"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.LifecyclePolicy = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const efs_generated_1 = require("./efs.generated");
// tslint:disable:max-line-length
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
// tslint:enable
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_7_DAYS"] = 0] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_14_DAYS"] = 1] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_30_DAYS"] = 2] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_60_DAYS"] = 3] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_90_DAYS"] = 4] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-efs-filesystem-performancemode
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * This is the general purpose performance mode for most file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * This performance mode can scale to higher levels of aggregate throughput and operations per second with a
     * tradeoff of slightly higher latencies.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-throughputmode
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     *  This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lifecyclePolicies: (props.lifecyclePolicy ? Array.of({
                transitionToIa: LifecyclePolicy[props.lifecyclePolicy],
            }) : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_a = props.provisionedThroughputPerSecond) === null || _a === void 0 ? void 0 : _a.toMebibytes(),
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        core_1.Tag.add(this, 'Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        subnets.subnetIds.forEach((subnetId) => {
            new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
        });
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.fileSystemId = attrs.fileSystemId;
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.FileSystem = FileSystem;
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
//# sourceMappingURL=data:application/json;base64,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