"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
//# sourceMappingURL=data:application/json;base64,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