"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegendPosition = exports.Color = exports.Shading = exports.SingleValueWidget = exports.GraphWidget = exports.AlarmWidget = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * A dashboard widget that displays metrics
 */
class GraphWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.props.left || [], this.props.right || []);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                    legend: this.props.legendPosition !== undefined ? { position: this.props.legendPosition } : undefined,
                    liveData: this.props.liveData,
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation
 */
var Shading;
(function (Shading) {
    /**
     * Don't add shading
     */
    Shading["NONE"] = "none";
    /**
     * Add shading above the annotation
     */
    Shading["ABOVE"] = "above";
    /**
     * Add shading below the annotation
     */
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/** blue - hex #1f77b4 */
Color.BLUE = '#1f77b4';
/** brown - hex #8c564b */
Color.BROWN = '#8c564b';
/** green - hex #2ca02c */
Color.GREEN = '#2ca02c';
/** grey - hex #7f7f7f */
Color.GREY = '#7f7f7f';
/** orange - hex #ff7f0e */
Color.ORANGE = '#ff7f0e';
/** pink - hex #e377c2 */
Color.PINK = '#e377c2';
/** purple - hex #9467bd */
Color.PURPLE = '#9467bd';
/** red - hex #d62728 */
Color.RED = '#d62728';
/**
 * The position of the legend on a GraphWidget.
 */
var LegendPosition;
(function (LegendPosition) {
    /**
     * Legend appears below the graph (default).
     */
    LegendPosition["BOTTOM"] = "bottom";
    /**
     * Add shading above the annotation
     */
    LegendPosition["RIGHT"] = "right";
    /**
     * Add shading below the annotation
     */
    LegendPosition["HIDDEN"] = "hidden";
})(LegendPosition = exports.LegendPosition || (exports.LegendPosition = {}));
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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