"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * A certificate managed by AWS Certificate Manager
 *
 * IMPORTANT: if you are creating a certificate as part of your stack, the stack
 * will not complete creating until you read and follow the instructions in the
 * email that you will receive.
 *
 * ACM will send validation emails to the following addresses:
 *
 *  admin@domain.com
 *  administrator@domain.com
 *  hostmaster@domain.com
 *  postmaster@domain.com
 *  webmaster@domain.com
 *
 * For every domain that you register.
 */
class Certificate extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: allDomainNames.map(domainValidationOption),
            validationMethod: props.validationMethod,
        });
        this.certificateArn = cert.ref;
        /**
         * Return the domain validation options for the given domain
         *
         * Closes over props.
         */
        function domainValidationOption(domainName) {
            let validationDomain = props.validationDomains && props.validationDomains[domainName];
            if (validationDomain === undefined) {
                if (core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                validationDomain = util_1.apexDomain(domainName);
            }
            return { domainName, validationDomain };
        }
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
//# sourceMappingURL=data:application/json;base64,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