"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
class AutoScalingGroupBase extends core_1.Resource {
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.notifications = [];
        this.securityGroup = new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            this.node.addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.autoScalingGroup.ref;
        this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
            service: 'autoscaling',
            resource: 'autoScalingGroup:*:autoScalingGroupName',
            resourceName: this.autoScalingGroupName,
        });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props);
        this.spotPrice = props.spotPrice;
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Add command to the startup script of fleet instances.
     * The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        // undefined is treated as 'true'
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launced
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custome group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: config.suspendProcesses !== undefined ? config.suspendProcesses :
            // Recommended list of processes to suspend from here:
            // https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
            [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
                ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS],
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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