"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resolver = exports.MappingTemplate = exports.Values = exports.AttributeValuesStep = exports.AttributeValues = exports.PartitionKey = exports.PrimaryKey = exports.SortKeyStep = exports.PartitionKeyStep = exports.Assign = exports.KeyCondition = exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = exports.GraphQLApi = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes.length) ? this.formatAdditionalAuthorizationModes(props.authorizationConfig.additionalAuthorizationModes)
                : undefined,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (defaultAuthorizationType === AuthorizationType.API_KEY ||
            ((_j = (_h = props.authorizationConfig) === null || _h === void 0 ? void 0 : _h.additionalAuthorizationModes) === null || _j === void 0 ? void 0 : _j.findIndex((authMode) => authMode.authorizationType === AuthorizationType.API_KEY)) !== -1) {
            const apiKeyConfig = ((_l = (_k = props.authorizationConfig) === null || _k === void 0 ? void 0 : _k.defaultAuthorization) === null || _l === void 0 ? void 0 : _l.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this.createAPIKey(apiKeyConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name, description, endpoint) {
        return new HttpDataSource(this, `${name}DS`, {
            api: this,
            description,
            endpoint,
            name,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        this._apiKey = key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
}
exports.GraphQLApi = GraphQLApi;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: (_a = this.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a http endpoint
 */
class HttpDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            httpConfig: {
                endpoint: props.endpoint,
            },
            type: 'HTTP',
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", ${this.arg}))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
      ${assignments.join(',')}
    }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the variables required for `renderTemplate`.
     */
    renderVariables() {
        return `#set($input = ${this.container})
      ${this.assignments.map(a => a.putInMap('input')).join('\n')}`;
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return '"attributeValues": $util.dynamodb.toMapValuesJson($input)';
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('UTF-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`
      ${values.renderVariables()}
      {
        "version": "2017-02-28",
        "operation": "PutItem",
        ${key.renderTemplate()},
        ${values.renderTemplate()}
      }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
/**
 * An AppSync resolver
 */
class Resolver extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.resolver = new appsync_generated_1.CfnResolver(this, 'Resource', {
            apiId: props.api.apiId,
            typeName: props.typeName,
            fieldName: props.fieldName,
            dataSourceName: props.dataSource ? props.dataSource.name : undefined,
            kind: props.pipelineConfig ? 'PIPELINE' : 'UNIT',
            requestMappingTemplate: props.requestMappingTemplate ? props.requestMappingTemplate.renderTemplate() : undefined,
            responseMappingTemplate: props.responseMappingTemplate ? props.responseMappingTemplate.renderTemplate() : undefined,
        });
        this.resolver.addDependsOn(props.api.schema);
        if (props.dataSource) {
            this.resolver.addDependsOn(props.dataSource.ds);
        }
        this.arn = this.resolver.attrResolverArn;
    }
}
exports.Resolver = Resolver;
//# sourceMappingURL=data:application/json;base64,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