"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi
 */
class RestApiBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * Gets the "execute-api" ARN
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @param method The method (default `*`)
     * @param path The resource path. Must start with '/' (default `*`)
     * @param stage The stage (default `*`)
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
}
exports.RestApiBase = RestApiBase;
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 *
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, props, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing RestApi that can be configured with additional Methods and Resources.
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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