"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(props.integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_a = options.authorizationScopes) !== null && _a !== void 0 ? _a : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ method: methodProps });
        }
    }
    /**
     * The RestApi associated with this Method
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled.
     *
     * @attribute
     */
    get methodArn() {
        if (!this.restApi.deploymentStage) {
            throw new Error(`Unable to determine ARN for method "${this.node.id}" since there is no stage associated with this API.\n` +
                'Either use the `deploy` prop or explicitly assign `deploymentStage` on the RestApi');
        }
        const stage = this.restApi.deploymentStage.stageName.toString();
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(integration) {
        if (!integration) {
            // use defaultIntegration from API if defined
            if (this.resource.defaultIntegration) {
                return this.renderIntegration(this.resource.defaultIntegration);
            }
            // fallback to mock
            return this.renderIntegration(new mock_1.MockIntegration());
        }
        integration.bind(this);
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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