"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
    }
    /**
     * Allows `Method` to access the integration props.
     *
     * @internal
     */
    get _props() {
        return this.props;
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        return;
    }
}
exports.Integration = Integration;
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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