"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextProvider = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const annotations_1 = require("./annotations");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers.
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @stability stable
 */
class ContextProvider {
    constructor() { }
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     * @stability stable
     */
    static getKey(scope, options) {
        jsiiDeprecationWarnings._aws_cdk_core_GetContextKeyOptions(options);
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    /**
     * @stability stable
     */
    static getValue(scope, options) {
        jsiiDeprecationWarnings._aws_cdk_core_GetContextValueOptions(options);
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Configure "env" with an account and region when ' +
                'you define your stack.' +
                'See https://docs.aws.amazon.com/cdk/latest/guide/environments.html for more details.');
        }
        const { key, props } = this.getKey(scope, options);
        const value = constructs_1.Node.of(scope).tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContextKey({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                annotations_1.Annotations.of(scope).addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
}
exports.ContextProvider = ContextProvider;
_a = JSII_RTTI_SYMBOL_1;
ContextProvider[_a] = { fqn: "@aws-cdk/core.ContextProvider", version: "1.134.0" };
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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