"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderIntrinsics = void 0;
/**
 * Turn CloudFormation intrinsics into strings
 *
 * ------
 *
 * This stringification is not intended to be mechanically reversible! It's intended
 * to be understood by humans!
 *
 * ------
 *
 * Turns Fn::GetAtt and Fn::Ref objects into the same strings that can be
 * parsed by Fn::Sub, but without the surrounding intrinsics.
 *
 * Evaluates Fn::Join directly if the second argument is a literal list of strings.
 *
 * Removes list and object values evaluating to { Ref: 'AWS::NoValue' }.
 *
 * For other intrinsics we choose a string representation that CloudFormation
 * cannot actually parse, but is comprehensible to humans.
 */
function renderIntrinsics(x) {
    if (Array.isArray(x)) {
        return x.filter(el => !isNoValue(el)).map(renderIntrinsics);
    }
    if (isNoValue(x)) {
        return undefined;
    }
    const intrinsic = getIntrinsic(x);
    if (intrinsic) {
        if (intrinsic.fn === 'Ref') {
            return '${' + intrinsic.args + '}';
        }
        if (intrinsic.fn === 'Fn::GetAtt') {
            return '${' + intrinsic.args[0] + '.' + intrinsic.args[1] + '}';
        }
        if (intrinsic.fn === 'Fn::Join') {
            return unCloudFormationFnJoin(intrinsic.args[0], intrinsic.args[1]);
        }
        return stringifyIntrinsic(intrinsic.fn, intrinsic.args);
    }
    if (typeof x === 'object' && x !== null) {
        const ret = {};
        for (const [key, value] of Object.entries(x)) {
            if (!isNoValue(value)) {
                ret[key] = renderIntrinsics(value);
            }
        }
        return ret;
    }
    return x;
}
exports.renderIntrinsics = renderIntrinsics;
function unCloudFormationFnJoin(separator, args) {
    if (Array.isArray(args)) {
        return args.filter(el => !isNoValue(el)).map(renderIntrinsics).join(separator);
    }
    return stringifyIntrinsic('Fn::Join', [separator, args]);
}
function stringifyIntrinsic(fn, args) {
    return JSON.stringify({ [fn]: renderIntrinsics(args) });
}
function getIntrinsic(x) {
    if (x === undefined || x === null || Array.isArray(x)) {
        return undefined;
    }
    if (typeof x !== 'object') {
        return undefined;
    }
    const keys = Object.keys(x);
    return keys.length === 1 && (keys[0] === 'Ref' || keys[0].startsWith('Fn::')) ? { fn: keys[0], args: x[keys[0]] } : undefined;
}
function isNoValue(x) {
    const int = getIntrinsic(x);
    return int && int.fn === 'Ref' && int.args === 'AWS::NoValue';
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVuZGVyLWludHJpbnNpY3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJyZW5kZXItaW50cmluc2ljcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQTs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQW1CRztBQUNILFNBQWdCLGdCQUFnQixDQUFDLENBQU07SUFDckMsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO1FBQ3BCLE9BQU8sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUM7S0FDN0Q7SUFFRCxJQUFJLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFFdkMsTUFBTSxTQUFTLEdBQUcsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ2xDLElBQUksU0FBUyxFQUFFO1FBQ2IsSUFBSSxTQUFTLENBQUMsRUFBRSxLQUFLLEtBQUssRUFBRTtZQUFFLE9BQU8sSUFBSSxHQUFHLFNBQVMsQ0FBQyxJQUFJLEdBQUcsR0FBRyxDQUFDO1NBQUU7UUFDbkUsSUFBSSxTQUFTLENBQUMsRUFBRSxLQUFLLFlBQVksRUFBRTtZQUFFLE9BQU8sSUFBSSxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEdBQUcsR0FBRyxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEdBQUcsR0FBRyxDQUFDO1NBQUU7UUFDdkcsSUFBSSxTQUFTLENBQUMsRUFBRSxLQUFLLFVBQVUsRUFBRTtZQUFFLE9BQU8sc0JBQXNCLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FBRTtRQUN6RyxPQUFPLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxFQUFFLEVBQUUsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQ3pEO0lBRUQsSUFBSSxPQUFPLENBQUMsS0FBSyxRQUFRLElBQUksQ0FBQyxLQUFLLElBQUksRUFBRTtRQUN2QyxNQUFNLEdBQUcsR0FBUSxFQUFFLENBQUM7UUFDcEIsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDNUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDckIsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO2FBQ3BDO1NBQ0Y7UUFDRCxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBQ0QsT0FBTyxDQUFDLENBQUM7QUFDWCxDQUFDO0FBekJELDRDQXlCQztBQUVELFNBQVMsc0JBQXNCLENBQUMsU0FBaUIsRUFBRSxJQUFTO0lBQzFELElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtRQUN2QixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztLQUNoRjtJQUNELE9BQU8sa0JBQWtCLENBQUMsVUFBVSxFQUFFLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7QUFDM0QsQ0FBQztBQUVELFNBQVMsa0JBQWtCLENBQUMsRUFBVSxFQUFFLElBQVM7SUFDL0MsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDMUQsQ0FBQztBQUVELFNBQVMsWUFBWSxDQUFDLENBQU07SUFDMUIsSUFBSSxDQUFDLEtBQUssU0FBUyxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFDNUUsSUFBSSxPQUFPLENBQUMsS0FBSyxRQUFRLEVBQUU7UUFBRSxPQUFPLFNBQVMsQ0FBQztLQUFFO0lBQ2hELE1BQU0sSUFBSSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUIsT0FBTyxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxLQUFLLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7QUFDaEksQ0FBQztBQUVELFNBQVMsU0FBUyxDQUFDLENBQU07SUFDdkIsTUFBTSxHQUFHLEdBQUcsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzVCLE9BQU8sR0FBRyxJQUFJLEdBQUcsQ0FBQyxFQUFFLEtBQUssS0FBSyxJQUFJLEdBQUcsQ0FBQyxJQUFJLEtBQUssY0FBYyxDQUFDO0FBQ2hFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIFR1cm4gQ2xvdWRGb3JtYXRpb24gaW50cmluc2ljcyBpbnRvIHN0cmluZ3NcbiAqXG4gKiAtLS0tLS1cbiAqXG4gKiBUaGlzIHN0cmluZ2lmaWNhdGlvbiBpcyBub3QgaW50ZW5kZWQgdG8gYmUgbWVjaGFuaWNhbGx5IHJldmVyc2libGUhIEl0J3MgaW50ZW5kZWRcbiAqIHRvIGJlIHVuZGVyc3Rvb2QgYnkgaHVtYW5zIVxuICpcbiAqIC0tLS0tLVxuICpcbiAqIFR1cm5zIEZuOjpHZXRBdHQgYW5kIEZuOjpSZWYgb2JqZWN0cyBpbnRvIHRoZSBzYW1lIHN0cmluZ3MgdGhhdCBjYW4gYmVcbiAqIHBhcnNlZCBieSBGbjo6U3ViLCBidXQgd2l0aG91dCB0aGUgc3Vycm91bmRpbmcgaW50cmluc2ljcy5cbiAqXG4gKiBFdmFsdWF0ZXMgRm46OkpvaW4gZGlyZWN0bHkgaWYgdGhlIHNlY29uZCBhcmd1bWVudCBpcyBhIGxpdGVyYWwgbGlzdCBvZiBzdHJpbmdzLlxuICpcbiAqIFJlbW92ZXMgbGlzdCBhbmQgb2JqZWN0IHZhbHVlcyBldmFsdWF0aW5nIHRvIHsgUmVmOiAnQVdTOjpOb1ZhbHVlJyB9LlxuICpcbiAqIEZvciBvdGhlciBpbnRyaW5zaWNzIHdlIGNob29zZSBhIHN0cmluZyByZXByZXNlbnRhdGlvbiB0aGF0IENsb3VkRm9ybWF0aW9uXG4gKiBjYW5ub3QgYWN0dWFsbHkgcGFyc2UsIGJ1dCBpcyBjb21wcmVoZW5zaWJsZSB0byBodW1hbnMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJJbnRyaW5zaWNzKHg6IGFueSk6IGFueSB7XG4gIGlmIChBcnJheS5pc0FycmF5KHgpKSB7XG4gICAgcmV0dXJuIHguZmlsdGVyKGVsID0+ICFpc05vVmFsdWUoZWwpKS5tYXAocmVuZGVySW50cmluc2ljcyk7XG4gIH1cblxuICBpZiAoaXNOb1ZhbHVlKHgpKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cblxuICBjb25zdCBpbnRyaW5zaWMgPSBnZXRJbnRyaW5zaWMoeCk7XG4gIGlmIChpbnRyaW5zaWMpIHtcbiAgICBpZiAoaW50cmluc2ljLmZuID09PSAnUmVmJykgeyByZXR1cm4gJyR7JyArIGludHJpbnNpYy5hcmdzICsgJ30nOyB9XG4gICAgaWYgKGludHJpbnNpYy5mbiA9PT0gJ0ZuOjpHZXRBdHQnKSB7IHJldHVybiAnJHsnICsgaW50cmluc2ljLmFyZ3NbMF0gKyAnLicgKyBpbnRyaW5zaWMuYXJnc1sxXSArICd9JzsgfVxuICAgIGlmIChpbnRyaW5zaWMuZm4gPT09ICdGbjo6Sm9pbicpIHsgcmV0dXJuIHVuQ2xvdWRGb3JtYXRpb25GbkpvaW4oaW50cmluc2ljLmFyZ3NbMF0sIGludHJpbnNpYy5hcmdzWzFdKTsgfVxuICAgIHJldHVybiBzdHJpbmdpZnlJbnRyaW5zaWMoaW50cmluc2ljLmZuLCBpbnRyaW5zaWMuYXJncyk7XG4gIH1cblxuICBpZiAodHlwZW9mIHggPT09ICdvYmplY3QnICYmIHggIT09IG51bGwpIHtcbiAgICBjb25zdCByZXQ6IGFueSA9IHt9O1xuICAgIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHgpKSB7XG4gICAgICBpZiAoIWlzTm9WYWx1ZSh2YWx1ZSkpIHtcbiAgICAgICAgcmV0W2tleV0gPSByZW5kZXJJbnRyaW5zaWNzKHZhbHVlKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuICByZXR1cm4geDtcbn1cblxuZnVuY3Rpb24gdW5DbG91ZEZvcm1hdGlvbkZuSm9pbihzZXBhcmF0b3I6IHN0cmluZywgYXJnczogYW55KSB7XG4gIGlmIChBcnJheS5pc0FycmF5KGFyZ3MpKSB7XG4gICAgcmV0dXJuIGFyZ3MuZmlsdGVyKGVsID0+ICFpc05vVmFsdWUoZWwpKS5tYXAocmVuZGVySW50cmluc2ljcykuam9pbihzZXBhcmF0b3IpO1xuICB9XG4gIHJldHVybiBzdHJpbmdpZnlJbnRyaW5zaWMoJ0ZuOjpKb2luJywgW3NlcGFyYXRvciwgYXJnc10pO1xufVxuXG5mdW5jdGlvbiBzdHJpbmdpZnlJbnRyaW5zaWMoZm46IHN0cmluZywgYXJnczogYW55KSB7XG4gIHJldHVybiBKU09OLnN0cmluZ2lmeSh7IFtmbl06IHJlbmRlckludHJpbnNpY3MoYXJncykgfSk7XG59XG5cbmZ1bmN0aW9uIGdldEludHJpbnNpYyh4OiBhbnkpOiBJbnRyaW5zaWMgfCB1bmRlZmluZWQge1xuICBpZiAoeCA9PT0gdW5kZWZpbmVkIHx8IHggPT09IG51bGwgfHwgQXJyYXkuaXNBcnJheSh4KSkgeyByZXR1cm4gdW5kZWZpbmVkOyB9XG4gIGlmICh0eXBlb2YgeCAhPT0gJ29iamVjdCcpIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuICBjb25zdCBrZXlzID0gT2JqZWN0LmtleXMoeCk7XG4gIHJldHVybiBrZXlzLmxlbmd0aCA9PT0gMSAmJiAoa2V5c1swXSA9PT0gJ1JlZicgfHwga2V5c1swXS5zdGFydHNXaXRoKCdGbjo6JykpID8geyBmbjoga2V5c1swXSwgYXJnczogeFtrZXlzWzBdXSB9IDogdW5kZWZpbmVkO1xufVxuXG5mdW5jdGlvbiBpc05vVmFsdWUoeDogYW55KSB7XG4gIGNvbnN0IGludCA9IGdldEludHJpbnNpYyh4KTtcbiAgcmV0dXJuIGludCAmJiBpbnQuZm4gPT09ICdSZWYnICYmIGludC5hcmdzID09PSAnQVdTOjpOb1ZhbHVlJztcbn1cblxuaW50ZXJmYWNlIEludHJpbnNpYyB7XG4gIGZuOiBzdHJpbmc7XG4gIGFyZ3M6IGFueTtcbn0iXX0=