"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatSecurityChanges = exports.formatDifferences = void 0;
const util_1 = require("util");
const colors = require("colors/safe");
const diff_template_1 = require("./diff-template");
const util_2 = require("./diff/util");
const format_table_1 = require("./format-table");
// from cx-api
const PATH_METADATA_KEY = 'aws:cdk:path';
/* eslint-disable @typescript-eslint/no-require-imports */
const { structuredPatch } = require('diff');
/**
 * Renders template differences to the process' console.
 *
 * @param stream           The IO stream where to output the rendered diff.
 * @param templateDiff     TemplateDiff to be rendered to the console.
 * @param logicalToPathMap A map from logical ID to construct path. Useful in
 *                         case there is no aws:cdk:path metadata in the template.
 * @param context          the number of context lines to use in arbitrary JSON diff (defaults to 3).
 */
function formatDifferences(stream, templateDiff, logicalToPathMap = {}, context = 3) {
    const formatter = new Formatter(stream, logicalToPathMap, templateDiff, context);
    if (templateDiff.awsTemplateFormatVersion || templateDiff.transform || templateDiff.description) {
        formatter.printSectionHeader('Template');
        formatter.formatDifference('AWSTemplateFormatVersion', 'AWSTemplateFormatVersion', templateDiff.awsTemplateFormatVersion);
        formatter.formatDifference('Transform', 'Transform', templateDiff.transform);
        formatter.formatDifference('Description', 'Description', templateDiff.description);
        formatter.printSectionFooter();
    }
    formatSecurityChangesWithBanner(formatter, templateDiff);
    formatter.formatSection('Parameters', 'Parameter', templateDiff.parameters);
    formatter.formatSection('Metadata', 'Metadata', templateDiff.metadata);
    formatter.formatSection('Mappings', 'Mapping', templateDiff.mappings);
    formatter.formatSection('Conditions', 'Condition', templateDiff.conditions);
    formatter.formatSection('Resources', 'Resource', templateDiff.resources, formatter.formatResourceDifference.bind(formatter));
    formatter.formatSection('Outputs', 'Output', templateDiff.outputs);
    formatter.formatSection('Other Changes', 'Unknown', templateDiff.unknown);
}
exports.formatDifferences = formatDifferences;
/**
 * Renders a diff of security changes to the given stream
 */
function formatSecurityChanges(stream, templateDiff, logicalToPathMap = {}, context) {
    const formatter = new Formatter(stream, logicalToPathMap, templateDiff, context);
    formatSecurityChangesWithBanner(formatter, templateDiff);
}
exports.formatSecurityChanges = formatSecurityChanges;
function formatSecurityChangesWithBanner(formatter, templateDiff) {
    if (!templateDiff.iamChanges.hasChanges && !templateDiff.securityGroupChanges.hasChanges) {
        return;
    }
    formatter.formatIamChanges(templateDiff.iamChanges);
    formatter.formatSecurityGroupChanges(templateDiff.securityGroupChanges);
    formatter.warning('(NOTE: There may be security-related changes not in this list. See https://github.com/aws/aws-cdk/issues/1299)');
    formatter.printSectionFooter();
}
const ADDITION = colors.green('[+]');
const CONTEXT = colors.grey('[ ]');
const UPDATE = colors.yellow('[~]');
const REMOVAL = colors.red('[-]');
class Formatter {
    constructor(stream, logicalToPathMap, diff, context = 3) {
        this.stream = stream;
        this.logicalToPathMap = logicalToPathMap;
        this.context = context;
        // Read additional construct paths from the diff if it is supplied
        if (diff) {
            this.readConstructPathsFrom(diff);
        }
    }
    print(fmt, ...args) {
        this.stream.write(colors.white(util_1.format(fmt, ...args)) + '\n');
    }
    warning(fmt, ...args) {
        this.stream.write(colors.yellow(util_1.format(fmt, ...args)) + '\n');
    }
    formatSection(title, entryType, collection, formatter = this.formatDifference.bind(this)) {
        if (collection.differenceCount === 0) {
            return;
        }
        this.printSectionHeader(title);
        collection.forEachDifference((id, diff) => formatter(entryType, id, diff));
        this.printSectionFooter();
    }
    printSectionHeader(title) {
        this.print(colors.underline(colors.bold(title)));
    }
    printSectionFooter() {
        this.print('');
    }
    /**
     * Print a simple difference for a given named entity.
     *
     * @param logicalId the name of the entity that is different.
     * @param diff the difference to be rendered.
     */
    formatDifference(type, logicalId, diff) {
        if (!diff || !diff.isDifferent) {
            return;
        }
        let value;
        const oldValue = this.formatValue(diff.oldValue, colors.red);
        const newValue = this.formatValue(diff.newValue, colors.green);
        if (diff.isAddition) {
            value = newValue;
        }
        else if (diff.isUpdate) {
            value = `${oldValue} to ${newValue}`;
        }
        else if (diff.isRemoval) {
            value = oldValue;
        }
        this.print(`${this.formatPrefix(diff)} ${colors.cyan(type)} ${this.formatLogicalId(logicalId)}: ${value}`);
    }
    /**
     * Print a resource difference for a given logical ID.
     *
     * @param logicalId the logical ID of the resource that changed.
     * @param diff      the change to be rendered.
     */
    formatResourceDifference(_type, logicalId, diff) {
        if (!diff.isDifferent) {
            return;
        }
        const resourceType = diff.isRemoval ? diff.oldResourceType : diff.newResourceType;
        // eslint-disable-next-line max-len
        this.print(`${this.formatPrefix(diff)} ${this.formatValue(resourceType, colors.cyan)} ${this.formatLogicalId(logicalId)} ${this.formatImpact(diff.changeImpact)}`);
        if (diff.isUpdate) {
            const differenceCount = diff.differenceCount;
            let processedCount = 0;
            diff.forEachDifference((_, name, values) => {
                processedCount += 1;
                this.formatTreeDiff(name, values, processedCount === differenceCount);
            });
        }
    }
    formatPrefix(diff) {
        if (diff.isAddition) {
            return ADDITION;
        }
        if (diff.isUpdate) {
            return UPDATE;
        }
        if (diff.isRemoval) {
            return REMOVAL;
        }
        return colors.white('[?]');
    }
    /**
     * @param value the value to be formatted.
     * @param color the color to be used.
     *
     * @returns the formatted string, with color applied.
     */
    formatValue(value, color) {
        if (value == null) {
            return undefined;
        }
        if (typeof value === 'string') {
            return color(value);
        }
        return color(JSON.stringify(value));
    }
    /**
     * @param impact the impact to be formatted
     * @returns a user-friendly, colored string representing the impact.
     */
    formatImpact(impact) {
        switch (impact) {
            case diff_template_1.ResourceImpact.MAY_REPLACE:
                return colors.italic(colors.yellow('may be replaced'));
            case diff_template_1.ResourceImpact.WILL_REPLACE:
                return colors.italic(colors.bold(colors.red('replace')));
            case diff_template_1.ResourceImpact.WILL_DESTROY:
                return colors.italic(colors.bold(colors.red('destroy')));
            case diff_template_1.ResourceImpact.WILL_ORPHAN:
                return colors.italic(colors.yellow('orphan'));
            case diff_template_1.ResourceImpact.WILL_UPDATE:
            case diff_template_1.ResourceImpact.WILL_CREATE:
            case diff_template_1.ResourceImpact.NO_CHANGE:
                return ''; // no extra info is gained here
        }
    }
    /**
     * Renders a tree of differences under a particular name.
     * @param name    the name of the root of the tree.
     * @param diff    the difference on the tree.
     * @param last    whether this is the last node of a parent tree.
     */
    formatTreeDiff(name, diff, last) {
        let additionalInfo = '';
        if (diff_template_1.isPropertyDifference(diff)) {
            if (diff.changeImpact === diff_template_1.ResourceImpact.MAY_REPLACE) {
                additionalInfo = ' (may cause replacement)';
            }
            else if (diff.changeImpact === diff_template_1.ResourceImpact.WILL_REPLACE) {
                additionalInfo = ' (requires replacement)';
            }
        }
        this.print(' %s─ %s %s%s', last ? '└' : '├', this.changeTag(diff.oldValue, diff.newValue), name, additionalInfo);
        return this.formatObjectDiff(diff.oldValue, diff.newValue, ` ${last ? ' ' : '│'}`);
    }
    /**
     * Renders the difference between two objects, looking for the differences as deep as possible,
     * and rendering a tree graph of the path until the difference is found.
     *
     * @param oldObject  the old object.
     * @param newObject  the new object.
     * @param linePrefix a prefix (indent-like) to be used on every line.
     */
    formatObjectDiff(oldObject, newObject, linePrefix) {
        if ((typeof oldObject !== typeof newObject) || Array.isArray(oldObject) || typeof oldObject === 'string' || typeof oldObject === 'number') {
            if (oldObject !== undefined && newObject !== undefined) {
                if (typeof oldObject === 'object' || typeof newObject === 'object') {
                    const oldStr = JSON.stringify(oldObject, null, 2);
                    const newStr = JSON.stringify(newObject, null, 2);
                    const diff = _diffStrings(oldStr, newStr, this.context);
                    for (let i = 0; i < diff.length; i++) {
                        this.print('%s   %s %s', linePrefix, i === 0 ? '└─' : '  ', diff[i]);
                    }
                }
                else {
                    this.print('%s   ├─ %s %s', linePrefix, REMOVAL, this.formatValue(oldObject, colors.red));
                    this.print('%s   └─ %s %s', linePrefix, ADDITION, this.formatValue(newObject, colors.green));
                }
            }
            else if (oldObject !== undefined /* && newObject === undefined */) {
                this.print('%s   └─ %s', linePrefix, this.formatValue(oldObject, colors.red));
            }
            else /* if (oldObject === undefined && newObject !== undefined) */ {
                this.print('%s   └─ %s', linePrefix, this.formatValue(newObject, colors.green));
            }
            return;
        }
        const keySet = new Set(Object.keys(oldObject));
        Object.keys(newObject).forEach(k => keySet.add(k));
        const keys = new Array(...keySet).filter(k => !util_2.deepEqual(oldObject[k], newObject[k])).sort();
        const lastKey = keys[keys.length - 1];
        for (const key of keys) {
            const oldValue = oldObject[key];
            const newValue = newObject[key];
            const treePrefix = key === lastKey ? '└' : '├';
            if (oldValue !== undefined && newValue !== undefined) {
                this.print('%s   %s─ %s %s:', linePrefix, treePrefix, this.changeTag(oldValue, newValue), colors.blue(`.${key}`));
                this.formatObjectDiff(oldValue, newValue, `${linePrefix}   ${key === lastKey ? ' ' : '│'}`);
            }
            else if (oldValue !== undefined /* && newValue === undefined */) {
                this.print('%s   %s─ %s Removed: %s', linePrefix, treePrefix, REMOVAL, colors.blue(`.${key}`));
            }
            else /* if (oldValue === undefined && newValue !== undefined */ {
                this.print('%s   %s─ %s Added: %s', linePrefix, treePrefix, ADDITION, colors.blue(`.${key}`));
            }
        }
    }
    /**
     * @param oldValue the old value of a difference.
     * @param newValue the new value of a difference.
     *
     * @returns a tag to be rendered in the diff, reflecting whether the difference
     *      was an ADDITION, UPDATE or REMOVAL.
     */
    changeTag(oldValue, newValue) {
        if (oldValue !== undefined && newValue !== undefined) {
            return UPDATE;
        }
        else if (oldValue !== undefined /* && newValue === undefined*/) {
            return REMOVAL;
        }
        else /* if (oldValue === undefined && newValue !== undefined) */ {
            return ADDITION;
        }
    }
    /**
     * Find 'aws:cdk:path' metadata in the diff and add it to the logicalToPathMap
     *
     * There are multiple sources of logicalID -> path mappings: synth metadata
     * and resource metadata, and we combine all sources into a single map.
     */
    readConstructPathsFrom(templateDiff) {
        for (const [logicalId, resourceDiff] of Object.entries(templateDiff.resources)) {
            if (!resourceDiff) {
                continue;
            }
            const oldPathMetadata = resourceDiff.oldValue && resourceDiff.oldValue.Metadata && resourceDiff.oldValue.Metadata[PATH_METADATA_KEY];
            if (oldPathMetadata && !(logicalId in this.logicalToPathMap)) {
                this.logicalToPathMap[logicalId] = oldPathMetadata;
            }
            const newPathMetadata = resourceDiff.newValue && resourceDiff.newValue.Metadata && resourceDiff.newValue.Metadata[PATH_METADATA_KEY];
            if (newPathMetadata && !(logicalId in this.logicalToPathMap)) {
                this.logicalToPathMap[logicalId] = newPathMetadata;
            }
        }
    }
    formatLogicalId(logicalId) {
        // if we have a path in the map, return it
        const normalized = this.normalizedLogicalIdPath(logicalId);
        if (normalized) {
            return `${normalized} ${colors.gray(logicalId)}`;
        }
        return logicalId;
    }
    normalizedLogicalIdPath(logicalId) {
        // if we have a path in the map, return it
        const path = this.logicalToPathMap[logicalId];
        return path ? normalizePath(path) : undefined;
        /**
         * Path is supposed to start with "/stack-name". If this is the case (i.e. path has more than
         * two components, we remove the first part. Otherwise, we just use the full path.
         * @param p
         */
        function normalizePath(p) {
            if (p.startsWith('/')) {
                p = p.substr(1);
            }
            let parts = p.split('/');
            if (parts.length > 1) {
                parts = parts.slice(1);
                // remove the last component if it's "Resource" or "Default" (if we have more than a single component)
                if (parts.length > 1) {
                    const last = parts[parts.length - 1];
                    if (last === 'Resource' || last === 'Default') {
                        parts = parts.slice(0, parts.length - 1);
                    }
                }
                p = parts.join('/');
            }
            return p;
        }
    }
    formatIamChanges(changes) {
        if (!changes.hasChanges) {
            return;
        }
        if (changes.statements.hasChanges) {
            this.printSectionHeader('IAM Statement Changes');
            this.print(format_table_1.formatTable(this.deepSubstituteBracedLogicalIds(changes.summarizeStatements()), this.stream.columns));
        }
        if (changes.managedPolicies.hasChanges) {
            this.printSectionHeader('IAM Policy Changes');
            this.print(format_table_1.formatTable(this.deepSubstituteBracedLogicalIds(changes.summarizeManagedPolicies()), this.stream.columns));
        }
    }
    formatSecurityGroupChanges(changes) {
        if (!changes.hasChanges) {
            return;
        }
        this.printSectionHeader('Security Group Changes');
        this.print(format_table_1.formatTable(this.deepSubstituteBracedLogicalIds(changes.summarize()), this.stream.columns));
    }
    deepSubstituteBracedLogicalIds(rows) {
        return rows.map(row => row.map(this.substituteBracedLogicalIds.bind(this)));
    }
    /**
     * Substitute all strings like ${LogId.xxx} with the path instead of the logical ID
     */
    substituteBracedLogicalIds(source) {
        return source.replace(/\$\{([^.}]+)(.[^}]+)?\}/ig, (_match, logId, suffix) => {
            return '${' + (this.normalizedLogicalIdPath(logId) || logId) + (suffix || '') + '}';
        });
    }
}
/**
 * Creates a unified diff of two strings.
 *
 * @param oldStr  the "old" version of the string.
 * @param newStr  the "new" version of the string.
 * @param context the number of context lines to use in arbitrary JSON diff.
 *
 * @returns an array of diff lines.
 */
function _diffStrings(oldStr, newStr, context) {
    const patch = structuredPatch(null, null, oldStr, newStr, null, null, { context });
    const result = new Array();
    for (const hunk of patch.hunks) {
        result.push(colors.magenta(`@@ -${hunk.oldStart},${hunk.oldLines} +${hunk.newStart},${hunk.newLines} @@`));
        const baseIndent = _findIndent(hunk.lines);
        for (const line of hunk.lines) {
            // Don't care about termination newline.
            if (line === '\\ No newline at end of file') {
                continue;
            }
            const marker = line.charAt(0);
            const text = line.slice(1 + baseIndent);
            switch (marker) {
                case ' ':
                    result.push(`${CONTEXT} ${text}`);
                    break;
                case '+':
                    result.push(colors.bold(`${ADDITION} ${colors.green(text)}`));
                    break;
                case '-':
                    result.push(colors.bold(`${REMOVAL} ${colors.red(text)}`));
                    break;
                default:
                    throw new Error(`Unexpected diff marker: ${marker} (full line: ${line})`);
            }
        }
    }
    return result;
    function _findIndent(lines) {
        let indent = Number.MAX_SAFE_INTEGER;
        for (const line of lines) {
            for (let i = 1; i < line.length; i++) {
                if (line.charAt(i) !== ' ') {
                    indent = indent > i - 1 ? i - 1 : indent;
                    break;
                }
            }
        }
        return indent;
    }
}
//# sourceMappingURL=data:application/json;base64,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