"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.diffUnknown = exports.diffResource = exports.diffParameter = exports.diffOutput = exports.diffMetadata = exports.diffMapping = exports.diffCondition = exports.diffAttribute = void 0;
const cfnspec = require("@aws-cdk/cfnspec");
const types = require("./types");
const util_1 = require("./util");
function diffAttribute(oldValue, newValue) {
    return new types.Difference(_asString(oldValue), _asString(newValue));
}
exports.diffAttribute = diffAttribute;
function diffCondition(oldValue, newValue) {
    return new types.ConditionDifference(oldValue, newValue);
}
exports.diffCondition = diffCondition;
function diffMapping(oldValue, newValue) {
    return new types.MappingDifference(oldValue, newValue);
}
exports.diffMapping = diffMapping;
function diffMetadata(oldValue, newValue) {
    return new types.MetadataDifference(oldValue, newValue);
}
exports.diffMetadata = diffMetadata;
function diffOutput(oldValue, newValue) {
    return new types.OutputDifference(oldValue, newValue);
}
exports.diffOutput = diffOutput;
function diffParameter(oldValue, newValue) {
    return new types.ParameterDifference(oldValue, newValue);
}
exports.diffParameter = diffParameter;
function diffResource(oldValue, newValue) {
    const resourceType = {
        oldType: oldValue && oldValue.Type,
        newType: newValue && newValue.Type,
    };
    let propertyDiffs = {};
    let otherDiffs = {};
    if (resourceType.oldType !== undefined && resourceType.oldType === resourceType.newType) {
        // Only makes sense to inspect deeper if the types stayed the same
        const typeSpec = cfnspec.filteredSpecification(resourceType.oldType);
        const impl = typeSpec.ResourceTypes[resourceType.oldType];
        propertyDiffs = util_1.diffKeyedEntities(oldValue.Properties, newValue.Properties, (oldVal, newVal, key) => _diffProperty(oldVal, newVal, key, impl));
        otherDiffs = util_1.diffKeyedEntities(oldValue, newValue, _diffOther);
        delete otherDiffs.Properties;
    }
    return new types.ResourceDifference(oldValue, newValue, {
        resourceType, propertyDiffs, otherDiffs,
    });
    function _diffProperty(oldV, newV, key, resourceSpec) {
        let changeImpact = types.ResourceImpact.NO_CHANGE;
        const spec = resourceSpec && resourceSpec.Properties && resourceSpec.Properties[key];
        if (spec && !util_1.deepEqual(oldV, newV)) {
            switch (spec.UpdateType) {
                case cfnspec.schema.UpdateType.Immutable:
                    changeImpact = types.ResourceImpact.WILL_REPLACE;
                    break;
                case cfnspec.schema.UpdateType.Conditional:
                    changeImpact = types.ResourceImpact.MAY_REPLACE;
                    break;
                default:
                    // In those cases, whatever is the current value is what we should keep
                    changeImpact = types.ResourceImpact.WILL_UPDATE;
            }
        }
        return new types.PropertyDifference(oldV, newV, { changeImpact });
    }
    function _diffOther(oldV, newV) {
        return new types.Difference(oldV, newV);
    }
}
exports.diffResource = diffResource;
function diffUnknown(oldValue, newValue) {
    return new types.Difference(oldValue, newValue);
}
exports.diffUnknown = diffUnknown;
/**
 * Coerces a given value to +string | undefined+.
 *
 * @param value the value to be coerced.
 *
 * @returns +undefined+ if +value+ is +null+ or +undefined+,
 *      +value+ if it is a +string+,
 *      a compact JSON representation of +value+ otherwise.
 */
function _asString(value) {
    if (value == null) {
        return undefined;
    }
    if (typeof value === 'string') {
        return value;
    }
    return JSON.stringify(value);
}
//# sourceMappingURL=data:application/json;base64,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