"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !exports.hasOwnProperty(p)) __createBinding(exports, m, p);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.diffResource = exports.diffTemplate = void 0;
const impl = require("./diff");
const types = require("./diff/types");
const util_1 = require("./diff/util");
__exportStar(require("./diff/types"), exports);
const DIFF_HANDLERS = {
    AWSTemplateFormatVersion: (diff, oldValue, newValue) => diff.awsTemplateFormatVersion = impl.diffAttribute(oldValue, newValue),
    Description: (diff, oldValue, newValue) => diff.description = impl.diffAttribute(oldValue, newValue),
    Metadata: (diff, oldValue, newValue) => diff.metadata = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffMetadata)),
    Parameters: (diff, oldValue, newValue) => diff.parameters = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffParameter)),
    Mappings: (diff, oldValue, newValue) => diff.mappings = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffMapping)),
    Conditions: (diff, oldValue, newValue) => diff.conditions = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffCondition)),
    Transform: (diff, oldValue, newValue) => diff.transform = impl.diffAttribute(oldValue, newValue),
    Resources: (diff, oldValue, newValue) => diff.resources = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffResource)),
    Outputs: (diff, oldValue, newValue) => diff.outputs = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffOutput)),
};
/**
 * Compare two CloudFormation templates and return semantic differences between them.
 *
 * @param currentTemplate the current state of the stack.
 * @param newTemplate     the target state of the stack.
 *
 * @returns a +types.TemplateDiff+ object that represents the changes that will happen if
 *      a stack which current state is described by +currentTemplate+ is updated with
 *      the template +newTemplate+.
 */
function diffTemplate(currentTemplate, newTemplate) {
    // Base diff
    const theDiff = calculateTemplateDiff(currentTemplate, newTemplate);
    // We're going to modify this in-place
    const newTemplateCopy = deepCopy(newTemplate);
    let didPropagateReferenceChanges;
    let diffWithReplacements;
    do {
        diffWithReplacements = calculateTemplateDiff(currentTemplate, newTemplateCopy);
        // Propagate replacements for replaced resources
        didPropagateReferenceChanges = false;
        if (diffWithReplacements.resources) {
            diffWithReplacements.resources.forEachDifference((logicalId, change) => {
                if (change.changeImpact === types.ResourceImpact.WILL_REPLACE) {
                    if (propagateReplacedReferences(newTemplateCopy, logicalId)) {
                        didPropagateReferenceChanges = true;
                    }
                }
            });
        }
    } while (didPropagateReferenceChanges);
    // Copy "replaced" states from `diffWithReplacements` to `theDiff`.
    diffWithReplacements.resources
        .filter(r => isReplacement(r.changeImpact))
        .forEachDifference((logicalId, downstreamReplacement) => {
        const resource = theDiff.resources.get(logicalId);
        if (resource.changeImpact !== downstreamReplacement.changeImpact) {
            propagatePropertyReplacement(downstreamReplacement, resource);
        }
    });
    return theDiff;
}
exports.diffTemplate = diffTemplate;
function isReplacement(impact) {
    return impact === types.ResourceImpact.MAY_REPLACE || impact === types.ResourceImpact.WILL_REPLACE;
}
/**
 * For all properties in 'source' that have a "replacement" impact, propagate that impact to "dest"
 */
function propagatePropertyReplacement(source, dest) {
    for (const [propertyName, diff] of Object.entries(source.propertyUpdates)) {
        if (diff.changeImpact && isReplacement(diff.changeImpact)) {
            // Use the propertydiff of source in target. The result of this happens to be clear enough.
            dest.setPropertyChange(propertyName, diff);
        }
    }
}
function calculateTemplateDiff(currentTemplate, newTemplate) {
    const differences = {};
    const unknown = {};
    for (const key of util_1.unionOf(Object.keys(currentTemplate), Object.keys(newTemplate)).sort()) {
        const oldValue = currentTemplate[key];
        const newValue = newTemplate[key];
        if (util_1.deepEqual(oldValue, newValue)) {
            continue;
        }
        const handler = DIFF_HANDLERS[key]
            || ((_diff, oldV, newV) => unknown[key] = impl.diffUnknown(oldV, newV));
        handler(differences, oldValue, newValue);
    }
    if (Object.keys(unknown).length > 0) {
        differences.unknown = new types.DifferenceCollection(unknown);
    }
    return new types.TemplateDiff(differences);
}
/**
 * Compare two CloudFormation resources and return semantic differences between them
 */
function diffResource(oldValue, newValue) {
    return impl.diffResource(oldValue, newValue);
}
exports.diffResource = diffResource;
/**
 * Replace all references to the given logicalID on the given template, in-place
 *
 * Returns true iff any references were replaced.
 */
function propagateReplacedReferences(template, logicalId) {
    let ret = false;
    function recurse(obj) {
        if (Array.isArray(obj)) {
            obj.forEach(recurse);
        }
        if (typeof obj === 'object' && obj !== null) {
            if (!replaceReference(obj)) {
                Object.values(obj).forEach(recurse);
            }
        }
    }
    function replaceReference(obj) {
        const keys = Object.keys(obj);
        if (keys.length !== 1) {
            return false;
        }
        const key = keys[0];
        if (key === 'Ref') {
            if (obj.Ref === logicalId) {
                obj.Ref = logicalId + ' (replaced)';
                ret = true;
            }
            return true;
        }
        if (key.startsWith('Fn::')) {
            if (Array.isArray(obj[key]) && obj[key].length > 0 && obj[key][0] === logicalId) {
                obj[key][0] = logicalId + '(replaced)';
                ret = true;
            }
            return true;
        }
        return false;
    }
    recurse(template);
    return ret;
}
function deepCopy(x) {
    if (Array.isArray(x)) {
        return x.map(deepCopy);
    }
    if (typeof x === 'object' && x !== null) {
        const ret = {};
        for (const key of Object.keys(x)) {
            ret[key] = deepCopy(x[key]);
        }
        return ret;
    }
    return x;
}
//# sourceMappingURL=data:application/json;base64,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