"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util = require("util");
const fs = require("fs-extra");
/* eslint-disable @typescript-eslint/no-require-imports */
const jsonDiff = require('json-diff').diff;
/* eslint-enable */
function line(fmt = '', ...param) {
    process.stdout.write(util.format(fmt, ...param));
    process.stdout.write('\n');
}
async function main() {
    const title = process.argv[2];
    const oldSpecFile = process.argv[3];
    const newSpecFile = process.argv[4];
    const newSpec = await fs.readJSON(newSpecFile);
    const oldSpec = await fs.readJSON(oldSpecFile);
    const out = jsonDiff(oldSpec, newSpec);
    // Here's the magic output format of this thing
    // If a key ends in __added, it got added, and the value
    //   is the new value.
    // If a key ends in __deleted, it got deleted, and the value
    //   is the old value.
    // If a value got changed, the value object will look like:
    //   { __old: ..., __new: ... }
    if (!out) {
        return; // no diff
    }
    const resourceTypeAdditions = new Set();
    const resourceTypeDeletions = new Set();
    const attributeChanges = new Array();
    const propertyChanges = new Array();
    const propertyTypeChanges = new Array();
    for (const key of Object.keys(out.ResourceTypes || {})) {
        classifyResourceTypeUpdate(key, out.ResourceTypes[key]);
    }
    for (const key of Object.keys(out.PropertyTypes || {})) {
        classifyPropertyTypeUpdate(key, out.PropertyTypes[key]);
    }
    line(`# ${title} v${newSpec.ResourceSpecificationVersion}`);
    line();
    line('## New Resource Types');
    line();
    resourceTypeAdditions.forEach(type => line(`* ${type}`));
    line();
    if (resourceTypeDeletions.size > 0) {
        line('## Removed Resource Types');
        line();
        resourceTypeDeletions.forEach(type => line(`* ${type}`));
        line();
    }
    line('## Attribute Changes');
    line();
    attributeChanges.forEach(x => line(x));
    line();
    line('## Property Changes');
    line();
    propertyChanges.forEach(x => line(x));
    line();
    line('## Property Type Changes');
    line();
    propertyTypeChanges.forEach(x => line(x));
    function classifyResourceTypeUpdate(resourceType, update) {
        const added = isAdded(resourceType);
        if (added) {
            resourceTypeAdditions.add(added);
            return;
        }
        const deleted = isDeleted(resourceType);
        if (deleted) {
            resourceTypeDeletions.add(deleted);
            return;
        }
        pushDownCompleteChanges(update);
        for (const key of Object.keys(update)) {
            switch (key) {
                case 'Properties':
                    for (const prop of Object.keys(update.Properties)) {
                        describeChanges(resourceType, prop, update.Properties[prop]).forEach(change => {
                            propertyChanges.push(change);
                        });
                    }
                    break;
                case 'Attributes':
                    for (const attr of Object.keys(update.Attributes)) {
                        describeChanges(resourceType, attr, update.Attributes[attr]).forEach(change => {
                            attributeChanges.push(change);
                        });
                    }
                    break;
                case 'Documentation':
                    describeChanges(resourceType, key, update.Documentation).forEach(change => {
                        attributeChanges.push(change);
                    });
                    break;
                default:
                    throw new Error(`Unexpected update to ${resourceType}: ${key}`);
            }
        }
    }
    function classifyPropertyTypeUpdate(propertyType, update) {
        var _a;
        const added = isAdded(propertyType);
        if (added) {
            const resourceType = added.split('.')[0];
            if (resourceTypeAdditions.has(resourceType)) {
                return; // skipping property for added resource types
            }
            propertyTypeChanges.push(`* ${added} (__added__)`);
            return;
        }
        const deleted = isDeleted(propertyType);
        if (deleted) {
            const resourceType = deleted.split('.')[0];
            if (resourceTypeDeletions.has(resourceType)) {
                return; // skipping property for added resource types
            }
            propertyTypeChanges.push(`* ${deleted} (__removed__)`);
            return;
        }
        if (Object.keys(update).length !== 1 && Object.keys(update)[0] === 'Properties') {
            throw new Error('Unexpected update to a resource type. Expecting only "Properties" to change: ' + propertyType);
        }
        for (const prop of Object.keys((_a = update.Properties) !== null && _a !== void 0 ? _a : {})) {
            describeChanges(propertyType, prop, update.Properties[prop]).forEach(change => {
                propertyTypeChanges.push(change);
            });
        }
    }
    /**
     * Push down mass changes to attributes or properties to the individual properties.
     *
     * An example will explain this best. JSON-diff will make the smallest diff, so if there
     * are new properties it will report:
     *
     * "Properties__added": {
     *    "Property1": { ... },
     *    "Property2": { ... },
     * }
     *
     * But we want to see this as:
     *
     * "Properties": {
     *    "Property1__added": { ... },
     *    "Property2__added": { ... },
     * }
     *
     * Same (but in reverse) for deletions.
     */
    function pushDownCompleteChanges(update) {
        for (const [category, entries] of Object.entries(update)) {
            const addedKey = isAdded(category);
            if (addedKey) {
                delete update[category];
                update[addedKey] = suffixKeys('__added', entries);
            }
            const deletedKey = isDeleted(category);
            if (deletedKey) {
                delete update[category];
                update[deletedKey] = suffixKeys('__deleted', entries);
            }
        }
    }
    function isDeleted(key) {
        return isSuffix(key, '__deleted');
    }
    function isAdded(key) {
        return isSuffix(key, '__added');
    }
    function isSuffix(key, suffix) {
        const index = key.indexOf(suffix);
        return index === -1 ? undefined : key.substr(0, index);
    }
    function suffixKeys(suffix, xs) {
        const ret = {};
        for (const [key, value] of Object.entries(xs)) {
            ret[key + suffix] = value;
        }
        return ret;
    }
    function describeChanges(namespace, prefix, update) {
        const changes = new Array();
        const added = isAdded(prefix);
        if (added) {
            changes.push(`* ${namespace} ${added} (__added__)`);
            return changes;
        }
        const deleted = isDeleted(prefix);
        if (deleted) {
            changes.push(`* ${namespace} ${deleted} (__deleted__)`);
            return changes;
        }
        if (typeof (update) !== 'object') {
            throw new Error(`Unexpected change for ${namespace}.${prefix} ${JSON.stringify(update)}`);
        }
        if ('__old' in update && '__new' in update) {
            changes.push(`* ${namespace} ${prefix} (__changed__)`);
            changes.push(`  * Old: ${update.__old}`);
            changes.push(`  * New: ${update.__new}`);
            return changes;
        }
        if (Array.isArray(update)) {
            changes.push(`* ${namespace} ${prefix} (__changed__)`);
            for (const entry of update) {
                if (entry.length !== 2) {
                    throw new Error(`Unexpected array diff entry: ${JSON.stringify(entry)}`);
                }
                switch (entry[0]) {
                    case '+':
                        changes.push(`  * Added ${entry[1]}`);
                        break;
                    case '-':
                        throw new Error(`Something awkward happened: ${entry[1]} was deleted from ${namespace} ${prefix}!`);
                    case ' ':
                        // This entry is "context"
                        break;
                    default:
                        throw new Error(`Unexpected array diff entry: ${JSON.stringify(entry)}`);
                }
            }
        }
        else {
            for (const key of Object.keys(update)) {
                for (const change of describeChanges(namespace, `${prefix}.${key}`, update[key])) {
                    changes.push(change);
                }
            }
        }
        return changes;
    }
}
main().catch(e => {
    process.stderr.write(e.stack);
    process.stderr.write('\n');
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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