# Copyright 2024 Adam McArthur
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import numpy as np
import pytest
import trimesh

from radstract.analysis.shapedistro.models import (
    COMPARISON_BINS,
    ShapeDistroModels,
    calculate_a3,
    calculate_d2,
    generate_distribution,
)


@pytest.fixture
def sample_trimesh():
    # Create a sample trimesh object for testing
    vertices = np.array([[0, 0, 0], [1, 0, 0], [0, 1, 0], [0, 0, 1]])
    faces = np.array([[0, 1, 2], [0, 1, 3], [0, 2, 3], [1, 2, 3]])
    return trimesh.Trimesh(vertices=vertices, faces=faces)


def test_calculate_a3(sample_trimesh):
    angles = calculate_a3(sample_trimesh)
    assert isinstance(angles, np.ndarray)
    assert angles.ndim == 1
    assert angles.dtype == np.float64
    assert len(angles) <= 1000
    assert np.all(angles >= 0)
    assert np.all(angles <= 180)


def test_calculate_d2(sample_trimesh):
    distances = calculate_d2(sample_trimesh)
    assert isinstance(distances, np.ndarray)
    assert distances.ndim == 1
    assert distances.dtype == np.float64
    assert len(distances) == 1000
    assert np.all(distances >= 0)


def test_generate_distribution(sample_trimesh):
    for model in ShapeDistroModels.ALL:
        bin_centers, hist = generate_distribution(sample_trimesh, model)
        assert isinstance(bin_centers, np.ndarray)
        assert isinstance(hist, np.ndarray)
        assert bin_centers.ndim == 1
        assert hist.ndim == 1
        assert len(bin_centers) == COMPARISON_BINS
        assert len(hist) == COMPARISON_BINS
        assert np.all(hist >= 0)

        # write code to load this file, and test that the values are the same
        # as the ones generated by the function
        npzfile = np.load(f"tests/test_data/{model}.npz")
        assert np.allclose(bin_centers, npzfile["bin_centers"])
        assert np.allclose(hist, npzfile["hist"])


def test_invalid_model(sample_trimesh):
    with pytest.raises(ValueError):
        generate_distribution(sample_trimesh, model="invalid")
