# Copyright 2024 Adam McArthur
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Integration tests for DICOM PDF encapsulation using ReportGenerator.
Tests the complete end-to-end workflow from PDF generation to DICOM encapsulation
and extraction with file similarity validation.
"""

import os
import tempfile
import time

import pydicom
import pytest

from radstract.testdata import Cases, download_case
from radstract.visuals.report_generator import ReportGenerator


class TestReportGeneratorDicomIntegration:
    """Integration tests for DICOM PDF encapsulation workflow."""

    @pytest.fixture
    def sample_dicom_file(self):
        """Download a sample DICOM file for testing."""
        dcm_file, _ = download_case(Cases.ULTRASOUND_DICOM)
        return dcm_file

    @pytest.fixture
    def report_generator(self):
        """Create a ReportGenerator with sample content."""
        report_gen = ReportGenerator(
            title="Integration Test Report",
            accent_color="#00bbff",
            footer_text="Generated by RADSTRACT Tests",
            footer_email="test@radstract.com",
        )

        # Add comprehensive content to make the test meaningful
        report_gen.add_subtitle("Test Summary", level=2)
        report_gen.add_paragraph(
            "This integration test validates the complete DICOM PDF "
            "encapsulation workflow including generation, encapsulation, "
            "extraction, and validation."
        )

        # Add highlights
        report_gen.add_highlights(
            report_success=True,
            highlight1="Integration Test",
            highlight1_label="Type",
            highlight2="DICOM Encapsulated PDF",
            highlight2_label="Format",
        )

        # Add technical data table
        headers = ["Component", "Status", "Validation"]
        data = [
            ["PDF Generation", "✓ Success", "Content Valid"],
            ["DICOM Encapsulation", "✓ Success", "Standards Compliant"],
            ["Data Extraction", "✓ Success", "Data Integrity"],
        ]
        report_gen.add_table(data, headers)

        # Add technical metadata
        report_gen.add_subtitle("Technical Details", level=2)
        metadata = {
            "SOPClassUID": "1.2.840.10008.5.1.4.1.1.104.1",
            "Modality": "DOC",
            "MIMEType": "application/pdf",
            "TestFramework": "pytest",
        }
        report_gen.add_json(metadata, title="Test Metadata")

        return report_gen

    def test_end_to_end_dicom_pdf_workflow(
        self, report_generator, sample_dicom_file
    ):
        """Test complete DICOM PDF workflow with file similarity validation."""
        with tempfile.TemporaryDirectory() as temp_dir:
            # File paths
            pdf_path = os.path.join(temp_dir, "test_report.pdf")
            dicom_path = os.path.join(temp_dir, "test_report.dcm")
            extracted_pdf_path = os.path.join(temp_dir, "extracted_report.pdf")

            # Step 1: Generate regular PDF
            pdf_success = report_generator.save_pdf(pdf_path)
            assert pdf_success, "PDF generation should succeed"
            assert os.path.exists(pdf_path), "PDF file should be created"
            assert (
                os.path.getsize(pdf_path) > 0
            ), "PDF file should not be empty"

            # Step 2: Generate DICOM with encapsulated PDF
            dicom_success = report_generator.save_to_dicom_study(
                output_path=dicom_path,
                dicom_tags=pydicom.dcmread(sample_dicom_file),
            )
            assert dicom_success, "DICOM generation should succeed"
            assert os.path.exists(dicom_path), "DICOM file should be created"
            assert (
                os.path.getsize(dicom_path) > 0
            ), "DICOM file should not be empty"

            # Step 3: Validate DICOM structure
            ds = pydicom.dcmread(dicom_path)
            assert hasattr(
                ds, "EncapsulatedDocument"
            ), "DICOM should contain encapsulated document"
            assert (
                ds.MIMETypeOfEncapsulatedDocument == "application/pdf"
            ), "MIME type should be PDF"
            assert (
                ds.DocumentTitle == "Integration Test Report"
            ), "Document title should match"

            # Step 4: Extract PDF from DICOM
            pdf_data = ds.EncapsulatedDocument
            assert len(pdf_data) > 0, "Extracted PDF data should not be empty"
            assert pdf_data.startswith(
                b"%PDF"
            ), "Extracted data should be valid PDF"

            # Save extracted PDF for comparison
            with open(extracted_pdf_path, "wb") as f:
                f.write(pdf_data)

            # Step 5: Load original PDF for comparison
            with open(pdf_path, "rb") as f:
                original_pdf = f.read()

            # Step 6: Perform comprehensive file similarity analysis
            self._validate_file_similarity(original_pdf, pdf_data)

    def _calculate_similarity(
        self, original_pdf: bytes, extracted_pdf: bytes
    ) -> float:
        """Calculate byte-by-byte similarity percentage between two PDF files."""
        if len(original_pdf) == 0:
            return 0.0 if len(extracted_pdf) > 0 else 100.0

        matching_bytes = sum(
            1 for a, b in zip(original_pdf, extracted_pdf) if a == b
        )
        return (matching_bytes / len(original_pdf)) * 100

    def _validate_file_similarity(
        self, original_pdf: bytes, extracted_pdf: bytes
    ):
        """Validate file similarity with detailed analysis and assertions."""
        # Basic size validation
        assert len(extracted_pdf) > 0, "Extracted PDF should not be empty"

        # Check they are the same size within 100 bytes
        assert abs(len(original_pdf) - len(extracted_pdf)) <= 10, "Files should be the same size within 100 bytes"

