# Copyright 2024 Adam McArthur
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Example demonstrating how to create a DICOM Encapsulated PDF using the ReportGenerator.
"""

import pydicom

from radstract.testdata import Cases, download_case
from radstract.visuals.report_generator import ReportGenerator

dcm_file, _ = download_case(Cases.ULTRASOUND_DICOM)


def main():
    # Create a report generator
    report_gen = ReportGenerator(
        title="Medical Analysis Report",
        accent_color="#00bbff",
        footer_text="Generated by RADSTRACT",
        footer_email="support@radstract.com",
    )

    # Add content to the report
    report_gen.add_subtitle("Executive Summary", level=2)
    report_gen.add_paragraph(
        "This report demonstrates the capability to encapsulate PDF documents "
        "within DICOM files using the DICOM Encapsulated PDF IOD."
    )

    # Add highlights
    report_gen.add_highlights(
        report_success=True,
        highlight1="DICOM Compliant",
        highlight1_label="Standard",
        highlight2="PDF Encapsulated",
        highlight2_label="Format",
    )

    # Add a table with technical data
    headers = ["Parameter", "Value", "Status"]
    data = [
        ["File Format", "DICOM Encapsulated PDF", "✓ Valid"],
        ["SOP Class UID", "1.2.840.10008.5.1.4.1.1.104.1", "✓ Standard"],
        ["MIME Type", "application/pdf", "✓ Correct"],
    ]
    report_gen.add_table(data, headers)

    # Add technical details
    report_gen.add_subtitle("DICOM Structure", level=2)
    dicom_structure = {
        "SOPClassUID": "1.2.840.10008.5.1.4.1.1.104.1",
        "Modality": "DOC",
        "MIMETypeOfEncapsulatedDocument": "application/pdf",
        "DocumentTitle": "Medical Analysis Report",
    }
    report_gen.add_json(dicom_structure, title="DICOM Metadata")

    dcm = pydicom.dcmread(dcm_file)

    # Save as DICOM Encapsulated PDF
    dicom_success = report_gen.save_to_dicom_study(
        output_path="debug/medical_report.dcm",
        dicom_tags=dcm,
    )

    # save the dcm locally for testing
    dcm.save_as("debug/medical_report_original.dcm")

    if dicom_success:
        print("✅ DICOM Encapsulated PDF created successfully!")
        print(f"📄 Regular PDF: debug/medical_report.pdf")
        print(f"🏥 DICOM file: debug/medical_report.dcm")
    else:
        print("❌ Failed to create DICOM Encapsulated PDF")


if __name__ == "__main__":
    main()
