"""
Module containing all the support method of the package.
"""
import io
import pkgutil
import re

import numpy as np
from nltk.stem.snowball import SnowballStemmer
from stop_words import get_stop_words
from textblob import Word

__other_words = {
    "it": [
        "gr",
        "g",
        "freschi",
        "fresche",
        "fresco",
        "fresca",
        "navel",
        "rubis",
        "uht",
        "galbusera",
        "igp",
        "dalfour",
        "vasetto",
        "barattolo",
        "vetro",
        "biologico",
        "biologica",
        "allevate",
        "terra",
        "italia",
        "cat",
        "sardegna",
        "it",
        "granarolo",
        "accadì",
        "william's",
        "artigianale",
        "d'italia",
        "az",
        "luganica",
        "agricola",
        "verga",
        "ml",
        "vaschetta",
        "gusti",
        "gusto",
        "spalmabile",
        "molle",
        "simile",
        "cavalieri",
        "classico",
        "circa",
        "xg",
        "bio",
        "contiene",
        "o'henry",
        "large",
        "medium",
        "small",
        "orange",
        "ferrero",
        "italiano",
        "pronto",
        "consumo",
        "milano",
        "pettinicchio",
        "wagyu",
        "campana",
        "palieri",
        "trentino",
        "st",
        "rapsodia",
        "petti",
        "parzialmente",
        "scremato",
        "cucina",
        "essicato",
        "siciliano",
        "dop",
        "naturale",
        "regina",
        "aperta",
        "parma",
        "affettato",
        "montare",
        "adulto",
        "julienne",
        "fili",
        "extra",
        "d",
        "krumiro",
        "pezzi",
        "spremere",
        "succo",
        "romano",
        "datterini",
        "sungold",
        "croccante",
        "fatto",
        "fatti",
        "bastoncini",
        "lavati",
        "mondati",
        "vergine",
        "baffo",
        "verde",
        "nero",
        "giallo",
        "azzurro",
        "caldo",
        "rosso",
        "rossa",
        "fettina",
        "panata",
        "asciutta",
        "asciutto",
        "tenero",
        "pronti",
        "consumo",
        "italia",
        "formia",
        "lt",
        "composta",
        "dalfour",
        "origine",
        "cuore",
        "spagna",
        "libano",
        "gran bretagna",
        "small",
        "cubettate",
        "preparato",
        "surgelati",
        "baffo",
        "panatura",
        "panato",
        "punta",
        "anca",
        "stagionato",
        "stagionata",
        "scaglie",
        "affumicata",
        "affumicato",
        "cubettato",
        "tritata",
        "tritato",
        "grattuggiato",
        "galline",
        "aperto",
        "antibiotici",
        "madras",
        "vaccina",
        "mezza",
        "luna",
        "pronta",
        "fette",
        "mix",
        "mascognaz",
        "cubetti",
        "bocconcini",
        "taggiasche",
        "rondelle",
        "tondo",
        "cotti",
        "forno",
        "pelle",
        "pinne",
        "gialle",
        "coltello",
        "eviscerato",
        "d'anca",
        "rosette",
        "fette",
        "altamura",
        "essicato",
        "petit",
        "royal",
        "summer",
        "bianco",
        "cubettata",
        "confettura",
        "%",
        "frutta",
        "mondate",
        "fettine",
        "cubetti",
        "filetto",
        "curry",
        "ciliegini",
        "carnaroli",
        "cotto",
        "grigliate",
        "trebbiano",
        "dry",
        "oltrepo",
        "pavese",
        "pan",
        "gran",
        "brut",
        "oltrepò",
        "metodo",
        "docg",
        "blanc",
        "de",
        "noir",
        "granella",
        "tropicale",
        "vegetale",
        "carnaroli",
        "blau",
        "dell’oltrepò",
        "quaquarini",
        "vigna",
        "poggio",
        "anna",
        "crudo",
        "silano",
        "alta",
        "qualità" "qualita",
        "salsa",
        "pronta",
        "secchi",
        "mare",
        "atlantica",
        "brodo",
        "bue",
        "caprese",
        "carpaccio",
        "provincia",
        "pavia",
        "igt",
        "coscia",
        "straccetti",
        "gialla",
        "tritate",
        "verdi",
        "rossi",
        "gialli",
        "fesa",
        "paillard",
        "tagliati",
        "filetti",
        "greciacroazia",
        "tonda",
        "crema",
        "affettati",
        "ortolana",
        "integrali",
        "integrale",
        "girella",
        "sardi",
        "suino",
        "sesamo",
        "camone",
        "all'olio oliva",
        "marmara",
        "lesso",
        "ciliegine",
        "artigianali",
        "giganti",
        "pepite",
        "mesi",
        "rino",
        "spumante",
        "quarto",
        "veraci",
        "code",
        "argentino",
        "gocce",
        "fusi",
        "tagliate",
        "spiedino",
        "rio",
        "velo",
        "dischetti",
        "arcobaleno",
        "m&m's",
        "precotta",
        "precotto",
        "lesse",
        "trito",
        "sottili",
        "nordico",
        "mezze",
        "estivo",
        "lievito",
        "base",
        "cremor",
        "tartaro",
        "fosfati",
        "semola",
        "grano",
        "duro",
        "lamon",
        "quadrate",
        "bastoncino",
        "lattosio",
        "dopparmigiano",
        "mesi",
        "valtellinesi",
        "feschi",
        "parboiled",
        "rotolini",
        "pz",
        "xgr",
        "xl",
        "scaloppe",
        "quadrati",
        "golden",
        "cruda",
        "bovinosuino",
        "impasto",
        "tortino",
        "puliti",
        "punte",
        "griglia",
        "nazionale",
        "cotta",
        "reale",
        "spicchi",
        "borghella",
        "surgelate",
        "nere",
        "bavarese",
        "petto",
        "grappolo",
        "gelo",
        "sp",
        "filone",
        "corta",
        "mediterranea",
        "bonduelle",
        "proporzioni",
        "variabili",
        "cubettati",
        "vapore",
        "germogli",
        "persico",
        "congelati",
        "misto",
        "fiocchi",
        "grigliare",
        "porzione",
        "almaverde",
        "diamond",
        "fuji",
        "pianta",
        "lavate",
        "sgusciati",
        "devenati",
        "sottofesa",
        "secche",
        "semi",
        "vivace",
        "rio",
        "qualità",
        "lessi",
    ],
    "en": [
        "crushed",
        "crumbles",
        "ground",
        "minced",
        "powder",
        "chopped",
        "sliced",
        "teaspoon",
        "tablespoon",
        "fluid",
        "ounce",
        "gill",
        "cup",
        "pint",
        "quart",
        "gallon",
        "gr",
        "ml",
        "l",
        "dl",
        "pound",
        "mg",
        "g",
        "kg",
        "mm",
        "cm",
        "m",
        "inch",
        "clove",
        "red",
        "bunch",
        "glass",
        "glasses",
        "fresh",
        "chopped",
        "a",
        "red",
        "bunch",
        "and",
        "or",
        "leaf",
        "chilli",
        "large",
        "extra",
        "sprig",
        "ground",
        "handful",
        "free",
        "small",
        "virgin",
        "range",
        "from",
        "dried",
        "sustainable",
        "black",
        "peeled",
        "higher",
        "welfare",
        "seed",
        "for",
        "finely",
        "freshly",
        "sea",
        "quality",
        "white",
        "ripe",
        "few",
        "piece",
        "source",
        "to",
        "organic",
        "flat",
        "smoked",
        "ginger",
        "sliced",
        "green",
        "picked",
        "the",
        "stick",
        "plain",
        "plus",
        "mixed",
        "your",
        "optional",
        "serve",
        "unsalted",
        "baby",
        "fat",
        "ask",
        "natural",
        "skin",
        "roughly",
        "into",
        "such",
        "cut",
        "good",
        "brown",
        "grated",
        "trimmed",
        "yellow",
        "dusting",
        "knob",
        "frozen",
        "on",
        "deseeded",
        "low",
        "runny",
        "balsamic",
        "cooked",
        "streaky",
        "rasher",
        "zest",
        "pin",
        "breadcrumb",
        "halved",
        "grating",
        "stalk",
        "light",
        "tinned",
        "dry",
        "soft",
        "rocket",
        "bone",
        "colour",
        "washed",
        "skinless",
        "leftover",
        "splash",
        "removed",
        "dijon",
        "thick",
        "big",
        "hot",
        "drained",
        "sized",
        "watercress",
        "fishmonger",
        "english",
        "dill",
        "raw",
        "worcestershire",
        "flake",
        "tbsp",
        "leg",
        "pine",
        "wild",
        "if",
        "fine",
        "herb",
        "shoulder",
        "cube",
        "dressing",
        "with",
        "chunk",
        "spice",
        "thumb",
        "garam",
        "new",
        "little",
        "punnet",
        "peppercorn",
        "shelled",
        "other",
        "chopped",
        "taste",
        "can",
        "sauce",
        "water",
        "diced",
        "package",
        "italian",
        "shredded",
        "divided",
        "all",
        "purpose",
        "crushed",
        "juice",
        "more",
        "bell",
        "needed",
        "thinly",
        "boneless",
        "half",
        "cubed",
        "jar",
        "seasoning",
        "extract",
        "sweet",
        "baking",
        "beaten",
        "heavy",
        "seeded",
        "tin",
        "uncooked",
        "crumb",
        "style",
        "thin",
        "coarsely",
        "spring",
        "chili",
        "cornstarch",
        "strip",
        "rinsed",
        "root",
        "quartered",
        "head",
        "softened",
        "container",
        "crumbled",
        "frying",
        "lean",
        "cooking",
        "roasted",
        "warm",
        "whipping",
        "thawed",
        "pitted",
        "sun",
        "kosher",
        "bite",
        "toasted",
        "split",
        "melted",
        "degree",
        "lengthwise",
        "romano",
        "packed",
        "pod",
        "anchovy",
        "rom",
        "prepared",
        "juiced",
        "fluid",
        "floret",
        "room",
        "active",
        "seasoned",
        "mix",
        "deveined",
        "lightly",
        "anise",
        "thai",
        "size",
        "unsweetened",
        "torn",
        "wedge",
        "sour",
        "basmati",
        "marinara",
        "dark",
        "temperature",
        "garnish",
        "bouillon",
        "loaf",
        "shell",
        "reggiano",
        "canola",
        "parmigiano",
        "round",
        "canned",
        "ghee",
        "crust",
        "long",
        "broken",
        "ketchup",
        "bulk",
        "cleaned",
        "condensed",
        "sherry",
        "provolone",
        "cold",
        "soda",
        "cottage",
        "spray",
        "tamarind",
        "pecorino",
        "shortening",
        "part",
        "bottle",
        "sodium",
        "cocoa",
        "grain",
        "french",
        "roast",
        "stem",
        "link",
        "firm",
        "asafoetida",
        "mild",
        "dash",
        "boiling",
        "chopped",
        "skin off",
        "bone out",
        "from sustrainable sources",
        "fillet steak",
        "curry",
        "cherry tomatoes",
        "carnaroli",
        "cooked",
        "grilling",
        "trebbiano",
        "dry",
        "Oltrepo",
        "bunting",
        "Pan",
        "great",
        "brut",
        "beyond",
        "method",
        "doc",
        "blanc",
        "de",
        "noir",
        "grain",
        "tropical",
        "vegetable",
        "carnaroli",
        "blau",
        "from beyond",
        "quaquarini",
        "vineyard",
        "knoll",
        "anna",
        "you",
        "raw",
        "silane",
        "high",
        "quality" "quality",
        "sauce",
        "ready",
        "buckets",
        "sea",
        "atlantic",
        "broth",
        "ox",
        "Caprese",
        "carpaccio",
        "province",
        "pavia",
        "igt",
        "thigh",
        "rags",
        "yellow",
        "chopped",
        "greens",
        "red",
        "Detective stories",
        "rump",
        "paillard",
        "cut yourself",
        "fillets",
        "greciacroatia",
        "round",
        "cream",
        "sliced",
        "vegetable garden",
        "integral",
        "whole wheat",
        "swivel",
        "Sardinians",
        "pig",
        "sesame",
        "camone",
        "with olive oil",
        "marmara",
        "boiled",
        "cherries",
        "artisanal",
        "giants",
        "nuggets",
        "months",
        "rino",
        "bubbly wine",
        "fourth",
        "true",
        "code",
        "Argentine",
        "drops",
        "fused",
        "cut",
        "skewer",
        "rio",
        "veil",
        "floppy disks",
        "Rainbow",
        "m & m's",
        "pre-cooked",
        "pre-cooked",
        "read",
        "chopped",
        "thin",
        "Nordic",
        "mezze",
        "summer",
        "yeast",
        "base",
        "cremor",
        "tartar",
        "phosphates",
        "semolina",
        "grain",
        "hard",
        "lamon",
        "square",
        "stick",
        "lactose",
        "Dopparmigiano",
        "months",
        "Valtellina",
        "feschi",
        "parboiled",
        "rolls",
        "pcs",
        "xgr",
        "xl",
        "scallops",
        "squares",
        "golden",
        "raw",
        "bovinosuino",
        "dough",
        "pie",
        "clean",
        "tips",
        "Grill",
        "national",
        "cooked",
        "real",
        "wedges",
        "borghella",
        "frozen",
        "black",
        "Bavarian",
        "chest",
        "bunch",
        "frost",
        "sp",
        "vein",
        "short",
        "Mediterranean",
        "bonduelle",
        "proportions",
        "variables",
        "diced",
        "vapor",
        "sprouts",
        "perch",
        "frozen",
        "mixed",
        "flakes",
        "to grill",
        "portion",
        "almaverde",
        "diamond",
        "fuji",
        "plant",
        "washed",
        "shelled",
        "devenati",
        "underside",
        "dry",
        "seeds",
        "boiled",
    ],
}


__categories = {
    "it": [
        "gelato",
        "sale",
        "pepe",
        "olio",
        "uova",
        "mozzarella",
        "pollo",
        "farina",
        "latte",
        "zucchero",
        "vino",
        "birra",
        "aceto",
        "colomba",
        "pizza",
        "riso",
        "te",
        "insalata",
        "carne",
        "verdure",
        "cioccolato",
        "funghi",
        "fungo",
    ],
    "en": [
        "ice cream",
        "salt",
        "pepper",
        "oil",
        "eggs",
        "egg",
        "mozzarella",
        "chicken",
        "flour",
        "milk",
        "sugar",
        "wine",
        "beer",
        "winegar",
        "dove",
        "pizza",
        "rice",
        "tea",
        "salad",
        "meat",
        "vegetable",
        "chocolate",
        "mushroom",
    ],
}


__adjectives_pattern = {
    "it": "[a-zA-Z]*ato",
    "en": "[a-zA-Z]*ed",
}


def __get_stop_words(language="en"):
    """
    Get the stop words of a selected language plus a collection of non-ingredient
    words usually associated with food and ingredients.

    :param language: the language of the stopwords.
    :return: a list containing all the stopwords of the language.
    """
    stop_words = get_stop_words(language)
    return stop_words + __other_words[language]


def __strip_numeric(string):
    """
    Remove all the numbers in a string.

    :param string: the string where the numbers must be removed.
    :return: a string without numbers.
    """
    return re.sub(r"\d+", "", string)


def __strip_multiple_whitespaces(string):
    """
    Remove all the multiple spaces in a string.

    :param string: the string where the multiple spaces must be removed.
    :return: a string without multiple spaces.
    """
    return re.sub(" +", " ", string)


def __categorize_words(string, language="en"):
    """
    Categorize a string if it contains one of the words in the
    categories dictionary based on the language. If the string
    doesn't contains any of that words, the string will not be modified.

    :param string: the string that can be categorized.
    :param language: the language of the ingredient and category.
    :return: a categorized string or the same string and a boolean to
    state if the string has been categorized or not.
    """
    category = __categories[language]
    word_list = string.split()
    intersection = list(set(word_list) & set(category))
    return (intersection[0], True) if intersection else (string, False)


def __remove_stopwords(string, language="en"):
    """
    Remove all the stopwords inside the provided string based on
    the provided language.

    :param string: the string where the stopwords must be removed.
    :param language: the language of the ingredient and stopwords.
    :return: a string without stopwords.
    """
    word_list = string.split()
    removed_list = [
        word for word in word_list if word not in __get_stop_words(language)
    ]
    return " ".join(removed_list)


def __remove_punctuation(string):
    """
    Remove all the punctuation symbols and characters in a string.

    :param string: the string where the punctation characters must be removed.
    :return: a string without punctuation.
    """
    return re.sub("[!@#£$.()/-]", "", string)


def __remove_adjectives(string, language="en"):
    """
    Remove all the adjectives in a string: adjectives are words that
    indicate a characteristic of the ingredient.
    (Words like fresh, frozen, ..., end with the same pattern)

    :param string: the string where the adjectives must be removed.
    :param language: the language of the ingredients and adjectives.
    :return: a string without adjectives.
    """
    return re.sub(__adjectives_pattern[language], "", string)


def __stem_word(string, language="en"):
    """
    Produce the stemming of the provided string.
    Stemming is the process of reducing the word to its word stem that
    affixes to suffixes and prefixes or to roots of words known as a lemma.
    Lemmatization only works with english words.

    :param string: the word to be lemmatized.
    :param language: the language of the word.
    :return: the word lemmatized.
    """
    lang = {
        "en": "english",
        "it": "italian",
    }
    stemmer = SnowballStemmer(language=lang[language])
    return stemmer.stem(string)


def process_ingredients(ing, language="en"):
    """
    Process all the ingredients string in order to retrieve only the word
    correspond to the single ingredients, without number, special charachters,
    punctuation, stopwords, adjectives and multiple whitespaces.

    :param ing: the string corresponding to the raw ingredient.
    :param language: the language of the ingredient.
    :return: a string corresponding to the single ingredient.
    """
    if not ing:
        return None
    ing = ing.lower()
    ing = ing.replace('"', "")
    ing, boo = __categorize_words(ing, language)
    if not boo:
        ing = __strip_numeric(ing)
        ing = __remove_punctuation(ing)
        ing = __remove_stopwords(ing, language)
        ing = __remove_adjectives(ing, language)
        ing = __strip_multiple_whitespaces(ing)
    ing = __stem_word(ing, language)
    return ing.strip()


def get_embedding(path):
    """
    Get the dataset of ingredients as a dictionary.

    :return: a dictionary representing the embedding
    """
    embedding_io = io.BytesIO(pkgutil.get_data(__name__, path))
    return np.load(embedding_io, allow_pickle=True).item()
