"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const aws_credentials_1 = require("./aws-credentials");
const pipeline_options_1 = require("./pipeline-options");
const stage_1 = require("./stage");
const github_action_step_1 = require("./steps/github-action-step");
const wave_1 = require("./wave");
const github = require("./workflows-model");
const yaml_file_1 = require("./yaml-file");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        // in order to keep track of if this pipeline has been built so we can
        // catch later calls to addWave() or addStage()
        this.builtGH = false;
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = props.preSynthed ?? false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = props.preBuildSteps ?? [];
        this.postBuildSteps = props.postBuildSteps ?? [];
        this.jobSettings = props.jobSettings;
        this.assemblyArtifactOptions = props.assemblyArtifactOptions ?? new pipeline_options_1.GithubAssemblyArtifactOptions();
        this.awsCredentials = this.getAwsCredentials(props);
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.workflowPath = props.workflowPath ?? '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowFile = new yaml_file_1.YamlFile(this.workflowPath);
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = props.runner ?? github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = props.publishAssetsAuthRegion ?? 'us-west-2';
    }
    /**
     * Parse AWS credential configuration from deprecated properties For backwards compatibility.
     */
    getAwsCredentials(props) {
        if (props.gitHubActionRoleArn) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove githubActionRoleArn)');
            }
            return aws_credentials_1.AwsCredentials.fromOpenIdConnect({
                gitHubActionRoleArn: props.gitHubActionRoleArn,
            });
        }
        if (props.awsCredentials) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove awsCredentials)');
            }
            return aws_credentials_1.AwsCredentials.fromGitHubSecrets({
                accessKeyId: 'AWS_ACCESS_KEY_ID',
                secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
                ...props.awsCredentials,
            });
        }
        return props.awsCreds ?? aws_credentials_1.AwsCredentials.fromGitHubSecrets();
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', options?.jobSettings);
        return stageDeployment;
    }
    /**
     * Add a Wave to the pipeline, for deploying multiple Stages in parallel
     *
     * Use the return object of this method to deploy multiple stages in parallel.
     *
     * Example:
     *
     * ```ts
     * declare const pipeline: GitHubWorkflow; // assign pipeline a value
     *
     * const wave = pipeline.addWave('MyWave');
     * wave.addStage(new MyStage(this, 'Stage1'));
     * wave.addStage(new MyStage(this, 'Stage2'));
     * ```
     */
    addWave(id, options) {
        return this.addGitHubWave(id, options);
    }
    addGitHubWave(id, options) {
        if (this.builtGH) {
            throw new Error("addWave: can't add Waves anymore after buildPipeline() has been called");
        }
        const wave = new wave_1.GitHubWave(id, this, options);
        this.waves.push(wave);
        return wave;
    }
    /**
     * Support adding stages with GitHub options to waves - should ONLY be called internally.
     *
     * Use `pipeline.addWave()` and it'll call this when `wave.addStage()` is called.
     *
     * `pipeline.addStage()` will also call this, since it calls `pipeline.addWave().addStage()`.
     *
     *  @internal
     */
    _addStageFromWave(stage, stageDeployment, options) {
        if (!(stage instanceof stage_1.GitHubStage) && options === undefined) {
            return;
        }
        const ghStage = stage instanceof stage_1.GitHubStage ? stage : undefined;
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', ghStage?.props?.gitHubEnvironment ?? options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', ghStage?.props?.stackCapabilities ?? options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', ghStage?.props?.jobSettings ?? options?.jobSettings);
    }
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        this.builtGH = true;
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        this.workflowFile.update(workflow);
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const contextValue = this.node.tryGetContext('cdk-pipelines-github:diffProtection');
        const diffProtection = contextValue === 'false' ? false : contextValue ?? true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || this.workflowFile.toYaml() !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        this.workflowFile.writeFile();
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Workflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else if (node.data.step instanceof github_action_step_1.GitHubActionStep) {
                    return this.jobForGitHubActionStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
            default:
                // The 'as any' is temporary, until the change upstream rolls out
                throw new Error(`GitHubWorfklow does not support graph nodes of type '${node.data?.type}'. You are probably using a feature this CDK Pipelines implementation does not support.`);
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -ex'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '${ASSET_HASH_NAME}=${assetId}' >> $GITHUB_OUTPUT`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                ...this.renderJobSettingParameters(),
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.awsCredentials.jobPermission(),
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToConfigureAws(this.publishAssetsAuthRegion),
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split('/').pop() ?? '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = this.stackProperties[stack.stackArtifactId]?.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                ...this.renderJobSettingParameters(),
                ...this.stackProperties[stack.stackArtifactId]?.settings,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.awsCredentials.jobPermission(),
                },
                ...this.renderGitHubEnvironment(this.stackProperties[stack.stackArtifactId]?.environment),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(region, assumeRoleArn),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1.2.0',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.awsCredentials.jobPermission(),
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        for (const dep of node.allDeps) {
            if (dep.data?.type === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v3',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v3',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    jobForGitHubActionStep(node, step) {
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.WRITE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                steps: step.jobSteps,
            },
        };
    }
    stepsToConfigureAws(region, assumeRoleArn) {
        return this.awsCredentials.credentialSteps(region, assumeRoleArn);
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v2',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return this.assemblyArtifactOptions.downloadAssemblySteps(CDKOUT_ARTIFACT, targetDir);
    }
    stepsToCheckout() {
        return [
            {
                name: 'Checkout',
                uses: 'actions/checkout@v3',
            },
        ];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return this.assemblyArtifactOptions.uploadAssemblySteps(CDKOUT_ARTIFACT, dir);
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
    renderJobSettingParameters() {
        return this.jobSettings;
    }
    renderGitHubEnvironment(environment) {
        if (!environment) {
            return {};
        }
        if (environment.url === undefined) {
            return { environment: environment.name };
        }
        return { environment };
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "@hojulian/cdk-pipelines-github.GitHubWorkflow", version: "0.0.6" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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