# -*- coding: utf-8 -*-
#
# Copyright The NOMAD Authors.
#
# This file is part of NOMAD. See https://nomad-lab.eu for further info.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

import numpy as np
from nomad.units import ureg
from nomad.metainfo import (
    MSection, Package, Quantity, SubSection, Datetime, Section)
from nomad.datamodel.data import EntryData, UseCaseElnCategory
from nomad.datamodel.results import (BandGap, BandStructureElectronic, ElectronicProperties, Material, OptoelectronicProperties, Properties, Results, SolarCell, Symmetry)


m_package = Package(name='perovskite_database')


def add_band_gap(archive, band_gap):
    '''Adds a band gap value (in eV) with the additional section structure for solar
    cell data.eV=
    '''
    if band_gap is not None:
        band_gap = BandGap(value=np.float64(band_gap) * ureg('eV'))
        band_structure = BandStructureElectronic(band_gap=[band_gap])
        electronic = ElectronicProperties(band_structure_electronic=[band_structure])
        archive.results.properties.electronic = electronic
        props = archive.results.properties.available_properties
        if not props:
            props = []
        props.append('electronic.band_structure_electronic.band_gap')
        archive.results.properties.available_properties = props


def add_solar_cell(archive):
    '''Adds metainfo structure for solar cell data.'''
    if not archive.results:
        archive.results = Results()
    if not archive.results.properties:
        archive.results.properties = Properties()
    if not archive.results.properties.optoelectronic:
        archive.results.properties.optoelectronic = OptoelectronicProperties()
    if not archive.results.properties.optoelectronic.solar_cell:
        archive.results.properties.optoelectronic.solar_cell = SolarCell()
    props = archive.results.properties.available_properties
    if not props:
        props = []
    if 'solar_cell' not in props:
        props.append('solar_cell')
    archive.results.properties.available_properties = props


class Ref(MSection):
    """Information about the source of the data. It describes who curated the data,
     the journal in which the data was published,
     the DOI number of the publication, the lead author and the publication date."""

    m_def = Section(
        a_eln=dict(lane_width='800px'))

    internal_sample_id = Quantity(
        type=str,
        shape=[],
        description="""
    This is your own unique cell identifier. With this text string alone, you should be able to identify this cell in your own internal data management system.
                    """,
        a_eln=dict(component='StringEditQuantity'))

    free_text_comment = Quantity(
        type=str,
        shape=[],
        description="""
    This could be anything given additional description to the cell that is not captured by any other field.
                    """,
        a_eln=dict(component='RichTextEditQuantity'))

    ID = Quantity(
        type=np.dtype(np.int64),
        shape=[],
        description="""
                    """)

    ID_temp = Quantity(
        type=np.dtype(np.int64),
        shape=[],
        description="""
                    """)

    # Add in the future auto filling from metadata.author
    name_of_person_entering_the_data = Quantity(
        type=str,
        shape=[],
        description="""
    Your name.
                    """)

    data_entered_by_author = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if you how enter the data also was involved in making the device or if you are a co-author of the paper where the data is presented.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    DOI_number = Quantity(
        type=str,
        shape=[],
        description="""
    The DOI number referring to the published paper or dataset where the data can be found. If the data is unpublished, enter “Unpublished”
Examples:
10.1021/jp5126624
10.1016/j.electacta.2017.06.032
Unpublished
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[])))

    lead_author = Quantity(
        type=str,
        shape=[],
        description="""
    The surname of the first author. If several authors, end with et al. If the DOI number is given correctly, this will be extracted automatically from www.crossref.org
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[])))

    publication_date = Quantity(
        type=Datetime,
        shape=[],
        description="""
    Publication date. If the DOI number is given correctly, this will be extracted automatically from www.crossref.org
                    """)

    journal = Quantity(
        type=str,
        shape=[],
        description="""
    nan
                    """)

    part_of_initial_dataset = Quantity(
        type=bool,
        shape=[],
        description="""
    nan
                    """)

    original_filename_data_upload = Quantity(
        type=str,
        shape=[],
        description="""
    nan
                    """)

    def normalize(self, archive, logger):
        from nomad.datamodel.datamodel import EntryMetadata
        import requests
        import dateutil.parser

        # Parse journal name, lead author and publication date from crossref
        if self.DOI_number:
            if not self.ID_temp:
                r = requests.get(f'https://api.crossref.org/works/{self.DOI_number}')
                temp_dict = r.json()
                # make sure the doi has the prefix https://doi.org/
                if self.DOI_number.startswith('10.'):
                    self.DOI_number = 'https://doi.org/' + self.DOI_number
                given_name = temp_dict['message']['author'][0]['given']
                familiy_name = temp_dict['message']['author'][0]['family']
                self.journal = temp_dict['message']['container-title'][0]
                self.publication_date = dateutil.parser.parse(temp_dict['message']['created']['date-time'])
                self.lead_author = given_name + ' ' + familiy_name
            if not archive.metadata:
                archive.metadata = EntryMetadata()
            if not archive.metadata.references:
                archive.metadata.references = []
                archive.metadata.references.append(self.DOI_number)
                if self.ID_temp is not None:
                    archive.metadata.references.append('https://doi.org/10.1038/s41560-021-00941-3')
                    archive.metadata.references.append('https://www.perovskitedatabase.com/')
                    archive.metadata.external_db = 'The Perovskite Database Project'
            for i, ref in enumerate(archive.metadata.references):
                if ref.startswith('10.'):
                    archive.metadata.references[i] = 'https://doi.org/' + ref


class Cell(MSection):
    """
    General information about the solar cell. It includes information about the device area,
    the layer stack sequence and the device architecture.
    """

    stack_sequence = Quantity(
        type=str,
        shape=[],
        description="""
    The stack sequence describing the cell. Use the following formatting guidelines
- Start with the substrate to the left and list the materials in each layer of the device
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If two materials, e.g. A and B, are mixed in one layer, list the materials in alphabetic order and separate them with  semicolons, as in (A; B)
- The perovskite layer is stated as “Perovskite”, regardless of composition, mixtures, dimensionality etc. There are plenty of other fields specifically targeting the perovskite.
- If a material is doped, or have an additive, state the pure material here and specify the doping in the columns specifically targeting the doping of those layers.
- There is no sharp well-defined boundary between a when a material is best considered as doped to when it is best considered as a mixture of two materials. When in doubt if your material is doped or a mixture, use the notation that best capture the metaphysical essence of the situation
- Use common abbreviations when possible but spell it out when there is risk for confusion. For consistency, please pay attention to the abbreviation specified under the headline Abbreviations found earlier in this document.
- There are several thousand stack sequences described in the literature. Try to find your one in the list of alternatives in the data template. If it is not there (i.e. you may have done something new) define a new stack sequence according to the instructions.
ExampleBelow are the 16 most common device stacks which represent close to half of all reported devices.
SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au
SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au
SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag
SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag
SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Al
SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag
SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag
SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon
SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon
SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Au
SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Au
SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ag
SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au
SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | Au
SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | Au
SLG | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Ag
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=sorted(['SLG | ITO | PTAA | Perovskite-sc | C60 | BCP | Cu', 'SLG | FTO | Ag-nancubes | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | NiO-c | Perovskite | TPA-3CN | BCP | Al', 'SLG | ITO | NiO-c | NiO-mp | Perovskite | ZnO-mp | ZnO-np | Ag', 'SLG | ITO | C60 | Perovskite | Spiro-MeOTAD | Cu', 'SLG | ITO | Graphene | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V852 | Au', 'SLG | FTO | SnO2-np | PCBM-60 | Perovskite | M109 | Au', 'SLG | ITO | MoO3 | PEDOT:PSS | Perovskite | C60 | Bphen | Ag', 'SLG | FTO | SnO2-np | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Carbon-tape', 'SLG | FTO | TiO2-c | Perovskite | TTz-1 | Au', 'SLG | ITO | Perovskite | C60 | Bphen | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuSCN | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | SWCNTs | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | SnO2-np | Perovskite | YD2-o-C8 | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | ME2 | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BMIM-OTF | ITO', 'SLG | ITO | SnO2-np | Perovskite | Theobromine | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-nc | Perovskite | Spiro-MeOTAD | Au', 'SLG  FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-np | Perovskite | YT3 | MoO3 | Ag', 'SLG | FTO | SnO2-c | Perovskite | EVA; MWCNTs | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PbS-QDs | Perovskite | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | ICBA | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 9,9’-di{6-[3-(2-(4-methylphenyl)vinyl)-9-carbazol9-yl]hexyl}-[3,3’]bicarbazole) | Au', 'SLG | ITO | ZnO-np | ZnO-nanospheres | PEI | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Cu:NiO | Perovskite | PC(70)BM | AZO | Al', 'SLG | FTO | TiO2-c | SiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | MgO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PFN; TT | Perovskite | PCBM-60 | Al', 'SLG | ITO | ZnO-c | Perovskite | P3HT | Au', 'PET | IZO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | WOx | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | ITIC | Au', 'SLG | FTO | SnO2-np | Perovskite | PBDT(T)(2F)T | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H18 | Au', 'SLG | ITO | ZnO-np | SnO2-np | C60-SAM | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Lif | Al', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | PN4N | Al', 'SLG | ITO | NiPcS4 | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Ag', 'PEN | AZO | ZnO-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | CS02 | Au', 'SLG | FTO | TiO2-c | SiO2-np | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-407 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Q198 | Carbon', 'SLG | ITO | PTAA | MoS2 | Perovskite | PCBM-60 | PFN | Al', 'SLG | FTO | SnO2-c | Perovskite | V1061 | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | [BMIM]BF4 | Ag', 'SLG | FTO | NiMgLiO-c | Perovskite | PCBM-60 | BCP | Bi | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTA3 | Au', 'SLG | ITO | CuI; CuSCN | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V859 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MeO-DATPA | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Ca | Al', 'SLG | FTO | WOx | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | pTPD | Perovskite | IPH | Ba | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | IBF-Ep | Ca | Al', 'SLG | ITO | Graphene oxide | PTFTS | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | In2S3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | Perosvkite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; PMMA | Perovskite | PMMA | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | Perovskite | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | SnO2-c | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | Perovskite | X51 | Ag', 'SLG | ITO | ZnO | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | ITO | SnO2-np | Perovskite | CuGaO2-np | CuSCN | Au', 'SLG | FTO | NiO | Perovskite | PCBM-60:PS | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TCP-OC8 | Au', 'PEN | ITO | HfO2 | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | P3HT | Au', 'PES | Ti | Graphene | NiO-c | Perovskite | PCBM-60 | AZO | Ag | AZO', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | PEDOT:PSS | ZnO-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | ITO | PTAA | Perovskite | ICBA | bis-C60 | Ag', 'SLG | PEI | Au | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | NiO-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-np | Perovskte | Spiro-MeOTAD | Au', 'SLG | FTO | MgO | SnO2-np | Ethanol amine | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NP2 | Au', 'SLG | ITO | TiO2-c | Perovskite | C201 | Ag', 'SLG | ITO | ZnO-c | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'PEN | ITO | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | ITO | nTi-MOF | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | C60 | SnO2 | ITO | Ag-grid', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnMgO | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | BCP | TeO2 | Ag', 'NOA63 | PEDOT:PSS | Perovskite | PCBM-60 | GaIn', 'SLG | FTO | ZnO-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | F4-TCNQ | Spiro-MeOTAD | Au', 'PET | ITO | PEDOT:PSS | Perovskite | EVA | PCBM-60 | BCP | Ag', 'PET | Ag-grid | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | NiO | Perovskite | PCBPy | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YKP03 | Ag', 'SLG | AZO | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PbS-QDs | Perovskite | P3HT | Au', 'SLG | ITO | SnO2-c | SnO2-mp | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | Perovskite | P3HT | PEDOT:PSS | ITO', 'SLG | ITO | Graphene oxide; PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Tetracene | Spiro-MeOTAD | Au', 'SLG | ITO | C60 | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TS-CuPc | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | CZ-STA | Au', 'SLG | FTO | TiO2-c | TiO2-nw | CdS | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | D149 | Spiro-MeOTAD | Au', 'b-CNF | IZO | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiO-c | Perovskite | SnO2-c | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | WY-2 | Ag', 'SLG | ITO | ZnO-c | Perovskite | CuSCN | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuPc | Spiro-MeOTAD | Au', 'Cu-nw; GFRHybrimer; ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-np | CPTA | Perovskite | P3CT | Au', 'SLG | FTO | TiO2-c | Perovskite | Th-PDI | Au', 'SLG | ITO | P3CT-K | Perovskite | PCBM-60 | C60 | Al', 'PET | ITO | PTAA | Perovskite | PCBM-60 | C60 | BCP | Cu', 'PEN | FTO | PEI | PCBM-60 | Perovskite | PTAA｜Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | Rhodamine 101 | LiF | Ag', 'SLG | ITO | TiO2-c | Perovskite | TPADPP-2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V1209 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Ag', 'SLG | ITO | SnO2-c | BA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | Mg0.15Zn0.85O-np | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-2,5 | Au', 'SLG | ITO | PEDOT:PSS | Graphene oxide | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | Cu2O | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiO-np | Spiro-MeOTAD | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | rGO | Au', 'SLG | FTO | TiO2-c | Perovskite | Dispiro-OMeTAD | Au', 'SLG | FTO | BiFeO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiPc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DEH | Ag', 'SLG | ITO | m-MTDATA | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | 4-aminobenzoic acid-SAM | Perovskite | P3HT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1,3-disubstituted azulene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoOx | Ag', 'SLG | FTO | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovsite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | CO2CO3 | Perovskite | PolyTPD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | n-dimethyloxamic acid | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | NiO-c | Perovskite | C60; PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PFN | Al', 'SLG | ITO | TiO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | bis-C60 | BCP | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | PEIE | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,6-Ben | Au', 'SLG | ITO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | PDI2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TBC | Au', 'SLG | ITO | PEDOT:PSS | X-OTPD | Perovskite | PCBM-60 | Al', 'SLG | FTO | NiO-c | CuGaO2-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | PTAA | Perovskite | PCBM-60 | Mg | Ag', 'SLG | FTO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | MoO3 | TPTPA | Perovskite | C60 | TmPyPB | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PST1 | Au', 'SLG | FTO | TiO2-c | TiO2-nanocones | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-np | NiO-np | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-nanofibers | Perovskite | PTAA | Au', 'SLG | ITO | WO3-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc-p-ZnPc 1 | Au', 'SLG | ITO | ZnO-np | Perovskite | Au', 'A.R.C. | SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | FAD | Perovskite | PCBM-70 | AgAl', 'SLG | ITO | NiO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | SiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | CsPbCl3-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | F6-TCNNQ; TaTm | Perovskite | C60 | C60; PhIm | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DIB; SeS2 | Au', 'PEN | ITO | SnO2-np | CPTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI-BTH1 | Ag', 'SLG | FTO | TiO2-c | TiBaO-nanosphere | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | BMIMBF4 | Spiro-MeOTAD | Au', 'SLG | ITO | PEIE | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | N-PDI | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | Ag-np; TiO2-c | TiO2-mp | Perovskite | Au', 'SLG | ITO | PEDOT:PSS | Pyrene | Perovskite | PCBM-70 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | BL40 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ag-sheet', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | LiF | Al', 'SLG | FTO | TiO2-c | Heparin-Na | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Al2O3-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2,8-bis-[2,2-bis(4-methoxyphenyl)ethenyl]-5,11-diethyl-5,11-dihidroindolo[3,2-b]carbazole | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | pi-PFE3 | Ag', 'SLG | FTO | TiO2-mp | Perovskite | HfO2 | Acetonitrile; I2; LiI; PMII; Propylene glycol; TBP | Graphite', 'PET | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | C60 | Perovskite | PPyra-TXA | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ag | Alq3', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Rhodamine 101 | Au | Al', 'SLG | FTO | TiO2-mp | Perovskite | Ag', 'SLG | ITO | 2,7-PCzTPA | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-70 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag | ITO', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | AZO-np | Ag', 'SLG | FTO | TiO2-c | Perovskite | PVK | Ag', 'SLG | ITO | PTAA | Perovskite | ITIC-Th | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Au-np | MgO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Perovskite | HATNAS3C7-C3h | Ag', 'SLG | FTO | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TFDIB | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | CuSCN | ITO', 'SLG | FTO | TiO2-nw | TiO2-c | Perovskite | Carbon', 'SLG | FTO | NiO-c | MoOx | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc-DPP-ZnPc | Au', 'SLG | PEDOT:PSS | NiO-c | Perovskite | C60 | bis-C60 | Ag', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS | FTO | SLG', 'SLG | ITO | BTF4 | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | HTM-2 (bifluorenylidene-based) | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | NiO-nanowalls | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | FTO | TiO2-c | Perovskite | TAE3 | Au', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Oleic-acid | P3HT | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | XPP | MoO3 | Ag', 'PET | ITO | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | NiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-c | KCl | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Al2O3-mp | Spiro-MeOTAD | Ag', 'SLG | ITO | CrO3 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60; ICBA | Ca | Al', 'SLG | FTO | TiO2-c | Al2O3-mp; Ag@TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ag', 'SLG | FTO | SnO2-np | Perovskite | Carbon', 'SLG | FTO | NiO-np | Perovskite | PCBM-60 | Au', 'SLG | ITO | PTAA | Perovskite | CMB-vTA | PCBM-60 | AZO | Ag', 'SLG | ITO | CuAlO2 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | PN4N | Ag', 'SLG | ITO | PCBM-70 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-np | M2 | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | Ca | C60 | Perovskite | TPTPA | TPTPA; MoO3 | MoO3 | Ag', 'PEN | ITO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | PCBM-60 | Perovskite | LGC-D013 | MoO3 | Ag', 'SLG | FTO | ZnO-c | CdS | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | POZ10 | Au', 'SLG | ITO | PEDOS:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTBDT | Ag', 'SLG | ITO | SnO2-np | OEABS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cu2ZnSnS4 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TFB | Au', 'SLG | ITO | Nb2O5 | Perovskite | Spiro-MeOTAD | Au', 'PET | Zeocoat | Graphene | P3HT | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | H-PheDOT | Au', 'SLG | PET | Graphene oxide | PTFTS | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | ZnO-c | ITO | MgF2', 'SLG | FTO | TiO2-c | Perovskite | PTZ2 | Au', 'SLG | ITO | MC-43 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | PT-DC | Ag', 'SLG | ITO | IrTiOx-c | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | C60 | LiF | Al', 'SLG | ITO | PTAA | Perovskite | PCB-C8 | AZO | Ag', 'SLG | ITO | NiO-c | NiO-mp | Perovskite | PCBM-60 | BCP | Al', 'HCLaminate | IZO | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | PTAA | Au | Organosilicate', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | MoOx | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | C60 | BCP | Au', 'SLG | FTO | TiO2-c | Perovskite | TPE-W4 | Ag', 'PEN | Graphene | Perovskite | PCBM-60 | Ag', 'SLG | FTO | ZnO-c | PCBM-60 | Perovskite | P3HT | Au', 'SLG | ITO | Graphene oxide | Perovskite | C60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TCP-OH | Au', 'SLG | ITO | ZnO-c | PCBM-60 | Perovskite | P3HT | MoO3 | Ag', 'A.R.C. | SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'PI | PEDOT:PSS | ZnO-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | ITO | ZnS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PII2T8T | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Carbon-nt | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | SP-01 | Au', 'SLG | ITO | Si-OMeTPA | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | SnO2-np | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NH-2,7 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cu', 'CdSeS-QDs | SLG | ITO | NiO-c | Perovskite | ADAHCl | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | CsBr | Perovskite | Carbon', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS | Al', 'SLG | ITO | PTAA | perovkite | PCBM-60 | Bphen | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TiS2-np | Au', 'SLG | FTO | TiO2-c | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PTCBI | Ag | WO3 | PTCBI | Ag', 'SLG | FTO | TiO2-c | Al2O3-nt | Perovskite | Spiro-MeOTAD | Au', 'PET | PEDOT:PSS | Perovskite | PCTDI | Cr | Au', 'SLG | ITO | TiO2-c | NAMF-Cl | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | NiPcS4 | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | PPDT2FBT | Au', 'SLG | ITO | SnO2-c | C60-5b | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | PEDOT:PSS', 'SLG | ITO | VOx | Perovskite | PCBM-60 | C60 | BCP | Al', 'PEI | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | V2Ox | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CS01 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Et2PrNMe-CF3BF3 | Al', 'SLG | FTO | ZnO-np | MgO | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | ZnO-nanoflake | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PFI-PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | PrC60MAI | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | P(NDI2DT-TTCN) | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ag | Au', 'PET | ITO | N-PDI | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2 | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | Perovskite | PCBM-60 | Spiro-MeOTAD | Au', 'PEN | ITO | X1 | Perovskite | PCBM-60 | C3-CBL | Ag', 'SLG | FTO | TiO2-c | Perovskite | Th-OMeTPA | Au', 'SLG | FTO | TiO2-c | ZrO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | 3TPYMB | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2TPA-2-DP | Au', 'SLG | ITO | NiO-np | PTAA | Perovskite | PCBM-60 | ZnO-np | ZnO | ITO | MgF2', 'SLG | ITO | PEDOT:PSS | Perovskite | GDY-Tz-CH2(CH2)16CH3; PCBM-60 | C60 | Al', 'PET | ITO | ZnO-c | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KM07 | MoO3 | Ag', 'SLG | ITO | MC6Cz-TPA | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | NiO | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | PBDTTT-E-T | MoO3 | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Bphen | Cu', 'SLG | FTO | TiO2-c | Perovskite | SWCNTs | Spiro-MeOTAD | Ag', 'SLG | FTO | SnO2-c | Perovskite | PtMePy | Au', 'SLG | ITO | PTAA | Perovskite | Ti | Cu', 'SLG | FTO | TiO2-c | Perovskite | TPDCN | Au', 'SLG | ITO | WO3 | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | SnO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | 2PDI-2S | PEIE | Ag', 'SLG | ITO | SnO2-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MPA-BTI | Perovskite | C60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | Spiro-OEtTAD | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | PrCMA | Ag', 'SLG | FTO | TiO2-mp | SAM1 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | P-R | Au', 'SLG | FTO | TiO2-c | PS:TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | PCBM-70 | Perovskite | benzo[1,2b:4,5b′]-dithiophene | Ag', 'SLG | ITO | Perovskite | PCBM-60 | Ca | Al', 'SLG | FTO | TiO2-nanofibers | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | SWCNTs | PEDOT:PSS | Ni-grid', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | AZO | ITO', 'SLG | FTO | NiO-c | Perovskite | PS | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | DTS | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Q216 | Carbon', 'SLG | ITO | PTAA | PS | Perovskite | PCBM-60 | Bphen | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC02 | Ag', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-c | Perovskite | Au', 'PET | ITO | ZSO-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PTAA | PMMA | Perovskite | PCBM-60 | Bphen | Al', 'SLG | FTO | TiO2-c | PCBA | Perobskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ba | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Cu | Cu2O | Ag', 'SLG | ITO | C60-lactone | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | NiO-np | Carbon', 'SLG | ITO | AZO-np | Perovskite | P3HT | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | M113 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M101 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | C12-carbazole | Ag', 'SLG | FTO | BenMeIM-Cl | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | ZnO-np | Al', 'SLG | ITO | PEDOT:PSS | 5,6,11,12-Tetraphenylnaphthacene | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | V2Ox | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Si-PO-2CN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cabon | FTO | SLG', 'SLG | FTO | NiMgLiO | Perovskite | TiO2-np | Sb', 'SLG | FTO | TiO2-c | TiO2-mp | PbI2 | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | DEME-TFSI | ITO', 'PET | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-TPA | Au | Ag', 'SLG | ITO | PTAA | PFN | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | COPV2 | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Ppz | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP; TZ-4 | Ag', 'SLG | ITO | SnO2-np | CPTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,6-Pyr | Au', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | M117 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-macroporous | Perovskite | Spiro-MeOTAD | Au', "SLG | ITO | SnO2-c | Perovskite | 4,4'-(5,10,11-Trihexyl-10,11-dihydro-5H-thieno[2′,3':4,5]pyrrolo [3,2-g]thieno[3,2-b][1,2,3]triazolo[4,5-e]indole-2,8-diyl)bis(N,N-bis(4- (hexyloxy)phenyl)aniline) | Au", 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SM13 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuMePc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | NaPTH | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | FTO | TiO2-c | SnO2-nanosheets | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-np | Perovskite | B-MWCNTs', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DM | Au', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | BTPA-TCNE | MoO3 | Au', 'SLG | ITO | ZnO-c | ZnO-c | Ba(OH)2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | PBTTTV-h | Au', 'SLG | FTO | TiO2-mp | Perovskite | CuSCN | Carbon | FTO | SLG', 'SLG | ITO | ZnO-np | Perovskite | P3HT | P3HT; PMMA | Ag', 'PET | SWCNTs | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | P3HT; SWCNTs; Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Al2O3-mp | Perovskite | PCBM-60 | Rhodamine 101 | Ag | Au', 'SLG | ITO | ZnCsO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | WO3-nw | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | ZnO-np | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | PTAA | PEDOT:PSS | ITO', 'SLG | ITO | PEDOT:PSS | Perovskite | PTTI-2 | Ag', 'SLG | Ti | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | IZO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-nt', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | PbS | Au', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC-Th | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DMF; I2; PVA; TBAI | Pt | FTO', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | SnO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HS-Ph-CN | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Fluorene-dithiophene | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-70 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | ICBA | Ca | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CBP | Au', 'SLG | FTO | PEI | PCBM-60 | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-np | Perovskite | Au', 'PET | SnO2-c | Au | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | AZO-np | Ag-nw', 'SLG | FTO | PTCBI | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | ZnO | PEIE | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | PANI | Perovskite | PCBM-60 | Au', 'SLG | ITO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | PCP-TPA | Au', 'SLG | FTO | TiO2-c | Perovskite | PTAA | Pd', 'SLG | FTO | CrO2 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTF-1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | SnO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'PET | Au | PEDOT:PSS | NPB | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | diPDI | TiO2 | Al', 'SLG | ITO | BCP | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | PTB7-Th | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | Fullerene-2b | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P(BDTT-tPPD) | Au', 'SLG | FTO | TiO2-mp | Perovskite | HfO2 | Spiro-MeOTAD | Graphite', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | TIPD | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | F-PDI | ZnO-np | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Ag@Au-np', 'SLG | ITO | F6-TCNNQ; TaTm | TaTm | Perovskite | C60 | C60; Phlm | Ag', 'SLG | ITO | SnO2-np | Perovskite | pentaerythritol tetrakis(3-mercaptopropionate) | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | DEA | Perovskite | PCBM-60 | PN4N | Al', 'SLG | ITO | NiO-c | Perovskite | SnO2-c | AZO | Ni | Al | Ni', 'SLG | ITO | PTEBS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTPAANT | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | XDB | MoO3 | Ag', 'SLG | ITO | ZnO-c | ZnO-nw | Perovskite | P3HT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA-2F | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPc‐OTPAtBu | Carbon', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | AZO-np | SnO2-c | Ag-nw', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPBTD-B[BMPDP]2 | Au', 'PEN | ITO | PEIE | PCBM-60 | Perovskite | P3HT | Au', 'PET | IZO | TiO2-c | TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR133 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Au | Pt', 'SLG | ITO | PEDOT:PSS | Perovskite | PDPP3T; PCBM-60 | Al', 'PEN | ITO | MFGO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DMFA-FA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphite | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z30 | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | AgAl-np | LiF | AgAl', 'SLG | ITO | NiO-np | Perovskite | ITIC | BCP | Ag', 'SLG | FTO | TiO2-mp | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | n-CuBuPc | Au', 'MgF2 | SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoOx | ITO | Au | MgF2', 'SLG | ITO | ZnO-np | PCBM-60 | Perovskite | P3HT | Au', 'SLG | FTO | NiO-c | Perovskite | TEACl | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | SrTiO3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | rGO | CuSCN | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPP-F | Au', 'SLG | ITO | Polythiophene | Perovskite | C60 | BCP | Ag', 'PET | ITO | CdS | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP; TZ-1 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | LD22 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | ITIC | PEIE | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Carbon; WO3-np', 'SLG | FTO | SnO2-c | Perovskite | CuHePc | Au', 'SLG | ITO | 3EGO-PPV | Perovskite | PCBM-60 | BCP | cu', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | Bphen | Ag', 'SLG | ITO | PTAA | LiF | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-nc | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | NiO-c | Perovskite | CSCNT@SnO2', 'SLG | FTO | TiO2-mp | Perovskite | NiO | Au', 'SLG | FTO | TiO2-mp | LPP | N719 | Perovskite | I2; KI; Propylene carbonate; Polyethylene glycol | Pt | FTO', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | P3HT | Au', 'SLG | ITO | SnO2-np | Perovskite | Graphene oxide | PTAA | Au', 'SLG | FTO | SnO2-nw | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO-c | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoS2 | Au', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC-Se | LiF | Ag', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZIF-8-40 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBSD:GD | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AgZn | Al', 'SLG | FTO | ZrO2-c | TiO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Ag', 'Ti | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | T102 | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | PCBM-60 | Au', 'SLG | FTO | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | tri-TPA | Au', 'SLG | FTO | 2H-TaS2 | Perovskite | P3HT | Ag', 'SLG | FTO | TiO2-c | ZrO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'PEN | ITO | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | LiF | Ag', 'PET | AZO | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | Ag | SnO2-c', 'SLG | FTO | NiMgLiO-c | Perovskite | PCBM-60 | TiO2-c | Ag', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Cu:NiO-np | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTPD | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | CPTA | Perovskite | PTAA | Au', 'SLG | ITO | TiO2-c | Perovskite | Carbon', 'Au | Perovskite | PCBM-60 | Ag', 'SLG | ITO | V1036 | Perovskite | C60 | BCP | Cu', 'SLG | FTO | ZnO-c | ZnO-nw | Au-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | [BMIM]BF4 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPD-4EtCz | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X26 | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Au', 'SLG | FTO | WO3 | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Ag-Nws | Spiro-MeOTAD | Perovskite | ZnO-np | ITO | SLG', 'SLG | ITO | SnO2-np | MSAPBS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuI | Perovskite | PCBM-60 | PEI | Ag', 'Polyimide | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | SWCNTs | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag | Al', 'SLG | FTO | NiO-c | Perovskite | CdS-np | Au', 'SLG | FTO | TiO2-c | TiO2-nanoballs | SiO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | CeO2-c | Perovskite | CuSCN | Au', 'SLG | ITO | SnO2-np | Perovskite | TTE-1 | Au', 'SLG | FTO | CeOx | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (CH3)3SPbI3 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | CaTiO3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | Cu0.67Cr0.33O2 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PDBS-PDI | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | P3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Carbon black', 'SLG | ITO | TiO2-c | PCBM-60 | WS-C60 | Perovskite | P3HT | MoO3 | Al', 'SLG | FTO | TiO2-c | ZnO-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | r-GO-BH | Perovskite | PCBM-60 | Ag', 'SLG | AZO | PCBM-60 | Perovskite | P3HT | Au', 'SLG | ITO | Graphene oxide | PFNBr | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | BrTPA-PCBM | bis-C60 | Ag', 'SLG | Ni:Au-grid | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'PEN | ITO | WOx | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | P3HT; SWCNTs | PEDOT:PSS | Ni-grid', 'SLG | ITO | br-4C | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M7-BR | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3 | Au', 'SLG | AZO | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nw | CuInS2-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | JY5 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsSnI3-QDs | Carbon', 'PET | AuCl3; Graphene | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Decaphenylcyclopentasilane | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | PCBM-60-np | Perovskite | PDCBT | Ta-Wox | Au', 'SLG | FTO | SnO2-c | 1‐butyl‐3‐methylimidazolium bromide | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ca | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2,7 BCz-OMeTAD | Au', 'SLNOA63 | CPI | Cr | Au-grid | EG-PEDOT:PSS | Ti | SnO2-c | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | CdS | Perovskite | Carbon | Ag', 'SLG | ITO | P3CT | Perovskite | C60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Cu', 'SLG | ITO | SnO2 | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | 2-PA-SAM | Perovskite | Spiro-MeOTAD | Au', 'PET | IWO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Carbon | Perovskite', 'SLG | ITO | PEDOT:PSS | Perovskite | N2200 | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OMeTPA-TPA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PTCBI | BCP | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | AgAl', 'SLG | ITO | NiCo2O4-np | Perovskite | PCBM-70 | Al', 'SLG | FTO | NiO-c | Perovskite | PMMA | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | P3HT | Au', 'SLG | ITO | PTAA | PFN | Perovskite | PCBM-60 | BCP | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | SnO2-c | Ag | SnO2-c', 'SLG | Ni | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO | CuSCN | Au', 'SLG | FTO | TiO2-c | Perovskite | DIQ-C6 | Au', 'SLG | FTO | TiO2-c | TiO2; Al2O3-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Zn2SnO4-np | Perovskite | PTAA | Au', 'SLG | ITO | MoO3 | TaTm | Perovskite | Perovskite | C60 | BCP | Ag', 'SLG | ITO | MoO3 | TPA-3,6-FLTPA-TPA | Perovskite | C60 | BCP | Ag', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | PCDTBT | MoO3 | Au', 'SLG | FTO | ZnO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OAI | DM | Au', 'SLG | FTO | TiO2-c | Perovskite | PPDI | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cu2O | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-nt | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PC(70)BM | AZO | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Al2O3-c | Au', 'SLG | ITO | TiO2-c | TiO2-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Y2 | Au', 'SLG | FTO | TiO2-c | Perovskite | CuPc | Carbon', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Zr(acac)4 | Al', 'SLG | FTO | TiO2-c | C60-SAM | Perovskite | HTM3 | Ag', 'SLG | FTO | Ag-np | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Graphene oxide | Perovskite | C60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | TiN | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPIE | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | (2Z,2′Z)-3,3′- (5,5′-(2,7-dioctyl-1,3,6,8-tetraoxo-1,2,3,6,7,8-hexahydrobenzo [lmn][3,8]phenanthroline-4,9-diyl)bis (thiophene-5,2-diyl))bis(2-(4-(trifluoromethyl)phenyl) acrylonitrile) | Ag', 'SLG | ITO | In2O3-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | KH560 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M6 | Au', "SLG | ITO | TiO2-c | Perovskite | N2,N2,N7,N7-tetrakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2,7-diamine | Au", 'PET | ITO | ZnO-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PDI | Au', 'PET | IZO | PTAA | Perovskite | PCBC6 | BCP | Ag', 'SLG | FTO | TiO2-c | Polystyrene-ns | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS | PEDOT:PSS:PSA', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BMIM-OTF | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MeO-BPZTPA | Au', 'SLG | FTO | TiO2-c | Dimethylbiguanide | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | IDT6CN-4F | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | Spiro-OprTAD | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Ag | Ta2O5', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Al', 'SLG | ITO | 2D-Sb | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TIPD | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiPc | V2O5 | Au', 'SLG | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-c | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Metal', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | C13-FAS | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBB-S-N | PCBM-60 | Al', 'SLG | ITO | SnO2-np | C60 | Perovskite | BTF4 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B74 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | AZO | NiAl -grid | MgF2', 'SLG | ITO | Ca | Perovskite | TAPC | MoO3; TAPC | MoO3 | Ag', 'SLG | ITO | TBDI | Perovskite | IPH | PDINO | Ag', 'SLG | ITO | NiO | BrBA | Perovskite | PCBM-60 | C60 | Ag', 'SLG | ITO | PEDOT:PSS | VOx | Perovskire | PCBM-60 | CIL | Al', 'PET | ITO | NiO-np | Perovskite | PCBM-60 | BCP | Ag', 'PET | IZO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | 5,6,11,12-Tetraphenylnaphthacene | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | rGO:CZTS0.75Se0.25 | Au', 'SLG | FTO | NiO-np | Perovskite | PCBM-60 | Ag', 'SLG | FTO | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TZ2 | Perovskite | C60 | BCP | Ag', 'SLG | ITO | SnO2-c | C3-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NaYF4:Yb:Er-np | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | TPD | Au', 'Willow glass | IZO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | Cd2SnO4 | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S101 | Au', 'PES | FTO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TZ1 | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | PTAA | MoO3 | Ag', 'PET | ITO | VOx | Cu phtalocyanine | Perovskite | C60 | BCP | Ag', 'SLG | FTO | SrTiO3-c | SrTiO3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S5 | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | PEIE | Cu', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | FTO | SLG', 'SLG | FTO | TiO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | OTS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | TiO2-mp | Al2O3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | ZnO-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TQ1d | Au', 'N-Graphene-QDs | SLG | FTO | TiO2-c | Perovskite | PTAA | Au', 'SLG | ITO | SnO2-c | Perovskite | HZ2 | Au', 'SLG | FTO | NiMgLiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HL-2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SWCNTs | Graphene oxide | PMMA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | rGO | Au', 'SLG | FTO | TiO2-c | Perovskite | SP-02 | Au', 'SLG | FTO | PCBM-60 | Perovskite | P3HT | Au', 'SLG | FTO | NiMgLiO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | rGO | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'PET | ITO | NiO-c | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | SnO2-c | Perovskite | ZnMe2Pc | Au', 'SLG | FTO | TiO2-c | Perovskite | P3HT | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; PMMA | Perovskite | PDPP-3T | MoOx | Ag', 'PET | Graphene | ZnO-np | Perovskite | PTAA | PEDOT:PSS | Graphene | PET', 'SLG | FTO | ZnO-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | ITO | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YN3 | Au', 'SLG | ITO | TaTm | Perovskite | C60 | BCP | Ag', 'SLG | FTO | CIGS | Perovskite | PCBM-60 | Zr(Acac) | Ag', 'SLG | ITO | CuI | Cu@CuI-nw | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiO-c | Perovskite | ZnO-np | Al', 'SLG | ITO | PEIE | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X18 | Au', 'SLG | FTO | TiO2-c | TiO2-nanowalls | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Azu-Hex | Au', 'SLG | ITO | NiO-c | Au-np | Perovskite | PCBM-60 | Ag', 'SLG | FTO | SnO2-np | PCBM-60 | Perovskite | M107 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | LiNiO-c | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-mp | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OIPC-Br | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FT73 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Corannulene-derivative | Al', 'PET | ITO | CPTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | ZnO-nanowells | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SYN1 | Au', 'SLG | ITO | SnO2-np | Perovskite | Caffeine | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | TFB | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEDOT | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | [BMMIm]Cl | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | P(NDI2DT-T2) | Al', 'SLG | ITO | SnO2-c | ABA-SAM | Perovskite | Spiro-MeOTAD | Au', 'Polyimide | ITO | ZnO-np | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FH-0 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnMgO | Al', 'SLG | ITO | PEDOT:PSS | Au@poly(4-styrenesulfonate) | Perovskite | PCBM-60 | BCP | Au', 'PEN | Planarization | SiN | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | PTZ-2 | Perovskite | PCBM-60 | Ag', 'SLG | ITO | P3CT-N | Perovskite | IT-4M | s-Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR145 | Au', 'SLG | FTO | ZnO-c | ZnO-nanofibers | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO-np | Carbon', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | PBDTT-SeDPP | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (OctPhO)8ZnPc | Au', 'SLG | FTO | TiO2-c | Perovskite | DCZ-OMeTPA | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AV-Carbon; MAI', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | SnO2-c | Cu | SnO2-c', 'PET | ITO | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | P3CT | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsSnBr2I-QDs | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | SAF-5 | MoO3 | Ag', 'PET | Ag-nw | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | BCP | Ag | SiO2 | ZnS | Ag | ZnS', 'SLG | FTO | SnO2-np | Perovskite | Cu3SbS4-np | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BChl-1 | Ag', 'SLG | ITO | ZnO-c | T2CA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Sym-HTPCH | Au', 'SLG | AZO | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PCBM-60 | Perovskite | PEDOT:PSS | Ag', 'SLG｜FTO | TiO2-np | Perovskite | Tea Catachinrich biomolecules | Ag | FTO', 'SLG | ITO | PB2T-SO | Perovskite | PCBM-60 | ZnO | Al', 'SLG | FTO | TiO2-c | TiO2-nanoflowers | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | PCBM-60 | Au | LiF', 'SLG | ITO | NiO-np | KCl | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | PEDOT:PSS | Ag | PEI | PCBM-60 | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CoPcNO2-COU | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | Zr(acac)4 | Au', 'SLG | ITO | NiO-np | Perovskite | Choline chloride | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Pt', 'SLG | FTO | SnO2-c | Perovskite | P3HT | Ag', 'PET | Graphene | PTAA | PEDOT:PSS | Perovskite | ZnO-np | TETA-Graphene | PET', 'SLG | FTO | TiO2-c | PEO | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | Au-np; Graphene; TFSA | Graphene-QDs | Perovskite | PCBM-60 | Al', 'SLG | ITO | PTAA | PMMA | Perovskite | HPDT | PCBM-60 | Bphen | Al', 'SLG | ITO | SnO2-c | SnO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | Cu12Sb4S13 | Au', 'SLG | FTO | Nb2O5; TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | C60 | Perovskite | PTAA | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | PDMS | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PN4N | Ag', 'SLG | ITO | Au-np; Graphene oxide | Perovskite | PCBM-60 | Ag', 'SLG | ITO | rGO; PMMA | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Al', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | ITO | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Sr2CeO4:Sm | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | BPTI | C60 | BCP | Al', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | PTAA | FTO | SLG', 'SLG | ITO | PEI | PCBM-60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | ITO | PTAA | Perovskite | C60 | BCP | Graphene', 'SLG | ITO | NiO-mp | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | FTO | Perovskite | Graphene | Ag', 'SLG | FTO | TiO2-c | Perovskite | Oleylmine | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | 3-PA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Y1 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag-nw | Dielectric-Mirror', 'SLG | FTO | TiO2-c | 3-OMe-SAM | Perovskite | P3HT | MoO3 | Au', 'SLG | FTO | TiO2-c | ZnO-c | Perovskite | CZTS-np | Au', 'SLG | ITO | SnO2-np | Perovskite | 2‐aminoterephthalic acid | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2 -np | Al', 'SLG | FTO | TiO2-c | Perovskite | TAE4 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc-th-ZnPc | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-c | Ag', 'SLG | FTO | ZnO-np | Perovskite | P3TAA | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-nt; P3HT | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au | Ni', 'SLG | ITO | CuGaO2-np | Perovskite | PCBM-70 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Bis-amide-TTF | Au', 'SLG | FTO | ZnO-c | MgO | TiO2-mp | Perovskite | TPB(2-TPTZ) | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Au | MoO3', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P6 | Au', 'SLG | ITO | ZnO-np | Perovskite | P3OHT | Ag', 'SLG | ITO | NiO-c | Perovskite | 2,6-Py | PCBM-60 | PEI | Ag', 'SLG | ITO | C60 | Perovskite | TAPC | MoO3; TAPC | MoO3 | Ag', 'SLG | ITO | Lignosulfonate; PEDOT:PSS; PDA | Perovskite | PCBM-60 | Al', 'SLG | ITO | PAF-68 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-np | Choline chloride; Glycerol | Perovskite | PCBM-60 | BCP | Ag', 'SLG | Cu-nw | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OMeTPA-DPP | Au', 'SLG | ITO | NiO-c | DEA | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TBP | Au', 'SLG | FTO | TiO2-c | Perovskite | 3,4-spiro | Au', 'SLG | FTO | Mg0.2Zn0.8O-np | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SO10 | Au', 'SLG | ITO | 3F-SAM | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp; ZrO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | ITIC | PEI | Ag', 'SLG | ITO | MoOx | TPTPA | Perovskite | C60 | Al', 'SLG | FTO | ZnO-c | Nb2O5 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | AZO | Ag', 'SLG | FTO | TiO2-c | Perovskite | CZTS | Au', 'SLG | ITO | P3CT-K | Perovskite | PCBM-60 | SnO2-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiCo2O4-np | Au', 'SLG | FTO | SnO2-c | Perovskite | Carbon | CuSCN', 'SLG | ITO | SnO2-c | SnO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO-np | ITO', 'SLG | ITO | PEDOT:PSS | Perovskite | ICTA | C60 | BCP | Al', 'SLG | FTO | TiO2-c | Ba0.8Sr0.2SnO3-np | Perovskite | PTAA | Au', 'SLG | FTO | ZrO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Carbon | Graphite', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | P3HT; SWCNTs | PEDOT:PSS | Ni-grid', 'SLG | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Perovskite | F22 | Ag', 'FPA-PDMS | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuAlO2; CuO | Perovskite | PCBM-70 | AZO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Spiro-MeOTAD | NaYF4-np | Au', 'PET | IZO | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CDTh-EtHex 2 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PYPH | LiF | Al', 'SLG | FTO | TiO2-c | Perovskite | pBBTa‐BDT1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | PCBM-60 | BCP | Perovskite | CuO | Ag', 'SLG | ITO | C60-SAM | Perovskite | PDCBT | MoOx | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60; PDI-DA | BCP | Ag', 'SLG | ITO | Spiro-MeOTAD | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | COPV5 | Au', 'SLG | ITO | ZnO-c | p-amino-benzoic acid-SAM | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | T60P | Au', 'PDMS | PET | Au-grid | PEDOT:PSS | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1d | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PVCz-OMeDPD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Bacteriorhodopsin | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-Me | Au | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Poly(TA) | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TPA-BP-OXD | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | FTO | Nb2O3-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PDPP3T | Au', 'PEN | ITO | NiO-np | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | CeO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Carbon-np; PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeTAD | Ag', 'PET | TCE | ZnO-c | Perovskite | PEDOT:PSS | Ag', 'SLG | ITO | SnO2-c | C60-5d | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | PDIN | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR131 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DFTAB | Au', 'SLG | ITO | P3CT-Na | Perovskite | C60; PDI | BCP | Ag', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | MoOx | ITO', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | FTO | NiO-c | BMIMBF4 | Perovskite | PCBM-60 | BCP | Cr | Au', 'SLG | ITO | ZnO-np | Perovskite | PTAA | Au', 'PI | Cu-grid | Graphene | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | ZnO | Al', 'SLG | FTO | TiO2-c | TiO2-mp | P3HT | Au', 'SLG | FTO | [EMIM]PF6-IL | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H64 | Au', 'SLG | ITO | PTAA | Perovskite | PCB-C4 | AZO | Ag', 'SLG | ITO | PTAA | Perovskite | PCBB-3N-3I | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HA1 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OTPA-ZnPc | Au', 'SLG | ITO | TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Mo | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-nt | PMMA', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Imidazolium iodide | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-Tri | Au', 'SLG | FTO | SnO2-c | 3-(1-pyridinio)-1-propanesulfonate | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-tBuBED | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PDI-Br | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Aminocaproic acid | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-mp | Perovskite | Ta2O5 | Spiro-MeOTAD | Carbon', 'SLG | FTO | ZnO-c | EMIM-PF6 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | PCBA | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'A.R.C. | SLG | ITO | 2PACz | Perovskite | C60 | BCP | Cu', 'SLG | ITO | NiO | Perovskite | C60 | bis-C60 | Ag', 'SLG | ITO | NiO-c | EPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PcM-Cou | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (1,6-di{3-[2-(4- methylphenyl)vinyl]carbazol-9-yl}hexane | Au', 'SLG | AZO | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | Al', 'SLG | ITO | NiMgO | PVP | Perovskite | PCBM-60 | ZnMgO | Al', 'PEN | FTO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI3HU-DTYM2 | BCP | Ag', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | M114 | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | Carbon-nt; Graphene oxide | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuSCN | Carbon-nt', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Carbon | Au', 'SLG | ITO | PFN | C60; PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | CZ-TA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | WY-3 | Ag', 'SLG | ITO | C70 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | PFN-Br | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Polypseudorotaxane | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | P3HT | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PFN-P1 | Ag', 'Foil | AZO | LiF | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | SnO2-c | AZO | Au', 'SLG | FTO | Zn2SnO4 | Zn2SnO4-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | pDPP5T-2 | WOx | Au', 'SLG | ITO | MoO3 | Perovskite | C60 | Ca | Al', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | PbS | Au', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | PTB7 | WOx | Au', 'SLG | FTO | PyCEE | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | 2-OMe-SAM | Perovskite | P3HT | MoO3 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI-PM | Al', 'SLG | FTO | TiO2-c | SiO2-nanocolumns | TiO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | InP-np | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuI | Cr | Pt | FTO', 'SLG | FTO | SnO2-np | Perovskite | EH44 | MoO3 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPc-Bu | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPPS | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | CuInSe2-QDs | Au', 'SLG | FTO | TiO2-3D | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon; PEMA', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ca | Al', 'SLG | ITO | SFT-TPA | Perovskite | C60 | BCP | Ag', 'PEN | ITO | SnO2-np | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PDINO | Ag', 'SLG | FTO | TiO2-c | Perovskite | IEICO; PBDTTT-E-T | MoO3 | Ag', 'SLG | ITO | PEDOT:LS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | CPEPh-Na | Perovskite | PCBM-60 | Al', 'PEN | ITO | TiO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Vox | Perovskite | N2200; PCBM-70; TT | Al', 'PET | ITO | Nb2O5 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S14 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM-M1 | Au', 'SLG | ITO | SnO2-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | CeOx | Ag', 'SLG | FTO | BaSnO3-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-c | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Ph-OMeTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon; MAI', 'PEN | ITO | Black P-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | PCBDAN | Ag', 'SLG | FTO | SnO2-c | SnO2-np | Perovskite | Spiro-MeOTAD | Al2O3-c | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Al', 'PET | Ag-nw; PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | TiO2-c | Perovskite | P3DDT | Ag', 'SLG | ITO | PCDTBT | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | CaTiO3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | EFGnPs-F | Al', 'SLG | ITO | NiO-c | Perovskite | ADAHCl | C60 | BCP | Ag', 'SLG | ITO | CuI | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | SnO2-c | Perovskite | CdTe@MAPbI3-QDs | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTPA-DSQT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FH-3 | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS | Au', 'SLG | ITO | SnO2-np | Perovskite | P3HT | Ag', 'PEN | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Bi2Te3', 'PET | ITO | Graphene | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | PEDOT:PSS | Au', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC-Se | C60 | BCP | Ag', 'SLG | ITO | ZnO-np | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuI | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S,Si‐heteropentacene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC01 | Ag', 'PEN | ITO | TiO2-c | Perovskite | Carbon-nt', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | IDTC4-TPA | Au', 'SLG | ITO | NiO-np | ME1 | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | TPA-TVT-TPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PPP | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | ZnO-np | Al', 'SLG | FTO | TiO2-c | Perovskite | rGO | FTO | SLG', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO | Al', 'SLG | FTO | TiO2-c | Perovskite | CTZS | Au', 'SLG | FTO | TiO2-c | ZIF-8 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | Py-C | Au', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | PCDTBT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BL08 | Au', 'SLG | FTO | TiO2-c | ZrO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | DR3T | Au', 'SLG | FTO | c-OTPD | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | ITO | PTB7-Th; PFN | Perovskite | PCBM-60 | Al', 'SLG | ITO | P3CT-Na | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | PDI2 | Perovskite | PCDTBT | MoO3 | Ag', 'Graphite | Perovskite | CdS | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Pt | Si', 'SLG | ITO | SiTP-OMeTPA | Perovskite | PCBM-60 | Al', 'SLG | ITO | XY1 | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | Perosvkite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-70 | Rhodamine 101 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | MgO | Perovskite | Montmorillonite | Spiro-MeOTAD | Au', 'SLG | ITO | Fullerene-SAM | Perovskite | CuSCN | Ta:Wox-np | Carbon', 'SLG | ITO | PEDOT | Perovskite | PCBM-60 | Al', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | H-Z3 | MoO3 | Ag', 'PEN | ITO | SnO2-QDs | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | BTPA | Perovskite | PCBM-60 | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | M115 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiZnO12-c | Perovskite | Spiro-MeOTAD | Au', 'PET | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Ag-nw | PCBM-60', 'SLG | ITO | VOx | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | MeO-2PACz | Perovskite | C60 | SnO2-c | Cu', 'SLG | FTO | SnO2 | Perovskite | Spiro-MeOTAD | MoOx | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AZ1 | Au', 'SLG | FTO | ZnO-c | Perovskite | CuI | Au', 'INVAR | ITO | SiO2 | ITO | NiO-c | Perovskite | PCBM-60 | IZTO', 'SLG | ITO | PEDOT:PSS | Perovskite | ZnO-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | THY-1 | Au', 'PET | IZO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Trimethylamine oxide | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | NaYF4:Yb:Tm-np | Spiro-MeOTAD | Au', 'PET | Ag-nw | Graphene | ZnO-np | Perovskite | PTAA | Au', 'SLG | ITO | Carbon-QDs; SnO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'PET | ITO | PTAA | Perovskite | Teflon | C60 | BCP | Al', 'SLG | FTO | TiO2-c | Graphene-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | Ag-Nws', 'SLG | ITO | Fullerene-SAM | Perovskite | P3HT | Ta:Wox-np | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuS | Au', 'SLG | ITO | ZnO-c | Perovskite | CuSCN | NPB | Ag', 'PET | Ag-grid | ZnO-c | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZIF-8 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | SnO2-c | AZO-c', 'SLG | ITO | SnO2-np | PCBM-60 | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | ITO | P3CT-Na | Perovskite | IDTCN | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X14 | Au', 'SLG | FTO | NiO-np | Perovskite | C60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2H-MoS2 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-Ca | Au', 'SLG | ITO | TiO2-c | NAMF-Br | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PTAA | Perovskite | C60 | SnS | Ag', 'SLG | ITO | NiO-np | PSS | Perovskite | PSS | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ST1 (4-(4-(bis(4-(4-(dibutylamino)styryl)phenyl)amino)styryl)-N,N-dibutylaniline) | Au', 'SLG | FTO | TiO2-c | Perovskite | DCZ-OMeTAD | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ca', 'SLG | ITO | PolyTPD | Perovskite | PCBM-60 | C60 | BCP | Al', 'ITO | PEN | WOx | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | VOx | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TBASBP | Perovskite | PCBM-60 | Al', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | BCP | Cu', 'SLG | ITO | PMA | TaTm | Perovskite | C60 | BCP | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | TIPD | Ag', 'SLG | FTO | SnO2-c | Perovskite | P-OR | Au', 'SLG | FTO | SnO2-c | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTB7 | Au', 'SLG | FTO | ZnO-np | MgO-EA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | ZnCdS | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | P3HT | Au | Ag', 'SLG | ITO | SnO2-np | Perovskite | Theophylline | Spiro-MeOTAD | Ag', 'SLG | FTO | Graphene | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | ZnO-nw | TiO2-np | Perovskite | P3HT | Ag', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | P3 | MoO3 | Ag', 'SLG | FTO | MoS2 | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | F33 | Ag', 'SLG | ITO | Perovskite | Diketopyrrolopyrrole | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | AZO-np | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | CuMe2Pc | Au', 'SLG | FTO | SnO2-c | Perovskite | 2,7-Bis(4,4′-dimethoxydiphenylamine)-9- (bis(methylsulfanyl)methylene)fluorene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HA2 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BuO-DATPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ph-TPA-6A | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X51 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Trux1 | Au', 'SLG | ITO | SnO2-np | Perovskite | Bp-OMe | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPs-TIPS | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon black; Carbon-nt; Graphite', 'SLG | ITO | SnO2-np | Perovskite | TTE-2 | Au', 'SLG | ITO | C60 | Perovskite | PH3T | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Graphite', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene oxide | PMMA | Au', 'SLG | ITO | PTAA | Perovskite | PCBB-S-N | Al', 'PET | ITO | CzPAF-TPA | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | ITO | TiO2-c | Perovskite | J2 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEH-8 | Au', 'SLG | ITO | ZnO-c | C-PCBSD | Perovskite | PDBT-co-TT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | N2,N2,N12,N12-Tetrakis(4-methoxyphenyl)-9-methyl-9H-naphtho[2,1-c]carbazole-2,12-diamine | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | B2Pin2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ZrO2 | MPTMS-SMA | Ag | MUTAB-SAM | PCBM-60 | Perovskite | PTAA | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X55 | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | COTT-2 | Au', 'SLG | FTO | NiO | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | SnO2-c | Perovskite | PBT | Au', 'SLG | ITO | F6-TCNNQ; TaTm | Perovskite | C60 | C60; Phlm | Ag', 'SLG | FTO | ZnO-c | Al2O3-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H2Pc-1 | Au', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | PEDOT | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | Ag', 'SLG | ITO | NiO-c | Perovskite | Bi2S3 | Au', 'SLG | ITO | CzPAF-TPA | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | SnO2-c | Perovskite | TFAP | Au', 'SLG | ITO | Graphene oxide | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT; PCBM-60 | Ag', 'SLG | ITO | NiO-c | Perovskite | C60 | SnO2-c | ITO | Au', 'SLG | FTO | TiO2-np | Perovskite | NiO-np | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsPbBrI2-np | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1b | Au', 'SLG | FTO | TiO2-c | TiOZn-nanosphere | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2 | Al', 'SLG | FTO | SnO2-np | PCBM-60 | Perovskite | X25 | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | C60 | Al', 'SLG | ITO | NiO-c | NiO-mp | Perovskite | ZnO-np | Ag', 'SLG | ITO | PEIE | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NP-BC | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Zr(acac)4 | PEI | Ag', 'A.R.C. | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'PET | Ag-grid | PEDOT:PSS; PEI | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-nw | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'PET | ITO | Ag-nw | ZnO-c | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PbI2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | cyclopenta[2,1-b; 3,4-b′]dithiophene | Ag', 'SLG | ITO | PTAA | Perovskite | 2PDI-0S | PEIE | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | EtO-DATPA | Au', 'SLG | ZnO0.95Sn0.05O1.05 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPDI | Carbon', 'SLG | ITO | TZ3 | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Triazine-Ph-OMeTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TQ1 | Au', 'SLG | FTO | TiO2; WOx | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | C60 | BCP | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI-ID(RR) | Al', 'SLG | ITO | TiO2-c | Perovskite | PDTSTTz-4 | MoO3 | Ag', 'SLG | ITO | NiO-np | TPI-4MEO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag | Au', 'SLG | ITO | SnO2-np | Perovskite | PEAI | PTAA | Au', 'SLG | ITO | SnO2-np | Perovskite | PDCBT | MoO3 | Ag', 'Regenerated cellulose film | Ag-nw | PEDOT:PSS | Perovskite | PCBM-60 | Au', 'SLG | ITO | NiO-c | NiO-nw | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnPc | Perovskite | C60 | BCP | Al', 'SLG | ITO | NiO-c | Perovskite | C60 | SnO2-c | ITO | Ag', 'SLG | ITO | SnO2-np | Perovskite | BTTI-C6 | Au', 'SLG | FTO | TiO2-c | C60 | Perovskite | P3HT | Au', 'SLG | FTO | Cu2BaSnS4 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-mp | Perovskite | HfO2 | Acetonitrile; I2; LiI; TBP | Graphite', 'SLG | FTO | TiO2-c | C60 | Perovskite | TCTA-BVP | Ag', 'SLG | FTO | TiO2-c | Perovskite | P3HT; Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD; PEDOT | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2-c | Al', 'SLG | ITO | 2-F-br-4C | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | NiO-c | SY4 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOS:PSS | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | bis-C60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60; PCBM-60-DPM-OE | Ag', 'SLG | ITO | SnO2-np | Perovskite | PEAI | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | PolyTPD | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | VO | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | ICBA | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | TDTP | LiF | Al', 'SLG | FTO | NiO | Perovskite | C60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | PBDTT | MoO3 | Ag', 'SLG | FTO | SnO2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-nanosphere | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | NiO | Perovskite | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C70 | BCP | Al', 'SLG | FTO | CdS | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | ZnO-np | Perovskite | PTAA | Au', 'SLG | ITO | Perovskite | Polyimid | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO-np | DPO | ITO', 'SLG | ITO | ZnO-np | NH4Cl | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuGaO2 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | FPI-PEIE | Ag', 'SLG | FTO | Au-np; SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | C60; Phlm | C60 | Perovskite | F6-TCNNQ; TaTm | Ag', 'SLG | ITO | PEI | ICBA | Perovskite | PTAA | MoO3 | Ag', 'SLG | ITO | WS2 | Perovskite | PCBM-60 | BCP | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PMMA | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | IPH | Ba | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S:DIB | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | t-BPTI | PDI-C4 | BCP | Al', 'SLG | FTO | AZO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-np | Perovskite | TiO2-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene oxide; NiO-c | Carbon', 'PET | ZnO-c | Ag | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | MPMIC60 | ZnO-np | Al', 'SLG | FTO | SnO2-c | SnO2-nw | Perovskite | Spiro-MeOTAD | Au', 'PEN | PEDOT:PSS | PEI | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | PEDOT | Au', 'SLG | ITO | SnO2-c | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | SnO2-c | C60 | Perovskite | TRUX-E-T | Au', 'SLG | FTO | TiO2-c | Perovskite | di-TPA | Au', 'SLG | FTO | TiO2-c | Perovskite | pm-TPE-4DPA | Ag', 'PET | ITO | NiO-c | Perovskite | C60 | bis-C60 | Ag', 'SLG | ITO | IBF-Ep | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HBT-ZnPc | Au', 'PET | ITO | PEDOT:PSS | Perovsktie | ZnO-np | Al', 'SLG | ITO | PEG; SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | AgAl', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPc | Carbon', 'SLG | ITO | P3CT-Na | Perovskite | PCBM-60 | C60 | Al', 'SLG | PEDOT:PSS | ITO | Perovskite | PCBM-60 | Al', 'SLG | ITO | FPI-PEIE | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AQ310 | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | Ag', 'SLG | FTO | TiO2-c | CsCl | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR216 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DM1P | Au', 'SLG | FTO | PTAA | PFN | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | C60 | Perovskite | Crosslinked TCTA-BVP | Ag', 'SLG | ITO | SnO2-np | Perovskite | P3HT | Au', 'SLG | ITO | V2Ox | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | TPA-OMeTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnChl | Ag', 'SLG | ITO | MoOx | F4-TCNQ | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Ca | C60 | Perovskite | TAPC | MoO3; TAPC | MoO3 | Ag', 'SLG | FTO | X1 | Perovskite | PCBM-60 | C3-CBL | Ag', 'SLG | FTO | TiO2-c | Perovskite-QDs | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | MoO3 | Perovskite | PCBM-60 | Al', 'SLG | ITO | VOx | Perovskite | C60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PN | Ag', 'SLG | FTO | SnO2-c | Perovskite | RE-ZnBu4Pc | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | XOP | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | HTM3 | Au', 'SLG | FTO | TiO2-c | rGO:TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CZTSe-QDs | Au', 'SLG | ITO | TiO2-c | Perovskite | J1 | Ag', 'SLG | ITO | TAPC | Perovskite | PCBM-60 | BCB | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Graphene | Au', 'SLG | FTO | SnO2-c | Perovskite | ZnBu4Pc | Au', 'SLG | FTO | TiO2-c | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DTBT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuInS | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | Graphene | PEDOT', 'SLG | FTO | TiO2-c | TiO2-mp | Carbon-np:Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au | ITO | Au', 'SLG | FTO | TiO2-bulk | Perovskite | Spiro-MeOTAD | Graphite', 'PET | ITO | PTAA | Perovskite | Fluorosilane | C60 | BCP | Al', 'SLG | ITO | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | PEDOT:PSS | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | B2T | C60 | Al', 'SLG | FTO | SnO2-mp | Perovskite | Spiro-MeOTAD | Au', 'PEN | Graphene | MoO3 | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TbT-1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Q221 | Au', 'PET | ITO | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | Au | MoO3', 'SLG | ITO | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-c | Perovskite | HTB-OMe | Au', 'PET | ITO | P3CT-Na | Perovskite | PCBM-60 | Ag', 'SLG | ITO | BV-FNPD | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | ODA-FeS2-np | Au', 'Ti-foil | TiO2-c | Perovskite | Spiro-MeOTAD | Cu | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; PS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2 | Al', 'SLG | FTO | NiO-c | Perovskite | BCP | Au', 'SLG | ITO | SnO2-np | Perovskite | Zn-ChL | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | NDIF1 | C60 | Ag', 'SLG | ITO | ZnO-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOS:PSS | Perovskite | C60; C70 | Ag', 'Pt | Perovskite | Al', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | C60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | HATNASOC7-Cs | Ag', 'SLG | FTO | CuGaO2-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO | Perovskite | PCBM-60 | ZSO-np | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PTCDI | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PDPP3T; PCBM-60 | Ca | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag | V2O5', 'SLG | FTO | NiO-c | NiO-mp | Perovskite | Ethyl acetate; I2; LiI; TBP; Urea | Pt | FTO | SLG', 'SLG | ITO | CPTA | BACl | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | TiO2-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SWCNTs | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | IDT1 | Au', 'SLG | ITO | P3CT-N | Perovskite | PCBM-60 | Bphen | Ag', "SLG | ITO | SnO2-c | Perovskite | 4,4'-((2-Hexyl-2H-benzo[d][1,2,3]triazole-4,7-diyl)bis(thiophene5,2-diyl))bis(N,N-bis(4-methoxyphenyl)aniline) | Au", 'SLG | ITO | P3CT-Na | Perovskite | PDI | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO', 'SLG | ITO:ATO | TiO2-c | TiO2-mp | Perovskite | Perylene | Au', 'SLG | ITO | PolyTPD | PFN | Perovskite | LiF | C60 | PEIE | SnO2-c | ZTO | Au', 'SLG | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | TiO2 | Al', 'SLG | AZO | Cu-nw | AZO | ZnO-c | PEI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiMgLiO | Perovskite | PCBM-60 | CeOx-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Silane-SAM | Perovskite | Carbon', 'SLG | ITO | CuI | Perovskite | C60 | BCP | Ag', 'MgF2 | Willow glass | IZO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | ITO | P3CT-Na | Perovskite | PCBM-60 | HDAC | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Carbon-QDs | Perovskite | Carbon', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | TZ3 | Ag', 'SLG | FTO | TiO2-c | 1-OMe-SAM | Perovskite | P3HT | MoO3 | Au', 'PEN | ITO | TiO2-c | SnO2-c | Perovskite | Graphene; P3HT | Carbon', 'SLG | FTO | IDIC | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SFX-TPA | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ACE-QA-ACE | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | Al', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au | Ag', 'PEN | ITO | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | FBT-TH4 | CuxO | Au', 'PET | ITO | ZnO-np | Perovskite | P3HT | Ag', 'SLG | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | NiO-c | NiO-np | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Bi2Te3 | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-np | Perovskite | COi8DFIC | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M:ON | Au', 'SLG | FTO | SnO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DEPT-SC | Au', 'SLG | FTO | Mg0.25Zn0.75O-np | Perovskite | P3HT | Au', 'SLG | Graphene | PCBM-60 | Perovskite | PTAA | Au', 'SU-8 | MoO3 | Au | PEDOT:PSS | Perovskite | PCBM-60 | Ca | Ag', 'SLG | ITO | c-TCTA | Perovskite | c-HATNA | bis-C60 | Ag', 'SLG | ITO | V2O5 | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ITO', 'SLG | FTO | SrTiO3 | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | F-graphene; P3HT | Ag', 'SLG | ITO | SnO2-c | NPC60 OH | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | ZrO2-mp | Perovskite | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-derivate11 | Al', 'SLG | ITO | NiO-c | Sn2O3-QDs | Perovskite | PCBM-60 | C60 | Ag', 'SLG | ITO | ZnO-c | ZnO-nw | Perovskite | MEH-PPV | PEDOT:PSS | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | 4Pa-SAM | Perovskite | PTAA | Au', 'SLG | ITO | 3-F-br-4C | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | Phenyltrichlorosilane | PCBM-60 | Ca | Al', 'SLG | ITO | 3,6-2,7-PCzTPA | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | Thiophene | Spiro-MeOTAD | Au', 'SLG | ITO | NiO | Perovskite | PCBM-60 | Ag', 'SLG | Zn0.9Sn0.1O1.10 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Azomethine | Au', 'SLG | ITO | NiO | Perovskite | PCBTh | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | I2; LiI; Methoxyacetonitrile | Pt | FTO', 'Transparent wood | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PTEG-1 | Ag', 'SLG | ITO | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Y3 | Au', 'SLG | ITO | SnO2-c | PCBM-60 | Perovskite | DPPZnP-TSEH; PCBM-60 | BCP | Ag', 'SLG | FTO | NiO-mp | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoOx | IZO | Au', 'SLG | ITO | SnO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | MoS2 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S,N-heteropentacene based HTM 2 | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | TPA3C | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiCo2O4-np | Spiro-MeOTAD | Au', 'SLG | ITO | PB2T-S | Perovskite | PCBM-60 | ZnO | Al', 'SLG | FTO | SnO2-np | PCBM-60 | Perovskite | M108 | Au', 'SLG | ITO | SnO2-np | Perovskite | PTAA; Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-np | Perovskite | Graphene oxide | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-np | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | IZO', 'SLG | Graphene | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | p-PFP-O | PTAA | Perovskite | PCBM-60 | PFN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V1207 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | HOOC-Ph-SH | Perovskite | Pentafluorobenzenethiol | Spiro-MeOTAD | Au', 'SLG | ITO | 4-F-br-4C | Perovskite | C60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | PBDB-T | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CW4 | Au', 'SLG | FTO | TiO2-c | Perovskite | PVP | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | THY-5 | Au', 'Foil | AZO | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | ITO | PEDOT:PSS | Perovskite | P4 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc | Au', 'SLG | FTO | TiO2-c | Perovskite | WO3 | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | F4 | BCP | Mg | Ag', 'SLG | ITO | NiO-c | Perovskite | ZnO-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO | MWCNTs', 'SLG | FTO | ZnO-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | Perovskite | Carbon-nt | Ag', 'SLG | ITO | DBTMT | Perovskite | C60 | BCP | Ag', 'PAA-PEG | Ti | PANI | Ti | Perovskite | PCBM-60 | Ti', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M104 | Ag', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | CuI | Au', 'SLG | ITO | PTAA | Perovskite | ICBA | C60 | BCP | Cu', 'SLG | ITO | PCBDAN | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-QDs | Perovskite | Carbon', 'SLG | ITO | MoO3 | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | H3PW12O4; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TcTa | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag | Ni', 'SLG | ITO | SnO2-np | Perovskite | D101 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | bis‐C60 | ITO', 'SLG | Au-np; Graphene | Graphene-QDs | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | MCA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Carbon black; Graphite | MWCNTs', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CDTh 1 | Au', 'SLG | ITO | Grafted rGO; Polyacrylonitrile | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | ZnO-np | MgO-EA | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-derivate10 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEDOT; Graphene | Au', 'SLG | ITO | B-γ-CsSnI3 | Cs2SnI6 | Au', 'SLG | ITO | Ni | NiMgO | Perovskite | PCBM-60 | ZnMgO | Al', 'SLG | ITO | ZnO-c | Graphene; ZnO-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | COTT-1 | Au', 'SLG | FTO | WOx | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CT2 | MoO3 | Ag', 'SLG | ITO | SnO2-c | Perovskite | HZ1 | Au', 'Ti-wire | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag-nw', 'SLG | ITO | SnO2-c | Perovskite | PEDOT:PSS | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Azu-Me | Au', 'SLG | ITO | PEDOT:PSS | PTPD | Perovskite | PCBM-60 | Bphen | Al', 'SLG | ITO | C60 | Perovskite | FTA1 | MoO3 | Ag', 'Au | Ni | Al | Ti | GaN | Perovskite | Graphene aerogel | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z33 | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | BCP | Cr | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | V1012 | Ag', 'SLG | FTO | ZnSnO4-c | ZnSnO4-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBA | Perovskite | Spiro-OMeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z2 | Au', 'SLG | ITO | PFB | Al2O3-np | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | MoS2; TiO2-np | Perovskite | Au', 'PET | APTES | AuCl3; Graphene | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuI | Au', 'SLG | ITO | PNP-BC | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | P3HT; SWCNTs | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SBFCz2 | Au', 'SLG | FTO | TiO2-c | ZnO-nw | AZO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P:ON | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag | Al', 'SLG | ITO | Ni-acetate | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Perovskite | BTT(DPP)3-C8 | Au', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | CuPc | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | SiW11O39-POM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuCrO2-np | Perovskite | C60 | BCP | Ag', 'SLG | FAZO | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Carbon-nt | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | ZnO-c | MgO | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | MoS2-nanosheets | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | 4-Aminobenzoic acid | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-nt | FTO | SLG', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | ITO', 'SLG | FTO | TiO2-c | PCBA | Perobskite | Hexakis[4-(N,N-di-p-methoxyphenylamino)phenyl]benzene | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | Graphite', 'SLG | ITO | NiO-np | Perovskite | ZnO | C60 | Ag', "SLG | ITO | TiO2-c | Perovskite | (N2,N2,N2',N2',N7,N7,N7',N7'-octakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2,2',7,7'-tetraamine) | Au", 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YT1 | Au', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | Al2O3 | Spiro-MeOTAD | Au', 'SLG | ITO | BDT-PTZ | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | Cu0.33Cr0.67O2 | Perovskite | PCBM-60 | Ag', 'Au | Ni | Al | Ti | GaN | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | MgO | TiO2-nw | Perovskite | Spiro-MeOTAD | AgAl', 'SLG | FTO | TiO2-mp | MK2 | Perovskite | Pt | SLG | FTO', 'SLG | FTO | SnO2-np | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | PEIE | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | SnO2-c | Perovskite | TAT-t BuSTy | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | ITO | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | CMB | Ag', 'SLG | ITO | Carbon-QDs; TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiMgLiO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DAI | DM | Au', 'SLG | ITO | MoOx | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | D205 | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Cu', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | ITO | MWCNTs', 'SLG | ITO | 4C | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | WOx | Perovskite | Spiro-MeOTAD | Au', 'PEN | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | NiO-np | PTAA | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | ITO | Ag', 'SLG | ITO | PolyTPD | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | TiO2-mp | Perovskite | 2,5-bis (5-(5-(5-hexylthiophen-2-yl)thiophen-2-yl) furan-2-yl) thiazolo[5,4-d] thiazole | Au', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PyThTPA | Au', 'SLG | ITO | ZnO-np | PCBM-60 | Perovskite | P3HT | PEDOT:PSS | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | THY-2 | Au', 'SLG | ITO | NiO-c | Perovskite | CdZnSeS-QDs | C60 | BCP | Cu', 'SLG | ITO | ZnO-np | C60 | Perovskite | PEDOT:PSS | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z35 | Au', 'SLG | ITO | SnO2-np | Perovskite | IEICO; PBDTTT-E-T | MoO3 | Ag', 'SLG | ITO | ZnO-c | C-PCBSD | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'Ti | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60; ICBA | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-421 | Au', 'SLG | AZO | ZnO-c | ZTO | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60 | Perovskite | Co-porphyrin | Au', 'SLG | FTO | TiO2-c | Perovskite | dly-1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Graphen Oxide | Perovskite | Spiro-MeOTAD | Au', 'Au | ZnO-c | Perovskite | Cu2O | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEDOT | FTO | SLG', 'SLG | ITO | NiO-c | Perovskite | PN-F25 | Ag', 'SLG | FTO | BaSnO3-nw | Perovskite | Spiro-MeOTAD | Carbon', 'SLG | FTO | TiO2-c | ZnPtriazine(gly)2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | BL50 | Au', 'SLG | FTO | PCBM-60 | BCP | Perovskite | NiO-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | JK-216D | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | SnO2-c | Perovskite | WO3 | Spiro-MeOTAD | Au', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | tetra{4-[N,N-(4,4'-dimethoxydiphenylamino)]phenyl}ethene | Au", 'SLG | ITO | NiO-c | Perovskite | HATNAS3C7 | Ag', 'SLG | ITO | PSS-g-PANI:PFI | Perovskite | PCBM-60 | Al', 'SLG | Au | OMeTP-SAM | Perovskite | CITP-SAM | Au', 'SLG | ITO | SnO2-c | Perovskite | Carbon', 'SLG | AZO | ZnO-nw | Perovskite | Carbon', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Carbon | FTO', 'SLG | FTO | NiO-c | Perovskite | MoOx | Au', 'Flexible | IZO | PEIE | C60 | BCP | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Oxo-Graphene | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | LHTM-2 | Au', 'SLG | ITO | SnO2-np | LiF | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-np | PEO; KI; I2 | Pt | FTO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z29 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | SWCNTs', 'SLG | FTO | NiMgLiO | Perovskite | PCBM-60 | TiO2-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PPy | Carbon', 'SLG | FTO | ZnO-np | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SrTiO3 | TiO2-mp | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | Perovskite | PBDT(2F)T | Ag', 'SLG | ITO | P3CT-K | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-3 | Ag', 'SLG | ITO | PEDOT:PSS | PTPADCF3FSONa | Perovskite | PCBM-60 | PN4N | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Asy-PBTBDT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ethyl acetate; I2; LiI; TBP; Urea | Pt | FTO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PCDTBT | Au', 'SLG | ITO | Perovskite | Au', 'SLG | FTO | SnO2-c | Perovskite | CuPc | Carbon', 'SLG | ITO | SnO2-np | Perovskite | CuSCN | MoOx | Au-grid', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | Carbon', 'SLG | FTO | SnO2-c | Perovskite | N-CuMe2Pc | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | SWCNTs | Au', 'SLG | FTO | ZnO-c | ZnO-nanofibers | PEI | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPEDOT-B[BMPDP]2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ATT-ODec | Au', 'SLG | FTO | TiO2-c | Perovskite | EDOT-OMeTPA | Au', 'SLG | FTO | Black P-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | WO3-c | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | SnO2-c | SnO2-np | Perovskite | PTAA | Au', 'SLG | ITO | TiO2-c | Perovskite | CuSCN | Au', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC | Rhodamine 101 | LiF | Ag', 'SLG | ITO | TiO2-nw | Perovskite | Spiro-MeOTAD | Au', 'ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | PCBA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PSe-PDI | Al', 'NaYF4:Eu-np | SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | WO3-c | Cs2CO3 | PCBM-60 | Perovskite | P3HT | Au', 'SLG | ITO | PEDOT:PSS | PSS-Na | Perovskite | PCBM-60 | PEI | Ag-nw', 'SLG | FTO | TiO2-c | Perovskite | PolyTPD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X54 | Au', 'Au | 4-methoxythiophenol | Perovskite | 4-chlorothiophenol | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | C6TBPH2 | Au', 'SLG | FTO | TiO2-c | PCBA | Perovskite | P1 | Au', 'SLG | FTO | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | SiO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Graphene oxide | PEDOT:PSS | Perovskite | PCBM-60 | Carbon-tape', 'SLG | ITO | SnO2-c | TiS2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ag@SnO2-nw | Spiro-MeOTAD | Au', 'SLG | ITO | c-OTPD | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | Zn2SnO4 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3HT | Al2O3-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnO-c | ZnO-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM-2 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskte | PCBM-70 | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60; Zn0.8Cd0.2S-np | Ag', 'SLG | ITO | Spiro-MeO-TPD | PEDOT:PSS | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S197 | Au', 'PET | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DTB | Au', 'SLG | ITO | ZnO-np | Perovskite | P3ODDT | Ag', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | PCA-1 | Au', 'SLG | SMPTMS-SMA | Ag | MUTAB-SAM | PCBM-60 | Perovskite | PTAA | Ag', 'SLG | ITO | OCNR; PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc-p-ZnPc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Azu-EH | Au', 'SLG | FTO | TiO2-c | PCBA | Perovskite | DTB | Au', 'SLG | ITO | NiO-c | Perovskite | BCP; PCBM-60 | Ag', 'Eu(TTA)2(Phen)MAA | SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | NiO | Ag | NiO | NaYF4 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DBT(QT-TPA)2 | Au | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ni-grid', 'SLG | ITO | TiO2-c | Perovskite | PCDTBT | MoO3 | Ag', 'SLG | ITO | SnO2-np | Perovskite | CuInS2-QDs | Au', 'SLG | ITO | Au-np; TiO2-np | C-PCBOD | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nt | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SY1 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | P1C1 | Au', 'SLG | ITO | CdS | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | pTPD | Perovskite | PCBM-60 | Ba | Ag', 'SLG | ITO | TiO2-np | Perovskite | DORDTS–TFBT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AZ2 | Au', 'SLG | FTO | TiO2-c | Perovskite | Cu2ZnSnS4 | Al', 'SLG | ITO | ZnO-c | Perovskite | Ag', 'SLG | ITO | PEDOT:PSS | GeO2 | Perovskite | PCBM-60 | Bphen | Ag', 'PET | ITO | NiO-c | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | PCBM-60; TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | ZnO-nw | Al2O3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | TiO2-mp | BaTiO3-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Zr(acac)4 | PEI | Ag | Ta2O5', 'SLG | ITO | PEDOT:PSS | Perovskite | LiF | Au | Ag', 'SLG | FTO | TiO2-c | Perovskite | PEG | Spiro-MeOTAD | Ag', 'SLG | ITO | Perovskite | C60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | NH4I | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PANI:PSS | Perovskite | PCBM-60 | Au', 'SLG | ITO | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Au | MoO3', 'SLG | ITO | NiO | Perovskite | PCBM-60 | CD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-TPM | Au', 'SLG | ITO | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | TiO2-c | Ag', 'MgF2 | Quartz | ITO | PTAA | Perovskite | PCBM-60 | AZO-np | PEI | Ag', 'SLG | FTO | TiO2-c | Perovskite | 3-hydroxypyridine | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60; MAI | Cu', 'SLG | ITO | PolyTPD | Al2O3-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | NaYF4:Yb:Er-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-IO | Perovskite | PTAA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | Ag', 'SLG | Ag | MoOx | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Cu | Ag | MoO3', 'SLG | ITO | ZnO-c | Perovskite | Au', 'SLG | ITO | LiF | PEDOT:PSS | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | NiMgLiO | Perovskite | 4-acetylpyridine | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | PCA-2 | Au', 'SLG | FTO | TiO2-c | TiO2-nw | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | ZnO-np | p-(F)-PO-TAZ | Perovskite | PB(NAP-Th)TBT | Ag', 'SLG | ITO | P3CT-N | Perovskite | TMTA | PCBM-60 | C60 | TPBi | Cu', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD-F | Au', 'SLG | Ni | Au | Perovskite | C60 | BCP | Al', 'SLG | FTO | TiO2-c | Perovskite | C60 | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Rodhamine 101 | Ag', 'SLG | ITO | HTM-1 (bifluorenylidene-based) | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | NiO-mp | Perovskite | Carbon', 'SLG | ITO | Perovskite | C60 | B4PyMPM | Ag', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | CuMe32c | Au', 'SLG | Zn0.85Sn0.15O1.15 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-hollow spheres; TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | L-alanine | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | CdS-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-nw; Zn2SnO4-np | Perovskite | PTAA | Au', 'SLG | ITO | SnO2-np | B2Cat2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | PN4N | Perovskite | PDCBT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Li-GO | Perovskite | Spiro-MeOTAD | Au', 'Y2O3:Eu3 | Au-np | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | PCBA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2; Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Au@SiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | I2; LiI; Methoxypropionitrile | Pt | FTO', 'SLG | ITO | CuSCN-nw | Perovskite | C60 | Bphen | Ag', 'SLG | ITO | NiO-c | NP-BC | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S,N-Heteroacene 2 | Au', 'SLG | ITO | AZO-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | ITO | SnO2-np | Perovskite | C12H10B2O4 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Yih-1 | Au', 'SLG | ITO | PCBM-60; PCBDAN | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | ZnSe | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT | Perovskite | PCBM-60 | Al', 'SLG | ITO | Perovskite | ICBA | BCP | Ag', 'SLG | ITO | MoS2 | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PVP | Ag', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | BCP | Al', 'SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | BCP | LiF | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60; F8TBT | Ag', 'SLG | ITO | WO3-np | Cs2CO3 | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | WO3-c | WO3-nw | Perovskite | P3HT | Au', 'SLG | ITO | ZnO-c | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-np | Perovskite | CeOx | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Carbon | FTO | SLG', 'PEN | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | PTAA | Au', 'PET | ITO | ZnO-c | Urea | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | PCBM-60 | Au', 'SLG | FTO | ZnO-c | CdSO | Perovskite | Spiro-MeOTAD | Ag', 'PET | ITO | CuPc | PEI | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | m-PYBrZnPor | Ag', 'SLG | ITO | ZnO-np | Perovskite | P3HT | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Zr(acac)4 | Au', 'SLG | FTO | SrSnO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 9,9-bis(3-(dimethylamino)propyl)-N2,N2,N7,N7-tetrakis(4-methoxyphenyl)-9H-fluorene- 2,7-diamine | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PCBDANI | LiF | Al', 'SLG | ITO | SnO2-np | Perovskite | NO2-PEAI | Spiro-MeOTAD | Au', 'SLG | ITO | JW8 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnO-np | PFS-FC | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | CuInS2 | Al2O3-np | Perovskite | TIPD; ZnO-np | Ag', 'SLG | PEDOT:PSS | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEIE | C60 | Perovskite | P3HT | MoOx | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Au', 'SLG | FTO | TiO2-c | Graphene-np; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | CsBr | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | APTES-SAM | Perovskite | PTAA | Au', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | CZ-STA; CZ-TA | Au', 'SLG | ITO | H6 | Perovskite | PCBM-60 | BCP | Au', 'PET | ITO | SnO2-np | Perovskite | PCPD2FBT:BCF | PEDOT:PSS | ITO | PET', 'SLG | FTO | TiO2-c | Graphydine-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | TTB-TTQ | Au', 'SLG | ITO | ZnO-c | WPF‐6‐oxy‐F | Perovskite | Spiro-MeOTAD | Au', 'SLG | PEDOT:PSS | Perovskite | C60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPc-DMP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | C5PcH2 | MoOx | Au', 'SLG | FTO | SnO2-nanospheres | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SrGO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc(tBu)4 | Au', 'SLG | ITO | ZnO | ZnO-nw | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BF002 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FePc-Cou | Au', 'SLG | MSA-PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | C60 | Rhodamine 101 | LiF | Ag', 'SLG | Ag-nw | AZO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | PBDTP-DTDPP | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | C60 | BCP | Al', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Cu-ribbon', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2,2'-[(4,5-Bis(2-ethylhexyl)-dithieno[2,3-d:2',3'-d']thieno[3,2-b:4,5-b']dipyrrole-2,7-diyl)-bis(4,3'-dihexyl-2,2'-bithien-5,5'-diyl)bis(methane-1-yl-1-ylidine)]dimalononitrile | Au", 'SLG | ITO | NiO-c | Perovskite | LiF | C60 | TmPyPB | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | PBDTT-SeDPP; PCBM-70 | MoO3 | Ag', 'SLG | FTO | TiO2-c | AAO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-CH3NH2 | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | P | Au', 'SLG | FTO | NiO-c | SDSB | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | NiO-c | Perovskite | C5-NCMA | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M104 | MoO3 | Ag', 'SLG | FTO | Graphene; TiO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | BMIMBF4 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC-Th | C60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | C60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | pp-TPE-4DPA | Ag', 'SLG | FTO | SnO2-np | TiO2-c | Perovskite | Carbon', 'SLG | ITO | WOx | I-Ph-PA_C60-C6-PA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | SrTiO3-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2 | Perovskite | Spiro-MeOTAD | Au', 'PEN | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | P3CT-Na | Perovskite | NDP-V | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTPA-DBQT | Au', 'SLG | FTO | TiO2-c | Perovskite | 3-Hexylthiophene | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | cyclopenta[2,1-b; 3,4-b′]dithiophene | SLG | ITO', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag-nw', 'SLG | ITO | P3CT-N | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | D103 | Au', 'PET | ITO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Poly-N-vinylcarbazole | SP-12 | Au', 'SLG | ITO | NiO-c | Perovskite | N2200 | bis-C60 | Ag', 'SLG | ITO | PEDOT:PSS | PCDTBT | Perovskite | PCBM-60 | BCP | Al', 'PEN | ITO | nTi-MOF | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2,7-Cbz-EDOT | Au', 'SLG | ITO | M118 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnO-c | JTCA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | Perovskite | PZn-DPPA-O | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZrAcac | Ag', 'SLG | ITO | NiO-c | Perovskite | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | A101 | Au', 'SLG | FTO | TiO2-c | Perovskite | DIP-C6 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'PET | ITO | SnO2-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | X-DVTPD | Perovskite | PC6BM | Bphen | Ag', 'SLG | FTO | PTCA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Al2O3-np | Perovskite | PCBM-60 | Ag', 'PET | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Graphene; SrTiO3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | ZnO-c | Perovskite | P3HT | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C70 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60 | Perovskite | Spiro-MeOTAD | ITO | MgF2', 'SLG | ITO | MoO3 | PTAA | Perovskite | C60 | BCP | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | TiO2-np | Perovskite | P3HT | Ag', 'Ag-nw; GFRHybrimer; ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskte | PCBM-60 | BCP | IZO', 'SLG | FTO | ZnO-c | AZO-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Graphite-nf | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiMgLiO | Perovskite | PCBM-60 | Ag', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoOx | ITO | MgF2', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | CuSCN-3D | Perovskite | C60 | Bphen | Ag', 'SLG | ITO | SnO2-np | ZnO-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamin 101 | Au', 'SLG | ITO | 3,6-PCzTPA | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | C60 | ZnO-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Sb2S3 | Perovskite | CuSCN | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | TMPA-Cl | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | JY6 | Ag', 'SLG | ITO | NiO-c | NiO-mp | Perovskite | PCBM-70 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | N3,N3,N9,N9‐tetrakis(4‐methoxyphenyl)xantheno[2,1,9,8‐klmna]xanthene‐3,9‐diamine | Au', 'SLG | FTO | Al2O3-c | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | C60 | SnO2 | ZTO | IZO', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | NPD | MoO3 | Au', 'SLG | ITO | Carbon-np; SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEI | PCBM-60 | Perovskite | T1 | MoO3 | Ag', 'SLG | ITO | ZnCsO | PCBA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ATT-OHex | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PD-10-DTTE-7 | Spiro-MeOTAD | Au', 'SLG | APTES; Graphene | PCBM-60 | Perovskite | PTAA | Au', 'SLG | ITO | C60 | Perovskite | FBA2 | MoO3 | Ag', 'SLG | ITO | NiO-c | DEA | Perovskite | PCBM-60 | PN4N | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO-np | Au', 'SLG | ITO | PEDOT:PSS-NH2-OH | Perovskite | PCBM-60 | Al', 'SLG | ITO | NiO-c | Perovskite | ZnO | PTFE | Ag', 'ITO | PEN | SnO2-np | Perovskite | PEDOT:PSS | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | N2200 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | G2 | Au', 'Ti-foil | TiO2-nt | Perovskite | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTA2 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Fullerene-2a | Ag', 'SLG | FTO | SnO2-c | Perovskite | CuPc | Au', 'SLG | ITO | PolyTPD | Perovskite | C60 | BCP | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnChl-2 | Ag', 'SLG | FTO | TMAH | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | LiMgNiO-c | Perovskite | PCBM-60 | Carbon-QDs | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M102 | Ag', 'SLG | ITO | NiO-np | Perovskite | C60; PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | C60 | Perovskite | PMMA | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Cu | MoOx', 'SLG | ITO | NO-Graphene-QDs | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | P3HT | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | C60 | Perovskite | SWCNTs | P3HT', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AS37 | Au', 'SLG | ITO | PTAA | Perovskite | C60 | SnO2 | BCP | Cu', 'SLG | FTO | TiO2-c | Perovskite | ZnPc | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-c | Perovskite | Spiro-MeOTAD | Ag | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BT | Au', 'SLG | FTO | TiO2-c | Au@SiO2-np; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Carbon', 'SLG | ITO | Cu2O | Perovskite | PCBM-60 | Ca | Al', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | TL- PEDOT:PSS | FTO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V862 | Au', 'SLG | FTO | PANI | Perovskite | PCBM-60 | Au', 'SLG | ITO | MoO3 | TPTPA | Perovskite | TmPyPB | C60 | TmPyPB | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTDTP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite-IO | Carbon-QDs | Spiro-MeOTAD | Ag', 'SLG | FTO | SnO2-c | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | MBA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SY4 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZTO | SnO2-c | Perovskite | PTAA | Au', 'SLG | ITO | rGO | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | WO3 | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | PCBM-60 | BCP | Ag', 'PET | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Pyridine | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60; PMMA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | POZ3 | Ag', 'SLG | ITO | PEDOT:PSS | PSS-Na | Perovskite | PCBM-60 | Ag-nw', 'SLG | In2O3:H | PTAA | Perovskite | PCBM-60 | ZnO-np | AZO | AlNi-grid', 'SLG | FTO | SnO2-nt | Perovskite | P3HT | Au', 'SLG | ITO | SnO2-np | EPA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | TPA2C | Au', 'SLG | ITO | NiO-np | PMMA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TRUX2 | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CZTS-QDs | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DOPT-SC | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Carbon-nt; TiO2-mp | Perovskite | PTAA | Au', 'SLG | ITO | Spiro-TBB | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | T30P | Au', 'SLG | ITO | PETDOT:PSS | Perovskite | PCBM-60 | AZO-np | Ag', 'SLG | FTO | TiO2-c | TiO2-nw | Al2O3 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | Zn0.8Sn0.2O1.2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | MoO3 | Au | MoO3', 'Ti | TiO2-nw | TiO2-np | Perovskite | Carbon-nt | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Carbon-mp | Perovskite | Carbon', 'PEN | ITO | SnS2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Bis-FITG | Ag', 'SLG | ITO | MoOx | PEDOT:PSS | Perovskite | PCBM-60 | TOPD | Ag', 'SLG | FTO | Ag:CuO-nanofibers | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PTAA | Perovskite | C60; C70 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | Polystyrene | PCBM-60 | BCP | Ag', 'SLG | ITO | TP-FTzF-TP | Perovskite | PCBM-60 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | MoS2 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BDT-4MeOTPA | Au', 'SLG | ITO | PEDOT:PSS | PEG | Perovskite | PCBM-60 | Ag', 'PET | ITO | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | Ti | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-derivate12 | Al', 'SLG | FTO | TiO2-mp | Perovskite | PTAA | Au', 'SLG | ITO | P3CT-N | Perovskite | C60 | BCP | Ag', 'SLG | FTO | Al:ITO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | NiPc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ODA-FeS2-np | Au', 'SLG | ITO | H2 | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | NiO-np | TPI-2MEO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | V1050 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1-Donecyl Mercaptan | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | SiO2-np; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuSCN | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | dly-2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Cu', 'SLG | ITO | CdS | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | HMe2Pc | Au', 'SLG | FTO | TiO2-c | Al2O3-nt | Perovskite | Spiro-MeOTAD | MoOx | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC-1 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BT-BTH | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Q197 | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | [BMPA-EDOT]3-TPA | Au', 'SLG | FTO | TiO2-c | Au-np | Perovskite | P3HT | Ag', 'SLG | ITO | P3HT | Perovskite | C60 | BCP | Cu', 'SLG | ITO | TiO2-c | Perovskite | PTAA | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | COPV3 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Cu', 'SLG | FTO | SnO2-c | Perovskite | Mo(tfd-COCF3)3 | SWCNTs | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | In2O3-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PTAA | Al2O3-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | Choline chloride | Spiro-MeOTAD | SWCNTs | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Sinapoyl malate | Perovskite | ZnP | Au', 'SLG | DWCNTs | PTAA | PFN-P2 | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | Perovskite | PbS-QDs | Au', 'SLG | AZO | Au | AZO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | Perovskite | CIGS-np | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DAHI | Spiro-MeOTAD | Au', 'SLG | Zn0.75Sn0.25O1.25 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | GABAHI | Perovskite | Spiro-MeOTAD | Ag | Au', 'SLG | FTO | SnO2-c | Perovskite | PdMePy | Au', 'SLG | FTO | TiO2-c | Perovskite | tris(4-(5-hexylthiophen-2-yl)phenyl)amine | MoOx | Ag', 'SLG | FTO | ZnO-np | Graphene | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTPD | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | H3 | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite-IO | Spiro-MeOTAD | Ag', 'SLG | FTO | SnO2-np; TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Cr2O3:Cr', 'SLG | FTO | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | C60 | TmPyPB | Ag', 'SLG | FTO | PCBM-60 | Carbon | Perovskite | PEDOT:PSS | Ag', 'SLG | ITO | CuI | Perovskite | ICBA | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | In2O3 | Perovskite | P3HT | Au', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | (3-Aminopropyl)trimethoxysilane | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | SnO2-np | BSO-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | VOx | X-DVTPD | Perovskite | PCBM-60 | FPI | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | FTO | TiO2-c | Perovskite | P3HT | MoO3 | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | ZnO-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1,3,6,8-tetrakis-(N,N-di-p-methoxyphenylamine)pyrene | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PPy | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | NiO-c | Carbon', 'SLG | ITO | MoO3 | TPTPA | Perovskite | B4PyMPM | C60 | TmPyPB | Ag', 'SLG | ITO | NiO-c | PFN-P2 | Perovskite | C60 | BCP | Ag', 'PEN | ITO | SnO2-np | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | Perovskite | P3OFHT | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Al2O3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HBZ-71 | Carbon | FTO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | THY-3 | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Phen-NaDPO | Ag', 'SLG | ITO | rGO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | NiMgLiO | Perovskite | TiO2-c | Ag', 'SLG | ITO | PVBT-SO3 | Perovskite | PCBM-60 | C60-N | Ag', 'SLG | ITO | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | NiO-mp | Perovskite | Carbon', 'SLG | ITO | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-QDs | Perovskite | Au@CdS | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | PTAA | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS | Au', 'Au | Perovskite | C60 | BCP | Au', 'SLG | Ni | NiO-c | Perovskite | PCBM-60 | PEIE | Cu', 'SLG | FTO | TiO2-c | Perovskite | PCPD2FBT:BCF | PEDOT:PSS | ITO | SLG', 'SLG | FTO | TiO2-c | Al2O3-mp | NiO-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Sinapoyl malate | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | PEI | ICBA | Perovskite | PTAA | MoO3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | SWCNTs | Ag', 'SLG | FTO | TiO2-np | Perovskite | PDCBT | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | V886 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | quart-p-phenylene1 | Au', 'SLG | FTO | TiO2-c | Perovskite | PCBM-60 | Au', 'SLG | ITO | F6-TCNNQ | TaTm | Perovskite | C60 | C60; Phlm | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | DNDIF3 | C60 | Ag', 'SLG | IZO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Al', 'PEN | ITO | C60 | C60-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Perovskite | P3HT | Au', 'SLG | ITO | TiO2-c | Perovskite | PCz | Ag', 'SLG | FTO | TiO2-mp | MgO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | M116 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | Ni | Au | Cu | Perovskite | C60 | BCP | Al', 'SLG | ITO | V1036:C2 | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Acetonitrile; B2; LiBr | Pt | FTO', 'SLG | FTO | ZnO-np | Perovskite | P3HT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YR3 | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiCo2O4 | Perovskite | C60; PCBM-60 | Zr(acac)4 | Ag', 'PET | ITO | FPI-PEIE | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au | Ag', 'PET | Ag-nw; Graphene oxide | PEDOT:PSS | Perovskite | PCBM-60 | PFN-P1 | Ag', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (2Z,2'Z)-2,2'-((10-(2-ethylhexyl)-10H-phenothiazine-3,7-diyl) bis(4,1- phenylene)) bis(3-(4-(diphenylamino) phenyl) acrylonitrile | Au", 'SLG | FTO | CdS-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | F6-TCNNQ; TaTm | TaTm | Perovskite | C60 | C60; PhIm | Ag', 'SLG | ITO | Au@TiO2-np; TiO2-np | Perovskte | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCB-C12 | AZO | Ag', 'SLG | ITO | P3CT-Na | PASP | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEDOT:PSS | FTO | SLG', 'SLG | FTO | CuI | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PPV | Au', 'SLG | ITO | MoO3 | NPB | PbI2 | C60 | BCP | Al', 'SLG | FTO | PEDOT:PSS | TPA-NAP-TPA | Perovskite | PCBM-70 | Ag', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | H-Z2 | MoO3 | Ag', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Au-np; NiO', 'SLG | FTO | PEDOT:PSS | NPB | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | TaTm | F6-TCNNQ; TaTm | Au', 'SLG | ITO | TAE | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnNc | Au', 'SLG | ITO | PTAA | Perovskite | Au', 'ZnSe-QDs | SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEI | PCBM-60 | Perovskite | PTAA | MoO3 | Au', 'PET | ITO | ZnO-c | ZnO-nanospheres | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | TiO2-nw | N-[3-(Trimethoxysilyl)propyl]ethylenediamine-SAM | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V2O5 | Au', 'SLG | ITO | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TbT-3 | Au', 'SLG | FTO | WO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | 1,2 ethanedithio | Perovskite | PZn-2FTPA | Au', 'PEN | ITO | NiO-np | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | FTO | TiO2-c | Perovskite | BTT(DPP-Th)3-EH | Au', 'SLG | ITO | Perovskite | C60 | BCP | Al', 'SLG | FTO | TiO2-c | SAED | Perovskite | ZnPc | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-mp | Perovskite | I2; KI; Propylene carbonate; Polyethylene glycol | Pt | FTO', 'SLG | FTO | TiO2-c | TiO2-mp | N719 | Perovskite | Pt | SLG | FTO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Al2O3 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Fu-OMeTPA | Au', 'SLG | ITO | SnO2-np | Perovskite | PTAA | Ag', 'SLG | ITO | NiO-np | Perovskite | C60; PCBM-60 | Zr(acac)4 | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | rGO-flakes | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTBT-th | Au', 'SLG | FTO | X1 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-411 | Au', 'PET | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'PET | ITO | P3CT-CH3NH2 | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS; PEG | Perovskite | C60 | BCP | Ag', 'SLG | FTO | P1-2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | P3CT-N | Perovskite | PCBM-60 | C60 | TPBi | Cu', 'SLG | FTO | TiO2-c | SnO2-mp | Perovskite | P2 | Au', 'SLG | ITO | Spiro-TAD | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Me-QTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZnO-mp | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | NiO-mp | Perovskite', 'SLG | TiO2-c | SnO2-c | TiO2-c | SnO2-c | TiO2-c | SnO2-c | TiO2-c | SnO2-c | TiO2-c | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | ITO', 'SLG | FTO | NiO-c | Al2O3-mp | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | P3CT-K | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | C60-NH2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnSO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | LiF | Al', 'SLG | FTO | PolyTPD | PFN | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | ZnO-c | ITO | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PEAI | PCBM-60 | BCP | Ag', 'SLG | FTO | C60 | Perovskite | Spiro-MeOTAD | Al', 'PEN | Ag-grid | ZnO-c | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BDTT:DPPD:TT | Au', 'SLG | ITO | V2O5 | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PCPDTBT | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | ZnOS | Perovskite | Carbon | Au', 'SLG | ITO | SnO2-np | Perovskite | Aminothiazolium iodide | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MeO-TPD | Au', 'SLG | ITO | SnO2-np | C60 | Perovskite | Spiro-TTB | VOx | ITO', 'SLG | FTO | TiO2-c | Perovskite | JY7 | Ag', 'PEN | ITO | PCBM-60 | Perovskite | PTAA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | LiF | Al', 'SLG | ITO | TiO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | PFN | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | CdS | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | NiO-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PEAI | PCBM-60 | BCP | SnO2-c | ITO | LiF', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | PCBM-60 | Ba | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SM09 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | MoO3 | Al', 'SLG | ITO | NiO-c | Perovskite | HATNASO2C7-Cs | Ag', 'SLG | ITO | MgO-c | Perovskite | Spiro-MeOTAD | Au', 'PEN | Graphene; MoO3 | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | FTO | TiO2-mp | D35 | Perovskite | Pt | SLG | FTO', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | TiO2 | Al', 'SLG | ITO | SnO2-np | Perovsite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | Carbon | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | ZnO-np | C60 | Perovskite | PEDOT:PSS | FrGO | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | EtheneDTPA | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | TPA4C | Au', 'SLG | ITO | NiO-c | Cysteine | Perovskite | PCBM-60 | Bphen | Al', 'SLG | ITO | PEIE | C60 | Perovskite | P3HT | MoO3 | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | B4PyMPM | Ag', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | SWCNTs | PMMA | Ag', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | IPH | PDINO | Ag', 'SLG | ITO | SnO2-np | Perovskite | 2,7-triphenylamine-carbazole | Au', 'PET | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ca | Al', 'SLG | FTO | TiO2-c | TiO2-mp | ZIF-8 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BzTA | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Carbon-QDs | Perovskite | Red Phosphorous-QDs | Carbon', 'SLG | FTO | TiO2-c | Perovskite | Cu2ZnSn4-np | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | N-Graphene', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | V950 | Ag', 'Ti | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag-np | ITO', 'SLG | FTO | TiO2-c | STHNP | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | NiO | Ag | NiO', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P2 | Au', 'SLG | ITO | PEDOT:PSS | CuI | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TIPS-Pentacene | Au', 'SLG | FTO | TiO2-nt | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | FTO | TiO2-c | SnO2-np | Perovskite | Spiro-MeOTAD | Carbon | FTO', 'SLG | ITO | ZTO | SnO2-c | PCBM-60 | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | IPA', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-4 | Au', 'SLG | ITO | MoO3 | Perovskite | Alq3; C60 | LiF | Ag', 'SLG | FTO | TiO2-c | Perovskite | DR3TBDTT; PDMS | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CoTh-TTPA | Au', 'SLG | ITO | SnO2-np | Perovskite | PTQ10 | PTAA | Ag', 'SLG | FTO | CF-Sp-BTh | Perovskite | PCBM-60 | Au', 'Willow glass | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TPA-ANR-TPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | EDOT-MPH | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Au | MoOx', 'SLG | FTO | PEDOT:PSS | Perovskite | MABr | PCBM-60 | BCP | Ag', 'SLG | ITO | PEI | PCBM-60 | Perovskite | PCDTBT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | mm-SFX-2PA | Au', 'SLG | ITO | MoS2 | Perovskite | PCBM-60 | BCP | LiF | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TPBi | Al', 'SLG | FTO | Perovskite | Metal', 'SLG | ITO | TiO2-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Florinated polymer | Spiro-MeOTAD | Ag', 'SLG | ITO | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | T80P | Au', 'SLG | FTO | TiO2-c | Perovskite | CuSCN | Au', 'SLG | ITO | NiO-c | Perovskite | ICBA | Al', 'SLG | FTO | NiO-c | PTZ-1 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | SrTiO3-np; TiO2-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | 2PACz | Perovskite | C60 | BCP | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ni | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | ITIC; PCBM-60 | PEI | Ag', 'SLG | ITO | PEDOT:PSS | PSS-Na | Perovskite | PCBM-60 | Ag', 'SLG | ITO | SnO2-c | Perovskite | NiO-c | ITO | SLG', 'SLG | FTO | TiO2-c | SnO2-c | Perovskite | Graphene; P3HT | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PAH 1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X60 | Au', 'SLG | ITO | P3CT-Na | Perovskite | C60 | BCP | Ag', 'SLG | FTO | SnO2-QDs | Perovskite | CsCuBr3-QDs | Carbon', 'SLG | ITO | MeO-TPD | PEDOT:PSS | Perovskite | C60 | Ag', 'SLG | FTO | ZnO-nw | Perovskite | Graphite | FTO', 'SLG | ITO | PEIE | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:SSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M103 | Ag', 'SLG | FTO | TiO2-c | Perovskite | NiO-np | Au', 'SLG | FTO | TiO2-c | Ba0.5Sr0.5SnO3-np | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-nanoplatelets | Perovskite | Au', 'SLG | FTO | TiO2-c | Perovskite | MoO3 | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-c | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTTP | Au', 'SLG | ITO | TiO2-c | Ag-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | NiO-c | Perovskite | Ag', 'SLG | ITO | f-C60 | C60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | ITO | SnO2-c | Perovskite | TATF8HBP | Ag', 'SLG | ITO | PEDOT:PSS | Black phosphorous QDs | Perovskite | PCBM-60 | ZrAcac | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPA-QA-DPA | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoOx | IZO', 'SLG | ITO | PEDOT:P(SS-co-TFPMA) | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BDT-4D | Au', 'SLG | FTO | 1-ethyl-3-methylimidazolium iodide | C60 | 1-ethyl-3-methylimidazolium iodide | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPA-ANT-DPA | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | PbTiO3 | Perovskite | Carbon', 'SLG | ITO | pBDT-BODIPY | PFN | Perovskite | PCBM-60 | Zr(acac)4 | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | C60 | BCP | Cu', 'SLG | ITO | PTAA | Perovskite | Polystyrene | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | SnO2-c | Perovskite | CuPc | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | TPA-PCBM | bis-C60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | GaIn', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEDOT:PSS | ITO | SLG', 'SLG | ITO | CuI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | P3Ct | Perovskite | PCBM-60 | C60 | BCP | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | ZnO-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTPA-5 | Au', 'SLG | FTO | TiO2-c | Perovskite | CuI | Graphite', 'SLG | ITO | NiO-np | Perovskite | TiO2-np | C60; PCBM-60 | Ag', 'PET | Graphene | ZnO-np | Perovskite | PTAA | PEDOT:PSS | TFSA-Graphene | PET', 'SLG | FTO | Nb2O5 | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AQ | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-mp | ZrO2-mp | Perovskite | Carbon | Galinstan', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Mo', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | SnO2-c | ITO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM-1 | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | SnO2-c | SnZnO | ITO', 'SLG | ITO | SnO2-np | C60-SAM | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ADAHI | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | F3 | BCP | Mg | Ag', 'PET | APTES; Graphene | PCBM-60 | Perovskite | PTAA | Au', 'SLG | ITO | TAPC | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | DPC60 | Perovskite | Spiro-MeOTAD | Au', "SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Field's metal", 'SLG | FTO | TiO2-c | Perovskite | Poly-N-vinylcarbazole | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | pSNT | Ag', 'PET | Graphene; TETA | ZnO-np | Perovskite | PTAA | PEDOT:PSS | Graphene; TFSA | PET | Ag', 'SLG | ITO | Aniline; rGO | CsCO3 | Perovskite | PffBT4T-2OD | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | CMB-vTA | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Mg | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | Sn', 'SLG | ITO | CuCrO2 | Perovskite | C60 | BCP | Ag', 'PET | PEDOT:PSS | Perovskite | PCBM-60 | GaIn', 'SLG | ITO | ZnTiO3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BPZTPA | Au', 'SLG | FTO | Nb2O5 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Cu1.75S | Au', 'PEN | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag | ITO', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PCBC | LiF | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60SB; TBAI | Ag', 'SLG | FTO | TiO2-c | AgInS2-QDs; TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | P3HT | Au', 'SLG | ITO | PMT | Perovskite | C60 | BCP | Ag', 'SLG | FTO | NiO-c | PTAA | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | VOx | Perovskite | PCBM-60 | Al', 'SLG | ITO | PTPD | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | PEDOT:PSS | PTPAFSONa | Perovskite | PCBM-60 | PN4N | Ag', 'SLG | FTO | NiO-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X50 | Au', 'PET | ITO | PEIE | CDIN | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-c | Perovskite | TATCz3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PCBM-60 | TmPyPB | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCTDI | Cr | Cu', 'SLG | ITO | PTAA | Perovskite | SiPc-Py-2 | PTCDA | PO-T2T | Cu', 'PET | ITO | CzPAF-TPA | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | FTO | TiO2-c | Perovskite | BP | CuSCN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | ZnO-c | Perovskite | PBTTT-14 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene oxide | CuBuPc | Au', 'SLG | FTO | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Cu:Ni-acetate | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | MATS | Al', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | IDTC6-TPA | Au', 'SLG | FTO | NiO-c | CuGaO2-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | JY5 | Ag', 'SLG | FTO | TiO2-c | ZnO-c | Perovskite | Spiro-MeOTAD | Pt | SLG | FTO', 'SLG | FTO | TiO2-c | Perovskite | COPV1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PolyTPD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | P(NDI2OD-T2); PCBM-60 | BCP | Ag', 'SLG | ITO | PolyTPD | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | ITO | Au-np; PEDOT:PSS | Perovskite | PCBM-60 | PFN | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Carbon-QDs | Perovskite | Carbon | Perovskite | Carbon-QDs | TiO2-mp | TiO2-c | FTO | SLG', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Poly(2-ethyl-2-oxazoline) | Ag', 'SLG | FTO | TiO2-c | Perovskite | MeOAc-Spiro-OMeTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M7-TFSI | Au', 'PEN | ITO | LiQ; PEIE | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | MWCNTs', 'SLG | ITO | F6-TCNNQ | TaTm | Perovskite | C60 | BCP | Ag', 'SLG | FTO | SnO2-np | Perovskite | PBDT(2H)T | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD-I | Au', 'SLG | FTO | TiO2-c | Perovskite | MEH-PPV | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SrCl2 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Th101 | Au', 'Ti-wire | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au-np', 'Textile | PEN | ITO | Au | Spiro-MeOTAD | Perovskite | PCBM-60 | SnO2-c | ITO | PEN', 'SLG | ITO | CA-Br; TPA-PT-C6 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | Na@Carbon-nanowalls', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PF8-TAA | Au', 'SLG | ITO | PolyTPD | Perovskite | C60 | BCP | CU', 'PET | ITO | SnO2-np | Perosvkite | Spiro-MeOTAD | Ag', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | MoOx | Au', 'PET | ITO | SnO2-np | Perosvkite | P3HT | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | Ag', 'SLG | ITO | c-OTPD; TPACA | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | Perovskite | Cu2NiSn4-np | Au', 'PEN | ITO | Zn2SnO4 | Perovskite | PTAA | Au', 'SLG | FTO | MgO | TiO2-mp | NiO-mp | Perovskite | Carbon', 'SLG | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | FPyBr | Al', 'SLG | ITO | NiO-c | SY1 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR122 | Au', 'SLG | FTO | Fe2O3-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-QDs | Perovskite | CsSnBr3-QDs | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PiF8-TAA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60; TDPP-CN4 | BCP | Ag', 'SLG | FTO | Al2O3; TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | ITIC | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PCBDANI | Al', 'SLG | ITO | MoO3 | CH3NH3PbI3 | C60 | TmPyPB | Ag', 'SLG | ITO | N,N‐di‐p‐methylthiophenylamine | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | ZnO-np | Al', 'SLG | ITO | ZnO-c | 1,2 ethanedithio | Perovskite | PZn-FTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | mGO | P3HT | Au', 'SLG | ITO | C60 | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TT-2,5-TPA | Au', 'SLG | ITO | NiO-c | Perovskite | FAD | PCBM-70 | AgAl', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B1 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | P3HT; PCBM-60Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Polymer1 | Au', 'SLG | ITO | PTAA | Perovskite | ITIC | Zn(acac)2 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTZ1 | Au', 'SLG | FTO | FeO | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | AgI-QDs | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-np | Perovskite | MWCNTs', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Cu9S5-np | Au', 'SLG | ITO | SnO2-np | Perovskite | Theophylline | PTAA | Ag', 'SLG | FTO | SnO2-c | SnO2-c | Perovskite | V1091 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | XY1 | Au', 'SLG | FTO | TiO2-c | Ag-np | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'PET | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag-np | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | D102 | Au', 'SLG | ITO | PTAA | Perovskite | ITIC-Th | Zn(acac)2 | Ag', 'SLG | TCO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'Ti-wire | TiO2-c | TiO2-nw | Perovskite | Carbon-nw', 'SLG | ITO | PEDOT:PSS | Perovskite | OCA | C60 | BCP | Ag', 'SLG | FTO | Perovskite | BCP | Au', 'SLG | ITO | Carbon-np; SnO2-np | Perovskite | CTAB | Spiro-MeOTAD | Au', 'SLG | AZO | Cu@Ni-nw | ZnO-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | ITO | Au', 'SLG | ITO | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NP1 | Au', 'SLG | ITO | MgO-c | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTPA-DTP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | POSS-SH | Spiro-MeOTAD | Au', 'SLS | FTO | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | BL25 | Au', 'SLG | FTO | BenMeIM-Cl | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoS2-QDs; rGO-flakes | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60 | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | FTO | TiO2-c | Perovskite | P3HT; Spiro-MeOTAD | Au | Ag', 'SLG | ITO | BPADF | PEDOT:PSS | Perovskite | C60 | Ag', 'SLG | ITO | TTA | Perovskite | PCBM-60 | C60 | BCP | Ag', 'PET | Ag-grid | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | SnO2@TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuO-nw | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | NiO-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | In2O3 | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiPc-Cou | Au', 'SLG | FTO | Zn2SnO4-c | rGO; Zn2SnO4-fiber | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO | ITO', 'SLG | ITO | TiO2-c | Perovskite | P3HT | Ag', 'SLG | ITO | TiO2-c | PBCM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS | Graphene', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Au | MoOx', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-derivate13 | Al', 'SLG | ITO | F6-TCNNQ; TaTm | TaTm | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | H101 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | CYTOP | PCBM-60 | PEIE | Ag', 'SLG | ITO | Perovskite | PCBM-60 | Methanol-hydroquinolatolithium | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-70 | Ag', 'SLG | ITO | SnO2-c | Perovskite | HZ3 | Au', 'PEG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | SWCNTs; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | AZO | Ag-nw | AZO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Spiro-MeOTAD | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | IZO', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Phen‐NaDPO | Ag', 'SLG | FTO | TiO2-c | Perovskite | [Fe(bpyPY4)](OTf)2.5 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | pBTT | Ag', 'SLG | ITO | NiO-c | PhNa-1T | Perovskite | PCBM-60 | Ag', 'SLG | ITO | P3CT-Na | Perovskite | TPE-ISO4 | C60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | C60-SAM | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Carbon black; Graphite', 'SLG | ITO | 1F-SAM | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2; SiO2-IO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO | Perovskite | PCBM-60 | PEOz | Ag', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | Spiro-MeOTAD | AgAl', 'SLG | ITO | CdSe-Qds | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | poly(DTSTPD-r-BThTPD) | Au', 'SLG | FTO | NiO-c | Perovskite | CSCNT@Al2O3-c | CSCNT@SnO2', 'SLG | FTO | TiO2-c | Perovskite | Cu2MnSn4-np | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | P3HT | MoO3 | Al', 'SLG | ITO | SnO2-c | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | ITO', 'SLG | ITO | PANI | Perovskite | C60 | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C70 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsSnBrI2-QDs | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | FTO | SLG', 'SLG | ITO | Ni | PEDOT:PSS | Perovskite | C60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SDTCz2F | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | MoO3 | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-70 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70; ZnO-np | TiO2-c | Al', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | ITO', 'SLG | ITO | ZnO-c | CsCO3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | MoOx | TPTPA | Perovskite | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPIO | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Pb', 'SLG | ITO | P3CT-Na | Perovskite | PCBM-60 | C60 | BCP | Ag', 'ITO | ZnO-c | Perovskite | Cu2O | ITO', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | SWCNTs | PMMA | Ag', 'SLG | FTO | NiO-c | Perovskite | C60 | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | PEO | Spiro-MeOTAD | Au', 'SLG | WO3 | Ag | WO3 | Perovskite | C60 | Bphen | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au | Ag', 'SLG | ITO | PCBZANT | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BI25 | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Ag | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | P(NDI2OD-T2) | Al', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | ITO | PEDOT:SAF | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | ZnO-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | PTAA | Perovskite | Polystyrene | C60 | BCP | Al', 'SLG | ITO | Cu2O | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPB-4-MOTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | V1013 | Ag', 'PET | Graphene | TiO2-c | PCBM-60 | Perovskite | Carbon-nt', 'SLG | ITO | Spiro-N | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | PTAA | Perovskite | 2PDI-4S | PEIE | Ag', 'SLG | ITO | MoO3 | Au | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | FDPI | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | ITO | poly(2-ethyl-2-oxazoline); PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | ZnO-c | C-PCBSD | Perovskite | PBTTT-14 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM | Au', 'SLG | FTO | TiO2-mp | Perovskite | Ethyl acetate; I2; LiI; TBP; Urea | Pt | FTO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TT80 | Au', 'SU-8 | Ca | Au | PEDOT:PSS | Perovskite | PCBM-60 | Ca | Ag', 'SLG | ITO | NBP-BC | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | CdS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | SAF‐OMe | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | DMAPA-C60 | Ag', 'SLG | AZO | SnO2-c | SnO2-mp | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | Ag-nw | ZnO-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PFN-OX; ZnO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Car-ETTA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | ZnO-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | P8TTT | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | Liq | Al', 'SLG | ITO | SnO2-c | Perovskite | PEAI | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-Fl | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | C3-CBL | Ag', 'SLG | FTO | SrTiO3 | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiMgO | Perovskite | PCBM-60 | ZnMgO | Al', 'SLG | PET | ZnO@rQD-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | V885 | Au', 'SLG | FTO | ZnO-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'MgF2 | Willow glass | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | ITO | SnO2-np | Perovskite | Imidazolium iodide | P3HT | Au', 'SLG | FTO | ZnO-c | ZnO-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Cu', 'SLG | FTO | TiO2-c | Perovskite | P3HT | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Sb2O3-QDs | Perovskite | Sn2O3-ns | PCBM-60 | Bphen | Ag', 'SLG | FTO | LiMgNiO-c | Perovskite | PCBM-60 | Ag', 'SLG | ITO | Spiro-TBB | Spiro-TBB | Perovskite | C60 | Ag', 'SLG | ITO | PEDOT:PSS | 4-bromobenzenediazonium tetrafluoroborate | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuO2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPcNO2-OPh | Au', 'SLG | FTO | ZnO-np | Perovskite | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | C60-N | Al', 'SLG | ITO | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Ag-grid | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | NaCl | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | ITO | F6-TCNNQ; TaTm | TaTm | Perovskite | C60; PhIm | Ag', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | PTZ-TPA | Au', 'PET | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PNDI-2T | PEIE | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ca | Al', 'SLG | ITO | P3HT | Perovskite | C60 | BCP | CU', 'SLG | FTO | TiO2-c | Perovskite | 3-Dodecylthiophene | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | PDPPT-TT | Ag', 'SLG | ITO | TiO2-c | Perovskite | Spiro-OiPrTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuInS2@ZnS-QDs | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Y2O3-c | Perovskite | Spiro-MeOTAD | Ag | Au', 'PEN | ITO | SnO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | 3-aminopropanoic acid-SAM | Perovskite | P3HT | Ag', 'PET | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'Paper | Au | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | Au | MoO3', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | Perovskite | Poly-N-vinylcarbazole | SP-11 | Au', 'SLG | FTO | CrOx | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PTTh | Perovskite | PCBM-60 | AZO-np | Au', 'SLG | ITO | F6-TCNNQ; TaTm | TaTm | Perovskite | C60; Phlm | Ag', 'SLG | FTO | Graphene; NDI; SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'PET | Ag-grid | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C70 | LiF | Al', 'Silk | Ag-nw | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiO-c | Perovskite | HATNA-F6 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H2Pc | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | Carbon | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Carbon-QDs | Perovskite | Phosphor-QDs | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Preovskite | Spiro-MeOTAD | Au', 'PET | Ag-mesh | PH1000 | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'NOA63 | ITO | PEDOT:PSS | Perovskite | PCBM-60 | MoO3 | Au | Ag | MoO3 | Alq3', 'SLG | FTO | TiO2-c | Perovskite | T40P | Au', 'SLG | ITO | MC8-TPA | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | PEDOT:PSS | Rubrene | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | PTAA｜Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CT4 | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AgAl', 'SLG | FTO | SnO2-c | Al2O3-mp | NiO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | PCDTBT1 | MoO3 | Au', 'SLG | FTO | ZnO-c | AZO-np | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | ZnO-np | Perovskite | PB(NAP-Th)TBT | Ag', 'SLG | FTO | TiO2-c | Perovskite | SCZF-5 | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | NBA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HBZ-70 | Carbon | FTO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | beta-Alanine-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | PDTSTTz | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | POSS-NH2 | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC | C60 | BCP | Ag', 'PEN | ITO | NiO-c | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Alq3 | Ag', 'SLG | FTO | TiO2-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | FASnI3 | C60 | BCP | Ag', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'PES | ITO | Perovskite | Spiro-MeOTAD | Au', 'PEN | ITO | TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FEH | Au', 'SLG | FTO | TiO2-c | Au-np; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Polystyrene | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | TiO2-c | Al | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | P3HT; SWCNTs | Ag', 'SLG | FTO | ZnO | Perovskite | Spiro-MeOTAD | Mo2O3 | Ag', 'SLG | FTO | MgO | NiO-mp | Perovskite | Carbon', 'SLG | ITO | TiO2-c | C60 | Perovskite | TaTm | TPBi | MoO3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OIPC-I | Au', 'SLG | FTO | SnO2-c | TiO2-c | Perovskite | CuPc | Carbon', 'SLG | ITO | Perovskite | PCBM-70; PTB7-Th | Ca | Al', 'SLG | ITO | Perovskite | PCBM-60 | Ag', 'SLG | ITO | ZnLiO | PCBA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | pTPD | Perovskite | IPB | Ba | Ag', 'SLG | ITO | Ti3C2Tx | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag | MoO3', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P(BDTT-ttPPD) | Au', 'SLG | ITO | NiO-np | Perovskite | EVA | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Au@Ag-np | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | 2,5‐bis(4,4′‐bis(methoxyphenyl)aminophen‐4′′‐yl)‐3,4‐ethylenedioxythiophene | Au', 'SLG | FTO | TiO2-np | PCBM-60 | Perovskite | PDCBT | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | TPE-Nme | Ag', 'SLG | ITO | PEDOT:PSS | Vox | Perovskite | PCBM-60 | Al', 'SLG | ITO | CuPc | PEI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | MeO-FDPA | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Polystyrene | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TSHBC-tBu | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SiO2-mp | Perovskite | CuPc | Carbon', 'SLG | FTO | ZnO-c | ZnO-nw | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR321 | Au', 'SLG | FTO | Ag-np; TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon', 'SLG | ITO | NiO-c | Perovskite | ICBA; PCBM-60 | Al', 'SLG | ITO | H5 | Perovskite | PCBM-60 | BCP | Au', 'MgF2 | Willow glass | ITO | PTAA | Perovskite | C60 | BCP | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | (DTYM-NDI-DTYA)2 | BCP | Ag', 'SLG | ITO | ZnO-c | ZnO-nw | Perovskite | Cu2O | ITO', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag | ZnS', 'SLG | FTO | SnO2-np | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | PTA | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PDPPDBTE | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ETPM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-422 | Au', 'PEN | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Au', 'SLG | ITO | TFB | Perovskite | C60 | B4PyMPM | Ag', 'SLG | FTO | TiO2-c | NaYF4:Yb:Er-np; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiPcS4 | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | Ag-nw; Chitosan | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Al', 'SLG | FTO | PTO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | NDT | Au', 'SLG | FTO | TiO2-c | Perovskite | CuPc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | AV-Carbon; MAI', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CdSe-QD | CsPbI3-QD | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | H-PDI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Carbon-QDs | Perovskite | Carbon-QDs | Carbon | Carbon-QDs | Perovskite | Carbon-QDs | TiO2-mp | TiO2-c | FTO | SLG', 'SLG | ITO | P3HT | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | NiO-c | Perovskite | PEAI | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro TTB | MoO3 | Au', 'SLG | FTO | Perovskite | Spiro-MeOTAD | CANP | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBM-60 | PbI2 | Perovskite | Spiro-MeOTAD | Au', 'MgF2 | Quartz | ITO | PTAA | CuSCN | Perovskite | PCBM-60 | AZO-np | PEI | Ag', 'SLG | ITO | SnO2-np | PN4N | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Al', 'SLG | ITO | ZnO-np | PFS-FTEG | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-fibres | Perovskite | Au', 'SLG | MSA-PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | C60 | Unknown | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | OTS | Perovskite | Spiro-MeOTAD | Ni-grid', 'SLG | ITO | PEDOT:PSS | MoO3 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | L-F | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | TPE-W1 | Ag', 'SLG | FTO | TiO2-c | ZnO-c | Perovskite | Spiro-MeOTAD | H2PtCl6', 'Willow glass | Graphene | Ag-nw | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | GaN | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | H-PDI | ZnO-np | Ag', 'SLG | ITO | TiS2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | Cr2O3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | C60-5a | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-nt | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | a-PTCDI | Al', 'SLG | FTO | CuO2 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | MPTS-SAM | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag-nw', 'SLG | ITO | Perovskite | C60 | BCP | Cu', 'PET | ITO | NiO-c | NiO-nw | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | YC-1 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | CuSCN | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene oxide | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuSCN | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | mp-SFX-2PA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Tetra-substituted azulene | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | TPA1C | Au', 'SLG | ITO | TAPC | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuI | Pt | FTO', 'SLG | ITO | NiO-np | Al2O3-mp | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | ITO | NiO-np | Perovskite | Ag', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | Ethyl acetate; I2; LiI; TBP; Urea | Pt | FTO | SLG', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | TET | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | XMP | MoO3 | Ag', 'PEN | ITO | PyCEE | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | nTi-MOF | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | GD; P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PVK | Ag', 'SLG | FTO | SnO2-c | Perovskite | Ome-TPA-CuPc | Au', 'SLG | FTO | (RhCp*Cp)2 | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-mp | Perovskite | Spiro-MeOTAD | Metal', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Au', 'SLG | ITO | SnO2-c | Au | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | NiO-mp | Perovskite | Au', 'SLG | ITO | TiO2-c | PNP | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | MgO-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | PCBM  Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Phen-NaDPO | Ag', 'SLG | ITO | TAPC | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | SnO2-c | PCBM-60 | Perovskite | PDCBT | Ta:WOx | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | D1 | Au', 'SLG | FTO | SnO2-np | PFN-Br | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Carpoic acid | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-2 | Au', 'SLG | ITO | In2O3-c | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | CuSCN | Au', 'SLG | ITO | NiO-np | Perovskite | C60; PCBM-60 | Zr(acac)4 | Ag', 'SLG | ITO | SnO2-np | Perovskite | Py | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuCrO2 | Au', 'SLG | FTO | ZnO-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon; MAI | Carbon', 'SLG | FTO | NiMgLiO-c | Perovskite | SFX-PDI4 | TiO2-c | Ag', 'SLG | ITO | PTAA | Perovskite | ICBA-tran3 | C60 | BCP | Cu', 'SLG | ITO | NiO-np | TPI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp; YVO4:Eu:Bi-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FDT | Au', 'SLG | FTO | NiO-c | Perovskite | C60-MPy | BCP | Ag', 'SLG | ITO | P3HT | Perovskite | PCBM-60 | Ag', 'SLG | FTO | NiO-c | Perovskite | TEABr | PCBM-60 | PEI | Ag', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | AU', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PCPDTBT | Ag', 'SLG | ITO | PVBT-TMA | Perovskite | PCBM-60 | C60-N | Ag', 'SLG | FTO | Al2O3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | SnO2-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S12 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cr', 'SLG | ITO | ZnO-c | 1,2-ethanedithiol | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z1011 | Au', 'SLG | FTO | TiO2-mp | Perovskite | I2-electrolyte | Pt | FTO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-OMeOTAD | Au', 'Ti | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Ag | MoOx', 'SLG | ITO | PTAA | Perovskite | C60 | BCP | Al', 'SLG | FTO | TiO2-c | Perovskite | DIQ-C12 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | ITIC | PCBM-60 | PEIE | Ag', 'SLG | FTO | SnO2-c | PCBM-60; PMMA | Perovskite | PMMA | Spiro-MeOTAD | Au', 'SLG | ITO | CuInS2 | Al2O3-mp | Perovsktie | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | P3TMAHT | Ag', 'SLG | FTO | SnO2-c | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PDI | Ag', 'SLG | FTO | TiO2-c | Perovskite | X59 | Au', 'SLG | ITO | P3OT | Perovskite | C60 | BCP | CU', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TAA14 | Au', 'SLG | FTO | TiO2-c | ZIF-8 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovsite | Boron subphthalocyanine chloride | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuH | Au', 'PDMS | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM5 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BL07 | Au', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Tetrakis(4-methoxyphenyl)spiro[cyclopenta[1,2-b:5,4-b']dipyridine-5,9'-fluorene]-2',7'-diamine | Au", 'SLG | FTO | TiO2-nanoleaves | MgO | Perovskite | Spiro-MeOTAD | Graphite', 'SLG | FTO | MTDATA | Perovskite | C60 | BCP | Cu', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | Carbon black', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | TZ1 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | POZ9 | Au', 'SLG | FTO | TiO2-c | Perovskite | NiO-np | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | SnO2-np | Dompamin-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-nanosphere | ZrO2-mp | Perovskite | Carbon', 'SLG | ITO | poly(1,4-phenylenevinylene) | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Sinapoyl malate | Perovskite | Co(II)P | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | diPDI:DMBI | TiO2 | Al', 'SLG | ITO | NiO-c | PEAI | Perovskite | PCBM-60 | AgAl', 'SLG | ITO | PEDOT:PSS | Perovskite | S-acetylthiocholine chlorde | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | Oleylamine | Au', 'SLG | ITO | SnO2-np | PCBM-60; PCBA | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'PET | ITO | PEDOT:PSS | NiO-c | Perovskite | PCBM-60 | TIPD | Al', 'SLG | ITO | MeO-2PACz | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FTO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | POZ2 | Ag', 'SLG | ITO | Graphene oxide; MoOx-np | Perovskite | PCBM-60 | Ag', 'SLG | ITO | JW6 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | P3CT-Na | Perovskite | PCBM-60 | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (n-BuO)4ZnPc | Au', 'SLG | FTO | NiO-c | Perovskite | CeOx | Ag', 'SLG | ITO | SnO2-c | Perovskite | SDF-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuSCN | Carbon', 'SLG | FTO | TiO2-c | BaSnO3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | CuI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | COTT-1 | COTT-2 | Au', 'SLG | ITO | Z9 | Perovskite | PCBM-60 | BCP | Ag', 'Carbon-nt-yarn | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Pt-Carbon-nt', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Cs2CO3 | Al', 'SLG | FTO | TiO2-c | MPMIC60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEI | PCBM-60 | Perovskite | PTAA | MoO3 | Ag', 'SLG | ITO | PDTON35 | SnO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | TiO2-c | Ag | TiO2-c | CPTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Trux2 | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO-np | DPO | Ag', 'SLG | FTO | TiO2-c | Perovskite | V911 | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | V997 | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | M112 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon | Ag', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | FTO | SnO2-c | Perovskite | SM09 | Ag', 'SLG | ITO | PTAA | PFN | Perovskite | PCBM-60 | BCP | Cu | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Mg | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | AgAu-mp', 'SLG | FTO | TiO2-c | Perovskite | CBP | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS | In', 'SLG | FTO | Perovskite | P3HT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Ag | Au', 'SLG | ITO | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MoO3 | TPTPA | Perovskite | 3TPYMB | C60 | TmPyPB | Ag', 'SLG | ITO | C60 | C60 | Perovskite | TaTm | F6-TCNNQ; TaTm | Au', 'SLG | ITO | SnO2-c | PEIE | PCBM-60 | Perovskite | Spiro-MeOTAD | MoOx | IO | ITO | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | HTM | Ag', 'SLG | ITO | ZnO-c | ZnO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | PEDOT:GSL | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | PTAA | PFN | Perovskite | PCBM-60 | PFN | Ag', 'SLG | FTO | TiO2-c | Perovskite | BDT-C1 | Ag', 'SLG | FTO | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-PT-OMeTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60-np | PrC60MA | PEI | Au', 'Steel | ZnO-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Carbon-nt', 'SLG | ITO | WOx | Perovskite | PCBM-60 | Ag', 'SLG | FTO | NiO | Perovskite | Q10 | BCP | Ag', 'Ti | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag-nw', 'SLG | ITO | SnO2-np | DTPA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | pBBTa‐BDT2 | Au', 'SLG | FTO | C60 | Perovskite | P3HT | Au', 'SLG | FTO | Carbon-mp | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Au', 'SLG | ITO | SnO2-np | Perovskite | CH3O-PEAI | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1c | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag | PMMA', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S,N-Heteroacene 1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTTP-CN | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Pervskite | CuSCN | Au', 'SLG | FTO | In2O3 | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PDPP3T | Ag', 'SrTiO3 | Sr2RuO4 | Perovskite | ITO', 'SLG | ITO | PEDOT:PSS | Dex-CB-MA | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CZTS0.25Se0.75; rGO | Au', 'SLG | ITO | Ts-CuPc | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | TiO2-c | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | Perovskite | MoO2-np | Au', 'SLG | AZO | Ag-grid | AZO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PCPDTBT | PEDOT:PSS | Au', 'SLG | ITO | NiO-np | Perovskite | C60 | BCP | Cu', 'SLG | ITO | NiO-c | Perovskite | ZnO-np | C60 | Ag', 'SLG | ITO | Spiro-TTB | Perovskite | LiF | C60 | SnO2-c | IZO | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Ag | Mox', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HPPHT | Carbon', 'SLG | Ag-nw | Graphene | ZnO-np | Perovskite | PTAA | Au', 'SLG | ITO | NiO-c | Perovskite | C60 | SnO2-np | Ag', 'SLG | ITO | SnO2-c | MgO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuO | Perovskite | PCBM-60 | Ca | Al', 'SLG | ITO | WOx | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PCDTBT | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | Ba(La)SnO3 | Perovskite | PTAA | Au', 'SLG | ITO | C60 | Perovskite | MoO3 | Ag', 'SLG | ITO | Spiro-TTB | Perovskite | C60 | TmPyPB | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | F1 | BCP | Mg | Ag', 'SLG | ITO | SnO2-np | Perovskite | PTAA | NiO-c | ITO | PEN', 'SLG | ITO | NiO-np | TPI-6MEO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-c | C9 | Perovskite | Spiro-MeOTAD | Au', 'PET | Graphene | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | PDI | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Ag | MoO3 | Back-reflector', 'SLG | FTO | WO3-c | WO3-nw | Cs2CO3 | PCBM-60 | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Black phosphorous nanosheets | Au', 'SLG | FTO | TiO2-c | TiO2-mp | C60-SAM | Perovskite | P3HT | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | DBP | PCBM-60 | Bphen | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H66 | Au', 'SLG | FTO | TiO2-c | Perovskite | Phenethylamine | Spiro-MeOTAD | Au', 'PET | ITO | PFN | C60; PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO-HMDS Scaffold | TiO2-c | C60 | Perovskite | PTAA | Ag', 'SLG | ITO | Ag-grid | AZO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA‐ANT‐TPA | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X23 | Au', 'SLG | ITO | PEDOT:PSS | TS-CuPc | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | Perovskite | DTS(IIThThHEX)2 | MoOx | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | CTAB | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DM2 | Au', 'SLG | ITO | PEIE | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Perovskite | Au', 'SLG | ITO | p-PFP-O | PTAA | Perovskite | PCBM-60 | PFN | Cu', 'SLG | ITO | CuI | PbPc | Perovskite | C60 | Bphen | Ag', 'SLG | ITO | PTAA | Perovskite | PCBB-OEG; PCBM-60 | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Polyrotaxane | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H111 | Au', 'SLG | ITO | rGO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | C60 | ZnSe | Ag', 'A.R.C. | SLG | ITO | PTAA | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | TiO2-mp | Nb2O5-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEG; ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'PET | Al2O3 | Ti | C60 | Perovskite | NiO-c | Ni | Al2O3 | PET', 'SLG | FTO | SnO2-np | PCBM-60 | Perovskite | Selenium | Au', 'SLG | ITO | SnO2-np | Perovskite | AL2O3-c | Spiro-MeOTAD | MoOx | ITO | MgF2', 'SLG | FTO | In2S3 | Perovskite | Spiro-MeOTAD | Au', 'PET | IZO | PTAA | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPc-OBu | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Au', 'SLG | ITO | ZnO-c | ZnO-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Benzoic acid | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | FTO | NiO-c | Perovskite | Au', 'SLG | ITO | TiO2-c | Perovskite | TPADPP-1 | Au', 'SLG | FTO | TiO2-nw | N719 | Perovskite | Au@CZTS-np | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BMIM-OTF | Al', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-70 | LiF | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | SnO2-c | SnO2-mp | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | Graphene | MoO3 | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | FTO | TiO2-c | PCBM-60 | C60-ETA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M:OO | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsSnI2.95F0.05 | FTO | SLG', 'SLG | ITO | DA-PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTF1 | Ag', 'SLG | ITO | MoO3 | Perovskite | PCBM-60 | Bphen | Ag', 'NOA88 | PEI | Au | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PTTI-1 | Ag', 'SLG | ITO | NiO-c | Perovskite | FSIP | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS; Sorbitol | Ag-grid | PET', 'SLG | FTO | Cu0.5Cr0.5O2 | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiO | PS | Perovskite | PS | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ni', 'SLG | FTO | TiO2-c | KY7F22-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | Au-np; Graphene; TFSA | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X61 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SCPDT-BiT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BChl-3 | Ag', 'SLG | FTO | SnO2-np | Perovskite | Hexamethyl-substituted subphthalocyanine | Au', 'SLG | ITO | PB2T-O | Perovskite | PCBM-60 | ZnO | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | H-DIPBI | Al', 'SLG | ITO | C60; PhIm | Perovskite | TaTm | F6-TCNNQ; TaTm | Ag', 'SLG | FTO | ZnO@rQD-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | f-C70 | Perovskite | P3HT | MoO3 | Ag', 'SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | NiO-c | NiO-mp | Perovsite | C60 | BCP | Al', 'SLG | ITO | H7 | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Co-porphyrins | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | C70 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | PbI2 | PDPP3T; PCBM-60 | Al', 'PEN | ITO | WOx | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | Perovskite | PDBT-co-TT | Au', 'PEN | ITO | rGO | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | PS | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | C60 | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | NiMgLiO | Perovskite | In2O3-c | Ag', 'SLG | FTO | TiO2-c | Perovskite | PTAA | Ag', 'SLG | FTO | TiO2-c | TiO2-helices | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Cobalt–porphyrin | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO | BCP | Ag', 'SLG | ITO | CzPAF-TPA | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OMeTPA-BDT | Au', 'SLG | FTO | NiO-c | PMMA | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HT-ZnPc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 4-(4-phenyl-4-alfa-naphthylbutadienyl)-N,N-di(4-methoxyphenyl)-phenylamine | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | F8T2e | Au', 'SLG | FTO | SnO2-np | C60 | Perovskite | V950 | Au', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SO8 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Mono-PCBOE | ZnO | Al', 'SLG | ITO | TiO2-c | Perovskite | DOR3T-TBDT | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI-ID | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | AgAl-np | Au-np | LiF | AgAl', 'PEN | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | YVO4:Eu | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | X-DVTPD | Perovskite | PCBM-60 | FPI | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ni', 'PEN | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | EP02 | Au', 'SLG | ITO | PEDOT:PSS | DPP-DTT | Perovskite | PCBM-60 | LiF | Ag', 'SLG | WO3 | Ag | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | Nb2O5-c | Nb2O5-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Thiourea | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | CuI | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3HT; PFN | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | MXene | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Carbon | Al', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | EH44 | Au', 'SLG | ITO | PT | Perovskite | C60 | BCP | Ag', 'SLG | FTO | SnO2-c | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MPA-BTTI | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR355 | Au', 'SLG | ITO | PTAA | PFN-Br | Perovskite | C60 | BCP | Cu', 'SLG | MSA-PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | C60 | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | Perovskite | PBDTTT-CT | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | TPE-DPP8 | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PFN-Br | Al', 'SLG | ITO | TiO2-c | Perovskite | Spiro-OBuTAD | Au', 'SLG | ITO | PANI:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | SnO2-mp | Perovskite | P1 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI-BTH2 | Ag', 'SLG | ITO | TRUX1 | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | PEG; PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-mp | Perovskite | HfO2 | CuI | Graphite', 'SLG | WO3 | Ag | WO3 | PEDOT:PSS | Perovskite | C60 | Bphen | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | Au', 'SLG | ITO | PTAA | MoS2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-c | Perovskite | Cu2O | Au', 'SLG | ITO | Py-COF | PTAA | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | AgAl', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTB3 | Au', 'SLG | ITO | C60(OH)16 | C60 | Perovskite | PH3T | MoO3 | Ag', 'PEN | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-c | IDTT2FPDI | Perovskite | PTAA | Ag', 'SLG | FTO | SnO2-c | KCl | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | C60; PhIm | C60 | Perovskite | TaTm | F6-TCNNQ; TaTm | Ag', 'PEN | ITO | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | ITO | P3CT-N | Perovskite | PCBM-60 | C60 | BCP | Cu', 'SLG | FTO | MgO | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | HTM | Metal', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,6-tris-(N,N-di-p-methoxyphenylamine)pyrene | Au', 'SLG | FTO | [EMIM]PF6-IL | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | CO2CO3 | Perovskite | PolyTPD | Au', 'SLG | FTO | PTAA | Perovskite | C60 | BCP | Cu', 'SLG | ITO | ZnO-c | APTES-SAM | Perovskite | PCBM-60 | Au', 'SLG | FTO | PbZrTiO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Cr | Au', 'SLG | FTO | SnO2-c | TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PTAA | Perovskite | PS | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon; TiO2-np', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Zr(acac)4 | Al', 'SLG | ITO | TiO2-c | C60 | Perovskite | PDTSTTz-4 | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60; PCBM-60 | bis-C60 | Ag', 'SLG | ITO | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | BBA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | ICBA | bis‐C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC-3 | Ag', 'SLG | ITO | PVBT-SB | Perovskite | PCBM-60 | C60-N | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-BPV-TPA | Au', 'PEN | ITO | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | Rhodamine 101 | LiF | Ag', 'SLG | ITO | CuInS2 | Al2O3-np | Perovskite | PCBM-60 | TIPD; ZnO-np | Ag', 'SLG | ITO | PCDTBT; PFN | Perovskite | PCBM-60 | Al', 'SLG | ITO | Au-np; TiO2-np | C-PCBSD | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-TPM | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Co3O4-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2-((2-(4-(2-ethylhexyl)-4H-dithieno[3,2-b:2′,3′-d]pyrrol-2-yl) thiazol-5-yl)methylene) malononitrile | Ag', 'PEN | ITO | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | H:MoO3-nanobelts', 'SLG | ITO | Cs2CO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | P(VDF-TrFE) | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P4 | Au', 'SLG | FTO | TiO2-c | Perovskite | TPE-W2 | Ag', 'PEN | ITO | ZnO-c | Perovskite | Carbon', 'SLG | FTO | PEDOT:PSS | Perovskite | C60 | BCP | Ag', 'SLG | ITO | Perovskite | PCBM-60 | ZnO | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SnS-np | NiO-np | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KM03 | MoO3 | Ag', 'SLG | FTO | SnO2-c | Perovskite | HOFP | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | TCPI | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-mp | Perovskite | 2,5-bis (5-(5-(5-hexylthiophen-2-yl) thiophen2-yl) thiophen-2-yl) thiazolo[5,4-d]thiazole | Au', 'SLG | ITO | ZTO | SnO2-c | C60 | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuInS2@ZnS-QDs | Au', 'SLG | ITO | AZO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | WOx | Perovskite | Spiro-MeOTAD | Ag', 'PET | Ag | MoOx | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Cu | Ag | MoO3', 'Ti-foil | TiO2-nt | Perovskite | Spiro-MeOTAD | Carbon-nt', 'SLG | ITO | PEDOT:PSS | Propionic acid | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CS03 | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | SnO2-c | Ba(OH)2 | Perovskite | PDCBT | MoOX | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TPAC0M | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-nw | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | Graphen', 'SLG | FTO | TiO2-mp | Perovskite | V841 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | Bphen | Ca | Ag', 'SLG | FTO | TiO2-c | KH550 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MoO3 | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (10-butyl-3,7-diphenylphenoxazine) | Au', 'SLG | ITO | NiO-c | Perovskite | PEAI | PCBM-60 | BCP | TeO2 | Ag', 'SLG | FTO | TiO2-c | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TBP | Ag', 'SLG | FTO | TiO2-c | Perovskite | 0F | Au', 'SLG | ITO | PSS-g-PANI | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | NiO-np | Perovskite | NiO-np | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | NiO-mp | Perovsite | ZnO | Al', 'SLG | FTO | Fe2O3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | Ome-DPA-CuPc | Au', 'SLG | FTO | TiO2-c | PCBDAN | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | CuIn1.5Se3-QDs | Au', 'SLG | FTO | ZnO | MgO | EA | TiO2-mp | Perovskite | Spiro-MeOTAD | CuI | Au', 'SLG | FTO | SnO2-np | Ethanol amine | Perovskite | Spiro-MeOTAD | Au', "SLG | FTO | SnO2-c | C60 | Perovskite | N4,N4'-(4,10-dimethyl-6H,12H-5,11-methanodibenzo[b,f][1,5]diazocine-2,8-diyl)bis(N4,N4',N4'-tris(4-methoxyphenyl)-[1,1'-biphenyl]-4,4'-diamine) | Ag", 'SLG | FTO | Au-grid | TiO2-c | SnO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | F-60; bis-C60 | Ag', 'SLG | ITO | C60 | Perovskite | BTF2 | MoO3 | Ag', 'SLG | FTO | NiMgLiO | Perovskite | CeOx-np | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | TiO2-c | Ag', 'SLG | ITO | NDI-P | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Graphene oxide | Perovskite | ZnO-np | Al', 'Unknown | Perovskite | Unknown', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | Fe2O3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | CzPF | Au', 'SLG | ITO | ZnO-np | Perovskite | Bifluo | Ag', 'SLG | ITO | TiO2-c | Perovskite | PCA-1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CJ-01 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HL-1 | Au', 'SLG | ITO | Graphene oxide | PTAA | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'PEN | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PTEG-1 | Ag', 'SLG | ITO | NiO-c | Perovskite | ZnO-np | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | TTC | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Poly[4,8-bis(2-(4-(2-ethylhexyloxy)phenyl)-5-thienyl)benzo[1,2-b:4,5b’]dithiophene-alt-1,3-bis(4-octylthien-2-yl)-5-(2-ethylhexyl)thieno[3,4-c]pyrrole-4,6-dione | Ag', 'SLG | FTO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PVBT-SO3 | Perovskite | C60/C70-N | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TT0 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-nt | PMMA | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | SnO2-c | Perovskite | PMMA | Carbon', 'SLG | ITO | TFB | Al2O3-np | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Ag-nw; PEDOT:PSS | Perovskite | PCBM-60; CTAB | Ag', 'SLG | ITO | SnO2-np | ImAcHcl | Perovskite | Spiro-MeOTAD | Au', 'PEN | ITO | Ti(OiPr)4 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-np | Perovskite | BDTS-2DPP | Au', 'SLG | FTO | ZnO-c | Perovskite | Graphene oxide | Ag', 'SLG | ITO | PEDOT:PSS | Graphene oxide; PEG | Perovskite | PCBM-60 | MoS2 | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | WT3 | Au', 'SLG | FTO | NiO-c | NiO-mp | Perovskite | Acetyl acetate; I2; LiI | Pt | FTO | SLG', 'SLG | ITO | PCBM-60; PDI-DA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | SnO2-c | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | C60 | AUH | Ag', 'SLG | ITO | MgZnO | Perovskite | Spiro-MeOTAD | MoOx | Al', 'PET | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Zn2SnO4 | Perovskite | PTAA | Au', 'SLG | FTO | F4-TCNQ | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | P3HT | FTO | SLG', 'SLG | ITO | TiO2-c | Na2SO4 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-nanosphere | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPB | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | LiF | Al', 'SLG | FTO | AZO-np | Perovskite | PTB7 | MoO3 | Al', 'SLG | FTO | CeOx | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PTEG-1 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MWCNTs', 'SLG | ITO | HTM-3 (bifluorenylidene-based) | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | PSS-Na | Perovskite | PCBM-60 | ZnO | Al', 'SLG | ITO | P3CT-K | Perovskite | PCBM-60 | ZnO-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B2 | Au', 'SLG | FTO | TiO2-c | SiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Montmorillonite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | TiO2-c | D35 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-3,4 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | HAT-CN | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,6-Cbz-EDOT | Au', 'SLG | ITO | NiO-np | Perovskite | C60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | SiO2-IO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | Spiro-MeOTAD | Au', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Poly[4,8-bis(2-(4-(2-ethylhexyloxy)3-fluorophenyl)-5-thienyl)benzo[1,2-b:4,5-b'] dithiophenealt-1,3-bis(4-octylthien-2-yl)-5-(2-ethylhexyl)thieno[3,4-c]pyrrole-4,6-dione | Ag", 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-Pc | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | CzP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | NiO-mp | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PVAc | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Cr', 'SLG | ITO | CuOx | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YN1 | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ACE‐ANT‐ACE | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SnS | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | cyclopenta[2,1-b; 3,4-b′]dithiophene | MoOx | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | TDPP-CN4 | BCP | Ag', 'SLG | ITO | NiO | Perovskite | C60; PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | VOx | Carbon', 'Eu-complex LDL | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', "SLG | ITO | TiO2-c | Perovskite | N2',N2',N7',N7'-tetrakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2',7'-diamine | Au", 'SLG | ITO | TiO2-c | Perovskite | HAB1 | Au', 'SLG | ITO | TiO2-mp | Perovskite | 4-(5-(5-(5-(5-(5-hexylthiophen-2-yl) thiophen-2-yl) thiophene-2-yl) thiazolo[5,4-d]thiazol-2-yl) thiophene2-yl)-N,N-diphenyl-benzenamine | Au', 'SLG | ITO | SnO2-np | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | IDT2 | Au', 'SLG | FTO | SnO2-c | Perovskite | RCP | Au', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | C60 | PAA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TPASBP | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskire | PCBM-60 | CIL | Ag', 'SLG | FTO | Ag-np; PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | NiO-np | Choline chloride | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | ZnO-c | AZO-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | EHCz-2EtCz | MoOx | Al', 'PET | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | CA-Br | Perovskite | PCBM-60 | BCP | Ag', 'SLG | PEDOT:PSS | PEI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-nw | SnO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | PS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | ZnO-c | Perovskite | PCBM-60 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Carbon | FTO', 'SLG | ITO | PTAA | PMMA | Perovskite | IDTT2FPDI | PCBM-60 | Bphen | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CdZnSe@ZnSe-QDs | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | d-PCBM-60; PCBM-60 | Ca | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | AZO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SbI3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | ZnO-nw | Au-np | Perovskite | Graphene', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Poly[4,8-bis(2-(4-(2-ethylhexyloxy)3,5-fluorophenyl)-5-thienyl)benzo[1,2-b:4,5-b']dithiophenealt-1,3-bis(4-octylthien-2-yl)-5-(2-ethylhexyl)thieno[3,4-c]pyrrole-4,6-dione | Ag", 'SLG | FTO | SnO2-c | Perovsite | Spiro-MeOTAD | Ag', 'PET | Graphene; TETA | PTAA | PEDOT:PSS | Perovskite | ZnO-np | Graphene | PET', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ba | Al', 'SLG | FTO | ZnO-c | ZnO-mp | PEI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M106 | Ag', 'NOA63 | MoO3 | Au | PEDOT:PSS | Perovskite | PCBM-60 | MoO3 | Au | Ag | MoO3 | Alq3', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | NiO | Perovskite | PCBM-60 | PN6 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-5 | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | ZnO-np | AZO | Ni | Al | MgF2', 'SLG | FTO | TiO2-mp | Perovskite | Ethyl acetate; I2; LiI; TBP | Pt | FTO | SLG', 'SLG | ITO | PEDOT:PSS | Perovskite | Graphene | Al', 'SLG | ITO | rGO | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag-nw', 'SLG | FTO | TiO2-c | TiO2-nc | Perovskite | P3HT | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | F2 | BCP | Mg | Ag', 'SLG | ITO | SnO2-c | Perovskite | Z2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cs-oleate | Spiro-MeOTAD | Au', 'SLG | ITO | PolyTPD | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SubPc | Au', 'SLG | FTO | Zn2SnO4-c | Zn2SnO4-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TPAC3M | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TBC-2 | Au', 'SLG | ITO:ATO | TiO2-c | Perovskite | Perylene | Au', 'SLG | ITO | NiCo2O4 | Perovskite | PS | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag | MoO3', 'SLG | FTO | TiO2-c | Perovskite | Al2O3-c | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60-ETA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO@C | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag | Teflon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Ag-nw', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon black; Graphite', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEL | PH 1000', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | Cu0.8Cr0.2O2 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPcNO2-OMFPh | Au', 'SLG | ITO | MoOx | Perovskite | C60 | BCP | Al', 'SLG | FTO | SnO2-c | Perovskite | CuBuPc | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YK2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | CsI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | CsPbI3-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | TiO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | C60-SAM | Perovskite | PDCBT | WOx | Au', 'Flexible | IZO | PEIE | C60 | B4PyMPM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MEH-PPV; PFN | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEIE | PCBM-60 | Perovskite | P3HT | Au', 'SLG | ITO | NPB | Perovskite | PCBM-60 | Al', 'SLG | FTO | NiMgLiO | Perovskite | 2-acetylpyridine | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X59 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Carbon', 'SLG | ITO | PTAA | Perovskite | C60 | BCP | IZO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BEDCE | Spiro-MeOTAD | Ag', 'SLG | ITO | NiPc | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | DPO | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BMIM-OTF | Au', 'SLG | ITO | Perovskite | Carbon-nt | PFN-Br | Ag', 'SLG | ITO | CuO | Perovskite | PC(70)BM | AZO | Al', 'SLG | ITO | SnO2-c | Perovskite | PDMS | CuSCN | Au', 'SLG | ITO | NiO-np | Perovskite | NDI-BiSe | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | m-MTDATA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | Carbon | Al', 'SLG | FTO | TiO2-c | TiO2-mp | C60-SAM | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskte | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AgAu-mp', 'ITO | PEN | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | rGO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTPA-4 | Au', 'SLG | ITO | ZnO-c | Perovskite | P3HT | Ag', 'SLG | FTO | TiO2-c | TiO2-nanoflowers | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | Perovskite | pPh-2MODPACz | Au', 'PEN | ITO | TiO2-np | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PbTiO3 | Perovskite | Carbon-nt', 'SLG | ITO | PEDOT:PSS | Perovskite | bis‐C60 | Ag', 'SLG | ITO | NiO-c | Perovskite | LiF | C60 | SnO2-c | ZnSnO2-c | ITO | Ag', 'SLG | FTO | SnO2-np | Preovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | ICBA | BCP | Ag', 'SLG | ITO | PEDOT:PSS | AuAg@SiO2-np | Perovskite | TPCBM | PFN | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Polyethylimine | Ag', 'SLG | ITO | PTAA | Perovskite | ICTA | BCP | Cu', 'SLG | ITO | PTAA | PMMA | Perovskite | PEAI | PCBM-60 | Bphen | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | IC60BA | bis-C60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | F16CuPc | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PIF8-TAA | Au', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR378 | Au', 'SLG | ITO | NiO-np | ME3 | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | PVK | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Al', 'SLG | ITO | TiO2-np | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PBT1-C | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTB1 | Au', 'SLG | ITO | TiO2-c | Perovskite | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | P1 | Ag', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | Au | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | ITO | Rubrene | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | PTMA-BP | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TT-3,6-TPA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Al', 'SLG | ITO | BTF3 | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | SnO2-c | C60 | Perovskite | PPyra-XA | MoO3 | Ag', 'SLG | ITO | PEI | PCBM-60 | Perovskite | PTAA | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PMDPT | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag-nw | ZnO-np', 'SLG | ITO | PEDOT:PSS | Perylene | Perovskite | PCBM-70 | C60 | BCP | Al', 'SLG | ITO | FPI-PEIE | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CIGGSe-np | Au', 'SLG | ITO | MoO3 | TBDI | Perovskite | IPH | PDINO | Ag', 'SLG | ITO | PFN | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | D205 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | X2 | Au', 'SLG | ITO | FPDI | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PTAA | Perovskite | 2PDI-3S | PEIE | Ag', 'SLG | FTO | ZnO-c | ZnO-nanodisks | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA; TPFB | Au', 'SLG | ITO | CdS-nw | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | SnO2-QDs | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | PN4N | Al', 'PET | ITO | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | AU', 'SLG | FTO | NiO-c | NiO-mp | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X22 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ag-nw | Spiro-MeOTAD | Au', 'SLG | FTO | C60 | Perovskite | CuPc | Au', 'SLG | ITO | L-H | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoOx | ITO', 'SLG | ITO | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PP-Spiro | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnChl-3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Graphene-QDs | Perovskite | Carbon', 'SLG | ITO | BDT-POZ | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnO-c | PCBM-70 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuSCN | Ag', 'SLG | FTO | TiO2-c | PCBA | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Al2O3-np | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | SnO2-np | Perovskite | PDO1 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | FNCA | C60 | BCP | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | TaTm | MoO3 | Au', 'SLG | ITO | ZnO-c | ZnO-nw | PCBM-60 | Perovskite | P3HT | Au', 'SLG | ITO | Spiro-MeOTAD | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | SnO2-np | Perovskite | NiO-np | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTB1 | Au', 'SLG | FTO | TiO2-c | Perovskite | C201 | Ag', 'SLG | rGO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | VB-MeO-FDPA | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Poly(ethylene oxide) | Carbon', 'SLG | FTO | TiO2-c | ITIC | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBSD | Perovskite | HPB-OMeDPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO-np | CuSCN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene oxide | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | mm-SFX-3PA | Au', 'SLG | FTO | SnO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H11 | Au', 'SLG | ITO | 2EGO-PPV | Perovskite | PCBM-60 | BCP | cu', 'SLG  FTO | TiO2-c | TiO2-mp | Perovskite | BX-OMeTAD | Au', 'PETUG | C60 | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | DNA-CTMA | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PT3HT | Ag', 'Flexible | IZO | PEIE | C60 | LiF | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | TPA-BP-OXD | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | p-PFP-O | PTAA | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Graphene oxide | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X19 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-OMe | Au | Ag', 'SLG | ITO | MFGO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | WS2 | Perovskite | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TDAB | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | ACR-TPA | Au', 'SLG | ITO | P3CT-Na | Perovskite | IDT6CN-4F | C60 | BCP | Ag', 'SLG | ITO | NiO-c | Mg(AcO)2 | Perovskite | C60 | BCP | Cu', 'SLG | ITO | PbI2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ag', 'SLG | FTO | NiMgLiO | Perovskite | 3-acetylpyridine | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BAI | DM | Au', 'SLG | ITO | PEDOT:PSS | NaI | Perovskite | PCBM-60 | AZO | BCP | Ag', 'SLG | ITO | SnO2-np | Li-TFSI | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PFN-P1 | Perovskite | PCBM-60 | PFN-P2 | Ag', 'SLG | ITO | PEDOT:PSS | Au-np; VOx | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SrGeO3 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | ZnO-c | 3-aminopropanioc acid-SAM | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PCBM-70 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | C60 | Perovskite | F22 | Au', 'SLG | FTO | NiO-c | Perovskite | ZnO-c | Ag', 'SLG | ITO | X-DVTPD | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P1Z2 | Carbon', 'SLG | FTO | TiO2-c | SnO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | Perovskite | COPV7 | Au', 'SLG | FTO | BaSnO3-mp | Perovskite | Spiro-MeOTAD | Carbon', 'SLG | FTO | TiO2-c | Perovskite | IEICO-4F | Spiro-MeOTAD | MoO3 | Au', 'SLG | FTO | ZnO-np | EA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | DEME-BF4 | ITO', 'SLG | FTO | NiMgO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | SnO2-np | Perovskite | CH3-PEAI | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PABA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | MEAI | Spiro-MeOTAD | Au', 'SLG | FTO | NiO | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiO-c | N749 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | MoS2; PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'Quartz | TaN | TaS2 | Perovskite | CdS | ITO', 'SLG | ITO | MoO3 | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | FTO | TiO2-c | CSOE | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS | Perovskte | PCBM-60 | Cr2O3 | Ag', 'PEN | ITO | SnO2-np | Perovskite | P3HT; Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | AZO-np | ITO', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | ITO | PolyTPD | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | AuAg | MoO3', 'SLG | FTO | TiO2-nw | Perovskite | CuSCN | Carbon', 'SLG | ITO | NiO-c | Perovskite | C60 | BCP | Ag', 'PET | Graphene; TETA | PTAA | PEDOT:PSS | Perovskite | ZnO-np | Graphene; TETA | PET | Ag', 'SLG | ITO | p-PFP-O | PTAA | Perovskite | PCBM-60 | PFN | Al', 'SLG | ITO | PEDOS:PSS | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Au@Ag-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsSnBr3-QDs | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | Perovskite | TiO2-mp | TiO2-c | FTO | SLG', 'Cellophane | TiO2-c | Ag | TiO2-c | CPTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | Perovskite | PTAA | NiO-c | Au | PEN', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YKP06 | Ag', 'SLG | FTO | NiO-np | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | Al2O3-c | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | PMMA | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | PS | Perovskite | PTAA | PEDOT:PSS | ITO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Spiro-MeOTAD | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Graphite ribbon', 'SLG | ITO | PTAA | Perovskite | PCBB-3N | PCBM-60 | Al', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Ti | Au', 'SLG | ITO | PEDOT:PSS | Carbon-nt | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nc | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TPA-NADT-TPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | In2O3-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-np | Perovskite | Zn-ChL | H2-Chl | Ag', 'SLG | FTO | NiO-c | TiO2-c | Perovskite | Carbon', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Ag@SnO2-nw | Perovskite | PCBM-60 | Al', 'SLG | ITO | styryl-functionalized GO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | C60; Phlm | Perovskite | TaTm | F6-TCNNQ; TaTm | Ag', 'SLG | FTO | TiO2-c | BZnTPP | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | ZTO-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1-(N,N-di-p-methoxyphenylamine)pyrene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | D149 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | N,N-bis-[7-(4,4′-dimethoxydiphenylamine)-9- (bis(methylsulfanyl)methylene)fluoren-2-yl]-4-methoxyaniline | Au', 'SLG | ITO | PEDOT:PSS | Au-nw; VOx | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | VOx | PEI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | NiO-c | Al2O3-mp; Au@SnO2-nw | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Phosphoniumfluorene | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC06 | Ag', 'SLG | FTO | SnO2-c | C60 | Perovskite | Spiro-MeOTAD | Ag', 'A.R.C. | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | Au | A.R.C.', 'SLG | ITO | ZnO-c | Perovskite | CzPAF‐SBF | Au', 'SLG | ITO | PCBM-60 | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S9 | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Ag | ITO', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | PCDTBTB | MoO3 | Au', 'SLG | ITO | VOx | Cu phtalocyanine | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PCBC | Al', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS | ITO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DMFA-FA | AU', 'SLG | ITO | NiO-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-nanosheets | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | SnO2-c | Perovskite | Au', 'SLG | ITO | SnO2-np | Perovskite | BTTI-C8 | Au', 'SLG | FTO | TiO2-np | Perovskite | Spiro-MeOTAD | Graphite', 'SLG | SU-8 | MoO3 | Au | PEDOT:PSS | Perovskite | PCBM-60 | Ca | Ag', 'SLG | FTO | ZnO-c | MgO | TiO2-mp | Perovskite | TPB(2-MeOTAD) | Au', 'SLG | FTO | TiO2-nanoplatelets | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60; Graphene | BCP | Ag', 'SLG | FTO | In2S3 | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SFXDAnCBZ | Au', 'SLG | ITO | PPN | Perovskite | C60 | BCP | Ag', 'SLG | ITO | SnO2-np | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | ICBA | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | IT-4f | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | AZO | Al', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Ag', 'SLG | ITO | Z7@MWCNTs | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X25 | Au', 'SLG | ITO | ZnO-c | Hexamethylenetetramine | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | DMZ | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TBC-3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon; NiO-np', 'SLG | ITO | Fullerene-SAM | Perovskite | CuSCN | Carbon', "SLG | ITO | TiO2-c | Perovskite | N2,N2,N3',N3',N6',N6',N7,N7-octakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2,3',6',7-tetraamine | Au", 'SLG | ITO | TiO2-c | Perovskite | PTAA | Au', 'SLG | ITO | PEDOT:MNSF | Perovskite | PCBM-60 | Ag', 'SLG | ITO | P3CT-Na | Perovskite | TPE-PDI4 | C60 | BCP | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | C60 | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P2 | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | C60 | BCP | Ag | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | MUTAB | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | B2F | C60 | Al', 'PET | ITO | SnO2-c | Al2O3-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MoO3 | TaTm | Perovskite | C60 | BCP | Ag', 'ITO | PTAA | Perovskite | PCBM-60 | Ti | Au', 'PET | ITO | PEDOT:PSS | 3-aminopropanoic acid-SAM | Perovskite | PCBM-60 | ZnO-np | Ag', 'PEN | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTP-1 | Au', 'SLG | ITO | NiO-np | Br-BPA-SAM | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YK1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Carbon', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | AZO | Ag', 'SLG | ITO | P3CT-Na | Perovskite | B2F | C60 | BCP | Ag', 'SLG | FTO | ZnO-c | Ethyl acetate | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 4,4′-(9-Methyl-9H-naphtho[2,1-c]carbazole-2,12-diyl)bis(N,N-bis(4-methoxyphenyl)aniline) | Ag', 'SLG | FTO | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | P3CT-Na | Perovskite | ITCP-M | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuSeCN | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Ca(acac)2 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | Ag-nw', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PANI | FTO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon | Spiro-MeOTAD', 'SLG | PEI | Au | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | PEDOT:PSS | PEI | Perovskite | PCBM-60 | BCP | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | AZO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | PTDPQ | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | HOOC-Ph-SH | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,6-cbz-EDOT | Au', 'SLG | FTO | Nb2O5-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P1 | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Zr(acac)4 | Au', 'SLG | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | ITO | CuInS2 | Al2O3-np | Perovskite | PCBM-60 | Ag', 'SLG | ITO | ZnO-c | WPF‐6‐oxy‐F | Perovskite | pTPA-DBTP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | AIGS-QDs | Carbon', 'SLG | ITO | TiO2-c | IPH | Perovskite | TaTm | F6-TCNNQ; TaTm | Au', 'MgF2 | SLG | FTO | TiO2-c | Perovskite | Perovskite-QD | PTAA | Au', 'SLG | FTO | TiO2-nanobundles | Perovskite | CF-BTz-ThR | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Au | Cu | MoOx', 'SLG | ITO | NiO-c | Perovskite | ITIC | BCP | Ag', 'SLG | FTO | NiO-mp | Perovskite | PCBM-60 | BCP | Al', 'Stainless steel | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Carbon-nt', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | IZTO', 'SLG | ITO | SnO2-np | Perovskite | Co3O4 | Carbon', 'SLG | FTO | Al:ITO-c | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | AZO', 'SLG | ITO | PEDOT:PSS | V2O5 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | Spiro-MeOTAD | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z34 | Au', 'PET | PEDOT:PSS | PEI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | bis‐C60 | Ag', 'SLG | ITO | JW7 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Q219 | Carbon', 'PS | ZnO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | a-DMEC70 | Al', 'SLG | ITO | ZnO-c | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | KAc', 'PET | Graphene; TETA | PTAA | PEDOT:PSS | Perovskite | ZnO-np | Graphene; TETA | PET', 'MgF2 | SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; PMMA | Perovskite | Spiro-MeOTAD | ITO | Au-grid | MgF2', 'SLG | ITO | H1 | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | PEDOT:PSS | Au-nanobipyramide; VOxs | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | NiMgLiO | 5-AVA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-a | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Al', 'SLG | ITO | C60 | Perovskite | FBA1 | MoO3 | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | Zn:SnO2 | Perovskite | CuPc | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | HOOC-C11H22-SH | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | pi-PFE4 | Ag', 'SLG | ITO | PTAA | Perovskite | IT-M | Zn(acac)2 | Ag', 'PET | Graphene | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Carbon-nt', 'SLG | ITO | PEDOT:PSS | PBDB‐T; ITIC | Perovskite | C60 | BCP | Ag', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,3'-(2,7-bis(bis(4-methoxyphenyl)amino)-9H-fluorene-9,9-diyl)bis(N-ethyl-N,N- dimethylpropan-1-aminium) bis(trifluoromethanesulfonyl)imide | Au", 'SLG | ITO | P3CT-Na | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Al | Ag', 'SLG | FTO | PEDOT:PSS | CuSCN | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | ZnO-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z28 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | AV-Carbon; MAI', 'SLG | ITO | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | ZnO-np | Al', 'PEN | ITO | TiO2-np | C70 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | p-DTS(FBTTh2)2 | MoOx | Ag', 'SLG | ITO | ZnO-c | Perovskite | Graphite | FTO', 'SLG | FTO | TiO2-c | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60; PTP-DFBT | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Mg | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsAc | Carbon', 'SLG | ITO | PTAA | PFN-P2 | Perovskite | LiF | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B[BMPDP]2 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | Al', 'SLG | ITO | DFTAB | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-mp | Perovskite | P3HT | Pt', 'SLG | FTO | Nb2O5 | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | Carbon', 'SLG | FTO | SnO2-c | Perovskite | asy-PBTBDT | Au', 'SLG | ITO | PTAA | Perovskite | C60 | SnO2 | IZO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbazole-based HTM | Au', 'SLG | ITO | AZO-np | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | cyclopenta[2,1-b; 3,4-b′]dithiophene | MoOx | SLG | ITO', 'SLG | FTO | TiO2-c | Perovskite | CuI | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Al2O3-mp | Me2N-DATPA | Ag', 'SLG | ITO | PEDOT:PSS | VOx | Perovskite | PCBM-60 | Al', 'SLG | ITO | SnO2-np | C60 | Perovskite | Spiro-MeOTAD | VOx | Au', 'SLG | ITO | PFN; ZnO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PCBM-60-mp | Perovskite | Spiro-MeOTAD | Ag', 'PET | PEDOT:PSS | ZnO-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPP-Ome | Au', 'Foil | AZO | PEIE | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | Perovskite | PTB7-Th | MoOx | Ag', 'SiO2-mp | SLG | ITO | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Au | Ag | MoOx', 'SLG | ITO | TiO2-c | Perovskite | CsOAc | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | CMO | Au', 'SLG | FTO | SnO2-np | Perovskite | PTAA | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | PPDIN6 | Ag', 'SLG | ITO | SnO2-np | Perovskite | PTT | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | 2,2’-BiPy | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DNA | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | PTAA | Au', 'SLG | ITO | PTAA | Perovskite | PDIN | Ag', 'SLG | ITO | C60 | Perovskite | TaTm | F6-TCNNQ; TaTm | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTPA-BDT | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Al', 'PEN | ITO | NiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-c | Perovskite | Spiro p-xylene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuP | Au', 'SLG | FTO | NiO-c | Perovskite | TiO2 | Ag', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | Ag-nw | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'PET | AZO | Ag | AZO | PEDOT:PSS | PolyTPD | Perovskite | PCBM-60 | Au', 'SLG | ITO | SnO2-np | Perovskite | Au', 'PEN | ITO | ZnO | PEIE | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | Perovskite | C60 | bis-C60 | ITO | Ni | Al', 'SLG | ITO | SnO2-nw | Perovskite | PTAA | Au', 'SLG | ITO | PEDOT:PSS | Porphyrin | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | ITO | SnO2-np | TPPO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | V1036:C4 | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Black phosphorous nanosheets | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI-ID(RS) | Al', 'SLG | ITO | SnO2-np | PbF2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | AZO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2,7-Ben | Au', 'SLG | FTO | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SFX-TPAM | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEI | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | DBFMT | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | PTCDA | PO-T2T | Cu', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | Graphite', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TAZ-[MeOTPATh]2 | Au', 'SLG | FTO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CZTSSe; rGO | Au', 'PET | ITO | SnO2-np | KCl | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-np | PEG; SnO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-np | Perovskite | C60; PCBM-60 | Ag', 'SLG | ITO | FrGO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPE-TPA-8A | MoO3 | Ag', 'SLG | ITO | (RuCp*mes)2 | C60 | Perovskite | TaTm | F6-TCNNQ; TaTm | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | Perovskite | HTM | Au', 'SLG | ITO | PEDOT | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | MAI | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | MoO3 | NPB | Perovskite | C60 | BCP | Al', 'SLG | ITO | PEDOT:PSS | VOx | Perovskire | PCBM-60 | Al', 'SLG | ITO | TFM | Perovskite | Carbon | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Al2O3 | Sym-HTPCH | Au', 'SLG | ITO | ZnO-c | Perovskite | CzPAF‐SBFN | Au', 'SLG | FTO | SnO2-c | Cs2SnO4 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Polyacrylonitrile | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Ni | NiMgO | PVP | Perovskite | PCBM-60 | ZnMgO | Al', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | PDINO | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | C70 | TmPyPB | Ag', 'SLG | ITO | P3CT | Perovskite | CPTA-E | Al', 'SLG | ITO | PolyTPD | Perovskite | PCBM-60 | C60 | BCP | Au', 'SLG | FTO | ZnO-np | Mg-EA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | FT-OMeTPA | Perovskite | PCBM-60 | BCP | Ag', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2,2'-[(4,5-Bis(2-ethylhexyl)-dithieno[2,3-d:2',3'-d']thieno[3,2-b:4,5-b']dipyrrole-2,7-diyl)-bis(3-hexylthien-5,5'-diyl)bis(methane-1-yl-1-ylidine)]dimalononitrile | Au", 'SLG | FTO | NiO-c | Perovskite | ZnO-np | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2TPA-1-DP | Au', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MnO3 | Ag', 'SLG | ITO | SnO2-np; ZnO-np | Al2O3-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | P3CT-K | CuZnS | Perovskite | PCBM-60 | ZnO | Al', 'SLG | FTO | TiO2-c | Zn2Ti3O8-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FT55 | Au', 'SLG | FTO | Zn2SnO4 | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PNVT-8 | ZnO | Al', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-029 | Ag', 'SLG | ITO | ZnO-nw | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | SnO2-np | Perovskite | Py-C | Au', 'SLG | TCO | TiO2-c | SrTiO3:TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | pBTTz | Ag', 'SLG | FTO | TiO2-mp | Perovskite | CuSCN | Carbon | FTO', 'SLG | ITO | NiO-c | PCDTBT | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | Nb2O5-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | C60-SAM | Al2O3-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | pi-PFE1 | Ag', 'SLG | FTO | NiO-c | Perovskite | CSCNT@Al2O3-c | CSCNT', 'Willow glass | AZO | SnO2-np | Perovskite | Spiro-MeOTAD | MoOx | Al', 'MgF2 | PET | ITO | 1‐benzyl‐3‐methylimidazolium chloride | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnSO | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | LiCoO2 | Perovskite | C60 | BCP | Ca | Al', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | PMMA | Spiro-MeOTAD | Au', 'SLG | FTO | CuI | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | NiO-c | Perovskite | NMPFP | Ag', 'SLG | ITO | C60 | Perovskite | F23 | Au', 'SLG | ITO | 2PACz | MeO-2PACz | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HMDI | Au', 'SLG | FTO | SnO2-c | Perovskite | ME6Bu-ZnPc | Ag', 'SLG | FTO | TiO2-nw | TiO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | TPCBM | PFN | Ag', 'SLG | FTO | TiO2-c | Perovskite | PT3HT | Ag', 'SLG | ITO | PEI | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | PTAA | CuGaO2-mp | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2TPA-3-DP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc | Carbon', 'SLG | ITO | C60 | Perovskite | SWCNTs | PTAA', 'SLG | FTO | TiO2-c | Perovskite | TAE1 | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | C8-BTBT | Perovskite | PCBM-60 | BCP I Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | C60 | BCP | Cu', 'SLG | ITO | Graphene oxide | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiS | Cr | Pt | FTO | SLG', 'MgF2 | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Carbon | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | C60-SAM | Perovskite | PCPDTBT | Ag', 'SLG | FTO | PEDOT | Perovskite | PCBM-60 | Au', 'SLG | ITO | NiCo2O4 | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | PTB7 | Perovskite | PCBM-70 | C70 | BCP | Ag', 'SLG | FTO | TiO2-np | CsBr | Perovskite | Carbon', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2 | BCP | Ag', 'SLG | FTO | TiO2-c | CdS | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | RCP-BTT | Au', 'SLG | ITO | PCT | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SWCNTs | PMMA | Au', 'PEN | ITO | NiO-np | Perovskite | PCBM-60 | TiO2-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | NbS2', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT; SWCNTs-PhOMe | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Cr | Au', 'SLG | ITO | SnO2-c; SWCNTs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CAS | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | BCP | AZO-c | Ag', 'SLG | FTO | TiO2-a | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | PDI-V | C60 | BCP | Ag', 'SLG | ITO | MoS2 | Perovskite | PCBM-60 | PFN | Al', 'SLG | FTO | ZTO | TiO2-mp | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | C60 | Perovskite | Thiophene | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene oxide | Au', 'SLG | ITO | TiO2-c | C60 | Perovskite | PDTSTTz | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTPA-OMeTPA | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCPDT-PDI | Al', 'SLG | FTO | Graphene | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBSD | Perovskite | HFB-OMeDPA | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | PFN | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | P4V4 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPor | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-mF | Au', 'SLG | FTO | Cu3PS4-np | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Fe3O4-np | Au', 'SLG | ITO | PTPD | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1‐adamantylamine | Spiro-MeOTAD | Au', 'SLG | FTO | ZnSO4-c | ZnSO4-mp | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | CuPc | PEI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEIE | PCBM-60 | Perovskite | Spiro-MeOTAD | MoOx | IO | ITO | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TDT-OMeTAD | Au', 'PET | ITO | SnO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Triazine-Th-OMeTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | CS05 | Au', 'PI | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | ZnO-np | Sinapoyl malate | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | Perovskite | P3OEGT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Si-QDs | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au | FTO', 'SLG | FTO | TiO2-nw | Sb2O3-np; SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PTAA | Perovskite | C60 | SnO2-c | ZTO | IZO', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | EFGnPs-H | Al', 'SLG | ITO | Graphene | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YT3 | Au', 'SLG | FTO | TiO2-c | TiO2-nanobeads | Perovskite | Spiro-MeOTAD | Au', 'SLG | AZO | Perovskite | P3HT | Au', 'SLG | FTO | ZnO-c | ZnO-nw | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Graphene-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | PTAA | MoO3 | Ag', 'SLG | ITO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-mp | Pbs-QDs | Perovskite | P3HT | Pt', 'SLG | FTO | BaCoF4 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60; F8BT | C60 | Al', 'SLG | ITO | SnO2-np | Perovskite | CI-GO | PTAA | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon-mp', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Rodhamine 101 | C60 | Rodhamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | WY-1 | Ag', 'SLG | ITO | CuPc | PEI | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | PCP‐Na | Perovskite | C60 | BCP | Al', 'SLG | FTO | TiO2-c | Perovskite | TPE-4DPA | Ag', 'SLG | ITO | GeO2 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | Perovskite | Al2O3 | CuPc(tBu)4 | Au', 'SLG | FTO | Graphene oxide | Perovskite | Graphene oxide | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | NDIF2 | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HS-Ph-SCH3 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | SP-12 | Au', 'SLG | FTO | Nb2O5 | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X1 | Au', 'SLG | FTO | TiO2-c | SnO2-c | Perovskite | PTAA | Au', 'SLG | FTO | C60 | Perovskite | FU7 | Au', 'SLG | ITO | PTAA | Perovskite | Ti | Au', 'SLG | FTO | SnO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | Carbon', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro TAD | MoO3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SiO2-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoO3 | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | IDTT-TPA | Au', 'SLG | FTO | Nb2O5 | (EMIM)PF6 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | PDI-T | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Au-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Na3C6H5O7 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | ZnO-np | Perovskite | Carbon', 'Carbon-nt-fiber | TiO2-c | TiO2-mp | Perovskite | P3HT; SWCNTs | Ag-nw', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2,7-Pyr | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | CBP | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | BCP | TiO2-c | Ag', 'SLG | FTO | SnO2-c | Perovskite | NP-SC6-TiOPc | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au | MoO3', 'SLG | FTO | TiO2-c | Perovskite | Ph-inv-OMeTPA | Au', 'SLG | ITO | TiO2-c | C60 | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-CN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | J61-ITIC | Carbon', 'SLG | ITO | P3CT-N | Perovskite | IT-4F | s-Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Poly(TA) | Perovskite | Poly(TA) | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | EH44 | Au', 'SLG | ITO | CuI | Perovskite | PCBM-60 | AZO-np | Ag', 'SLG | FTO | TiO2-c | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | Graphene | Graphene-QDs | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | SnO2-mp | Perovskite | P3 | Au', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Au-np | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CW5 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PDPT | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | F-R-COOK | Ag', 'SLG | FTO | SnO2-c | Perovskite | P3HT; SWCNTs | Spiro-MeOTAD | Au', 'Si | MgO | SrRuO3 | SrRuO3 | Perovskite | ITO', 'SLG | ITO | SnO2-np | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | PTAA | Ag', 'PET | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | TiO2 | Al', 'PET | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Ca | C60 | Perovskite | TPTPA | MoO3 | Ag', 'SLG | FTO | TiO2-c | Ag@TiO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Titanylphthalocyanine | Au', 'SLG | FTO | SnO2-np | Graphene-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | bis-PCBM; DMC | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | SnO2-np | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | Perovskite | TPE-NMe | Ag', 'PEN | ITO | Au-np | NiO-c | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PFO | Au', 'SLG | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | CuInS2 | Al2O3-np | Perovskite | Ag', 'SLG | ITO | CPE-K | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Al2O3 | CuSCN | Au', 'SLG | FTO | SnO2-c | Perovskite | CuEtPc | Au', 'Ti-wire | TiO2-c | TiO2-np | Perovskite | Spiro-MeOTAD | Au | Au-wire', 'SLG | ITO | SnO2-np | Perovskite | NiO-np | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Me-BPZTPA | Au', 'SLG | ITO | SnO2-np | Perovskite | PMAA; Spiro-MeOTAD | Spiro-MeOTAD | Au', 'SLG | ITO | C60-SAM | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene; TSHBC | Au', 'PET | ITO | Nb2O5; TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | SnO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-nanocones | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PTAA | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | NiO-np | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ni', 'SLG | FTO | PolyTPD | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | MEH-PPV | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | V2O5 | Au', 'SLG | ITO | SnO2-np | Perovskite | TFB | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | O5H-OMeDPA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag | ITO | Ag', 'SLG | FTO | TiO2-c | Graphene | Perovskite | Spiro-MeOTAD | Au', 'PEN | ITO | NiO-np | Perovskite | PCBM-60 | Al', 'PET | FTO | NiO-c | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-np | C60 | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | IPH | PFNBr | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | T101 | Au', 'SLG | ITO | NiO-c | NiO-c | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | IZO', 'SLG | ITO | PTAA | LiF | Perovskite | C60 | BCP | Ag', 'SLG | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | MoOx | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Pervskite | P3TAA-co-P3HT | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | BCP | cu', 'SLG | ITO | PEDOT:PSS | PFN | Perovskite | PCBM-60 | Zr(acac)4 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PDINO | Al', 'SLG | ITO | SnO2-c | Perovskite | TATSFHBP | Ag', 'SLG | ITO | H4 | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | C60; C70 | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | VB-DAAF | Perovskite | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Al2O3-np | CuBuPc | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | PDINO | Ag', 'SLG | ITO | ZnO-nanospheres | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | sGO | Perovskite | PCBM-60 | BCP | Ag', 'PEN | ITO | CdSe | PCBM-60 | Perovskite | CuPc | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | MgO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-c | C60 | Perovskite | PTAA | Au', 'SLG | ITO | P3CT-N | Perovskite | IT-4H | s-Bphen | Ag', 'SLG | ITO | N1,N3,N5-tris(4-n-butylphenyl)-N1,N3,N5-triphenylbenzene-1,3,5-triamine | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BEDN | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Zn(acac)2 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Graphene', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag-nanocubes | BCP | Ag | MoO3', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon', 'SLG | FTO | SnO2-n | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DHPT-SC | Au', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | P3HT | WO3 | Ag', 'SLG | ITO | ZnO-c; Graphene-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BChl-2 | Ag', 'SLG | ITO | TiO2-c | Perovskite | PTB7 | Au', 'SLG | ITO | Ag-np; NiO-c | Perovskite | PCBM-60 | AgAl', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PAH 2 | Au', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | MoOx | ITO', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | PFN | Ag', 'SLG | ITO | Graphene oxide | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,3′,5,5′-tetrasubstituted 1,1′-biphenyl | Au', 'PEN | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2 | Perovskite | V1160 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PFN | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA-1F | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CJ-02 | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | PEDOT:PSS', 'SLG | ITO | SnO2-np | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | F101 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | LiSPS | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Graphene | Perovskite | Spiro-MeOTAD | Au', 'FTO | TiO2-c | Perovskite | Carbon', 'SLG | FTO | NiO-c | MoOx | Perovskite | ZrO2-c | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon; NiS', 'SLG | ITO | Z7 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | In2O3 | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | DS2 | Ag', 'SLG | FTO | ZnO-c | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | IEICO; PBDB-T | MoOx | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-70 | ZnO-np | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP I Ag', 'SLG | FTO | TiO2-c | Perovskite | PCE-10 | MoO3 | Ag', 'SLG | FTO | PTPD | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | GO-nanoribbons | Perovskite | ZnO-np | Al', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | AZO-np | SnO2-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTBCPE | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZIF-8-20 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TT1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoO3 | CuPc | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | ITO | MgF2', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Mg | Ag', 'SLG | ITO | CdS | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MnS | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | MgO | Perovskite | Spiro-MeOTAD | Au', 'SLG | SWCNTs-HNO3 | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | ITO | SWCNTs | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | TiO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPBC | Au', 'PEN | ITO | NiO-np | Perovskite | PCBM-60 | Ag', 'SLG | ITO | NiO-c | Perovskite | Nb2O5 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | F-C60 | Ag', 'SLG | FTO | SnO2 | Perovskite | Spiro-MeOTAD | MoOx | IAI', 'SLG | ITO | SnO2-c | Perovskite | CZTPA-1 | MoO3 | Au', 'SLG | ITO | NiO-c | PTAA | Perovskite | C60 | SnO2 | BCP | Cu', 'Quartz | Graphene | C60 | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HS-Ph-NO2 | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PEI-HI | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | Perovskite | Carbon', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | SnO2-c | ITO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuSCN | rGO | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P5 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1,2-Bis[3,6-(4,4`-dimethoxydiphenylamino)-9H-carbazol-9-methyl]benzene | Au', 'SLG | ITO | BTF1 | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | C60-ETA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | Bphen | Ag', 'SLG | ITO | TiO2-c | Perovskite | P3HT | MoO3 | Ag', 'SLG | MPTMS-SMA | Ag | MUTAB-SAM | PCBM-60 | Perovskite | PTAA | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | Ag', 'SLG | ITO | SnO2-c | Perovskte | Spiro-MeOTAD | Ag', 'PET | ITO | TiS2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | MoO3 | Au', 'SLG | ITO | ZnO-np | Perovskite | PTAA | PEDOT:PSS | Au', 'SLG | ITO | SnO2-np; ZnO-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | Carbon-tape', 'SLG | ITO | TiO2-c | C60 | Perovskite | TOPO | TaTm | MoO3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H65 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM | Ag', 'SLG | FTO | AgAu-np | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NBNDD | Perovskite | PCBM-60 | Al', 'SLG | FTO | SnO2-c | Graphene | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'Willow glass | AZO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | 4-HI-PVK | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | PTAA | Spiro-MeOTAD | Au', 'SLG | ITO | C60 | Perovskite | SWCNTs', 'SLG | ITO | PTAA | PFN-P2 | Perovskite | C60 | BCP | Cu', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | PEIE | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Li4Ti5O12 | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Bphen | Al', 'SLG | FTO | TiO2-c | rGO; TiO2-nanofibers | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Aminocaproic acid; Caproic acid | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au | Ag-nw', 'SLG | ITO | P3CT-Na | Perovskite | TPE-DPP6 | C60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | PFN | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP; TZ-2 | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | PEIE | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-BPFN-TPA | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Azu-Bu | Au', 'SLG | Ag | MoOx | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | ITO', 'SLD | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PbS-QDs | Au', 'SLG | FTO | TiO2-c | TiO2-nt | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | FTO', 'SLG | ITO | PEDOT:PSS | Perovskite | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M3 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Triazine-Flu | Au', 'SLG | FTO | NiO-c | NiO-mp | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | A102 | Au', 'SLG | FTO | NiO | Perovskite | PTEG-1 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnBChl | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | F8BT | C60 | Al', 'SLG | ITO | TPAC-SAM | Perovskite | C60 | BCP | Ag', 'SLG | FTO | CuSCN | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | Alkoxy-PTEG | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | TPFB | Au', 'SLG | FTO | SrTiO3 | Graphene; Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V886 | Au', 'PET | ITO | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Perovskite | TPFPB | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-BP-TPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | OMeTPA-FA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoOx | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | EH44 | MoO3 | Al', 'SLG | ITO | NiO-c | Perovskite | PCBM-60; PDI-DA | Ag', 'SLG | ITO | ZnO-c | Perovskite | EDT; PbS | Au', 'SLG | ITO | Perovskite | Spiro-MeOTAD | Au | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | FTO | TiO2-c | SnO2-c | Perovskite | P3HT | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FA-MeOPh | Au', 'SLG | ITO | A-PDTON | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | ZnO-c | CPTA; PbI2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | ZnO-c | Ag', 'SLG | FTO | TiO2-c | Perovskite | CMP | Au', 'SLG | ITO | SnO2-np | DPC60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | C60 | Perovskite | TaTm | TaTm:F6-TCNNQ | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | pi-PFE2 | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | ZnO-np | AZO | NiAl | MgF2', 'SLG | ITO | CuAlO2 | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'Cu | CuI | Perovskite | ZnO-np | Ag-nw', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TBAI | Ag', 'SLG | ITO | PTAA | PFN | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | MoOx | ICO', 'SLG | ITO | Cu0.67Cr0.33O2 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | Nb2O5 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | PEDOT | Au', 'SLG | ITO | ZnO-c | ZnMgO-c | ZnO-nw | PCBM-60 | Perovskite | PEDOT:PSS | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | BCP | Ga2O3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | BL51 | Au', 'SLG | FTO | ZnO-c | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | ITO | MoS2 | Perovskite | C60 | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | BaTiO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | ZnO-c | Perovskite | CuI | Carbon', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | PffBT4T-2OD | WOx | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V1221 | Au', 'SLG | ITO | NiO-c | MOF-808 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PNDI20D-TT | ZnO | Al', 'SLG | FTO | TiO2-c | Perovskite | InP-np | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TP1 | Au', 'SLG | ITO | SnO2-np | Perovskite | Py-OMe | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2TPA-4-DP | Au', 'SLG | ITO | NiO-c | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-nanofibers | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Carbon | FTO | SLG', 'Ti-foil | TiO2-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS | TCA | PET:Ni-mesh', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MPA | Au', 'SLG | FTO | TiO2-np | Perovskite | Al2O3-np | TPA-ZnPc | Au', 'SLG | FTO | SnO2-c | Perovskite | CuMePy | Au', 'SLG | FTO | Nafion; PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | Ag-np; NiO-c | Perovskite | Poly(9-vinylcarbazole) | PCBM-60 | AgAl', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | PEDOT:PSS | PDMS', 'SLG | ITO | NPB | Perovskite | PCBM-60 | PDI-Br | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FU7 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | t-BPTI | C60 | BCP | Al', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | WOx | Au', 'SLG | ITO | r-GO-HBS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | SnO2-np | Perovskite | 3,6-triphenylamine-carbazole | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Au', 'SLG | ITO | SnO2-np | C60 | Perovskite | Spiro-TTB | VOx | Au', 'SLG | ITO | PTAA | Perovskite | LiF | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MWCNTs; ONC3', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | P3HT | MoO3 | Al', 'SLG | FTO | TiO2-c | TiO2-IO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | CoO | Perovskite | PCBM-60 | Ag', 'SLG | FTO | C60 | Perovskite | F4-TCNQ | CuPc | Au', 'SLG | ITO | TiO2-c | Perovskite | MEH; PPV | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2FBTA-2 | Au', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | PMMA | DTP-C6Th | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | NiO-mp | SWNCTs', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Y4 | Au', 'SLG | FTO | 1-ethyl-3-methylimidazolium iodide | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ATO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | PSQ2 | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | C12-silane-SAM | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TSHBC-CF3 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | ITO | MgF2', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | MAAc', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | mp-SFX-3PA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PDBD-T | Au', "SLG | ITO | SnO2-c | Perovskite | 4,4'-(5,10,11-Trihexyl-10,11-dihydro-5H-thieno[2′,3':4,5]pyrrolo [3,2-g]thieno[3,2-b][1,2,3]triazolo[4,5-e]indole-2,8-diyl)bis(N,N-bis(4- methoxyphenyl)aniline) | Au", 'SLG | FTO | TiO2-c | Perovskite | C102 | Au', 'SLG | AZO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | NiO-c | Perovskite | (4AMP)I2 | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuCrO2-np | Carbon', 'SLG | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | IDT6CN | Rhodamine 101 | LiF | Ag', 'SLG | ITO | ZnO-c | ZnO-mp | PEI | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | YC-1 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-np | PCBM-60 | Perovskite | M110 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TOPD | Ag', 'SLG | FTO | ZnSe | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | NiO-c | Perovskite | C60 | BCP | Ag', 'SLG | ITO | ZnO-c | C3-SAM | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | ZnO-np | ZnO-nanospheres | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC-2 | Ag', 'SLG | ITO | PCBM-60 | BCP | Perovskite | PEDOT:PSS | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | TrNBr | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | CdS | Perovskite | PTAA | Au', 'SLG | ITO | TiO2-c | C60-SAM | SiO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PCBM-60 | Perovskite | CuPc | Carbon', 'SLG | FTO | TiO2-nw | N719 | Perovskite | CZTS-np | Au', 'SLG | ITO | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-nanosphere | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MTA | Au', 'SLG | FTO | TiO2-c | Perovskite | po-TPE-4DPA | Ag', 'SLG | FTO | Perovskite | KY7F22-np | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PBDTT-FTTE | Ag', 'SLG | FTO | TiO2-c | Perovskite | MeO-PheDOT | Au', 'PEN | ITO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | PTAA | Perovskite | CMB | AZO | Ag', 'SLG | ITO | SnO2-c | SnO2-c | Perovskite | NiO-np | Ag', 'SLG | FTO | SnO2-c | Perovskite | EVA; SWCNTs | Spiro-MeOTAD | Au', 'SLG | ITO | Perovskite | PCBM-60 | PFN | Al', 'SLG | ITO | SnO2-np | Perovskite | 2-MP | Spiro-MeOTAD | Au', 'SLG | FTO | WO3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | Fe2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Porphyrin-H2 | Au', 'SLG | ITO | VOx | X-DVTPD | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | Perovskite | Carbon-nanowalls', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ATT-OBu | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPcNO2-OBFPh | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BDT:TT | Au', 'SLG | ITO | NiO-c | Perovskite | HATNAS3C7-Cs | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | Carbon-nt; PEI | Ag', 'SLG | ITO | PCBM-60; PEI | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | FTO | TiO2-c | TiO2-mp | Carbon-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Ag-np; TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO | Perovskite | PCBBz | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M1 | Au', 'SLG | ITO | PFBT-Na | Perovskite | PCBM-60 | Al', 'SLG | FTO | SnO2-c | Perovskite | PTAA | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Zr(acac)4 | Ag | CsF', 'SLG | FTO | TiO2-mp | Perovskite | V842 | Au', 'SLG | ITO | Spiro-MeOTAD | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | nPrO-DATPA | Au', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | PDBT-T1 | WOx | Au', 'SLG | FTO | TiO2-c | Perovskite | CuPc(tBu)4 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | iPrO-DATPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon; Graphite; PANI', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuInS2@ZnS-QDs | Carbon; LPP', 'SLG | FTO | TiO2-mp | Perovskite | V873 | Au', 'SLG | FTO | NiO-c | Perovskite | PPEA | PCBM-60 | BCP | Ag', 'SLG | ITO | PANI-PAMPSA | Perovskite | C60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | PTZDPP-2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Chl-1 | Ag', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag-nw | SLG', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | TiO2-c | PPDI-F3N | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TCl-PDI | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Carbon-nt | Au', 'SLG | ITO | Perovskite | PCBM-60 | Isopropanol-hydroquinolatolithium | Ag', 'SLG | ITO | NiO-np | PAS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-c | Perovskite | PDCBT | MoOX | Au', 'SLG | FTO | TiO2-c | Perovskite | Cu2CoSn4-np | Au', 'SLG | FTO | C60 | BenMeIM-Cl | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SO9 | Au', 'SLG | ITO | Ti3C2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | alfa-NPD | MoO3 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PH 1000', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuSCN | Au', 'SLG | FTO | Nb2O5-c | Perovskite | CuPc | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,6 BCz-OMeTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | tetra-TPA | Au', 'SLG | FTO | Graphene oxide; TiO2-np | Perovskite | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Au', 'SLG | FTO | TiO2-c | Perovskite | PTAA | MoOx | Ag', 'SLG | ITO | Perovskite | PCBM-60 | Al', 'SLG | ITO | PCBM-60 | EGME | Perovskite | PTAA | Au', 'SLG | ITO | NiO-c | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P1Z1 | Carbon', 'SLG | FTO | SnO2-np | CPTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PEI | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEA2PBI4 | Carbon', 'SLG | ITO | TN-P | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-nanosheets | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | PMMA | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PFN-Br | Ag', 'SLG | ITO | Cu2O | Perovskite | C60 | BCP | Ag', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC-Th | Rhodamine 101 | LiF | Ag', '42P2O5-22Li2O-22ZnO-12Sm2O3-2CeO2 | FTO | Perovskite | Metal', 'SLG | ITO | NiO-np | Glycerol | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnO-c | Urea | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Perovskite | P3HT | Ag', 'SLG | ITO | NiO-c | PNP-BC | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au | FTO', 'SLG | ITO | CuOx | Perovskite | C60 | BCP | Ag', 'SLG | ITO | ZnO-c | Perovskite | PZn-DPPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | NiO-mp | Perovskite | Carbon', 'SLG | ITO | SnO2-c | C60-5f | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | WO3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | PTAA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Al', 'SLG | ITO | NiO-np | Perovskite | PEAI | PCBM-60 | BCP | Ag', 'SLG | Ti | Au | Perovskite | ITO', 'SLG | FTO | TiO2-c | Perovskite | DFBT(DTS-FBTTh2)2 | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Th-PDI | Au', 'SLG | ITO | NiO-c | 1bb | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PCBM-60; PEI | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PTAA | Perovsktie | ZnO-np | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | F-PDI | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | Perovskite | Au', 'SLG | ITO | PASQ-IDT | Perovskite | C60 | BCP | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | 1,2-diaminoethane | C60 | BCP | Ag | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PVAc | Graphene oxide | Carbon', 'SLG | FTO | ZnO-c | AZO-mp | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nt | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Phen-NaDPO; Sn(SCN)2 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Au', 'SLG | ITO | NiO-c | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | NbOH5 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | CuO | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PMMA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TAZ-[MeOTPA]2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Cu; Cu2O', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Au', 'PET | Graphene; TETA | ZnO-np | Perovskite | PTAA | PEDOT:PSS | Graphene | PET', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Carbon-nt', 'SLG | FTO | PEDOT:PSS | Perovskite | PV-PDI | Al', 'SLG | ITO | CuI | Cu@CuI-nw | Perovskite | PCBM-60; ZnO-np | Ag', 'SLG | ITO | Graphene | AuCl3 | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | SnO2-c | TiO2-mp | PbI2 | Spiro-MeOTAD | Au', 'SLG | ITO | C60 | Perovskite | H1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DPA-TPM | Au', 'SLG | FTO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTB-BO | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SM01 | Ag', 'PET | ITO | PTCA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | r-GO-NH | Perovskite | PCBM-60 | Ag', 'SLG | ITO | SnO2-np | Perovskite | PMAA; Spiro-MeOTAD | Spiro-MeOTAD | AgAu', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | PEI | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Au', 'PET | Ni-mesh:PH1000 | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TPP-OMeTAD | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cz-Pyr | Au', 'SLG | FTO | TiO2-c | Perovskite | MeO-TPD | Au', 'SLG | ITO | CuO | Perovskite | PCBM-70 | AZO | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Mg | Al', 'SLG | ITO | Porphyrin | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S2 | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | IZO | Au', 'SLG | ITO | CPE-Na | Perovskite | PCBM-60 | Al', 'PEN | ITO | C60 | PAA | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | N2200 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | SQ2 | Perovskite | Spiro-MeOTAD | Au', 'SiO2-hollow | SLG | ITO | TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC04 | Ag', 'SLG | ITO | SnO2-np | C60 | Perovskite | Spiro-MeOTAD | VOx | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MWCNTs; ONC2', 'SLG | ITO | ZnO-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | ZnO-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-c | Perovskite | IT-4F; PBDB-T-SF | MoO3 | Ag', 'Phosphor-in-glass | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-N | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | SWCNTs | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | PVP | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | THY-4 | Au', 'SLG | ITO | SnO2-np | Perovskite | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | C60 | BCP | Ag', 'PET | ITO | PTAA | Perovskite | PVDF-TrFE | C60 | BCP | Al', 'SLG | ITO | NiO-np | Perovskite | NDI-TriSe | Ag', 'SLG | ITO | NiO-c | Perovskite | ZnO-np | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Au | MoO3', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEDOT:PSS | Carbon', 'SLG | ITO | Cu2O | Perovskite | PCBM-60 | Au', 'SLG | ITO | PTAA | Perovskite | C60 | PEI | Ag', 'SiO2 | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | AgAl', 'SLG | FTO | SnO2-c | Perovskite | PTEG | Ag', 'SLG | ITO | C60 | C60-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Al2O3-mp | Perovskite | Au', 'SLG | FTO | Graphene oxide | Perovskite | Graphene oxide; TiO2-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Au-np; NiO', 'SLG | ITO | V2O5 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | Rubrene | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C70 | Ag', 'SLG | FTO | TiO2-c | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TPE-S | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | BL52 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PhCOOH | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Cr | Au', 'SLG | ITO | TiO2-c | PCBB-2CN-2C8 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | sGO | Perovskite | PCBM-60 | Ag', 'SLG | FTO | AZO-np | Perovskite | PCE-10 | MoO3 | Al', 'SLG | ITO | WO3-nw@PEDOT | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | LD29 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SM-1 | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPcNO2-OPh | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag,', 'SLG | FTO | TiO2-c | TiO2-mp | 5-aminovaleric acid | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | C70 | BCP | Ag', 'SLG | ITO | In2O3-c | Perovskite | Spiro-MeOTAD | Au', 'Ag-nw; Graphene; Polycarbonate; Polystyrene | MoO3 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | Au-np | NiO-c | Perovskite | PCBM-60 | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | TiO2-c | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X62 | Au', 'SLG | ITO | C60; PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60; PS-b-PEO | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | Ru(acac) | Au', 'SLG | FTO | TiO2-c | Perovskite | P3HT:Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | PSS-Na | Perovskite | PCBM-60 | PEI | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TbT-2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-405 | Au', 'SLG | FTO | SnO2-QDs | Perovskite | CsBiBr3-QDs | Carbon', 'SLG | FTO | TiO2-c | SnO2-np | Perovskite | PDO2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | CBA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 5,7-disubstituted azulene | Au', 'SLG | FTO | C60 | Perovskite | Carbon', 'SLG | FTO | NiO-c | Perovskite | C60-HPy | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | CuMePc | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | Spiro-MeOTAD | Pd', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | CS04 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | 1,2 ethanedithio | Perovskite | PZn-3FTPA | Au', 'SLG | FTO | ZnO-c | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-Tetra | Au', 'SLG | ITO | PTAA | Perovskite | Ag', 'PEN | ITO | NiO-np | Perovskite | C60 | BCP | Ag', 'SLG | FTO | C3N5 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ATT-OMe | Au', 'SLG | ITO | ZnO-c | WPF‐6‐oxy‐F | Perovskite | pDPA-DBTP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | FAAc', 'SLG | ITO | NiO-np | PSS | Perovskite | PSS | PCBM-60 | ZnO-np | ITO', 'SLG | ITO | PFN-2TNDI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z1012 | Au', 'SLG | FTO | SnO2-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | Perovskite | PABA | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MWCNTs; ONC1', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FA-CN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X36 | Au', 'SLG | Ag-nw; Graphene oxide | PEDOT:PSS | Perovskite | PCBM-60 | PFN-P1 | Ag', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Fused-F (Tris[[4-[3,3'-dihexylsilylene-2,2'-bithiophene]-7-[5′′-n-hexyl-(2,2′; 5′,2′′-terthiophen\ne)-5-yl]-benzo[c]-[1,2,5]thiadiazole]-2,6,10-yl]-4,4,8,8,12,12-hexamethyl-4H,8H,12\nHbenzo[1,9]quinolizino [3,4,5,6,7,-defg]acridine ) | Au", 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CW3 | Au', 'PET | PEDOT:PSS | Ag-nw | SnO2-np | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuCo2O4 | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | ITO | PTAA | Perovskite | C60 | BCP | Cu', 'Au | Ni | Al | Ti | GaN | Perovskite | hexagonal-BN | Perovskite | Graphene aerogel | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | IPFB | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | POZ6-2 | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | LiF | Al', "SLG | FTO | TiO2-c | Perovskite | 9,9'-([1,2,5]thiadiazolo[3,4-c]pyridine-4,7-diylbis(4,1- phenylene))bis(N3,N3,N6,N6-tetrakis(4-methoxyphenyl)-9H-carbazole-3,6-diamine) | Au", 'SLG | FTO | CdSe | Perovskite | CuPc | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | NiO-c | Perovskite | PCBM-60 | Au | ITO | PEN', 'Carbon-nt-fiber | TiO2-c | TiO2-mp | Perovskite | P3HT; SWCNTs | Ag-nw | PMMA', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P2Z1 | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | TPE-DPP12 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnP | Au', 'SLG | ITO | PolyTPD | PFN | Perovskite | LiF | C60 | BCP | Au', 'SLG | FTO | SnO2-c | Perovskite | CuPrPc | Au', 'SLG | ITO | YC-3 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | Ba0.9Sr0.1SnO3-np | Perovskite | PTAA | Au', 'SLG | ITO | ZnO | MgZnO | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | PEDOT:PSS | Al', 'SLG | ITO | TPA-NAP-TPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P1 | Au', 'SLG | ITO | C60:Phlm | C60 | Perovskite | TaTm | TaTm:F6-TCNNQ | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | ZnO-c | Al', 'SLG | ITO | TiO2-c | PCBA | Perovskite | Spiro-MeOTAD | WO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | IEICO | MoO3 | Ag', 'SLG | FTO | ETM | Perovskite | HTM | Metal', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Chl-2 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au | Al2O3-c', 'SLG | ITO | PEDOT:GSL | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Pyrimidine | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B58 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Sinapoyl malate | Perovskite | Co(II)P; Co(III)P | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H1 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Br-PDI | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | D-C60 | Al', 'SLG | FTO | TiO2-nc | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | C60 | BCP | Al', 'PDMS | SLG | ITO | PTAA | Perovskite | C60 | BCP | Cu', 'SLG | ITO | Spiro-MeOTAD | PEDOT:PSS | Perovskite | C60 | Ag', 'SLG | ITO | 2EGO-PPV | PFN-P2 | Perovskite | PCBM-60 | BCP | cu', 'SLG | FTO | NiO-c | Perovskite | Fulleropyrrolidinium Iodide | Ag', 'SLG | ITO | ZnO-c | SnO2; TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuPc | PTAA | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | DMEC-60 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H112 | Au', 'SLG | FTO | TiO2-c | Perovskite | 3-Butylthiophene | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PTCBI | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60-MCM | Ag', 'SLG | FTO | BCP | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | ZnO-np | Perovskite | Ag', 'PET | ITO | ZnO-np | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z3 | Au', 'SLG | FTO | SnO2-mp | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | ZnO-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-MeOPh | Au', 'SLG | FTO | SnO2-c | SnO2-c | Perovskite | V1056 | Au', 'PET | ITO | PEDOT:PSS | Perovskite | C60 | Al', 'SLG | FTO | TiO2-c | Perovskite | PO-Spiro | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Pt', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TPBi | Ag', 'SLG | FTO | SnO2-np | Perovskite | NiO-c | FTO', 'PET | WO3 | Ag | WO3 | PEDOT:PSS | Perovskite | C60 | Bphen | Ag', 'SLG | ITO | NPB; PTAA | Perovskite | PCBM-60 | Al', 'SLG | FTO | CuSCN | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEDOT | Au', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | PDCBT | WOx | Au', 'SLG | ITO | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | PIDT-DFBT | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | TiO2-c | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | Ca | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M3; PCBM-60 | Au', 'SLG | FTO | ZnO-c | Perovskite | P3HT | Au', 'SLG | FTO | NiCoO4 | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO-c | Carbon', 'SLG | FTO | TiO2-c | AAO | Perovskite | MoOx | ITO', 'SLG | FTO | TiO2-c | Perovskite | MWCNTs; Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | BPB-M | LiF | Ag', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | PEDOT:PSS | PEDOT:PSS | PDMS', 'SLG | FTO | ZnO-c | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | FTO | TiO2-c | Perovskite | Pyridine | Spiro-MeOTAD | Au', 'Mica | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | NiO-c | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | NaYF4-np; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Carbon-QDs | Au', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | IPH | Ba | Ag', 'SLG | Graphene | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | ITO | SnO2-np | Perovskite | EHCz-MeFl | MoOx | Al', 'SLG | ITO | NiO-np | Perovskite | NDI-Se | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTh; Graphene | Au', 'SLG | ITO | P3CT-Na | Perovskite | TPE-DPP4 | C60 | BCP | Ag', 'Ti-wire | TiO2-nt | Perovskite | Spiro-MeOTAD | Carbon-nt', 'SLG | FTO | NiO-c | Perovskite | TEAI | PCBM-60 | PEI | Ag', 'SLG | ITO | PFN-OX | Perovskite | Spiro-MeOTAD | Au', 'SLG | Ag-nw | Graphene oxide | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | NaYF4:Yb:Tm-np | Perovskite | NaYF4:Yb:Tm-np | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | ZnO-np | Al', 'SLG | ITO | TiO2-c | CdSe-tetrapod | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | NiO-mp | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | ITO | MC8-9-NPC | Perovskite | PCBM-60 | LiF | Al', 'PEN | SWCNTs | MoO3 | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Cz-N | Au', 'SLG | ITO | P3CT | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | W', 'SLG | ITO | NiO | PS | Perovskite | PCBM-60 | Ag', 'PEN | ITO | Al2O3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-mp | Perovskite | HfO2 | PTAA | Graphite', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TB-ZnPc | Au', 'SLG | ITO | PolyTPD | Perovskite | PCBM-60 | BCP | Al', 'SLG | ITO | SnO2-np | Perovskite | PTAA | MoO3 | Al', 'SLG | FTO | TiO2-c | Perovskite | NiO-c | Ni', 'SLG | FTO | TiO2-c | Perovskite | Au-np; P3HT | Au', 'SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon | Graphite', 'SLG | ITO | PMA | Perovskite | C60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Si-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuPc | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; PMMA | BAI | Perovskite | BAI | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | Perovskite | P3HT | P3HT; PMMA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTA1 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | C60 | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | ZnO-nw | Al2O3-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | BCP | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-410 | Au', 'SLG | ITO | TiO2-c | NAMF-H | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | SnO2-mp | Perovskite | Carbon', 'SLG | FTO | BaSnO3 | Perovskite | NiO | FTO | SLG', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Bis-FIMG | Ag', 'SLG | ITO | PTB7-Th | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPcNO2-OBFPh | Au', 'SLG | ITO | PTAA | Perovskite | IDIC | C60 | BCP | Cu', 'SLG | ITO | P3CT-Na | Perovskite | IDT6CN-TM | BCP | Ag', 'SLG | FTO | Au-np | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | EGO-PPV | Perovskite | PCBM-60 | BCP | cu', 'SLG | FTO | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | ZnO-c | Perovskite | Co-Porphyrin | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | ITO | NiO-c | NBP-BC | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | c-HATNA | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphite | Pt', 'SLG | ITO | PEDOT:PSS | Perovskite | P2 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X2 | Au', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon Paper', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag | MoO3', 'SLG | ITO | PEDOT:PSS | Perovskite | DS1 | Ag', 'SLG | ITO | C60; Phlm | C60 | Perovskite | TaTm | F6-TCNNQ; TaTm | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Bphen; Ir(MDQ)2(acac) | Ag', 'SLG | FTO | TiO2-c | Perovskite | NiO@C', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Pt', 'SLG | ITO | P3CT | Perovskite | PFPDI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoS2-QDs | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZIF-8-10 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B63 | Au', 'SLG | FTO | TiO2-c | TiO2-np | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PCBM-60 | BCP | Perovskite | PEDOT:PSS | Al | Ag', 'SLG | ITO | P3CT-Na | Perovskite | IDTCN | C60 | BCP | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | PEI | D-Sorbito; PEDOT:PSSl | Ag-nw | PET', 'SLG | ITO | SnO2-np | PCBA-60 | Perovskite | PTA | MoO3 | Ag', 'SLG | ITO | SnO2-np | Perovskite | EHCz-3EtCz | MoOx | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PVDF-HFP | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | IDT6CN | C60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | C60 | Ag', 'SLG | ITO | PEDOT:PSS | Al2O3-mp | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPD-4MeTPA | Au', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | AZO-c', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | TiO2 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | T103 | Au', 'SLG | ITO | PEDOT:PSS | PbI2 | Perovskite | PCBM-60 | Ag', 'SLG | ITO | EDTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | LiF | C60 | SnO2-c | ITO | Ag', 'SLG | ITO | Perovskite | PCBM-60 | BCP | Al', 'PET | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG  FTO | TiO2-c | TiO2-mp | Perovskite | BTX-OMeTAD | Au', 'SLG | ITO | ZnO-c | ZnO-nw | Perovskite | MDMO-PPV | MoO3 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PDI | C60 | BCP | Al', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MoOx | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SWCNTs | Ag', 'SLG | FTO | TiO2-c | ZnO-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Perovskite | PCBM-60-PCP | Ag', 'SLG | ITO | CuCrO2 | C60; PCBM-60 | Perovskite | Zr(acac)4 | Ag', 'SLG | FTO | TiO2-c | SiO2-np | TiO2-c | SiO2-np | TiO2-c | SiO2-np | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PPDIDTT | PCBM-60 | Bphen | Ag', 'SLG | ITO | SrTiO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTT-1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM-P1 | Au', 'SLG | ITO | ZnO | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | ITO | NiO-c | UiO-66 | Perovskite | PCBM-60 | BCP | Ag', 'PEN | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | SnO2-c | Perovskite | SWCNTs | Spiro-MeOTAD | Au', 'SLG | AZO | ZnO-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SY2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Tetrakis-Triphenylamine | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | T(EDOT-TPA)2 | Au | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 1a | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-epoxy | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZnS | Perovskite | PTAA | Au', 'SLG | ITO | TAPC | Perovskite | C60 | BCP | Ag', 'PEN | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | ZSO-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | PTAA | Perovskite | PCBM-60 | BCP | Au', 'PET | ITO | ZnO-c | Perovskite | PTAA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | Al', 'SLG | ITO | NiO-np | Perovskite | ICBA | PrC60MAI | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | P(NDI2OD-T2) | Ag', 'SLG | ITO | SnO2-np | C60 | Perovskite | BTF1 | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | PrCMA | Ag', 'PAA-PEG | Ti | PANI | Perovskite | PCBM-60 | Ti-grid', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | SnO2-c | SnZnO | Al', 'SLG | ITO | Graphene oxide; ZnO-np | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SQ2 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | P(NDI2OD-TET) | Al', 'SLG | ITO | PTAA | PFN | Perovskite | PCBM-60 | PFN | Cu-CFNs', 'SLG | ITO | ICMA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 4-(4-Phenyl-4-alfa-naphthylbutadienyl)-N,N-di(4-tolyl)-phenylamine | Au', 'SLG | ITO | PTAA | Perovskite | ICBA | C60 | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | BCP | Cu', 'MgF2 | Willow glass | AZO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | ITO | MoO3 | TPBi | TaTm | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | Ag', 'SLG | FTO | TiO2-c | Perovskite | KY7F22-np | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; PTAA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | MgO-c | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NTPA | Perovskite | PCBM-60 | Ag', 'SLG | ITO | SnO2-c | C60-5g | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nw | Perovskite | CuI | Au', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PFB | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon Black | Carbon', 'SLG | ITO | CuInS2 | Al2O3-mp | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PFN; PTPD | Perovskite | PCBM-60 | Al', 'SLG | ITO | CuS-np | Perovskite | C60 | BCP | Ag', 'SLG | ITO | ZnO-c | Perovskite | PZn-TPA | Au', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MWCNTs', 'SLG | Graphene | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-c | Ag-nw', 'SLG | ITO | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-nw | ZrO2-mp | Perovskite | Carbon', 'SLG | ITO | TiO2-c | Perovskite | Au', 'SLG | ITO | CuSCN-2D | Perovskite | C60 | Bphen | Ag', 'SLG | ITO | SnO2-c | C60 | Perovskite | PPyra-ACD | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | PSS-Na | Perovskite | PCBM-60 | ZnO | PEI | Al', 'SLG | FTO | SnO2-c | C60 | Perovskite | PMMA | DTPC8-ThDTPA | Au', 'SLG | ITO | NiO-c | Perovskite | CeOx | Ag', 'SLG | FTO | SnO2-c | Perovskite | NiO-c | Ni', 'NOA88 | PEI | Au | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | MoO3 | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | ITO | TPASB | Perovskite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PN4N | Al', 'SLG | FTO | C70 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | IZO | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Poly TPD-NPD | Perovskite | PCBM-60 | BCP | Ag', "SLG | ITO | TiO2-c | Perovskite | N3',N3',N6',N6'-tetrakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-3',6'-diamine | Au", 'SLG | ITO | C60 | Perovskite | IDF-DiDPA | MoO3 | Ag', 'SLG | FTO | TiO2-mp | Perovskite | Carbon | PEDOT:PSS | FTO | SLG', 'PET | Carbon-nt; P3HT | PEDOT:PSS | Perovskite | Carbon-nt:PCBM-60', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z26 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MWCNTs; Spiro-MeOTAD | Au', 'SLG | ITO | MeO-PPV | PFN-P2 | Perovskite | PCBM-60 | BCP | cu', 'SLG | FTO | TiO2-c | Perovskite | 2F | Au', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | Bphen | Al', 'PEN | ITO | SnO2-np | Perovskite | PTAA | NiO-c | Au | PEN', 'SLG | FTO | CZTS | Perovskite | PCBM-60 | Ag', 'SLG | resist | ITO | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | NiO-c | Perovskite | PEAI | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PbPc | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | AZO-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; Poly(N-vinylcarbazole) | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuOx | Perovskite | PCBM-60 | Ag', 'SLG | ITO | C60 | Perovskite | H2 | Au', 'SLG | ITO | DFH | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PFN; PTPD | Perovskite | PCBM-60 | TiO2 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | CNTPA-PCBM | bis-C60 | Ag', 'SLG | Ag | TiO2-c | Perovskite | Au', 'SLG | FTO | TiO2-c | SiO2-np | TiO2-c | SiO2-np | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | MoO2 | ITO', 'SLG | ITO | PTPD | PFN | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TSHBC | Au', 'PEN | ITO | SnO2-np | Perovskite | PTAA | NiO-c | ITO | PEN', 'PET | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YT2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | T-MWCNTs', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | V1004 | Ag', 'SLG | ITO | ATO | TiO2-c | TiO2-mp | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | AS44 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SnS-np; ZnS-np | NiO-np | Carbon', 'SLG | ITO | NiO-c | Perovskite | PbS | PCBM-60 | Al', 'SLG | FTO | ZnO-c | ZnO-mp | ZrO2-mp | Perovskite | Carbon', 'SLG | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | SnO2-np | BBA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | H-Z1 | MoO3 | Ag', 'SLG | ITO | ZnO-c | ZnO-nw | Perovskite | Rubene | P3HT | Ag', 'SLG | FTO | TiO2-c | Perovskite | PbS-QDs | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3TI | Au', 'SLG | ITO | ZnO-np | ATAA | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | Mg0.05Zn0.95O-np | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PANI | Carbon', 'SLG | ITO | NiO-c | Perovskite | N2200; PFN-Ox | bis-C60 | Ag', 'SLG | ITO | PCBM-60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | Au | pV3d3 | Al2O3', 'SLG | FTO | NiO-c | Perovskite | C60-BPy | BCP | Ag', 'SLG | ITO | p-PFP-O | PTAA | Perovskite | PCBM-60 | PFN | Ag', 'SLG | ITO | ZnO-np | SnO2-np | Perovskite | PSQ1 | MoO3 | Ag', 'SLG | ITO | SnO2-np | TiO2-np | Perovskite | PTAA | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PDTP-DFBT:PCBM-60 | Ca | Al', 'SLG | FTO | ZnO-c | ZnO-nw | Perovskite | P3HT | Ag', 'SLG | ITO | PTAA | Perovskite | PCBB-S-N | C60 | BCP | Cu', 'SLG | ITO | TPAC2M | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | 1-ethyl-3-methylimidazolium iodide | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perylene | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO-np | ITO | MgF2', 'SLG | ITO | TiO2-c | DMF | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', "SLG | ITO | SnO2-c | Perovskite | 4,4'-((2-Hexyl-2H-benzo[d][1,2,3]triazole-4,7-diyl)bis(thiophene5,2-diyl))bis(N,N-bis(4-(hexyloxy)phenyl)aniline) | Au", 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | WT3 | MoOx | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M4 | Au', 'SLG | FTO | TiO2-c | Perovskite | PTB7 | MoO3 | Ag', 'Flexible | IZO | PEIE | C60 | 3TPYMB | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | Perovskite | PQT-12 | Au', 'SLG | ITO | V2O5 | Perovskite | PCBM-60 | ZnO | Ag', 'PET | ITO | P3CT-Na | Perovskite | PCBM-60 | HDAC | Ag', 'SLG | FTO | TiO2-mp | N719 | Perovskite | Pt | SLG | FTO', 'SLG | ITO | WOx | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | SnO2-np | Choline Chloride | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-nw | SiO2-nanocolumns | TiO2-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-nanoballs | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Spiro-MeOTAD | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PDVT-10 | Au', 'SLG | FTO | NiO-c | Perovskite | SnO2-c | AZO-c', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Porphyrin-H1 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | TiO2-c | BaSnO3-np | Perovskite | PTAA | Au', 'SLG | ITO | SnO2-np | Perovskite | OMe-TATPyr | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PM-Spiro | Au', 'Ti | TiO2-nw | Perovskite | PEDOT | ITO | PEN', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPB-2-MOTPA | Au', 'SLG | ITO | PTAA | Perovskite | Cu', 'SLG | FTO | NiO | Perovskite | PCBM-60 | LiF | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | CMB-vTA | AZO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YC03 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | S-acetylthiocholine chloride | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | SnO2-c | IZO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 4-(2-(4-(Bis(4-(hexyloxy)phenyl)methyl)phenyl)-9-methyl-9H-naphtho[2,1-c]carbazol-12-yl)-N,N-bis(4-(hexyloxy)phenyl)aniline | Ag', 'SLG | FTO | TiO2-c | Perovskite | BP-DC | Ag', 'SLG | ITO | PolyTPD | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | PEDOT:PSS | X-QUPD | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Mix-DMEC70 | Al', 'SLG | FTO | SnO2-c | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | ITIC | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 4,8-bis-\n(5-bromothiophene-2-yl)-benzo thiadiazole | Au', 'SLG | ITO | Pentacene | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | AZO-np | SnO2-c | ITO', 'SLG | FTO | PEDOT:PSS | Perovskite | PC61BEH | Al', 'SLG | FTO | TiO2-c | Al2O3-mp | NiO-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Ca | C60 | Perovskite | TAPC | MoO3; TAPC | Ag', 'SLG | ITO | YC-2 | Perovskite | PCBM-60 | BCP | Ag', 'PET | MSA-PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | C60 | Rhodamine 101 | LiF | Ag', 'SLG | ITO | ZnO-c | PCBM-70 | Perovskite | Ag', 'SLG | FTO | TiO2-c | Perovskite | P3HT | Graphite', 'SLG | ITO | NiO-c | Perovskite | Nb2O5 | PCBM-60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V950 | Au', 'Cellulose paper | Carbon black | Perovskite | C60 | BCP | Cu | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Y2A2 | Au', 'SLG | ITO | LiQ; PEIE | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Z8 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | C60-SAM | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | B-DIPBI | Al', 'SLG | FTO | TiO2-c | Alq3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | Ca | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR360 | Au', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | PTAA | Au', 'SLG | FTO | Cu2O | Perovskite | SiO2-c | ZnO-c | Ag', 'SLG | ITO | Spiro-E | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-nanoleaves | Perovskite | Spiro-MeOTAD | Graphite', 'SLG | ITO | s-PANI:PSS | Perovskite | PCBM-60 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PDPP-3T | MoOx | Ag', 'Willow glass | Ti | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | IZO', 'PET | In2O3 | ZnO-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | CeO2-c | TiO2-c | Perovskite | CuSCN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | La2O3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | Graphene-QDs | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | asy-PBTBDT | Au', 'SLG | FTO | NiO-c | Perovskite | MAI | PCBM-60 | PEI | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | bis-PCBM | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | DMEC-70 | LiF | Al', 'SLG | ITO | Graphene oxide; PEDOT:PSS | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | Perovskite | P3HT | MoO3 | Ag', 'PET | ITO | Graphene | ZnO-QDs | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | Spiro-TAD | PEDOT:PSS | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | PCBM-60 | Perovskite | PCDTBT1 | MoO3 | Au', 'SLG | ITO | PEDOT:PSS | Perovsksite | PCBM-60 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TmPyPB | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | F16CuPc | Bphen | Al', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Ag', 'SLG | FTO | KY7F22-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | Perovskite | Bifluo-OMeTAD | MoO3 | Ag', 'SLG | ITO | C60 | Perovskite | IDF-TeDPA | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Benzylamine | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO-np | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Ag-nw', 'SLG | FTO | TiO2-mp | N719 | Perovskite | Dye | I2; KI; Propylene carbonate; Polyethylene glycol | Pt | FTO', 'SLG | FTO | Cu0.2Cr0.8O2 | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-AZO | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-404 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | TOPO | Carbon', 'SLG | ITO | TPA | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | ZnO-nw | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CZTS; rGO | Au', 'SLG | ITO | PTAA | Perovskite | C60 | B4PyMPM | Ag', 'Willow glass | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | FTO | NiO-c | n-Butylamine | Perovskite | C60 | BCP | Ag', 'SLG | ITO | SnO2-c | Perovskite | OMe-TATPyr | Au', 'MgF2 | PET | ITO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Carbon-QDs | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO | Perovskite | PCBM-60:C60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | PbSe | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | NiO-np | Graphene oxide | Perovskite | GO-Li | TiO2-c | Al', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MnOx | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | T5H-OMeDPA | Au', 'SLG | FTO | TiO2-c | MgO | TiO2-mp | ZrO2-mp | NiO-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Triazine | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | LiF | Al', 'SLG | ITO | P3CT-Na | Perovskite | ITCPTC-Se | Rhodamine 101 | LiF | Ag', 'PES | PEDOT:PSS | Ag | PEI | PCBM-60 | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | FTO | TiO2-c | CsPbBr3-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | PEI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | P3HT | Ag', 'PET | TiO2-c | Ag | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TPP-SMeTAD | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CT3 | MoO3 | Ag', 'SLG | ITO | SnO2-c | C60-5e | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | Graphene', 'SLG | FTO | Perovskite | HTM | Ag', 'SLG | FTO | TiO2-c | Perovskite | PHPT-py | Au', 'SLG | FTO | TiO2-c | Perovskite | Graphene', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | Ca | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Ag | MoO3', 'SLG | ITO | SnO2-c | ZnTiO3-mp | Perovskite | Spiro-MeOTAD | Au', 'PDMS | PEDOT:PSS | Perovskite | PCBM-60 | PEI | PEDOT:PSS | PDMS', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuInS2-QDs | Au', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | Al2O3-mp | MeO-DATPA | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MnS | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEH-3 | Au', 'Carbon-np | SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | MoO3 | Perovskite | PCBM-60 | Al', 'SLG | ITO | TiO2-c | Perovskite | PTAA | MoO3 | Au', 'SLG | FTO | TiO2-np | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KIPIG', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MAI | Spiro-MeOTAD | Au', 'PET | ITO | PEI | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | NiO-c | Perovskite | CSCNT@Al2O3-c | CSCNT@SnO2', 'SLG | ITO | MC6Cz-9-NPC | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | AZO-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | BL38 | Au', 'SLG | FTO | (RhCp*Cp)2 | PTCBI | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-Bi | Au', 'SLG | FTO | ZnO-np | MgO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | CL1-2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon-nt | Carbon', 'SLG | FTO | PTEG-1 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Au@SiO2-nw; TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Yih-2 | Au', 'SLG | ITO | TiO2-c | TiO2-mp | MgO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | SrTiO3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | PEDOT:PSS | Au', 'MgF2 | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PEIE | SnO2-np | ITIC | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-K | Perovskite | PCBM-60 | ZnO | Al', 'PET | ITO | ZnO-np | PCBM-60 | Perovskite | P3HT | PEDOT:PSS | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | ITO | TiO2-c | SnO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | CuI | Perovskite | PCBM-60 | ZnO-np | Ag', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C70 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | COPV4 | Au', 'SLG | ITO | ITO-nw | TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | PIF8-TAA | Au', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | TZ2 | Ag', 'SLG | ITO | Perovskite | C60 | Bphen | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTSe-1 | Au', 'SLG | ITO | PTAA | Perovskite | CMB-vTA | AZO | Ag', 'SLG | ITO | C60; PhIm | C60 | Perovskite | F6-TCNNQ; TaTm | Ag', 'SLG | ITO | SY3 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Spiro-TTB | PEDOT:PSS | Perovskite | C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsPbBr2I-np | Carbon', 'SLG | FTO | TiO2-c | TiO2-nt | ZrO2-mp | Perovskite | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskte | PCBM-60 | TIPD | Al', 'SLG | TiO2-mp | Ti:TiO2 | ZrO2-mp | Perovskite | Au', 'PET | ITO | PEDOT:PSS | PEI-HI | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | PCBCB | Perovskite | Thiophene | Spiro-MeOTAD | Au', 'SLG | ITO | In2O3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | f-C70 | C60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | ITO | TB(MA) | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | TPD | MoO3 | Au', 'SLG | FTO | NiO-c | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PVP | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Au | Al', 'NOA63 | Au | PEDOT:PSS | Perovskite | PCBM-60 | MoO3 ∣ Au ∣ Ag ∣ MoO3 | Alq3', 'SLG | ITO | PEDOT:PSS | VB-DAAF | Perovskite | C60 | BCP | Al', 'PI | Ag-np | PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'PEN | ITO | CuPc | PEI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | ZnO-mp | Perovskite | Spiro-MeOTAD | Carbon', 'SLG | ITO | 3C | Perovskite | C60 | BCP | Ag', 'SLG | ITO | NiO-c | SY2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CdSe-QD | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | C60 | BCP | CU', 'SLG | FTO | Perovskite | CuPc | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | iDM1 | Au', 'SLG | ITO | PCBM-60 | ICL | Perovskite | PTAA | Au', 'SLG | ITO | VOx | APPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PANI-PAMSA | Perovskite | C60 | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | PH 1000', 'SLG | FTO | TiO2-c | Al2O3-mp | Perovskite | P3HT | SWCNTs | PMMA | Cu', 'SLG | Au; NiO-c | Perovskite | C60 | BCP | Al', 'SLG | ITO | TiO2-np | Perovskite | DERDTS-TBDT | MoO3 | Ag', 'SLG | FTO | TiS2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | TPD | Perovskite | PCBM-60 | Ba | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | PFN | Au', 'SLG | ITO | PTAA | Perovskite | PCBB-S-N | PCBM-60 | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BT41 | Au', 'SLG | Cu | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | LiF | Al', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | Ag', 'Nanopaper | TiO2 | Ag | TiO2-c | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | SnO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | Nb2O5 | PCBM-60 | (EMIM)PF6 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | 2,4-spiro | Au', 'PET | Au-np; Graphene; TFSA | Graphene-QDs | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Perovskite | Diazo-OMeTPA | Au', 'ITO | PET | ZnO2-c | ZnO-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | Perovskite | PCBM-60 | C60 | BCP | Cu', 'Ti | TiO2-c | Perovskite | PTAA | Graphene | PDMS', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BDT0FMeDPA | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PFN; ZnO-np | Al', 'PET | Graphene; TETA | ZnO-np | Perovskite | PTAA | PEDOT:PSS | Graphene; TFSA | PET', 'SLG | FTO | SnO2-c | TiO2-c | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-mp | Z907 | Perovskite | Pt | SLG | FTO', 'SLG | FTO | TiO2-c | [EMIM]PF6-IL | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | ZnO-nw | Perovskite | Graphene', 'SLG | FTO | CuCrO2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Perovskite | PCBM-60 | Ethanl-hydroquinolatolithium | Ag', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | Al', 'SLG | FTO | SnO2-c | Perovskite | P-SC6-TiOPc | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | PdMe2Pc | Au', 'SLG | ITO | PEDOT:PSS | Etylene glycol | Perovskite | Liq | Al', 'SLG | ITO | PN-P | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | P3DT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M2 | Au', 'SLG | FTO | TiO2-np | Perovskite | TPA-ZnPc | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Sinapoyl malate | Perovskite | Co(III)P | Au', 'SLG | FTO | SnO2-c | TiO2-mp | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Al', 'SLG | ITO | CPTA | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | N719 dye | Perovskite | PCBM-60 | Ag', 'SLG | ITO | SnO2-c | PA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuInS2 | ZnS-QDs | Carbon', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | ZnO-np | Perovskite | Spiro-MeOTAD | Ag', 'PET | AZO | Ag | AZO | PEDOT:PSS | Perovskite | PCBM-60 | Au', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | 3-Ethylthiophene | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | ZrO2 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-CPDT | Au', 'SLG | ITO | TiO2-np | PCBM-70 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | CuGaO2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | NaYF2@SiO2-np | Perovskite | Carbon', 'SLG | FTO | TiO2-c | SnO2-QDs | Perovskite | Spiro-MeOTAD | Au', 'Ti | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | P3HT | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2-mp | Perovskite | Graphite', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2-((5-(4-(2-ethylhexyl)-4H-dithieno[3,2-b:2′,3′-d]pyrrol-2-yl) thiophen-2-yl)methylene) malononitrile | Ag', 'SLG | ITO | TiO2-np | C-PCBOD | Perovskite | Spiro-MeOTAD | Au', 'Epoxy | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | SnO2-c | Ag | SnO2-c | SnO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B186 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KTM3 | Au', 'SLG | ITO | PEDOT:PSS | CrO3 | Perovskite | PCBM-60 | Bphen | Ag', 'SLG | ITO | P3CT-Na | Perovskite | TPE-DPP16 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | p-DTS(FBTTh2)2 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H-Di | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | [BMPA-BTD]3-TPA | Au', 'SLG | ITO | TiO2-c | Perovskite | P3HT | MoO3 | Al', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | CuPc | Au', 'SLG | FTO | PEDOT:PSS | Perovskte | PCBM-60 | Ag', 'SLG | ITO | ZnO-nanoflake | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnS2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | CuI | Perovskite | PCBM-60 | BCP | Ag', 'SLG | Graphene; TFSA | Graphene-QDs | Perovskite | PCBM-60 | Al', 'SLG | ITO | SnO2-np | Perovskite | BTTI-C12 | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60; PCDTBT | Ag', 'Au | Ni | Al | Ti | GaN | Perovskite | hexagonal-BN | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | OTES:APTES-SAM | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M105 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | LiF | Au', 'SLG | ITO | AZO-c | Perovskite | Ag', 'SLG | ITO | P3CT-Na | Perovskite | IDT6CN-TM | C60 | BCP | Ag', 'SLG | FTO | NiO | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NH-2,6 | Au', 'Perovskite | C60 | BCP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Al2O3 | Ethyl acetate; I2; LiI; NMBI; Urea | Pt | FTO', 'SLG | FTO | CoOx | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | SP-11 | Au', 'SLG | ITO | SFT-TPAM | Perovskite | C60 | BCP | Ag', 'SLG | FTO | ZnO-c | Al2O3-c | Perovskite | Spiro-MeOTAD | Au', 'PET | Ag-grid | PEDOT:PSS | PTAA | Perovskite | PCBM-60 | Al', 'SLG | FTO | ZnO-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | PDI2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DDOF | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphene oxide; NiO-c | Au', 'SLG | ITO | NiO-np | 2,2’-BiPy | Perovskite | PCBM-60 | Zr(acac)4 | Ag | CsF', 'PEN | ITO | SnO2-c | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | FTO | TiO2-c | Perovskite | NiO | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Glycine | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | ZnO-c | Perovskite | PDQT | Au', 'A.R.C. | SLG | FTO | TiO2-c | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2; CoCr-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | Z1 | Au', 'SLG | FTO | NiO-np | Perovskite | PCBM-60 | BCP | Au', 'SLG | ITO | ZnO-c | PVP | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | ZnO-c | Perovskite | FDT | Ag', 'SLG | ITO | EGO-PPV | PFN-P2 | Perovskite | PCBM-60 | BCP | cu', 'SLG | ITO | 3EGO-PPV | PFN-P2 | Perovskite | PCBM-60 | BCP | cu', 'SLG | FTO | TiO2-c | Perovskite | DR3TBDTT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | CuOxNy | Au', 'SLG | FTO | NiMgLiO | Perovskite | SnO2-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | NiO | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nw | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | AZO-np | Al | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuFeO2-np | Au', 'PDMS | PET | PEDOT:PSS | Perovskite | C60 | BCP | Cu | Parylene-film', 'SLG | ITO | Spiro-TTB | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | MeoTPD | MoO3 | Au', 'SLG | ITO | V2O5 | P3CT-K | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | M4; PCBM-60 | Au', 'SLG | ITO | C60 | Perovskite | FTA2 | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PhCz-4MeOTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | EtheneTTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | LHTM-1 | Au', 'PET | ITO | TiO2-c | Perovskite | PTAA | Au', 'SLG | SnO2-c | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PET-OMeDPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-c | Perovskite | Ethyl acetate; I2; LiI; NMBI; Urea | Pt | FTO', 'SLG | FTO | Zn2SnO4-c | Zn2SnO4-fiber | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | Cu2O | Perovskite | PCBM-60 | Al', 'SLG | ITO | CuSCN | Perovskite | C60 | Bphen | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | SnO2 | Perovskite | Spiro-MeOTAD | MoO3 ∣ ITO', 'SLG | ITO | NiO-np | M3 | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Cs2CO3; MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Au@SiO2-np | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ph-TPA-4A | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | pTPD | Perovskite | PCBB | Ba | Ag', 'SLG | ITO | 2F-SAM | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | PTAA | PFNBr | Perovskite | PCBM-60 | BCP | Ag', 'SLG | Cu | NiO-c | Perovskite | PCBM-60 | PEIE | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60; PMMA | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | Choline chloride | C60 | BCP | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | Zr(acac)4 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | ICBA | bis-C60 | Ag', 'SLG | FTO | Perovskite | PCBM-60 | C60 | BCP | Cu | Au | BCP', 'SLG | FTO | TiO2-c | Perovskite | CGS | Au', 'SLG | ITO | PTAA | TFPPy-ETTA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Al-np; PEDOT:PSS | Perovskite | PCBM-60 | PFN | Al', 'SLG | FTO | NiO-c | NiO-mp | Perovskite | PCBM-60 | Al', 'SLG | FTO | Perovskite | Ag', 'Textile | PEN | ITO | Au | Spiro-MeOTAD | Perovskite | SnO2-c | ITO | PEN', 'SLG | ITO | SnO2-np | ImAcHcl | Perovskite | PTAA | Au', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | PMMA | DTPC13-ThTPA | Au', 'SLG | CuS | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnChl-4 | Ag', 'SLG | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'PET | ITO | SnO2-np | SnO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Adamantane | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Ca | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Th-PDI | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM4 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | V1021 | Ag', 'SLG | ITO | CoOx | Perovskite | PCBM-60 | Ag', 'SLG | ITO | Au@TiO2-nw; TiO2-np | Perovskte | Spiro-MeOTAD | Au', 'SLG | FTO | NDI; SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | ZrO2@TiO2-mp | Perovskite | Carbon', 'SLG | ITO | NiO-c | Perovskite | PCBM-60 | ZnO | IZTO', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | IDT-TPA | Au', 'SLG | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag | WO3', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-70 | Fe2O3 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DIPO-Ph4 | Au', 'SLG | FTO | SnO2-np; TiO2-np; MXene | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | M111 | Au', 'SLG | SnO2 | SiO2 | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-c | Perovskite | C60 | BCP | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | Fullerene-2c | Ag', 'SLG | FTO | PCBCB | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CuCrO2 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | NiO-c | Perovskite | FA-PDI2 | TiO2-c | Ag', 'SLG | ITO | NiO | Perovskite | Nb2O5 | Ag', 'SLG | ITO | FB-OMeTPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | MoS2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnChl-1 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Azu-Oct | Au', 'SLG | FTO | ZnO-np | Perovskite | P3TAA-co-P3HT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ph-TPA-2A | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR373 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PCPDTBT | Au', 'SLG | ITO | TiO2-c | Perovskite | P3OT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | CdSe-QDs | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPc-flu-ZnPc | Au', 'SLG | ITO | SnO2-c | C60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | SnO2-np | Perovskite | EH44 | MoOx | Al', 'SLG | FTO | TiO2-c | TiO2-nw | TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'Ag-np | SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Graphite | FTO | SLG', 'SLG | ITO | C60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCTDI | Cr | Au', 'SLG | FTO | SnO2-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | YN2 | Cu', 'SLG | ITO | PEDOT:PSS | P3HT | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Cu', 'SLG | ITO | NiO-c | 1ab | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | Mg0.1Zn0.9O-np | Perovskite | P3HT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HMPDI | Au', 'SLG | ITO | TaTm | Perovskite | C60 | C60; Phlm | Ag', 'SLG | ITO | CA-Br; TPA-PT-C6 | Perovskite | Polyimid | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | PANI | FTO | SLG', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEH-1 | Au', 'SLG | ITO | ZnO-c | ZnO-nw | CdS-QDs | Perovskite | MEH-PPV | PEDOT:PSS | Au', 'SLG | ITO | CuI | PEDOT:PSS | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | Me4NBr | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | BAFB | ZnO-np | Ag', 'SLG | In2O3:H | PTAA | Perovskite | PCBM-60 | ZnO-np | AZO-c | Ni | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTB7-TH | Ag', 'SLG | FTO | SnO2-np | Perovskite | CZTS | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | apv-EC | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,6Cz-TPA | Au', 'SLG | ITO | C60 | Perovskite | IDF-SFXPh | MoO3 | Ag', 'SLG | FTO | SnO2-QDs | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | PTAA | Perovskite | Plastic foam | C60 | BCP | Al', 'Unknown', 'PEN | ITO | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PDI-EH | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | ZnO-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | ZnS | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | DPA-ANR-DPA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | Bphen | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | MoO3 | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BF003 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CT1 | MoO3 | Ag', 'PET | ITO | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | C-PDTON | ZnO | Perovskite | Spiro-MeOTAD | Ag', 'PEN | ITO | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | bis‐C60 | Ag', 'SLG | ITO | SnO2-np | Perovskite | NH2-POSS | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | PEI | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DBC-OMeDPA | Au', 'SLG | FTO | SnO2-c | C60-SAM | Perovskite | PMMA | DTPC8-ThTPA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | tert-CuBuPc | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | CdSe | LiF | Au | Ag', 'SLG | ITO | Au-np; TiO2-np | Perovskite | Spiro-MeOTAD | Au', 'PET | ITO | PTAA | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | ITO | Graphne oxide | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TaTm | Perovskite | PCBM-60 | AZO-np | Au', 'SLG | ITO | NiO-c | Perovskite | C60 | SnS | Ag', 'SLG | FTO | SnO2-c | Perovskite | CuSCN | Carbon', 'SLG | FTO | TiO2-c | Perovskite | SWCNTs', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TQ2 | Au', 'SLG | FTO | TiO2-c | C60-BCT | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | SnO2-c | Perovskite | V1000 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | 4-Aminobutyric acid | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTAA | MoO3 | ITO | MgF2', 'SLG | FTO | PEDOT:PSS | Perovskite | F8TBT | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SiO2 | MWCNTs', 'SLG | ITO | TPL | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | ZnO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Al2O3-c | TiO2-mp | Perovskite | PTAA | Au', 'SLG | ZnO | ITO | Ag-nw | ITO | ZnO-c | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PARA1 | Au', 'SLG | ITO | CuOx | Perovskite | C60 | BCP | Al', 'SLG | FTO | SnO2-c | PCBM-60 | Perovskite | CuPc | Au', 'SLG | FTO | CoCuO | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SDTFCz2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | C60-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | C101 | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag-nw', 'SLG | ITO | PTAA | Perovskite | C60 | ZnSe | Ag', 'SLG | ITO | NiO-c | Perovskite | C60 | SnO2-c | ITO | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FeS2 | Au', 'SLG | ITO | TiO2-c | Perovskite | TaTm | F6-TCNNQ; TaTm | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Graphene-QDs | Perovskite | CISZ-QDs | Carbon', 'SLG | ITO | Trux-OMeTAD | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | PTAA | Perovskite | C60MC12 | AZO-np | Ag', 'SLG | FTO | TiO2 | Perovskite | MEH-PPV | Carbon', 'SLG | FTO | ZnO-nw | ZnO-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Aniline | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | MgO | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiMgLiO | Perovskite | ZnO-c | Ag', 'SLG | ITO | TiO2-c | C60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TPPI | Al', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MWCNTs; Spiro-MeOTAD | Au', 'SLG | ITO | NiO | Perovskite | C60 | Zr(AcAc) | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Bphen | Cs2CO3 | Ag', 'Willow glass | ITO | PTAA | Perovskite | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TB4-ZnPc | Au', 'SLG | ITO | NPB | Perovskite | C60 | BCP | Al', 'Polyimide | In2O3:H | PTAA | Perovskite | PCBM-60 | ZnO-np | AZO | Ni | Al', 'SLG | ITO | Fullerene-SAM | Perovskite | P3HT | Carbon', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | C60 | Ba | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD; X60 | Au', 'PEN | Ag-nw | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Au', 'SLG | ITO | C60 | Perovskite | SWCNTs | Spiro-MeOTAD', 'SLG | FTO | NiO-c | Al2O3-mp; Au@SnO2-np | Perovskite | PCBM-60 | BCP | Au', 'SLG | FTO | TiO2-c | Perovskite | PTB7:Th | Au', 'SLG | ITO | Ag-np; PEDOT:PSS | Perovskite | PCBM-60 | PFN | Al', 'Eu(TTA)2(Phen)MAA | SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | ZnO-c | ZnOS | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | C60-N | Ag', "SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (2Z,2'Z)-2,2'-(((2,4-dimethylphenyl) azanediyl) bis([1,1'-biphenyl]-4',4-diyl)) bis(3-(4-(diphenylamino) phenyl) acrylonitrile | Au", 'SLG | FTO | TiO2-c | TiO2-mp | beta-alanine | Perovskite | Spiro-MeOTAD | Au', 'PEN | ITO | SnO2-c | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsPbI3-np | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ti3C2', 'Ti | TiO2-np | Perovskite | PEDOT | ITO | PEN', 'PET | PEDOT:PSS | Perovskite | PCTDI | Cr | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZPPHT | Carbon', 'SLG | ITO | MoOx | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | PCBM-60 | AZO-np | ITO', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTB7 | Ag', "SLG | FTO | TiO2-c | Perovskite | 5,5',5''-(5,5',5''-(nitrilotris(benzene-4,1-diyl))tris(furan-5,2-diyl))tris(2-octylisoindoline-1,3-dione) | MoOx | Ag", 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CdSe-QDs | Spiro-MeOTAD | Au', 'Willow glass | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | Au | PDMS Nanocone', 'SLG | FTO | SnO2-c | 3-(1-pyridinio)-1-propanesulfonate | Perovskite | asy-PBTBDT | Au', 'SLG | ITO | SnS2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Li-TFSI; TBPAu', 'SLG | FTO | PEDOT:PSS | Perovskite | Me4NI | PCBM-60 | BCP | Ag', 'SLG | ITO | PEIE | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Fullerenol | Perovskite | P3HT | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Carbon | Carbon-fiber', 'PES | AZO | CuNW | AZO | SnO2-np | Perovskite | P3HT; Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Cs2CO3 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X60(TFSI)2 | Au', 'SLG | FTO | CuO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PFO | Al2O3-np | Perovskite | PCBM-60 | BCP | Ag', 'Y2O3:Eu3 | SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nanosphere | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | ZrO2-mp | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P3HT-MoS2 | Au', 'SLG | FTO | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | Perovskite | Al2O3-c | PEDOT:PSS | Au', 'SLG | ITO | PEIE | NDI-H | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | PFN-P2 | Perovskite | C60 | LiF | BCP | Cu', 'SLG | ITO | ZnO-c | WPF-6-oxy-F | Perovskite | pDPA-DBTP | Au', 'SLG | FTO | SnO2-c | SnO2-nw | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Ag-nw; TiO2-np | Perovskite | Spiro-MeOTAD | Ag', 'PET | Graphene | PTAA | PEDOT:PSS | Perovskite | ZnO-np | Graphene | PET', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEIE | Ag', 'SLG | ITO | PEDOT:PSS | Si-np | Perovsksite | PCBM-60 | Al', 'SLG | PEDOT:PSS | PEI | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | Graphitic carbon nitride | PCBM-60; Graphene | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | BCP; PCBM-60 | Ag', 'SLG | ITO | ZnO-np | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NbOx | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | Perovskite | PCBM-60 | Al', 'SLG | ITO | C60-SAM | SnO2-c | PCBM-60 | Perovskite | P3HT | WOx | Au', 'SLG | ITO | TiO2 | Perovskite | Spiro-MeOTAD | Ag', 'PET | ITO | PEDOT:PSS | Perovskite | F8BT | Al', 'SLG | FTO | TiO2-c | KH570 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | Perovskite | C60 | Bphen | Ag', 'SLG | ITO | NiO-c | Perovskite | PEAI | PCBM-60 | BCP | SnO2-c | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S,N-heteropentacene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MEH-PPV | Au', 'SLG | FTO | TiO2-c | Perovskite | BTT(DPP)3-EH | Au', 'SLG | FTO | CrOx | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPD | HAT-CN | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsPbBr3-np | Carbon', 'SLG | FTO | ZnO-c | PCBM-60 | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BDT-2D | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | LCS01 | Au', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | PDPP4T | Au', 'PET | ITO | SnO2-c | SnO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | AZO | Ba(OH)2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS｜Perovskite | PCBM-60 | C60 | BCP | Al', 'SLG | ITO | CuI | Perovskite | PCBM-60 | Ag', 'SLG | ITO | BTF2 | Perovskite | PCBM-60 | LiF | Ag', 'SLG | FTO | TiO2-c | Perovskite | 3-Methylthiophene | Spiro-MeOTAD | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | BPTI | PDI-C4 | BCP | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | FT37 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PO-Spiro | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P(BDTT-SePPD) | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (BMPA-EDOT)3-TPA | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-420 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | G1 | Au', 'SLG | FGZO | SnO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MoO3 | TPA-2,7-FLTPA-TPA | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | V1225 | Au', 'SLG | FTO | SnO2-c | APTES-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-nw | Perovskite | Carbon', 'PEN | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | PCBA | Perovskite | P3 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PCBM-60 | Ag', 'SLG | ITO | PEDOT:PSS | pTPD | Perovskite | PCBH | Ba | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TBC-1 | Au', 'SLG | FTO | TiO2-c | CoCr | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | C60 | Al2O3-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | MoOx | TaTm | Perovskite | C60 | BCP | Ag', 'SLG | FTO | SnO2-c | PMMA | Perovskite | PMMA | Spiro-MeOTAD | Au', 'PET | ITO | c-TCTA | Perovskite | c-HATNA | bis-C60 | Ag', 'Nb:SrTiO3 | Perovskite | ITO', 'SLG | ITO | TiO2-c | TiO2-mp | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | ZnOS | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-nanowalls | Diethanolamine | Perovskite | PCBM-60 | Zr(acac)4 | Ag', 'SLG | ITO | TiO2-c | Perovskite | rGO-4FPH | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS｜Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD:P3HT | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | bis-PCBOE | ZnO | Al', 'SLG | FTO | CdSe | PCBM-60 | Perovskite | CuPc | Carbon', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Al', 'SLG | ITO | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Au', 'SLG | FTO | SnO2-np | Perovskite | Cz-OMeTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X21 | Au', 'SLG | ITO | PCP-Na | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | B-MWCNTs', 'SLG | ITO | XSln847 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | HB-Cz | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Candle soot | FTO', 'SLG | Ag | PCBM-60 | Perovskite | PTAA | Ag', 'SLG | ITO | NiO-c | Perovskite | C60 | SnO2-c | ITO', 'SLG | ITO | PII2T8TSi | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PTAA | Perovskite | CdI2 | C60 | BCP | Cu', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Pt-sheet', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | Au', 'SLG | FTO | Li4SiW12O40 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CsPbI3-QD | Spiro-MeOTAD | Au', 'SLG | ITO | C60 | Perovskite | FBA3 | MoO3 | Ag', 'SLG | ITO | TiO2-np | C70 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SM | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S,N-heteropentacene based HTM 1 | Au', 'SLG | ITO | Spiro-s | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 2FBTA-1 | Au', 'SLG | FTO | SnO2-c | Perovskite | N-CuMe2Pc; P3HT | Au', 'SLG | FTO | TiO2-c | PCBSD | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PTAA | Perovskite | PCBM-60; Sb-Carbon-nw | BCP | Au', 'PET | ITO | PEDOT:PSS | Perovskite | P(NDI2DT-TTCN) | BCP | Au', 'SLG | FTO | TiO2-c | Al2O3-c | Perovskite | PTAA | Au', 'SLG | ITO | NiO-c | Mercaptoethylamine chlorate | Perovskite | PCBM-60 | Bphen | Al', 'SLG | ITO | ZnO-c | TiO2-c | PCBM-60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | TiO2-c | Perovskite | C202 | Ag', 'SLG | ITO | IDTT2FPDI | Perovskite | PTAA | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PT-PDI | Al', 'SLG | ITO | Z10 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag | PMMA | Teflon', 'PET | IZO | PTAA | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z25 | Au', 'SLG | ZnO | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | PEDOT:PSS | Perovskite | Cytop | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-c | Perovskite | Carbon-nt | Spiro-MeOTAD | Ag', 'Polyester-satin textile | Polyurethane | Carbon-nt; PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | ITO | CONs | PEDOT:PSS | Perovskite | PCBM-60 | TiO2 | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | C60; PCBM-60 | Zr(acac)4 | Ag', 'PET | FTO | NiO-np | Perovskite | C60 | bis-C60 | Ag', 'SLG | ITO | MSAPBS | Perovskite | Spiro-MeOTAD | Au | Ag', 'SLG | ITO | XSln1453 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | PSS-Na | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | JK-217D | Ag', 'SLG | FTO | PEDOT:PSS | PTAA | Perovskite | Me4NBr | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2-np | Perovskite | PEDOT:PSS | MoO3 | Ag', 'SLG | FTO | Fe2O3 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | C60 | TmPyPB | Ag', 'SLG | IWO | SnO2-c | KCl | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | PCBA | Perovskite | P2 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon black; MWCNTs', 'PET | ITO | PEDOT:PSS | Perovskite | PCBM-60 | TiO2 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | H16 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | P:OO | Au', 'SLG | ITO | SnO2-np | Perovskite | TCPBr | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTPA-6 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTB-DCB21 | Au', 'PET | ITO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | NiO-c | Perovskite | ZnO | C60 | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Ag-nw | C60', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | MoS2 | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KR353 | Au', 'SLG | FTO | SnO2-c | Perovskite | 1‐adamantylamine hydrochloride | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | SnO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Zn2SnO4-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | TiO2-c | Perovskite | DOR3T-TBDT; PCBM-70 | MoO3 | Ag', 'SLG | FTO | SnO2-c | Perovskite | Carbon', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CZTS0.5Se0.5; rGO | Au', 'SLG | FTO | TiO2-nw | N719 | Perovskite | Pt-np | Au', 'SLG | FTO | TiO2-c | Perovskite | NiCoO | Al | Al203', 'SLG | ITO | MeO-PPV | Perovskite | PCBM-60 | BCP | cu', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | bis‐C60 | Ag', 'SLG | FTO | Au-nw | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | PEDOT:PSS | Graphene | PMMA | PDMS', 'SLG | ITO | SnO2-c | NiCl2 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | CsAc | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TTB2 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | (2Z,2′Z)-3,3′-(5,5′-(2,7-dioctyl-1,3,6,8-tetraoxo-1,2,3,6,7,8-hexahydrobenzo[lmn][3,8] phenanthroline-4,9-diyl)bis(thiophene-5,2-diyl))bis(2-(3,5-bis (trifluoroomethyl)phenyl) acrylonitrile) | Ag', 'SLG | FTO | SnO2-np | Perovskite | CsPbBr3-np | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | Perovskite | H6Bu-ZnPc | Ag', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | C60 | Ag', 'SLG | ITO | MEH-PPV-20 | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | COPV6 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM3 | Au', 'SLG | ITO | ZnO-c | SnO2-np | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | PTAA | Perovskite | C60 | Ag', 'SLG | ITO | PEDOT | Perovskite | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Metal', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTEG | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | FTO | ZnO-np | Perovskite | P3HT | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | LiF | Au', 'SLG | FTO | TiO2-mp | Perovskite | Carbon | FTO | SLG', 'SLG | FTO | PEDOT:PSS | Perovskite | 2-HI-PVK | PCBM-60 | BCP | Ag', 'SLG | ITO | f-C60 | Perovskite | P3HT | MoO3 | Ag', 'SLG | ITO | SnO2-c | Perovskite | CZTPA-2 | MoO3 | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | BCP; TZ-3 | Ag', 'SLG | FTO | ZnO | Au-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | SnO2-c | SnO2-c | Perovskite | V1102 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS | FTO | SLG', 'SLG | ITO | ZnO-np | Perovskite | PbS-QDs | MoO3 | Au | Ag', 'SLG | FTO | ZnO@SnO2-np | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | VB-MeO-FDPA | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | P3CT-Na | Perovskite | TPE-PDI4 | Rhodamine 101 | LiF | Ag', 'SLG | ITO | ZnLiO | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | C60-BCT-Au-NP | Perovskite | PTAA | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | apv-T | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HS-Ph-OCH3 | Spiro-MeOTAD | Au', 'PEN | ITO | MC8-9-NPC | Perovskite | PCBM-60 | LiF | Al', 'SLG | FTO | TiO2-c | Perovskite | 2mF-X59 | Au', 'SLG | ITO | ZnO-c | PO-TAZ | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | NiO-c | Al2O3-mp | Perovskite | PCBM-60 | Rhodamine 101 | Au | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEH-2 | Au', 'PET | ITO | ZnO-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | TS-CuPc | MoO3 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | QCAPZ | LiF | Al', 'SLG | FTO | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-mp | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | CZTS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | Perovskite | Spiro-MeOTAD | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | Graphite', 'SLG | ITO | NiO-c | DEA | Perovskite | C60 | PN4N | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Q222 | Au', 'SLG | ITO | SnO2-np | Perovsite | Spiro-MeOTAD | Ag', 'SLG | ITO | CuSCN | Perovskite | PCBM-60 | AZO-np | Ag', 'SLG | ITO | VB-DAAF | Perovskite | C60 | BCP | Al', 'SLG | FTO | PolyTPD | Perovskite | C60 | BCP | Ag', 'SLG | FTO | NiO-c | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | Sr2CeO4:Eu | Perovskite | Spiro-MeOTAD | Au', 'PET | Ag-nw | FZO | TiO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | Silica-gel-electrolyte | Carbon', 'SLG | FTO | NiO-c | Perovskite | PPA | PCBM-60 | BCP | Ag', 'SLG | ITO | PTB7 | Perovskite | PCBM-60 | PEI | Ag', 'PET | Graphene | PEDOT:PSS | Perovskite | C60 | BCP | Ag', 'SLG | ITO | Au-np | NiO-c | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | TaTm | Perovskite | C60 | Ag', 'SLG | ITO | SnO2-np | ZnO-c | Perovskite | PTAA | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DM1 | Au', 'PDMS | Graphene | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPA-QA-TPA | Ag', 'PET | Ag-mesh:PH1000 | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | SnO2 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | SnO2 | Perovskite | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 3,8,13-tris[2,2-bis(4-methoxyphenyl)ethenyl]-5,10,15-triethyl-10,15-dihydro-5H-indolo-[3,2-a:3′,2′-c]carbazole | Au', 'SLG | ITO | DH-MeO-FDPA | Perovskite | PCBM-60 | PEI | Ag', 'Ti | TiO2-nt | Perovskite | Spiro-MeOTAD | Au', 'SLG | MSA-PEDOT:PSS | PEDOT:PSS | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuSCN | Graphite | Cu-tape', 'SLG | FTO | TiO2-c | Si-nw | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | NiO-c | Perovskite | TPA-3CN | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PS | C60 | BCP | Ag', 'SLG | ITO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | ITO | PEIE | SnO2-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | Cu-BHT | PEDOT:PSS | Perovskite | PCBM-60 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Tetracene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS', 'SLG | FTO | TiO2-c | TiO2-mp | PCBM-60 | Perovskite | Spiro-MeOTAD | ITO | Au', 'SLG | FTO | NiMgLiO | Perovskite | 4-DMABA | PCBM-60 | BCP | Ag', 'SLG | ITO | NiO-c | Perovskite | PN-F50 | Ag', 'PEG | ITO | PEDOT:PSS | Perovskite | bis-C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | DMFA-TPA | Au', 'SLG | ITO | Graphene oxide | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | NaYF4:Yb:Tm-np | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | Rhodamine 101 | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PFN-P2 | Ag', 'SLG | ITO | PBT | Perovskite | C60 | BCP | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | tris-PCBOE | ZnO | Al', 'SLG | ITO | C60-SAM | Perovskite | PDCBT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PTPD2 | Au', 'SLG | FTO | SnO2-c | Perovskite | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | CdSe-QDs; PCBM-60 | Rhodamine 101 | LiF | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEH-9 | Au', 'SLG | FTO | ZnO-c | TiO2-mp | Perovskite | 18-crown-6 ether | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PEO; KI; I2 | Pt | FTO | SLG', 'SLG | ITO | NiO-np | Perovskite | PCBM-60 | ZnO | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | TiO2-c | Perovskite | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | DTAB | Ag', 'PET | ITO | C60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | P3CT-Na | Perovskite | NDP-V | C60 | BCP | Ag', 'SLG | FTO | SnO2-c | Perovskite | Spiro-MeOTAD | Rubrene | Au', 'SLG | ITO | PEDOT:PSS | Perovskite | Ti | Au', 'SLG | ITO | NiO-np | Perovskite | C60; PCBM-60 | bis-C60 | Ag', 'SLG | FTO | ZnO-c | ZnO-nw | Perovskite | NiO | Au', 'PEN | ITO | PEDOT:PSS | Perovskite | PCBM-60 | PEI | Ag', 'SLG | FTO | Graphene oxide | Perovskite | TiO2-c | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Spiro-MeOTAD | PEDOT:PSS | Ag-nw | PDMS', 'SLG | ITO | SnO2-c | CBA-SAM | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-np | PCBM-60 | Perovskite | BTPA | MoO3 | Au', 'SLG | ITO | PEDOT:PSS | PFI | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | Perovskite | SWCNTs | PMMA', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | KM05 | MoO3 | Ag', 'SLG | FTO | TiO2-c | Perovskite | 2D-PT | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | CuBuPc | Au', 'SLG | FTO | Poly-EDOT-C60 | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | SrGO | Perovskite | PCBM-60 | BCP | Ag', 'SLG | ITO | PEIE | CDIN | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-nw | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'Moth eye PDMS | SLG | ITO | PTAA | Perovskite | C60 | BCP | Cu', 'SLG | ITO | PEDOT:PSS | Perovskite | Br-PDI | ZnO-np | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | 4b | Au', 'SLG | ITO | TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | NDI-PhE | Al', 'SLG | ITO | PVBT-SO3 | Perovskite | PCBM-60 | C60/C70-N | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | (OctPhO)8CuPc | Au', 'SLG | SWCNTs | MoO3 | PEDOT:PSS | Perovskite | C60 | BCP | LiF | Al', 'SLG | ITO | PEDOT:PSS | PolyTPD | Perovskite | Au', 'SLG | ITO | VB-Me-FDPA | Perovskite | PCBM-60 | PEI | Ag', 'SLG | ITO | MoO3 | Perovskite | C60 | BCP | Al', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | bis-C70 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TDAC | Au', 'SLG | ITO | C60; Phlm | C60 | Perovskite | TaTm | F6-TCNNQ; TaTm | Au', 'SLG | ITO | NiO-c | SY3 | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SGT-409 | Au', 'SLG | ITO | Si-OMeTPA | Perovskite | PCBM-60 | Al', 'SLG | ITO | ZnO-c | WPF‐6‐oxy‐F | Perovskite | mDPA-DBTP | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | X51 | Ag', 'SLG | ITO | SnO2-np | C60 | Perovskite | BTF2 | Au', 'PET | TCE | ZnO-c | Perovskite | P3HT | Ag', 'SLG | ITO | CuO | Perovskite | PCBM-60 | ZnO-np | Al', 'SLG | ITO | WO3-nw@PEDOT | PEDOT:PSS | Perovskite | PCBM-60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | ZnPy | Au', 'SLG | ITO | Perovskite | PCBM-60 | C60 | BCP | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | PBDTTT-C | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HTM2 | Au', 'SLG | ITO | NiO-c | PEDOT:PSS | Perovskite | PCBM-60 | bis-C60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon | CsAc', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPBS | Au', 'PEN | ITO | PhNa-1T | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | HPDI | Au', 'SLG | FTO | C60 | [EMIM]PF6-IL | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | SnO2-c | Perovskite | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | Perovskite | CuSCN | Graphene | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Ph-TPM | Au', 'SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | SnO2-np | Sinapoyl malate | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | ZnO-c | TiO2-c | Perovskite | Spiro-MeOTAD | MoO3 | Al', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | BTPA-3 | Au', 'SLG | FTO | PEI | PCBM-60 | Perovskite | PTAA｜Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Carbon-QDs | Spiro-MeOTAD | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | S7 | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | TPD-4MeOTPA | Au', 'SLG | ITO | ZnO-c | Perovskite | PZn-TPA-O | Au', 'SLG | ITO | PFN | PCBM-60 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | PolyTPD | PFN | Perovskite | LiF | C60 | SnO2-c | ZTO | Au', 'Mn:CsPbCl3-QDs | SLG | FTO | TiO2-c | Perovskite | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Al2O3-mp | Perovskite | Carbon; WO2-np', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SO7 | Au', 'SLG | ITO | PbS-QDs | Perovskite | PCBM-60 | Al', 'PEN | ITO | SnO2 | Perovskite | Spiro-MeOTAD | Au', 'SLG | ITO | SnO2-c | Perovskite | HPB-OMe | Au', 'SLG | ITO | ZnO-c | PCBM-60 | Perovskite | Spiro-MeOTAD | MoO3 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | SWCNTs | Graphene oxide | Au', 'SLG | ITO | SnO2-np | C60 | Perovskite | BTF3 | Au', 'SLG | FTO | TiO2-c | Au@TiO2-np; TiO2-mp | Perovskite | Spiro-MeOTAD | Ag', 'SLG | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Cs2CO3 | Ag', 'SLG | ITO | CuI-np | Perovskite | C60 | BCP | Ag', 'PEN | ITO | PEDOT:PSS | Perovskite | PCBM-60 | Ag', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Q205 | Carbon', 'SLG | ITO | SnO2-c | Perovskite | Carbon-nt | Spiro-MeOTAD | Au', 'SLG | FTO | TiO2-c | TiO2-mp | Perovskite | Z1013 | Au']))))

    area_total = Quantity(
        type=np.dtype(np.float64),
        unit=('cm**2'),
        shape=[],
        description="""
    The total cell area in cm2. The total area is defined as the area that would provide photovoltaic performance when illuminated without any shading, i.e. in practice the geometric overlap between the top and bottom contact.
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    area_measured = Quantity(
        type=np.dtype(np.float64),
        unit=('cm**2'),
        shape=[],
        description="""
    The effective area of the cell during IV and stability measurements under illumination. If measured with a mask, this corresponds to the area of the hole in the mask. Otherwise this area is the same as the total cell area.
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    number_of_cells_per_substrate = Quantity(
        type=np.dtype(np.int64),
        shape=[],
        description="""
    The number of individual solar cells, or pixels, on the substrate on which the reported cell is made
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    architecture = Quantity(
        type=str,
        shape=[],
        description="""
    The cell architecture with respect to the direction of current flow and the order in which layers are deposited. The two most common are nip (also referred to as normal) and pin (also referred to as inverted) but there are also a few others, e.g. Back contacted
- nip architecture means that the electrons are collected at the substrate side. The typical example is when a TiO2 electron selective contact is deposited between the perovskite and the substrate (e.g. SLG | FTO | TiO2-c |Perovskite | …)
- pin architecture means that it instead is the holes that are collected at the substrate side. The typical example is when a PEDOT:PSS hole selective contact is deposited between the perovskite and the substrate (e.g. SLG | FTO | PEDOT:PSS |Perovskite | …)
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown', 'Pn-Heterojunction', 'Front contacted', 'Back contacted', 'pin', 'nip', 'Schottky'])))

    flexible = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the cell flexible and bendable.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    flexible_min_bending_radius = Quantity(
        type=np.dtype(np.float64),
        unit=('cm'),
        shape=[],
        description="""
    The maximum bending radius possible without degrading the cells performance
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    semitransparent = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the cell is semi-transparent, which usually is the case when there are no thick completely covering metal electrodes.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    semitransparent_AVT = Quantity(
        type=np.dtype(np.float64),
        shape=[],
        description="""
    The average visible transmittance in the wavelength range stated in the next field
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    semitransparent_wavelength_range = Quantity(
        type=str,
        shape=[],
        description="""
    the wavelength range under which the average visible transmittance is determined
Example:
400; 720
350; 770
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['nan; nan', '800; 1200'])))

    semitransparent_raw_data = Quantity(
        type=str,
        shape=[],
        description="""
    A link to where the data file for the measurement is stored
- This is a beta feature. The plan is to create a file repository where the raw files for stability data can be stored and disseminated. With the link and associated protocols, it should be possible to programmatically access and analyse the raw data.
                    """)

    def normalize(self, archive, logger):
        add_solar_cell(archive)
        if self.stack_sequence:
            archive.results.properties.optoelectronic.solar_cell.device_stack = self.stack_sequence.split(' | ')
        if self.architecture:
            archive.results.properties.optoelectronic.solar_cell.device_architecture = self.architecture
        if self.area_total or self.area_measured:
            if self.area_measured and not self.area_total:
                archive.results.properties.optoelectronic.solar_cell.device_area = self.area_measured
            else:
                archive.results.properties.optoelectronic.solar_cell.device_area = self.area_total


class Module(MSection):
    """
    Specific section containing information if the reported device is a module.
    """

    Module = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the cell is a module composed of connected individual sub cells
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    number_of_cells_in_module = Quantity(
        type=np.dtype(np.int64),
        shape=[],
        description="""
    The number of cells in the module
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    area_total = Quantity(
        type=np.dtype(np.float64),
        unit=('cm**2'),
        shape=[],
        description="""
    The total area of the module in cm2. This includes scribes, contacts, boundaries, etc. and represent the module’s geometrical footprint.
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    area_effective = Quantity(
        type=np.dtype(np.float64),
        unit=('cm**2'),
        shape=[],
        description="""
    The active area of the module in cm2.
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    JV_data_recalculated_per_cell = Quantity(
        type=bool,
        shape=[],
        description="""
    The preferred way to report IV data for modules is to recalculate the IV data to average data per sub-cells in the module. That simplifies downstream comparisons, and it ensures that there is no erroneous transformation that otherwise may occur when error checking the IV data. Mark this as TRUE if the conversation is done.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))


class Substrate(MSection):
    """
    Information about the substrate used in the device. It describes the `substrate stack sequence`,
    the `substrate area`, the `substrate thickness`, and its provenance or fabrication method.
    """

    stack_sequence = Quantity(
        type=str,
        shape=[],
        description="""
    The stack sequence describing the substrate.
- With the substrate, we refer to any layer below the electron transport layer in a nip-device, and any layer below the hole transport layer in a pin-device.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If two materials, e.g. A and B, are mixed in one layer, list the materials in alphabetic order and separate them with semicolons, as in (A; B)
- Use common abbreviations when appropriate but spell it out if risk for confusion.
- There are a lot of stack sequences described in the literature. Try to find your one in the list. If it is not there (i.e. you may have done something new) define a new stack sequence according to the instructions.
ExampleBelow are some of the most common substrates
SLG | FTO
SLG | ITO
PET | ITO
PEN | ITO
SLG | AZO
PET | IZO
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=sorted(['SLG | Ag-nw', 'Textile | PEN | ITO', 'PET | Ag', 'PEN | Graphene', '42P2O5-22Li2O-22ZnO-12Sm2O3-2CeO2 | FTO', 'SLG | Cu-BHT', 'Mica | ITO', 'NOA88 | PEI | Au', 'Si | ITO', 'PEN | Graphene | MoO3', 'PET | Al2O3', 'PES | Ti | Graphene', 'SLG | Ag | Unknown', 'PEN | Ag-nw', 'Cu', 'Transparent wood | ITO', 'SLG | AZO', 'none', 'Ti-foil', 'Cellulose paper | Carbon black', 'SLG | FAZO', 'PDMS | PET | Au-grid', 'PEN | SWCNTs | MoO3', 'SrTiO3 | Sr2RuO4', 'SLG | In2O3:H', 'Epoxy | ITO', 'PES | ITO', 'Polyimide | ITO', 'Cellophane | TiO2 | Ag | TiO2', 'SiO2-mp | SLG | ITO', 'Polyimide | In2O3:H', 'Polyester-satin textile', 'NOA63 | Au', 'SLG | PEDOT:PSS', 'b-CNF | IZO', 'SLG', 'SLG | ITO:ATO', 'SLG | PET', 'Flexible | IZO', 'PET | TCE', 'SLG | AZO | Ag-nw | AZO', 'PET | PEDOT:PSS', 'Si', 'Pt', 'SLG | SU-8 | MoO3 | Au', 'SLG-HAMC-patterned | FTO', 'SLG | Cd2SnO4', 'PDMS | Graphene', 'PET | APTES; Graphene', 'PDMS | PET', 'SLG | Ni:Au-mesh', 'PEN | AZO', 'SLG | Ag-nw | Graphene oxide', 'SLG | Graphene | MoO3', 'Au | Ni | Al | Ti | GaN', 'Quartz | TaN', 'SLG | FTO | Au-grid', 'PDMS', 'PI | PEDOT:PSS', 'SLG | ITO | Au-grid | ITO', 'SLG | Au', 'SLG | PEDOT:PSS | Ag | PEI', 'Quartz | ITO', 'SLG | ITO | Au-grid | AZO', 'PEI | ITO', 'PES | FTO', 'PET | Ni-mesh:PH1000', 'PET | IWO', 'PI | Ag-np | PEDOT:PSS', 'PET | Ag-nw | FZO', 'SLG | WO3 | Ag | WO3', 'Willow glass | Graphene | Ag-nw', 'PAA-PEG | Ti', 'SLG | Au-np; Graphene; TFSA', 'PET | Ag-grid | PEDOT:PSS', 'PI', 'SLG | Ti', 'SLG | AZO:F', 'ITO | PET', 'SLG | MPTMS-SMA | Ag | MUTAB-SAM', 'PAA-PEG | Ti | PANI | Ti', 'SLG | Ag-nw; Graphene oxide', 'SLG | Graphene', 'SLG | ITO', 'Nb:SrTiO3', 'SLG | SWCNTs-HNO3', 'SiO2-hollow | SLG | ITO', 'PI | Cu-grid | Graphene', 'PEN | Graphene; MoO3', 'PET | Ag-grid', 'SLG | MSA-PEDOT:PSS', 'SLG | Ag-nw | AZO', 'NOA63 | MoO3 | Au', 'NOA63', 'PS', 'PES | PEDOT:PSS | Ag', 'Ag-nw; Graphene; Polycarbonate; Polystyrene', 'NOA63 | ITO', 'PET | ITO | Ag-nw', 'SLG | IZO', 'SLG | Graphene; TFSA', 'SLG | TiO2-c | SnO2-c | TiO2-c | SnO2-c | TiO2-c | SnO2-c | TiO2-c | SnO2-c | TiO2-c | ITO', 'SLNOA63 | CPI | Cr | Au-grid', 'SLG | APTES; Graphene', 'SLG | Au-np; Graphene', 'PET | AZO', 'Unknown', 'SLG | AZO | Cu-nw | AZO', 'Carbon-nt-yarn', 'PDMS | SLG | ITO', 'Cellophane | TiO2-c | Ag', 'SLG | ITO | ITO', 'PET | PEDOT:PSS | Ag-nw', 'PET | Ag-mesh:PH1000', 'PEN | Planarization | SiN | ITO', 'PET | AuCl3; Graphene', 'Silk | Ag-nw | PEDOT:PSS', 'PET | Graphene', 'PET | IZO', 'PET | Ag-nw; PEDOT:PSS', 'Quartz | Graphene', 'HCLaminate | IZO', 'Ti-wire', 'SLG | ITO-HMDS Scaffold', 'SLG | Cu-nw', 'PET | Ag-nw; Graphene oxide', 'PET | Ag-nw | Graphene', 'SLG | ZrO2 | MPTMS-SMA | Ag | MUTAB-SAM', 'PET | In2O3', 'Ti', 'PET', 'Carbon-nt-fiber', 'PEN | ITO', 'SLG | TiO2-c | Ag', 'Regenerated cellulose film | Ag-nw', 'SLG | Ag', 'PET | Graphene; TETA', 'Si | SiO2', 'SU-8 | Ca | Au', 'PET | FTO', 'PET | Ag-mesh | PH1000', 'PES | AZO | CuNW | AZO', 'Willow glas | Ti', 'PET | WO3 | Ag | WO3', 'SLG | Ni', 'Graphite', 'PET | Ag-nw', 'FPA-PDMS', 'Willow glass | ITO', 'SLG | SWCNTs', 'INVAR | ITO', 'SLG | SnO2 | SiO2 | FTO', 'PEG | ITO', 'PETUG', 'SLG | ZnO | ITO | Ag-nw | ITO', 'Ti-sheet', 'Unknown | ITO', 'SLG | FTO', 'PET | ITO', 'PES | AZO | Ag-nw | AZO', 'Willow glass | AZO', 'SLG | ITO | Ni', 'SLG | DWCNTs', 'PET | Au', 'Paper | Au', 'Steel', 'Nanopaper | TiO2 | Ag', 'PET | Au-np; Graphene; TFSA', 'PET | MSA-PEDOT:PSS', 'SU-8 | MoO3 | Au', 'SLG | PEI | Au', 'SLG | resist | ITO', 'PEN', 'SLG | SWCNTs | MoO3', 'SLG | WO3 | Ag', 'ITO | PEN', 'Ag-nw; GFRHybrimer; ITO', 'SLG | Ag-nw | Graphene', 'SLG | IWO', 'SLG | rGO', 'SLG | AZO | Au | AZO', 'Stainless steel', 'SLG | TCO', 'SLG | Au-grid | AZO', 'Willow glass | IZO', 'PET | SWCNTs', 'Cu-nw; GFRHybrimer; ITO', 'PEN | FTO', 'PEN | Ag-grid', 'SLG | Cu', 'SLG | FGZO', 'PET | APTES | AuCl3; Graphene', 'Foil | AZO', 'PET | AZO | Ag | AZO']))))

    thickness = Quantity(
        type=str,
        shape=[],
        description="""
    A list of thicknesses of the individual layers in the stack. Use the following guidelines
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous filed.
- State thicknesses in nm
- Every layer in the stack have a thickness. If it is unknown, state this as ‘nan’
- If there are uncertainties, state the best estimate, e.g write 100 and not 90-110
- If you only know the total thickness, e.g. you have a 2 mm thick commercial FTO substrate and you do not know how thick the FTO layer is, state that as ‘2 | nan’
Example
2.2 | 0.1
2 | nan
nan | nan | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'nan | 140.0', '2.0 | 0.0004', 'nan | 0.4', '3.0', '0.03 | 0.00015 | 2.0', 'nan | 332.0', 'nan | 35.0 | nan | 35.0', '2.0 | 0.06', 'nan | 100.0', '0.15', '398.0', 'nan | 500.0', 'nan | 0.2', 'nan | 0.6', 'nan | 0.1', '2.2 | nan', 'nan | 180.0', 'nan | nan', '1.0 | 0.14', '2.2 | 0.2', '3.0 | nan', '2.0', 'nan | 220.0', 'nan | 0.5', 'nan | 0.13', 'nan | 0.22', '150.0', 'nan | 60.0', '200.0', 'nan | 0.15', 'nan | 0.25', '1.0 | 0.15', 'nan | 0.04', '0.175 | 0.0025', '2.2', '1.0 | 0.13', '2.0 | 0.6', 'nan | 40.0'])))

    area = Quantity(
        type=np.dtype(np.float64),
        unit=('cm**2'),
        shape=[],
        description="""
    The total area in cm2 of the substrate over which the perovskite is deposited. This may be significantly larger than the cell area
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    supplier = Quantity(
        type=str,
        shape=[],
        description="""
    . The supplier of the substrate.
- Most substrates in the perovskite field are bought commercially, but if it is made in the lab, state this as “lab made”
- If the supplier is unknown, stat that as‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example:
Lab made
NGO
Pilkington
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'AimCore Technology', 'Lumtec', 'Yingkou OPV Tech New Energy Co.', 'Sigma Aldrich', 'Yingkou YouXuan', 'Unknown', 'Automatic Research GmbH', 'Xiang Science & Technology', 'Solaronix', 'OPV Technology Corp', 'Furuuchi Chemical', 'Thin Film Devices Inc.', 'Mekoprint OC50', 'Luminiscence Technology Corporation', 'Naranjo', 'Lab made', 'CSG Holding Co', 'Eastman Chemical Company', 'Naranjo substrates', 'Nippon Sheet Glass Co.', 'Pilkington', 'Advanced Election Technology', 'Kintec', 'Shen Zhen Hua Nan Xiang Cheng Factory', 'Hartford Glass Co.', '3M', 'HeptaChroma', 'Merck', 'Delta Technologies', 'Ossila', 'AMG-Tech', 'Greatcell Solar', 'Xinyan Technology', 'Zhuhai Kaivo'])))

    brand_name = Quantity(
        type=str,
        shape=[],
        description="""
    . The specific brand name of the substrate. e.g NGO11, TEC15, etc.
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'TCO-XY15', 'Unknown', 'TEC15', 'TEC8', 'TEC14', 'Trizact 3000', 'FTO22-7', 'TEC7', 'TEC7.5', 'NSG10', 'DHS-FTO22-15N', 'TEC9', 'TCO22-7', 'TECS'])))

    deposition_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    . A list of the deposition procedures for the substrate
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- Make sure that you describe as many layers as there are layers in the stack. Otherwise it will be difficult to interpret which layer the deposition procedure is referring to. It should thus be as many vertical bars in this field as when describing the substrate stack.
- When more than one reaction step, separate them by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the deposition procedure for a layer unknown, state that as‘Unknown’
- If a substrate is bought commercially and you do not know, indicate this by the label “Commercial”
- This category was included after the initial project release wherefor the list of reported purities are short, so be prepared to expand on the given list of alternatives in the extraction protocol.
Example
Commercial | Commercial
Commercial | Sputtered >> Sputtered
Commercial | ALD
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Commercial | Commercial', 'Commercial | Electrodeposition >> Spin-coating', 'Commercial | Sputtering', 'Commercial | Electrospinning', 'Unknown', 'Commercial | Sputtering >> Sputtering', 'Commercial | commercial', 'Spin-coating | Commercial', 'Photolithography | Spin-coating | Lamination', 'Commercial | Laser patterning >> Spin-coating', 'Commercial | Spin-coating', 'CVD', 'Commercial | Magnetron sputtering', 'Commercial | ALD | Doctor blading | ALD'])))

    surface_roughness_rms = Quantity(
        type=np.dtype(np.float64),
        unit=('nm'),
        shape=[],
        description="""
    The root mean square value (RMS) of the surface roughness expressed in nm
- If not known, leave this field blank
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    etching_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    . For the most common substrates, i.e. FTO and ITO it is common that part of the conductive layer is removed before perovskite deposition. State the method by which it was removed
- If there is more than one cleaning step involved, separate the steps by a double forward angel bracket (‘ >> ‘)
- This category was included after the initial project release wherefor the list of reported purities are short, so be prepared to expand on the given list of alternatives in the extraction protocol.
Example
Zn-powder; HCl >> Mecanical scrubbing
Laser etching
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Unknown', 'Photolithography', 'Dipping in HCl', 'Laser etching', 'Zn-powder; HCl >> Mecanical scrubbing'])))

    cleaning_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    . The schematic cleaning sequence of the substrate. The Extraction protocol does not capture the fine details in the cleaning procedures, e.g. times, temperatures, etc. but state the general sequence. Refers to the cleaning of the entire substrate before the deposition of the rest of the cell stack starts.
- If there is more than one cleaning step involved, separate the steps by a double forward angel bracket (‘ >> ‘)
- If more than one procedure is occurring simultaneously, e.g. Soap washing an ultrasonic bath, separate simultaneously occurring steps with a semicolon.
- This category was included after the initial project release wherefor the list of reported purities are short, so be prepared to expand on the given list of alternatives in the extraction protocol.
Example
Helmanex >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> UV-ozone
Piranha solutionion
Piranha solutionion >> UV-ozone
Soap
Soap >> Ultrasonic bath
Soap >> Ultrasonic bath >> Ethanol; Ultrasonic bath >> Acetone >> UV-ozone
Soap >> Ultrasonic bath >> UV-ozone
Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=sorted(['Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> heating >> UV-Ozone', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath', 'Oxygen plasma', 'Soap >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> IPA >> O2-plasma', 'Soap >> Ultrasonic bath >> UV-Ozone', 'Water >> Acetone >> Ethanol >> UV-Ozone', 'De-ionized Water >> Ultrasonic bath >> Soap >> Ultrasonic bath >> IPA >> Ultrasonic bath', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> UV-Ozone', 'Micro-90 detergent >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath', 'Soap >> Water >> Ethanol', 'Helmanex >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> UV-Ozone', 'UV-Ozone', 'Soap >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> DI Water', 'Soap >> Water >> Ultrasonic bath >> IPA >> O2-plasma', 'Extran 300 >> Ultrasonic bath >> IPA', 'DI Water >> Ethyl alcohol >> Acetone', 'Soap >> Water >> Acetone >> IPA >> UV-Ozone', 'Toluene >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Water >> Ultrasonic bath', 'Helmanex >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath', 'Soap >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> UV-Ozone', 'Water >> Acetone >> IPA', 'detergent >> deionized water >> isopropanol >> acetone >> UV-Ozone', 'Hellmanex >> solution >> DI >> water >> acetone >> IPA', 'Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> Ethanol >> UV-Ozone', 'Acetone >> Water >> IPA >> Nitrogen flow', 'Detergent >> Ultrasonic bath >> DI-Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> O2 plasma', 'Helmanex >> Ultrasonic bath >> Water >> IPA >> Ultrasonic bath >> Acetone >> microwave plasma', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> IPA >> UV-Ozone', 'Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> UV-Ozone', 'Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone', 'Soap >> Water >> Ultrasonic bath >> Ethanol >> O2-plasma', 'Soap >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Plasma', 'Soap >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> IPA', 'Soap >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> IPA >> UV-Ozone', 'Detergent >> Ultrasonic bath >> alkali liquor >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath', 'Soap >> Water >> Ultrasonic bath >> Ethanol >> UV-Ozone', 'Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Water >> Ultrasonic bath', 'Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath', 'Mucasol >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Water >> Ultrasonic bath', 'Soap >> Water >> Ultrasonic bath >> Ethanol', 'Acetone >> Ultrasonic bath >> Abs Ethanol >> Ultrasonic bath >> DI Water', 'Soap >> Ultrasonic bath >> Water', 'Soap >> Water >> Acetone >> Water', 'Soap >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> UV-Ozone', 'Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> Drying in oven >> UV-Ozone', 'Unknown >> O2 plasma', 'Water >> Ultrasonic bath >> IPA >> Ultrasonic bath >> O2-plasma', 'Detergent >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> UV-Ozone', 'Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> UV-Ozone', 'H2O2/HCl/H2O = 1:1:5 >> acetone >> isopropyl alcohol', 'Soap >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> Water >> Ultrasonic bath >> O2-plasma', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> IPA >> UV-Ozone', 'DI Water >> Ultrasonic bath >> Ethanol', 'DI Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> UV-Ozone', 'Unknown >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> DI Water >> Ultrasonic bath', 'Mucasol >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> IPA >> UV-Ozone', 'Acetone >> Water >> IPA >> Nitrogen flow >> Corona Treatment 0.74 kW', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Plasma', 'DI Water >> Ultrasonic bath >> Helmanex >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Methanol >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> UV-Ozone', 'Acetone >> Ultrasonic bath >> Methanol >> Ultrasonic bath', 'Soap >> Water >> Ultrasonic bath >> Acetone; IPA; Ethanol >> UV-Ozone', 'Soap >> Acetone >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol >> O2-plasma', 'Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> Ethanol >> Ultrasonic bath', 'Detergent >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath', 'Acetone >> Ultrasonic bath >> Methanol >> Ultrasonic bath >> DI Water', 'Soap >> Ultrasonic bath >> Ethanol-HCl >> Ultrasonic bath >> Acetone >> Water >> Heating', 'Unknown', 'Water >> Ethanol >> IPA', 'Helmanex >> Ultrasonic bath >> IPA >> Ultrasonic bath >> Acetone >> UV-Ozone', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> UV-Ozone', 'DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath', 'Helmanex >> Ultrasonic bath >> Ethanol >> Acetone', 'Soap >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> lPA;Acetone; Water >> Ultrasonic bath >> UV Ozone', 'Water >> Acetone >> IPA >> O2-plasma', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> Plasma', 'Helmanex >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> DI Water', 'NaOH >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> DI Water >> Acetone', 'Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> O2-plasma', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethano >> Ultrasonic bath >> UV-Ozone', 'Soap >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> Plasma', 'Acetone >> Ultrasonic bath >> Isopropyl alcohol >> Ultrasonic bath >> de-ionized Water', 'Detergent >> Ultrasonic bath >> DI-Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> air plasma', 'Helmanex >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> UV-Ozone', 'Soap >> peroxide/ammonia >> Ultrasonic bath >> Methanol >> Ultrasonic bath >> IPA >> UV-Ozone', 'Soap >> DIWater >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> N2 blowing >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> UV-Ozone', 'alconox-detergent >> Ultrasonic bath >> deionized water >> Ultrasonic bath  >> acetone >> Ultrasonic bath >> isopropanol >> Ultrasonic bath', 'Detergent >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >>  >> DI-Water >> Ultrasonic bath >> UV-Ozone', 'Water >> Acetone >> Ethanol >> IPA', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol', 'Helmanex >> Acetone >> IPA >> O2-plasma', 'DIWater >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> UV-Ozone', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> O2-plasma', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> lPA >>  >> Ethanol >> O2-plasma', 'Soap >> Water >> Toluene >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> O2-plasma', 'detergent >> acetone >> isopropanol >> ethanol', 'Helmanex >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> O2 plasma', 'Acetone >> Ultrasonic bath >> Methanol >> Ultrasonic bath >> Water', 'Soap >> Water >> Acetone >> IPA >> Ethanol >> Water >> UV-Ozone', 'Soap >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> UV-Ozone', 'Mucasol >> Ultrasonic bath >> Acetone >> Utrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> Ozone', 'Acetone >> IPA >> O2 plasma', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> O2-plasma', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> O2-plasma', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol >> Utrasonic bath >> UV-Ozone', 'Acetone >> Ultrasonic bath >> Methanol >> Ultrasonic bath >> Water >> Ultrasonic bath', 'Acetone >> IPA >> Water', 'Helmanex >> Ultrasonic bath >> DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> IPA >> Ultrasonic bath', 'Water >> Acetone >> IPA >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath', 'Soap >> Acetone >> Ethanol >> Water >> UV-Ozone', 'NaOH Ethanolic solution >> Water >> detergent >> Water >> Dry air', 'Soap >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol >> UV-Ozone', 'Soap >> Ultrasonic bath >> de-ionized Water >> Ultrasonic bath >> Acetone >> UV-Ozone >> Ethanol >> UV-Ozone', 'Acetone >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Water >> Ultrasonic bath >> UV-Ozone', 'DI Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> Ultrasonic bath >> UV-Ozone', '2.5 M NaOH >> Ultrasonic bath >> Water >> Detergent >> Milli-Q Water >> Annealed 30 min at 500℃', 'DIWater >> Ultrasonic bath >> Ethanol >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> Oven drying >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> Water >> Ultrasonic bath >> Ethanol >> Ultrasonic bath', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ethanol >> Ultrasonic bath >> IPA >> UV-Ozone', 'Helmanex >> Ultrasonic bath >> DI Water >> Ethanol >> 2-propanol >> Ultrasonic bath', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA >> O2-plasma', 'Soap >> Water >> Ultrasonic bath >> Acetone >> Ultrasonic bath >> IPA']))))

    def normalize(self, archive, logger):
        add_solar_cell(archive)
        if self.stack_sequence:
            archive.results.properties.optoelectronic.solar_cell.substrate = self.stack_sequence.split(' | ')


class ETL(MSection):
    """
    A section to describe information related to the Electron Transport Layer (**ETL**).
    """

    stack_sequence = Quantity(
        type=str,
        shape=[],
        description="""
    The stack sequence describing the electron transport layer. Use the following formatting guidelines
- With the ETL, we refer to any layer between the substrate and the perovskite in a nip-device, and any layer between the perovskite and the back contact in a pin-device.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If two materials, e.g. A and B, are mixed in one layer, list the materials in alphabetic order and separate them with semicolons, as in (A; B)
- If no electron transport layer, state that as ‘non’
- Use common abbreviations when appropriate but spell it out if risk for confusion.
- If a material is doped, or have an additive, state the pure material here and specify the doping in the columns specifically targeting the doping of those layers.
- There is no sharp well-defined boundary between when a material is best considered as doped or as a mixture of two materials. When in doubt if your material is best described as doped or as a mixture, use the notation that best capture the metaphysical essence of the situation.
- There are a lot of stack sequences described in the literature. Try to find your one in the list. If it is not there (i.e. you may have done something new) define a new stack sequence according to the instructions.
ExampleBelow are some of the most common electron transport layers
TiO2-c | TiO2-mp
TiO2-c
PCBM-60
PCBM-60 | BCP
SnO2-np
C60 | BCP
SnO2-c
TiO2-c | TiO2-mp | ZrO2-mp
ZnO-c
PCBM-60 | C60 | BCP
PCBM-60 | LiF
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['3TPYMB', 'PCBM-60; ICBA', 'SnO2 | C60', 'SnO2-np | MSAPBS', 'TiO2-c | TiO2-mp | PbS-QDs', 'ZTO | SnO2-c | C60', 'TmPyPB | C60 | TmPyPB', 'CeOx', 'TiO2-c | TiO2-np | TiO2-mp', 'TiO2-c | TiO2-mp | CsBr', 'PCBM-60 | CeOx', 'C60; PCBM-60 | BCP', 'ZTO | SnO2-c | PCBM-60', 'SnO2-c | C60', 'TiO2-c | TiO2-nw | CdS', 'TiO2-c | TiO2-mp | NH4I', 'SnO2-c | C60-5d', 'NDI-PhE', 'PCBM-60 | Zr(acac)4', 'PCBM-70 | C60 | AUH', 'SnO2', '2PDI-0S | PEIE', 'PCBM-60 | TiO2-np', 'PCBM-60 | DEME-BF4', 'ZnO-np | p-(F)-PO-TAZ', 'TiO2-c | TiO2-mp | Sinapoyl malate', 'Cyptop | PCBM-60 | BCP', 'PCTDI', 'PCBM-60 | PPDIN6', 'FA-PDI2 | TiO2-c', 'SnO2-mp | TiO2-c', 'ZnO0.95Sn0.05O1.05', 'ZnO | PEIE', 'a-PTCDI', 'SnO2-np | TiO2-c', 'TiO2-c | SiO2-np; TiO2-mp', 'TiO2-c | TiO2-mp | NaYF2@SiO2-np', 'MoS2', 'PCBM-63', 'TiO2-c | Al2O3-c', 'SnO2-np | PbF2', 'PCBM-60 | DPO', 'PEG; SnO2-np', 'SnO2-np | BBA', 'ZnO0.9Sn0.1O1.1', 'PCBM-60 | BCP | AZO', 'TiO2-c | Ce2O-mp', 'S-acetylthiocholine chlorde | C60 | BCP', 'TiO2-c | TiO2-mp | D205', 'Ti(OiPr)4', 'PCB-C8', 'WO3', 'PCBM-60 | AZO-np | BCP', 'TiO2-c | In2O3', 'FDA | PCBM-70', 'AZO | ZnO-nw', 'PCBM-60 | BCP; TZ-1', 'SnO2-c | Au | SnO2-c | TiO2-mp', 'TiO2-c | TiO2-mp | BaTiO3', 'TiO2-c | TiO2-mp | Caproic acid', 'SnO2-np | EPA', 'SnO2-np | C60-SAM', 'PTAA', 'SnO2-np; ZnO-np', 'ICBA', 'TiO2-c | TiO2-mp | Sb2S3', 'TPE-ISO4 | C60 | BCP', 'NDI-ID (RS)', 'IDTT2FPDI', 'SrTiO3-c | SrTiO3-mp', 'ZnO | 1,2-ethanedithiol (10 mM)', 'C60; C70', 'PCBM-60 | Isopropanol-hydroquinolatolithium', 'TiO2-c | MgO | TiO2-nw', 'PN-F25', 'PbI2', 'TiO2-c | WOx', 'TiO2-c | BaSnO3-mp', 'TiO2-c | MPMIC60', 'ZnO0.75Sn0.25O1.25', 'SnO2-c | NPC60 OH', 'PCBM-60 | C3-CBL', 'TiO2-c | TiO2-mp | ZrO2@TiO2-mp', 'ICBA-tran3 | C60 | BCP', 'TiO2-c | H3PW12O4; TiO2-mp', 'LiF | C60 | TmPyPB', 'TiO2-c | SiO2', 'MgO | TiO2-mp', 'C60 | Rhodamine 101', 'PCBM-60; PDI-DA | BCP', 'ZTO | TiO2-mp | ZrO2-mp', 'PCBM-60 | PDINO', 'C60 | PCBM-60', 'TiO2-c | SnO2@TiO2-np', 'PDI-T | C60 | BCP', 'NDI-BTH1', 'PCBM-60 | Rodhamine 101', 'Ba(OH)2', 'KY7F22-np', 'F8TBT', 'PCBM-60 | CMB-vTA', 'TiO2-nt', 'PCBB-S-N | PCBM-60', 'TiO2-nanobundles', 'PCB-C12', 'TiO2-c | SnO2-c | TiO2-mp | ZrO2-mp', 'TiO2-c | ZnSe', 'SnO2-c | TiO2-np', 'ZnO-c | PCBM-70', 'TiO2-c | OTS', 'TiO2-mp | Z907', 'TiO2-c | CsCl', 'PSe-PDI', 'Zn2SnO4-c', 'PCBM-60 | PCBC', 'ITCPTC-Se | C60 | BCP', 'TiO2-nanoplatelets', 'PTEG-1', 'WOx | SnO2-np', 'IDTCN | C60 | BCP', 'Zn2SnO4', 'Mono-PCBOE | ZnO', 'TiO2-c | TiO2-nanocones', 'TiO2-c | PCBDAN', 'HATNAS3C7', 'F-PDI | ZnO-np', 'PCBM-60 | PCBDAN', 'PCBM-60; PEI', 'CdSe | LiF', 'TiO2-nw | CuInS2-QDs', 'PCBM-70', 'ZnO-nw | ZnO-c', 'TMTA | PCBM-60 | C60 | TPBi', 'PNDI-2T | PEIE', 'Graphene oxide; TiO2-np', 'TiO2-c | Benzoic acid', '3TPYMB | C60 | TmPyPB', 'YVO4:Eu | TiO2-c | TiO2-mp', 'SnO2-np | PCBA', 'TiO2-c | ZnO-c | TiO2-mp', 'SnO2-c | PCBM-60-np', 'TiO2-c | WO3', 'TPE-DPP8 | C60 | BCP', 'C60-HPy | BCP', 'SnS2', 'TiO2-c | BaSnO3-np', 'IPH', 'C60 | BCP', 'TiO2-c | TiO2', 'SnO2-c | KCl', 'PCBM-70; PTB7-Th', 'TiO2-c | TiO2-nc', 'SnO2-np | PN4N', 'PCBM-60 | F-C60', 'PS | PCBM-60', 'SnO2-np | ImAcHcl', 'PCBM-60 | ZnMgO', 'Sn2O3-ns | PCBM-60 | Bphen', 'Au-np; TiO2-np | C-PCBOD', 'PCBM-60 | ZnO | PEI', 'PCBM-60 | Bphen; Ir(MDQ)2(acac)', 'PEIE | C60', 'ITCPTC-Se | Rhodamine 101 | LiF', 'SnO2-c | Ba(OH)2', 'TiO2-c | 2-OMe-SAM', 'TMPA-Cl | C60 | BCP', 'ZnO-np | NH4Cl', 'ETM', 'SnO2-c | Al2O3-mp | NiO-np', 'ZnO-np | PFS-FC', 'TiO2-c | Graphene', 'PC61BTh | BCP', 'TiO2-c | Zn2Ti3O8-mp', 'ZnO0.855Sn0.15O1.15', 'Nb2O5 | PCBM-60 | Bphen', 'SnO2-np | Sinapoyl malate', 'ZnOS', 'PCBM-60 | Rhodamine 101 | C60 | BCP | Rhodamine 101 | LiF', 'TiO2-c | TiO2-mp | BaTiO3-mp', 'Phenyltrichlorosilane | PCBM-60', 'ICPA | BCP', 'TiO2-c | ZnO-nanoflake', 'TiO2-c | TiO2-mp | PCBM-60; PS', 'TiO2-c | SiW11O39-POM', 'TiO2-c | SiO2-np | TiO2-c | SiO2-np | TiO2-c', 'TiO2-c | SiO2-np | TiO2-c', 'ZnO-c', 'TiO2-c | D35', 'SnO2-np | PCBA-60', 'ZnO-c | TiO2-mp', 'TiO2-c | TiO2-mp | PCBM-60; PTAA', 'SnO2-c | IDTT2FPDI', 'PCBM-60 | CMB', 'TiO2-c | SiO2-IO', 'TiO2-c | TiO2-mp | Graphene', 'ZnO-c | 4-methoxybenzoic acid', 'ZnO-c | CPTA', 'PCBM-60; Sb-Carbon-nw | BCP', 'ZnO-c | 1,2 ethanedithio', 'TiO2-c | TiO2-mp | 3-aminopropanoic acid-SAM', 'Ca', 'TPE-DPP6 | C60 | BCP', 'SnO2-nw | TiO2-c', 'PCBM-60 | TPPI', 'SnO2-np | KCl', 'P3HT', 'ZnO-c | ZnSO', 'CPTA-E', 'TiO2-c | TiO2-mp | PCBM-60; Poly(N-vinylcarbazole)', 'SnO2-nanosheets | C60', 'TiO2-c | CPTA', 'ZnO', 'ZnO-c | ZnO-mp | PEI', 'TiO2 | Al2O3-mp', 'TiO2-c | Dimethylbiguanide', 'PCBM-60 | AgAl-np | Au-np | LiF', 'TiO2-c | Graphene; SrTiO3', 'f-C70', 'Fe2O3-mp', 'TiO2-c | TiO2-mp | SnO2', 'IDT6CN-TM | BCP', 'PCBM-60 | PFN-P2', 'ZnO-np | EA', 'ZnO-np | ZnO-nanospheres', 'PCBM-60 | Ca(acac)2', 'PCBM-60 | Zn(acac)', 'TiO2-c | TiO2-mp | Carbo-QDs', 'ITO-nw | TiO2-np', 'IZO', 'TiO2-c | TiO2-mp | ZnO-mp', 'PPDIDTT | PCBM-60 | Bphen', 'PDIN', 'ZnSO4-c | ZnSO4-mp', 'TiO2-c | Fullerenol', 'PCBM-60; ICBA | BCP', 'PCBM-60-MCM', 'CMB-vTA | PCBM-60 | AZO', 'SrTiO3 | TiO2-mp', 'nTi-MOF | PCBM-60', 'COi8DFIC | BCP', 'ZnO-np | PEI', 'PEAI | PCBM-60 | BCP | SnO2-c', 'In2O3-c | PCBM-60', 'CeO2-c | TiO2-c', 'H-PDI', 'SnO2-c | TiO2-mp', 'ZnO-c | ZnO-nw | SnO2-c', 'MgO-c | SnO2-c', 'TiO2-c | TiBaO-nanosphere', 'PCBM-60 | TmPyPB', 'TiO2-c | KH560', 'ZnO | 1,2-ethanedithiol (5 mM)', '4-acetylpyridine | PCBM-60 | BCP', 'TiO2-c | TiO2-mp | Trimethylamine oxide', 'DMEC-70 | LiF', 'TiO2-c | ZnO-nw | AZO-c', 'PEDOT:PSS', 'IT-4f | BCP', 'PCBM-60; PMMA', 'C70', 'PCBM-60 | ZnO-np | AZO', 'PCBM-60 | PCBC | LiF', 'WO3 | TiO2-c', 'TiO2-c | SWCNTs; TiO2-mp', 'LiF', 'TiO2-mp | MgO', 'ZnO-c; Graphene-np', 'Graphene | Al2O3-mp', 'PCMB-60:BCP', 'SnO2-np | C60', 'Graphene-QDs', 'ZnO-c | PVP', '(RhCp*Cp)2 | PTCBI', 'SnO2-c | ABA-SAM', 'PCBM-70 | C60', 'TiO2-c | TiO2-nanofibers', 'AZO-np', 'IDT6CN | Rhodamine 101 | LiF', 'PCBM-60 | C70 | BCP', 'PCBM-60 | TOPD', 'TiO2-c | TiO2-mp | SbI3', 'SnO2-c | PMMA', 'PCBM-60 | Methanol-hydroquinolatolithium', 'LiF | C60 | PEIE | SnO2-c | ZTO', 'PCBM-60 | PDIN', 'ZnO-c | AZO-np', 'PCBM-60 | ZnO-np | BCP', 'TiO2-mp', 'B-DIPBI', 'NDI; SnO2-np', 'QCAPZ | LiF', 'ZnO-c | ZnO-nw', 'ZnO-c | C3-SAM', 'Carbon-nt | PFN-Br', 'TiO2-c | AAO', 'AgAu-np | TiO2-c', 'TiS2', 'pBTTz', 'PCBM-60-PCP', 'TiO2-c | PCBM-60 | WS-C60', 'TiO2-c | TiO2-mp | Cs2CO3', 'NDI-BTH2', 'SrTiO3 | Al2O3-mp', 'TiO2-c | Polystyrene-ns | TiO2-mp', 'SnO2-nanospheres', 'PCBM-60 | PCBDANI', 'ZnO-c | AZO-mp', 'ZnO-np | PCBM-60', 'NDI-PM', 'C60 | TmTyPB', 'Graphene-np; TiO2-mp', 'TiO2-c | TiO2-mp | Si-np', 'PCBM-62', 'PFPDI', 'PCBM-60 | TiO2', 'TiO2-c | TiO2; Al2O3-mp', 'TiO2-c | TiO2-hollow spheres; TiO2-mp', 'PFN | C60; PCBM-60', 'PCBM-60 | PN4N', 'PCBM-60 | [BMIM]BF4', 'PNVT-8 | ZnO', 'IT4H | s-Bphen', 'TiO2-c | Ba0.9Sr0.1SnO3-np', 'TiO2-c | TiO2-mp | ETPM', 'TiO2-c | TiO2-mp | MBA-SAM', 'SnO2-c | TiO2-c | TiO2-mp', 'PEDOT:PSS | PEI', 'IDTCN | Rhodamine 101 | LiF', 'Carbon-QDs; SnO2-c', 'Q10 | BCP', 'TiO2-mp | Pbs-QDs', 'PCBM-60 | ZnO', 'PDBS-PDI', 'PTTI-1', 'PCBM-70 | BCP', 'ZnO-c | PEI', 'C60 | C60-np', 'TiO2-c | TiO2-mp | Au@Ag', 'C70 | TmPyPB', 'TiO2-mp | MK2', 'TiO2-c | CSOE', 'PDI-V | C60 | BCP', 'PCBM-60 | BCB', 'ZnO | C60', 'TiO2-c | SiO2-nanocolumns | TiO2-nw', 'ZnO@rQD-np', 'Graphene; TiO2-c | Al2O3-mp', 'PEI | ICBA', 'TiO2-np | C60; PCBM-60', 'Polyimid | PCBM-60 | BCP', 'ZnO-c | ZnMgO-c | ZnO-nw | PCBM-60', 'P(NDI2DT-TTCN) | BCP', 'PFN-2TNDI', 'CeOx-np', 'TiO2-c | ZnO-np', 'CeO2-c', 'PCBM-70 | ZrO2', 'Aniline; rGO | CsCO3', 'IDT6CN-4F | BCP', 'TiO2-c | TiO2-mp | Au-np', 'PCBM-60 | TiO2 | BCP', 'SnO2-c | PA-SAM', 'BenMeIM-Cl', 'PEI', 'PCBB-S-N | C60 | BCP', 'TiO2-c | NaYF4:Yb:Er-np', 'PCBM-60 | Phosphoniumfluorene', 'ZnO-c | ZnO-nw | TiO2-np', 'Ti | C60', 'DS1', 'PCBM-60 | Bphen | Cs2CO3', 'TiO2-c | ZnO-nw | Al2O3-c', 'PCBM-60 | Rhodamine 101 | LiF', 'TiO2-c | TiO2; SiO2-IO', 'PCBM-60 | BCP; TZ-2', 'TiO2-c | TiO2-mp | n-dimethyloxamic acid', 'CPTA', 'ZnO-c | Ethyl acetate', 'TDTP | LiF', 'bis-C60', 'PCBM-60 | AZO-np', 'TiO2-c | TiO2-mp | Nb2O5', 'TiO2-mp | ZrO2-mp', 'PCBM-60', 'BPTI | C60 | BCP', 'CdS-np', 'f-C60 | C60', 'PCBM-60 | Bphen | Cs2CO3; MoO3', 'TiO2-c | TiO2-mp | Al2O3', 'SnO2-QDs | PCBM-60', 'PCBM-60 | Carbon-QDs', 'ZnO-c | Urea', 'PEIE | PCBM-60', 'TiO2-c | TiO2-mp | PCBM-60; PMMA | BAI', 'TiO2-c | TiO2-mp | Al203-c', 'Me4NBr | PCBM-60 | BCP', 'PCBM-derivative10', 'In2O3', 'TiO2-c | 60', 'TiO2-c | TiO2-nanowalls', 'PCBB-S-N', 'bis-PCBOE | ZnO', 'EG-PEDOT:PSS | Ti | SnO2-c | TiO2-c', 'TiO2-c | 3-OMe-SAM', 'TiO2-c | TiO2-np', 'PDPT', 'SnO2-c | NiCl2', 'PCBM-60 | FPI-PEIE', 'Zn2SnO4 | Zn2SnO4-nw', 'Carbon-mp | TiO2-mp', 'SnO2-c | C3-SAM', 'PCBM-60 | MoS2', 'TiO2-c | MoOx', 'Mg0.1Zn0.9O-np', 'ICBA; PCBM-60', 'TiO2-c | PCBM-60 | C60-SAM', 'Zn:SnO2', 'TiO2-c | TiO2-mp | PbTiO3', 'PCBM-60; F8BT | C60', 'TiO2-c | CsAc', 'Zn2SnO4-c | rGO; Zn2SnO4-fiber', 'PCBM-60 | C60SB:TBAI', 'SrSnO3', 'TPE-DPP16 | C60 | BCP', '(DTYM-NDI-DTYA)2 | BCP', 'Polystyrene | C60 | BCP', 'TiO2-nw | TiO2-nw', 'Al2O3-mp', 'TiO2-c | Graphdiyne-QDs', 'Plastic foam | C60 | BCP', 'TiO2-c | TiO2-nanobeads', 'TiO2-c | PCBM-60 | PbI2', 'PCBM-60 | ZnO-np | ZnO-c', 'C60-N', 'NDI-P', 'Al:ITO-c', 'Al2O3-c | TiO2-c | TiO2-mp', 'PN-P', 'TiO2-c | ZnCdS', 'AZO-np | PCBM-60', 'PCBM-60 | BMIM-OTF', 'TiO2-c | TiO2-helices', 'f-C60', 'SnO2-c | Carbon-QDs', 'TiO2-c | TiO2-mp; YVO4:Eu:Bi-np', 'none', 'LiSPS | PCBM-60', 'ZnO-np | Graphene', 'N2200; PFN-Ox | bis-C60', 'ZnO@SnO2-np', 'MgO | SnO2-np | Ethanol amine', 'TiO2-c | TiO2-mp | 4‐Aminobenzoic acid', 'Cs2CO3', 'TiO2-c | Si-nw', 'Fullerene @ F2', 'PCBM-60; F8TBT', 'HAT-CN | BCP', 'TiO2-nanoleaves | MgO', 'TiO2-c | C60-BCT', 'TiO2 | SnO2-c | SnO2-np', 'TiO2-nanofibers', 'CdS-c | TiO2-mp', 'CdSe-QDs; PCBM-60 | Rhodamine 101 | LiF', 'ITCPTC-Th | Rhodamine 101 | LiF', 'PCBM-60 | Rhodamine 101 | C60 | Rhodamine 101 | LiF', 'S-acetylthiocholine chloride | C60 | BCP', 'PCBM-60 | AZO-np | SnO2-c', 'ZnO-c | ZnO-c', 'PCBM-60 | Polyethylimine', 'C60; PDI | BCP', 'BrTPA-PCBM | bis-C60', 'BaSnO3-np', 'TiO2-c | TiO2-mp | PhCOOH', 'IC60BA | bis-C60', 'Li4SiW12O40', 'Au@TiO2-np; TiO2-np', 'TiO2-np', 'TiO2-np | PCBM-60 | TiO2-c', 'TiO2-c | TiO2-mp | ZIF-8-40', 'TiO2-c | Al2O3-nt', 'TiO2-c | TiO2-mp | MXene', 'ZnO-np | Sinapoyl malate', 'ZnO-np | MgO-EA', '2-HI-PVK | PCBM-60 | BCP', 'TiO2-c | 1-OMe-SAM', 'LiF | C60 | SnO2-c', 'TiO2-c | PCBB-2CN-2C8', 'TiO2-c | MCA', 'TiO2-c | TiO2-mp | Y2O3-c', 'PCBH', 'TiO2-np; Nb2O5', 'C60:PhIm | C60', 'TiO2-c | TiO2-mp | C60 | ZrO2-mp', 'PCBCB', 'Carbon | TiO2-mp', 'TiO2-np | PCBM-60', 'TiO2-c | Al2O3-mp', 'C60-SAM', 'TiO2-c | TiO2-nw | N-[3-(Trimethoxysilyl)propyl]ethylenediamine-SAM', 'ZnO-c | ZnO-nw | TiO2-c', 'PbS | PCBM-60', 'SnO2-np | DTPP', 'ZnO | 1,2-ethanedithiol (15 mM)', 'TiO2-c | [EMIM]PF6-IL', 'TTC | C60 | BCP', 'C60 | SnO2-np', 'TiO2-mp | TiO2-c', 'Au-nw | TiO2-c', 'ZnO-nanospheres', 'Nb2O5 | (EMIM)PF6', 'F-PDI', 'SnO2-c | C60-5a', 'TiO2-c | Ag-nw; TiO2-np', 'TPA-PCBM | bis-C60', 'SnO2-np | TPPO', 'PCBM-60 | PVP', 'SnO2-c | C60-5b', 'PyCEE', 'TiO2-c | TiO2-np | ZrO2-mp', 'TiO2-c | SrTiO3', 'TiO2-c | SrTiO3-mp', 'TiO2-c | ZnO-nw', 'ITIC; PCBM-60 | PEI', 'FSIP | C60 | BCP', 'PTCDA | PO-T2T', 'PCBM-60 | CIL', 'C60 | C60; Phlm', 'PMDPT', 'TiO2-c | TiO2-mp | PbSe', 'TPE-DPP4 | C60 | BCP', 'TiO2-c | PCBM-63', 'Acetyl acetate; I2; LiI', 'Au-np; TiO2-np | C-PCBSD', 'TiO2-c | NaYF4:Yb:Er-np; TiO2-mp', 'TiO2-c | TiO2-mp | HOOC-Ph-SH', 'CPTA | BACl', 'H-DIPBI', 'SnO2-c | C60-SAM', 'PCBM-60; CTAB', 'Zr(acac)4', 'TiO2-c | TiO2-nanoflowers', 'C5-NCMA | bis-C60', 'MC-43', 'TiO2-np | C70', 'TiO2-c | Au-np; TiO2-mp', 'C60 | C60', 'MoO3 | Au | SnO2-c | TiO2-mp', 'TiO2-c | IPH', 'PCBM-60 | ZnO-c', 'C60 | LiF | BCP', 'N2200 | bis-C60', 'Graphene | ZnO-QDs', 'ZnO-c | T2CA', 'PCBM-60 | PCBM-60', 'P(NDI2OD-TET)', 'TiO2-c | NiO', 'TiO2-c | TiO2-mp | D149', 'PCBM-60 | AZO', 'ZnO-c | ZnO-nanodisks', 'Fullerene-2a', 'Ru(acac)', 'PCBM-60 | Phen-NaDPO', 'PCBM-60 | bis-C70', 'PCBM-60; Graphene | BCP', 'TiO2-c | TiO2-mp | Aminocaproic acid; Caproic acid', 'Graphene nanoribbons hPDI2-Pyr-hPDI2', 'TiO2-c | TiO2-mp | 4‐Aminobutyric acid', 'BAFB | ZnO-np', 'SnO2-c | C60-5e', '2PDI-3S | PEIE', 'PEIA | PCBM-60 | BCP', 'Graphene', 'Graphene; NDI; SnO2-np', 'PCBM-60 | CD', 'MATS | PCBM-60', 'PCBM-70 | C70 | BCP', 'MABr | PCBM-60 | BCP', 'PTEBS', 'PEDOT:PSS | PEI | PCBM-60', 'SnO2-mp', 'PMMA | PCBM-60', '2PDI-4S | PEIE', 'TiO2-nw', 'Au-np; TiO2-np', 'C60 | Zr(AcAc)', 'PEAI | PCBM-60 | PEI', 'C60; Phlm', 'ZnO-mp | ZnO-np', 'PCBM-60 | PEOz', 'TiO2-c | TiO2-mp | ZIF-8-20', 'PFN-OX:ZnO', 'In2S3', 'ZnLiO', 'TiO2-c', 'C60 | TmPyPB', 'SnO2-np | SnO2-c', 'PTCDI | BCP', 'PCBM-60 | TBAI', 'SnO2-c | CBA-SAM', 'MAI | PCBM-60 | PEI', 'PEIE | SnO2-np | ITIC', 't-BPTI | C60 | BCP', 'Carbon-QDs; TiO2-c', 'P2', 'PEI | PCBM-60', 'PCBM-60 | FPI', 'Bphen', 'Au-np | TiO2-c', 'TiO2-fibres', 'TiO2-nanosphere', 'ZnO-c | 3, 4, 5- trimethoxybenzoic acid', 'C60-SAM | C60 | BCP', 'PCBM-derivative11', 'SnO2-nanosheets', 'TiO2-nw | N719', 'TiO2', 'TiO2-c | TiO2-mp | TiO2-c', 'CdS', 'PCBM-60 | Carbon-nt; PEI', 'C60; PCBM-60', 'C60 | B4PyMPM', 'SnO2-QDs', 'PCBM-60; Zn0.8Cd0.2S-np', '(4AMP)I2 | PCBM-60 | BCP', 'C60; PCBM-60 | bis-C60', 'ZnO-c | ZnOS | TiO2-mp', 'SnO2-np | OTES:APTES-SAM', 'PDI | C60 | BCP', 'TiO2-c | (3-Aminopropyl)trimethoxysilane', 'SnO2-c | BA-SAM', 'IDTT2FPDI | PCBM-60 | Bphen', 'ZnO-np | BCP', 'ZnO-c | ZnO-nanofibers', 'GaN', 'CSCNT@Al2O3-c', 'PCBM-60 | C60', 'IT4M | s-Bphen', 'SnO2-np | B2Cat2', 'PCBM-60 | PFN-Br', 'Fe2O3-np', 'ICBA | bis-C60', 'ZnO-c | Graphene; ZnO-np', 'ZnS', 'Corrannulene-derivative', 'D-C60', 'PCPDT-PDI', 'TiO2-3D', 'TiO2-c | TiO2-mp | ZnS', 'PCBM-60 | FPyBr', 'DNDIF3', 'SnO2-np | ZnO-c', 'ZrO2-c | TiO2-c | Al2O3-mp', 'C60-lactone', 'IT4F | s-Bphen', 'Zn2SnO4-np', 'Ag-np; TiO2-c | TiO2-mp | ZrO2-mp', 'PCBM-60 | PFN; ZnO-np', 'ZnO-nanoflake', 'TiO2-c | TiO2-mp | Carbon-QDs', 'PCBM-60 | Ethanl-hydroquinolatolithium', 'Fullerene-2b', 'ITCPTC | C60 | BCP', 'PFN; ZnO', 'PCBM-60 | PrC60MAI', 'TiO2-c | TiO2-mp | Al2O3-mp | TiO2-c', 'PEG; ZnO-np', '2,6-Py | PCBM-60 | PEI', 'TiO2-nanosphere | TiO2-mp', 'MoS2-nanosheets', 'BPTI | PDI-C4 | BCP', 'SnO2-np; ZnO-np | Al2O3-mp', 'MgO | SnO2-np', 'Nb2O5-c | TiO2-mp', 'TiO2-c | Au-np', 'FPI-PEIE', 'TiO2-np | C-PCBOD', 'LiF | C60 | SnO2-c | ZTO', 'TiO2-c | AgInS2-QDs; TiO2-c', 'ATO | TiO2-c | TiO2-mp', 'TiO2-c | CdS', 'Nb2O3 | Al2O3-mp', 'TiO2-c | TiO2-mp | MgO', 'PEIE | SnO2-np', 'Mg0.2Zn0.8O-np', 'N-PDI', 'Fe2O3-c', 'Graphene oxide; TiO2-c', 'F16CuPc | Bphen', 'TiO2-c | TiO2-nanosphere | ZrO2-mp', 'TiO2-c | TiO2-nw', 'SnO2-np', 'SnO2-np | Ethanol amine', 'PCBM-60 | C60/C70-N', 'Zn2SnO4-c | Zn2SnO4-mp', 'ZnO-np | TiO2-mp', 'Fullerene-2c', 'P(NDI2DT-T2)', 'WO3-c | WO3-nw | Cs2CO3 | PCBM-60', 'TiO2-c | Thiourea', 'TiO2-c | ZnO', 'SnO2-nw; Zn2SnO4-np', 'PCBM-60 | BCP | TiO2-c', 'TiO2-c | ZTO-mp', 'ITIC', 'TiO2-c | ZIF-8', 'PCBM-70; ZnO-np | TiO2-c', 'ZnO-c | 4-dimethoxybenzoic acid', 'PCBM-60 | Cs2CO3', 'ZnO-c | p-amino-benzoic acid', 'ZnO-c | SnO2; TiO2-mp', 'ZnO-c | ZnO-nanospheres', '2PDI-2S | PEIE', 'TiO2-c | Ps', 'C60 | Bphen', 'ZnO-c | ZnO-mp | ZrO2-mp', 'SnO2-c | SnO2-np', 'WO3-c', 'IDT6CN-TM | C60 | BCP', 'C60; PCBM-60-DPM-OE', 'PSS | PCBM-60 | BCP', 'TiO2-c | SrTiO3-np', 'PCBM-60 | C60-ETA', 'TiO2-c | Al3O3-mp', 'TiO2-c | TiO2-mp | ZrO2-mp | Co3O4-mp', 'TiO2-c | PPDI-F3N', 'PDTP-DFBT:PCBM-60', 'PCBM-60 | Si-np', 'SnO2-c | C60-5g', 'TiO2-c @ {001} facet', 'PCBB', 'TiO2-c | PCBSD:GD', 'C3N5', 'PCBM-60; PDI-DA', 'nTi-MOF', 'PCBM-60 | pi-PFE3', 'PCBM-derivative12', 'C60 | PEI', 'PCBM-60 | ZrAcac', 'N2200 | PCBM-60', 'ZnO-np | MgO-EA | TiO2-mp', 'NbOH5', 'TiO2-c | TiO2-mp | N719', 'Boron subphthalocyanine chloride | PCBM-60', 'TiO2-c | CsBr', 'TCl-PDI', 'FNCA | C60 | BCP', 'P3', 'SnO2-c | APTES-SAM', 'TiO2-c | C60-BCT-Au-NP', 'PCBM-60 | Ppz', 'ZnO-c | JTCA', 'Ti', 'PCBM-60 | CMB-vTA | AZO', 'Nb2O5 | PCBM-60', 'PS | C60 | BCP', 'BaSnO3-mp', 'NDI3HU-DTYM2 | BCP', 'OCA | C60 | BCP', 'DS2', 'CuCrO2 | C60; PCBM-60', 'SnO2-np | PCBM-60', 'GDY-Tz-CH2(CH2)16CH3; PCBM-60 | C60', 'C60 | ZnO-c', 'C60-BPy | BCP', 'PCBM-60 | ZnO-np', 'Nb2O5 | PCBM-60 | (EMIM)PF6', 'PCBM-70 | C60 | BCP', 'TiO2 | ITIC', 'PCBM-60 | Phen-NaDPO:Sn(SCN)2', 'PCBM-60 | P3HT; PCBM-60', 'TiO2-c | TiO2-mp | Al2O3-c', 'HATNAS3C7-C3h', 'MgZnO', 'TiO2-c | SiO2-mp', 'PDI | PDI', 'ZnO-np | MgO | TiO2-mp', 'TiO2-c | TiO2-mp | ZrO2-mp | Silane-SAM', 'SrTiO3 | Graphene; Al2O3-mp', 'TiO2-c | SnO2-np', 'C60 | SnO2 | ZTO', 'SnO2-c | Cs4SnO4', 'ZnCsO', 'TiO2-c | Na2SO4', 'WOx', 'PCBM-60; TDPP-CN4 | BCP', 'TiO2-c | TiO2-mp | PABA-SAM', 'PFN | PCBM-60', 'TiO2-c | KH550', 'Teflon | C60 | BCP', 'a-DMEC70', 'EVA | PCBM-60 | BCP', 'SnO2-c | MgO-c', 'TiO2-c | Al2O3-mp | NiO-np', 'PTO', 'PCBM-60; TiO2-np', 'TiO2-c | Alq3', 'GO-Li | TiO2-c', 'ZnO-c | TiO2-c', 'CITP-SAM', 'NDI-Se', 'ZnO0.85Sn0.2O1.2', 'PCBM-70 | Fe2O3-np', 'TiO2-c | TiO2-mp | Poly(TA)', 'TEAI | PCBM-60 | PEI', 'ZnO-c | AZO-c', 'IPH | PFNBr', 'PC61BPy | BCP', 'Ag-np; TiO2-c | TiO2-mp', 'ICTA | C60 | BCP', 'Fulleropyrrolidinium Iodide', 'DMEC-60 | LiF', 'ZnO-np', 'TiO2-c | SiO2-np | TiO2-c | SiO2-np | TiO2-c | SiO2-np | TiO2-c', '1,2-diaminoethane | C60 | BCP', 'TiO2-c | CoCr', 'ZnSO | TiO2-mp', 'SnO2-np | ZnO-np', 'TiO2-mp | D35', 'ZnO-c | PCBM-60', 'HATNASOC7-Cs', 'TiO2-c | TiO2-mp | PTDPQ', 'TiO2-c | Ba0.5Sr0.5SnO3-np', 'SnO2-c | SnO2-nw', 'Mg0.05Zn0.95O-np', 'PCBM-60 | TPBi', 'MPMIC60 | ZnO-np', 'TiO2-c | CdSe-tetrapod', 'IPB', 'TiO2-c | TiO2-mp | NaPTH', 'TiO2-mp | SAM1', 'PCBM-70 | Lif', 'TiO2-c | C60-NH2', 'PCBM-60 | P4N4', 'Ethyl acetate; I2; LiI; TBP; Urea', 'SnO2-np | Dompamin-SAM', 'Ti | Au', 'MgO-c | TiO2-mp', 'SnO2-np | TiO2-np', 'PN', 'ZnO-c | CdS', 'TiO2-c | PCBA', 'TiO2-c | In2O3-c', 'SnO2-c | SnO2-nw | TiO2-c', 'WO3-np', 'C60; C70 | BCP', 'Fe2O3', 'TiO2-c | Heparin-Na', 'WO3-np | Cs2CO3 | PCBM-60', 'PCBM-60 | EFGnPs-F', 'C60 | SnS', 'ZnO-mp', 'PCBM-60-np | PrC60MA | PEI', 'TiO2-c | TiO2-mp | PCBM-60; PVK', 'C60 | SnO2-c | ZTO', 'TiO2-np | C60', 'NDI-TriSe', 'TPA-3CN | BCP', 'TiO2-c | ZnO-nw | ZnO-c', 'PEIE | NDI-H', 'ITCP-M | BCP', 'Alq3; C60 | LiF', 'PDPP3T; PCBM-60', 'PCBM-60 | Ag-np | BCP', 'TiO2-c | STHNP', 'SrTiO3', 'B4PyMPM | C60 | TmPyPB', 'Ti3C2Tx', 'TiO2-c | graphite nanofibers', 'TiO2-c | TiO2-mp | Al2O3-c | ZrO2-mp', 'SnO2-c | SnO2-mp', '[EMIM]PF6-IL', 'HATNAS3C4', 'TiP2-c | PCBM-60', 'Nb2O5-c', 'TiO2-c | SrTiO3-np; TiO2-nw', 'Poly(9-vinylcarbazole) | PCBM-60', 'TiO2-c | PNP', 'SnO2-c | Ag | SnO2-c | SnO2', 'ZnO | Au-nw', 'ZnSnO4-c | ZnSnO4-mp', 'PCBM-60 | PFN', 'SnO2-c | Al2O3-mp', 'PCBM-60 | C70', 'P1', 'PCBM-70 | AZO-np', 'TiO2-c | TiO2-mp | Al2O3-mp | NiO-mp', 'PCBM-60 | ZSO-np', 'ZnCsO | PCBA', 'ZnO-c | WPF‐6‐oxy‐F', 'Al2O3-c', 'Carbon', 'PCBM-60 | PCBDANI | LiF', 'TiO2-c | TiO2-mp | MTPS-SAM', 'IDT6CN-4F | C60 | BCP', 'TiO2-c | ZnO-c | C60', 'PCBM-60; PS-b-PEO', 'TiO2-c | TiO2-mp | Li-TFSI', 'PCBM-60 | P3TMAHT', 'PCBM-70 | TiO2', 'TiO2-c | TiO2-IO', 'AZO-c', 'PDI | BCP', 'TiO2-c | Au@SiO2-nw; TiO2-mp', 'PCBM-60 | C60-N', 'TiO2-c | TiO2-mp | In2O3', 'TiO2-c | CuI', 'PCBM-60 | BCP; TZ-3', 'ZnO-c | Al2O3-mp', 'TiO2-c | TiO2-nw | Al2O3', 'C60; PhIm', 'PCBM-60 | bis-C60', 'PEAI | PCBM-60 | BCP', 'pSNT', 'Bi2S3', 'SnO2-c', 'ZnO-c | CPTA; PbI2', 'ZnO-c | C-PCBSD', 'ZnO-c | Ag | TiO2-c', 'ZnO-c | ZnO-nw | Au-np', 'BaCoF4', 'Unknown', 'ZnO-c | C60', 'TiO2-c | TiO2-mp | PCBM-60', 'SnO2-c | 3-(1-pyridinio)-1-propanesulfonate', 'Graphitic carbon nitride | PCBM-60; Graphene | BCP', 'PCBM-60 | LiF', 'TiO2-c | TiO2-mp | 4PA-SAM', 'Graphene nanoribbons hPDI3-Pyr-hPDI3', 'TiO2-np | PCBM-70', 'PCBM-60 | TBAOH', 'PCBM-70 | ZnO-np', 'F8BT | C60', 'Carbon-nt', 'SnO2-c | TiS2', 'HPDT | PCBM-60 | Bphen', 'Nb2O5-c | Nb2O5-mp', 'Al2O3; TiO2-c', 'TiO2-c | Sr2CeO4:Eu', 'PEIE | TiO2-c | TiO2-mp', '1-ethyl-3-methylimidazolium iodide | C60 | 1-ethyl-3-methylimidazolium iodide', 'ITIC | BCP', 'N2200', 'C60 | LiF', 'TiO2-c | Ag | TiO2-c', 'TEACl | PCBM-60 | PEI', 'ZnO | MgO | EA | TiO2-mp', 'ITCPTC | Rhodamine 101 | LiF', 'IDIC', 'PCBM-60 | PDI-Br', 'SFX-PDI4 | TiO2-c', 'In2O3-c', 'ZnO-nw', 'Graphene oxide', 'TiO2-c | 3-PA-SAM', 'ITIC | PEIE', 'ZnO-c | ZnO-nw | CdS-QDs', 'TiO2-c | TiO2-mp | Al2O3-mp', 'PCBM-60 | ZnO-np | DPO', 'TiO2-c | Ba0.8Sr0.2SnO3-np', 'C60 | PN4N', 'TiO2-c | TiO2-mp | ZrO2-mp | NiO-mp', 'TiO2-c | SnO2-c', 'PSS | PCBM-60 | ZnO-np', 'TiO2-c | TiO2-mp | La2O3', 'PCBM-60 | SnO2-c', 'PFN-OX', 'PVDF-TrFE | C60 | BCP', 'ZnO-np | ZnO-nanospheres | PEI', 'TPE-PDI4 | C60 | BCP', 'PEABr | PCBM-60 | BCP', 'PbZrTiO3', 'TiO2-c | TiO2-nanofibres', 'TiO2-c | SiO2-np', 'ICBM | bis-C60', 'Mix-DMEC70', 'N2200 | ZnO-np', 'TiO2-c | SnO2-np | BSO-mp', 'LiF | C60 | BCP', 'Fluorosilane | C60 | BCP', 'C60(OH)16 | C60', 'TiO2-c | TiO2-mp | MAI', 'ZnO-c | ZnO-nw | ZrO2-mp', 'SnO2-nw', 'B2F | C60 | BCP', 'PCBB-S-N | PCBM-60 | C60 | BCP', 'SnO2-np | Li-TFSI', 'PCBM-60 | pFN-Br', 'PCBM-60 | Cr2O3', 'TiO2-mp | TiO2-c | TiO2-nw', 'TiO2-nw | TiO2-c', 'PO-TAZ | ZnO-c', 'PCBB-3N-3I | PCBM-60', 'PCBM-60 | PN6', 'TiO2-c | TiO2-mp | Graphene oxide', 'SnO2-np | SnO2-mp', 'LiF | C60', 'PCBM-60; MAI', 'PCBM-60 | TiO2-c', 'PCBM-60 | Poly(2-ethyl-2-oxazoline)', 'N2200; PCBM-70; TT', 'Br-PDI | ZnO-np', 'MoS2; TiO2-np', 'TPE-PDI4 | Rhodamine 101 | LiF', 'TiO2-mp | N719 | LPP', 'IDT6CN | C60 | BCP', 'PCBM-60 | bis-C61', '(RhCp*Cp)2 | C60', 'Br-PDI', 'C-PDTON | ZnO', 'SrGeO3', 'PCBM-60 | DEME-TFSI', 'Nb2O5 | TiO2-mp', 'ZnO-c | MgO | TiO2-mp', 'TiO2-c | TiO2-mp | SnO2-mp', 'P(NDI2OD-T2)', 'SnO2-np | PEG; SnO2-mp', 'PYPH | LiF', '(2Z,2′Z)-3,3′-(5,5′-(2,7-dioctyl-1,3,6,8-tetraoxo-1,2,3,6,7,8-hexahydrobenzo[lmn][3,8] phenanthroline-4,9-diyl)bis(thiophene-5,2-diyl))bis(2-(3,5-bis (trifluoroomethyl)phenyl) acrylonitrile)', 'TiO2-nw | Sb2O3-np; SnO2-np', 'bis-PCBM', 'SnO2-c | BMIMBF4', 'ITIC-Th | Zn(acac)2', 'ITCPTC-Se | LiF', 'TiO2-c | SnO2-np | PCBM-60', 'In2S3 | TiO2-c | TiO2-mp', 'TiO2-c | C60', 'TiO2-mp | Ti:TiO2 | ZrO2-mp', 'Cs2SnI6', 'TiO2-c | KH570', 'TiO2-c | TiO2-mp; TiO2-nw', 'ZnO-np | C60', 'PNDI20D-TT | ZnO', 'PCBDAN', 'SnO2-QDs | C60', 'd-PCBM-60; PCBM-60', 'TiO2-c | Ag-nw', 'PEIE | TiO2-c', 'NDIF1', 'PCBM-60 | C60 | Rhodamine 101 | LiF', 'TPE-DPP12 | C60 | BCP', 'TiO2-nw | TiO2-np', 'ZrO2-c | PCBM-60', 'ZnO-np | ATTA', 'ZnO-c | CdSO', 'Black P-QDs', 'MSAPBS', 'SnO2-c | PCBM-60', 'TiO2-c | SAED', 'ZnO-np | ZnO-nw', 'TiO2-c | PCBM-62', 'TiO2-c | TiO2-mp | CBA-SAM', 'AZO', 'PCBM-60-mp', 'SnO2-c | SnO2-c', 'B2F | C60', 'BenMeIM-Cl | C60', 'PCBC6 | BCP', 'B2T | C60', 'TiO2-c | TiO2-mp | HOOC-C11H22-SH', 'c-HATNA | bis-C60', 'TiO2-np | TiO2-c', 'C60 | SnO2', 'TiO2-mp | N719', 'SnO2-np; TiO2-np; MXene', 'TiO2-c | NaYF4:Yb:Tm-np', 'CdS | PCBM-60', 'ZnO-np | TiO2-np', 'Fullerene @ F4', 'SnO2-np | PCBM-60; PCBA | C60', 'ZnO-c | SnO2-np', 'PCBM-60 | m-PYBrZnPor', 'ZnO-c | ZnO-nw | PCBM-60', 'TiO2-c | ZrO2-c | Al2O3-mp', 'PCBM-60 | DMAPA-C60', 'NDI-ID', 'MgO-c', 'PCBM-60 | BCP | AZO-c', 'TiO2-c | SnO2-QDs', 'TiO2-c | TiO2-mp | Bacteriorhodopsin', '[EMIM]PF6-IL | C60', 'ZnO-c | 3-aminopropanioc acid-SAM', 'TiO2-c | TiO2-c', 'TiO2-c | PCBM-60', 'SnO2-c | TiO2-c', 'Graphene | TiO2-mp', 'TiO2-c | TiO2-mp | GABAHI', 'WO3-c | WO3-nw', 'pBTT', 'TiO2-c | TiO2-mp | ZrO2-mp | Carbon-mp', 'ZnSe', 'ADAHCl | C60 | BCP', 'C60', 'PCBM-60 | Zr(acac)4 | PEI', 'PCBM-60; PCBDAN', 'Poly-EDOT-C60', 'TiO2-c | TiO2-mp | NBA-SAM', 'C60/C70-N', 'PCBM-60 | Rhodamine 101', '4-methoxythiophenol', '4-HI-PVK | PCBM-60 | BCP', 'C60 | ZnSe', 'ZnO-c | EMIM-PF6', 'C60 | bis-C60', 'TiO2-c | TiO2-mp | Graphene-QDs', 'NDP-V | Rhodamine 101 | LiF', 'CYOP | PCBM-60 | PEIE', 'LiF | C60 | SnO2-c | ZnSnO2-c', 'CuInS2 | Al2O3-np', 'TiO2-c @ {101} facet', 'ZnO | MgZnO', 'EDTA', 'ZnO-np | SnO2-np | C60-SAM', 'Nb2O5', 'C60 | BCP | LiF', 'PCBM-60 | C60 | TPBi', 'TiO2-c | PDI2', 'TiO2-c | MgO-c | TiO2-mp', 'SiPc-Py-2 | PTCDA | PO-T2T', 'PCBB-3N | PCBM-60', 'TiO2-c | TiO2-mp | MgO-c', 'TiO2-c | TiO2-mp | Au-np | MgO', 'PCBM-70 | Rhodamine 101', 'TiO2-c | TiO2-mp | CdSe-QDs', 'In2O3-c | SnO2-c', 'SnO2-np | APTES-SAM', 'Ag-np | TiO2-c', 'SnO2-nt', 'CDIN', 'TIPD; ZnO-np', 'TiO2-c | TiO2-mp | 4-aminobenzoic acid-SAM', 'ZnO-c | ZnO-nanowells', 'TiO2-c | TiO2-mp | C60-SAM', 'TiO2-c | TiO2-mp | ZrO2-mp | PbTiO3', 'Fullerene-SAM', 'SnO2-c | 1‐butyl‐3‐methylimidazolium bromide', 'PCBM-60 | PPy', 'TiO2-c | TiO2-nanosphere', 'TiO2-c | CsPbBr3-QDs', 'TiO2-c | TiO2-mp | B2Pin2', 'TiO2-c | TiO2-mp; ZrO2-mp', 'TiO2-c | NAMF-Cl', '4-DMABA | PCBM-60 | BCP', 'ZnO-c | ZTO | PCBM-60', 'TiO2-c | TiO2-mp | GO-Li', 'NiO-c', 'F8BT', 'TiO2-c | TiO2-mp | Cs2CO2', 'PCBM-60 | Bis-FITG', 'ZnO-c | Al2O3-c | TiO2-mp', 'MgO', 'TMAH', 'C60 | [EMIM]PF6-IL', 'Ca | C60', 'TiO2-c | C60-ETA', 'TiO2-c | C60-SAM | SiO2-np', 'TN-P', 'diPDI | TiO2', 'ZnO-c | Cs2CO3', 'Graphene oxide; ZnO-np', 'ZnO-np | PFS-FTEG', 'PCBM-60 | Et2PrNMe-CF3BF3', 'TiO2-c | TiO2-nt | ZrO2-mp', 'TiO2-c | TiO2-nt', 'TiO2-c | SnO2', 'SnO2 | PCBM-60', 'f-C70 | C60', '1‐benzyl‐3‐methylimidazolium chloride', 'SnO2-c | PCBM-60; PMMA', 'PCBM-60 | CeOx-np', 'TiO2-macroporous', 'ICMA', 'TiO2-c | TiO2-mp | PcM-Cou', 'TiO2-c | SnO2-nanosheets', 'HATNASO2C7-Cs', 'P4', 'PV-PDI', 'PCBM-60 | PrCMA', 'TiO2-c | TiOZn-nanosphere', 'TiO2-c | TiO2-mp | TiN', 'TiO2-c | Ag@TiO2-nw', 'BCP', 'Choline chloride | C60 | BCP', 'ZnO-c | Al2O3-c', 'BaSnO3-nw', 'TDPP-CN4 | BCP', 'PCBM-60 | Bphen', 'Cr2O3', 'NDP-V | C60 | BCP', 'Carbon-np; SnO2-np', 'SnO2-np | DPC60', 'ZnO-np | MgO', 'TiO2-c | TiO2-mp | ZIF-8', 'BiFeO3', 'TiO2-c | PBCM-60', 'PCBM-60 | BCP | Ga2O3', 'TiO2-c | TiO2-nw | TiO2-np', 'CMB | AZO', 'C60-MPy | BCP', 'CdI2 | C60 | BCP', 'Me4NI | PCBM-60 | BCP', 'LiQ; PEIE | PCBM-60', 'C-PDTON | SnO', 'C60MC12 | AZO-np', 'SnO2-c | NaOH', 'TiO2-c | ZrO2-mp', 'TiO2-c | NaYF4-np; TiO2-mp', 'PCBM-60 | Bis-FIMG', 'SnO2-c | ZnTiO3-mp', 'TiO2-c | TiO2-mp | SQ2', 'TiO2-np | CsBr', 'TiO2-c | TiO2-mp | SiO2-mp', 'C60 | Al2O3-mp', 'PCBM-70 | TiO2-c', 'DPC60', 'PCBM-60 | MUTAB', 'FeO | PCBM-60', 'ZnO-np | SnO2-np', 'ZnO-c | APTES-SAM', 'SnO2-c; SWCNTs', 'PN-F50', 'NDI-BiSe', 'ZnO-c | ZnO-mp', 'TiO2-c | Ag-np | TiO2-c', 'SnO2-np | LiF', 'TiO2-c | bis-PCBM; DMC', 'PCBM-60 | AgAl-np | LiF', 'PCBM-60 | C60 | BCP', 'PCBM-60 | pi-PFE4', 'TiO2-c | TiO2-mp | Al2O3-np', 'TiO2-nanocones | TiO2-mp', 'TiO2-c | PCBM-60 | C60-ETA', 'PC61BEH', 'TiO2-c | TiO2-mp | ZIF-8-10', 'Mg0.25Zn0.75O-np', 'PCBB-OEG; PCBM-60 | C60 | BCP', 'NMPFP', '2H-TaS2', 'ICBA | BCP', 'C70 | LiF', 'TiO2-c | TiO2-mp | PCBM-60; PMMA', '1-ethyl-3-methylimidazolium iodide', 'TiO2-c | MgO | TiO2-mp | ZrO2-mp', 'SrTiO3 | TiO2-mp | Al2O3-mp', 'BaSnO3', 'CMB-vTA | AZO', 'TiO2-c | NiO-np', 'ITIC-Th | BCP', 'PCBM-70 | ZrO3', 'FPI-PEIE | PCBM-60', 'TiO2-c | PCBSD', 'ZnO-c | ZnO-nw | Al2O3-c', 'PCBM-60 | AZO | BCP', 'Zn2SnO4 | TiO2-mp', 'BCP | C60', 'ZnSO', 'TiO2-c | NAMF-H', 'CdZnSeS-QDs | C60 | BCP', 'TiO2-c | TiO2; CoCr-mp', 'IPH | PDINO', 'PCBM-60 | HDAC', 'C60 | SnO2-c', 'Ag-nanocube | TiO2-c', 'SnO2-c | C9', 'NbOx', 'IDIC | C60 | BCP', 'NiO-c | TiO2-c', 'ICBA | PrC60MAI', 'TiO2-c | TiO2-mp | CdS', 'AZO-c | ZnO-nw', 'Fullerene @ F3', 'SnO2-c | TiO2-mp | ZrO2-mp', 'CdS-nw', 'PCBM-60 | F-R-COOK', 'TiO2-c | 2-PA-SAM', 'TiO2 | C60', 'TiO2-nt | ZnO-c', 'SnO2-c | C60-5c', 'PT-PDI', 'NDI-ID (RR)', 'TEABr | PCBM-60 | PEI', 'FPDI', 'MgO | SrRuO3 | SrRuO3', 'PCBM-60 | TrNBr', 'TiO2-c | TiO2-mp | Au@SiO2-np', 'PCBM-60 | EFGnPs-H', 'TiO2-c | Sr2CeO4:Sm', 'TiO2-c | TiO2-mp', 'ZnO | PTFE', 'ZnO-c | TiO2-c | PCBM-60', 'C60 | 3TPYMB', 'PCBM-60 | Rodhamine 101 | C60 | Rodhamine 101 | LiF', 'PCBM-60; ZnO-np', 'PCBM-60 | TIPD', 'PCBM-60 | pi-PFE1', 'C60; PCBM-60 | Zr(acac)4', 'TiO2-c | TiO2-nw | SiO2-nanocolumns | TiO2-nw', 'PCBM-60 | F-C60; bis-C60', 'TiO2-c | BZnTPP', 'C60 | BenMeIM-Cl', 'Au-np; SnO2-np', 'ZnO-c | PCBA', 'PC61BBz | BCP', 'SnO2-np; TiO2-c', 'SiO2-c | ZnO-c', 'CdSe-QDs', '(2Z,2′Z)-3,3′- (5,5′-(2,7-dioctyl-1,3,6,8-tetraoxo-1,2,3,6,7,8-hexahydrobenzo [lmn][3,8]phenanthroline-4,9-diyl)bis (thiophene-5,2-diyl))bis(2-(4-(trifluoromethyl)phenyl) acrylonitrile)', 'PDI-HE', 'PCBM-60 | SnO-c | SnZnO', 'ZSO-np | PCBM-60', 'SnO2-c | Al2O3-np', 'PCB-C4', 'PCBM-60 | Zr(Acac)', 'PEIE', 'PTTI-2', 'ZnO-c | Nb2O5', 'SnO2-np | CPTA', 'TiO2-c | SnO2-mp', 'TiO2-c | TiO2-nanoballs | SiO2', 'TiO2; WOx', 'PCBM-70 | LiF', 'TiO2-c | Zn2SnO4-mp', 'PCBM-60 | C60 | LiF', 'ITCPTC-Th | C60 | BCP', 'ZnO | 1,2-ethanedithiol', 'TiO2-c | NAMF-Br', 'TiO2-c | Au@SiO2-np; TiO2-mp', 'TiO2-c | CsPbCl3-QDs', 'WO3-c | Cs2CO3 | PCBM-60', 't-BPTI | PDI-C4 | BCP', 'TiO2-c | ZnO-c', 'TiO2-c | TiO2-mp | SnO2-c', 'SnO2-np | Graphene-QDs', 'SnO2-c | KOH', 'PCBM-60 | BCP', 'ZnO | ZnO-nw', 'TiO2-c | CsPbI3-QDs', 'HATNAS3C7-Cs', 'TiO2-c | Al2O3-mp; Ag@TiO2-np', 'TiO2-c | KY7F22-np', 'PCBM-60 | Carbon', 'ZnO-np | Mg-EA', 'SnO2-np | Choline Chloride', 'SnO2-np | TiO2-mp', 'TiO2-c | C60-SAM | Al2O3-np', 'ITIC | PCBM-60 | PEIE', 'Au@TiO2-nw; TiO2-np', 'TiO2-c | DMF', 'ICBA | C60 | BCP', 'CNTPA-PCBM | bis-C60', 'ZnTiO3-c', 'Mg0.15Zn0.85O-np', 'PCBM-70 | Bphen', 'PCBM-60 | PEI', 'PCBM-60 | DTAB', 'TiO2-c | ZnPtriazine(gly)2', 'ITCPTC-Th | LiF', 'PCBM-60; PCDTBT', 'PFN', 'H-PDI | ZnO-np', 'tris-PCBOE | ZnO', 'TiO2-c | TiO2-nw | TiO2-c', 'PCBM-60 | PEIE', 'TiO2-c | PS:TiO2-mp', 'Fullerene @ F1', 'TiO2-c | Au@TiO2-np; TiO2-mp', 'ZTO | SnO2-c', 'TiO2-c | Au@Ag | TiO2-mp', 'TiO2-c | TiO2-mp | CaTiO3-c', 'HfO2 | SnO2-np', 'DBP | PCBM-60 | Bphen', 'IBF-Ep', 'C60; PhIm | C60', 'PCBM-70 | AZO', '1-ethyl-3-methylimidazolium iodide | C60', 'SnO2-c | Graphene', 'TiO2-c | rGO:TiO2-nanofibrse', 'TiO2-c | TiO2-mp | CaTiO3', 'TiO2-c | [BMIM]BF4', 'C70 | BCP', 'P(NDI2OD-T2); PCBM-60 | BCP', 'In2O3 | PCBM-60', 'CdS | TiO2-c', 'TiO2-c | TiO2-mp | ZrO2-mp', 'ZnO-c | Hexamethylenetetramine', 'TiO2-c | Graphene-QDs', 'BCP; PCBM-60', 'PCBM-60 | AZO-np | PEI', 'PCBM-60 | BCP | LiF', 'C60 | PCBM-60 | BCP', 'ITIC | Zn(acac)2', 'PCBM-60 | TIPD; ZnO-np', 'ZnO-c | ZnO-nanofibers | PEI', 'CeOx | PCBM-60', 'SnO2-c | PEIE | PCBM-60', 'C60 | PAA', 'ZnLiO | PCBA', 'GAN', 'TiO2-c | TiO2-mp | beta-Alanine-SAM', 'TiO2-c | Al2O3-c | TiO2-mp', 'TiO2-c | TiO2-mp | CsI', 'TiO2-c | C60-SAM', 'C60 | TiO2-c', 'PCBM-60 | ZnO-np | ACO-c', '3-acetylpyridine | PCBM-60 | BCP', 'TiO2-c | TiO2-nanosphere | TiO2-mp', 'PCBM-60 | pi-PFE2', 'ZrO2-np', 'C60-SAM | SnO2-c | PCBM-60', 'PCBM-60 | BCP; TZ-4', 'BPB-M | LiF', 'SnO2-np | OEABS', 'PCBM-60 | Lif', 'NDIF2', 'ZnO-c | ZnO-c | Ba(OH)2', 'TiO2-c | TiO2-mp | Aminocaproic acid', 'C60 | Zr(acac)4', 'PCBM-60 | CTAB', 'HATNA-F6', 'PCBM-60 | ZnO-np | PEI', '(RuCp*mes)2 | C60', 'PTCBI', 'PTCBI | BCP', 'PCBM-60 | EGME', 'TiO2-nanocones', 'IT-M | Zn(acac)2', '2-acetylpyridine | PCBM-60 | BCP', 'C60 | C60; PhIm', 'SnO2-nw | SnO2-nw', 'Ti3C2', 'Al:ITO-c | SnO2-c', 'WOx | I-Ph-PA_C60-C6-PA', 'WOx | C60', 'SnO2-np | PFN-Br', 'PEIE | PCBM-60 | Bphen', 'TiO2-c | SrTiO3:TiO2-mp', 'TiZnO12-c', 'TiO2-c | Carbon-nt; TiO2-mp', 'PTCA', 'SnO2-c | SnO2-mp | TiO2-c', 'C60; Phlm | C60', 'PCBM-60 | ICL', 'PCBM-60 | Alq3', 'TiO2-c | TiO2-nanoballs', 'TiO2-nanoleaves'])))

    thickness = Quantity(
        type=str,
        shape=[],
        description="""
    A list of thicknesses of the individual layers in the stack. Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous filed.
- State thicknesses in nm
- Every layer in the stack have a thickness. If it is unknown, state this as ‘nan’
- If there are uncertainties, state the best estimate, e.g write 100 and not 90-110
Example
200
nan |250
100 | 5 | 8
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '40.0 | 50.0', '50.0 | 7.0', '40.0 | nan', '5.0 | 65.0 | nan', '50.0 | 200.0', 'nan | 400.0 | 1700.0', '20.0 | 400.0', '40.0 | 3.0 | 2.0 | 2.0', '40.0 | 150.0', '14000.0', '50.3', '27.0 | 600.0', '40.0 | 210.0', 'nan | 1100.0', '25.0 | 150.0', 'nan | 600.0', 'nan | 2000.0 | nan', '0.0', '30.0 | 351.0', '12.7 | 40.0 | 6.0', '27.0 | 175.0', 'nan | 950.0', 'nan | 100.0 | 2000.0', '40.0 | 353.0', '1800.0', '60.0 | 12.0', 'nan | 480.0 | 900.0', 'nan | 400.0 | nan', 'nan | 600.0 | 1200.0', '160.0 | 40.0', '141.0 | 200.0', '2.4', '50.0 | 1000.0 | 1000.0', 'nan | 325.0', 'nan | 1000.0 | 4500.0', '614.0', '87.0', '8.0 | nan | 1.0', '70.0', 'nan | 5400.0', 'nan | 1000.0 | 1008.0', '500.0', '85.0 | nan', '80.0 | 365.0', '4.0 | nan', '76.0', '90.0 | 400.0', '45.0 | 4.0', '6.0 | 20.0 | 8.0', '35.0 | 150.0', '100.0 | nan | 2.0', '100.0 | 300.0', '2.0 | 4.0 | nan', '50.0 | 50.0 | 300.0', '30.0 | 440.0', '14.0 | nan', '2.0 | 1.0', '31.1', '240.0', '50.0 | 100.0 | 400.0 | 50.0', '101.0 | 500.0', 'nan | nan | 7.0', 'nan | 320.0', '75.0 | 500.0', '95.0 | 365.0', '50.0 | 480.0', '30.0 | 600.0', 'nan | 25.0', 'nan | 1000.0 | 1010.0', '60.0 | 2500.0', '30.0 | 200.0 | nan', '40.0 | 300.0', '1.0 | 25.0 | 6.0', '120.0 | 0.5', '26.0', '55.0 | 5.0', 'nan | 30.0 | 10.0', '700.0 | 80.0', '50.0 | 6.3', 'nan | 500.0 | 500.0 | 10000.0', '20.0 | nan | 20.0', '60.0 | 460.0', '80.0 | 120.0', 'nan | 2000.0 | 2000.0', '9.0', '72.0 | 200.0', '20.0 | 20.0 | 20.0', 'nan | 600.0 | nan', '30.0 | 12000.0', '50.0 | 200.0 | 115.0', 'nan | 270.0', '5000.0', '50.0 | 250.0', 'nan | 500.0 | 3000.0', '700.0 | 2400.0', '30.0 | 130.0', '70.0 | nan', '17.0 | 350.0', '90.0 | 300.0', '80.0 | 1050.0 | 1.0', '70.0 | 80.0 | nan', '6.0 | 54.0', '36.0', '50.0 | 30.0', '23.0 | nan | 1.0', 'nan | 100.0', 'nan | 440.0', '50.0 | 1.0', 'nan | 220.0', '15.0 | 50.0', '50.0 | 180.0', 'nan | 55.0', '106.0 | nan', '25.0 | 15.0', '100.0 | 130.0 | nan', '80.0 | 1400.0', '23.0 | 7.0', '50.0 | 620.0', 'nan | 500.0 | 1300.0', '80.0 | 300.0 | 1400.0', '40.0 | 10.0', '368.0 | nan', '30.0 | 240.0', 'nan | 20.0 | 5.0', 'nan | 2000.0 | 1000.0', '58.0 | 200.0', '0.2 | nan', '135.0', '2.0 | nan', 'nan | 120.0', '15.0 | 6.0', '30.0 | 20.0 | 8.0', '90.0 | 200.0', '5.5', '200.0 | 700.0 | 1000.0', '40.0 | 40.0 | 150.0 | 150.0', 'nan | 750.0', '8.0 | nan | nan', '80.0 | 900.0 | 1.0', '1.2 | 10.0 | 8.0 | 2.0', '277.0', '60.0 | 24.0', '50.0 | 400.0 | 400.0', '27.0 | 275.0', 'nan | 500.0 | 500.0', 'nan | 6.0 | nan | nan', '20.0 | 30.0', '50.0 | 5.0 | 5.0', 'nan | 1300.0', '60.0 | 15.0', '10.0 | nan | 1.0', 'nan | 290.0', '50.0 | 1000.0 | 800.0', '91.0', '60.0 | 30.0', '15.0 | 180.0', '7.5', '405.0', '100.0 | 30.0 | 10.0', '80.0 | 20.0', '13.0 | 20.0 | 8.0', '30.0 | 20.0', 'nan | 100.0 | nan', '65.0 | nan', '50.0 | 50.0 | 6.3', 'nan | 453.0', '30.0 | 350.0', '5.0 | 20.0 | 8.0', '40.0 | 5.0 | 40.0', '40.0 | 3.0', '75.0', '20.0 | 5.0', 'nan | 0.6', '40.0 | 3.0 | 2.0 | 0.5', 'nan | 180.0', 'nan | 600.0 | 500.0', 'nan | 1050.0 | 350.0', 'nan | 5502.0', '22.0', '10.0 | 40.0', '50.0 | nan | 30.0 | 10.0 | 0.05 | 0.8', '35.0 | 40.0', '63.0', '60.0 | 3.0', '34.0', '50.0 | 300.0 | 500.0', '10000.0 | nan | nan', '100.0 | 60.0', '43.0 | nan | 1.0', 'nan | nan | 305.0', '100.0 | 600.0', '70.0 | 80.0 | nan | nan', '280.0', '60.0', '40.0 | 360.0', '45.0 | nan', '7.0', '440.0', '400.0', 'nan | 0.7', '25000.0', '1.5', 'nan | 450.0', '128.0', '20.0 | nan', 'nan | 660.0', '600.0 | 340.0', '208.0 | 200.0', '40.0 | 250.0', '50.0 | 150.0', 'nan | 1000.0 | 1004.0', '50.0 | nan | 1.0', 'nan | 200.0', '12.0', '60.0 | 120.0', '800.0 | 200.0', '100.0', '40.0 | 10.8', 'nan | 1073.0', '70.0 | 130.0', '55.0 | 75.0', '20.0 | 260.0', '15.0 | 1.0', 'nan | nan | 1.5', '15.0 | 400.0', '55.0 | nan', '38.69', '120.0 | 700.0', '16.39', '60.0 | 530.0', '3.0 | 25.0 | 6.0', '70.0 | 10.0', '75.0 | 365.0', '50.0 | 600.0', 'nan | 3600.0', '30.0 | 550.0', '290.0', '100.0 | 1.3', 'nan | 20.0 | 0.6', '55.0 | nan | nan', '50.0 | 540.0', '43.0 | 10.0', '50.0 | 100.0', '40.0 | 5.0', '20.0 | 25.0 | nan', '60.0 | 1500.0', '50.0 | 800.0 | 700.0', '35.0 | 15.0', '30.0 | 150.0', '40.0 | 7.5', 'nan | 15.0 | 6.0', '140.0 | 380.0', '10.0 | 150.0', 'nan | 670.0', '20.0 | 6.0', 'nan | 350.0 | nan', '17.0 | 170.0', '80.0 | 60.0', '20.0 | 10.0 | 10.0', '500.0 | 3000.0', '90.0 | 60.0', 'nan | 20000.0', '15.0 | 20.0', 'nan | 170.0', '60.0 | 50.0 | 45.0', '22.0 | 150.0', '60.0 | nan | 10.0', '7.0 | nan', '44.0', '45.0 | 100.0', '550.0', '5.0 | 30.0', '65.0 | 30.0', '140.0 | nan', '119.0 | 35.0', '50.0 | 1000.0 | 500.0', '3.0 | 50.0', '80.0 | 300.0', '120.0', '50.0 | 500.0', 'nan | 1000.0 | 1011.0', '110.0', '20.0 | 15.0', '100.0 | 340.0', 'nan | 15.0 | 150.0', '68.0 | nan', 'nan | 7.0 | nan | nan', '150.0 | 2.0', '65.0 | 10.0', '40.0 | 20.0 | 40.0', '590.0', '770.0', '90.0 | nan', '70.0 | 20.0', '17.5 | 100.0', '50.0 | 3.0', '75.0 | 200.0', '60.0 | 7.5', '85.0 | 150.0', '33.0 | 200.0', '15.0 | 220.0', 'nan | 452.0', 'nan | 700.0', '5.0 | 7.5 | 20.0', '100.0 | 4.3', '100.0 | 2.0', '800.0 | 340.0', 'nan | 1000.0 | 1002.0', '60.0 | 0.5', '47.03', '210.0', '5.0 | 80.0', '29.0 | 20.0 | 8.0', '40.0 | 260.0', '251.0', '0.5 | 90.0 | 7.0', '42.0 | nan | 1.0', 'nan | 3.0 | 1.0', '17.0 | 140.0', '129.0', '30.0 | 15.0', '14.0 | nan | nan', '50.0 | 220.0', 'nan | 10.0', 'nan | 0.0', 'nan | 9400.0', '6.0 | 340.0', '40.0 | 345.0', '50.0 | 75.0', '17.0 | 150.0', 'nan | 604.0', '30.0 | nan | 200.0', '100.0 | 2.1', '50.0 | 420.0', '40.0 | 3.0 | 2.0 | 1.0', '10.0 | 75.0', 'nan | 550.0 | 2500.0', '10.0 | 50.0', 'nan | 1000.0 | 1005.0', 'nan | 6.0', '100.0 | nan', 'nan | 1000.0 | 2000.0', 'nan | 510.0', '40.0 | 8.0', '1900.0', '90.0', 'nan | 857.0 | 2000.0', '32.0 | nan', '10.0 | 340.0', '89.0 | nan', 'nan | 130.0', '10.0 | 180.0', '40.0 | 4.0', '300.0 | 500.0', '668.0', '142.0 | 20.0 | 8.0', '40.8 | 8.0', '29.0 | nan | 1.0', 'nan | 200.0 | 200.0', 'nan | 800.0', '15.0 | 150.0', '21.0 | 150.0', '47.6', 'nan | 40.0 | 6.0', '25.0 | 187.0', '40.0 | 30.0', '20.0 | nan | nan', 'nan | 70.0 | 350.0', '31.0 | nan | 1.0', 'nan | 20.0 | 10.0', '25.0 | 4.0', 'nan | 1470.0', 'nan | 25.0 | 10.0', '40.0 | 140.0', '1.0 | 30.0 | nan | nan | nan', '28.0 | nan', '11.0', 'nan | 480.0 | 480.0', '56.0 | 250.0', 'nan | 801.0', '50.0 | 2.3', '30.0 | 200.0', 'nan | 53.0', 'nan | 15.0 | 25.0', '5.0 | 150.0', '10.0 | 350.0', '10.0 | 5.0', '60.0 | 80.0', 'nan | 1800.0', '80.0 | 280.0 | 1.0', 'nan | 60.0 | 1000.0', '70.0 | 170.0', '80.0 | 450.0 | 1.0', '30.0 | 1050.0', '30.0 | 8.0', '300.0 | 340.0', 'nan | 380.0', '50.0 | 400.0', '100.0 | 8.3', '11.0 | nan', 'nan | 1.0 | 1.0', '10.0 | 270.0', '23.0 | 20.0', 'nan | 700.0 | 5.0', 'nan | 460.0', '27.0 | 80.0 | 27.0 | 80.0 | 27.0 | 80.0 | 27.0', 'nan | 23.0 | 7.0', 'nan | 125.0 | nan', '11.0 | 120.0', '25.0 | 200.0', '166.9', '3.0 | 40.0 | 6.0', '30.0 | 40.0', 'nan | 13.0', 'nan | 1.0', '30.0 | 2.0', '50.0 | 350.0 | 2.5', '300.0 | 4.0', '8.0', 'nan | 8.0', 'nan | 3.0', 'nan | 360.0', '70.0 | 400.0', '30.0 | 9.0', '20.0 | 10.0', '40.0 | 365.0', '4.0 | 350.0', '60.0 | 140.0', '22.5 | 10.0', '200.0 | nan', '30.0 | 8.0 | 1.0', 'nan | nan | nan | 15.0', '90.0 | 45.0', '70.0 | 500.0 | 2000.0', '100.0 | 6.0', '30.0 | 250.0', 'nan | 548.0', '10.0 | 20.0', '50.0 | 156.0', '30.0 | 450.0 | 500.0', '1.3 | 50.0 | nan', '65.0', 'nan | 45.0', '40.0 | 600.0', '11.2 | 3.0', 'nan | 60.0', '43.9', '55.0 | 8.0', '40.0 | 380.0', '5.0 | nan | 1.0', '5.0 | 20.0', 'nan | 25000.0', '40.0 | 400.0 | 6.0', '31.0 | 150.0', '244.0', '120.0 | 7.0', '47.0', 'nan | 7.0', '50.0 | 1000.0 | 1200.0', '24.0 | 500.0 | nan', 'nan | 125.0', 'nan | 12.0', '28.0 | nan | nan', '1.0 | nan', '5.0 | 45.0', 'nan | 500.0 | nan | 2000.0', '30.0 | 500.0', '200.0 | 20.0', 'nan | 1000.0 | 1012.0', 'nan | 2000.0 | 1000.0 | nan', '97.0 | nan', 'nan | 800.0 | 500.0', '100.0 | 1000.0 | 500.0', '50.0 | 80.0', '49.0', '40.0 | 468.0', '31.2', 'nan | 610.0', '60.0 | 2.5', '30.0 | 910.0', '8.0 | 50.0', '20.0 | 20.0 | nan', 'nan | 450.0 | 400.0', 'nan | 52.0', '30.0 | 1.0 | 8.0', '50.0 | 80.0 | nan', '10.0 | 17.0', '67.0', '30.0 | 7.0', '45.0 | 25.0', 'nan | 430.0', '1.0 | 15.0 | 10.0', '24.6', '60.0 | 5.0', 'nan | 23.0 | 8.0', '13.3 | 40.0 | 6.0', 'nan | 118.0', '380.0', '50.0 | 1.6', '700.0', '50.0 | 300.0 | nan | nan', '300.0 | 1.0', '40.0 | 3.0 | 1.0', 'nan | nan | 1.2', 'nan | 400.0 | 900.0', '1.0 | 30.0 | 8.0', '60.0 | 100.0', '37.0', '110.0 | nan', '7.5 | 20.0', 'nan | 150.0 | nan', '45.0 | 1.0', '5.0', '900.0', '20.0 | 3.0', '50.0 | 440.0', '8.9 | 40.0 | 6.0', '500.0 | 3000.0 | 10000.0', '15.0 | 300.0', 'nan | 240.0', '10.0 | 48.0', '45.0 | 125.0', '20.0 | 7.0', '80.0 | 720.0 | 1.0', '30.0 | 100.0', '840.0', '8.3 | 40.0 | 6.0', 'nan | nan | 4.0', '15.0 | 40.0', 'nan | 3.5', '20.0 | 250.0', '600.0', '12000.0 | nan', '40.0 | 174.0', '10.0 | 130.0', '6.0 | nan', '40.0 | 100.0', '105.0', '100.0 | 500.0', '69.0', '17.0 | 80.0', 'nan | 300.0 | 150.0', '68.5', '10000.0 | nan', '40.0 | 7.0', '70.0 | 40.0', 'nan | nan | 3.0', '110.0 | 700.0', 'nan | 30.0 | 8.0', 'nan | 1250.0', '30.0 | 342.0', '4.0 | 80.0', '45.0 | 8.0', '0.5 | 25.0 | 6.0', '70.0 | 500.0', 'nan | 8300.0', '0.84 | 3.0', '10.0 | 1.0', 'nan | 5000.0', '30.0 | 386.0', '1.0 | 10.0 | 6.0', '300.0 | 3.0', '60.0 | 13.0', '64.0 | 200.0', 'nan | 300.0 | 400.0', '100.0 | 800.0', '50.0 | 3.8', '45.0 | 2.0', '1.5 | nan | nan', '50.0 | 400.0 | nan', '117.0', '25.0 | 3.0', '36.0 | 20.0 | 8.0', 'nan | 1000.0 | 4000.0', 'nan | 25.0 | 8.0', '52.0 | 270.0', 'nan | 50.0', 'nan | 5800.0', '30.0 | 170.0', '60.0 | 270.0', '60.0 | 50.0 | 70.0', '300.0 | nan', 'nan | 480.0 | 600.0', '30.0 | 175.0', 'nan | 451.0', '30.0 | 310.0', '2.5 | 10.0', 'nan | 390.0', '35.0 | 2.0', 'nan | 1000.0 | 1500.0', '189.0', '55.0 | 365.0', '80.0 | nan', '6.0', '100.0 | 20.0', '20.0 | 100.0', '80.0 | 190.0', '40.0 | 1000.0 | 1000.0', '61.0', '20.0 | 80.0', '80.0 | 500.0', '55.0 | 0.5', '1.0 | 340.0', '50.0 | 280.0', 'nan | 20.0 | 8.0', '15.0', '52.2', 'nan | 1000.0 | 3000.0', '100.0 | nan | 10.0', '430.0', '41.0 | 10.0', '70.0 | 80.0', '1.0 | 30.0 | nan | nan', '80.0 | 1.0', '50.0 | 700.0 | 450.0', 'nan | 800.0 | 1500.0', '60.0 | 280.0', '80.0 | 200.0', '20.0 | 8.0', '100.0 | 365.0', 'nan | 1200.0', '27.0 | nan', '1000.0', '25.0 | nan', '16.0 | 150.0', 'nan | 500.0 | nan | 1200.0', 'nan | 238.0', '15.0 | 15.0', 'nan | 650.0', 'nan | 40.0', '40.0 | 1.0', '80.0 | 800.0', 'nan | 850.0', '8.0 | 40.0', '30.0 | 150.0 | nan', '390.0', '70.0 | 350.0', '10.0', '80.0 | 3.0', '42.0', '30.0 | 45.0', '30.0 | nan', '20.0 | 1.0', '65.0 | 250.0', '52.0 | 10.0', '40.0 | 2.0', 'nan | 1000.0 | 1007.0', '15.0 | 260.0', 'nan | 2.8', '51.0 | 270.0', '5.0 | 10.0', '25.0 | 0.5', '30.0 | 201.0', 'nan | 40.0 | 8.0', '5.0 | 12.0 | 20.0', '265.0 | nan', '90.0 | 350.0', 'nan | 500.0 | nan | 3000.0', 'nan | 100.0 | 400.0', '5.0 | 20.0 | 7.0', '8.0 | 340.0', 'nan | nan | 4.8', '42.0 | 10.0', '3.0 | 200.0', 'nan | 5500.0', 'nan | 450.0 | 1200.0', '30.0', '35.0 | 30.0', '20.0 | 12.0', '88.0 | nan', '70.0 | 150.0', '745.0', '20.0 | 18.0', 'nan | 480.0', 'nan | nan | 2.0', '3.2 | 40.0 | 6.0', '10.0 | nan | nan', '30.0 | 200.0 | 1.9', '152.0', '26.0 | nan', '10.0 | 15.0', '10.0 | 400.0', '60.0 | 20.0', 'nan | 1.2', 'nan | 20.0 | 3.0', '60.0 | 190.0', 'nan | 285.0', '30.0 | 265.0', '50.0 | 430.0', '80.0 | 550.0 | 1.0', '62.0 | nan', '60.0 | 260.0', 'nan | 2000.0 | 1100.0', '50.0 | 190.0', '140.0', 'nan | 1500.0 | 2000.0', '70.0 | 100.0', '20.0 | 150.0', 'nan | 3900.0', '6.0 | nan | nan', '60.0 | 40.0', 'nan | 618.0', 'nan | 400.0', '80.0 | 175.0', '40.0 | 30.0 | 30.0', '50.0 | 13.6', '24.0 | nan | nan', '15.0 | 10.0', '89.0', 'nan | 580.0', '100.0 | 800.0 | 1000.0', '220.0', 'nan | 210.0 | nan | nan', '20.0 | 40.0', 'nan | nan | 450.0', '35.0 | 0.1', '40.0 | 1900.0', '47.0 | 34.0', '30.6', '30.0 | 2000.0', 'nan | 160.0', '55.0 | 250.0', '40.0 | 3.0 | 1.0 | 1.0', '40.0 | 500.0', '17.0 | 200.0', '70.0 | nan | 150.0', '60.0 | 200.0 | nan', '100.0 | 400.0', '30.0 | 0.0', 'nan | 500.0 | 3000.0 | 10000.0', 'nan | 40.0 | 9.0', '24.0 | 8.0', '60.0 | nan | 5.0', '15.0 | 365.0', '95.0', 'nan | nan | 121.0', '60.0 | 350.0', '42.4', 'nan | 400.0 | 10.0', '5.6 | 40.0 | 6.0', '50.0 | 40.0', '20.0 | 7.5', '130.0 | nan', '100.0 | 1000.0 | 1000.0', '193.0', '50.0 | 8.0', '73.0', '57.0 | 250.0', '1300.0', 'nan | nan | 258.0', '2000.0', '40.0 | 180.0 | nan', 'nan | 27.0 | 1.0', '5.0 | 7.0', '10000.0 | 5.0', '17.0 | nan', '55.0 | 300.0', '175.0', '60.0 | 70.0', 'nan | nan | 20.0', 'nan | nan | 40.0 | 8.0', '55.0 | 10.0', '50.0 | 800.0 | 1200.0', '30.0 | 216.0', '112.0', '17.0 | 230.0', '2.0 | 20.0 | 8.0', '30.0 | 308.0', '33.0 | nan', 'nan | 500.0 | 1000.0', '101.0', '3.0 | nan | nan', 'nan | 330.0', '10.0 | 10.0 | 10.0', '50.0 | 30.0 | 8.0', '30.0 | 7.5', '32.0 | nan | nan', 'nan | 224.0', '176.0', '341.0', '60.0 | 50.0 | 150.0', '34.0 | nan | 1.0', 'nan | 480.0 | nan', 'nan | 80.0', 'nan | 540.0', '650.0', '60.0 | 50.0 | 30.0', '220.0 | nan', '60.0 | 8.0', '60.0 | 360.0', '60.0 | 200.0', '10.0 | 25.0', '30.0 | 1730.0', 'nan | 280.0', '24.0 | 30.0', 'nan | 110.0', '100.0 | 450.0', '56.0 | nan', '37.8 | 8.0', '20.0 | 2.0', '80.0 | 10.0', '80.0 | 30.0', 'nan | 500.0 | 1000.0 | 10000.0', '40.0 | 190.0', 'nan | 8000.0', '70.0 | 1.0', 'nan | 1000.0 | 1001.0', 'nan | 380.0 | nan', '150.0', '50.0 | 19360.0', 'nan | 70.0', '20.0 | 220.0', 'nan | 140.0', '30.0 | 910.0 | nan', 'nan | 1000.0', '30.0 | 30.0', '46.0 | nan', 'nan | 20.0', '43.8', '15.0 | nan', '56.0 | 200.0', '30.0 | 70.0', 'nan | 7.5', '39.4', '40.0 | 165.0 | nan', 'nan | 1050.0', '30.0 | nan | nan', '33.0', '51.0', '50.0 | 0.8', '1.0 | 5.0', '30.2', '16.0 | nan', 'nan | 600.0 | 2500.0', '60.0 | 50.0', '50.0 | 300.0', '100.0 | 150.0', 'nan | 500.0 | 1300.0 | 70.0', '27.0 | 20.0 | 8.0', '100.0 | 390.0', 'nan | nan | 1.0', '54.0 | 250.0', '2.0 | 340.0', '5.0 | 1.0', 'nan | 585.0', '45.0 | 3.0', '25.0', '1.0 | 20.0', '72.0 | nan', '80.0 | 1000.0', '2.0 | 40.0', '40.0 | 350.0', '100.0 | 700.0 | 1500.0', '60.0 | 6.0', '25.0 | 6.0', '15.0 | 175.0', '35.0 | nan', '100.0 | 220.0', '30.0 | 1000.0 | 2000.0', '100.0 | 844.0', '30.0 | 100.0 | 100.0', '10.0 | nan', '40.0 | 200.0', '35.0 | 365.0', '3.0 | nan', '45.0 | 30.0', 'nan | nan | 75.0', '115.0 | nan', '40.0 | 6.0', '40.0 | 40.0 | 5.0', '45.0 | 55.0', 'nan | 480.0 | 500.0', '30.0 | 400.0', 'nan | 550.0', '60.0 | 60.0', '40.0 | 230.0', '100.0 | nan | 5.0', 'nan | 160.0 | nan', '125.0', 'nan | 30.0', '250.0 | 100.0', '550.0 | nan', '4.0 | 0.0', '50.0 | 170.0', '50.0 | 250.0 | 300.0', 'nan | 15.0', 'nan | 400.0 | 600.0', '20.0 | 240.0', '510.0', '24.0 | nan', 'nan | 79.0', '80.0 | 150.0', '20.0 | 200.0', 'nan | 275.0', 'nan | 10.0 | 3.0', 'unkown', '0.8 | nan', '4.0', '50.0 | 585.0', '7.0 | 20.0 | 8.0', '40.0 | 120.0', 'nan | 850.0 | nan', '740.0', '8.0 | nan', '30.0 | 330.0', '252.0', 'nan | 230.0', '31.7 | 6.0', '0.5', '54.0 | 10.0', '10.0 | 20.0 | 8.0', '88.0 | 200.0', '100.0 | 15.0 | 1.0', '995.0', '350.0', '20.0 | 600.0', '3.0', '76.2', 'nan | 300.0 | 1100.0', '20.0 | nan | 5.0', '4.0 | 20.0 | 8.0', '43.0 | nan', '1.0 | 10.0', 'nan | nan', '50.0 | 130.0', '100.0 | 8.4', '80.0 | 2.0', '20.0 | 160.0', '100.0 | 15.0 | 5.0', 'nan | 500.0 | 1500.0', 'nan | 500.0 | nan', '160.0', 'nan | 564.0', '100.0 | 8.0', '20.0 | 50.0', '18.0 | nan', '25.0 | 100.0', '40.0 | 20.0 | 8.0', '3.0 | 0.0', 'nan | 210.0', '25.0 | 35.0', '40.0 | 165.0', '15.0 | 45.0', 'nan | 900.0', '55.0', '10.0 | 210.0', '23.0 | nan', '60.0 | 1.0', '60.0 | 365.0', '17.0', '300.0 | 200.0', '5.0 | 60.0', '42.0 | nan', '70.0 | 350.0 | nan', '700.0 | 420.0', '17.0 | 5.0', '19.0', 'nan | 1000.0 | 12000.0', 'nan | 30.0 | 7.0', '79.0', '40.0 | 150.0 | 150.0', '18.0', '34.0 | 200.0', '130.0', '39.0', '30.0 | 1000.0', 'nan | 90.0', '80.0 | 5.0', '310.0', '10.0 | 100.0', 'nan | 300.0 | 6.0', 'nan | 10000.0', '70.0 | 180.0', '78.0', '100.0 | 0.6', '60.0 | 500.0', 'nan | 2200.0', '65.0 | 2.0', '30.0 | 50.0', '120.0 | 2.0', '235.0', 'nan | 1000.0 | 60.0 | 1000.0', '1050.0', '45.0 | 7.5', 'nan | 8.0 | 5.0', '25.0 | 400.0', '40.0 | 400.0', '20.0 | 35.0', '140.0 | 0.5', '45.0 | 80.0', '90.0 | 7.0', 'nan | 25.0 | 7.0', '80.0 | 170.0', '74.0 | 200.0', '10.0 | 370.0', '27.0 | 80.0 | 27.0 | 80.0 | 27.0', '30.0 | 6.0', '141.0 | nan', 'nan | 410.0', '65.0 | 300.0', '50.0 | 300.0 | nan', '525.0', '20.0 | nan | 10.0', '50.0 | 1000.0 | 1800.0', '370.0', '30.0 | 850.0', '56.1', '80.0 | 500.0 | 1400.0', '35.0 | 200.0', '24.0', 'nan | 480.0 | 300.0', '30.0 | 400.0 | 450.0', 'nan | 473.0', 'nan | 8600.0', '50.0 | 130.0 | 1.0', '200.0 | 200.0', 'nan | 375.0', '41.2', '23.0 | 8.0', '30.0 | 260.0', 'nan | 1500.0', 'nan | 75.0 | 1400.0', '50.0 | 0.3', '38.0 | nan', '0.5 | 7.5', '27.0 | 60.0 | 27.0 | 60.0 | 27.0 | 60.0 | 27.0', '60.0 | 20.0 | 7.0', '5.0 | 400.0', 'nan | 8.0 | 1.0', '50.0 | 500.0 | 500.0', 'nan | 700.0 | 1800.0', '100.0 | 350.0', '460.0', '30.0 | 5.0', '60.0 | 450.0', '35.0 | nan | 1.0', '45.0 | 7.0', 'nan | 230.0 | nan', '100.0 | 250.0', '100.0 | 200.0', 'nan | 222.0', '80.0', '31.0 | nan', '12.0 | nan', '45.0 | 300.0', '131.0 | 200.0', 'nan | 5.0', '46.0', '320.0', 'nan | 195.0', '50.0 | 830.0', '40.0 | 80.0', '1.0 | 30.0 | 6.0', '20.0 | 130.0', '25.0 | 120.0', '50.0 | 10.0', '39.0 | 206.0 | 186.0', '50.0 | 16.0', '25.0 | 5.0', '60.0 | 93.0', 'nan | nan | 213.0', 'nan | nan | 1500.0', '80.0 | 200.0 | 200.0', 'nan | 1400.0', '30.0 | 1.0', '8.0 | 23.0', 'nan | 250.0', '24.0 | 500.0', '100.0 | 10.0', '12.0 | 600.0 | 600.0', '70.0 | 90.0', '95.0 | 20.0 | 8.0', '50.0', 'nan | 405.0', '180.0', '45.0', '40.0 | 70.0', '22.0 | 80.0', '70.0 | 800.0', 'nan | 260.0', 'nan | 400.0 | 400.0', '40.0 | 6.0 | 2.0', 'nan | 800.0 | 2500.0', '123.0', '50.0 | 350.0', '60.0 | 10.0', '120.0 | nan', '40.0 | 2000.0 | nan', '60.0 | 150.0', '5.0 | 35.0', '41.0', '130.0 | 40.0', '0.5 | 50.0', '60.0 | 900.0', '114.0', '5.0 | nan', 'nan | 2.5', 'nan | 1000.0 | 1006.0', '1.0 | 7.5', 'nan | 175.0', '1.0', '4.0 | 340.0', '54.0', '50.0 | nan | nan', '80.0 | 22.0', '32.0 | 200.0', '38.0 | 20.0 | 8.0', '17.0 | 110.0', '630.0', '50.0 | 130.0 | 100.0', '20.0 | 5.0 | 10.0', '1.1 | nan', '50.0 | 450.0', '145.0 | nan', '25.0 | 8.0', '80.0 | 400.0', '50.0 | 50.0', '35.0 | 8.0', '30.0 | 500.0 | 2000.0', '85.0 | 30.0 | 7.0', 'nan | nan | 8.0', '80.0 | 600.0', '21.0', '60.0 | 560.0', 'nan | 5501.0', '1000.0 | 340.0', '50.0 | 1400.0 | 500.0', '420.0', '45.0 | 200.0', '50.0 | 640.0', '70.0 | 100.0 | nan', '85.0', '20.0 | 20.0 | 8.0', '60.0 | 250.0', '53.0 | 250.0', '25.0 | 500.0', 'nan | 150.0', '70.0 | 600.0', '50.0 | 700.0 | 700.0', 'nan | 370.0', '50.0 | 700.0 | 500.0', 'nan | 36.0 | 1.0', '5.0 | 40.0', '60.0 | 750.0', '620.0', '10.0 | 200.0', '184.2', '22.0 | nan', '13.0', '50.0 | 15.0', '13.0 | 80.0', '20.0 | 10.0 | 0.5', 'nan | 1000.0 | 1000.0', 'nan | nan | nan', 'nan | 250.0 | nan', '20.0', '30.0 | 300.0', '31.0 | 20.0 | 8.0', '1400.0', '65.0 | 0.5', 'nan | 717.0', '25.0 | 500.0 | 1000.0', '2.3', 'nan | 4800.0', '90.0 | 440.0', 'nan | 790.0', '1100.0', '30.0 | 1650.0', '25.0 | 7.0', '92.0', '60.0 | 700.0', 'nan | 600.0 | 2000.0', '30.0 | 120.0', '30.0 | 80.0', '20.0 | 300.0', 'nan | 90.0 | nan', '55.0 | 650.0', '0.0 | 340.0', '250.0', '35.0 | 300.0', '19.0 | nan | nan', '10.0 | 10.0', '60.0 | 20.0 | 8.0', 'nan | 1.5', '60.0 | 600.0', '60.0 | 550.0', '37.0 | 1.0 | 1.0', '40.0 | 55.0', '3.0 | 57.0', '60.0 | 28.0', '33.7 | 67.5', '57.0 | nan', '50.0 | 7.5', 'nan | nan | 5.0', '100.0 | 0.5', '6.0 | 2.0', '27.0 | 70.0 | 27.0 | 70.0 | 27.0 | 70.0 | 27.0', '60.0 | 300.0', '50.0 | 270.0', '60.0 | 400.0', '610.0', '44.2 | 8.0', 'nan | 1000.0 | 1009.0', 'nan | nan | 1.3', 'nan | 0.5', '700.0 | 1970.0', '55.0 | 2.0', 'nan | 35.0', '70.0 | 5.0', '35.0', '60.0 | 125.0', '100.0 | 5.0', '30.0 | 10.0', '8.0 | 80.0', '40.0 | 3.0 | 3.0 | 1.0', '3.0 | 20.0 | 8.0', 'nan | 2.0', '50.0 | 60.0', '2.5', '60.0 | nan', '50.0 | 27.0', '21.0 | nan', 'nan | 350.0 | 150.0', '80.0 | 350.0', '20.0 | 20.0', '60.0 | 4.0', '25.0 | 10.0', '23.0', '35.0 | 10.0 | 1.0', '10.0 | 9.0', '30.0 | 230.0', '65.0 | 350.0', '100.0 | 35.0', '30.0 | 650.0', '550.0 | 2500.0', '54.0 | nan', 'nan | nan | 50.0', '40.0', '126.0 | nan', '230.0', '51.0 | 250.0', '60.0 | 420.0', '3.5', '28.0', '28.0 | 8.0', 'nan | 45.0 | nan', '50.0 | nan', 'nan | 300.0', '30.0 | 600.0 | 2500.0', 'nan | 400.0 | 500.0', '780.0', '48.8', 'nan | 2000.0', '0.8', '60.0 | 1300.0', 'nan | 1000.0 | 1003.0', '74.4', '60.0 | 7.0', '100.0 | 30.0', 'nan | 700.0 | 700.0', '27.0 | 80.0 | 27.0', '122.0', '300.0 | 2.0', '55.0 | 30.0', '181.0 | nan', '2.0', '2.0 | 0.0', '48.0 | 150.0', '130.0 | 10.0', '52.0 | 200.0', '6.0 | 10.0', '80.0 | 2100.0', '30.0 | 340.0', '50.0 | nan | 1200.0', '400.0 | 17.0', '60.0 | 105.0', 'nan | nan | 10.0', 'nan | 600.0 | 3000.0', '10.0 | 40.0 | nan', 'nan | 20.0 | 6.0', '44.0 | 200.0', '60.0 | 160.0', '0.5 | nan', '40.0 | 40.0', '80.0 | 1800.0', '64.7', '50.0 | nan | nan | 1200.0', '68.0', '30.0 | 172.0', '30.0 | 5.5', '50.0 | 250.0 | nan', '200.0', '20.0 | 370.0', '150.0 | nan', '80.0 | 8.0', '70.0 | 2.0', '30.0 | 264.0', '1.6 | 40.0 | 6.0', '100.0 | 40.0', 'nan | 500.0 | 2000.0', '30.0 | 258.0', '10.0 | 30.0', '80.0 | 2300.0', '40.0 | 450.0', '90.0 | 100.0', '15.0 | 20.0 | 8.0', '930.0', 'nan | 500.0 | 750.0', '25.0 | 20.0 | 5.0', '60.0 | 1000.0 | 1000.0', '20.0 | 180.0', '38.0', 'nan | 20.0 | 7.0', '40.0 | 350.0 | 1000.0', 'nan | 51.0', '270.0', '62.0', '170.0', 'nan | 12000.0', '99.4', '76.0 | 200.0', 'nan | 520.0', '130.0 | 20.0', '50.0 | 400.0 | 450.0', '10.1 | 3.0', '40.0 | 118.0', 'nan | nan | 167.0', '20.0 | 450.0', '45.0 | 0.5', 'nan | 20.0 | 7.5', '100.0 | 413.0', 'nan | 54.0', '8.0 | 100.0', '300.0', '190.0 | 40.0', 'nan | 500.0 | 600.0', '200.0 | 400.0', '30.0 | nan | 150.0', '450.0', '88.0', '60.0 | 9.0', '31.0 | 200.0', 'nan | 350.0', 'nan | 1700.0', '150.0 | 10.0', '5.8 | 40.0 | 6.0', '54.0 | 150.0', '33.0 | 1.0', '27.0', '66.0', 'nan | 4.0', '30.0 | 3.0', '10.0 | 35.0', '1200.0', '48.0 | 8.0', '17.5', 'nan | 2300.0', '1.7 | 40.0 | 6.0', '60.0 | 1000.0', '50.0 | 700.0 | 250.0', '60.0 | nan | nan', '2.0 | 25.0 | 6.0', '20.0 | nan | 1.0', '150.0 | 860.0', '800.0', '50.0 | 630.0', 'nan | 190.0', '48.0 | nan | 1.0', '55.0 | 0.5 | 30.0 | 0.5 | 0.5', '32.0', '1000.0 | 60.0 | 150.0', '227.5', '40.0 | 20.0', '20.0 | 150.0 | 1.0', '149.0', '40.0 | 390.0', '52.0 | 250.0', 'nan | 500.0 | 1400.0', '95.0 | nan', '100.0 | 3.3', '30.0 | 180.0', '120.0 | 365.0', '40.0 | 15.0', '20.0 | 60.0', '383.0', '25.0 | 187.0 | nan', 'nan | 500.0', 'nan | 38.0 | 1.0', '60.0 | 50.0 | 15.0', '11.0 | 150.0', '45.0 | 10.0', '96.0 | nan', '100.0 | 370.0', '40.0 | 20.0 | nan | 90.0', '40.0 | 400.0 | nan', '60.0 | 50.0 | 75.0', 'nan | 200.0 | nan', '40.0 | 175.0', '40.0 | 14.0', '70.0 | 50.0', '2.0 | 4.0 | 10.0', '60.0 | 2.0', '50.0 | 400.0 | 1500.0', '65.0 | 150.0', '30.0 | 500.0 | 1000.0', '25.0 | 300.0', '50.0 | 210.0', '50.0 | 120.0', '50.0 | 6.0', '41.7 | 8.0', '50.0 | 5.0', '80.0 | 40.0', '80.0 | 225.0', '400.0 | 200.0', '16.0', '80.0 | 250.0', '100.0 | 1.0', '10000.0', '30.0 | 450.0'])))

    additives_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    List of the dopants and additives that are in each layer of the ETL-stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous fields.
- If several dopants/additives, e.g. A and B, are present in one layer, list the dopants/additives in alphabetic order and separate them with semicolons, as in (A; B)
- If no dopants/additives, state that as “Undoped”
- If the doping situation is unknown, stat that as‘Unknown’
Example
Undoped | Li-TFSI
TiCl4
Nb
Undoped | Undoped | Undoped
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Undoped | Li-FTSI', 'PFO | Unknown', 'Er; Yb', 'Unknown | B; F; PEI', 'Unknown | Zr', 'Al(NO3)3\xa0· 9H2O', 'PMo12 | Unknown | Unknown', 'Unknown | TiCl4; Mg', 'nan | Al', 'nan | Au-np', 'N-DPBI', 'Unknown | Li-TFSI; TiCl4', 'Unknown | Au-np; Li-TFSI', 'Bi2O2S-np', 'PEI', 'Unknown | CsBr', 'Unknown | Au@Ag-np; Li-TFSI | Unknown', 'N-Graphene', 'Unknown | I', '1-butyl-3-methylimidazolium tetrafluoroborate', 'Unknown | NaCo2O4', 'H2PtCl6', 'Polystyrene', 'CH3NH3I', 'CsAc', 'UV', 'Unknown | CsI', 'Unknown | N', 'Unknown | Oleic acid', 'Unknown | Ruthenium', 'N-Graphene-nanosheets | Unknown', 'Unknown | Ag@SiO2 | Unknown', 'Unknown | NaYF4:Yb3:Er:@SiO2-np | Unknown', 'Unknown | Eu', 'Unknown | N; Ta', 'PMMA | Unknown', 'Y', 'Unknown | BF4', 'Undoped | Undoped', 'Galliumnitrate\xa0hydrate', 'PFNOX', 'bis-C60', 'Ta', 'Eu(NO3)3·6H2O', 'Acetylacetone', 'Ni', 'Al(NO3)3\xa0· 9H2O; La(NO3)3\xa0· 6H2O', 'Ga', 'Pyridine | Undoped', 'Ag', 'Unknown | NaYF4:Yb:Er-np', 'Mg', 'Unknown | Zn', 'Cs', 'Unknown | FeN3O9', 'Unknown | Ag@SiO2', 'ITIC | Undoped', 'InCl3', 'Unknown | TiO2-nw', 'In2O3', 'Unknown | Zn0.5Cd0.5S-np', 'DBU', 'Undoped | Mg | Undoped | Undoped', 'Unknown | TiCl4; Ethyl cellulose', 'D35', 'rGraphene oxide', 'Unknown | Ag-np', 'nan | MAI', 'nan | FK209', 'Urea', 'nan | Thiourea', 'EDTA', 'Tetraisopropil titanate butanol | TiCl4', '(RuCp*mes)2 | Undoped', 'Unknown | N-DBPI', 'TiCl4 | Nb', 'Nb2O5', 'Unknown | SDBAC', 'Ethanolamine', 'CoSe', 'Unknown | ZnGa2O4:Eu(III)', '1H molecule | Undoped', 'Ti', 'nan | Au-nw', 'CF4', 'AlCl3', 'Hydrogen | Undoped', 'Li-TFSI; Mg(TFSI)2', 'Unknown | Pluronic P-123', 'Al', 'Unknown | TiCl4 | Unknown', 'Ga2O3', 'Sb | Unknown', 'NaCl | Unknown', 'Al | Undoped', 'Al | nan', 'Unknown | Mg', 'Unknown | Er', 'DMOPA | Unknown', 'Dopamine', 'Undoped | P', 'Glycine', 'Unknown | Au@SiO2-np', 'Er', 'Phosphorene nanocomposites', 'ethanolamine', 'Unknown | NaYF4:Yb3:E3', 'oTb', 'Unknown | Fe', 'oTb | Unknown', 'Unknown | Graphene oxide', 'Yttrium', 'Graphene oxide | Unknown', 'F', 'Unknown | Zn0.25Cd0.75S-np', 'Unknown | EA | Unknown', 'Unknown | Au@TiO2 NPs', 'Unknown | Unknown | Cu', 'Undoped | p-Toluenesulfonic acid', 'SnCl2', 'Unknown | Nb', 'TMAH', 'tert-butanol', 'BF4', 'CF3NaO2S', 'Unknown | ZnCdS-np', 'Nb | TiCl4', 'Unknown | TPFPB; LiClO4', 'Unknown | TPFPB', 'N:Graphene-oxide', 'Unknown | Terpineol', 'Unknown | Unknown | Li-TFSI', 'Cs2CO3', 'Unknown | Au@Ag-np; Li-TFSI', 'Graphene oxide | Graphene oxide', 'Li-TFSI; Mg-TFSI | Li-TFSI', 'Li', 'La(NO3)3\xa0· 6H2O', '2,2,2-trifluoroethanol', 'Urea | Unknown', 'RGraphene oxide | RGraphene oxide', 'Unknown | PEG', 'Graphene; TiCl4 | Graphene', 'NbCl5', 'Cl', 'DPM2; OC10H21', 'Triton X-100', 'Li-TFSI; TiCl4', 'TiCl4 | TiCl4', 'TiCl4 | Decamethylcobaltocene', 'Unknown | SiW12', 'AgInS2-QD', 'Unknown | Li-TFSI | Unknown', 'Graphdiyne', 'Oleamide', 'Ti(acac)2', 'nan | Li-TFSI', 'F8BT | Unknown', 'Phen-I', 'Unknown | Graphene oxide | Unknown', 'Unknown | Mg; Er', 'Unknown | Ag-nw', 'CNT', 'Unknown | Li-TFSI', 'Bi | nan', 'TBABF4 | Unknown', 'Y2O3', 'Undoped | Undoped | Cu', 'Unknown | rGraphene oxide', 'nan | Ga', 'Yb', 'F | F', 'Unknown | H20', 'Chlorine', 'Ta | Undoped', 'Unknown | H', 'Unknown | SnCl2', 'DMOAP | Unknown', 'Unknown | Li; Er; Yb', 'TiCl4 | Ag-np', 'Ionic liquid', 'Unknown | NbCl5', 'nan | AuAg-np', 'nan | TiCl4', 'Unknown | Ge-np', 'Li-TFSI', 'Unknown | Al | Unknown', 'Unknown | BaTiO3', 'Fe | nan', 'EA', 'Unknown | N-DPBI', 'HCl', 'TAA', 'Unknown | Au@SiO2', 'Graphene-nanosheets | Unknown', 'Unknown | Au@Pt@Au-np | Unknown', 'TiCl4 | Undoped | Undoped', 'Er; Mg', 'DIO', 'Li-TFSI | nan', 'NaCl', 'Unknown | Triethylamine; HCl | Unknown', 'LiCl2; PEG', 'Cd; Y', 'Unknown | CeO2', 'SWCNTs | SWCNTs', 'Pluronic P123 surfactant', 'TiCl4 | Na-TFSI', 'DMBI', 'CoCl2', 'Unknown | Co-TFSI', 'Ga(acac)3 | Ga(acac)3', 'DPM; OE', 'Unknown | Triton 100-X', 'KCl', 'KOH', 'Potassium O-hexyl xanthate', '2-CP', 'Ru | Unknown', 'SnOCl2', 'Al(NO3)3; Ethanolamine', 'DMOAP | Undoped', 'Unknown | SrO', 'Li2CO3', 'Ru', 'BaSnO3', 'Fe', 'Compound 2 | Unknown', 'Zn', 'Zr | N', 'Unknown | Unknown | Unknown | MoO3', 'Unknown | HI', 'Unknown | TiCl4 | Tm; Yb', 'Unknown | Al; In', 'Unknown | TiO4', 'Unknown | Li-TSFI', 'CTAB | Unknown', 'Unknown | Ta', 'Unknown | Ethylene cellulose; TiCl4', 'Phosphotungstic acid; WCl6', 'Undoped', 'PNDI-2T | Undoped', 'Unknown | Yb:Er', 'TiCl4 | Unknown', 'Mxene | Mxene', 'Nb; TiCl4', 'NACl', 'nan | CsBr', 'Unknown | Undoped', 'Ethyl cellulose | Unknown', 'Glucose', 'Unknown | Unknown | Nb', 'Unknown | LiFTSI', 'Sn | Unknown', 'SnCl2; MercaptoPropionic acid; Urea', 'Al | Unknown', 'HCl | Unknown', 'TOPD', 'Unknown | Li-TFSI; TiCl4 | Unknown', 'Tantalum(V) ethoxide', 'Acetic Acid', 'MAI | Unknown', 'IL-BF4', 'Undoped | AlCl', 'Unknown | La', 'B-TiO2', 'CNT, DMSO', 'RuCl3', 'Unknown | InCl3 | Unknown', 'rGraphene oxide | Unknown', 'Unknown | TiCl4; SWCNTs', 'Unknown | Graphene', 'Unknown | NaYF4:Yb:Tm-np', 'TiCl4', 'Undoped | Li-TFSI', 'Carbon-np', 'Unknown | Li', 'In | Unknown | Unknown', 'Unknown | Unknown | NaYbF4:Ho-np', 'nan | TAA', 'Mg | Unknown', 'Unknown', 'Samarium', 'Unknown | Au-np; TiCl4', 'TiCl4 | Undoped', 'nan | Nb', 'TiCl4 | nan', 'K', 'nan | Ag', 'Unknown | YCl3', 'Graphene', 'Unknown | SWCNTs', 'Undoped | W', 'Yb:Er', 'Unknown | Ag', 'Pyridine | Unknown', 'In', 'Unknown | Ho; Yb; Mg', 'N; PEI', 'Unknown | ZnGa2O4-np', 'Graphene | Graphene', 'Nb', 'H2O', 'PEIE', 'Unknown | 2,6-Py | Unknown', 'Unknown | Cl', 'Ga | Undoped', 'HMB', 'TBAPF6', 'Ethanolamine | Unknown', 'Unknown | Sb | Unknown', '4,4′-BiPy | nan', 'W2(hpp)4', 'N-Graphene | Unknown', 'DBU | Unknown', 'La', 'Unknown | F127', 'Unknown | Yb', 'Unknown | CTAB', 'N2H8S', 'Unknown | Zn0.75Cd0.25S-np', 'Unknown | TiCl4', 'TBAB', 'PF6', 'Unknown | SnOCl2', 'BIZ | Undoped', '2H molecule | Undoped', 'Al2O3', 'LiCl', 'MoCl5', 'PMMA', 'Li-TFSI; Mg(TFSI)2 | Li-TFSI', 'F; Sn', 'N', 'Gd', 'TiCl4 | Li-TFSI', 'Co', 'Nb(OCH2CH3)5', 'n-DMBI', 'Unknown | CeO2:Eu', 'TiCl4 | polystyrene', 'Unknown | B', 'V | Undoped', 'SnCl4', 'TBABF4', 'Unknown | Ba(OH)2', 'TiAcAc', 'Unknown | AlCl3:NH3', 'Graphene-QDs', 'Titanium acetylacetonate', 'Unknown | In', '2,6-Py | Unknown', 'Bphen', 'Unknown | Cs', 'Guanidinium chloride', 'N-DMBI | Unknown', 'Zr', 'Unknown | Ni | Unknown', 'Unknown | Carbon-np', 'ZnCl2', 'In | Unknown', 'Oleic Acid', 'C60-substituted catechol | C60-substituted catechol', 'TiCl4 | Unknown | Unknown', 'CdS', 'Undoped | TiCl4', 'Stearic acid; EDA', 'N2', 'Unknown | TAA', '2,2′-BiPy | nan', 'Undoped | Acetylacetone; Triton X-100', 'Cd', 'Carbon-QDs', 'W', 'Bi | Unknown', 'Sn', 'Unknown | F', 'Undoped | TiCl4 | TiCl4', 'Unknown | Sn', 'Unknown | PVC-g-POEM', 'Unknown | rGraphene oxide; Li-TFSI', 'Undoped | Undoped | Undoped', '2D graphene', 'Mxene | Mxene | Unknown', 'Graphdiyne | Graphdiyne', 'NH4Cl', 'Nb | Unknown', 'Na2S', 'AgNO3 | Undoped | Undoped', 'TiCl4 | K-TFSI', 'MAI | Unknown | Unknown', 'TaCl', 'Unknown | In | Unknown', 'Unknown | Polystyrene', 'TaCl5', 'Unknown | DMBI', 'Fe(NO3)3 | Unknown', 'Mg | Undoped | Undoped', 'Unknown | Al', 'PS', 'IZ | Undoped', 'CTAB', 'NH4F', 'Unknown | HCl', 'CeOx; TiCl4 | Unknown', 'TiAc2', 'Unknown | Sb', 'Unknown | Li; Mg | Unknown', 'Unknown | Li; Ho; Yb', 'Triethylamine | Unknown', 'Sb', 'NOBF4', 'Unknown | B; F', 'Nb | Undoped', 'Unknown | Li-acetat', 'P123 | Unknown', 'Unknown | Li-FTSI', 'Unknown | InCl3', 'Na2CO3', 'Ag; In | Undoped', 'K2CO3', 'nan | Cs2CO3', 'DMBI | Unknown', 'DPM2; OE2', 'Unknown | Unknown | Unknown | Co2O3', 'Unknown | PAAb-PEO', 'Unknown | Er; Yb', 'Ti3C2', 'titanium diisopropoxide bis(acetylacetonate)', 'Unknown | PEI', 'Unknown | NaYF4:Yb3:Er-np | Unknown', 'TiCl4 | Cs-TFSI', 'Unknown | Sr', 'Unknown | Unknown | TiCl4', 'CdCl2', 'Unknown | Unknown | Unknown | NiO', 'Ti(acac)2 | TiCl4', 'Undoped | nan', 'rGS', 'DMOAP', 'C60-substituted catechol', 'Unknown | Unknown | Unknown | CuO', 'ClGD | Unknown', 'Unknown | Li-TFSI; TBP', 'Mo', 'PFNOX; Polystyrene', 'Li2SiO3'])))

    additives_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the dopants/additives.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If more than one dopant/additive in the layer, e.g. A and B, separate the concentration for each dopant/additive with semicolons, as in (A; B)
- For each dopant/additive in the layer, state the concentration.
- The order of the dopants/additives must be the same as in the previous filed.
- For layers with no dopants/additives, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used.
- The preferred way to state the concentration of a dopant/additive is to refer to the amount in the final product, i.e. the material in the layer. When possible, use on the preferred units
o wt%, mol%, vol%, ppt, ppm, ppb
- When the concentration of the dopant/additive in the final product is unknown, but where the concentration of the dopant/additive in the solution is known, state that concentration instead. When possible, use on the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
5 vol%; nan | 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | 0.3 M
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '4 % | nan', 'nan | 0.04 | 0.04', '1', '2', 'nan | 0.5 mg/ml', '0.005', '5 vol%', '3 mol%', '50 vol%', 'nan | 12 %', 'nan | 0.5 %', '0.2', '0.1 mol%', '0.005 M', '1 mol%', '0 | 50 ppm', '100 vol%', '0.33', '0.01', '1.0 mol%', '0.025 M | nan', '0.003', '0.1', '0.66', '0.4 % | nan', '2.5 vol%', '0.05 % | nan', '0.07', '40 mM | nan', '10 mol%', '0.025 M', '0.15 M | 20 wt% | 0.5 mg/ml', '0.05', '2 mol%', 'nan | 7.5 % | nan | nan', '0.15', '0.2 mM/ml', '0.015', '0.3 mM', '2 % | nan', '1.2 mg/ml', '0.5 % | nan', '1.5 vol%', 'nan | 1 %', '3.0 mol%', '0.5', '75 vol%', '6 wt%', '0.5 %; 0.5 %', '0.4 mg/ml', '0.15 M | nan', 'nan | 0.04', '0 | 100 ppm', '0.355 vol%', '0.15 M | 20 wt% | 1 mg/ml', '0.02', '0.007', '0.2 wt%', '2.4 vol%', '0.6 M', '7.5 % | nan | nan', '50 mM', '6.25 wt% | nan', '0.1 mM/ml', '0.15 mM', '0.6 mM', 'nan | nan', '0.4 mM/ml', 'nan | 0.1 M', '0.04 M', '0.001', 'nan | 6 %', '2.0 mg/ml', '0.15 M | 20 wt% | 2 mg/ml', 'nan | 9 %', '0.1 wt%', 'nan | 3 %', '12.5 wt% | nan', '0.5 mol%', '3 % | nan', '0 | 10000 ppm', '20 mg/ml | 0.5 mg/ml', '0.01 M', '5 % | nan', '0.8 mg/ml', '2.5 wt% | nan', '5 mol%', '0.5 wt%', '1.6 mg/ml', ' undoped', '1.5 wt%', '1 % | nan', '100 mM | nan | nan', '1 wt%', '0.15 M | 20 wt% | 4 mg/ml', '0.45 mM', 'nan | 100 mg/ml', '0.3 mM/ml', '3 wt%', 'nan | nan | nan', 'nan | 5 %', '0.0025', '5 wt%', '25 vol%', '0 | 1000 ppm', '0.03'])))

    deposition_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    The deposition procedures for the ETL stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate them by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Thermal annealing is generally not considered as an individual reaction step. The philosophy behind this is that every deposition step has a thermal history, which is specified in a separate filed. In exceptional cases with thermal annealing procedures clearly disconnected from other procedures, state ‘Thermal annealing’ as a separate reaction step.
- Please read the instructions under “Perovskite. Deposition. Procedure” for descriptions and distinctions between common deposition procedures and how they should be labelled for consistency in the database.
Example
Spin-coating
Spin-coating | Spin-coating
Spray-pyrolys | Spin-coating
Evaporation | Evaporation
Spin-coating | Evaporation
CBD
Spray-pyrolys
Spin-coating | Evaporation | Evaporation
Spray-pyrolys >> CBD | Spin-coating >> CBD
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=sorted(['Spin-coating | Spin-coating >> Spin-coating', 'Spin-coating | Spin-coating >> CBD', 'ALD | ALD | Spin-coating', 'Spin-coating | Spin-coating', 'Magnetron sputtering | Spin-coating | Spin-coating', 'Sputtering', 'Electrospraying | Hydrothermal | Spin-coating', 'Evaporation | CVD', 'Evaporation | Co-evaporation', 'Spin-coating | Sputtering', 'Sputtering | Spray-pyrolys | Spin-coating', 'Electrodeposition | Spin-coating', 'DC Magnetron Sputtering | Electrochemical anodization', 'Brush painting | Evaporation', 'Spray-pyrolys | Screen printing | Spray-coating | Screen printing', 'Electrochemical anodization', 'Spin-coating >> Spin-coating >> Spin-coating', 'Photo-thermal CVD', 'Spray-pyrolys | Electrodeposition | Screen printing | Screen printing', 'Hydrothermal | Hydrothermal | Hydrothermal', 'Spin-coating | Spin-coating >> CBD >> Rinsing', 'ALD | Spray-pyrolys | Spin-coating', 'Spin-coating | Spin-coating >> Spin-coating | Spin-coating >> CBD', 'Comersial | Spin-coating', 'Spray-pyrolys | Screen printing | CBD', 'Spray-pyrolys | PVD', 'Spin-coating | Lithography', 'Sputtering >> CBD >> CBD', 'Evaporation | Evaporation | ALD | ALD', 'Spin-coating | Dipp-coating', 'Spin-coating | Spin-coating | Spin-coating', 'Spin-coating | Electrospraying', 'Aerosol-assisted CVD | Spin-coating', 'CBD | Screen printing | Dipp-coating', 'Dipp-coating | Doctor blading | Doctor blading', 'Spray-pyrolys | Spin-coating >> Hydrothermal >> Ion exchange >> CBD', 'Electrodeposition | Screen printing', 'E-beam evaporation | Spin-coating', 'PVD', 'Condensation | Spin-coating', 'Evaporation | ALD | ALD', 'Spray-coating | Spin-coating | Spin-coating', 'Inkjet printing | Evaporation', 'Temperature gradient solid-phase sintering', 'Unknown | Screen printing', 'Spray-pyrolys', 'Spin-coating >> CBD | Spin-coating', 'Spin-coating | ALD | Evaporation', 'Drop-infiltration', 'Solution combustion | Spin-coating', 'ALD | Spin-coating', 'CVD | CVD | Spin-coating', 'CVD | Doctor blading', 'Electrospraying | Electrospraying', 'Screen printing', 'Spin-coating | Ultrasonic spray', 'Meniscus coating | Evaporation', 'Spin-coating | E-beam evaporation', 'Spin-coating >> Hydrothermal', 'Evaporation | CVD >> ALD', 'Spray-pyrolys | Spin-coating >> CBD | Spin-coating', 'Doctor blading | Dipp-coating', 'CBD >> Screen printing >> CBD | Screen printing | Screen printing', 'Sputtering | Screen printing | Screen printing', 'Spin-coating | CBD | Spin-coating', 'Spin-coating | Lithography | Spin-coating', 'Spin-coating | Evaporation | Evaporation | Evaporation', 'Spray-coating | Solvothermal', 'Spin-coating >> Hydrothermal | Spin-coating', 'Spin-coating | Spin-coating | Evaporation', 'Sputtering | E-beam evaporation', 'Doctor blading | Evaporation', 'Spray-pyrolys | Screen printing | Spray-pyrolys | Screen printing', 'Dipp-coating | Spin-coating >> Plasma treatment', 'E-beam evaporation | E-beam evaporation', 'Dipp-coating | Spin-coating', 'CVD | Spin-coating', 'CBD | Spin-coating >> CBD', 'RF sputtering | Spin-coating', 'Spin-coating | PVD', 'Spray-pyrolys | Spray-coating', 'Spin-coating | ALD', 'Spin-coating | Doctor blading | Doctor blading', 'Spin-coating | Spin-coating | Spin-coating | Spin-coating | Spin-coating', 'RF Magnetron sputtering', 'Spray-pyrolys | Spin-coating >> Spin-coating', 'Thermal oxidation', 'Unknown | Screen printing | Screen printing', 'Spin-coating | Evaporation >> Electrohemical anodization >> Etching', 'Spin-coating | Hydrothermal | Doctor blading', 'Roller coating', 'Spray-pyrolys | Spin-coating | Spin-coating', 'Evaporation | Evaporation | CVD', 'Spray-pyrolys | Screen printing', 'Unknown | Unknown', 'Slot-die coating', 'Spin-coating >> CBD', 'Spin-coating | Evaporation', 'Ultrasonic spray', 'Dipp-coating | Spin-coating | Spin-coating', 'Spin-coating | Electrodeposition', 'Spin-coating | Spin-coating | Hydrothermal', 'Sputtering >> Spin-coating | Spin-coating', 'Sputtering >> Oxidation | Spin-coating', 'Ultrasonic spray pyrolysis', 'Spin-coating | Spin-coating | Spin-coating | Spin-coating', 'Spin-coating | Slot-die coating', 'Spin-coating | Hydrothermal', 'SILAR', 'Inkjet printing', 'Spin-coating | Spin-coating >> Dipp-coating', 'Condensation | Evaporation | Evaporation', 'Spin-coating | Spin-coating | SILAR', 'Evaporation | Evaporation', 'Spin-coating | RF sputtering', 'Spin-coating | Sputtering >> Hydrothermal >> ALD', 'Spin-coating | Doctor blading', 'ALD | Spin-coating | Hydrolysis', 'Spin-coating | Air brush spray', 'ALD | Hydrolysis', 'CBD | Spin-coating | Spin-coating', 'Hydrothermal | Spin-coating', 'Spin-coating | Spin-coating | Spin-coating | Spin-coating | Spin-coating | Spin-coating | Spin-coating', 'Spray-coating | Evaporation', 'Dipp-coating | Hydrothermal | CBD', 'Electrochemical anodization | CBD', 'Sputtering | Hydrothermal | ALD', 'Dipp-coating | Hydrothermal', 'Evaporation | Sputtering', 'Dipp-coating', 'Spin-coating | Hydrolysis', 'Spin-coating | SILAR', 'Oxidation | Dipp-coating', 'DC Magnetron Sputtering | Spin-coating', 'ALD | Flame aerosol | ALD', 'CBD | Hydrothermal', 'Spin-coating | Dipp-coating | Spin-coating', 'Spin-coating | Spin-coating | Evaporation | Evaporation | Evaporation', 'ALD | Evaporation', 'Meniscus-coating', 'Spin-coating | Solvothermal', 'Magnetron sputtering | Hydrothermal', 'ALD | Screen printing', 'Spray-pyrolys | Evaporation >> Electrochemical anodization', 'Spin-coating | Magnetron sputtering >> Oxdation', 'Hydrothermal >> Solvothermal', 'Electrochemical anodization | Electrochemical anodization', 'Magnetron sputtering', 'Substrate vibration assisted dropcasting', 'RF Magnetron Sputtering', 'Spin-coating | Hydrothermal >> Solvothermal etching', 'Chemical etching >> Thermal oxidation', 'Sputtering >> Electrochemical anodization | Spin-coating', 'Spin-coating | Electrospinning | Spin-coating', 'Spray-pyrolys | Doctor blading', 'Doctor blading | Spin-coating | Spin-coating', 'Spin-coating | Spin-coating | SILAR method', 'Evaporation | Evaporation | CVD | CVD', 'Spin-coating | Hydrothermal >> Spin-coating', 'E-beam evaporation | Spin-coating | Spin-coating', 'Spray-pyrolys | Spin-coating | Spray-pyrolys', 'RF sputtering | CBD', 'Reactive sputtering', 'E-beam evaporation >> CVD | ALD', 'Spray-pyrolys | Screen printnig', 'Spin-coating | Dropcasting', 'Spray-pyrolys | Evaporation', 'Spray-pyrolys >> CBD', 'DC Sputtering | Spin-coating', 'Spin-coating | Sputtering | Spin-coating', 'CBD | Spin-coating | Dipp-coating', 'Spin-coating | Spin-coating | RF sputtering', 'CVD >> ALD', 'Spray-pyrolys | Spin-coating >> CBD', 'Spray-pyrolys | Spin-coating | Screen printing', 'RF plasma sputtering | Spin-coating', 'Sputtering | Evaporation', 'Slot-die coating | Slot-die coating', 'Doctor blading | Hydrothermal', 'Spin-coating | Spin-coating | Spin-coating | Dipp-coating', 'Spin-coating >> litography', 'Spin-coating', 'ALD | Screen printing | Screen printing', 'Sputtering | Spin-coating | Spin-coating', 'Spray-coating | Spray-pyrolys | Spin-coating', 'Spray-coating', 'Electrospraying | Hydrothermal', 'Spin-coating | Spin-coating | Doctor blading', 'Spin-coating >> Spin-coating >> Spin-coating | Spin-coating', 'Screen printing | Screen printing | Screen printing | Screen printing', 'Co-evaporation | Evaporation', 'Screen printing | Screen printing', 'Spin-coating | Spin-coating >> CBD | Spin-coating', 'Spin-coating | Spin-coating | Screen printing', 'Spray-pyrolys | Blow-drying', 'Screen printing | Screen printing | Screen printing', 'Spray-coating | Spray-coating', 'Spin-coating >> Spin-coating | Spin-coating', 'Spin-coating | Screen printing >> CBD', 'Unknown | Hydrothermal', 'Hydrothermal | CBD', 'Spin-coating | Spin-coating | Dipp-coating', 'Solution combustion', 'CBD | Spin-coating', 'Spin-coating | Evaporation >> Anodisation >> Oxidation >> Etching', 'CBD | Hydrothermal | Dipp-coating', 'Electrodeposition | Electrodeposition', 'Spray-pyrolys | Screen printing | Screen printing | Dipp ccoating', 'Spray-pyrolys | Hydrothermal', 'Spray-pyrolys | Dropcasting', 'Spin-coating | Unknown', 'Spin-coating >> Evaporation | Evaporation', 'Spin-coating | Lamination', 'Spin-coating | E-beam evaporation >> Electrochemical anodization', 'Spray-pyrolys >> Hydrothermal | PVD-OAD', 'Sputtering | Dipp-coating', 'Spray-pyrolys | PVD | PVD | PVD', 'CBD >> Screen printing >> CBD | Screen printing', 'Dropcasting | Spin-coating | Spin-coating', 'Spin-coating | Spray-coating', 'Sputtering | Sputtering', 'Spin-coating | Evaporation | Spin-coating | CBD', 'Pulsed laser deposition | Spin-coating', 'Electrodeposition | CBD', 'Spray-coating | Spin-coating', 'Photo-thermal CVD >> Spin-coating', 'Oxygen plasma treatment', 'Spin-coating | Spin-coating | Unknown | Unknown', 'Spin-coating | Hydrothermal | LBLAR', 'Lamination', 'Dipp-coating | Spin-coating | Spin-coating | CBD', 'Evaporation | Evaporation | ALD', 'Evaporation | Evaporation | Evaporation', 'Spin-coating | Unknown | Unknown', 'Magnetron sputtering | Hydrothermal | ALD', 'Dipp-coating >> CBD', 'Spray-pyrolys | Screen printing | Spin-coating', 'Spray-pyrolys | ALD', 'Hydrothermal >> Dipp-coating', 'Evaporation', 'RF Magnetron sputtering | RF Magnetron sputtering', 'Spin-coating | Evaporation | Spin-coating', 'Unknown', 'Electrospraying', 'Spin-coating | RF Magnetron Sputtering', 'Sputering', 'CVD', 'Spin-coating | Spin-coating >> Hydrothermal', 'Slot-die coating | Spin-coating', 'Spin-coating >> Spin-coating >> CBD', 'Evaporation | Unknown', 'Spin-coating | Hydrothermal | Dipp-coating', 'Doctor blading', 'CBD | CBD', 'Spin-coating | Hydrothermal | Spin-coating | Spin-coating', 'Sputtering | Spin-coating', 'Spray-pyrolys >> Hydrothermal | Spin-coating', 'Spin-coating | Electrospraying | Spin-coating', 'Dipp-coating | Screen printing | Screen printing', 'Hydrothermal | Dipp-coating', 'Electrospinning', 'Spin-coating | Spin-coating | Spin-coating | ALD', 'Electrospraying | Hydrothermal | Hydorthermal', 'Spin-coating | CBD', 'Spray-pyrolys | Screen printing | Screen printing | Screen printing', 'DC Magnetron Sputtering', 'CVD | CVD | Evaporation', 'Electrodeposition | Spin-coating >> CBD', 'Hydrothermal', 'Electrodeposition', 'Spray-pyrolys | Spin-coating', 'Spin-coating | Hydrothermal | SILAR', 'Dipp-coating | Evaporation', 'DC Sputtering >> Electrochemical anodization', 'ALD | Spin-coating | Spin-coating', 'Electrospraying | Electrospraying | Spin-coating', 'Spin-coating | Electrodeposition | Spin-coating', 'Electrospraying | Spin-coating', 'CVD | CVD', 'CBD | Evaporation', 'Spin-coating >> Spin-coating', 'Spray-pyrolys | Electrospinning', 'Sputtering | CBD', 'Electrodeposition | Hydrothermal', 'Dipp-coating | Dipp-coating', 'CBD | Screen printing', 'Spin-coating | Hydrothermal | Sputtering', 'Spin-coating >> Solvent annealing', 'Spin-coating | Electrodeposition >> CBD', 'Evaporation | Spin-coating', 'Sputtering | Sputtering | Sputtering', 'Magnetron sputtering >> Electrochemical anodization | Electrodeposition', 'Spray-pyrolys | Spin-coating >> Dipp-coating', 'Hydrolysis', 'Hydrothermal | Spin-coating | CBD', 'Spray-pyrolys | ALD | Spin-coating', 'Screen printing | Spin-coating', 'Evaporation | ALD', 'Spin-coating >> Spin-coating >> Spin-coating | Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating', 'Spray-pyrolys | Spray-pyrolys', 'Spin-coating >> CBD >> Rinsing >> Rinsing | Spin-coating', 'RF sputtering', 'Sputtering | Pulsed laser deposition | Hydrothermal | Spin-coating', 'Spray-pyrolys | PVD | PVD', 'CBD | Screen printing | Screen printing', 'Spin-coating | Hydrothermal | CBD', 'Evaporation >> Oxidation | Spin-coating', 'Unknown | Unknown | Unknown', 'Spin-coating | Hydrothermal | Spin-coating', 'Spin-coating | Spin-coating >> Hydrothermal | Spin-coating', 'Spray-pyrolys | Electrospraying', 'Sputtering >> Electrochemical anodization', 'DC Reactive Magnetron Sputtering', 'Spin-coating | Spin-coating | Sputtering', 'Spin-coating | Spray-pyrolys', 'Frequency Magnetron Sputteirng', 'ALD | Microwave hydrothermal', 'Spin-coating | Screen printing', 'Magnetron sputtering | Spin-coating', 'Lamination | Spin-coating', 'CVD | Spray-pyrolys', 'Spray-pyrolys | Screen printnig | Screen printing', 'Spray-pyrolys | Screen printing | Screen printing', 'Spin-coating | Spin-coating | Evaporation | Evaporation | Evaporation | Evaporation', 'ALD | Magnetron sputtering | ALD | ALD', 'Spray-pyrolys | Screen printing | Evaporation | Screen printing', 'Solvothermal', 'Spray-pyrolys >> CBD | Spin-coating', 'E-beam evaporation | CVD', 'Spin-coating >> CBD | Evaporation', 'Doctor blading | Spin-coating | Dipp-coating', 'Slot-die coating | Evaporation', 'Spray-pyrolys | Spin-coating | Dipp-coating', 'CBD', 'Spin-coating | Electrospinning', 'Spray-pyrolys | CVD', 'E-beam evaporation >> CVD', 'CBD >> Rinsing >> Rinsing', 'Magnetron sputtering >> Electrochemical anodization', 'Unknown | Unknown | Spin-coating', 'ALD | Flame aerosol', 'Sputtering | Electrodeposition | Spin-coating', 'Sputtering | Hydrothermal', 'Spin-coating | Hydrothermal | Evaporation', 'E-beam evaporation', 'Langmuir-Blodgett deposition', 'Unknown | Spin-coating', 'CBD | Hydrothermal >> Etching', 'Spray-pyrolys | Inkjet-Printed', 'Spin-coating | Spin-coating | CBD', 'Dropcasting', 'Hydrothermal | Hydrothermal', 'Pulsed laser deposition', 'CBD | Inkjet printing | Inkjet printing', 'Spin-coating | Spin-coating | Evaporation | Evaporation', 'Spin-coating | Hydrothermal | Hydrothermal', 'Spray-pyrolys | Ultrasonic spray', 'Unknown | CVD', 'Spin-coating | Evaporation | Spin-coating | Spin-coating', 'Spin-coating | Spin-coating | ALD', 'Evaporation | Evaporation | Spin-coating | ALD | ALD', 'Spin-coating | Screen printing | Screen printing', 'Doctor blading | Doctor blading', 'Spin-coating | Evaporation | Evaporation', 'Spray-pyrolys >> Hydrothermal', 'Spin-coating | Transfer', 'Spin-coating | Hydrothermal | ALD', 'ALD']))))

    deposition_aggregation_state_of_reactants = Quantity(
        type=str,
        shape=[],
        description="""
    The physical state of the reactants
- The three basic categories are Solid/Liquid/Gas
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the aggregation state associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Most cases are clear cut, e.g. spin-coating involves species in solution and evaporation involves species in gas phase. For less clear-cut cases, consider where the reaction really is happening as in:
o For a spray-coating procedure, it is droplets of liquid that enters the substrate (thus a liquid phase reaction)
o For sputtering and thermal evaporation, it is species in gas phase that reaches the substrate (thus a gas phase reaction)
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Liquid
Gas | Liquid
Liquid | Liquid >> Liquid
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Liquid | Liquid', 'Liquid | Liquid >> Liquid | Liquid >> Liquid', 'Liquid >> Liquid >> Liquid >> Liquid | Liquid', 'Gas | Liquid | Liquid', 'Liquid | Liquid | Liquid | Liquid', 'Liquid >> Liquid >> Liquid | Liquid', 'Gas', 'Liquid | Liquid >> Liquid', 'Liquid >> Liquid >> Liquid', 'Gas | Gas', 'Liquid | Gas', 'Gas | Gas | Gas', 'Unknown', 'Liquid | Liquid | Liquid', 'Liquid >> Liquid', 'Gas | Gas >> Gas', 'Gas >> Gas', 'Liquid | Liquid | Gas', 'Liquid', 'Liquid | Gas | Gas', 'Liquid >> Liquid | Liquid', 'Liquid | Gas | Liquid', 'Liquid | Liquid >> Liquid >> Liquid', 'Unknown | Gas | Gas', 'Gas >> Liquid | Liquid', 'Solid | Solid', 'Gas >> Liquid >> Liquid', 'Gas | Liquid'])))

    deposition_synthesis_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The synthesis atmosphere
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Vacuum | N2
Air | Ar; H2O >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['N2 | Vacuum', 'N2', 'Air | Air >> Air', 'Air >> Air | Air', 'Air >> Air >> Air', 'Unknown', 'Air | N2', 'Vacuum | Vacuum | Vacuum', 'Vacuum >> Vacuum >> Unknown', 'N2 | N2 | N2', 'N2 | > N2', 'N2 | N2', 'Vacuum | Vacuum >> Vacuum', 'Ar | Ar', 'Ar', 'N2 | Air', 'Air | Air | Air', 'Air | Vacuum', 'N2 | N2 | Vacuum', 'Air | Air', 'Ar; O2 | Air', 'Vacuum | Air', 'Air | Vacuum | Vacuum', 'Air | Air >> Air | Air >> Air', 'Air | Air | Air | Air', 'Dry air', 'Vacuum | Vacuum', 'Air', 'Dry air | Dry air', 'Vacuum', 'Air | Ar', 'Ar; O2', 'Vacuum >> Vacuum', 'Dry air | Vacuum', 'Air >> Air', 'Air; O2 | Air', 'Vacuum | N2', 'N2 | Vacuum | Vacuum'])))

    deposition_synthesis_atmosphere_pressure_total = Quantity(
        type=str,
        shape=[],
        description="""
    The total pressure during each synthesis step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- Pressures can be stated in different units suited for different situations. Therefore, specify the unit. The preferred units are:
o atm, bar, mbar, mmHg, Pa, torr, psi
- If a pressure is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 100 pa and not 80-120 pa.
Example
1 atm
0.002 torr | 10000 Pa
1 atm >> 1 atm | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1 Torr', '1 atm >> 1 atm', '0.00002 Torr | 1 Torr >> 1 Torr', '1 atm | 1 atm', '0.000001 mbar | 0.0000001 mbar', '1 *10-6bar | 1 *10-6bar', '0.0000002 Torr', '0.0000001 mbar | 0.0000001 mbar | 0.0000001', '0.0005 Pa | 0.0005 Pa', '0.00002 Torr | 1 Torr', '1 atm | 1 atm >> 1 atm', '0.000001 mbar', '0.0000048 Torr | 0.0000048 Torr', '0.0000001 Torr', '1 atm | 1 atm | 1 atm | 1 atm', '0.005 Torr | 1 atm', 'nan >> nan | nan', '0.000005 mbar | 0.000005 mbar', '0.00001 Pa | 0.00001 Pa', '0.000001 mbar | 0.000001 mbar', 'nan | nan', '1 atm | 1 atm | 1 atm', 'nan | 0.000001 mbar', 'nan | 1 atm', '0.000001 Torr', '0.0075 Torr', '1 atm >> 1 atm | 1 atm', 'nan |  E-6torr', '1 atm | 0.000001 mbar', 'nan | 0.000009 mbar', '0.00002 Torr', 'nan | 5 E-4mbar', '1 Torr >> 1 Torr', 'nan | 0.0005 Pa | 0.0005 Pa', '1 atm'])))

    deposition_synthesis_atmosphere_pressure_partial = Quantity(
        type=str,
        shape=[],
        description="""
    The partial pressures for the gases present during each reaction step.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the partial pressures and separate them with semicolons, as in (A; B). The list of partial pressures must line up with the gases they describe.
- In cases where no gas mixtures are used, this field will be the same as the previous filed.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 0.99 atm; 0.01 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1 Torr', '1 atm >> 1 atm', '0.0004023 Torr; 0.0046 Torr | 1 atm', '0.001 Torr; 0.004 Torr | 1 atm', '0.00002 Torr | 1 Torr >> 1 Torr', '1 atm | 1 atm', '0.000001 mbar | 0.0000001 mbar', '1 *10-6bar | 1 *10-6bar', '0.00002 Torr | 1 Torr', '1 atm | 1 atm >> 1 atm', '0.3 Torr', '0.000001 mbar', '1 atm | 1 atm | 1 atm | 1 atm', 'nan >> nan | nan', 'nan | 0.3 Torr', '0.0065 Torr; 0.001 Torr', '1 atm; 1 bar | 1 atm', '0.000001 mbar | 0.000001 mbar', 'nan | nan', '1 atm | 1 atm | 1 atm', 'nan | 1 atm', '0.000001 Torr', '1 atm >> 1 atm | 1 atm', '1 atm | 0.000001 mbar', '0.00002 Torr', '1 Torr >> 1 Torr', '0.0004545 Torr; 0.004545 Torr | 1 atm', '1 atm', '1 amt'])))

    deposition_synthesis_atmosphere_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relative humidity during each deposition step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the relative humidity associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns
- If the relative humidity for a step is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 35 and not 30-40.
Example
35
0 | 20
25 >> 25 | 0
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '25', '30 >> 30', '0 >> 0', '15', '48 | 48', '0 | 0 >> 0', '20 | 20', '25 | 25', '30 | 30', '40', '30; 30', '50 | 50', '20', 'nan >> nan | nan', '50 | 50 | 50', '30', '35 | 0', 'nan | nan', '30 | 30 | 30 | 30', '0 | 0', '55; nan', '48', '30 | 30 | 30', '30 >> 30 | 30', '35 | 35 >> 35', '30 | 0', '35 | 35'])))

    deposition_solvents = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in each deposition procedure for each layer in the stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvents associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the solvents in alphabetic order and separate them with semicolons, as in (A; B)
- The number and order of layers and deposition steps must line up with the previous columns.
- For non-liquid processes with no solvents, state the solvent as ‘none’
- If the solvent is not known, state this as ‘Unknown’
- Use common abbreviations when appropriate but spell it out when risk for confusion
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
DMF
Acetonitil; Ethanol | Ethanol
None | Chlorobenzene
H2O >> H2O | Methanol
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Ethanol | Ethanol', 'Unknown | IPA', 'Dichlorobenzene | none', 'Acetyl acetone; IPA | Water', 'Ethanol | anhydrous 1-butanol | Water >> Water; Ethanol >> Methanol; Water >> Water; Ethanol', '1-butanol >> 1-butanol | Ethanol', 'Ethanol; distilled Water >> nitric acid >> NaOH aqueous solution >> distilled Water; HCl | distilled Water', 'IPA | Unknown | Unknown', 'IPA >> Water | Ethanol; Water', 'Ethanol | Ethanol | Water', 'none >> 1-Butanol; IPA | Terpineol', 'IPA | Chlorobenzene', 'Ethanol | Terpineol; 2-Methoxy ethanol', '2-methoxyethanol | Chlorobenzene', 'Unknown | Unknown', 'none', 'Ethanol | P25; polyethylenglycol; OP; Water', 'Unknown >> Water >> Water >> Ethanol | Ethanol', 'n-butyl alcohol | n-butyl alcohol', 'IPA | Methanol >> Water >> Water', 'IPA >> Water | Ethanol', 'Water >> Water >> Ethanol', 'IPA | IPA', 'Chlorobenzene', '1-Butanol; IPA | Etanol >> Water', 'Ethanol | Dichlorobenzene', '1-Butanol | Ethanol | DMF', 'Ethanol', 'urea | hydrochloric acid | thioglycolic acid | SnCl2·2H2O | DI water', '2-methoxyethanol; Ethanolamine', '1-butanol | Water', 'butanol | Unknown', 'butanol | Ethanol', 'Chloroform | IPA', 'Water | 1-butanol >> Water', 'IPA | TiO2-np', 'butanol | IPA | Water', 'Ethanol | Ethanol | Ethanol', 'Terpineol | Terpineol | none', 'Ethanol; distilled Water >> nitric acid', 'Water | Ethanol', 'Chlorobenzene | none | none', 'Chloroform', 'Chlorobenzene | Ethyl alcohol', 'Acetyl acetone; IPA | Ethanol', 'Ethanol >> Water', '2-Butanol | Chlorobenzene', 'Water >> Water', 'Ethanol | Unknown | Unknown', '1-butanol | Unknown', 'IPA | none', 'Chlorobenzene | IPA | none', 'Ethanol | Ethanol | Ethanol | none', 'o-xylene >> tetrabutylammonium hydroxide 30-hydrate', 'Ethanol; distilled Water >> nitric acid >> none >> distilled Water; HCl', '2-methoxyethanol | Methanol', 'IPA; Ethanol | Terpineol | Unknown', 'Anh ethanol', 'Acetonitil; Acetyl aceton; IPA | Ethanol >> acetonitrile', 'Chloroform | Isopropyl alcochol', 'Ethanol | Ethanol | Ethanol | Unknown', 'Water | Unknown', '2-methoxyethanol >> 2-Butanol | Chlorobenzene', 'IPA | Water', 'none | Terpineol', 'Chlorobenzene | Ethanol', 'ethonal | 2-methoxyethanol | terpineol | acetonitrile', 'Water >> Ethanol | a-Terpineol >> Ethanol', 'Unknown >> Water | Chlorobenzene', 'Dicholorobenzene | IPA', 'Water | Methanol', 'Methanol | Water', 'Ethanol | Ethanol | Unknown', 'Ethanol >> Ethanol', '1-butanol  ethanol >> Water', 'IPA | Ethanol', 'Water | Water', 'Unknown | none | Unknown', 'Chloroform; IPA', '1-butanol | Ethanol', 'Ethanol | Water', 'n-butylalcohol | Ethanol', '1-Butanol >> 1-Butanol >> 1-Butanol | Acetylacetone; Polyethylene glycol; Triton X-100', 'Acetonitil; Acetyl aceton; IPA | Ethanol', 'Ethanol; HCl | Ethanol', 'Ethanol | Chlorobenzene', 'IPA >> Water | none', 'Dichlorobenzene', 'Acetyl acetone; Ethanol | Ethanol', '1-Butanol; IPA | Terpineol', 'Unknown >> Water', 'Unknown', 'Ethanol >> Water | Ethanol', 'none | Ethanol', 'Ethanol | Ethanol >> acetonitrile', 'Chlorobenzene | none', 'Chlorobenzene; DCB | IPA', 'Water | Chlorobenzene', 'Ethanol >> Ethanol | IPA', 'n-butyl alcohol | Unknown', 'terpineol; Ethanol', '1-Butanol', '2-methoxyethanol | Water', 'Water2; Water | Chlorobenzene', 'Chlorobenzene; Octane | none', 'IPA >> IPA >> IPA | Unknown >> Unknown >> Unknown >> Unknown >> Unknown >> Unknown >> acetonitrile', 'IPA; Ethanol | Ethanol', 'none >> Water >> Water', 'IPA >> Ethanol', 'Water', 'Chlorobenzene | 2-methoxyethanol', 'Anisole; tetralin; TAA', 'Water | IPA', 'none | Chlorobenzene', 'none | none', 'Unknown | Ethanol', 'none | terpineol; Ethanol', '1-butanol', 'Ethanol; Water', 'Ethanol | Methanol', '1-butanol | Ethanol >> Water | Ethanol >> Water', '1-Butanol; IPA | 1-Butanol; Ethylcellulose; Llauric acid; Terpineol', 'Chlorobenzene | Methanol', '2-methoxyethanol', '2-methoxyethanol >> 2-Butanol', 'IPA; Etanol | Etanol', 'Chlorobenzene | Unknown', 'n-butylalcohol | 2-methoxyethanol', 'IPA >> Water', 'n-butyl alcohol', 'Dichlorobenzene | IPA', 'IPA; Ethanol | none', 'n-butylalcohol', 'Ethanol | anhydrous 1-butanol', 'IPA | Unknown', 'Ethanol | Unknown', 'Ethanol | none', '1-Butanol | Ethanol', 'Methanol; n-butanol; Chloroform', 'Chlorobenzene | Trifluorethanol', 'Chlorobenzene | Unknown | Unknown', 'Chlorobenzene | Water', 'Chlorobenzene | Chlorobenzene | Ethanol', 'Water2; Water', 'deionized water', 'n-butyl alcohol | Ethanol', 'IPA', 'n-butyl alcohol | n-butyl alcohol | Unknown', 'Ethanol | IPA', 'Chlorobenzene | IPA'])))

    deposition_solvents_mixing_ratios = Quantity(
        type=str,
        shape=[],
        description="""
    The mixing ratios for mixed solvents
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent mixing ratios associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- For pure solvents, state the mixing ratio as 1
- For non-solvent processes, state the mixing ratio as 1
- For unknown mixing ratios, state the mixing ratio as ‘nan’
- For solvent mixtures, i.e. A and B, state the mixing ratios by using semicolons, as in (VA; VB)
- The preferred metrics is the volume ratios. If that is not available, mass or mol ratios can be used instead, but it the analysis the mixing ratios will be assumed to be based on volumes.
Example
9; 0.6; 0.4 | 1
1 >> 1 | 1
9; 1 | 3; 2
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    deposition_solvents_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of all the solvents.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For non-liquid processes with no solvents, mark the supplier as ‘none’
- If the supplier for a solvent is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Sigma Aldrich
Sigma Aldrich; Fisher | Acros
none >> Sigma Aldrich; Sigma Aldrich | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Sigma Aldrich | Unknown >> Unknown', 'Sigma Aldrich | Sigma Aldrich | Unknown >> Unknown; Sigma Aldrich >> Unknown; Unknown >> Unknown; Sigma Aldrich', 'Acros Organics; Sigma Aldrich', 'sigma aldrich', 'Sigma Aldrich; Sigma Aldrich | Unknown >> Unknown', 'Sigma Aldrich', 'Alfa Aesar', 'Sigma-Aldrich', 'Unknown', 'NanoPac >> Unknown | NanoPac', 'Millipore Sigma | Milllipore Sigma', 'Unknown | Unknown', 'Sigma Aldrich | Sigma Aldrich', 'Sigma Aldrich | Unknown >> Unknown | Unknown >> Unknown', 'Unknown >> Unknown | Unknown', 'Sigma Aldrich; Sigma Aldrich', 'Wako; Wako; Wako; Unknown', 'Unknown >> Unknown | Alfa Aesar', 'Sigma Aldrich | Fischer Scientific', 'Kanto Chemical Tokyo; Unknown', 'Unknown | Sigma Aldrich', 'Unknown >> Sigma Aldrich; Sigma Aldrich | Sigma Aldrich', 'NanoPac >> Unknown', 'Sigma Aldrich; Sigma Aldrich; Fisher | Sigma Aldrich >> Acros', 'Sigma Aldrich | Unknown', 'Sigma Aldrich; Sigma Aldrich | Sigma Aldrich', 'Unknown; Sigma Aldrich', 'Sigma Aldrich; Unknown >> Sigma Aldrich >> Sigma Aldrich >> Unknown; Sigma Aldrich | Unknown', 'Sigma Aldrich | Sigma Aldrich >> Sigma Aldrich', 'Kanto Chemical Tokyo; Unknown | Unknown', 'Alfa Aesar | Alddin', 'Sigma Aldrich | Unknown | Unknown', 'Sigma Aldrich | Sigma Aldrich | Unknown', 'Nacalai Tesque', 'Sigma Aldrich; Unknown >> Sigma Aldrich', 'Unknown | Wako Pure Chemical Industries; Nacalai Tesque; Sigma Aldrich', 'Unknown | Alfa Aesar', 'Unknown; Unknown | Unknown', 'Sigma Aldrich; Unknown >> Sigma Aldrich >> Unknown >> Unknown; Sigma Aldrich', 'Sinopharm Chemical Reagent Co. Ltd.'])))

    deposition_solvents_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the solvents used.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For non-liquid processes with no solvents, state the purity as ‘none’
- If the purity for a solvent is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
Puris; Puris| Tecnical
none >> Pro analysis; Pro analysis | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Puris | Unknown >> Unknown | Unknown >> Unknown', 'Puris | Unknown >> Unknown', 'Puris | Puris | Unknown >> Unknown; Puris >> Unknown; Unknown >> Unknown; Puris', '99.8%; 99.5% | Uknown >> Unknown', 'Technical | Puris', '99.5% | 99.5%', 'Unknown', '99.8% | 99.5%', '99.8% | Unknown | Unknown', 'Unknown | Unknown', 'Unknown | Unknown | Unknown', 'Unknown >> Unknown | Unknown', '0.998', '99.8% >> Unknown | 99.8%', 'Puris', 'Puris; Puris', '99.8 >> Unknown | 99.8%', '99.8% >> Unknown', 'Pro analysis; Pro analysis; Pro analysis | Puris >> Pro analysis', 'Unknown; Puris', 'Puris | Puris', 'Anhydrous 99.8%'])))

    deposition_reaction_solutions_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    The non-solvent precursor chemicals used in each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemicals associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several compounds, e.g. A and B, list the associated compounds in alphabetic order and separate them with semicolons, as in (A; B)
- Note that also dopants/additives should be included
- When several precursor solutions are made and mixed before the reaction step, it is the properties of the final mixture used in the reaction we here describe.
- The number and order of layers and reaction steps must line up with the previous columns.
- For gas phase reactions, state the reaction gases as if they were in solution.
- For solid-state reactions, state the compounds as if they were in solution.
- For reaction steps involving only pure solvents, state this as ‘none’
- If the compounds for a deposition step is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Titanium diisopropoxide bis(acetylacetonate) | TiO2-np
C60 | BCP
Titanium diisopropoxide bis(acetylacetonate) | TiO2-np >> Li-TFSI
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'nan | TiO2-np Anatase; TiO2-np Rutile', 'PEIE >> nTi-MOF | PCBM-60', 'TiO2-np; titanium diisopropoxide bis(acetylacetonate)', 'C60 | TDMASn', 'TiOx >> TiOx', 'Zinc Acetate dehydrate; ethanolamine | 3, 4, 5- trimethoxybenzoic acid', 'SnO2-np | PCBM-60; PEG', 'SnCl4', 'ICBA | BCP', 'SnO2-np | Choline Chloride', 'Titanium diisopropoxide bis(acetylacetonate) >> Titanium diisopropoxide bis(acetylacetonate) | CsAc', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2-np >> TiCl4 >> nan', 'TiCl4 | TiO2 paste', 'SnCl2', 'Titanium isopropoxide; diethanolamine | Titanium tetrabutanolate', 'SnO2-np', 'PCBM-60 | tetrakisdimethylamino-tin; H2O >> tetrakisdimethylamino-tin; H2O', 'Titanium diisopropoxide bis(acetylacetonate) | Titanium diisopropoxide bis(acetylacetonate) | (3-aminopropyl)trimethoxysilane (APTMS)', 'nan >> TiCl4 >> nan >> nan | TiO2-np', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2-np', 'Titanium(IV) isopropoxide >> nan >> nan >> NaOH >> nan | magnesium methoxide', 'Acetylacetone; Titanium isopropoxide | TiO2 paste NR30-D', 'Titanium isopropoxide (TTIP); HCl | TiO2 paste 18NRT', 'Zinc acetate dihydrate; KOH; ZnO', 'PCBM-60 | PFN', 'PEIE >> nTi-MOF', 'PCBM-60 | PEI', 'Zinc Acetate dehydrate; ethanolamine | 2-methoxybenzoic acid', 'PCBM | BCP', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2-np | ZrO-np', 'diethanolamine; Titanium isopropoxide; H2O', 'SnO2 | C60', 'Ti >> NaOH >> HCl', 'nan | TiO2-np', 'Zinc Acetate dehydrate; ethanolamine | 4-dimethoxybenzoic acid', 'TiO2 np', 'Titanium oxysulfate | PDI-glass', 'ZnO-np', 'SnCl2 | C60', 'Titanium diisopropoxide bis(acetylacetonate) >> Titanium diisopropoxide bis(acetylacetonate) >> Titanium diisopropoxide bis(acetylacetonate) | TiO2-np', 'PCBC6 | BCP', 'Titanium isopropoxide (TTIP); acetyl acetone >> NR30-D; ethanol', 'synthesized | synthesized | synthesized', 'tetrakisdimethylamino-tin; H2O', 'PCBM-60 | PEIE', 'Zinc Acetate dehydrate; ethanolamine', 'TiO2-np; Ethylcellulose', 'SnO2-np | 4-Bromobenzoic acid', 'nan | TiO2-np; Ethylcellulose', 'SnO2-np | ZnO-np', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2-np >> Li-TFSI', 'TPE-DPP-8 | C60 | BCP', 'SnCl5', 'HCl; Titanium isopropoxide | TiO2 paste | nan', 'TiO2 pellets | TTIP', 'SnO2', 'Titanium diisopropoxide bis(acetylacetonate) | Al2O3-np | Cu:NiO-np', 'Titanium diisopropoxide bis(acetylacetonate) | MgCl6 6H2O; SnCl2 2H2O | TiO2-np | ZrO2-np', 'Titanium isopropoxide (TTIP); HCl', 'TiO2-np; titanium diisopropoxide bis(acetylacetonate) | PPDI-F3N', 'PCBM-60 | Rhodamine 101 | LiF', 'B2F | C60 | BCP', 'Titanium diisopropoxide bis(acetylacetonate) | CaSc2O4:0.15Eu3+', 'titanium tetrachloride', 'Titanium diisopropoxide bis(acetylacetonate); TiO2-np', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2 paste NR30-D', 'SnO2-np | KOH', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2 paste NR30-D | SnCl4', 'Titanium diisopropoxide bis(acetylacetonate); TiO2-np; PCBM', 'Titanium isopropoxide | TiO2 paste', 'Acetylacetone; Titanium diisopropoxide bis(acetylacetonate) | TiO2 paste', 'Titanium diisopropoxide bis(acetylacetonate) | Titanium diisopropoxide bis(acetylacetonate)', 'Titanium isopropoxide | TiO2-np | Zn(NO3)2.6H2O >> nan >> Na2S >> nan', 'tantalum(V) ethoxide in titanium; Titanium diisopropoxide bis(acetylacetonate)', 'PCBM-70 | TiO2', 'TPE-DPP-6 | C60 | BCP', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2-np >> TiCl4', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2 paste | Al2O3 paste', 'Titanium Orthotitanate | TiO2-np', 'PCBM-60', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2 paste', 'SnO2 2H2O', 'nan | Al2O3-np', 'Titanium diisopropoxide bis(acetylacetonate)', 'Titanium diisopropoxide bis(acetylacetonate) >> TiCl4 | C60', 'Titanium isopropoxide (TTIP); HCl | C60 solution', 'nan | TiO2-np Rutile', 'PCBM-60 | Bphen', 'PEIE; C60', 'PCBM-60 | tetrakisdimethylamino-tin; H2O', 'SnCl2.H2O | ethanol | ( 3-aminopropyl)triethoxysilane (APTES) | IPA', 'Tetraisopropyl orthotitanate | TiO2-np', 'Titanium isopropoxide', 'tetrakisdimethylamino-tin; H2O >> tetrakisdimethylamino-tin; H2O', 'PCBM-70', 'SnO2-np | NaOH', 'Titanium tetrachloride', 'Titanium diisopropoxide bis(acetylacetonate); vanadium(V)oxytriethoxide | TiO2 powder; polyethylene glycol >> acetylacetone; triton X-100', 'C60 | LiF | BCP', 'HCl; Titanium isopropoxide | TiO2 paste', 'Titanium(IV) isopropoxide >> nan >> nan >> NaOH >> nan', 'Titanium diisopropoxide bis(acetylacetonate) | Polystyrene latex microsphere solution >> TiCl4 | TiO2-np >> TiCl4', 'synthesized | BCP', 'Tetra-nbutyl titanate; diethanolamine >> TiCl4 | TiO2 paste >> TiCl4 | ZrO2 paste', 'Titanium oxysulfate', 'C60', 'nan | TiO2 paste 18NRT', 'nan | TiO2-np | ZrO-np', 'Titanium(IV) isopropoxide >> nan', 'SnCl2 | PCBM-60', 'PCBM-60 | PFN-Br', 'nan | C60-SAM', 'HCl; TiCl3', 'C60; (RuCp*mes)2', 'PCBM-60 | Rhodamine 101', 'TiOx >> TiOx >> TiOx', 'TiCl4', 'PCBM-60 | AZO', 'Titanium diisopropoxide bis(2,4-pentanedionate) | In(NO3)3·xH2O', 'nan | TiO2-np >> TiCl4', 'TPE-DPP-16 | C60 | BCP', 'Titanium isopropoxide | Ethanol', 'PCBM | TBAOH', 'CdI2 | C60 | BCP', 'Titanium isopropoxide >> TiCl4 | TiO2-np', 'nan | TiO2-np Anatase', 'Titanium diisopropoxide bis(acetylacetonate) >> TiCl4 | TiO2-np', 'Zinc acetate; Tin Acetate', 'TiOx', 'Titanium isopropoxide >> TiCl4 | PCBA', 'Titanium tetrabutanolate', 'titanium diisopropoxide bis(acetylacetonate) | TiO2-np | ZrO2-np', 'Titanium diisopropoxide bis(acetylacetonate) >> TiCl4', 'Titanium isopropoxide; diethanolamine | TiO2-np >> TiCl4', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2-np | SbI3', 'TiO2-crystalline anatase np', 'Titanium isopropoxide (TTIP)', 'nan | nan', 'Tetrabutyl titanate | TiO2-np', 'TiO2-c | PCBM-61', 'nan | TiO2-np Anatase >> CdSO4; CS(NH2)2; NH3.H2O', 'SnCl2.H2O | ethanol', 'Titanium tetrachloride | TiO2-np', 'MgCl6 6H2O; SnCl2 2H2O | TiO2-np | ZrO2-np', 'TiCl4 | PCBM-60', '1,4,5,8-Naphthalenetetracarboxylic dianhydride; (R)-(-)-aminoindane', 'Ti', 'Tetrabutyl titanate', 'Tetra-nbutyl titanate; diethanolamine | TiO2 paste NR30-D', 'COi8DFIC | BCP', 'Titanium isopropoxide >> TiCl4', 'TiO2 -np | Ti | ZrO2-np', 'C60 | BCP', 'Acetylacetone; IPA; tetrabutyl titanate; Triton X100 | TiO2 paste', 'NDI3HU-DTYM2 | BCP', 'TiO2-anatase >> Titanium diisopropoxide bis(acetylacetonate) | TiO2-np', 'titanium tetrachloride | TiO2-np >> titanium tetrachloride', 'Titanium diisopropoxide bis(acetylacetonate) >> Titanium diisopropoxide bis(acetylacetonate)', 'TiCl4 >> TiCl4', 'Titanium diisopropoxide bis(acetylacetonate); niobium(V)ethoxide | TiO2 powder; polyethylene glycol >> acetylacetone; triton X-100', 'SnO2-np | PCBM-60', 'PCBM-70 | Rhodamine 101', 'Titanium diisopropoxide bis(acetylacetonate) | SnCl4', 'PCBM | Bis-C60', 'PEIE', 'Nb | TiO2-np', 'Titanium isopropoxide | PCBM-60', 'titanium isopropoxide', 'ITIC | BCP', '(DTYM-NDI-DTYA)2 | BCP', 'PCBM-60 | ZnO-np', 'TiO2-nw', 'Titanium isopropoxide | TiO2-np >> Li-TFSI', 'TiCl4 >> nan >> nan', 'Titanium diisopropoxide bis(2,4-pentanedionate)', 'Titanium diisopropoxide bis(acetylacetonate) >> Titanium diisopropoxide bis(acetylacetonate) | TiO2-np', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2 paste 18NR-T', 'MoS2-nanosheets', 'Titanium diisopropoxide bis(acetylacetonate); acetylacetone', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2 powder; polyethylene glycol >> acetylacetone; triton X-100', 'PCBM-60 | C60 | LiF', 'Tetrabutyl titanatein; HCl', 'Zinc Acetate dehydrate; ethanolamine | PCBM-60', 'TPE-DPP-12 | C60 | BCP', 'SnO2-np | ethylphosphonic acid', 'TiO2-anatase | TiO2-np', 'HCl; Titanium isopropoxide', 'Titanium isopropoxide | TiO2-np', 'IT-4f | BCP', 'Titanium diisopropoxide bis(acetylacetonate); tantalum(V)ethoxide | TiO2 powder; polyethylene glycol >> acetylacetone; triton X-100', 'tin chloride bihydrate', 'PCBM-61 | BCP', 'Titanium tetraisopropoxide; acetylacetone | TiO2-np | Carbon-QDs', 'Titanium diisopropoxide bis(acetylacetonate) | TiO2 paste | ZrO2 paste | carbon paste', 'Titanium diisopropoxide bis(acetylacetonate) >> titanium tetrachloride | TiO2-np', 'CdSO4; thiourea; NH4OH', 'HCl; Titanium isopropoxide | 2-mIm; Zn(NO3)2', 'PCBM | LiF', 'PCBM-60 | BCP', 'SnCl4 | 1‐butyl‐3‐methylimidazolium bromide', 'Titanium tetraisopropoxide; acetylacetone | TiO2-np', 'PCBM | PNDI-2T | LiF', 'tetraamminezinc(II) hydroxide', 'HCl; Titanium isopropoxide | TiO2 paste 22NR-T Solaronix', 'Titanium isopropoxide | HCl | Ethanol', 'titanium (diisopropoxide) bis(2,4-pentanedionate) | TiO2-np'])))

    deposition_reaction_solutions_compounds_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemical suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For gas phase reactions, state the suppliers for the gases or the targets/evaporation sources that are evaporated/sputtered/etc.
- For solid state reactions, state the suppliers for the compounds in the same way.
- For reaction steps involving only pure solvents, state the supplier as ‘none’ (as that that is entered in a separate filed)
- For chemicals that are lab made, state that as “Lab made” or “Lab made (name of lab)”
- If the supplier for a compound is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Dysole; Sigma Aldrich; Dyenamo; Sigma Aldrich
Sigma Aldrich; Fisher | Acros
Lab made (EPFL) | Sigma Aldrich >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Pegasus; Uppsala University', 'Unknown | Pegasus; Uppsala University >> Pegasus; Uppsala University', 'Sinopharm Chemical Reagent Co., Ltd; Macklin | Macklin | Shanghai MaterWin New Material', 'Fisher Scientific; Greatcell Solar', "Xi'an p-OLED Corp. | Xi'an p-OLED Corp.", 'Unknown | Unknown', 'Sigma Aldrich | Sigma Aldrich', 'Sigma Aldrich | Dysole >> Sigma Aldrich >> Unknown', 'Solenne B.V. | Sigma-Aldrich', 'Sinopharm', 'Merck | Solaronix', 'ENB Korea >> Sigma Aldrich', 'Nichem Fine Technology | Nichem Fine Technology', 'Sigma Aldrich | Alfa-Aesar >> Junsei Chemicals | Dyesol >> Junsei Chemicals', 'Nichem Fine Technology Co. Ltd.', 'ENB Korea >> Sigma Aldrich >> ENB Korea', 'Nano-c', 'Sigma Aldrich | Dysole', 'Sigma Aldrich | Sigma Aldrich | Sigma Aldrich', 'nano-c; Unknown', 'Sinopharm | America Dye Sources', 'CBMM | Dyesol', 'Acros', '1-Material Inc', 'Sigma Aldrich', 'Nichem', 'Solenne | Sigma Aldrich', 'Sigma Aldrich | Dysole >> Sigma Aldrich', 'Millipore Sigma | Milllipore Sigma', 'Sinopharm; Sinopharm', 'Han Feng Chemical | P-OLED', 'Unknown | Dyesol', 'Sigma Aldrich; Synthetized', 'Lab made', 'Unknown | Sigma Aldrich', 'Unknown >> Unknown | JGC Catalysts and Chemicals Ltd.', 'Unknown | Dyesole', 'Sigma Aldrich | Sinopharm Chemical Reagent Co., Ltd; Macklin | Macklin | Shanghai MaterWin New Material', 'Sigma Aldrich | Lab-made', 'Xi’an Polymer Light Technology Corp', 'Unknown | Degussa', 'Sigma Aldrich >> Unknown', 'Sigma Aldrich | Unknown >> Sigma Aldrich', 'Sigma Aldrich; Solaronix; Solaronix', 'Alfa Aesar', 'Sigma Aldrich | Unknown >> Sinopharm', "Xi'an Polymer Light Technology Corp. | Xi'an Polymer Light Technology Corp.", 'Unknown', 'Solaronix | Unknown', 'NanoPac >> synthsized', 'Unknown | Nichem Fine Technology', 'Nano-C | Alfa-Aesar', 'Pegasus; Uppsala University >> Pegasus; Uppsala University', 'Sigma Aldrich; Sigma Aldrich', 'Sigma Aldrich | Dyesol >> Junsei Chemicals', 'Aldrich; Aldrich | Nippon Aerosil; Nacalai Tesque >> Wako Pure Chemical; Wako Pure Chemical', 'Ossila | Unknown', 'Alfa-Aesar', '1-Material', 'Aldrich | Nippon Aerosil; Nacalai Tesque >> Wako Pure Chemical; Wako Pure Chemical', 'Sigma Aldrich | SureChem', 'Unknown; Sigma Aldrich', "Lab made | Xi'an Polymer Light Technology Corp. | Xi'an Polymer Light Technology Corp.", 'Aldrich; Solaronix Ti-Nanoxide N/SP', 'NanoPac >> Lab made | NanoPac', '1-Material | 1-Material', 'Nano-C', 'Sigma Aldrich >> Unknown >> Sigma Aldrich >> Unknown', 'Sigma Aldrich | Greatcell', 'Unknown | Dyesole | Solaronix', 'Unknown | Pegasus; Uppsala University', 'Solarmer Material | 1-Material', 'Sigma Aldrich | Solaronix PST-18NR | Sigma Aldrich | Borun New Material Technology', 'Unknown | 1-Materials', 'Luminescence Technology Corp', 'Luminescence Technology Corp | Xi’An Polymer Light Technology Corp', 'Sigma Aldrich >> ENB Korea', 'Alfa Aesar >> Unknown >> Unknown', 'Sigma Aldrich | Dyesol', 'Unknown | Sigma Aldrich; Degussa', 'ITASCO >> Unknown >> Unknown', 'Alfa-Aesar | Lab made', 'Sigma Aldrich >> Unknown >> Sigma Aldrich >> Unknown | Sigma Aldrich', 'Frontier Carbon Corp. | Tokyo Chemical Industry', 'Sigma Aldrich | Dyesol | Sigma Aldrich >> Unknown >> Sigma Aldrich >> Unknown', 'Unknown | NanoCleantech', 'Sigma Aldrich | Aerosil', 'America Dye Sources'])))

    deposition_reaction_solutions_compounds_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the compound purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, i.e. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For reaction steps involving only pure solvents, state this as ‘none’ (as that is stated in another field)
- If the purity for a compound is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
99.999; Puris| Tecnical
Unknown >> Pro analysis; Pro analysis | none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Pro analysis', 'Pro analysis; Unknown', '99%; 99.99%', 'Puris | Unknown >> Puris | Unknown >> Puris', '0.97', '98%; 99.8%', '99.99% >> Unknown', 'Unknown >> Unknown >> Unknown >> Unknown | Unknown', 'Unknown', 'Unknown | Unknown', 'Puris | Unknown', 'Unknown; 97%', 'Pro analysis; Puris | Puris >> Puris', 'Pro analysis | Tecnical >> Pro analysis', '0.98', 'Unknown >> Unknown | Unknown', 'Puris', '0.999', 'Technical | Unknown', '96% | Unknown', 'Unknown >> Unknown >> Unknown >> Unknown', 'Unknown >> Unknown', '99.5% | 99%', '99.99% >> Unknown | Unknown', 'Puris | Pro-analysis', 'Unknown | 99%', '99.9% | Unkown', 'Puris | Unknown >> Puris', 'Pro analysis | Puris >> Puris', 'Unknown | Unknown | Puris >> Unknown >> Puris >> Unknown', 'Unknown | 99.8%'])))

    deposition_reaction_solutions_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the non-solvent precursor chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the concentrations associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated concentrations and separate them with semicolons, as in (A; B)
- The order of the compounds must be the same as in the previous filed.
- For reaction steps involving only pure solvents, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used. When possible, use one of the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml, wt%, mol%, vol%, ppt, ppm, ppb
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
0.2 M; 0.15 M| 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.05 M', '0.15 M | 33.33 wt%', '15 wt%', '16 mg/ml', '15 mg/ml | nan', '0.42 M', '120 mg/ml; 640 mg/ml', '25 mg/ml | nan', '20 mg/ml; nan', '52 wt% | 208 mg/ml; 14 mg/ml | 20 wt% | 20 wt%', '0.15 M | 0.20 M', '20 mg/ml | 30 mg/ml', 'nan | nan >> 80 mM', '8 mg/ml | nan', '20 mg/ml | 1 mg/ml', '15 mg/ml | 1 mg/ml', '2 mg/ml | 0.6 mg/ml | nan', '164 mg/ml; 50 mg/ml', '1 M; 0.5 M', '14 vol%; 20 vol%', '3 mg/ml | nan', '0.2 M', 'nan >> 0.04 M >> nan >> nan | 20 wt%', '0.15 M | 0.15 M', '13 wt% | 20 wt%', '2.564 vol%; 25 wt%; nan; nan', '10 wt% | 20 wt%', '5 mg/ml', '12 wt% | 20 wt%', '208 mg/ml; 14 mg/ml | 20 wt% | 20 wt%', '20 mg/ml | nan | nan', 'nan | nan >> 20 mM', '2.5 wt%; 20 wt%', '5 mg/ml | nan', '0.03 M', '15 mg/ml; 2 mg/ml', '0.254 M; 0.02 M; 0.1 mg/ml', '0.15 M; nan', '2 mg/ml | nan | nan', '0.16 M', '33 wt% | 10 mg/ml; 10 mg/ml', 'nan | nan >> 100 mM', '20 wt%', '0.2 M; 2 M | 150 mg/ml', '25 wt%; 28 wt%', '400 mM >> nan >> nan', '75 wt% | nan', '0.15 M | 7.5 mg/ml', '20 mg/ml | 0.1 wt%', '11 wt% | 20 wt%', '20 mM; 150 mg/ml', '0.2 M | nan', '200 mM >> nan >> nan', '2 vol%; 15 vol%', '15.38 vol%', '20 mg/ml | 0.2 %', 'nan | 10 mg/ml', '0.286 vol%; 0.2 wt%', '0.3 M; 10 mg/ml', 'nan; 0.9 vol%', '20 mg/ml; 0.5 mg/ml', '0.3 M', 'nan | 2.0 M', '1.8 wt% | nan', '20 mg/ml', '40 mg/ml | 0.5 mg/ml', '8.6 vol%; 5.71 vol% >> 12 wt%', '14 wt% | 20 wt%', '6 vol% | 125 mg/ml', '0.2 M | 15 mg/ml', '6 vol% | 150 mg/ml >> 10 mg/ml', '0.355 vol%; 6.4 vol%', '4 wt%', '46.7 vol% >> nan', '17 wt% | 20 wt%', '8 mg/ml', '2 wt%', '100 mg/ml; 2.8 mg/ml | 10 mg/ml', '7.5 vol% | 28 %', '0.08', '16 wt% | 20 wt%', '2 wt% | 1 mg/ml', '20 mg/ml | nan | 0.5 mg/ml', 'nan | nan >> 0.1 M', 'nan | 150 mg/ml', '0.3 M >> 40 mM | nan', '20', '0.15', '40 mg/ml | 11.11 vol%', '20 mM >> 0.11 mg/ml >> 20 mM', '0.04 M | 20 mg/ml', 'nan | nan | 5 mM | nan', '15 wt% | 20 wt%', '100 mM >> nan >> nan', '2.5 wt%', '1.25 vol% | nan | nan', '20 wt% | 20 wt%', '46.7 vol% >> nan >> 20 mg/ml >> nan | nan', 'nan | 28.6 vol% >> 0.02 M', '0.3 M >> 40 mM', '20 mg/ml | 0.2 wt%', '5.26 vol% >> 4.2 vol%', '10 vol%', 'nan | 4 wt%', '6 vol% | 10 wt% | 1 mM', '40 mM', '0.0267', '0.3 M >> 40 mM | 14 wt%', '2.2 vol% >> 0.22 vol%', '2.67 wt% | 20 mM', '10 mg/ml', 'nan | 0.1 g/0.8mL >> 0.1 M', 'nan | 0.04 M', '0.2 M | 5 mg/ml', '75 wt% | 25 wt% >> 0.05 M >> nan', 'nan >> nan | 22.2 wt%', '20 mg/ml | 40 mg/ml', '0.15 M | nan', 'nan >> 1 M >> 0.4 M', 'nan | 22.22 wt%', '0.2 M | 10 mg/ml', '20 mg/ml >> 0.7 mg/ml', '18 mg/ml', 'nan | 22.2 wt%', 'nan | 1.5 M', '2 vol%; 15 vol% | 22 mg/ml; 9.9 mg/ml', '0.125 M >> 0.025 M | 0.1 mg/ml', '20 mg/ml | 1.5 mg/ml', '0.15 M | 130 mg/ml >> 0.02 M', 'nan | 0.12 g/ml | 0.2 M >> nan >> 0.1 M >> nan', '0.15 M', '2.4 vol%', 'nan | nan >> 40 mM', '13.3 mg/ml; 3.6 vol%; 0.098 vol%', '20 mg/ml | 2.5 wt%', '15 wt% | 0.5 mg/ml', '20 mg/ml; 0.8 mg/ml', '14.6 vol%; 1 M; 22.2 wt%', 'nan | 41.6 wt% | nan', '20 mg/ml | 4.61 mg/ml', '33 wt% | 10 mg/ml', '0.3 M | 33 vol% >> 0.04 M | 28.6 wt% >> 0.04 M', '0.3 M | 28.6 wt% >> 0.04 M', '18 wt% | 20 wt%', '0.7 vol%; 8 vol% | 1.8 wt%', '0.15 M | 15.15 mg/ml >> 0.02 M', '0.5 M; 0.5 M | nan', 'nan | 0.5 M', '0.15 M | 0.3 M | nan', '0.15 M >> 0.3 M | 60 mg/ml', '10 vol% | nan', '2.25 vol%', 'nan | 14.3 wt%', '0.1 M', '22 wt% | 20 wt%', '2 mg/ml', '11.4 wt%', 'nan | nan', '10 mg/ml | 0.5 mg/ml', '164 mg/ml; 50 mg/ml | 0.7 mg/ml', '30 mg/ml | 0.5 mg/ml', '33 wt%', '0.04 M', '15 mg/ml | 2 mg/ml', '15 wt% | 1 mg/ml', '10 mg/ml | nan', '20 mg/ml | 0.6 mg/ml', '5.3 mg/ml; nan', '30 mg/ml', '16 wt% | 6.67 wt% | 0.5 wt%', '25 wt%28wt%', '23.6 mg/ml; 22.77 mg/ml; 6 mg/ml', '0.15 M | 2 :7wt', '20 mg/ml | 5 mg/ml', '0.4 wt% | nan', '20 mg/ml | 0.5 mg/ml', '30 mg/ml | nan', '46.7 vol% >> nan >> 20 mg/ml >> nan', 'nan | 22 wt%', '250 mg/ml', 'nan | 1.0 M', 'nan >> 0.15 M | nan', '1.43 mg/ml | nan', 'nan | 0.08 wt%', '1.5 mM; 1.5 mM; 30 %', '5.3 mg/ml; nan | 0.1 mg/ml', '20 mg/ml | nan', '0.15 M >> 0.30 M', '5 mol%; 10 vol%', '2.67 wt%', 'nan | 0.25 mg/ml', '0.15 M | 10 vol%', '0.254 M; 0.02 M', 'nan | 4 mg/ml', '10 vol% >> 10 vol%', '0.15 M >> 0.3 M >> 0.3 M | nan', '0.4 wt%', '200 mM | 22 wt%', '0.2 M | 0.066 wt%', '0.5 M; 0.5 | nan >> 0.04', '0.15 M | 0.10 M', '0.2 wt% >> 8 mg/ml | nan', '19 wt% | 20 wt%', '0.15 M | 0.05 M', '0.15 M; 20 wt%; 0.0002 M', '0.15 M; 20 wt%', '2 M; 16.67 wt%', '20 mg/ml | 20 mg/ml', 'nan; 60 mg/ml', '6 mg/ml', '15 mg/ml', '0.2 M | 7.5 mg/ml', '7 mg/ml | nan', 'nan; 4 mg/ml', '100 mg/ml; 2.8 mg/ml', '3 mol%; 10 vol%', '10 wt% | 0.8 mg/ml', '10 vol% >> 10 vol% | 0.1 M', '3 wt%', '200 mM', '10 vol% | 25 wt%', '10 vol% | 18 wt%', '1 mol%; 10 vol%', '6 vol%; 0.67 vol%', '0.15 M | 0.3 M', '22 mg/ml', '21 wt% | 20 wt%', 'nan | 0.7 mh/ml', '20 mg/ml | 10 mg/ml', '30 mg/ml | 0.1 wt%', '300 mM >> nan >> nan', 'nan | 0.12 g/ml', '0.2 wt% >> 8 mg/ml', 'nan | 0.2 M', '1 M'])))

    deposition_reaction_solutions_volumes = Quantity(
        type=str,
        shape=[],
        description="""
    The volume of the reaction solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the volumes associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The volumes refer the volumes used, not the volume of the stock solutions. Thus if 0.15 ml of a solution is spin-coated, the volume is 0.15 ml
- For reaction steps without solvents, state the volume as ‘nan’
- When volumes are unknown, state that as ‘nan’
Example
0.1
0.1 >> 0.05 | 0.05
nan | 0.15
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['1.0 | Unknown', '54.35', '0.9 >> 0.1 >> 15.0', '10.0', '125.0; 65.0; Unknown', 'Unknown >> Unknown >> Unknown >> Unknown | Unknown', 'Unknown', '7.0', '250.0 | Unknown', 'Unknown | Unknown', '100.0; 100.0', '8.0', 'Unknown >> Unknown | Unknown', '10.2', '20.0 | 30.0', '250.0', '0.369; 0.07 | Unknown', '50.0 | Unknown | Unknown | Unknown', '50.0 | Unknown | Unknown', 'Unknown | 0.1', '10.0 | 0.05 >> 0.1', '2.2; 2.2; 2.8', '0.15 | 0.3', 'Unknown >> Unknown >> Unknown >> Unknown', 'Unknown >> Unknown', '0.6; 0.4 >> 0.04', '30.0 | Unknown', '1.0', '1.0 | 1.0'])))

    deposition_reaction_solutions_age = Quantity(
        type=str,
        shape=[],
        description="""
    The age of the solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the age of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- As a general guideline, the age refers to the time from the preparation of the final precursor mixture to the reaction procedure.
- When the age of a solution is not known, state that as ‘nan’
- For reaction steps where no solvents are involved, state this as ‘nan’
- For solutions that is stored a long time, an order of magnitude estimate is adequate.
Example
2
0.25 |1000 >> 10000
nan | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['0.0 | Unknown', '0.0 | 0.0', '0.0 | 1.0', '10.0', '0.5 | Unknown | Unknown', 'Unknown >> Unknown >> Unknown >> Unknown | Unknown', 'Unknown', 'Unknown | Unknown', '0.5 | Unknown | Unknown | Unknown', '8.0', 'Unknown >> Unknown | Unknown', '6.0', '0.0 | 6.0', 'Unknown >> Unknown >> Unknown | 12.0', '30.0', 'Unknown >> Unknown >> Unknown >> Unknown', 'Unknown >> Unknown', 'Unknown | Unknown >> 0.66', '1.0', '0.5 | 1000.0 >> 1000.0', '4.0'])))

    deposition_reaction_solutions_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the reaction solutions.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a reaction solution undergoes a temperature program, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons, e.g. 25; 100
- When the temperature of a solution is unknown, state that as ‘nan’
- For reaction steps where no solvents are involved, state the temperature of the gas or the solid if that make sense. Otherwise state this as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume an undetermined room temperature to be 25
Example
25
100; 50 | 25
nan | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown | 300', '', '25', '25 | 25 >> 70 >> 25', '70 >> 25 >> 25', '40; ', '500 | 25', '70 | 25 >> 70', 'Unknown | Unknown >> Unknown | Unknown >> 70', '70', '40;  >> 40; ', '25 >> 70 >> 25 >> 25 | 25', '80', '25 | 25', 'Unknown', 'Unknown | Unknown', '100', '25 | 25 >> 25', '25 | 25 >> 70', '77', '25 | Unknown | Unknown', '70 | Unknown', 'Unknown >> Unknown | Unknown', 'Unknown >> 220 >> Unknown', 'Unknown | Unknown >> 70', '75', 'Unknown | 200', '25 | 25 >> 90', '25 | 80', '60', ' | 40; ', '25 | 70; 25', '150 | Unknown', '25 >> 70', 'Unknown >> 80', '60 | Unknown', '100 | 25', 'Unknown | 105', 'Unknown >> 80; 450 >> 150 >> 600 | Unknown', 'Unknown >> 80; 450', '25 | Unknown | Unknown | Unknown', 'Unknown | 450', '450', ' | 40;  >> 40; ', '230 | Unknown', '25 >> 70 | 25', '25 >> 70 | 25 | 25', 'Unknown >> 80; 450 >> 150 >> 600', '200 | Unknown', '25 >> 70 | Unknown', '450 | Unknown', '25 | Unknown'])))

    deposition_substrate_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the substrate.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the substrates (i.e. the last deposited layer) associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The temperature of the substrate refers to the temperature when the deposition of the layer is occurring.
- If a substrate undergoes a temperature program before the deposition, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- When the temperature of a substrate is not known, state that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume that an undetermined room temperature is 25
Example
125; 325; 375; 450 | 25 >> 25
100
nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[' | 60', '450 >> 70 | 25', '25', '', '25 | 25 >> 70 >> 25', '300', '70 >> 25 >> 25', '60 | 25', '500 | 25', '70 | 25 >> 70', '70', '25 >> 70 >> 25 >> 25 | 25', '80', '25 | 25', '450 | 25 >> 25', 'Unknown', 'Unknown | Unknown', '100', '500 | 25 >> 25', '25 | 25 >> 70', '500', '450 | 95', '150', '77', '70 | Unknown', '450 | 25', '450 | 25 | 25', 'Unknown >> 220 >> Unknown', '25 | 25 | 25 | 25', '60', '25 | 25 >> 90', '70 | 25', '25 >> 70', '90', 'Unknown | Unknown >> Unknown | Unknown >> Unknown', '100 | 25', ' | 90 >> 90', '300 >> Unknown | Unknown', '450 | Unknown | Unknown', '450', 'Unknown | Unknown >> Unknown', '25 | 25 | 25', ' | 90', '90 >> 90', '455 >> Unknown', '500; 25', '455 | 25', '500 | 25 | 25', '455 | Unknown', 'Unknown | 120', '15 | 15', '25 | Unknown'])))

    deposition_thermal_annealing_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperatures of the thermal annealing program associated with depositing the layers
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing temperatures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- If no thermal annealing is occurring after the deposition of a layer, state that by stating the room temperature (assumed to 25°C if not further specified)
- If the thermal annealing program is not known, state that by ‘nan’
Example
450 | 125; 325; 375; 450 >> 125; 325; 375; 450
50 | 25
nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['80 | Unknown', '100 >> 500 | 70', '25', ' | 60', '125 >> 500 | 500', '', '25 | 150 >> 500', '450 | 510', '500 >> 500 >> 500 | 500 >> 500 >> 500 >> 500 >> 500 >> 500 >> 500', '500 | 25', '80.0 | Unknown', '180', '135; 500 | 500', '400', '450 | 95; 350', 'Unknown | 125; 325; 375; 450; 500', 'Unknown | Unknown', '130', '450 | 150', '450 | 125; 375; 450 >> 450', '135; 500', '120 | 450', '125; 500 | Unknown', 'Unknown | 100', '160; 500', '150 >> 180', '25 | 80', '75', '500 | 500 >> 500', '420', '25 | 150', '125 >> 500 | 125 >> 500', '100 | 25', '450 | 100 >> 500', '500 | 70; 500', '450; 500 | 20 >> 500 >> 20', '500 | 150; 500', '455 | 100 >> 500', '90 >> 90', '100; 180', '400 | 400', '125 | 500 >> 25; 500', '501', '140', 'Unknown | 500', '130 | 70', '410', '450 | 125; 250; 350; 450; 500', '480 | 500', '160', '125 >> 500 | 125 >> 500 | 125 >> 500', ' | 90 >> 90', '100 >> 500 | 100 >> 500', '550', '450 | 100 >> 450', '450 | 25 | 500', '500 | 550', '450 | 125; 500', '500 | 70 >> 70 | 500 >> 500', '150 >> 500', '510 >> 70 >> 510 | 510', '25 | 325; 375; 450; 500', '70', '125 >> 450 | Unknown', '150 | 500 | 500 | 500', '550 | 550 >> 550', '500 | 125; 500', '500 | 550 >> 500', '200 | 25', '500 | 450 >> 450', 'Unknown | 125 >> 500', '100', '500 | 550 | Unknown >> Unknown >> Unknown >> 450', '100 >> 500 | 100 >> 125 >> 325 >> 375 >> 450 >> 500', '455 >> 100; 500', '500', '150 | 500', '90 | 50', '500 >> 70 >> 500 | 500', '165', '125 | 125; 500 | 25', '500 | Unknown', '125; 125', '100; 150; 185', '125; 150 >> 150; 190', '20', '150 >> 550', '25 >> 125; 500 | Unknown', '450 | 450 >> 150', '120 | 120', '450 | 120; 500', '120 >> 500 | 120 >> 500', '150 | 25', 'Unknown | 180', '300 >> 500 | 550', '60 | Unknown', '120 >> 500', '25 | 25 | 25', '250 | 250 | 250', '150 >> 290 >> 200 | 80', '125 | 125; 550 >> 25; 500', '550 | 300', '450 | 500 | 500', '120 >> 500 | 500', '125 | 500 >> 25 >> 500', '501 | 450', '25 >> 25 >> 500', '125 >> 500 >> 500', '125 >> 450 | 500', '500 | 70', '100; 185', '120 | 500', '455 | 100; 500', '25 | 25', '550 | 150', 'Unknown', '125 >> 450 | 125 >> 450', '500 | 125 >> 500', '500 >> 70 | 100; 500 >> 70 | 500', '500 | 500 | 25', '150', '500 | 450', '500 | 500 | 450', '25 | 480', '500 | 500', '500 | 25 | 400', '100 | 100', '125 | 500', '500 >> 500 | 500 | 25', '510 >> 25 >> 25 >> 510 | 510', '150 | 120', '200 | 150', '25 | 125 >> 500', '90', '180 >> 400 >> 450', '500 | 120', '25; 500', '25 >> 500 | 100; 500', '450 | 500 | 190', 'Unknown | 450', '500 | 110 >> 500', '450', '100 | 80', '80; 500 | 500', '150 | 150', '500; 25', '118 | 500 >> 500', '480', '150 >> 290 >> 200', '125 >> 125 >> 125; 500 | 120; 500', '300 | 500 | 400 | 100', '100 | Unknown', '185', '150; 500', '70 >> 180', '450 | 100', '25 | 125 >> 500 | 150 >> 500', '125 | 125; 500', '25 | 0', '100; 100', '165 | 100', '500 | 120 >> 500', '300', '150 | 100', '120 >> 500 | 150', '25 >> 25 >> 100', '450 | 500', '150 >> 500 | 125 >> 500', '25 | 500', '80', '170', '200', 'Unknown | 70; 500', '125 | 100; 550', '500 | 500 >> 450', '450 | 500 >> 450', '450 | 450', '120', '90 | Unknown', '500 >> 500 | Unknown', '110', '500 | 50', '25 | 500 | 500', '450 >> 450', '500 | 500 | 500', '25 | 500 >> 25; 500', '450 | 450 | Unknown', '80.0', '60', '510 | 540', '150 | Unknown', '125 >> 450', '450 | 500 | 400', '410 | 400', ' | 90', '150 | 450 >> 500', '100; 500', '450 | 125 >> 500', '100; 150; 185 | 100', '450 | 450 >> 500', '100; 150; 185 | 25', '450 | 450 | 450'])))

    deposition_thermal_annealing_time = Quantity(
        type=str,
        shape=[],
        description="""
    The time program associated to the thermal annealing program.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing times associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the associated times at those temperatures and separate them with semicolons.
- The annealing times must align in terms of layers¸ reaction steps and annealing temperatures in the previous filed.
- If a time is not known, state that by ‘nan’
- If no thermal annealing is occurring after the deposition of a layer, state that by ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 20 and not 10-30.
Example
30 | 5; 5; 5; 30 >> 5; 5; 5; 30
60 | 1000
nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['5.0 | 30.0', '15.0 | 10.0', '90.0 | 90.0 >> 30.0', '30.0; 30.0 | 30.0', 'Unknown >> Unknown >> 60.0', '5.0 | 5.0; 60.0', '60.0; 60.0; 60.0 | Unknown', 'Unknown | Unknown | 40.0', '60.0 | 60.0', '30.0 | 10.0; 30.0', '5.0 | 5.0; 60.0 | Unknown', '60.0', '5.0', '0.0 | 30.0 | 30.0', '120.0', '45.0 | 5.0; 5.0; 30.0 >> 30.0', '180.0', 'Unknown | Unknown', '45.0', '120.0 | 30.0 >> 30.0', 'Unknown | 2.0', '210.0 >> 60.0 >> 60.0', '5.0 >> 5.0 >> 5.0; 30.0 | 5.0; 30.0', '20.0 >> 60.0 | 30.0 >> 5.0 >> 5.0 >> 5.0 >> 5.0 >> 15.0 >> 15.0', '20.0; 10.0', '10.0 | 180.0', 'Unknown | 5.0; 5.0; 5.0; 5.0', '30.0 | 30.0 | Unknown', '10.0 >> 30.0 | 30.0', '10.0 | 10.0', '30.0 | 30.0 | 1440.0', '2.0', '60.0 | 15.0; 60.0', '40.0 >> 10.0; 30.0', '5.0 | Unknown', 'Unknown >> 20.0 | 30.0', '5.0 >> 30.0', '30.0 | 30.0; Unknown', '15.0 | 15.0', '5.0 | 60.0 >> Unknown >> 60.0', '90.0 | 5.0 >> 5.0 | 90.0 >> 30.0', '5.0 >> 30.0 | 5.0 >> 30.0 | 5.0 >> 30.0', '60.0 | 60.0 | 60.0', '2.0 | 2.0', '4.0', '30.0 | 0.0', '40.0 | 10.0 >> 30.0', '60.0 | 10.0', '53.0 | 35.0', '30.0 | 30.0 | 60.0', '2.0 >> 30.0 >> 30.0', '60.0 | Unknown', '120.0 | 10.0 >> 30.0', '5.0 | 0.0', '10.0', 'Unknown >> 5.0; 30.0 | Unknown', '15.0 >> 30.0', '30.0 | 30.0', '25.0 | 5.0 >> 20.0', '10.0 >> 10.0 >> 10.0 | 5.0', '15.0 | 30.0', '12.0', '30.0 >> 40.0 >> 30.0 | 30.0', '25.0 | 25.0', '5.0 | 5.0; 120.0', 'Unknown | Unknown | Unknown', '10.0 >> 60.0 | 60.0', '30.0 >> Unknown >> Unknown >> 30.0 | 30.0', '5.0 >> 30.0 | Unknown', '10.0 >> 10.0 >> 10.0', 'Unknown | Unknown; 20.0', 'Unknown; Unknown', '5.0; 30.0 | Unknown', '2.0 | 30.0', 'Unknown >> 30.0 | 5.0; 30.0', '5.0 >> 60.0 | 60.0', '40.0 | 10.0; 30.0', '10.0 >> 60.0', '25.0 | 0.0', '0.0 | 0.0 | 0.0', '18.0 | 28.0 >> 28.0', '30.0 >> 30.0 | 30.0 | Unknown', '30.0 | 60.0 | Unknown >> Unknown >> Unknown >> 30.0', 'Unknown | 60.0', '45.0 | 45.0 >> 30.0', '35.0 | 10.0 >> 30.0', '30.0 | 10.0', '60.0; 60.0', '10.0 >> 120.0', '6.0 | 30.0', '5.0; 30.0 >> 5.0; 60.0', '120.0 | 30.0', '30.0 >> 60.0', 'Unknown; 500.0', '0.0 | 0.0', '30.0 | 15.0', '53.0', 'Unknown | 5.0; 5.0; 15.0; 15.0', 'Unknown | 30.0', '5.0 >> 30.0 | 5.0 >> 30.0', 'Unknown', '10.0 | 30.0', '20.0 | 20.0', '30.0 | 60.0 >> 30.0', 'Unknown | 10.0; 30.0', '30.0 | 20.0; 10.0; 10.0; 10.0; 30.0', '5.0 | 30.0 >> Unknown; 30.0', '20.0 | 60.0', '0.0 | 30.0', '5.0; 60.0', '80.0 | Unknown | 110.0', '10.0; 30.0 | 30.0', '30.0', '25.0 | 5.0 >> 30.0 | 5.0 >> 30.0', '30.0 | 40.0 | 40.0', '60.0 >> 60.0 | Unknown', '5.0 | 5.0; 60.0 >> Unknown; 30.0', '5.0 >> 30.0 | 30.0', '30.0 | Unknown', '25.0', '45.0 | 10.0', '100.0 | 15.0; 30.0', 'Unknown | Unknown >> 15.0', '30.0 >> 30.0', '20.0; 30.0', 'Unknown | Unknown; 30.0', '30.0 | 20.0', '10.0 | 10.0 | 10.0', '10.0 >> 30.0 | 10.0 >> 30.0', '10.0 | 30.0 >> 30.0', '30.0 | 30.0 | 30.0', '60.0 | 20.0', '30.0 | 5.0', '30.0 | 10.0 >> 30.0', 'Unknown | 20.0', '240.0 >> 60.0', '1440.0', 'Unknown >> Unknown >> 180.0', '25.0 | 500.0', '30.0 >> 30.0 >> 30.0 | 50.0', '20.0 | 10.0 >> 30.0', '15.0 >> 30.0 | 10.0 >> 30.0', '30.0 | 60.0', '60.0; 60.0; 60.0 | 10.0', '60.0 | 30.0 | 30.0', '420.0 >> 60.0 >> 60.0', '0.0 | 10.0 >> 30.0', '1.0 | 30.0 >> 30.0', 'Unknown | 30.0 >> Unknown; 30.0', '30.0 >> 30.0 | 30.0; 30.0 >> 30.0 | 30.0', '30.0 | 30.0 >> 30.0', '10.0 >> 30.0 | 10.0', '10.0 | Unknown', '10.0 | 10.0; 60.0', '30.0 | 30.0 >> Unknown', '10.0 | 30.0 | 30.0 | 30.0', '12.0 | Unknown', '20.0', '45.0; Unknown', '10.0 | 30.0 | 30.0 | 15.0', '10.0; 30.0', '60.0; 60.0; 60.0', 'Unknown | 15.0 >> 30.0', '300.0 >> 60.0 >> 60.0', '10.0 >> 10.0 >> 10.0 | 10.0 >> 10.0 >> 10.0 >> 10.0 >> 10.0 >> 10.0 >> 10.0', '0.0; 30.0', '15.0'])))

    deposition_thermal_annealing_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere during thermal annealing
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each annealing step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the atmosphere is a mixture of different gases, i.e. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas.
- This is often the same as the atmosphere under which the deposition is occurring, but not always.
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Vacuum | N2
Air | Ar >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['N2', 'Air; Air; Air | N2', 'Ambient | Ambient | Ambient', 'Ambient | N2', 'Air | Air >> Air', 'Air | Air | Vacuum', 'Ambient | Unknown', 'Ambient >> Ambient | Ambient', 'Air; chlorobenzene', 'Unknown', 'Air; Air; Air', 'Air | N2', 'Unknown | Unknown', 'Vacuum >> Vacuum >> Unknown', 'Steam', 'N2 | N2', 'Ambient | Ambient >> Ambient | Ambient >> Ambient', 'Air | Air | Air', 'N2 | N2 | Vacuum', 'Ambient', 'Air | Air', 'Ambient | Ambient | Ambient >> Ambient >> Ambient >> Ambient', 'Unknown | Ambient', 'Ambient | Ambient >> Ambient', 'Air | Air | Air | Air', 'Dry air', 'Vacuum | Vacuum', 'Air', 'Dry air | Dry air', 'Ambient | Ambient', 'Ambient >> Ambient >> Ambient', 'Vacuum', 'Air | Ar', 'Dry air | Vacuum'])))

    storage_time_until_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    The time between the HTL stack is finalised and the next layer is deposited
- If there are uncertainties, only state the best estimate, e.g. write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1.0', '0.25', '0.1', '0.16', '0.3'])))

    storage_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere in which the sample with the finalised HTL stack is stored until the next deposition step.
Example
Air
N2
Vacuum
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Ar', 'N2', 'Unknown', 'Air'])))

    storage_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relive humidity under which the sample with the finalised HTL stack is stored until next deposition step
- If there are uncertainties, only state the best estimate, e.g. write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '40.0', '0.9', '30.0', '20.0'])))

    surface_treatment_before_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    Description of any type of surface treatment or other treatment the sample with the finalised ETL-stack undergoes before the next deposition step.
- If more than one treatment, list the treatments and separate them by a double forward angel bracket (‘ >> ‘)
- If no special treatment, state that as ‘none’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example:
none
Ar plasma
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Water', 'Plasma', 'ZnAl-LDH and thermal annealing', 'Ozone', 'UV', 'UV-Ozone', 'CO2', 'H2', 'He plasma', 'Washed with methanol', 'Wash with IPA', 'O2 plasma', 'Reactive ion etching'])))

    def normalize(self, archive, logger):
        add_solar_cell(archive)
        if self.stack_sequence:
            archive.results.properties.optoelectronic.solar_cell.electron_transport_layer = self.stack_sequence.split(' | ')


class Perovskite(MSection):
    """
    This section contains information about the properties of the absorber layer. It describes
    the `chemical formula`, the `dimensionality`, the `bandgap`,
    or the `list of dopants and additives` that are in the perovskite layer.
    """

    single_crystal = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the cell is based on a perovskite single crystal
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    dimension_0D = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the cell is based on a perovskite quantum dots. Perovskite nanoparticle architectures can also be counted here unless they more have the characteristics of a standard polycrystalline cell
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    dimension_2D = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the cell is based on 2D perovskites, i.e. a layered perovskite with a large A-cation
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    dimension_2D3D_mixture = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the cell is based on a mixture of 2D and 3D perovskites. This is sometimes referred to as reduced dimensional perovskites (but not as reduced as to be a pure 2D perovskite)
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    dimension_3D = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE for standard three-dimensional perovskites with ABC3 structures. TRUE also for the case where the bulk of the perovskite is 3D but where there exist a thin 2D-caping layer
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    dimension_3D_with_2D_capping_layer = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the bulk of the perovskite layer is 3D but there is a top layer with lower dimensionality.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    dimension_list_of_layers = Quantity(
        type=str,
        shape=[],
        description="""
    A list of the perovskite dimensionalities
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- In most cases, there will be only one layer
- For a perovskite that is a mixture of a 2D and a 3D phase, mark this is as2.5
Example
3
3 | 2
0
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '3.0 | 1.0', '3.0 | 0.0', '2.5', '0.0 | 0.0', '2.0 | 3.0 | 2.0', '3.0 | 0.0 | 0.0 | 0.0', '3.0', '2.0 | 3.0', '3.0 | 3.0', '3.0 | 2.0', '3.0 | 0.0 | 0.0', '1.5', '2.0', '3.0 | 2.0 | 0.0', '1.0', '0.0', '3.0 | 0.0 | 0.0 | 0.0 | 0.0', '1.0 | 3.0'])))

    composition_perovskite_ABC3_structure = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the photo-absorber has a perovskite structure
- The typical perovskite has an ABC3 structure and that is clearly a TRUE
- This category is inclusive in the sense that also 2D perovskite analogues should be labelled as TRUE
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    composition_perovskite_inspired_structure = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the photo absorber does not have a perovskite structure. In the literature we sometimes see cells based on non-perovskite photo absorbers, but which claims to be “perovskite inspired” regardless if the crystal structure has any resemblance to the perovskite ABC3 structure or not. This category is for enabling those cells to easily be identified and filtered.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    composition_a_ions = Quantity(
        type=str,
        shape=[],
        description="""
    List of the A-site ions in the perovskite structure
- We have experimented with letting users write the perovskite structure and from that extract ions and coefficients. Due to the multitude of formatting variations, that has not worked out very well, wherefor we now define the perovskite ion by ion.
- List all the A-site ions in alphabetic order and separate them by semicolons
- For ions which labels are three characters or longer, enclose them in parenthesis. That improves readability and simplifies downstream data treatment.
- In case of a layered perovskite structure, separate layers by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- Only include ions that go into the perovskite structure. Ions that only are found in secondary phases, or amorphous grain boundaries, or that disappears during synthesis, should instead be added as dopants/additives in the field dedicated to dopants and additives.
o On example is Rb in MAFAPbBrI-perovskites. As far as we know, Rb does not go into the perovskite structure, even if that was believed to be the case in the beginning, but rather form secondary phases. For MAFAPbBrI-perovskites, Rb should thus not be considered as a A-site cation, but as a dopant/additive.
Example:
MA
FA; MA
Cs; FA; MA
(5-AVA); MA
Cs; FA; MA | (PEA)
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=sorted(['', '(TFEA); Cs; FA; MA', '(PEA) | Cs; FA; MA | (PEA)', 'HA; MA', '(CHMA); MA', 'FA; HA', 'Cs | BA; Cs', 'MA | Cs | Cs | Cs | Cs', '(ALA); BA; MA', '(TEA); MA', '(PGA); BA; MA', 'BA; Cs; FA; MA', 'Cs; FA; MA | (PEA)', 'EA; MA; PEA', 'Cs; FA | Cs', '(BYA); MA', 'Cs; FA; K; MA', 'MA | Cs | Cs', '(PEI); MA', '(PEA); MA | MA', 'MA; PA', '(NMA); MA', '(BZA); (HAD); MA', 'Bi', '(PDA); MA', 'Cu', '(BzDA); Cs; FA; MA', '(TBA); MA', 'Cs; (DMA); MA', '(CPEA); MA', 'FA; MA | (A43)', '(Anyl)', 'BA; Cs; MA', '(HdA)', 'HA', 'FA; MA | DA', '(TMA)', '(CIEA); MA', 'MA | FA', '(HEA); Cs; FA', '(BZA)', '(BdA)', 'BA; FA', '(OdA)', 'K; BA', 'Rb', '(5-AVA); MA', '(ALA); MA', 'Cs; FA; GU', 'DI; FA', '(iPA)', 'FA; MA | (MIC3)', '(5-AVAI); Cs; FA', 'FA; MA | (MIC2)', 'Cs; FA; PDA', 'Cs; FA | Cs; FA', 'Cs; MA', '(mF1PEA); MA', 'Cs; FA; MA | PA', '(PDMA)', 'Cs; FA; MA | (FPEA)', 'EA; FA', 'MA | (BEA)', 'FA | (ODA)', '(ImEA)', 'MA | BA', 'BA; MA', 'Cs; FA; MA | (mFPEA)', '(C6H4NH2)', 'MA | Cs | Cs | Cs', '(IEA); MA', 'FA; K; MA', '(PMA)', 'Cs; FA; MA | (pFPEA)', 'IM', 'Cs; HA', 'FA; PN', 'Cs; FA; GU; MA', '(ThMA); MA', 'FA', '(DPA); MA; PA; PA', 'FA; MA | (C8H17NH3)', 'AN; FA; MA', '(DMA); MA', 'Cs; FA; MA | (oFPEA)', '(3AMP); FA; MA', 'Bi; La', 'Cs; MA; FA | BA', 'AN; Cs; MA', '(BZA); MA', '(Ace); MA', 'MA | (PPEA)', 'FA; MA | (C4H9NH3)', '((CH3)3S)', 'MA | MA', '(4ApyH)', '(Br-PEA); MA', 'FA; GU', 'IA; MA', '(DMA)', 'BA; GA; MA', 'FA | Cs', '(PBA); MA', 'Aa; MA', 'Ag', '(PMA); MA', '(DAP)', '(MTEA); MA', 'MA | Cs', 'AN; MA', 'BU; FA', '(CHMA); Cs; MA', 'Cs | Cs', 'Cs', '(PDMA); MA', 'FA; MA | PEA', 'MA | (MIC1)', '(PEA); FA; MA', '(3AMPY); MA', 'Cs; EA; FA', '(PTA); MA', '(PEA) | MA', 'MA | CA', '(PEA); Cs', '(N-EtPy)', 'MA | (EU-pyP)', 'GU; MA', '(PEA); BA; FA', 'Cs; FA; MA | BA', '(PEA); (F5PEA); Cs; FA; MA', 'Cs; FA; nan', '(n-C3H7NH3)', '(PGA); MA', 'FA; MA | (PEA)', '(PEA); BA; MA', '(PEA); FA', '(F-PEA); MA', 'Ba; K', 'Cs; Rb', 'Cs; FA; MA | Cs', 'Ca; MA', 'BA', 'Cs; Li', '(iso-BA); MA', '(PyrEA)', 'Cs; FA; MA | (CH3)3S', 'Cs; FA; Rb', 'BA; Cs', '(BEA); MA', 'Cs; FA; MA | (EPA)', '(NEA); BA; MA', 'FA; MA | (HTAB); FA; MA', 'FA | EDA', '(1.3-Pr(NH3)2)', 'FA; MA | BA; FA', '(BEA); Cs; FA; MA', '(PEI)', 'MA | (BI)', 'MA | (PEA)', 'MA | (C4H9N2H6)', '(NH4)', 'K', '(ThFA); MA', 'Ag; Cs; Rb', 'EA', 'EDA; FA', 'FA; Rb', 'Cs; FA; MA | (FEA)', '(TBA); Cs; FA; MA', 'Cs; Ag', 'Cs | MA', '(NH4); FA; MA', '(Anyl); MA', '(PEA); MA', 'Cs; Na', 'IM; MA', '(4AMP); MA', '(F3EA); BA; MA', 'BA; FA; MA; PMA', '(EDA); FA; MA', 'MA | (PPA)', 'HDA', 'Sr', '(pF1PEA); MA', 'MA | BA; MA', 'BE; FA', '(4AMPY); MA', 'FA; MA | (C6H13NH3)', 'Cs; FA | (PA)', 'Cs; FA | (PEA)', 'GA; MA', 'FA; MA | TA', '(oF1PEA); MA', 'EA; MA; NEA', '(BDA); MA', 'BA; Cs; FA', 'Cs; FA; MA', '(H-PEA); MA', 'Cs; K', '(PEA); Cs; FA; MA', 'Cs; FA | (PMA)', 'GU', '(PBA); BA; MA', 'FA | (PEA)', '(3AMP); MA', '(PEA); (F5PEA)', 'BA; FA; MA', '(ThMA); FA', 'BDA; Cs', '(BIM); MA', '(CH3ND3)', '(GABA); MA', 'FA; MA | (FEA)', 'La', 'FA; OA', 'FA; MA | (NH4); FA', '(APMim)', '(F5PEA); Cs; FA; MA', 'Cs; FA; GA', 'FA; MA | (MIC1)', 'Cs; FA; MA | (A43)', 'Ag; Cs', 'Cs | FA', '(PDMA); FA', 'FA; MA', 'DA; FA', '(PEA); Cs; MA', 'FA; MA | OA', 'Cs; FA; MA | NMABr', 'MA', 'FA; TN', 'Cs; FA; MA | HA', '(f-PEA)', 'Cs; FA; MA; Rb', 'Bn', '(Ada); FA', 'Ca; FA; MA', '(AVA) | MA | (BI)', '(PEA); Cs; FA', 'Cs; FA; MA | (PEI)', '(6-ACA); MA', '(DAT); MA', '(5-AVA); FA', '(PEA) | MA | (PEA)', '(BDA)', '(PyEA); MA', '(F5PEA)', '(THM); MA', 'MA | (MIC3)', '(PDA); Cs; FA', '(5-AVA); Cs; MA', 'FA; GU; MA', 'FA; MA | BA', 'Cs; FA', '(Cl-PEA); MA', '(AVA); MA', '(PMA); FA', 'PA', 'EA; MA', 'FA; PR', '(AVA) | MA', '(4FPEA); MA', '(PEA)', 'MA | (MIC2)', 'BA; GU; MA']))))

    composition_a_ions_coefficients = Quantity(
        type=str,
        shape=[],
        description="""
    A list of the perovskite coefficients for the A-site ions
- The list of coefficients must line up with the list of the A-site ions
- If a coefficient is unknown, state that with an ‘x’
- If there are uncertainties in the coefficient, only state the best estimate, e.g. write 0.4 and not 0.3-0.5.
- A common notation is ‘1-x’. Write that as x
- If the coefficients are not known precisely, a good guess is worth more than to state that we have absolutely no idea.
Examples:
1
0.83; 0.17
0.05; 0.79; 0.16
1.5; 0.5
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['1', '0.2', '0.83; 0.17'])))

    composition_b_ions = Quantity(
        type=str,
        shape=[],
        description="""
    List of the B-site ions in the perovskite structure
- We have experimented with letting users write the perovskite structure and from that extract ions and coefficients. Due to the multitude of formatting variations, that has not worked out very well, wherefor we now define the perovskite ion by ion.
- List all the B-site ions in alphabetic order and separate them by semicolons
- In case of a layered perovskite structure, separate layers by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- Only include ions that go into the perovskite structure. Ions that only are found in secondary phases, or amorphous grain boundaries, or that disappears during synthesis, should instead be added as dopants/additives in the field dedicated to dopants and additives.
Example:
Pb
Sn
Pb; Sn
Bi
Pb | Pb
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Pb; Sn | Pb; Sn', 'Ni; Pb', 'Al', 'Pb; Sm', 'Cu; Sb', 'Ag; Bi', 'Pb | Pb | Pb', 'Pb | Pb | Pb | Pb | Pb', 'Bi; Te', 'Mn; Pb', 'Pb; Zn', 'Y', 'Au', 'Pb; Sr', 'Fe', 'Sn', 'Cu', 'Bi', 'Hg; Pb', 'Ca; Pb', 'Sn | Sn', 'Cu; Pb; Sn', 'Ge; Pb', 'Pb | Pb', 'Ni', 'Bi; Pb', 'Cu; Pb', 'Sb', 'Mg; Pb', 'Hg', 'Co; Pb', 'Ge; Sn', 'Pb; Tb', 'Pb; Sn', 'Pb', 'Pb; Sb', 'Sb; Sn', 'Pb | Ba; Pb', 'Mn', 'Sn | Pb', 'Fe; Pb', 'Ti', 'In; Pb', 'La; Pb', 'Nb; Ni', 'Pb | Pb | Pb | Pb', 'Ge; Sb', 'Bi; Fe; Cr', 'Bi; Sb', 'Ge', 'Ba; Pb', 'Eu; Pb'])))

    composition_b_ions_coefficients = Quantity(
        type=str,
        shape=[],
        description="""
    A list of the perovskite coefficients for the B-site ions
- The list of coefficients must line up with the list of the B-site ions
- If a coefficient is unknown, mark that with an ‘x’
- If there are uncertainties in the coefficient, only state the best estimate, e.g. write 0.4 and not 0.3-0.5.
- A common notation is ‘1-x’. Write that as x
- If the coefficients are not known precisely, a good guess is worth more than to state that we have absolutely no idea.
Examples:
1
0.83; 0.17
x; x
0.5; 0.5 | 1
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '7.8', '0.008; 0.992', '0.625; 0.375', '9', '0.6; 0.6', '0.1; 0.9', '0.875; 0.125', '1', '0.05; 0.85; 0.1', '2', '0.85', '0.6', '0.85; 0.15', '0.93', '1.2; 0.8', '0.93; 0.07', '41', '0.9999; 0.0001', '2.4; 1.8', '0.98', '0.5; 0.5', '0.45 | 9', '0.003; 0.997', '0.97; 0.03', '0.07; 0.93', '11', '1 | 1; 1', '0.95; 0.1', '0.995', '2.6', '1 | 3', '0.025; 0.975', '0.2; 0.8', '0.016; 0.984', '0.748; 0.252', '0; 0.19', '4.0', '1 | 1 | 1 | 1', '0.0118; 0.9882', '4.8; 3.6', '0.99', '0.6; 0.4', '0.02; 0.98', '0.031; 0.969', '0.4; 0.6', '0.875', '0.94; 0.06', '0.99; 0.01', '0.05', '0.99999; 0.00001', '1.4; 0.6', '0.09; 0.91', '0.664; 0.336', '0.54', '100', '0.999; 0.001', '0.075; 0.925', '7', '0.25; 0.75', '20', '0.96; 0.04', '8', '0.15; 0.85', '0.5; 0.500', '0.05; 0.95', '0.063; 0.937', '0.57', '1 | 1 | 1', '1 | 2', '0.20; 0.80', '1.0', '6.1', '0.05; 0.9; 0.05', '11.2', '1 | 1', '0.01; 0.99', '4', '0.05; 0.8; 0.15', '10', '0.997; 0.003', '29', '3', '0.916; 0.084', '0.014; 0.986', '1.8; 0.2', '3.14', '0.8; 0.2', '0.95; 0.05', '2.7', '6', '0.10; 0.90', '40', '0.50; 0.50', '1; 0.6', '0.7; 0.3', '0.08; 0.92', '0.25', '0.4; 0.6 | 0.4; 0.6', '23', '0.84; 0.84', '0.005; 0.995', '0.98; 0.02', '3; 2.4', '0.832; 0.168', '3; 1', '1.8; 1.2', '1; 3', '0.995; 0.005', '61', '1.9; 0.1', '0.88; 0.12', '0.95', 'x', '2.2', '0.7; 0.255', '2; 1; 1', '0.06; 0.94', '0.38; 0.62', '0.97', '0.03; 0.97', '5', '0.125; 0.875', '1; 1', '0.9; 0.1', '0.66; 0.33', '80', '0.9', '0.375; 0.625', '0.58; 0.42', '12.9', '0.997', '1.1', '60', '0.3; 0.7', '30', '0.04; 0.96', '0.92; 0.08', '1.6; 0.4', '9.5', '0.75; 0.25', '0.45', '4 | 1', '0.0094; 0.9906', '0.37; 0.6255', '0.93; 0.03', '0.65; 0.35', '1 | 1 | 1 | 1 | 1', '4.4'])))

    composition_c_ions = Quantity(
        type=str,
        shape=[],
        description="""
    List of the C-site ions in the perovskite structure
- We have experimented with letting users write the perovskite structure and from that extract ions and coefficients. Due to the multitude of formatting variations, that has not worked out very well, wherefor we now define the perovskite ion by ion.
- List all the A-site ions in alphabetic order and separate them by semicolons
- For ions which labels are three characters or longer, enclose them in parenthesis. That improves readability and simplifies downstream data treatment.
- In case of a layered perovskite structure, separate layers by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- Only include ions that go into the perovskite structure. Ions that only are found in secondary phases, or amorphous grain boundaries, or that disappears during synthesis, should instead be added as dopants/additives in the field dedicated to dopants and additives.
o One example is chloride in MAPbI3. As far as we know, Cl does not go into the perovskite structure even if that was believed to be the case in the beginning. For MAPbI3 Cl should thus not be considered as a C-site cation, but as a dopant/additive.
Example:
I
Br; I
Br
Br; I| I
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Br; I | Br; I', '', '(BF4); I', 'O', 'I | Br; I | I', 'S', 'Br | Br; I', 'Cl', 'Br', 'I | Br', 'I | I; Br', 'Br | I', 'Br; I | I', 'Br; F; I', 'Br; I', 'I | I | Br; I | Br; I', '(PF6); PF6', '(SCN); I', 'I | I', 'F; I', 'Cl; I', 'I', 'I; (SCN)', 'I | I | I', 'Cl; I | Cl', 'Br; Cl', 'I; SCN', 'Br | Br', 'I | I | Br; I', 'I | I | Br; I | Br; I | Br; I', 'I | I; (PF6)', 'Br; Cl; I', 'Br | Br; Cl', 'I | Br; I'])))

    composition_c_ions_coefficients = Quantity(
        type=str,
        shape=[],
        description="""
    A list of the perovskite coefficients for the C-site ions
- The list of coefficients must line up with the list of the C-site ions
- If a coefficient is unknown, mark that with an ‘x’
- If there are uncertainties in the coefficient, only state the best estimate, e.g. write 0.4 and not 0.3-0.5.
- A common notation is ‘1-x’. Write that as x
- If the coefficients are not known precisely, a good guess is worth more than to state that we have absolutely no idea.
Examples:
3
0.51; 2.49
0.51; 2.49 | x
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    composition_none_stoichiometry_components_in_excess = Quantity(
        type=str,
        shape=[],
        description="""
    Components that are in excess in the perovskite synthesis. E.g. to form stoichiometric MAPbI3, PbI2 and MAI are mixed in the proportions 1:1. If one of them are in excess compared to the other, then that component is considered to be in excess. This information can be inferred from data entered on the concentration for all reaction solutions but this gives a convenient shorthand filtering option.
- If more than one component is in excess, order them in alphabetic order and separate them by semicolons.
- It there are no components that are in excess, write Stoichiometric
Examples:
PbI2
MAI
Stoichiometric
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'SnI2', 'FA', 'Cl', 'SnF2; Sn', 'BiBr3', 'SnCl2', 'PbI2; SnI2', 'FAI', 'I2', 'Sn', 'MAI; PEAI; DMF', 'CH3ND3I', 'Stoichiometric | Stoichiometric', 'PbAc2', 'MAI; FAI', 'PbBr2; PbI2', 'MAI; PEAI', 'RbI', 'MA', 'FAI; MABr', 'MABr; PbI2', 'NH4Cl', 'HCl', '5-AVAI', 'I', 'Pb', 'Stoichiometric', 'PbI2 | nan', 'SrI2', 'CsI', 'BiI3', 'MAI', 'AgI', 'CsBr', 'MABr', 'PbCl2', 'CsI; MACl', 'MACl', 'SbI3', 'PbI2; PbBr2', 'Stoichiometric | nan', 'PbI', 'SnBr2', 'PbBr2', 'PbI2'])))

    composition_short_form = Quantity(
        type=str,
        shape=[],
        description="""
    nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['FAMAPbBrI | (MIC3)PbI', 'CsBiPbI', 'BAFAMAPbI', 'FAPbI | (ODA)PbI', 'CsFAPbI | CsPbI', '(PMA)FAPbI', 'CsRbSnI', 'MAPbI | (BEA)PbI', '(PEA)FAMAPbI', 'BAPbI', 'CsPbI | CsPbI', 'AgCsBiI', 'MAPbI | MABaPbI', 'MAPbI | (MIC1)PbI', 'CsPbBr | CsPbI', '(PEA)CsFAPbI', 'EAMAPEAPbI', 'nanBiI', 'GUPb(SCN)I', 'MAMnPbI', 'MAPbISCN', '(5-AVA)MAPbI', 'IMPbI', '(mF1PEA)MAPbI', 'MAPb(SCN)I', 'FAGeSbCl', '(BEA)CsFAMAPbBrI', 'nanPbI', 'CsEAFAPbBrI', 'CsPbBrI | FAPbBrI', 'BAMASnI', 'CsCuPbBr', 'CsPbBr | MAPbI', '(1.3-Pr(NH3)2)PbI', 'FAMAPbBrI | (FEA)PbI', 'MAPbI | (PPEA)PbI', 'FAPbSnI', 'BACsFAMAPbI', 'CsMgPbBr', 'CsHAPbI', '(F-PEA)MAPbI', '(C6H4NH2)CuClI', 'CsPbBr | FAPbBr', '(DAT)MAPbI', '(EDA)FAMAPbI', '(NH4)SbBrI', 'CsPbBrI | BACsPbBrI', 'MAMgPbI', 'CsNaBiI', '(IEA)MAPbI', '(PGA)BAMAPbI', 'BACsFAPbSnI', 'FASnI', 'MASnBr', 'MAPbI | FAPbBrI', 'HAPbI', 'CsPbBrI | CsPbI', 'EAMANEAPbI', 'CsPbSrBr', 'CsFAMAPbI', '(PEA)CsFAMAPbBrI', '(APMim)Pb(PF6)PF6', '(5-AVA)CsMAPbI', '(oF1PEA)MAPbI', 'SrTiO', 'EAMAPbI', '(PMA)MAPbI', 'FATNSnI', 'CuBiI', 'nanBiO', '(n-C3H7NH3)PbCl', '(PBA)BAMAPbI', 'CsFAPbSnI', '(NH4)SbI', 'CsFAPbBrI', 'HDABiI', 'CsRbPbBr', '(NH4)SbBr', 'AgCsBiBr', 'MAPbBr', '(PDA)CsFAPbI', 'CsFAPbBr', 'CsBaPbI', 'BACsPbI', 'MAPbSrI', 'HAMASnI', '(F5PEA)(PEA)CsFAMAPbBrI', '(4AMPY)MAPbI', 'MAPbI | (MIC2)PbI', '(OdA)PbI', 'MAPbI | (C4H9N2H6)PbI', 'MAEuPbI', 'CsPbSnI', 'FAPbI | EDAPbI', 'RbPbI', 'BDACsPbBrI', '(Anyl)MAPbI', 'EAFASnI', '(4ApyH)SbI', '(PEA)BAMAPbI', '(TMA)SnI', 'CsPbZnBr', 'AgBiI', 'CsPbSnBrI', '(ImEA)PbI', 'FAPbBrI', 'MAPbI | MAPbI', 'FAMAPbI | (A43)PbI', 'MANiPbI', 'BAKBiTeO', 'MASnI', '(BDA)PbI', 'FAMAPbBrI | OAPbI', 'MAHgI', '(F5PEA)CsFAMAPbBrI', '(4FPEA)MAPbI', 'MAPbI | (EU-pyP)PbI', 'CsMAPbSnBrI', 'CsFAPbBrI | (PEA)PbBrI', '(PEA)PbI | CsFAMAPbBrI | (PEA)PbI', '(iso-BA)MAPbI', 'MAPbI | MAPbBrI', 'CsFAPbI | CsFAPbI(PF6)', 'MAPbBrCl', 'MAPbI | BAMAPbI', 'FAMAPbBrI | (C4H9NH3)PbI', 'EDAFASnI', 'CsNiPbBr', 'MAInPbI', '((CH3)3S)SnClI', 'CsBaPbBrI', '(C6H4NH2)CuBrI', 'FAPbI | CsPbI', 'MACuPbBrI', '(Br-PEA)MAPbBrI', 'CsFAMAPbBr', 'FAPbCl', 'MABaPbI', '(AVA)PbI | MAPbI | (BI)PbI', 'BAGAMAPbI', '(PEI)MAPbI', '(PEA)FAMASnI', 'MAPbI | MAPbBr', '(PDMA)MAPbI', 'FAMASnI', 'CsPb(SCN)I', 'CsFAMAPbBrI | (A43)PbI', 'CsFAPbBrI | (PA)PbBrI', 'EAMAPbBr', 'FASnBrI', '(PEA)MAPbI', '(HEA)CsFAPbBrI', 'MACuPbSnBrI', 'IAMAPbI', '(ThMA)MAPbI', 'CsPbBr | FAPbBrCl', 'MAAuBr', 'BACsFAPbBrI', 'CsFAMAPbBrI | HAPbI', 'CsFARbPbI', 'CaMAPbBrI', 'CsMAPbI', 'FASnI | (PEA)SnI', 'FAPbBrCl', 'MACuBrCl', 'FAHAPbI', '(PEA)MASnI', 'MACoPbI', '(3AMP)MAPbI', '(PBA)MAPbI', 'MAPbSnBr', 'CsFAMAPbBrI | CsPbI', 'FAMAPbI', 'FAMAPbSnI', '(F5PEA)(PEA)PbI', 'MAPbI | (BI)PbI', 'MAPAPbI', 'AgBi(SCN)I', 'FAPbBr', '(6-ACA)MAPbI', 'CsMAPbSnClI', '(PEI)PbI', 'FAMAPbSnBrI', 'MANiClI', 'MAGeBrI', 'MAPbClI', 'CsTiBr', 'CsSnI', 'MAPbBrI', '(CPEA)MAPbI', 'FAMAGeSnI', 'MAPbI | BAPbI', 'BiLaFeO', '(PEA)MAPbClI', 'BiFeO', 'FAMAPbBrI', '(PDMA)PbI', 'CsGeSnI', 'FAMAPbBr', 'FAKMAPbBrI', 'MASbSnI', 'FAMAGePbBrI', '(5-AVA)FASnI', 'CsFAMAPbI | (FEA)PbI', 'CsFAMAPbBrI | (pFPEA)PbI', 'FAMAPbBrI | DAPbI', '(4ApyH)BiSbI', '(DMA)MAPbI', 'CsFAMAPbBrI | (EPA)PbI', 'MASbI', 'IMMAPbI', '(BDA)MAPbI', '(F5PEA)PbI', 'BAFAPbI', 'MAGeI', 'LaYS', 'MAPbI | (PPA)PbI', 'CsPbBr | FAPbBrI', 'FAMAPbBrI | PEAPbI', 'nannannan', 'CsBiI', '(PEA)PbI', 'MACaPbI', 'nanSnI', 'CsFAMAPbBrI | CsPbBrI', 'FAMAPbBrI | BAFAPbI', '(TEA)MAPbI', 'MAPbZnI', 'CsSbI', 'CsFAGAPbI', 'FAMAPbI | (NH4)FAPbI', 'BACsMAPbI', 'CsFASnI', 'BnSnI', 'HAMAPbI', 'FAPbI | (PEA)PbI', 'FAPbBrClI', 'CsPbBr | CsPbBrI', '(NMA)MAPbI', 'CsGePbBrI', 'CsPbBrI', 'FAMAPbSnI | (PEA)PbSnI', '(3AMPY)MAPbI', '(PGA)MAPbI', 'CsPbBr', 'BAFASnI', 'EAPbI', 'CsFAPDAPbI', '(N-EtPy)SbBr', '(PDMA)FAPbI', 'BAFAPbClI', 'MASnI | MAPbI', 'CsFAGAPbBrI', 'CsCaPbBr', 'MAPbI | CsPbBr', 'CsNaPbBr', 'AgCsSbI', 'CsLiPbBr', '(pF1PEA)MAPbI', '(NH4)FAMAPbBrI', 'BAMAPbI', 'MAPbI', 'MAPbI | CsPbI | CsPbBrI', '(PyEA)MAPbI', '(PEA)BAFASnI', '(BZA)PbI', 'CsFAPbI | CsFAPbI', 'MACuPbI', 'DIFAPbI', 'CsPbBr | CsPbBr', 'CsLaPbBrI', 'CsFAMAPbI | NMABrPbIBr', 'CsPbZnBrI', 'MASnBrI', 'CsPbBrI | CsPbBrI', '(ThMA)FAPbI', 'ANCsMAPbI', 'CsFAGUPbI', '(PEA)FAMASnBrI', 'HASnI', 'CsSnBr', '(CHMA)MAPbI', '(TFEA)CsFAMAPbBrI', 'CsFAMAPbBrI | (FPEA)PbI', '(PEA)FASnI', 'CsFAMAPbBrI', 'CsFAKMAPbBrI', 'MAPb(BF4)I', '(BZA)(HAD)MAPbI', '(BIM)MAPbI', '(ALA)MAPbI', 'CsPbSmBr', '(PEA)CsPbI', 'MAPbI | (MIC3)PbI', 'GUPbI', 'CsFAPbI', 'FABiPbI', '(BEA)MAPbI', '(PEA)CsMAPbI', '((CH3)3S)SnBrI', '(TBA)MAPbI', '(Anyl)PbI', 'nanCuSbI', 'MAAlCl', 'CsBaPbBr', 'CsGeI', '(GABA)MAPbI', '(Ada)FAPbI', '(CH3ND3)PbI', 'BACsFAMAPbBrI', 'MAHgPbI', 'CsMAPbBrI', '(BdA)PbI', 'CsFAMAPbBrI | (mFPEA)PbI', 'BUFAPbI', 'CsPbBrFI', 'CsRbPbBrI', 'AgCsRbBiBr', '(CHMA)CsMAPbI', '(BzDA)CsFAMAPbBrI', 'FAMAPbBrI | BAPbI', 'FAMAPbBrI | (HTAB)FAMAPbBrI', 'ANMAPbI', '(PEA)MAPbClI | MAPbCl', 'MAPbI | CAPbI', '(PEA)FAPbI', 'CsPbI', '(BYA)MAPbI', 'FAMAPbBrI | (MIC1)PbI', 'PAPbI', 'CsFAGUPbBrI', '(APMim)PbBrI', 'MAPbSnBrI', 'FAMAPbBrI | (C8H17NH3)PbI', 'MASnBrClI', '(Cl-PEA)MAPbClI', 'nanBiCrFeO', '(5-AVAI)CsFAPbI', '(F3EA)BAMAPbI', '(BZA)MAPbI', 'FAPRPbI', '(f-PEA)PbI', 'MAPbI | CsPbI | CsPbBrI | CsPbBrI', 'FABiI', '(DMA)PbI', '(PEA)MAPbBrI', 'CsMAPb(SCN)I', 'CsFAMASnBrI', 'BAFAPbBrI', 'MAPbSbBrI', 'MAPbI | (PEA)PbI', 'FAMAPbBrI | (MIC2)PbI', 'CsMAPbSnI', '(H-PEA)MAPbI', 'CsFAMAPbBrI | BAPbBrI', 'CsFAMAPbBrI | BAPbI', 'RbSbI', 'BAGUMAPbI', 'FARbPbI', 'MAPbI | CsPbI | CsPbBrI | CsPbBrI | CsPbBrI', 'AaMAPbI', 'CsFAMAPbBrI | (PEI)PbI', 'CsFAKMAGePbBrI', 'CsFAMAPbBrI | (CH3)3SPbI', 'MASnCl', 'KSbI', 'BaKNbNiO', 'CsFAPbBrI | (PMA)PbBrI', '(CIEA)MAPbI', '(PEA)CsPbBrI', 'FAMAPbI | TAPbI', 'CsFAPbI | (PEA)PbI', '(iPA)PbI', '(4AMP)MAPbI', 'AgBiBr', 'CsSnBrI', 'CsEuPbBrI', 'AgCsBiSbBr', '(ThFA)MAPbI', 'FAOASnI', '(DMA)CsMAPbI', 'MAMnI', 'CsKPbBr', 'ANFAMAPbI', 'CsFAMAPbSnBrI', 'CsFAPbBrClI', 'CsFAMAPbBrI | PAPbI', '(HdA)PbI', 'CsFAMAPbSnI', '(AVA)PbI | MAPbI', 'FAPbClI', '(ALA)BAMAPbI', 'CsAgBiBr', 'GUSnI', '(TBA)CsFAMAPbBrI', 'CsPbI | FAPbI', '(Ace)MAPbI', 'FAPbI', '(PDA)MAPbI', 'CsPbTbBr', '(PEA)PbI | MAPbI', 'FAMAPbBrI | (C6H13NH3)PbI', 'MAPbSbI', 'GAMAPbI', 'CsFAGUMAPbBrI', 'BEFAPbI', 'CsFAMAPbBrI | (PEA)PbI', '(3AMP)FAMAPbI', '(PEA)CsPbBr', '(AVA)MAPbI', '(PyrEA)PbI', 'FAGUSnI', '(Cl-PEA)MAPbI', '(PTA)MAPbI', '((CH3)3S)SnI', 'MAFePbI', 'DAFASnI', 'BAMAPbSnI', 'MAPbSnI', '(PEA)PbI | MAPbI | (PEA)PbI', 'MABiSbI', 'FAGUMAPbI', '(MTEA)MAPbI', 'CsMASnI', 'CsFAMARbPbI', '(DPA)MAPAPAPbI', 'FAPNSnI', 'CsFAMARbPbBrI', '(DAP)PbI', 'FASnBr', '(NEA)BAMAPbI', '(PEA)FAMAPbBrI', '(PMA)CuBr', 'GUMAPbI', 'MABiI', 'CaFAMAPbBrI', 'CsFAMAPbBrI | (oFPEA)PbI', 'MAPbI | CsPbI', '(THM)MAPbI', 'CsPbSrBrI', 'MASnFI', 'CsFAnanPbI', 'MAPbCl'])))

    composition_long_form = Quantity(
        type=str,
        shape=[],
        description="""
    nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Cs0.2FA0.6MA0.2PbBr0.095I0.905', '(PEA)2FA0.5MA3.5Pb5I16', 'Cs0.2FA0.8Pb0.75Sn0.25I3', '(THM)0.05MA0.95PbI3', 'FA0.026MA0.974PbI3', 'CsPbBrI2 | CsxPbBrI2', '(PEA)0.2BA1.8MA3Pb4I13', 'Cs0.17FA0.83Pb0.995Sn0.005I3', '(PEA)2MA9Pb10I31', 'Cs0.05FA0.79MA0.16Pb0.58Sn0.42Br0.52I2.48', 'FA0.7MA0.3PbBr0.1I2.9', 'FA0.83MA0.13PbBr0.39I2.61', '(NH4)6.8FA0.15MA2.125Pb7.8Br0.45I23.97', 'FA0.75K0.1MA0.15PbBr0.55I2.55', 'Cs0.97Na0.03PbBr3', 'Cs0.2FA0.75MA0.05PbBr0.51I2.49', 'Cs0.09FA0.77MA0.14PbBr0.42I2.58', 'MASnI3', 'FASnI3 | (PEA)2SnI4', 'Cs0.10FA0.83MA0.07PbBr0.51I2.49', 'Cs0.07FA0.725MA0.115PbBr0.45I2.55', 'FA0.85MA0.15PbBr0.21I2.79', 'Ag3Bi1.0Br6', 'Cs0.2MA0.8PbI3', '(PDMA)FA2Pb3I10', '(PEA)2MA59Pb60I181', 'GUPb(SCN)2.2I0.8', 'Cs0.05FA0.79MA0.16Pb0.5Sn0.5I3', 'MAPbI3 | MAPbBrI2', 'Cs0.9Ag3Bi2.6I9', 'MACu0.05Pb0.9Sn0.05Br0.1I2.9', 'FAPbCl3', 'Cs0.10FA0.75MA0.15PbBr0.45I2.55', 'Cs02FA0.8PbBr0.42I2.58', 'CsPbBr3 | FAPbBr2I', 'Cs0.3FA0.7PbBr0.256I0.744', 'Cs0.07FA0.9MA0.03PbBr0.24I2.76', 'Cs0.1MA0.9Pb0.9Sn0.1Br0.3I2.7', 'MAIn0.25Pb0.75I3', 'MAHg0.1Pb0.9I3', 'Cs0.05MA0.95Pb0.95Sn0.1Br0.15I2.85', 'Cs0.25FA0.75PbBr0.6I2.4', 'Cs0.1FA0.74MA0.1530PbBr0.51I2.49', 'Cs0.125FA0.875PbBr0.375I2.625', 'MAPb(SCN)I2', 'MAPb1.0BrI2', 'MAPbBr0.6I2.4', '(PGA)0.2BA1.8MA3Pb4I13', 'CsPbBr3 | CsPbBrI2', 'Cs0.1FA0.7MA0.2PbBr0.3I2.7', 'Cs0.10FA0.36MA0.54PbBr0.2I2.8', '(Ace)0.08MA0.92PbI3', '(Anyl)2MA3Pb4I12', '(PBA)0.5BA1.5MA3Pb4I13', 'Cs0.05FA0.85MA0.15PbBr0.45I2.70', 'FA0.67MA0.33PbBr2I', 'FA0.5MA0.5PbBr0.3I2.7', 'Cs0.15FA0.85PbBr0.87I2.13', '(PEA)2MA19Pb20I61', 'Cs0.05FA0.79MA0.16PbBr0.03I2.97', 'Cs0.1FA0.85Rb0.05PbI3', '(Ace)0.03MA0.97PbI3', 'Cs0.1FA0.85MA0.05PbBr0.15I2.85', 'FA0.83MA0.17PbBr0.46I2.54', 'Cs0.02FA0.83MA0.17PbBr0.51I2.49', 'FA0.57MA0.43PbBr0.13I2.87', 'MAGeI3', 'HAMAPbI3', 'FA0.15MA0.85PbI3', 'CsSnBr2.5I0.5', 'HASnI3', 'Cs0.05FA0.93GU0.02PbI3', 'MACu0.01Pb0.99Br0.01I2.99', 'CsNi0.03Pb0.97Br3', 'Cs0.05FA0.78MA0.12PbBr0.51I2.49', 'MA3Bi2I10', 'FASnBr3', 'IM0.3MA0.30.7PbI3', 'Cs0.05FA0.76MA0.16PbBr0.48I2.52', 'EA0.15MA0.85PbI3', 'MASnI3 | MAPbI3', '(BZA)1.85(HAD)0.15MA2Pb3I10', 'Cs0.1FA0.9PbBr0.095I0.905', 'FA0.57MA0.43PbI3', 'Cs0.09FA0.77MA0.14PbBr0.42I2.58 | (FPEA)2PbI4', 'Cs0.05FA0.15MA0.85PbI3', 'CsEu0.07Pb0.93BrI2', 'Cs0.2FA0.8PbBr0.28I2.72', 'MACa0.01Pb0.99I3', '(THM)0.075MA0.925PbI3', 'Cs0.40MA0.60PbI3', 'FA0.43MA0.57PbBr0.13I2.93', '(Ada)2FA2Pb3I10', 'Cs0.17FA0.83Pb0.4Sn0.6I3', 'MAPbBr0.01I2.99', '(BIM)0.1MAPbI3.2', 'Cs0.96K0.04PbBr3', 'Cs0.17FA0.83PbBr0.3I2.7', 'FA0.85MA0.15PbBr3', 'Cs0.05MA0.95PbBr1.5I1.5', 'Cs0.2668FA0.666MA0.0664PbBr0.095I0.905', 'FA0.85MA0.15PbBr0.45I2.55 | BA0.5FA1.5PbI4', 'MAPbBr0.78I2.22', 'Cs0.05MA0.17FA0.76PbBrI2 | BA2PbBr3.2I0.8', 'FA0.95MA0.05PbBr0.15I2.75', 'Cs0.3FA0.7PbI3', 'Cs0.34MA0.66PbI3', '(PEA)0.03MA0.97PbI3', 'IM0.3MA0.7PbI3', 'nanPbI2', '(5-AVA)0.05Cs0.05MA0.9PbI3', 'MAPbBr0.56I2.44', 'Cs0.17FA0.83PbBr0.51I2.5', 'FA0.93MA0.03PbBr0.09I2.91', 'MASnBr3', 'FA0.37MA0.63PbI3', 'MAPb0.75Sn0.25Br2.4I0.6', 'MAPb0.38Sn0.62I3', '(PEA)BAFA3Sn4I13', 'MAPb0.97Sb0.03Br2.94I0.09', 'EA2MA10Pb11I34', 'FA0.12MA0.8PbBr0.12I2.88', 'Cs0.05FA0.285MA0.665PbI3', 'BA2MAPb2I7', 'Cs0.025FA0.825MA0.15PbBr0.45I2.55', 'FA0.8MA0.2Pb1.0I3', 'MA3PbBr0.32I2.68', 'MAPbBr0.27I2.73', 'HAPbI4', 'FA0.15PN0.85SnI3', 'CsEu0.05Pb0.95BrI2', 'AN0.15FA0.5MA0.8PbI3', 'FA0.84MA0.16PbBr0.51I2.49', 'Cs0.2FA0.8PbBr0.12I2.88', 'Cs0.05FA0.81MA0.14PbI3', 'FA0.048MA0.952PbBrI2.91', 'BA2Cs0.6FA3.4Pb3Sn2.4I17', 'MA2CuBr2Cl2', 'Cs0.05FA0.8075MA0.1425PbBr0.45I2.55', 'MAPb0.75Sb0.25I3', 'Cs0.2FA0.66MA0.14Pb0.5Sn0.5Br0.5I2.5', 'Cs0.1FA0.75MA0.15PbBr0.35I2.65', 'FAPbBrI2', 'FA8PR2Pb9I28', '(DAT)MA2Pb3I10', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | (pFPEA)PbI3', '(PEA)2PbI4 | Cs0.1FA0.74MA0.13Pb1Br0.39I2.48 | (PEA)2PbI4', 'Cs0.065FA0.79MA0.145PbBr0.49I2.51', 'CsPbBr3', 'Cs0.1FA0.9PbBr0.1I2.9 | (PMA)PbBr0.1I2.9', 'Cs0.05FA0.95MAPbI3', 'Cs0.17FA0.83PbBr2.49I0.51', '((CH3)3S)2SnClI5', 'FA0.7GU0.3SnI3', 'CsBa0.1Pb0.9BrI2', 'Cs0.05FA0.8265MA0.1235PbBr0.51I2.49', 'BA2MA23Sn4I13', 'FA0.33PbBr1.5I1.5', 'CsPb0.88Zn0.12BrI2', 'FA0.57MA0.43PbBr0.13I2.91', 'FA0.85MA0.15Pb1.0Br0.45I2.55', '(MTEA)2MA4Pb5I16', 'FA0.8MA0.2PbBr0.6I2.40', 'FA0.17MA0.83PbBrI2', 'IM0.05MA0.95PbI3', '(4AMP)MA3Pb4I13', 'Cs0.15AgBi3I8.5', 'Cs0.8MA0.2PbI3', 'Cs0.2FA0.8PbBr0.9I2.1', 'FA0.85MA0.15PbBr0.45I2.55 | PEA2PbI4', 'Cs0.4FA0.2MA0.4PbI3', 'MAGeBrI2', 'MAPbBr0.045I2.955', 'FA0.75MA0.25PbBr0.45I2.55', 'MAIn0.10Pb0.90I3', 'Cs0.05FA0.79MA0.16Pb0.5Sn0.5Br0.5I2.5', 'FAPbI3 | (ODA)2PbI4', '(PEA)2MA8Pb9Br11.2I16.8', 'Cs2FAnanPbI3', 'HA0.4MA0.6SnI3', 'Cs0.05FA0.79MA0.16PbBr0.15I0.85', 'FA0.97MA0.03PbBr0.09I2.91 | (MIC1)2PbI4', 'FAPb0.75Sn0.25I3', 'FA4GeSbCl12', 'Cs0.1FA0.1MA0.8Pb1.0I3', 'Cs0.1MA0.9PbBr1.2I1.8', 'MAPb0.4Sn0.6Br1.2I1.8', 'BA2FA2Pb3I10', 'MAPb0.75Sn0.25Br1.2I1.8', 'MAPbBr0.1I2.9', 'CsPbBr0.45I2.55', 'Cs0.1MA0.9Pb0.7Sn0.255I3', 'FA0.9MA0.1PbBr0.15I2.85', 'Cs0.2FA0.66MA0.14Pb0.25Sn0.75Br0.5I2.5', 'CsMAPbI3', 'MAIn0.05Pb0.95I3', 'Cs0.2FA0.66MA0.14SnBr0.5I2.5', 'Cs0.2FA0.8PbBr0.6I2.4', 'GUPb(SCN)3I', 'Cs0.1FA0.36MA0.54PbBr0.2I2.8', 'FA0.85MA0.15Pb1.0Br0.15I0.85', 'FA0.15MA0.85PbBr0.15I2.85', 'Cs0.05FA0.7885MA0.1615PbBr0.4845I2.4155', 'FA0.66MA0.34PbI3', 'Cs0.05FA0.79MA0.16Pb0.916Sn0.084Br0.52I2.48', 'CsSnBr0.3I2.7', 'GU0.25MA0.75PbI3', '(TBA)0.03MA0.97PbI3', 'FA0.125MA0.875PbBrI2', 'IM0.25MA0.250.75PbI3', 'FA0.5MA0.5PbBr0.13I2.92', 'MAPb0.3Sn0.7I3', 'BAFA10Pb11I34', 'FA0.83MA0.17PbBr0.52I2.48', 'FA0.3MA0.7PbBr2.1I0.9', 'CsPbBrI2', 'IMPbI3', 'CsPbBr3 | CsPb2Br5', 'Cs0.15FA0.75MA0.1PbI3', 'CsNi0.003Pb0.997Br3', 'MASnBr0.42I2.58', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | (mFPEA)PbI3', 'Cs0.05FA0.80MA0.15PbBr0.43I2.57', 'Cs0.88K0.12PbBr3', '(CIEA)0.03MA0.97PbI3', '(PyEA)2MA8Pb9I28', 'CsPbBrxIx', 'FA0.3MA0.7PbBr0.225I2.775', 'FA0.75MA0.75PbI3', 'AgCs2BiBr5.9', 'HA0.1MA0.9PbI3', '(GABA)0.1MA0.9PbI3', 'FA0.83MA0.17PbBr0.03I2.97', 'Cs0.04FA0.96PbI3', '(PEA)2MAPb2I4', 'Cs0.1FA0.83MA0.17PbBr0.51I0.249', 'FA0.85MA0.15PbBr0.5I2.5', 'CsAgBiBr6', 'FA0.55MA0.45PbI3', 'FAPbBr2.43Cl0.57', '(F5PEA)2PbI4', 'FASn1.0I3', 'Cs0.25FA0.75PbBr0.51I2.49', 'MAMnI3', 'FA0.7MA0.3PbBr0.256I0.744', 'Cs0.05FA0.85MA0.1PbBr0.66I2.34', 'FAxMAxPbBrxIx', 'FA0.7MA0.3Pb0.7Sn0.3I3', 'FA0.3MA0.7PbBr0.3I2.7', 'Cs0.005Pb0.995Br3', 'Cs0.05FA0.7885MA0.1615PbBr0.34I2.66', 'AN0.06MA0.94PbI3', '(f-PEA)2PbI4', 'MAPbBrI', 'BA2Cs0.15FA0.57MA2.28Pb4I13', 'Cs0.17FA0.83PbI3', 'Cs0.4FA0.6PbBr0.256I0.744', 'CsPbBr3I', 'Ca0.2MA0.8PbBr0.6I2.4', 'CsBa0.3Pb0.7BrI2', 'Ag3Bi2I9', 'Cs0.15FA0.85PbBr2.1I0.9', 'IM0.4MA0.40.6PbI3', 'Cs0.1FA0.75MA0.15PbBr0.45I2.55', 'Cs0.2FA0.32MA0.48PbBr0.2I2.8', 'BA2MA3Pb4I13', 'Cs0.14FA0.65MA0.21PbBrI2', 'CsPbBr2.7I0.3', 'FA0.83MA0.17Ge0.03Pb0.97Br0.3I2.7', 'CsPbBr2.8I0.2', 'MAHg0.05Pb0.95I3', 'Cs0.4MA0.6PbI3', 'MAPbBr0.08I2.92', 'Cs0.08FA0.78MA0.14PbBr0.45I2.55', 'MAEu0.1Pb0.9I3', 'Cs0.005Pb0.995Br1.99I1.01', 'Cs0.05FA0.79MA0.16Pb0.25Sn0.75Br0.5I2.5', 'CsPbBr0.75I2.25', '(PEA)2PbI4 | MAPbI3', '(HEA)2Cs0.9FA8.1Pb10Br3.3I29.7', 'Cs0.05FA0.7885MA0.16150PbBr0.51I2.49', 'Cs0.2FA0.8PbBr1.5I1.5', 'CsEu0.03Pb0.97BrI2', '(PEA)0.1MA0.9SnI3', '(NH4)8.5FA0.15MA2.04Pb9.5Br0.45I29.24', 'Cs0.25FA0.75Pb0.37Sn0.6255I3', 'Cs0.2FA0.8PbBr0.15I2.85', 'Cs0.1MA0.9Pb0.9Sn0.1Br0.2I2.8', 'Cs0.05FA0.79MA0.16PbBr0.52I2.48', 'MAPbBr0.237I2.763', 'CsPbBr3 | CsPbI3', 'FAPbBr1.25Cl0.35I1.45', 'Cs0.03FA0.97PbI3', 'Cs0.06FA0.87MA0.07PbBr0.12I2.88', 'FAPb0.5Sn0.5I3', 'FA0.75MA0.25PbI3', 'Cs0.04FA0.71GU0.1MA0.15PbBr0.5I2.49', 'EA0.5MA0.5PbBr3', 'Cs0.03FA0.77MA0.2PbBr0.46I2.54', 'Cs0.5FA0.75MA0.1PbBr0.51I2.49', 'MAPbBr0.06I2.94', 'FA0.10MA0.9PbI3', 'Cs0.05FA0.78MA0.13PbBr0.45I2.55', 'MAPb0.75Sn0.25Br1.8I1.2', 'CsPb0.97Zn0.03Br3', 'Cs0.05FA83MA17PbBr0.51I2.49', 'Cs0.17FA0.83PbBr0.5I2.5', 'FA0.85MA0.15PbBr0.45I2.55 | BA1.5FA0.5PbI4', 'MAFe0.1Pb0.9I3', 'Cs0.12FA0.88PbBr0.36I2.54', 'FA0.5MA0.5PbBr3', 'FA0.85MA0.15PbBr0.45I2.55 | (FEA)2PbI4', 'Cs0.15(DMA)0.85MA0.15PbI3', 'Ca0.05FA0.8075MA0.1425PbBr0.45I2.55', 'BA2Cs0.15MA2.85Pb4I13', 'Cs0.05FA0.79MA0.16Pb0.75Sn0.25Br0.5I2.5', 'MAPbBr0.09I2.1', 'Cs0.06FA0.58MA0.36PbBr0.12I2.88', 'IM0.1MA0.9PbI3', 'Cs0.05FA0.79MA0.16PbBr0.3I2.7', 'Cs0.17FA0.83Pb0.3Sn0.7I3', 'Cs0.05FA0.85MA0.1PbBr0.45I2.55 | (PEA)2PbI4', 'MABi2I9', 'FA0.26MA0.74PbI3', 'FA0.85MA0.15PbBr0.3I2.7', 'FA0.85MA0.15PbBr0.50I2.50', 'FA0.17MA0.83PbBr0.5I2.5', 'MA3BiI2', 'FA0.83MA0.17PbBr0.39I2.61', '(NH4)5.1FA0.15MA1.7Pb6.1Br0.45I18.7', 'Cs0.05FA0.7917MA0.1583PbBr0.5I2.5', 'Cs0.01FA0.94Rb0.05PbI3', 'MABiSbI9', '(NH4)3Sb2Br3I6', '(PEA)0.15FA0.85SnI3', 'FA3OA2Sn4I13', 'RbPbI3', 'Cs0.1MA0.9PbBr0.1I2.9', '(PEA)2Cs9Pb10I34', 'Cs0.1FA0.77MA0.13PbBr0.4I2.6', 'CsPbBr3 | FAPbBr3', 'CsSnBrI2', 'Cs0.2FA0.8Pb0.3Sn0.7I3', '(NH4)11.9FA0.15MA1.7Pb12.9Br0.45I39.1', 'Cs0.05FA0.81MA0.14PbBr0.45I2.55 | CsPbBr1.85I1.15', 'FA0.8MA0.2PbBr0.2I2.8', 'Cs0.09FA0.77MA0.14PbBr0.42I2.58 | (PEA)2PbI4', 'BA2MA3Pb4I9', 'FAPbBr0.44I2.56', '(PEA)2FA3Pb4I13', 'MAPb(BF4)2.85I0.15', '(CHMA)2MA39Pb40I121', 'MA3Bi2I11', 'FA0.5MA0.5SnI3', 'Cs0.21Ag3Bi3.14I9', '(oF1PEA)2MA4Pb4I13', 'HA2MA3Pb4I13', 'CsPbBr3 | FAPbBr1.5Cl1.5', 'CsFAMAPbBrI', '(PEA)2Cs1.77FA57.23Pb60I181', 'FA0.75MA0.25SnI3', 'Cs0.05FA0.7885MA0.1441PbBr0.3I2.7', 'MAPbBr1.5I1.5', 'Cs0.085FA0.915PbBr0.45I2.55', '(PEI)2MA6Pb7I22', 'MASnBr1.5Cl0.5I', 'Cs0.05FA0.79MA0.16PbBr0.47I2.53', 'Cs0.15FA0.75MA0.1PbBr0.5I2.5', '(PEA)x(F5PEA)xCs0.15FA0.64MA0.2PbBr0.6I2.4', 'Cs0.2FA0.8PbBr1.2I1.8', 'CsPb0.99Sr0.01BrI2', 'Cs0.92K0.08PbBr3', 'MAHg0.075Pb0.925I3', 'FA0.9MA0.1PbBr0.03I2.91', 'Cs0.05FA0.79MA0.16PbBr2.51I2.49', 'FA0.95MA0.05PbBr0.1I2.9', 'CsBa0.1Pb0.9I3', '(BYA)2MA3Pb4I13', 'Cs0.07FA0.93PbI3', 'MA2PA3Pb4I13', 'FAPbCl0.5I2.5', 'Cs0.0664FA0.8668MA0.0664PbBr0.095I0.905', '(Ace)0.2MA0.8PbI3', 'MAPb0.9Sr0.1I3', 'Cs0.75FA0.25PbI3', 'FA0.83MA0.17PbBr0.17I2.83', '(PEA)2Cs59Pb60I181', 'BA2MA2Pb3I9', '(HEA)2Cs1.9FA17.1Pb20Br9.45I53.55', 'FA0.28MA0.72PbI3', '(iso-BA)0.5MA0.75PbI3.25', 'Cs0.15FA0.85PbBr0.12I2.88', 'FA0.88MA0.12PbBr0.15I2.85', 'Cs0.92Li0.08PbBr3', 'Cs0.175FA0.750MA0.075PbBr0.36I2.64', '(BZA)1.8(HAD)0.2MA2Pb3I10', '(3AMP)MA3Pb4I13', 'Cs0.06FA79MA0.15PbBr0.45I2.55', 'FAPb0.625Sn0.375I3', '(PMA)0.67FA0.33PbI3', '(TBA)0.01MA0.99PbI3', 'Cs0.15MA0.85PbBr1.2I1.8', 'MAPbBr1.16I1.74', 'FA0.83MA0.17PbBr0.51I2.49', 'FA0.85MA0.15PbBr0.15I0.85', '(PEA)2Pb2I4', 'MASbSnI9', 'Cs2Pb(SCN)2I', 'FA0.85MA0.15PbBr0.45I0.85', 'Cs0.88Na0.12PbBr3', 'nanBi2O6', 'CsPbBrF0.12I1.88', '(PEA)2Cs0.45FA2.55Pb4I13', 'Cs0.25FA0.75PbBr0.60I2.40', 'EA0.92FA0.08SnI3', '(PDA)0.05Cs0.15FA0.8PbI3', '(APMim)PbBrI3', 'MASbI3', '(PEA)1.4BA0.6MA3Pb4I13', 'FA0.95MA0.15PbBr0.45I2.5075', 'MAPbBr0.9I2.1', 'Cs0.05FA0.75MA0.15PbBr0.15I2.85', 'FA0.90TN0.10SnI3', 'MAMn0.1Pb0.9I3', 'Cs0.1FA0.9PbBr0.09I2.91', 'MA2Au2Br6', 'FA0.75MA0.25Pb0.75Sn0.25I3', 'FA0.85MA0.15PbBr0.55I2.55', 'Cs0.05FA0.38MA0.57PbBr0.2I2.8', 'MA2PA6Pb7I22', 'Cs0.04FA0.82MA0.14PbBr0.42I2.58', 'Cs0.05FA0.8K0.03MA0.12Ge0.03Pb0.97Br0.3I2.7', 'MAPbBr2.85I0.15', 'Cs0.05FA0.79MA0.16PbBr1.2I2.6', 'FA0.75MA0.25PbBr0.25I2.79', 'CsSnI3', '(5-AVAI)0.02Cs0.05FA0.93PbI3', 'Cs0.05FA0.79MA0.16PbBr0.41I2.59', 'AN0.5MA0.5PbI3', 'MACo0.2Pb0.8I3', '(TBA)0.1Cs0.05FA0.71MA0.14PbBr0.51I2.49', 'MAEu0.04Pb0.96I3', 'Cs0.17FA0.83Pb0.6Sn0.4I3', 'Cs0.1FA0.75MA0.15PbBr0.46I2.54', 'MA2SnI6', 'FA0.75MA0.25PbBr0.75I2.25', 'MAPbBr0.19I2.81', '(5-AVA)0.05MA0.95PbI3', '((CH3)3S)2SnI6', 'Cs0.06FA0.94PbBr0.03I2.97', 'AN0.15Cs0.5MA0.8PbI3', 'MAPbI3 | CsPbI3', 'FA0.2PN0.8SnI3', 'BA2Cs0.2FA0.6MA3.2Pb5I16', 'Cs0.05FA0.81MA0.14PbBr0.42I2.58', 'Cs0.1FA0.9PbBr0.6I2.4', '(3AMP)FA0.9MA2.1Pb4I13', 'Cs0.02MA0.98PbBr0.06I2.94', 'Cs0.07FA0.785MA0.145PbBr0.45I2.55', 'Cs0.01Pb0.99Br1.98I1.02', 'FAHA2Pb2I7', 'Cs0.5FA0.5PbI3', 'Cs0.5FA0.85PbI3', 'FA0.2MA0.8Pb1.0I3', 'Cs0.1FA0.1MA0.8PbI3', 'CsPbBr0.21I2.79', 'EA2MA8Pb9I28', 'Cs0.15FA0.71MA0.1PbBr0.39I2.61', 'MAPbI3 | MAPbBr1.2I1.8', 'Cs0.05FA0.82MA0.13PbBr0.13I2.87', 'Cs0.1FA0.76MA0.14PbBr0.51I2.49', 'Bn2SnI4', 'Cs0.14FA0.83MA0.17PbBr0.51I0.249', 'Cs0.05FA0.79MA0.16PbBr0.51I', 'MACa0.02Pb0.98I3', 'FA0.9MA0.1PbBr0.03I2.97', 'CsFAPbBrClI', 'Cs0.15FA0.26MA0.59PbI3', 'MAPb0.92Sb0.08I3', '(mF1PEA)MA4Pb4I13', 'FA0.7MA0.3PbBr0.10I2.90', 'CsFA0.83MA0.17PbBr0.5I2.5', 'Cs0.04MA0.96PbI3', 'PA2PbI4', '(PEA)0.1FA0.15MA0.75SnI3', '(PDA)0.03Cs0.15FA0.82PbI3', 'CsCu0.003Pb0.997Br3', 'AgBiI4', 'FA0.17MA0.83PbBr2.5I0.5', 'Cs0.1FA0.9PbBr0.225I2.775', '(Ace)0.1MA0.9PbI3', 'Cs0.05FA0.8MA0.15PbBr0.55I2.55', 'Cs0.17FA0.83Pb0.5Sn0.5I3', 'Cs0.07FA0.79MA0.14PbBr0.45I2.55', 'MAPb0.75Sn0.25I', '(PEA)2FA3Sn4I13', 'Cs0.17FA0.83Pb0.8Sn0.2I3', '(Cl-PEA)2MA2Pb3ClI10', 'Cs0.05MA095PbI3', '(PEA)2Cs59Pb60Br120.67I60.33', 'MAPbBr0.7I2.3', '(ThFA)2MA2Pb2I7', '(EDA)0.04FA0.29MA0.67Pb1.0I3', 'FA0.97MA0.03PbBr0.09I2.91', 'FA0.7MA0.3PbBr0.9I2.1', 'FA0.83MA0.17PbBr0.17I0.83', 'Cs0.1FA0.75MA0.15PbBr1.5I1.5', 'Cs0.16FA0.8MA0.04PbBr1.5I1.5', 'CsPb0.6Sn0.4I3', 'Cs0.05FA0.76MA0.16PbBr0.49I2.51', 'EA2MA19Pb20I61', 'MAPb0.375Sn0.625I3', '(F-PEA)2MA2Pb3I10', 'Cs0.05FA0.85MA0.1PbBr0.5I2.5', 'FA0.92MA0.08PbI3', 'FA0.7MA0.3PbBr0.095I0.905', 'AN0.3MA0.7PbI3', 'Cs0.05FA0.81MA0.14PbBr0.45I2.55', 'FA0.6MA0.4PbBr0.4I2.6', 'Cs0.05FA0.79MA0.16PbBr0.30I2.70', 'CsPbBr1.5I1.5', '(PEA)0.1FA0.9SnI3', 'Cs2NaBiI6', 'FA0.95MA0.05PbBr0.15I2.85 | BA2PbI4', 'MAPbI3 | MAPbBr0.9I2.1', 'MAPbBr2.5I0.5', 'Cs0.12MA0.88PbBr0.36I2.64', 'MA0.98PbI3', 'MAPbBr2.52I0.48', 'Cs0.17FA0.83PbBr0.17I0.83', 'FA0.1MA0.9PbI3', 'MAPbBr1.62I1.38', 'Cs0.1FA0.6MA0.3PbBr0.095I0.905', 'Cs0.05FAxMAxPbBrxIx', 'FA0.83MA0.17PbBr0.51I2.59', 'Cs0.025FA0.7MA0.25Rb0.025PbI3', 'MAPbI3-xClx', 'FA0.84MA0.16PbBr0.48I2.52', 'Cs0.25FA0.75PbBr0.9I2.1', 'IM0.2MA0.8PbI3', 'FA0.2MA0.8PbBr0.45I2.55', 'FA0.2MA0.8PbI3', 'MACu0.1Pb0.9I3', 'Cs0.08FA0.92PbBr3', 'FA0.83MA0.17PbBrI2', 'CsBi2I7', 'FA0.4MA0.6PbBr0.6I2.4', 'FAPbBr2.67Cl0.33', 'CsPb0.9Sn0.1Br2I', 'MAPbBr2.91I0.09', 'Cs0.15FA0.75MA0.1PbBr0.4I2.6', '(EDA)0.04FA0.28MA0.68Pb1.0I3', '(CHMA)2MA2Pb3I10', 'MAPbBr0.4I2.6', 'Aa0.05MA0.95PbI3', '(NH4)3Sb2Br6I3', 'GU0.1MA0.9PbI3', 'HA0.2MA0.8SnI3', 'Cs0.1FA0.9PbBr0.3I2.7', 'MAGeBr0.15I2.75', 'CsPbI3', 'Cs0.04FA0.8MA0.16PbBr0.49I2.51', 'MASnBr2I', 'Cs0.12FA0.83MA0.05PbBr1.2I1.8', 'Cs0.05FA0.88MA0.07PbBr0.24I2.76', 'Cs0.45FA0.55PbI3', 'MACo0.063Pb0.937I3', 'Cs0.05FA0.827MA0.123PbBr0.369I2.631', 'MAPb0.99Sr0.01I3', 'MAPbBr0.09I2.91', 'Cs0.05FA0.75MA0.2PbI3', 'MASb1.2Sn0.8I9', '(TBA)0.1MA0.9PbI3', 'Cs0.99Rb0.01PbBrI2', 'Cs0.02FA0.98PbI3 | Cs0.57FA0.43PbI3', '(PDA)MA3Pb4I13', 'MAPbISCN', 'Cs0.17FA0.83PbBr0.30I2.7', 'FAPbI3 | EDA22Pb3I10', '(CPEA)2MA5Pb6I19', 'MACu0.0118Pb0.9882Br0.2I2.8', '(DPA)2MA3PAPAPb4I13', 'FA0.85MA0.15PbBr0.27I2.73', '(C6H4NH2)CuBr2I', 'Cs0.25FA0.75PbBr0.6I2.40', 'BA2MA2Pb3I10', '(3AMP)FA0.75MA2.25Pb4I13', 'MACo0.016Pb0.984I3', 'MAPbBr0.02I2.98', '(CIEA)0.05MA0.95PbI3', 'Cs0.07FA0.93PbBr0.06I2.94', 'nanCu3SbI6', 'Cs0.05MA0.95PbBr0.9I2.1', 'Cs0.11FA0.74MA0.15PbBr0.51I2.49', 'Cs0.15FA0.85PbBr0.81I2.19', 'Cs0.05Pb0.95Br1.9I1.1', '(PBA)2MA3Pb4I13', 'CsGe0.3Pb0.7BrI2', 'MAPbBr1.26I1.74', 'Cs0.05FA0.93GA0.02PbI3', 'MACa0.10Pb0.90I3', 'MAPbI3 | MAPbI3', 'FA0.25MA0.75Pb0.75Sn0.25I3', '(PEA)0.6BA1.4FA3Sn4I13', 'MAPbBr0.39I2.61', 'Cs0.97Rb0.03PbBr3', '(4ApyH)SbI4', 'Cs0.17FA0.83PbBr0.81I2.19', 'FA0.87MA0.13PbBr0.51I2.49', 'AgCs2Bi0.75Sb0.25Br6', 'FA0.84MA0.16PbBr0.5I1.5', 'Cs0.1FA0.75MA0.15Pb0.25Sn0.75Br0.5I2.5', 'MAPb0.93Sb0.07I3', '(PEA)2Cs79Pb80I241', 'FA0.3MA0.67PbI3', 'CsPbBrF1.88I0.12', '(H-PEA)2MA2Pb3I10', 'Cs0.05FA0.75MA0.2PbBr0.3I2.7', '(BEA)2MA3Pb4I13', 'FAPbBr1.25Cl0.25I1.5', 'EA0.4MA0.6PbI3', 'Cs0.79FA0.16MA0.1PbBrI2', 'MAPb1.0Br0.6I2.4', 'Cs0.05FA0.78MA0.16PbBr0.5I2.5', 'FA0.8MA0.8PbBr0.06I2.96', 'Ca0.05MA0.95PbBr0.15I2.85', 'MAPbBr0.30I2.70', 'FA0.3MA0.7Pb1.0I3', 'FA0.7MA0.3Pb0.5Sn0.5I3', 'KBABiTeO6', 'CsPb0.97Tb0.03Br3', 'MAPbBr0.2I2.7', 'MA2PA4Pb5I16', 'Cs0.30FA0.70PbI3', 'Cs0.07FA0.81MA0.1146PbBrI2', 'FA0.8MA0.2PbBr0.45I2.55', 'MAPbI3 | (PEA)2PbI4', 'Cs0.05FA0.95PbBr0.15I2.85', '(PDA)0.04Cs0.15FA0.81PbI3', 'Cs0.05FA0.79MA0.17PbBr0.5I2.5', 'Cs0.3FA0.6MA0.1PbBr0.256I0.744', 'MA3Bi2I9', 'BAGUMA4Pb5I16', 'MAPb1.0Br0.3I2.7', 'Cs0.05FA0.65MA0.3PbBr0.15I2.85', 'CsPb0.98Sr0.02BrI2', 'MAPbBr0.075I2.95', '(CPEA)2MA2Pb3I10', 'Cs0.15FA0.71MA0.14PbBr0.45I2.55', 'MAPbBr0.03I2.7', 'AN0.4MA0.6PbI3', 'Cs0.05MA0.95PbI3', 'Cs0.07FA0.78MA0.15PbBr0.45I2.55', 'Cs0.05FA0.79MA0.11Rb0.05PbBr0.39I2.61', '(APMim)PbBr2I3', 'Cs0.05FA0.80MA0.15PbI2.55', 'Cs0.1FA0.9PbI3 | (PEA)2PbI4', '(CPEA)2MA4Pb5I16', 'Cs0.05FA0.83MA0.17PbI3', 'Cs0.05FA0.81MA0.14PbBr0.57I2.43', '(TBA)0.3MA0.7PbI3', 'Cs0.1FA0.77MA0.13PbBr0.39I2.48 | (PEA)2PbI4', 'Cs0.05FA0.81MA0.14PbBr0.39I2.61', 'Cs0.07FA0.785MA0.115PbBr0.45I2.55', 'MAPbBr0.75I2.25', 'BA2Cs0.08MA3.92Pb5I16', 'Cs0.05FA0.92MA0.3PbBr0.09I2.91', 'FA0.57MA0.43PbBr0.04I2.96', 'BDACs4Pb5Br4.8I11.2', 'IM0.03MA0.97PbI3', '(PBA)BAMA3Pb4I13', 'MAPbBr0.15I2.85', 'Cs0.32FA0.58GA0.1PbBr0.81I2.19', 'FA0.6MA0.4Sn0.6I3', 'GU0.17MA0.83PbI3', 'Cs0.04FA0.80MA0.16PbBr0.51I2.49', 'FAxMAPbBrxI', 'BAMA3Pb4I13', 'Cs0.02MA0.98PbI3', 'AgCs2Bi0.5Sb0.5Br6', 'MAPbBr0.033I2.97', 'Cs0.05FA0.83MA0.17PbBr0.45I2.55', 'EA0.5MA0.5PbI3', 'FA0.83MA0.17PbBr2.5I0.5', '(BZA)1.9(HAD)0.1MA2Pb3I10', 'GUPb(SCN)2.6I0.4', 'MAPbI3 | MABaPbI3', 'FA0.9MA0.1PbBr0.256I0.744', 'FA3Bi2I9', 'Cs0.10FA0.75MA0.15PbBr0.51I2.49', 'Cs0.04FA0.92MA0.04PbI3 | (FEA)2PbI4', 'MAPbI3 | MAPbBr3', 'Cs0.1FA0.81MA0.09PbBr0.17I2.83', 'Cs0.1FA0.75MA0.15PbBrI2', 'CsPb0.5Sn0.5BrI2', '(EDA)0.01FA0.29MA0.7Pb1.0I3', 'Cs0.1MA0.9PbI3', '(PEA)0.1FA0.15MA0.75SnBr0.24I2.76', 'Cs0.01FA0.99PbI3', 'CsPb1.0Br1.8I1.2', 'BAFA60Pb61Cl4I180', 'FA0.98MA0.02PbBr0.06I2.94', '(PEA)2MA39Pb40I121', 'FA0.85MA0.10PbBr0.3I2.7', 'MA0.1Mg0.1Pb0.9I3', 'MAPbBr0.60I2.40', 'Cs0.075FA0.75MA0.175PbBr0.33I2.67', 'Cs0.2FA0.8PbI3', 'Cs0.025FA0.475MA0.5Pb0.5Sn0.5I3', 'MA2CuBr3.5Cl0.5', 'FA0.95MA0.05PbBr0.15I2.85 | DA2PbI4', '(PDA)MAPb2I7', 'MAPb0.2Sn0.8Br0.4I2.6', 'CsPb1.0Br1.5I1.5', 'FA0.8MA0.2PbBr0.095I0.905', 'MASnF0.4I2.6', 'BU2FA8Pb9I28', 'MASnBr2.64I0.36', '(4AMP)MA2Pb3I10', 'Cs0.15FA0.85PbBr0.45I2.55', 'BA2MA3Pb4I12', 'Cs0.05FA0.07MA0.25PbI3', 'Cs0.04FA0.96PbBr3', 'Cs2Bi3I9', 'MABa0.01Pb0.99I3', 'Cs0.05FA0.79MA0.16Pb0.748Sn0.252Br0.52I2.48', 'Cs0.2FA0.8PbBr1.05I1.95', 'FA0.1MA0.9Pb1.0I3', 'FA0.78MA0.21PbBr0.21I2.79', 'BA2Cs0.3FA1.7Pb1.8Sn1.2I10', 'MA0.1Mn0.1Pb0.9I3', '(BEA)0.5MA3Pb3I10', 'FA0.38MA0.62PbI3', 'MAPbBr0.43I2.57', 'FA0.81MA0.19Pb0.57I2.33', 'FAPbI3', 'Cs0.05FA0.79MA0.16PbBr0.6I2.4', 'Cs0.2FA0.72MA0.08PbBr0.03I2.97', 'Cs0.133FA0.733MA0.133PbBr0.095I0.905', 'CsPbI3 | FAPbI3', 'MAPb0.9Sn0.1I3', '(ThMA)2MA2Pb3I10', 'Cs0.32FA0.58GU0.1PbBr0.81I2.19', 'Cs0.1FA0.75MA0.13PbBr0.45I2.55', 'GU0.50MA0.50PbI3', 'MA0.05Pb0.95I3', 'Cs0.06FA0.94PbI3', '(HEA)2Cs2.9FA26.1Pb30Br9.3I83.7', 'Cs0.17FA0.83Pb0.9999Sn0.0001I3', 'Cs0.1FA0.135MA0.765PbBr0.45I2.55', 'MAPb0.50Sn0.50Br1.2I1.8', 'FA0.81MA0.15PbBr0.45I2.55', 'nanPb2I8', '(TBA)0.5Cs0.03FA0.4MA0.08PbBr0.51I2.49', 'Cs0.80K0.20PbBr3', 'MAPbCl0.06I2.94', 'Cs0.05FA0.85MA0.1PbBr0.03I2.97', '(Anyl)2MA2Pb3I9', 'Cs0.1FA0.54MA0.36PbI3', 'CsBi0.06Pb0.94I3', 'Cs0.67FA0.33PbBr0.75I2.25', 'CsFASnI3', 'Cs0.1FA0.65MA0.25PbI3', 'Cs0.05FA0.77MA0.16PbBr0.48I2.52', 'MAPb0.85Sn0.15I3', 'FAPbBr0.45I2.55', '(N-EtPy)SbBr6', 'MAPbBr3', 'MAPbBr0.33I2.67', 'MAPb0.25Sn0.75I3', 'Cs0.14FA0.83MA0.03PbBr0.51I2.49', 'Cs0.05FA0.89MA0.06PbBr0.18I2.82', 'FA0.2MA0.8PbBr0.15I2.85', 'Cs0.15FA0.8GA0.05PbBr0.45I2.55', '(THM)0.1MA0.9PbI3', '(BZA)1.95(HAD)0.05MA2Pb3I10', 'FAMAPbBr3I', 'Cs0.05MAPbBr0.45I2.55', 'Cs0.05FA0.6MA0.35PbBr0.3I2.7', 'Cs0.20FA0.80PbBr1.2I1.8', 'Cs0.4FA0.6PbBr0.9I2.1', 'FAPbI3 | (PEA)2PbI4', 'Cs0.5Rb0.5SnI3', 'Cs0.1MA0.9SnI3', 'Cs0.05FA0.79MA0.12PbBr0.39I2.61', 'CsCa0.03Pb0.97Br3', 'Cs0.05FA0.79MA0.15PbBr0.45I2.55', 'Cs0.15FA0.85PbI3', 'MASbI2', 'MA0.125Pb0.875I3', 'MACu0.05Pb0.95Br0.05I2.95', '(DAP)PbI4', 'Cs0.05FA0.7885MA0.1615PbBr0.51I2.49', 'AN0.09MA0.91PbI3', 'Cs0.07FA0.81MA0.12PbBr0.38I2.62', 'MASb1.6Sn0.4I9', '(NMA)2MA39Pb40I121', 'MAPbI3 | (EU-pyP)2PbI4', 'Cs0.15FA0.85PbBr0.3I2.7', 'FASnBr0.24I2.76', 'FA0.5MA0.5PbI3', 'FAMAPbBrI', 'MAGeBr0.3I2.9', 'Cs0.091FA0.758MA0.152PbI3', 'Cs0.17FA0.83PbBr1.8I1.2', 'BDACs2Pb3Br3I7', 'FA0.85GU0.15SnI3', 'MAPb0.95Sn0.05Br0.1I2.9', 'Cs0.2FA0.2MA0.6PbI3', 'Cs0.05FA0.95SnI3', 'MAPb0.8Sr0.2I3', 'Ag2BiI5', 'Cs0.05FA0.85MA0.10PbBr0.45I2.55', 'Cs0.05FA0.81MA0.14PbBr0.4I2.6', 'Cs0.07FA0.7MA0.23PbBr0.69I2.31', 'FA0.65MA0.35PbI3', '(TMA)SnI3', '(PEI)2MA2Pb2I10', '(AVA)2PbI4 | MA1Pb1I3 | (BI)2PbI4', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | BAPbI4', 'CsPbBr3 | FAPbBr1.5I1.5', 'MAPbI3 | (BI)2PbI2', 'GU0.14MA0.86PbI3', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | CsPbI3', 'GU0.125MA0.875PbI3', 'Cs0.05FA0.79MA0.16PbBr1.5I1.5', 'CsSn0.05I2.95', 'MAPbBr0.25I2.75', 'Cs0.2FA0.8PbBr0.3I2.7', 'Cs0.3FA0.7Pb0.7Sn0.3I3', 'Cs0.25FA0.75Pb0.6Sn0.4I3', 'Cs0.04FA0.82MA0.14PbBr0.45I2.55', 'Cs0.0664FA0.666MA0.2668PbBr0.256I0.744', 'Cs0.05FA0.79MA0.16PbBr0.75I2.25', 'FA0.1MA0.9PbI3 | TAPbI3', 'Cs0.05FA0.79MA0.16PbBr0.45I2.55', 'Cs0.05FA0.87MA0.0782PbBrI2', 'MAPbI3 | (C4H9N2H6)PbI4', 'Cs0.6MA0.4PbI3', 'BA2MA4Pb5I16', '((CH3)3S)2SnBr2I4', 'Cs0.05FA0.05MA0.9PbBr0.05I2.95', 'MAPbBr0.2I2.8', 'MAPb0.5Sn0.5Br0.6I2.4', '(Ace)0.15MA0.85PbI3', 'CsPbBr', 'EA2MA2Pb3I10', 'Cs0.05FA0.788MA0.162PbBr0.5I2.5', 'DI2FA8Pb9I28', 'MAPbI3 | CsPbBr3', 'Ca0.1MA0.9PbBr0.3I2.7', '(PEA)1.8BA0.2MA3Pb4I13', 'Cs0.09FA0.58MA0.33PbBr0.65I2.35', 'CsPbBrF1.78I0.22', 'FA0.9MA0.1PbI3', 'CsSnBr0.5I2.5', 'FAPbBr0.15I2.85', 'Cs0.30MA0.70PbI3', '(NH4)10.2FA0.15MA1.7Pb11.2Br0.45I34', 'FA0.96MA0.04PbBr0.12I2.88', 'Cs0.05FA0.79MA0.16Pb0.54I2.46', 'CsLa0.02Pb0.98BrI2', 'MAPbBr0.04I2.96', 'MAPb0.4Sn0.6Br0.9I2.1', 'IM0.025MA0.0250.975PbI3', 'AN0.2MA0.8PbI3', 'Cs0.02FA0.98PbI3', 'FA0.6MA0.4PbBr1.2I1.8', '(DMA)0.05MA0.95PbI3', 'BA2MA2Sn3I10', 'Cs0.94Na0.06PbBr3', 'FA0.75MA0.25Ge0.2Sn0.8I3', 'MAPb0.75Sn0.25I3', '(PEA)2Cs3Pb4I13', 'FA0.125MA0.875PbI3', '(CHMA)2MA3Pb4I13', 'FA0.6MA0.4Pb0.4Sn0.6Br0.48I2.52', 'Cs0.05FA0.49MA0.16PbBr0.51I2.49', 'Cs0.05FA0.83MA0.12PbBr0.36I2.64', 'Ag2Bi3I11', '(PEA)FASnI3', 'Cs0.05FA0.79MA0.16PbI3', 'Cs0.15FA0.75MA0.1PbBr0.1I2.9', 'Cs0.15FA0.71MA0.14PbBr0.75I2.25', 'MAGeBr0.3I2.7', 'Cs0.05FA0.81MA0.14PbBr0.45', 'MAPb0.5Sb0.5I3', '(PEA)2MA5Pb4Cl2I10 | MA3PbCl2', 'FA0.38MA0.57PbI3', 'Cs0.025FA0.81MA0.15PbBr0.45I2.5', 'BA2Cs4MA35Pb40I121', 'Cs0.05FA0.79MA0.16PbBr0.39I2.61', 'AgCs1.7Rb0.3BiBr6', 'Cs0.09MA0.91PbI3', '(NH4)1.7FA0.15MA1.7Pb2.7Br0.45I8.5', 'GU0.05MA0.95PbI3', 'Cs0.13FA0.87PbBrI2', 'MAPbI3', 'Cs0.05FA0.8MA0.15PbBr0.75I1.25', 'MAPbBr2.25I0.75', '(ThMA)2FA4Pb5I16', 'Cs0.21FA0.56MA0.23PbBr0.06I2.94', '(BDA)MA3Pb4I13', 'Cs0.15FA0.8Rb0.05PbI3', 'MABa0.1Pb0.9I3', '(PEA)xCs0.15FA0.64MA0.2PbBr0.6I2.4', 'Cs0.05MA0.95Pb0.95Sn0.05Cl0.1I2.9', 'Cs0.08FA0.81MA0.12PbBr0.35I2.65', 'Cs0.3FA0.6MA0.1PbBr0.095I0.905', 'GU0.2MA0.8PbI3', 'FA0.5MA0.5PbI4', 'FA0.67MA0.33PbBr0.5I2.5', 'Cs0.05FA0.7885MA0.1615PbBr0.1I0.9', 'FA0.9MA0.1PbBr0.3I2.7', '(NH4)6.8FA0.15MA2.04Pb7.8Br0.45I24.14', 'FA0.65K0.2MA0.15PbBr0.55I2.55', 'FA0.625MA0.935PbI', '(HEA)2Cs1.9FA17.1Pb20Br6.3I56.7', 'FA0.975MA0.025PbI3', 'Cs0.02Pb0.98Br1.96I1.04', 'FA0.95MA0.05PbBr0.15I2.85 | OA2PbI4', 'FA0.95MA0.05PbBr0.15I2.85', 'CsLa0.03Pb0.97BrI2', 'FA0.75MA0.15PbBr0.45I', 'Cs0.05FA0.79MA0.160Pb1.0Br0.3I2.7', '(PEA)2MAPbI4', 'Cs0.25FA0.75PbBrI2', 'Cs0.94Rb0.06PbBr3', 'Cs0.05FA0.15MA0.8PbI3', '(NH4)3.4FA0.15MA2.04Pb4.4Br0.45I13.94', 'CsPb0.997Zn0.003Br3', 'CsBi3I10', 'Cs0.15FA0.75MA0.1PbBr0.3I2.7', 'CsPb0.3Sn0.7I3', 'Cs0.15FA0.85SnI3', 'Cs0.06FA0.79MA0.15PbBr0.45I2.55', 'CsCu0.01Pb0.99Br3', 'FA0.17MA0.83PbBr1.5I1.5', 'Cs0.15FA0.85PbBr0.25I2.75', 'CsPb0.75Sn0.25Br2I', 'MAPbCl3', 'FA0.14MA0.86PbBr0.42I2.58', 'MAHgI3', 'Cs0.23MA0.77PbI3', 'GUPb(SCN)1.8I1.2', '(DMA)0.075MA0.925PbI3', '(PEA)2Cs39Pb40Br40.33I80.67', '(TFEA)2Cs0.225FA7.425MA1.35Pb10Br4.65I26.35', 'FA0.9GU0.1SnI3', 'MA3Bi2I13', 'Cs3Bi2I9', 'BA2PbI4', 'FA0.7MA0.3PbI3', 'Cs0.1FA0.9PbBr0.9I2.1', 'FASnBrI2', 'FAPb0.375Sn0.625I3', 'Cs0.2FA0.8PbBr0.32I2.68', 'Cs0.2FA0.6MA0.2PbBr0.256I0.744', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | PA2PbI4', 'MAPb0.98Zn0.02I3', '(n-C3H7NH3)PbCl3', '(PTA)2MA3Pb4I13', 'Cs0.2FA0.8PbBr0.256I0.744', 'Cs0.1MA0.9Pb0.25Sn0.75I3', '((CH3)3S)2SnBrI5', 'MAPbI3 | FAPbBrI2', 'FA0.83MA0.17PbBr0.51', 'FA0.1MA0.9PbBr0.3I2.9', 'BA2CsPb2I7', 'MAPbI3 | (BEA)PbI4', 'Cs0.10MA0.90PbI3', 'CsPb0.97Sr0.03Br3', 'FA0.67MA0.33PbBr0.33I2.67', 'Cs0.05FA0.83MA0.12PbBr0.5I2.5', 'CsBi0.04Pb0.96I3', '(TBA)0.3Cs0.04FA0.55MA0.11PbBr0.51I2.49', 'MAPbI3 | (MIC1)2PbI4', 'Ag3BiI3(SCN)3', 'FAPbBr0.6I2.4', 'Cs0.003Pb0.997Br3', 'CsNi0.005Pb0.995Br3', 'Cs0.05FA0.788GU0.032MA0.129PbBr0.51I2.49', 'Cs0.07FA0.73MA0.20PbBr0.47I2.53', 'FA0.02MA0.98PbI3', 'FA0.975MA0.025PbBr0.075I2.925', 'Cs0.17FA0.83Pb0.98Sn0.02I3', 'MAPb0.95Sn0.05I3', 'Cs0.10FA0.81MA0.09PbBr0.03I2.97', '(NH4)6.8FA0.15MA1.7Pb7.8Br0.45I23.8', 'FAPbBr0.25I2.75', 'MAPb0.2Sn0.8I3', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | (oFPEA)PbI3', 'MAPb1.0Br0.66I2.33', 'BA2FA3Sn4I13', 'MAPbBr0.11I2.89', 'Cs0.10MA0.90Pb(SCN)0.15I2.85', '(PEA)2Cs99Pb100I301', 'MAPbBr0.51I2.49', 'FA0.85MA0.15PbBr0.45I2.45', 'FA0.88MA0.12PbI3', 'Cs0.05FA0.79MA0.16Pb1.0Br0.51I2.49', 'MAPbI3 | (MIC3)2PbI4', 'FAPbBr0.095I0.905', 'FAPb0.4Sn0.6I3', 'MAPb0.8Sn0.2I3', 'MAPb0.75Sn0.25Br0.9I2.1', 'Ag2CsSb2I3', 'Cs0.17FA0.83PbBr1.5I1.5', 'Cs0.08MA0.92PbBr0.24I2.76', 'Cs0.05FA0.70MA0.25PbI3', 'MAPb0.4Sn0.6I3', 'MAPb0.97Zn0.03I3', 'MACu0.05Pb0.95Br0.1I2.9', 'FA0.85MA0.15Pb0.45I2.55 | (NH4)8FA2.4Pb9I28.4', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | (PEI)2PbI4', 'Cs0.02FA0.82MA0.16PbBr0.51I0.249', 'Cs0.88FA0.12PbI3 | Cs0.88FA0.12PbIx(PF6)x', 'FA0.97MA0.03PbBr2.91I0.09', 'FA0.976MA0.024PbBr0.075I2.925', 'CsPbBr0.6I2.4', 'FA0.94MA0.6PbBr0.06I2.94', '(PEA)0.5MA0.5PbI3', 'FA0.67MA0.33PbI3', 'Cs0.17FA0.83PbBr0.45I2.55', 'FA0.94MA0.06PbBr0.06I', 'FA0.9MA0.1PbBr0.1I2.9', 'Cs0.05FA0.81GU0.025MA0.11PbBr0.39I2.61', '(PEA)2Cs59Pb60Br181', 'FA0.85MA0.15PbBr0.03I2.97', 'HDABiI5', 'CsPb0.93I3', 'FA0.33PbBr3', 'MAPbBr0.075I2.925', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | (PEA)PbI3', 'CsCu0.03Pb0.97Br3', '(HEA)2Cs1.9FA17.1Pb20Br0I63', 'FA0.6MA0.4Pb0.4Sn0.6Br0.18I2.82', '(PyrEA)PbI4', 'BA2MA3Pb4.0I13', 'FA0.1MA0.9Pb0.9Sn0.1I3', '(TBA)0.2Cs0.04FA0.63MA0.13PbBr0.51I2.49', 'BA2Cs1.5MA2.85Pb4I13', '(NH4)3.4FA0.15MA1.7Pb4.4Br0.45I13.6', '(PEA)0.4MA0.6PbI3', 'FA0.58MA0.42PbI3', 'Cs0.05FA0.7MA0.25PbI3', 'Cs0.05FA0.85MA0.15PbBr0.75I2.25', 'FA0.6MA0.4PbSn0.6I0.4', 'CsGe0.5Sn0.5I3', 'CsPbBrF0.22I1.78', 'GAMA4Pb4I13', 'BA2MA10Pb11I34', 'EA3MA2NEAPb4I13', 'MA2PbI4', 'Cs0.15FA0.85PbBr0.15I2.85', 'MASb1.8Sn0.2I9', 'FA0.75MA0.25Ge0.05Sn0.95I3', 'CsPb0.25Sn0.75Br2I', 'Cs0.2FA0.66MA0.14PbBr0.5I2.5', 'Cs0.0664FA0.666MA0.2668PbBr0.095I0.905', 'FA0.83MA0.17PbI3', 'MAPb1.0Br0.45I2.55', 'Cs0.05FA0.80MA0.15PbI3', 'Cs0.1MA0.90PbBr0.3I2.70', 'FA0.85MA0.15PbI5', 'Cs0.06FA0.78MA0.16PbI3', '(PEA)0.1MA0.9PbI3', 'Cs0.84K0.16PbBr3', 'Cs0.17FA0.83Pb0.99999Sn0.00001I3', '(BDA)MAPb2I7', 'FA0.2MA0.8PbBr0.3I2.9', 'BDACs3Pb4Br3.9I9.1', 'Cs0.1FA0.747MA0.153PbBr0.17I0.83', 'FAPbBr1.5I1.5', 'MA0.03Mg0.03Pb0.97I3', '(1.3-Pr(NH3)2)0.5Pb1.0I3', 'FA0.83MA0.17PbBr0.6I2.4', 'Cs0.20MA0.80PbI3', 'FA0.1MA0.9PbBrI2.8', 'MAPb0.75Sn0.25Br0.6I2.4', 'Cs0.15MA0.85PbI3', '(NH4)6.8FA0.15MA1.275Pb7.8Br0.45I23.375', 'CsPbBr0.3I2.7', 'FA0.84MA0.16PbBr0.50I2.50', 'Cs0.07FA0.775MA0.145PbBr0.45I2.55', 'FA0.7MA0.3PbBr0.3I2.9', '(pF1PEA)2MA4Pb4I13', '(ALA)2MA3Pb4I13', 'FA0.9MA0.1PbBr0.095I0.905', 'CsHA2Pb2I7', 'Cs0.175FA0.75MA0.075PbBr0.33I2.67', 'MA2PA8Pb9I28', '(IEA)2MA2Pb2I7', 'FAMAPbI3', 'FA0.3MA0.7PbBr0.45I2.55', 'MAPb0.97Sn0.03Br0.06I2.94', 'MAPb0.6Sn0.4Br0.4I2.6', 'MAPb0.4Sn0.6Br2.1I0.9', 'Cs0.25FA0.75PbBr0.3I2.7', 'Cs0.25FA0.75PbI3 | CsPbI3', 'Cs0.1FA0.9PbBr3', '(TEA)2MA3Pb4I14', 'Cs0.06FA0.77MA0.17PbBr0.17I0.83', 'Cs0.05FA0.79MA0.16Pb0.84Sn0.84Br0.52I2.48', 'FA0.85MA0.15PbBr0.15I2.85', 'Cs0.08FA0.92SnI3', 'Cs0.05FA0.5MA0.45Pb0.5Sn0.5I3', 'CsPbBr0.2I2.8', 'HA2MAPb2I7', 'CsPbBr1.5I1.5 | FAPbBr1.5I1.5', 'Cs0.05FA0.788GU0.129MA0.032PbBr0.51I2.49', 'Cs0.15MA0.85PbBr0.45I2.55', '(Anyl)2PbI3', 'MAPb0.93Sb0.03I3', 'MAPb0.25Sb0.75I3', 'Cs0.5FA0.5PbBr0.51I2.49', 'MA0.75Sn0.25I3', 'Cs0.07FA0.81MA0.12PbBr0.39I2.61', '(Cl-PEA)2MA3Pb4I13', 'MAPbBr0.06I2.24', 'Cs0.05FA0.83MA0.12PbBr0.49I2.51', 'Cs0.05FA0.8MA0.15PbBr0.15I0.85', 'MAPb0.75Sn0.25Br0.3I2.7', 'Cs0.45FA0.55PbBr0.15I2.85', 'MAHg0.2Pb0.8I3', 'CsBa0.4Pb0.6BrI2', 'Cs0.05MA0.95PbBr1.2I1.8', 'Cs0.05FA0.28MA0.67PbI3', 'FA0.8MA0.15PbBr0.45I2.55', 'Ag4Bi7I25', 'FA0.82MA0.18PbBr0.53I2.47', 'FA0.95GU0.05SnI3', 'Cs0.17FA0.83MAPbBr2.59I0.51', 'FA0.8MA0.2PbBr0.6I2.4', 'FA0.81MA0.19PbBr0.5I2.5', 'AgCs1.9Rb0.1BiBr6', 'FA0.5MA0.5PbBr0.45I2.55', 'FA0.5MA0.5Pb0.5Sn0.5I3', 'AgCs2BiBr5.5', 'CsPbBr0.15I2.85', 'Cs0.1MAPbBr0.45I2.55', 'FA0.4MA0.6PbBr0.1I2.9', 'MAPb0.96Sb0.04I3', 'FA0.85MA0.15Pb0.6Sn0.4Br0.45I2.55', 'FABi3I10', 'FA0.285GU0.05MA0.665PbI3', 'Cs0.96Li0.04PbBr3', 'CsPb0.995Zn0.005Br3', 'Cs0.1FA0.2MA0.7PbI3', 'MASnCl3', 'MAPb0.9Sb0.1I3', 'FA0.25MA0.75PbI', 'FA0.5MA0.5PbBr0.25I2.75', 'MAPbBr1.2I1.8', '(GABA)0.5MA0.5PbI3', 'MAPb(BF4)2.80I0.2', 'EA0.3MA0.7PbI3', 'Cs0.05FA0.80MA0.15PbBr0.51I2.49', 'Cs0.07FA0.78MA0.15PbBr0.51I2.49', 'Cs0.225FA0.75MA0.025PbBr0.33I2.67', 'FA0.3MA0.7PbI3', 'MAPbBr0.6Cl2.4', 'HA2PbI4', 'Cs0.05FA0.8MA0.15PbBr0.5I2.5', 'FA0.5MA0.5PbBr1.5I1.5', 'Cs0.05FA0.8MA0.15PbI3', 'Cs0.05FA0.19MA0.76PbI3', 'Cs0.24FA0.76PbI3', 'FA0.83MA0.17PbBr2I', 'Cs0.05FA0.788GU0.065MA0.097PbBr0.51I2.49', 'Cs0.1FA0.75MA0.15PbBr0.5I2.5', '((CH3)3S)2SnCl2I4', 'Cs0.06FA0.67MA0.27PbBr0.3I2.7', 'Cs0.1FA0.7MA0.2Pb0.5Sn0.5I3', '(5-AVA)2FA4Sn5I16', 'FA0.81MA0.15PbBr0.45I2.51', 'FA0.6MA0.4Pb0.6Sn0.6I3', 'MAPbBr2.1I0.9', 'Cs0.05FA0.875MA0.075PbBr0.225I2.775', 'FA0.75MA0.25PbBr0.25I2.75', 'MAPb(BF4)2.90I0.1', 'Cs0.02FA0.37MA0.61PbBr0.04I2.96', 'Cs0.06FA0.78MA0.16PbBr0.54I2.46', 'GUMA3Pb3I10', 'FAPb0.2Sn0.8I3', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | (CH3)3SPbI3', 'MACu0.05Pb0.8Sn0.15Br0.1I2.9', 'Cs0.2FA0.8PbBr0.095I0.905', 'Cs0.05FA0.79MA0.16Pb0.832Sn0.168Br0.52I2.48', 'Cs0.15FA0.85PbBr0.9I2.1', 'Cs0.04FA0.81MA0.14PbBr0.43I2.57', 'CsGe0.1Pb0.9BrI2', 'Cs0.05FA0.81MA0.14PbBr0.43I2.57', 'FA0.5MA0.5PbBr2I', 'FABi0.05Pb0.95I3', 'FAPbBr0.05I2.95', 'Cs0.05MA0.95PbBr0.3I2.7', 'Cs0.05FA0.7885MA0.1615PbI3', 'Cs0.75MA0.25PbI3', 'MAPbBr0.36I2.64', 'Cs0.15FA0.76MA0.09PbBr0.03I2.97', 'Cs0.12FA0.5MA0.38PbBr0.96I2.04', 'FA0.83MA0.17PbBr0.0I2.51', 'Cs0.3FA0.2MA0.5PbI3', '(GABA)0.05MA0.95PbI3', 'MABiI2', 'Cs0.17MA0.83PbI3', 'MASnBr1.2I1.8', 'MAPbIx', 'MAPb1.0ClI2', 'BA2MA34Pb5I15', 'Cs0.05MA0.95Pb0.95Sn0.05Br0.1I2.9', 'Cs0.05FA0.85MA0.15PbBr0.45I2.55', '(3AMP)MA2Pb3I10', 'CsGe0.2Pb0.8BrI2', '(NEA)0.2BA1.8MA3Pb4I13', 'Cs0.05FA0.86MA0.09PbBr0.3I2.7', 'CsCu0.005Pb0.995Br3', 'MAPbBr0.12I2.88', 'FA0.93MA0.07PbBr0.21I2.79', '(NMA)2MA59Pb60I181', '(PEA)2MA5Pb4Cl2I10', 'FA0.4MA0.6PbBr0.2I2.8', '(PBA)1.5BA0.5MA3Pb4I13', 'MAPb1.0Br1.5I1.5', 'AgBiI7', 'FA0.75MA0.25Ge0.1Sn0.9I3', 'CsPb0.97Sm0.03Br3', 'Cs0.05FA0.83MA0.17PbBr0.51I2.49', 'FA0.83MA0.17PbBr0.22I2.78', 'FA0.75MA0.25PbBr0.51I2.49', 'FA0.33PbI3', 'FA0.85MA0.15PbBr0.6I2.4', 'Cs0.05FA0.83MA0.17PbBr0.36I2.64', 'FAPbBr3', 'FA0.3MA0.7PbBr0.15I2.85', 'FA0.33MA0.66Pb0.66Sn0.33I3', 'BAFA60Pb61Br4I180', 'MABiI3', 'AgCs2BiBr5.4', 'FA0.95MA0.05PbBr0.15I2.85 | (HTAB)0.03FA0.95MA0.05PbBr0.15I2.85', 'Cs0.05FA0.75MA0.11PbBr0.39I2.61', 'Cs0.1MA0.9PbBrI2', '(PGA)2MA3Pb4I13', 'Cs0.3FA0.7PbBr3', 'MAPbBr1.74I1.26', 'Cs0.05FA0.7885MA0.1615PbBr0.3I2.7', 'Cs0.05FA0.788GU0.162PbBr0.51I2.49', 'MAPbBr1.77I1.23', 'AgCs2BiBr5.8', 'Cs0.05FA0.83MA0.17PbBr0.17I0.83', 'Cs0.1MA0.9Pb0.5Sn0.5I3', 'Cs0.3Ag3Bi2.2I9', 'CsPbBr0.31I2.69', 'Cs0.15FA0.72MA0.13PbBr0.45I2.55', 'MAPbBr1.7I1.3', 'FAPbBr0.3I2.7', 'FA0.33PbBrI2', 'Cs0.1FA0.27MA0.63PbI3', 'Cs0.15FA0.75MA0.1PbBr0.7I2.3', 'CsPb0.5Sn0.5Br2I', 'Cs0.98Li0.02PbBr3', 'MAPbBr1.8I1.2', 'MAPb0.95Zn0.05I3', 'GUPbI3', 'Cs0.1FA0.79MA0.16PbBr0.51I2.49', 'Cs0.05FA0.8075MA0.1425PbBr0.51I2.49', 'FA0.8TN0.2SnI3', 'FA0.45MA0.55PbI3', 'Cs0.05FA0.79MA0.16PbBr0.54I2.46', 'FA0.83MA0.17PbBr3', 'Cs0.1FA0.765MA0.135PbBr0.45I2.55', 'Cs0.08FA0.76MA0.16PbBr0.51I2.49', 'BA2PbI3', 'IM0.2MA0.20.8PbI3', 'Cs0.1FA0.75MA0.15PbBr0.55I2.55', 'MAPb1.0Br0.15I2.85', 'nanBiI3', 'MACu0.014Pb0.986I3', 'CsBa0.03Pb0.97Br3', '(3AMP)FA0.6MA2.4Pb4I13', 'MAGeBr0.6I2.4', 'MAPb0.95Sb0.05I3', 'Cs0.04FA0.80MA0.16PbBr0.50I2.50', 'MAAlCl4', 'AgCs2BiBr5.7', 'GU0.075MA0.925PbI3', 'Cs0.02FA0.84MA0.14PbBr0.05I2.95', '(5-AVA)0.1MA0.9PbI3', 'Cs0.2FA0.6MA0.2PbBr0.33I2.67', 'FA0.5MA0.5PbBr0.5I2.5', 'Cs0.1MA0.9Pb0.75Sn0.25I3', 'Cs0.05FA0.81MA0.14Pb0.95Br0.43I2.42', 'AgCs2BiBr6', '(PEA)2MA29Pb30I91', 'MAPb0.7Sn0.3I3', 'FA0.07MA0.93PbI3', 'Cs0.05FA0.80MA0.15PbBr0.25I2.75', 'MAPb1.0I3', 'MAPb0.6Sn0.4I3', 'Cs0.15FA0.85PbBr0.6I2.4', 'Cs0.1FA0.9PbBr0.1I2.9 | (PEA)PbBr0.1I2.9', 'EA2MA4Pb5I16', 'MA0.01Pb0.99I3', 'IM0.05MA0.050.95PbI3', 'FA0.67MA0.33PbBr1.5I1.5', 'Cs0.05FA0.81MA14PbBr2.55I0.45', 'FA0.83MA0.17PbBr0.5I2.5', 'Cs0.1FA0.7MA0.2PbBr0.2I2.8', 'FAPbCl0.45I2.55', 'FA0.29MA0.71PbBr0.3I2.7', 'FAPbBr', 'BA2MA3Pb3SnI13', '(PDA)MA2Pb3I10', '(PEA)2Cs39Pb40I121', 'FA0.6MA0.4PbI3', 'Cs0.16FA0.8MA0.04PbBr1.83I1.17', 'SrTiO3', 'FA0.66MA0.33PbBr0.45I2.55', '(GABA)0.025MA0.975PbI3', 'Cs0.06FA0.8MA0.14PbBr0.45I2.55', 'FA0.15MA0.85PbBr2.55I0.45', 'FA0.85MA0.15PbBr0.451I2.55', '(BZA)2MA2Pb3I10', 'MAPbI3 | Cs1Pb1I3 | Cs1Pb1Br0.3I2.7 | Cs1Pb1Br0.7I2.3 | CsPbBrI2', '(PEA)2Cs9Pb10Br10.33I20.67', 'FAPbBr0.1I2.9', 'Cs0.2FA0.24MA0.56PbI3', 'CsPbBrI2 | CsPbBrI2', 'MAPb0.5Sn0.5I3', '(3AMPY)MA3Pb4I13', 'Cs0.05FA0.5MA0.5PbBr1.5I1.5', 'Cs0.5FA0.5PbI3 | CsPbI3', 'MAPbI3 | FAPbBr1.5I1.5', '(THM)0.025MA0.975PbI3', 'Cs0.17FA0.75MA0.08PbBr0.39I2.61', '(6-ACA)0.038MA0.962PbI3', '(PEA)2PbI4', 'FA0.83MA0.17PbBr0.47I2.53', 'Cs0.15Ag3Bi3I9', 'MAPbBr1.6I1.4', 'MAPb0.97Sr0.03I3', '(BDA)MA2Pb3I10', 'FA0.6MA0.4Pb0.4Sn0.6Br0.12I2.88', 'Cs0.05FA0.83MA0.12PbBr0.51I2.49', '(PEI)2PbI4', 'Cs0.17FA0.83Pb0.999Sn0.001I3', 'Cs0.08FA0.09PbBr3', 'Cs0.05FA0.7885MA0.1615PbBr0.4845I2.5155', 'Cs0.05FA0.85MA0.5PbBr0.25I2.75', 'MAPbI3 | CA2PbI4', 'Ag4Bi5I19', 'MAPb0.8Sn0.2Br0.4I2.6', 'FA0.92MA0.08PbBr0.24I2.76 | (C8H17NH3)2PbI4', 'MAPb0.75Sn0.25Br1.5I1.5', 'FA0.4MA0.6PbBr0.3I2.7', 'CsSnBr2.7I0.3', 'MACo0.008Pb0.992I3', '(OdA)PbI4', 'Cs0.1FA0.9PbBr0.135I2.865', '(PEA)2MA4Pb5I16', 'CsPb0.94Zn0.06BrI2', 'FA0.05PN0.95SnI3', 'FA0.97MA0.03PbBr0.09I2.91 | (MIC3)2PbI4', 'CsPb0.97Sr0.03BrI2', 'MAPbBrI2', '(CPEA)2MA3Pb4I13', 'Cs0.02FA0.15PDA0.82PbI3', '(NH4)6.8FA0.15MA0.85Pb7.8Br0.45I22.95', 'MA4PbI6', 'Cs0.05FA0.8075MA0.1425PbBr0.3I2.7', '(PEA)0.4BA1.6MA3Pb4I13', 'MAEu0.06Pb0.94I3', '(NH4)6.8FA0.15MA2.21Pb7.8Br0.45I24.31', 'MAPb0.4Sn0.6Br1.5I1.5', 'BA0.15FA0.85SnI3', 'FA0.95MA0.05PbBr01.51I2.85', 'Cs0.05FA0.85MA0.1PbBr0.1I2.9', 'MAPbBr2I', 'Cs0.06FA0.79MA0.15PbBr0.51I2.49', 'Cs0.08FA0.92PbI3', 'Cs0.75FA0.25PbI3 | CsPbI3', 'FA0.83MA0.17PbBr0.49I2.51', 'MAPbBr0.5I2.5', 'Cs0.1665FA0.667MA0.1665PbBr0.33I2.67', '(EDA)0.005FA0.3MA0.695Pb1.0I3', 'GU0.025MA0.975PbI3', 'Cs0.08FA0.78MA0.14PbBr0.42I2.38', '(CHMA)2MAPb2I7', '(NH4)5.1FA0.15MA2.04Pb6.1Br0.45I19.04', 'Cs0.1FA0.6MA0.3PbBr0.256I0.744', 'FA0.85MA0.15Pb0.45I2.55', 'FA0.75MA0.25PbBr0.25I2.77', 'Cs0.05FA0.81MA0.15PbBr0.45I2.55', 'Cs0.15FA0.85Pb0.375Sn0.625I3', '(NH4)8.5FA0.15MA1.7Pb9.5Br0.45I28.9', 'Cs0.2FA0.8PbBr0.16I2.84', 'Cs0.05FA0.89MA0.6PbBr0.06I2.94', '(BZA)2PbI4', '(PEA)2CsPb2I7', 'Cs0.05FA0.28MA0.67PbBr0.54I2.46', '(PEA)2Cs7Pb8I25', 'MAPb0.75Sn0.25Br2.1I0.9', '(NH4)3Sb2I9', 'EA2MA6Pb7I22', 'CsSnBr0.6I2.4', 'MAPb0.995Sb0.005I3', 'EA2MA5Pb6I19', 'MA3PbCl2', 'Cs0.88Rb0.12PbBr3', 'Cs0.133FA0.733MA0.133PbBr0.256I0.744', 'FA0.67MA0.33PbBr3', 'MAPbBr1.41I1.59', 'MAPb0.7Sn0.255I3', 'Cs0.1FA0.7MA0.2PbBr0.5I2.5', '(PEA)2Cs9Pb10I31', '(PEA)MAPbI3', 'Cs0.05FA0.75MA0.2PbBr0.51I2.49', 'FA0.6MA0.4Pb0.4Sn0.6I3 | (PEA)2Pb0.4Sn0.6I4', '(PDMA)FA2Pb23I7', '(Br-PEA)2MA2Pb3BrI10', 'CsBi9I28', 'Cs0.05FA0.788MA0.162PbBr0.51I2.49', 'BA2FA0.6MA2.4Pb4I13', 'FA0.83MA0.17PbBr0.3I2.7', 'Cs0.2MA0.8Pb0.5Sn0.5I3', '(DMA)PbI3', 'AN0.015MA0.985PbI3', 'Cs0.91Na0.09PbBr3', 'BA2MA2Pb4I13', '(Anyl)2MAPb2I6', 'FA0.75MA0.25PbBr0.24I2.76', 'FA0.83MA0.17PbBr0.37I2.63', 'Cs0.05FA0.79MA0.16PbBr0.48I2.52', 'Cs0.07FA0.93PbBr0.15I2.85', 'Cs0.05FA0.84MA0.11PbBr0.1I2.9', 'Cs0.01MA0.99PbBr0.03I2.97', 'FAPbBr0.5I2.5', 'MAPbBr0.45I2.55', 'Cs0.17FA0.83Pb0.7Sn0.3I3', 'Cs0.1FA0.75MA0.13PbBr0.45I2.55 | (A43)2PbI4', 'Cs0.8FA0.69MA0.23PbBr0.9I2.1', 'MAPb(Br0.7I0.3)xCl3-x', 'MAPb0.75Sn0.25Br2.7I0.3', 'Cs0.1FA0.76MA0.14PbBr0.51I2.49 | (EPA)2PbI4', 'CsSnBr1.5I1.5', 'Cs0.17FA0.83Pb0.99Sn0.01I3', 'Cs0.05FA0.788MA0.162PbBr0.3I2.7', 'MAPbBr0.84I2.16', 'Cs0.05FA0.46MA0.49PbBr0.12I2.88', 'GU0.15MA0.85PbI3', '(CH3ND3)PbI3', '(F3EA)0.12BA1.88MA3Pb4I13', '(PEA)2MA5Pb6I19', 'MAPb0.99Sb0.01I3', 'Cs0.14FA0.86PbI3', 'MAPbBr0.48I2.52', '(CHMA)2Cs4MA35Pb40I121', 'MAPb1.0Br3', 'FA0.11MA0.89PbI3', 'CsPbBr3 | FAPbBr2Cl', 'MAPbI3 | Cs1Pb1I3 | CsPbBr0.3I2.7', 'FA0.35MA0.65PbBr0.13I2.94', 'FA0.8MA0.2PbBr0.2I0.8', 'CsPbI3 | CsPbI3', 'MAPbI3 | BAPbI4', 'MASb1.9Sn0.1I9', 'Cs0.05FA0.57MA0.38PbI3', 'MAPbBr0.18I2.82', 'MA3PbI3', 'Cs0.02FA0.79MA0.16PbBr0.551I2.49', 'FA0.9MA0.1PbBr0.2I2.7', 'CsPbBrF1.72I0.28', '(PDA)MAPbI4', 'Cs0.1FA0.75MA0.15Pb0.75Sn0.25Br0.5I2.5', 'FAMAPbBr0.45I2.55', 'MAPbBr0.21I2.79', 'K3Sb2I9', 'CsPbBr0.09I2.91', 'FA0.75MA0.25PbBr0.25I2.80', 'AgBi2I7', 'Cs0.05FA0.9Rb0.05PbI3', 'FA0.84MA0.16PbBr0.4I2.6', 'Cs0.07FA0.89GU0.02PbI3', '(HEA)2Cs1.9FA17.1Pb20Br3.15I59.85', 'Cs0.05FA0.79MA0.16PbBr0.65I2.35', 'FA0.85MA0.85PbI3', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49 | HAPbI4', 'Cs0.17FA0.83PbBr0.6I2.4', 'FAPb0.7Sn0.3I3', '(BDA)PbI4', 'Cs0.05FA0.76MA0.16PbBr1.5I1.5', 'Cs0.15FA0.75MA0.1PbBr0.2I2.8', 'FA0.85TN0.15SnI3', 'MAPb0.625Sn0.375I3', 'MAPb0.99Zn0.01I3', 'LaYS3', 'FA0.8K0.05MA0.15PbBr0.55I2.55', '(5-AVA)0.03MA0.97PbI3', 'MAPbBr0.225I2.775', 'Cs0.85Rb0.15PbBr3', 'FA0.5MA0.5PbBrI2', '(CIEA)0.1MA0.9PbI3', '(ThMA)MA4Pb5I16', 'FABiPbI3', 'Cs0.2FA0.8Pb0.25Sn0.75I3', 'FAPb0.875Sn0.125I3', 'FA0.46MA0.64PbBr0.24I0.76', '(TFEA)2Cs0.475FA15.675MA2.85Pb20Br9.15I51.85', '(TBA)0.15Cs0.04FA0.67MA0.14PbBr0.51I2.49', '(PEA)2Cs79Pb80Ix', 'FA0.97MA0.03PbBr0.09I2.91 | (MIC2)2PbI4', 'Cs0.09FA0.83MA0.08PbBr0.15I2.85', 'FA0.87MA0.13PbI3', 'MAIn0.20Pb0.80I3', 'nanSnI6', 'CsEu0.09Pb0.91BrI2', '(PMA)2MAPbI3', 'Ag4Bi9I31', 'Cs0.13FA0.72MA0.16PbBr0.51I2.49', 'MAPb0.4Sn0.6Br1.8I1.2', 'FA0.83MA0.17PbBr0.02I2.98', 'MACa0.03Pb0.97I3', 'Cs0.05FA0.85MA0.1PbBr3.0', 'FA0.15MA0.85PbBr0.45I2.55', 'MAPbBr1.14I1.86', 'Cs0.05FA0.79MA0.16PbBr0.5I2.5', 'MAMg0.09Pb0.91I3', 'MAHg0.15Pb0.85I3', '(APMim)Pb(PF6)PF63', 'FAPb0.125Sn0.875I3', 'CsLa0.01Pb0.99BrI2', 'CsPb1.0Br3', '(CHMA)2MAPbI4', 'CsPb0.98I3', 'FA0.79MA0.16PbBr0.51I2.49', 'MAPbBr2.97I0.03', 'MAPbBr0.3I2.7', 'Cs0.05FA0.8MA0.15PbBr0.4I2.6', '(PMA)2MA3Pb4I13', 'GUMA4Pb4I13', 'Cs0.03FA0.37MA0.6PbBr0.025I2.975', 'FA0.83MA0.17PbBr1.2I1.8', 'Cs0.2FA0.8PbBr0.2I2.8', 'MAPbBr0.87I2.13', 'FAPb0.9Sn0.1I3', 'MAGeBr0.45I2.55', 'FA0.75Sn0.25I3', 'FAPb0.5Sn0.5I', 'CsSnBr2I', 'MAPbBr0.14I2.86', 'Cs0.1FA0.75MA0.15PbBr0.51I2.49', 'Cs0.05FA0.85MA0.10PbBr0.39I2.61', 'Cs0.15FA0.71MA0.14PbBr0.9I2.1', 'FA0.4MA0.6PbI3', '(PEA)0.8MA0.5PbI3.2', 'MASnBr1.8Cl0.2I', 'MABi3I10', 'MAPbBr0.083I2.92', 'FA0.67MA0.33PbBrI2', 'Cs0.05FA0.54MA0.41PbBr0.06I2.94', 'Cs0.17FA0.83PbBrI2', 'Cs0.05FA0.8MA0.15PbBr0.47I2.53', 'FA0.87MA0.13PbBr0.13I2.83', '(AVA)2PbI4 | MAPbI3', 'Cs0.05FA0.76GU0.075MA0.11PbBr0.39I2.61', 'Cs0.1FA0.7MA0.2PbI3', 'FA0.8MA0.2PbBr0.256I0.744', 'CsPb0.95I3', 'Cs0.1FA0.81MA0.09PbBr0.1I2.9', 'Cs2SnI6', 'FA0.85MA0.15PbBr0.46I2.54', 'FA0.75MA0.25Sn1.0I3', 'Cs0.1FA0.9SnI3', 'GU0.75MA0.25PbI3', '(3AMP)FA0.45MA2.55Pb4I13', 'EA2MA3PEAPb4I13', '(DMA)0.025MA0.975PbI3', 'Cs0.05FA0.76MA0.19PbBr0.6I2.4', '(BDA)MAPbI4', 'Cs0.04FA0.8MA0.16PbBr0.45I2.55', 'Cs0.1FA0.75MA0.15Pb0.5Sn0.5Br0.5I2.5', 'FA0.5MA0.5PbI', '(TBA)0.5Cs0.05FA0.75MA0.15PbBr0.51I2.49', 'Cs0.05FA0.79MA0.16PbBr0.49I2.51', 'FA0.59MA0.41PbI3', 'BAFA40Pb41I124', 'MASb2I9', 'Cs0.02FA0.98SnI3', 'Cs0.05FA0.68MA0.26Pb0.75Sn0.25Br0.4I2.6', 'FA0.79MA0.21PbI3', 'FA0.10PN0.90SnI3', 'FA0.2MA0.8PbBr2.4I0.6', 'CsPb0.1Sn0.9I3', 'MACu0.03Pb0.97Br0.03I2.97', 'Cs0.08FA0.78MA0.14PbBr0.42I2.58', 'Cs0.15FA0.71MA0.14PbBr0.6I2.4', 'Cs0.06FA0.78MA0.16PbBr0.18I2.82', 'FAPbBr2.25Cl0.75', 'MAPb0.95I3', 'BE2FA9Pb9I28', 'MAPb0.4Sn0.6Br0.4I2.6', 'FA0.83MA0.17PbBr1.5I1.5', 'Cs0.04FA0.8MA0.16PbBr0.15I0.85', '(PEA)0.67MA0.33PbI3', 'Ag0.152Bi3I9.5', 'HA0.1MA0.9SnI3', 'FA0.25MA0.75SnI3', '(Ace)0.02MA0.98PbI3', 'FA0.0MA0.13PbBr0.13I2.87', 'FA0.5MA0.5PbBr2.5I0.5', 'FA0.072MA0.928PbBrI2.86', 'Cs0.05FA0.79MA0.16PbBrI2', 'Cs0.03Pb0.97Br3', 'BA0.5MA0.75PbI3.25', 'FA0.4MA0.6PbBr0.025I2.975', 'Cs0.05FA0.55MA0.4PbBr2.88I0.12', 'FA0.7Rb0.3PbI3', 'FA0.88MA0.12PbBr0.36I2.64', 'FA0.4MA0.6PbI4', 'FAPbBr2.1Cl0.9', 'MA3Sb1.4Sn0.6I9', 'Cs0.0125FA0.4875MA0.5Pb0.5Sn0.5I3', '(PDMA)MA5Pb6I19', 'BA2MAPb2I6', 'MA0.01Mg0.01Pb0.99I3', 'Cs0.1FA0.9PbBr0.1I2.9', 'Cs0.05FA0.8MA0.15PbBr0.15I2.85', 'CuBiI4', 'CsPb0.9I3', 'GUMA3Pb3I12', 'Cs0.05FA0.79MA0.16SnBr0.5I2.5', 'MAPb0.97Sb0.03I3', '(CHMA)2MA4Pb5I16', 'BA2MA39Pb40I121', '(PDA)0.01Cs0.15FA0.84PbI3', 'MAPb(BF4)2.95I0.05', 'MA3PA2Pb4I13', 'Cs0.05MA0.95PbBr0.6I2.4', 'Cs0.01Pb0.99Br3', 'EA2PbI4', 'MAPb0.97I3', 'MAPbBr0.07I2.97', '(F3EA)0.04BA1.96MA3Pb4I13', 'Cs0.19FA0.81PbBr0.54I2.46', 'FA0.6MA0.4Pb0.4Sn0.6I3', 'MANi0.1Pb0.9I3', 'CsMg0.03Pb0.97Br3', 'BA2MA3Sn4I13', 'Cs0.05MA0.79PbBr0.3I2.7', 'Cs0.17FA083PbBr0.6I0.24', 'MAPbI3 | (PPA)PbI4', 'CsPb1.0Br1.2I1.8', 'Cs0.05FA0.83MA0.12PbBr0.45I2.55', 'FA0.83MA0.17PbBr0.33I2.67', 'MAPb0.4Sn0.6Br0.6I2.4', 'FA0.71MA0.29PbBr0.42I2.58', 'CsEu0.01Pb0.99BrI2', 'BA0.52GA0.15MA0.67PbI3.33', 'Cs0.3MA0.7PbI3', 'DA2FA3Sn4I13', 'FA0.024MA0.976PbBrI2.955', 'MABa0.03Pb0.97I3', 'BA2Cs0.1FA2.36MA0.48Pb3Br1.7I0.83', 'Cs0.05FA0.94MA0.01PbBr0.03I2.97', '(F3EA)0.2BA1.8MA3Pb4I13', 'Cs0.1FA0.9PbI3', 'BA2Cs0.08FA1.36MA2.56Pb5I16', 'FA0.62MA0.38PbBr0.13I2.90', 'MA2PA10Pb11I34', 'Cs0.2FA0.8SnI3', 'MAPb0.4Sn0.6Br2.4I0.6', 'FA0.8MA0.2PbI3', '(4ApyH)Bi0.2Sb0.8I4', 'Cs0.05FA0.75MA0.1PbBr0.3I2.9', 'FA0.75MA0.25PbBr0.25I2.76', 'Rb3Sb2I9', 'FA0.86MA0.15PbBr0.45I2.55', 'FA0.8MA0.2Pb0.5Sn0.5I3', '(iPA)3PbI5', 'Cs0.12FA0.88PbBr0.36I2.64', 'FA0.3MA0.7PbBr0.6I2.4', 'CsPbBr3 | MAPbI3', 'MAPb0.4Sn0.6BrI3', 'CsFAPbBrI', 'BA2FA2.4MA0.6PMAPbI13', 'FAPbBr2I', 'MAEu0.08Pb0.92I3', 'FA0.87MA0.13PbBr0.51I2.61', 'FA0.4MA0.6Pb1.0I3', 'FAPb1.0I3', 'FA0.92MA0.08PbBr0.24I2.76 | (C4H9NH3)2PbI4', 'FA0.73MA0.23PbBr0.13I2.89', 'FA0.6MA0.4PbBr0.256I0.744', 'FA0.8GU0.2SnI3', 'Cs0.005FA0.81MA0.14PbBr0.45I2.55', 'Ag0.15Bi4I11.5', '(HEA)2Cs3.9FA35.1Pb40Br12.3I110.7', 'CsFAPbBr0.2I2.8', '(C6H4NH2)CuCl2I', 'CsPbBr3 | CsPbBr3', 'Cs0.08FA0.76MA15PbBr0.51I2.49', 'Cs0.05FA0.8MA0.15PbBr0.42I2.58', 'CsPb0.99Zn0.01Br3', 'MAEu0.02Pb0.98I3', '(DMA)2PbI4', 'Cs0.04FA0.80MA0.16PbBr0.5I2.5', 'CsBi0.05Pb0.95I3', 'Cs0.05FA0.75GU0.075MA0.10PbBr0.39I2.61', 'MA2PA4Pb3I10', 'FA0.17MA0.83PbBr3', 'Cs2TiBr6', 'FA0.85MA0.85PbBr0.45I2.55', '(F5PEA)xCs0.15FA0.64MA0.2PbBr0.6I2.4', 'CsPbBrI2 | BA2CsPb2BrI6', 'Cs0.5MA0.5PbI3', 'Cs0.05FA0.79GU0.05MA0.11PbBr0.39I2.61', 'GUSnI3', 'FA0.3MA0.7PbBr0.3I2.9', 'Cs0.25FA0.75PbI3', 'Cs0.1FA0.85MA0.15PbBr0.45I2.55', 'MAPbBr0.81I2.19', 'EA2MA9Pb10I31', 'Cs0.13FA0.87PbBr0.39I2.61', 'Ba0.1K0.9Nb0.95Ni0.05O3', '(AVA)0.05MA0.95PbI3', '(DMA)0.1MA0.9PbI3', 'MA3Sb2I9', 'FAPb0.6Sn0.4I3', 'BA2FAPb2I7', 'FA0.17MA0.83PbBr2I', 'FA0.83MA0.17PbBr0.4I2.6', 'BDACsPb2Br2.1I4.9', 'FA0.5MA0.5PbBr0.12I2.88', '(PMA)2CuBr4', 'FA0.3MA0.7PbBr0.48I2.52', 'MAPb0.25Sn0.75Br1.2I1.8', 'MAPb0.99I3', '(PEA)2MA2Pb3I10', 'Cs0.8Rb0.2SnI3', 'MAPb0.65Sn0.35I3', 'FAPbBr3I', '(PEA)0.8BA1.2MA3Pb4I13', 'FA0.85MA0.15PbBr0.45I2.55 | BAFAPbI4', '(4AMPY)MA3Pb4I13', 'CsNi0.01Pb0.99Br3', '(HdA)PbI4', 'FA0.5MA0.5Pb0.75Sn0.25I3', 'FA0.6MA0.4PbBr0.3I2.9', 'MAPb1.0Cl3', '(PEA)2Cs59Pb60Br60.33I120.67', 'FA0.54MA0.46PbI3', 'Cs0.05FA0.855MA0.095PbBr0.285I2.565', '(BDA)MA4Pb5I16', 'CsPbBr1.9I1.1', '(PEA)2PbI4 | MA1Pb1I3 | (PEA)2PbI4', 'Cs0.11MA0.89PbI3', 'Cs0.06FA0.94PbBr3', 'MA2Pb(SCN)2I2', 'AgCs2BiBr5.6', 'MAPbI3 | BA2MA2Pb3I310', 'Cs0.06MA0.94PbI3', 'BiFeO3', 'Cs0.09FA0.91PbBr0.12I2.88', 'Cs0.05FA0.84MA0.11PbBr0.2I2.8', 'FA0.87MA0.13PbBr0.39I2.61', 'Cs0.1FA0.9PbBr0.1I2.9 | (PA)PbBr0.1I2.9', 'MAIn0.15Pb0.85I3', 'Cs0.05FA0.788GU0.097MA0.065PbBr0.51I2.49', 'FA0.6MA0.4Pb0.4Sn0.6Br0.3I2.7', 'FA0.05MA0.95PbI3', 'CsPbBr3 | CsPbBr2I', 'Cs0.35FA0.65PbI3', 'Cs0.05FA0.79MA0.16PbBr0.51I2.51', 'FA0.2MA0.8PbBr0.6I2.4', 'Cs0.06FA0.78MA0.16PbBr0.51I2.49', 'Cs0.02FA0.95MA0.03PbBr0.09I2.91', 'Cs0.15FA0.65MA0.20PbBr0.6I2.4', 'CsPbBr2I | CsPbI3', 'FA0.81MA0.19PbBr0.54I2.46', 'Cs0.1FA0.76MA0.14PbBr0.45I2.55', 'CsxFAxPbI3', 'Cs0.17FA0.83Pb0.9Sn0.1I3', 'MAPb0.75Sn0.25Br3', '(ImEA)PbI4', 'Cs0.01FA0.76MA0.14PbBr0.45I2.55', 'FA0.8MA0.2PbBr0.22I', 'Cs0.06FA0.79MA0.15PbBr1.8I1.2', 'Cs0.05FA0.5MA0.45PbBr0.04I2.96', 'IM0.005MA99.5PbI3', 'EA0.2MA0.8PbI3', 'Cs0.05FA0.76MA0.19PbBr0.57I2.32', 'CsPbBrF0.28I1.72', 'Cs0.10FA0.75MA0.15PbBr0.50I2.50', 'Cs0.05FA0.7885MA0.1625PbBr0.45I2.55', '(Anyl)2MA34Pb5I15', 'Cs0.5FA0.4MA0.1PbBr0.51I2.49', '(BdA)PbI4', 'FASnI3', 'IM0.5MA0.50.5PbI3', 'Cs0.05FA0.81MA0.14PbBr0.51I2.49', '(PEA)2FA0.85MA0.15Pb2Br1.05I5.95', 'FAPbBr0.35I2.65', 'Cs0.1FA0.75MA0.15SnBr0.5I2.5', 'Cs0.15FA0.255MA0.595PbI3', 'Cs3Sb2I9', 'MACo0.1Pb0.9I3', 'MASnBr0.6I2.4', 'FA0.33PbBr2.5I0.5', 'Cs0.80MA0.20PbI3', 'CsBa0.2Pb0.8BrI2', 'Cs0.16FA0.8MA0.04PbBr1.71I1.29', 'Cs0.3FA0.7PbBr0.095I0.905', 'FA0.75GU0.25SnI3', 'MAPb0.93I3', 'MASnBrI2', 'PAPbI3', 'MACu0.0094Pb0.9906I3', 'BA2Cs1.2FA7.65Pb4.8Sn3.6I28', 'Cs0.05FA0.76MA0.19PbI3', 'FAPbBrxIx', '(GABA)0.2MA0.8PbI3', 'Cs0.05FA0.81MA0.11PbBr0.45I2.55', '(Ace)0.05MA0.95PbI3', 'FAPbBr2Cl', 'MAPbBr0.3I2.9', 'Cs0.1FA0.78MA0.13PbI3', 'GU2PbI4', '(ALA)0.2BA1.8MA3Pb4I13', 'Ag3BiI6', 'Cs0.05FA0.83MA0.12PbBr0.39I2.61', 'FAPbBr2.5I0.5', 'BA2Cs0.15FA0.85Pb1.2Sn0.8I7', 'Cs0.05FA0.95PbI3', '(EDA)0.015FA0.29MA0.695Pb1.0I3', '(PEA)1.6BA0.4MA3Pb4I13', 'nannannan', '(PEA)2FA8Sn9I28', 'CsPb0.9Zn0.1Br2I', 'Cs0.05FA0.79MA0.16Pb1.1Br0.51I2.49', 'FAPb3Br6I', 'MAPbBr0.8I2.2', 'FA0.125MA0.875PbI', 'Cs0.1FA0.83MA0.17PbBr0.51I2.49', 'MAPb0.95Sr0.05I3', 'MAPbBr0.015I2.985', 'FA0.3MA0.7PbBr0.54I2.46', 'FA0.84MA0.16PbBr0.45I2.55', 'Cs0.2FA0.664MA0.336PbBr1.05I1.95', 'MAPbI3 | (PPEA)PbI4', 'MAPb0.9Zn0.1I3', 'Cs0.4FA0.6PbBr0.095I0.905', 'CsGeI3', 'FA0.85MA0.15PbBr0.55I2.45', 'FA0.83MA0.17PbBr0.50I2.50', 'MACo0.4Pb0.6I3', 'Ag0.153Bi3I10.5', 'MAPbI3 | Cs1Pb1I3 | Cs1Pb1Br0.3I2.7 | CsPbBr0.7I2.3', 'Cs0.05FA0.45MA0.5Pb0.5Sn0.5I3', 'Cs0.08FA0.8MA0.12PbBr0.36I2.64', 'CsPbBrI', 'Cs0.05FA0.85MA0.1PbBr0.45I2.45', 'Cs0.15FA0.51MA0.34PbI3', 'Cs0.1FA0.9PbBr0.51I2.49', 'FA0.4MA0.6PbBr1.8I1.2', 'IM0.01MA0.99PbI3', '(PDMA)PbI4', 'MACu0.05Pb0.85Sn0.1Br0.1I2.9', 'FA0.6MA0.4PbBr0.1I2.9', 'MA3Bi2I12', 'FA0.85PbBr0.08I2.92', 'FA0.5MA0.5PbSnI3', 'MAPb0.5Sn0.500I3', '(PEA)1.4BA0.6FA3Sn4I13', 'FA0.25MA0.75PbBr0.12I2.88', 'EA0.1MA0.9PbI3', 'BAPbI4', 'Cs0.94Li0.06PbBr3', 'Cs0.05FA0.79MA0.16PbBr0.51I2.49', 'MAPbBr0.05I2.95', 'Cs0.17FA0.83PbBr0.8I2.2', 'Cs0.07FA0.77MA0.16PbBr0.50I2.50', 'MAPb0.9Sn0.1Br0.2I2.8', 'MANiCl2I', 'Cs0.2668FA0.666MA0.0664PbBr0.256I0.744', 'Cs0.17FA0.83PbBr0.75I2.25', 'Cs0.70FA0.30PbI3', 'Cs0.05FA0.79MA0.16PbBr1.2I1.8', 'Cs0.0664FA0.8668MA0.0664PbBr0.256I0.744', 'FA0.5MA0.5PbBr0.13I2.87', 'FA0.85MA0.15PbBr0.51I2.49', 'Cs0.2FA0.8PbBr0.36I2.64', '(PEA)0.33MA0.67PbI3', 'Cs0.1FA0.74MA0.13PbBr0.39I2.48', 'Cs0.05FA0.7885MA0.1615PbBr1.2I1.8', 'Cs0.05FA0.38MA0.57PbI3', 'Cs0.03MA0.97PbBr0.09I2.91', 'CsPbBr2I', 'FA0.1MA0.9PbBr2.7I0.3', 'MAPb0.85Sb0.15I3', 'Cs0.05FA0.80MA0.15PbBr0.45I2.55', '(PEA)2MA3Pb4I13', 'MACo0.031Pb0.969I3', 'Cs0.03FA0.91Rb0.05PbI3', 'FA0.85MA0.15PbBr0.45I2.55', 'FA0.3MA0.7PbBr0.9I2.1', 'MASnBr1.5I1.5', 'FA0.75MA0.25PbBr0.25I2.78', 'MAPbBr2.13I0.87', 'Cs0.05FA0.93MA0.11PbBr0.40I2.6', 'Cs0.17FA0.83PbBr0.51I2.49', 'Cs0.24FA0.76PbBrI', 'CsBi0.01Pb0.99I3', 'MAHg0.3Pb0.7I3', '(PEA)0.05MA0.95PbBr1.2I1.8', 'FA0.6MA0.4Pb0.4Sn0.6Br0.24I2.76', 'FA0.6MA0.4Pb1.0I3', 'FA0.75MA0.25PbBr0.5I2.5', 'Cs0.05FA0.79MA016PbBr0.5I2.5', 'BA2FAMA2.64Pb5I16', 'Cs0.2FA0.66MA0.14Pb0.75Sn0.25Br0.5I2.5', 'Cs0.15FA0.25MA0.6PbI3', 'BA2MA6Pb7I22', 'Cs0.05FA0.79MA0.16PbBr0.31I2.7', 'Cs0.88FA0.12PbI3', 'Cs0.17FA0.83PbBr1.2I1.8', 'Cs0.05FA0.79MA0.16PbBr0.50I2.5', 'FA0.85MA0.15PbBr0.45I2.55 | BA2FAPbI4', 'Cs0.1FA0.77MA0.13PbBr0.39I2.48', '(HEA)2Cs1.9FA17.1Pb20Br12.6I57.54', '(4FPEA)2MA4Pb5I16', 'Cs0.05FA0.80MA0.15PbBr0.33I2.67', 'CsPbBr2.9I0.1', 'BAMA2Pb2I7', 'FA0.33MA0.67PbI3', 'MAHg0.025Pb0.975I3', 'Cs0.1FAxMAxPbBrxIx', 'Cs0.03FA0.945MA0.025PbBr0.075I2.925', 'MAPbBr2.16I0.84', 'BAFA60Pb61I184', 'FA0.67MA0.33PbBr2.5I0.5', '(PEA)0.12BA0.9MA3Pb4I13', 'Cs0.05FA0.7885MA0.1615PbBr0.4845I2.51555', 'MAPbBr0.226I2.774', 'FA0.48MA0.52PbI3', 'FASnBr0.75I2.25', 'MAPb(BF4)0.05I2.95', '(THM)0.2MA0.8PbI3', 'CsPb0.95Sr0.05BrI2', '(BEA)0.5Cs0.15FA2.36MA0.48Pb3Br1.7I0.83', 'Cs0.125FA0.75MA0.125PbBr0.33I2.67', 'BA2MA3PbI4', 'Cs0.17FA0.83Pb0.95Sn0.05I3', 'Cs0.3FA0.7PbBr0.15I2.85', 'FA0.33PbBr2I', 'MAPb0Sn0.19I3', 'IM0.6MA0.60.4PbI3', 'FA0.76MA0.15PbBr0.48I2.42', 'Cs0.05FA0.85MA0.1PbBr0.3I2.7', 'Cs0.04FA0.92MA0.04PbI3', 'Cs0.1FA0.9PbBr0.256I0.744', 'Cs0.02FA0.98PbBr3', 'FAPb0.25Sn0.75I3', 'MAPb0.98Sb0.02I3', 'FA0.92MA0.08PbBr0.24I2.76 | (C6H13NH3)2PbI4', 'MACa0.1Pb0.9I3', '(CIEA)0.01MA0.99PbI3', 'MAPbBrxIx', 'MAPbBr0.9I', 'MAPb0.125Sn0.875I3', 'Cs0.05FAPbI3', 'FA0.25MA0.75PbI3', 'Cs0.05FA0.16MA0.79PbBr0.51I2.49', 'Cs0.17FA0.83PbBr0.15I2.85', 'FA0.66MA0.34PbBr0.42I2.58', 'Cs0.05FA0.795MA0.16PbBr0.51I2.5', 'Cs0.05FA0.79MA0.16PbBr0.16I0.84', 'FA0.44MA0.56PbBr0.33I2.67', 'HA0.3MA0.7SnI3', 'AgCs1.8Rb0.2BiBr6', 'Cs0.02FA0.38MA0.6PbBr0.025I2.975', 'Cs0.3MA0.7Pb0.5Sn0.5I3', 'MAPbBr0.066I2.93', 'MAPb0.875Sn0.125I3', 'Cs0.06FA0.84MA0.10PbBr0.41I2.59', 'Cs0.17FA0.83PbBr0.4I2.6', 'FA0.8MA0.2PbBr0.13I2.88', 'nanBi2FeCrO6', 'Cs0.05FA0.79MA0.16Pb0.664Sn0.336Br0.52I2.48', 'Cs0.05FA0.747K0.05MA0.153PbBr0.51I2.49', '(NH4)3Sb2Br9', '(PDA)0.02Cs0.15FA0.83PbI3', 'CsSnBr3', 'CsBiI4', 'FA0.2MA0.8PbBr0.3I2.7', '(TBA)0.05MA0.95PbI3', 'MAPbBr0.03I2.97', 'CsBi0.025Pb0.975I3', 'Cs0.91Rb0.09PbBr3', 'Cs0.4FA0.6PbBr1.05I1.95', 'MAPb0.999Sb0.001I3', 'Cs0.08FA0.69MA0.23PbBr0.63I2.37', 'Bi0.95La0.05FeO3', '(PEA)0.25(F5PEA)0.75PbI4', 'Cs0.2FA0.65MA0.15PbBr0.55I2.55', 'FA0.57MA0.43PbBr0.39I2.61', 'Cs0.05FA0.79MA0.17PbBr0.51I2.49', 'MAPbI3 | BA2PbI4', 'BA2FA60Pb61Br4I180', '(PDA)2MA3Pb4I13', 'CsFAPbI3', 'CsFAPbBr3I', 'Cs0.08FA0.55MA0.37PbI3', 'Cs0.05FA0.8MA0.15PbBr0.45I2.55', 'Cs0.05FA0.7885MA0.1615PbBrI2', 'Cs0.08MA0.92PbI3', '(BzDA)Cs0.45FA7.2MA1.35Pb10Br2.17I28.83', 'FA0.95TN0.05SnI3', 'FA0.17MA0.83PbI3', '(PEA)0.05MA0.95PbI3', 'Cs0.05FA0.82MA0.13PbBr0.45I2.55', 'EDA0.01FA0.99SnI3', 'Cs0.05FA0.90MA0.05PbBr0.15I2.85', 'Cs0.1FA0.76MA0.14PbBr0.55I2.55', 'Cs0.2FA0.75MA0.05PbBr0.33I2.67', '(PEI)2MA4Pb5I16', 'Cs0.05FA0.8265MA0.1235PbBr0.39I2.61', 'Cs0.06FA0.83MA0.17PbBr0.51I0.249', 'AN0.03MA0.97PbI3', 'BA2MA8Pb9I28', 'Cs0.05FA0.79MA0.16PbI3 | NMABrPbIBr', 'Cs0.17FA0.83PbBr1.2I2.8', 'CsPb2Br5', 'MAPbBr1.25I75', 'Cs0.1FA0.9PbBr0.315I2.685', 'HA2MA2Pb3I10', 'IM0.15MA0.150.85PbI3', 'Cs0.14FA0.86PbBr0.27I2.29', 'FA0.85MA0.15PbI3', 'Cs0.05FA0.09MA0.05PbBr0.15I2.85', 'BA0.67MA0.67PbI3.33', 'FA0.87MA0.13PbBr0.5I2.5', 'FA0.33PbBr0.5I2.5', 'MAPb3I3', 'MA0.05Mg0.05Pb0.95I3', '(NH4)1.7FA0.15MA2.04Pb2.7Br0.45I8.84', 'CsxMAxPbxBrxIx', 'FAPbBr0.256I0.744', 'FASnBr0.5I2.5', 'Cs0.1FA0.75MA0.24PbBr0.51I2.49', 'MABa0.05Pb0.95I3', 'MAPbI3 | (MIC2)2PbI4', 'Cs0.17FA0.87PbBr0.36I2.64', 'Cs0.2FA0.8PbBr0.75I2.25', '(PEA)2Cs4MA35Pb40I121', 'Cs0.03FA0.81MA0.16PbBr0.50I2.50', 'Cs0.1FA0.747MA0.153PbBr0.51I2.49', 'AN0.1MA0.9PbI3', 'MA0.5PA0.5PbI3', 'Cs0.05FA0.83MA0.17PbBr0.39I2.61', 'FA0.6MA0.4PbBr0.095I0.905', '(TFEA)2Cs0.725FA23.925MA4.35Pb29Br13.65I77.35', 'FA0.12MA0.88PbBr0.25I2.75', '(PEA)2MA4Pb4I13', 'Cs2AgBiBr6', 'CsPb1.0BrI2', 'Cs0.60MA0.40PbI3', '(PDMA)FA3Pb4I13', '(EDA)0.02FA0.29MA0.69Pb1.0I3', 'BA2Cs0.45FA2.55Pb2.4Sn1.8I14', 'Cs0.1FA0.9MAPbI3', '(PEA)0.4FA0.6PbI3', '(TFEA)2Cs0.975FA32.175MA5.85Pb40Br18.15I102.85', 'Cs0.01FA0.39MA0.6PbBr0.025I2.975', 'IA0.05MA0.95PbI3', 'Cs0.15EA0.75FA0.1PbBr0.3I2.7', 'Cs0.05FA0.75MA0.15Rb0.05PbI3', 'FA0.25MA0.75PbBr0.25I2.75', 'Cs0.017FA0.83MA0.15PbBr0.51I2.49', 'FA0.1MA0.9PbI3 | (A43)2PbI4', 'Cs0.11FA0.89PbBr0.18I2.82', 'Cs0.05FA0.79MA0.15PbBr0.51I2.49', 'Cs0.25FA0.75Pb0.5Sn0.5I3', 'CsPb0.85I3', 'BA0.1Cs0.15FA0.75PbBr0.3I2.7', 'FA0.92MA0.08PbBr0.24I2.76', 'CsFAPbBr0.3I2.7', 'IM0.1MA0.10.9PbI3', 'MAPb1.0Br0.33I2.66', 'Cs0.2FA0.8Pb0.5Sn0.5I3', 'MACa0.05Pb0.95I3', 'FAPbI3 | CsPbI3', 'Cs0.05FA0.7885MA0.1615PbBr0.45I2.55'])))

    composition_assumption = Quantity(
        type=str,
        shape=[],
        description="""
    The knowledge base from which the perovskite composition is inferred. Is the assumed perovskite composition based on the composition of the precursor solutions and the assumption that the final perovskite will have the same composition (i.e. Solution composition), or is it based on literature claims (i.e. Literature) or has it been experimentally verified with some technique, e.g. XRD, EDX, XRF, etc.?
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Literature', 'Solution composition | XRD', 'TEM', 'XPS', 'Solution composition', 'Solution composition | Solution composition', 'EDX', 'Experimental verification', 'XRD'])))

    composition_inorganic = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the perovskite does not contain any organic ions.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    composition_leadfree = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if the perovskite is completely lead free.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    additives_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    List of the dopants and additives that are in the perovskite
- If the perovskite is layered (e.g. 3D perovskite with a 2D caping layer), separate the layers by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If several dopants/additives, e.g. A and B, are present in one layer, list the dopants/additives in alphabetic order and separate them with semicolonsas in (A; B)
- If no dopants/additives, state that as “Undoped”
- If the doping situation is unknown, stat that as‘Unknown’
- Include any non-solvent that does not go into the perovskite structure. This includes compounds that are found in secondary phases, or amorphous grain boundaries, or that disappears during synthesis.
o One example is Rb in MAFAPbBrI-perovskites. As far as we know, Rb does not go into the perovskite structure, even if that was believed to be the case in the beginning, but rather form secondary phases. For MAFAPbBrI-perovskites, Rb should thus not be considered as a A-site cation, but as a dopant/additive.
o One other example is chloride in MAPbI3. As far as we know, Cl does not go into the perovskite structure even if that was believed to be the case in the beginning. For MAPbI3 Cl should thus not be considered as a C-site cation, but as a dopant/additive.
Example
Cl
Undoped
5-AVAI
SnF2
Ag; Cl; rGO
Rb
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '4-Aminophenyl Boronic acrid; Cl', 'FABr', 'BMImI; LiI; L-iTFSI', 'Cl; Phosphatidylcholine', 'PLMF', 'Cl | ZnCl2', 'DACl', 'CH3CONH2', 'rGO-g-Ptet', '(PEY)2PbI4', 'oxo-G/DA', 'MEH-PPV; TBP', 'PbI2-bipyridine', '1,10-diiododecane; Cl', 'AA; Cl', 'NiI2', 'Benzenamine hydrochloride', 'BrPh-Thr; bis-PCBM', 'HCOOH', 'OctAm', 'CsPbI3', 'CH2CHCH2 (CC2)-ionic liquid', 'Polyethylenimines (PEIs), MW 600', 'EP-PDI', 'EtOH; MA', 'EDACl2', 'Cl; PTAI', 'bis-DMEC60', 'DMF', 'CuBr', 'Cl; PEO', 'Tb', 'ILPF6', 'KIPIG-500', 'Cl; K; Sn', '1‐methyl‐3‐(1H,1H,2H,2H‐tridecafluorooctyl)‐imidazolium iodide', 'SCN; SnF2', 'Cl; PbS-np', 'NbF5', 'PbF2', 'PEI', 'n-butyl amine', 'Cl; MDACl2', 'NaI', 'PF-0', 'Cl; DPE', 'methylammonium hypophosphite; L-α-phosphatidylcholine; PEABr', 'PVDF', 'SrI2', 'Starch', 'P(VDF-TrFE)', 'C8Br; Cl', 'GU(SCN)', 'PCBB-OEG', 'C4H10INO2; Cl', 'ITIC', 'NH4I', 'Carbazole-C4', 'Polystyrene', 'CsAc', 'BMIm', 'SCN', 'FACl; SnF2', 'BaAc2', 'PMM', '1‐butyl‐3‐methylimidazolium bromide', 'RbBr', 'PVAm.HI', 'C16H36BrN', 'PPC', 'organicD‐pi‐A', 'p-phenylenediamine', 'Pyrrole', 'TPA', 'Cl; CNT-NH2', 'Cl; CuI', 'Cl; CuPc(tBU)4', 'HPbI3 | Cl', 'Undoped | Undoped', '1,4-dibromobutane; Cl', 'CuBr2; RbI', 'PABA∙HI', 'PPAI', 'In; Br', 'Cl; Spiro', 'Rubrene', 'PbAc2; H2O', 'PCBM-60', 'Ni', 'Ag', 'CuBr2; CsI', 'EDA; SnF2', 'ASCl; Cl', 'Cl; SrCl2', 'Mg', '1,8- Diazabicyclo[5.4.0]undec-7-ene', 'Melanin', 'HoNH3Cl', 'MACl, NMP | Undoped', 'AsI3 | NH4Cl', 'Cs', 'Isatin-Cl', 'KSCN', 'TPE-DPP4', 'Phenylammonium; HI', 'DCL97', 'MoS2', 'PPS', 'P25', 'MABr; MACl', '1‐butyl‐3‐methylimidazolium iodide', 'Caprolactam', 'Cl; Urea', 'BAOAc', 'AgI', 'TEP', 'Cl; EAI', 'CSA', 'HPA; Pb(AcO)2', 'InCl3', 'DMA', 'C60', 'HI; HBr', 'Cl; PNVF–NVE', 'Phosphonic acid', 'BAI', 'Aminobenzonitrile', 'C3N5', 'MAPbBr3-QDs', 'Cl; Cu(Tiurea)I', 'Cl; Hypophosphorous acid', 'MPTS; Cl', 'Imidazole', 'Cl; I2', 'C60; PMMA', '(adamantan‐1‐yl)methanammonium; HI', 'Hl', 'UiO', 'ABA', 'Cl; PCBM', 'PEG-[60]fullerenes', 'p-Si-np', '5-AVAI; CuSCN', 'Cl; Formamidineacetate', 'HMTA', 'Cl; NH4Cl; CuBr2', 'Acetate; HPA', 'Pb(OAC)2', 'Urea', 'Cl; Thiourea', 'MgI', 'FeOOH-QDs', 'ZnI', 'C3H7NH3I; Cl', 'FAAc', 'TBA', 'L-alfa-phosphatidylcholine', 'NaOAc', 'Co(Ac)2', 'CPTS; Cl', '(SnF2(DMSO))2', 'PEA2PbI4', 'RbI', 'Polyethylenimines (PEIs), MW 70000', 'PTAA', '2‐(6‐bromo‐1,3‐dioxo‐1H ‐benzo[de ]isoquinolin‐2(3H )‐yl)ethan‐1‐ammonium iodide', 'Hexamethylphosphoramide', 'CsPbBr3-np', 'Cl; PEG-NH2', 'DRCN5T', 'Carbonnanowalls', 'HaHc', 'Diethylamine hydrochloride', 'PEO', 'Mn', 'Hexylamine hydrochlorid', 'C6Br; Cl', 'NMP', 'poly(ethyleneimine); Carbon-nt', 'b-PEI; Cl', 'Ti', 'Sr', 'PbCl2; KCl', 'MACl', '1,4-diiodobutane; Cl', 'BYK333', 'BP-QDs', 'C6H14INO2; Cl', 'exMMT', 'Spiro', 'Lead acetate', 'CEA', 'Cl; Au-CZTS', 'PbI2', 'Methylpyrrolidone', 'H2P2O6', 'C-PCBOD', 'IEICO-4F', 'GaAA3', 'beta-cyclodextrin', 'n-butyl amine; Cl', 'P123', 'tetra-tert-butyl-metal free phthalocyanine', 'PEG; Cl', 'HMImCl', 'Pb(DDTC)2', 'EDAl2; SnF2', 'Anilinium Iodide', 'MA(SCN)', 'PS; PMMA', '5-AVAI; PCBM-60', 'E-g-C3N4-np', 'D-alanine', 'Acetate', 'YD2-o-C8', '4-ABPACl', 'PDMAI', 'PVA', 'FIm', '3-Aminophenyl Boronic acrid; Cl', 'Polyethyleneglycol; Polyvinylpyrrolidone', 'EtOH; H20; Pb(SCN)2', 'Polyurethane', '1-butyl-4-amino-1,2,4-triazoliumiodine', 'CuSCN', 'DPSI', 'Glycine', 'GUBr', 'p-CH3OC6H4; p-t-BuC6H4', 'MA', 'Nd', 'Cl; TPPI', 'MACl, NMP', 'GAI', 'Er', 'ZnPc', 'Pb(CH3CH2COO)2', 'Tetracyanoquinodimethane', 'PPA', 'KIPIG-600', 'PTS | Cl', 'Piperazin; SnF2', 'CH2I2', 'Cl; Liquid crystals', 'MOF', 'CuBr2; NaI', 'YbAc3', 'NiCl2', 'Butylammonium iodide', 'Cl; INIC2', 'TTABr', 'HI; PEAI', 'Melaminium iodine', 'Lysine', 'Yttrium', 'MAPbI3-QDs', 'DMBI-2-Th', '1,3:2,4-di-O-dimethylbenzylidene-d-sorbitol', 'Ag@SiO2-nw', 'Poly(styrene-co-butadiene)', 'E2CA', 'Cl; MAH2PO2', 'KIPIG-550', 'Benzoic acid hydroiodide', 'Cl; DRCN5T', 'Guanidinium', '5-AVAI; Formamide', 'Guanidinium-SCN', 'SA-2', '1-ethyl-4-amino-1,2,4-triazoliumiodine', 'SnCl2', '1,6-diaminohexane dihydrochloride', 'DOI', 'CaI2', 'Cl; DIO', 'CNDs@K', 'Cl; CsF', 'PCBPEG-4k', 'CdI2', 'CsPbBr3', 'TPPCl', 'Cl; DMF', 'APSA', '1‐butyl‐3‐methylimidazolium chloride', 'Cl; PEI', 'pyr-fullerene', 'tetrabutylammonium chloride', 'DMBI-2-Th-I', 'Sm(acac)3', 'CH3I; Cl', 'Rb', 'PAI', 'SrCl2 | MACl', 'ZnO-np', 'NiO', 'MACl; poly[9,9-bis(3′-(N,N-dimethylamino)-\npropyl)-2,7-fluorene)-alt-2,7-(9,9-dioctylfluorene)]) (PFN-P1) in chlorobenzene; PFN-P2 (in ethanol', 'OTG1', 'DF-C60; SnF2', 'PF-1', 'Undoped | Mn', 'BiI3', 'Li', 'MABr', 'PbCl2', 'MACSN', 'TBAI3', 'HATNA', 'IBr', 'MXene', 'AuAg@SiO-np', 'Cl; H2O', 'P(EO/EP)', 'Protic ionic liquid', 'Cl; HI', 'Cl; InCl3', '5-AVAI; HBF4', 'Formamide; PEA', 'Li-TFSI; TBP', 'AVAI', 'Cl; Cu:NiO-np; Graphite', 'MOF-525', 'Rb; Thiourea', 'Li-TFSI; LiF; TBP', 'DMAI', 'NMA', 'SQ63', 'Cl', 'NH4Ac', '1,8-octanedithiol', 'Hydrazinium chloride; SnF2', 'Butylamineiodide', 'FAI', 'tetra-tert-butyl‑silicon phthalocyanine bis(trihexylsilyloxide)', 'S-Carbon-nt', 'HPbI3', 'NaF', 'BCP', 'DETAI3', 'H2O; Pb(SCN)2', 'Cl; TBAB', 'C6H5C2H4NH3', 'Diphenylidonium hexafluoroarsenate; PCBOD', 'NO3', 'Eu', 'Butanediaminedihydroiodide', 'PEACl', 'PbAc', 'FU11', 'Phenylethylammonium iodide', 'rGO-g-PDDT', 'Ag-nw', 'Graphdiyne', 'Chitosan', 'H2O; KI', 'C4F8I2', '1-ethylpyridinium chloride', 'SWCNTs', 'CNT', 'DMEC60', 'J71', 'MACl; PbBr2', '2Ph-ox', 'NEP', 'LiBr', '4-fluorophenylethylammine', '1-octyl-4-amino-1,2,4-triazoliumiodine', 'PCBM-70', 'PMPS', 'DMI', 'Diiodomethane', 'PEOXA', 'InCl2', 'Yb', 'PDAI', 'CuI2', 'MAAc', 'SrCl2', 'Side-chain liquid crystalline polymer (SCLCP)', 'SnF2; TMA', 'Cl; I3', 'A43', 'K; Rb', 'Al; Cl', 'OTG2', 'methylammonium hypophosphite (MHP); L-α-phosphatidylcholine (LP); 1,3-diaminopropane (DAP)', 'BEA', 'TiO2-np', 'Acetic acid; HCl; n-propylamin; Pb(Ac)2', 'C4Br; Cl', 'C60-PYP', 'Ethane\xad1,2\xaddiammonium', 'Pb(NO3)2', 'ASA', 'CB', 'Li-TFSI', 'BiFeO3-np', 'Carbon-np; Urea', 'PE10', 'GA', 'BTA; SnF2', 'methylammonium hypophosphite; L-α-phosphatidylcholine; PEACl', 'TMA', 'SnF2', 'CuCl; PbCl2', 'Poly(amicacid)', 'Cl; RbBr', 'PbC2O4', 'Thiourea', 'LiI', 'Graphdiyne-QDs', 'terephthalonitile', 'As; NH4; Cl', 'FACl', 'NH4Cl; DMSO', 'Cl | nan', 'Acetonitrile', '2-(1H-pyrazol-1-yl)pyridine', 'GITC', 'TDZT', 'HCl', 'N-methyl-2-(3,5-bis(perfluorooctyl)phenyl)-3,4-fulleropyrrolidine', 'MAPbBr3', 'SmI2; SnF2', 'HI; Mercapto-tetrazolium', 'TiI2', 'GuaI', 'Cl; PEG', 'Carbon-nt-g-P3HT', 'Cl; Fe(acac)3', 'NaYF4:Yb:Er-np', '5-AVAI; Cl', 'CsBr', 'Hydrophosphoric acid; rGO; PbAc2', 'N-Carbon-QDs', 'MAH2PO2', 'PTMA-H; Rb', 'F127', 'Black Phophorous', 'GeI2', 'PCBM-60; PbF4', 'GuCl', 'L-α- phosphatidylcholine', 'CsPbBrCl2-QDs', 'Ag-NPs', '2-pyridylthiourea', 'EtOH', '1-Donecyl Mercaptan', 'Au@Ag@SiO2-np', 'PFPA', 'PbS-QDs-AI-ligand', 'DIO', 'H2PO3', 'Cl; ZnO-np', 'Polyimide', 'EDAI2 | SnF2', 'NaCl', 'CsPbr3', 'I3', 'EDA', 'CH3SH', 'Cl; IPFB', 'Ti3C2Tx', 'Methylammonium formate', 'MACl; PFN-P2 (in ethanol', 'Cl; NH4Cl', 'Cl; NH4Br', 'HPbI3; PTABr', 'tetra-tert-butyl‑germanium naphthalocyanine bis(trihexylsilyloxide)', 'PE', 'NH4SCN', 'Phenylethyl-ammonium iodide', 'Acetamidine hydrochloride', 'Polythiocyanogen', 'Cl; Guanidinium', '3,4-dihydroxybenzhydrazide; MACl', 'KCl', 'Sm', 'CH2CCH (CC3)-ionic liquid', 'GASCN', 'TBAI; Cl', 'Cl; HCOOH', 'Octoxynol', 'PEA5', 'Cl; Formamide; Guadinium', 'n-BAI; Rb', '5F-PCDM-60', 'CuI; PbCl2', 'Au@TiO2 NPs', 'TOPO', 'Cl; Cu(thiourea)Cl', 'PbF4', 'M13 bacteriophage', 'OAc', 'PVC', 'EAPP', 'CH2O2; SnF2', 'SbCl3', 'n-Si-np', 'Trimesic acid', 'CsCl', 'Acetate; SrI2', 'N-cyclohexyl-2-pyrrolidone', 'DPP-DTT', 'Hydrazine', 'KHQSA', 'Butylated hydroxytoluene', 'RbI; KI', 'Acetate; HAc', 'Ethyl cellulose; Cl', 'Cl; TBAC', 'C2H6INO2; Cl', 'FeI2', '4-MSA', 't-BAI', 'Cu', '1,2,4-triazole', 'Zn', 'KBr', '3F-PCBM-60', 'Cl; Eu(acac)3', 'Cl; Cu(thiourea)I', 'TDZDT', 'Acetate; Cl', 'Polyvinylalkolhol', 'Agarose', 'Cl; KI', 'PMA', '3BBAI; Cl', 'Thenolytrifluoroacetone', 'Cl; PCBM-60', 'Cl; PCBM-60; PEG', 'HBr', 'MLAI', '5-AVAI', 'PVDF-HFP', 'Cl; HI; KOH', 'D3', 'Cl; KBr', 'Cl; PNVF-NVEE Microgels', 'PVA; SnF2', 'Undoped', 'C6F5I', 'PTAI', 'Cl; EA۰HCl', 'Rb; SrI2', 'g-C3N4', 'Carbon', 'DEACl', 'MAI', 'EACl', 'C3A; PEA', 'THTO', 'Acetate; H2O; Hypophosphorous acid', 'Adipic acid; Cl', '3-aminopropyl (3-oxobutanoic acid) functionalized silica nanoparticles; Cl', 'OA; ODE; OLA', 'D1', 'CHP', 'SA-1', 'Ethylene‐diammonium; SnF2', '1,8-dibromooctane; Cl', 'GN-GQDs', 'BMImI; LiI', 'ZnI2', 'h-TAc', 'F4-TCNQ', 'A10C60', 'Ca', 'Ag; Cl; rGO', 'Cl; HBr', 'PEABr', 'MQW', 'Cl; NO3', 'Mn; Cl', 'PDMS', 'H2O; KCl', 'Cl; MAAc', 'BMIMBF4', 'C60(QM)2', 'PMMA; PbCl2', 'H3PO2; Acetate', 'PANI', 'Acetic acid', 'Al2O3-np', 'PC', 'Ce', 'PEG', 'Caffeine', 'SQ45', '3-phenyl-2-propen-1-amine iodide', 'HBr; Hi', 'SQ81', 'MnCl2', 'Monoammonium zinc porphyrin', '1,8-diiodooctane', 'BMImI', 'Cl; DMSO', 'Cl; DL-tartaricacid', 'CDTA; SnF2', 'ThMAI', 'I', 'TPPi', 'Imidazolium', 'FEAI', '1-hexyl-3-methylimidazoliumchloride; HMImCl', 'EuCl3', 'Octylammonium iodide', 'CHCl', 'Dithizone', '4-vinylbenzylammonium', 'H2O; TEOS', 'Cl; Formic acid', 'iPAI', '1,3-diaminopropane', 'p-phtalic acid', 'Cl; NH3SO3', 'Lu', 'H2O; Ti3C2Tx', 'Cl; TBPI', 'Cl; NiO-np', 'Rhodanina', 'CH3COCHCOCH3', 'MAOAc', 'Cl; SDS', 'Cu:NiO-np', 'SP-3D-COF 2', 'CU(SCN)', 'Cl; ITIC', 'Cl; PEDOT:PSS', 'PbAc2; PbCl2', 'PAA', 'Carbon-np', 'alfa-cyclodextrin', 'Eu-pyP', 'MWCNTs', 'TSC', 'Cl; Diiodooctan', 'CsBr; Cl', 'BHT', 'Unknown', 'SbBr', 'Cl; HCl', 'PbS-QDs-MAI-ligand', 'N2H5Cl', 'Si-nc', 'Octylammonium', '1-hexyl-3-methylimidazoliumchloride', 'PMMA; Rb', 'CH3NH2', '4-(1H-imidazol-3-ium-3-yl) butane-1-sulfonate', 'ligands', 'N-methylimidazole', 'DIFA', 'Formamide', 'Styrene', 'YbCl3', 'K', 'MeO-PEAI', 'BrPh-ThR', 'OA; OLA', 'SnF2; Uric Acid', 'SnCl', 'Cl; TBP', 'tetra-ammonium zinc porphyrin', 'Ho', 'Graphene', 'Carbon-nt-g-PDDT', 'PFPAI', 'Graphitic carbin nitride (g-C3N4)', 'Cl; PAA', 'TEOS', 'Cl; KCl', 'Carbon-nt', 'F-PEAI', 'J61', 'Ethyl cellulose', 'Ethyleneglycol', 'Citric acid; Cl', 'Cl; Pb(SCN)2; SnF2', 'D2', 'pyP', 'In', 'dimethyl itaconate', 'GdF3', 'BAI; PEG', 'Formic acid', 'Cl; PCDTBT', 'Nb', 'GABr; Pb(SCN)2', 'APPA', 'H2O', 'Ag@SiO2', 'NH4Br', 'gamma-cyclodextrin', 'Ba', 'PEAI', 'J51', 'N2200', 'DNA', 'TiO2-nw', 'A@SiO2-np-nw; Cl', 'Graphene-nanofibers', 'Methylamine', 'G-NH2', 'BaCl2; Cl', 'GASCN; MACl', 'SnF2; TFEACl', 'Polyacrylonitrile', 'Acetic acid; Cl', 'La', 'LFA', '[BMMIm]Cl', 'PCBPEG-20k', 'B-alanine', 'NH2CONH2', 'P(VDF-TrFE-CTFE)', 'Polyvinylbutyral', 'Graphene oxide', 'EAI', 'C70', 'C4H8I2', 'Cesium phenethyl xanthate', 'NH4Ac2', 'Benzylamine hydroiodide', 'LiCl', 'FPEAI', 'GuaSCN; SnF2', 'PbS-np', 'Hydroquinone', 'CuI', 'enI2; SnF2', 'Cl; PbS', 'PMMA', 'H2O; KBr', 'HC(NH2)2I–NaI', 'PbC2N2S2', 'g-CN', '1,8-octanedithiol; Cl', 'Cellulose-CDHC', 'HMPA', 'rGO-g-P3HT', 'Cellulose-HEC', 'Gd', 'NaSCN', 'NH4Cl; NH4SCN', 'PA', 'TACl', 'I2; Thiourea', 'CsPbBr3-nw', 'Benzene‐1,4‐diboronic acid; Cl', 'methylammonium hypophosphite; L-α-phosphatidylcholine', 'MABF4', 'Co', 'BA; HI; HBr', 'Cl; PVP', 'IPA HCl', 'CaCl2', 'F-N2200', 'CuCl2', 'Ag; Cl', 'TCA', 'TBP', 'CuBr2; KI', 'Cl; CuSCN', 'Benzoquinone', 'KI', 'Al2O3-np; Cl', 'Isobutylamine hydroiodide', 'Graphene-QDs', 'MAPbCl3', '2,9,16,23-tetra-tert-butyl-29H,31H-phthalocyanine', 'Formamide; Guadinium', 'Bi', 'Cl; Y', 'NiO-np', 'GuI', '1,8-diiodooctane; Cl', 'SP-3D-COF 1', 'Cl; Cu', 'rGO', 'HBr; HI', 'HA', 'OAm', 'Cl; Graphene', 'Nano-carbon', 'HAc', 'PEA20', 'ACN', 'Hypophosphorous acid', 'Cl; TPPCl', 'IPFB; PbCl2', 'CsPbBr-np', 'MgAc', 'PEA0', 'In2-6', 'FPEAI; Mn', 'Cl; Y(acac)3', 'Cl; NH4I', 'ZnCl2', 'CsI', 'TiO2', 'Polyethylenimines (PEIs), MW 10000', 'Diiodooctane', 'PbS-QDs', 'Phenylethyl ammonium iodide', '3-(5-Mercapto-1H-tetrazol-1-yl)benzenaminium iodide; HI', 'CdS; Cd(SCN2H4)2Cl2', 'PbAc tri-hydrate; H3PO2', 'MA3Bi2Br9', 'BmPyPhB', 'Ag-rGO; Cl', 'CdS', 'Cl; DMA', 'Methimazole', '2-Phenylethylamine Hydroiodide', 'Ba; Cl', 'BAI; GAI', 'Az', 'Cl; C-PCBSD', 'Diethylammoniumchloride; PCBM-60', 'hypophosphorous acid', 'CH2CN (CN)-ionic liquid', 'ITIC-Th', 'PbCl2; Phenol', 'False', 'Br', 'Cl; NAP 1-(3-aminopropylpyrrolidine)', 'Cd', 'Carbon-QDs', 'PTMA; Rb', 'Cl; DTA', 'g-CNI', 'Sn', 'PCBM-nw', '5-AVAI; Acetamide', 'Cl; CZTS', 'MnCl2; ZnCl2', 'Cl; MA', 'Cl; SCN', 'ITIC; PCBM-60', 'NH4OAc', 'NH4Cl; SnF2', 'Au-np', 'TMS', 'PVP', 'Cl; MoOx-np', 'NaAc', 'Benzoicacid', 'FAOAc', 'Tea Polyphenol', 'Cl; Sr', 'Carbon-nt; PDDT', 'DOI; PbCl2', 'Nickel phtalocyanine', 'OAI', '5-AVAI; Urea', 'n-butylammoniumbromide', 'en; SnF2', 'PbCl2; Phosphatidylcholine', 'NH4Cl', 'CQD', 'Pb(OAc)2', 'Levulinic acid', 'EC', 'PbSCN2', 'KI; I2', 'In2-4', 'NH4BF4', 'EDEA', 'Hydrophosphoric acid; PbAc2', 'Graphdyine', 'TOAB', 'OTAB', 'Pb(SCN)2; SnF2', 'Diethylammoniumchloride', 'Tetraethylorthosilicate', 'trihydrazine dihydriodide (THDH)', 'DA2PbI4', 'Cl; DL-lacticacid', 'l-alfa-phosphatidylcholine; Methylammoniumhypophosphite; NH4Cl', 'HI | Undoped', 'Graphdiyne QDs', 'Lead acetate trihydrate', 'PEAI; SnF2', 'BE2PbI4', 'Cl; FAH', 'ABS', 'PbCl2; TBP', 'PS', 'CsI; OIH', 'J50', 'NH4F', 'acac; Cl', '3DHG', 'BA', 'PEA; SnF2', 'Pb(Ac)2', 'OTG3', 'NH4H2PO2', 'PEA', 'Eu(Ac)3', 'Acetate; H2O', '1-allyl-3-methylimidazolium chloride', 'H3PO2', 'EE', 'NO3-C3N4', 'Cl; MWCNTs', 'BCP; Cl', 'Zr(AC)4', 'Terephthalic acid', 'SnS-QDs', '1-benzyl-3-methylimidazolium chloride', 'Sb', 'Cl; TPPBr', 'di-iodomethane', 'NH4', 'ZnAc2', 'Ethylenediammonium', 'Cl; DL-malicacid', 'Thiosemicarbazide', 'Guanidinium; HI', 'ETI', 'Phosphonic acid; Aluminium acetylacetonate', '1,3:2,4-di-O-methylbenzylidene-d-sorbitol', 'CH3CH2COO', 'SbI3', 'SnBr2', 'xDMAI', 'Cl; SrAl2O4:Eu2+:Dy3+', 'CuBr; PbCl2', 'MACL', 'Cl; SnF2', 'PTN-Br; SnF2', 'SnI2', 'bis-PCBM', 'HI', 'BMII', 'HPA', 'In2-2', 'Poly(diallyldimethylammoniumchloride)', 'Cl; K', 'I2', 'ME', 'Pyrazine; SnF2', 'DPPS', 'Ascorbic acid', 'PbAc2', 'CuBr2', 'IDTBR', 'PbCl2; PbAc2', 'Cl; ICBA', 'Carbon-nt; P3HT', 'P(VDF-TrFE-CFE)', 'BaI2', 'ALAI; Cl', 'TMTA', 'Formamidinium chloride; formamidinium hypophosphite; Phenylethylammonium chloride', 'DMSO', 'NaYF4:Tb:Er-np', 'Cl; NH4Cl; CuBr', 'PTB7', 'SnF2; PMMA', 'Pb(SCN)2', 'l-alfa-phosphatidylcholine; Methylammoniumhypophosphite', 'DAGCl', '1-chloronaphthalene; Cl', 'Acetate; HCl', 'CdCl2', 'Cl; V2Ox', 'Cl; IEICO-4F', '1-chloronaphthalene', 'C70; Cl', 'a-Ge; Cl; H-np', 'methylammonium hypophosphite; L-α-phosphatidylcholine; PEAI', 'C4H9NH3I; Cl', 'C-PANI', 'MAI; FACl', 'PU', 'TFBA'])))

    additives_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the dopants/additives.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If more than one dopant/additive in the layer, e.g. A and B, separate the concentration for each dopant/additive with semicolons, as in (A; B)
- For each dopant/additive in the layer, state the concentration.
- The order of the dopants/additives must be the same as in the previous filed.
- For layers with no dopants/additives, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used.
- The preferred way to state the concentration of a dopant/additive is to refer to the amount in the final product, i.e. the material in the layer. When possible, use on the preferred units
o wt%, mol%, vol%, ppt, ppm, ppb
- When the concentration of the dopant/additive in the final product is unknown, but where the concentration of the dopant/additive in the solution is known, state that concentration instead. When possible, use on the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
5 vol%; nan | 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | 0.3 M
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.25 wt%', '0.125', '0.225', '24 mg/ml', '0.025; nan', '0.35', '2 %; 0.1 %', '0.66; 0.0875', '2', '0.0775', '0.005', '0.0207', '0.05 %', '0.1 M; 10 wt%', '5 vol%', '0.1 mol', '0.005 M', '0.057', '0.003; nan', '6 mol%; 10 mol%', '0.22', '0.6 mg/ml; nan', 'nan; 1', '3.7 wt%', '0.36', '0.099', 'nan; 8 mol%', '0.6 vol%', '0.03 M', '5.6 vol%; nan', '2.6', ' PEG', '2.5 wt%; 0.5 wt%', '0.1666666666666667', '0.03125', '0.035; 1.5 vol%', '0.002 mg/ml', '8 %; 4 %33%', '0.0003 mmol/ml', '0.3 vol%', '3.3 %; nan', '2.5 vol%', '0.2 mol%; 10 mol%', '0.18', '0.0001 mol%', '0.01 mol%', '0.515 M', '0.07', '20 µmol/ml', '2.5 wt%; 1.5 wt%', '0.9 wt%', '0.006', '0.5 mg/perovskite_mmol', '30 mmolL-1', '0.000525', '2.5 wt%; 0.25 wt%', '0.321; 0.009', '0.05', '2.4', '0.00167', 'nan; 5 %', '0.4 wt.%', '0.05 wt%', '6.4', '1.05', '0.04; 0.17', '0.0125', '0.3 mg/perovskite_mmol', '2.5 wt%; 0.35 wt%', '8', '0.084', '10 vol%', '0.05 %; nan', '0.0001 mmol/ml', '0.035; 3 mg/ml', '0.0007', '0.03; 0.02', '0.0625 \u202c', '1.0', '0.14', '10 wt%', '0.16; 0.5', '60 mol%', 'nan | 20 mg/ml', '0.00068', 'nan | 13 mg/ml', 'nan; 4 mol%', '0.02', '1.2 mg/ml; 20 mg/ml', 'nan | 27 mg/ml', '0.0003', '0.333; 0.009', '3.9 vol%', '0.4; 0.01', '0.7', '0.9 mg/ml; 15 mg/ml', '6 mol%; 20 mol%', '0.05; nan', '0.66; 1', '0.04; 0.0015', 'nan; 0.66 %', '0.034; 0.034', '3 mg/ml; nan', '15; 10', '50 mg/ml; 1 mg/ml', '0.3 mg/ml', '6.8 e-05', '0.03 mM; 0.09 mM', '0.048', '1.2 M', '0.312; 0.018', '3 mg/ml', '0.0005 mmol/ml', '0.005 wt%', '4.5', '0.1 vol%', 'nan; 0.0125', '60 mol% | 6 mg/ml', 'nan; 0.0166 %', '0.336', '150 mol%', '12 wt%', '10 %; 7.5 %', 'nan | 5 mg/ml', '4 mg/ml; nan', '3 %; nan', '0.9 mM', '30 mM', '0.03 vol%', '12.8', '10 ppm', '0.089 mg/ml', '0.5 wt%', '10 g/ml', '0.06 mg/ml', '0.13 wt%', '0.028; nan', '2.5 %; 10 %', 'nan; 1.0', '0.1; 0.15', '15 %; 7.5 %', '8.0', 'nan; 1 wt%', '23 mol%', '0.006; 0.05', '6 mg/ml', 'nan; 1 mg/ml', '15 mg/ml', '0.262 M', '0.02 M', 'nan; 20 %', '0.09', '0.45', '0.016', '1 ppm', '7 mol%', '0.333', '5 mg/ml; nan', '0.0085', '0.5 wt.%', '0.09375', '0.15 wt%; 0.05 wt%; 0.42 mol%', '5 mol%; 20 mg/ml', '2.0 wt%', '50 mol%', '5 wt%', '50', '5 %; nan', '2 wt.%', '4 mg/ml; 4 mg/ml; 4 mg/ml', '2.3', '0.05 M', '0.03', '0.0034', '16 vol%', '15 wt%', '0.01 ug/ml', '2.5 wt%; 0.3 wt%', 'nan; 0.02', '0.10 wt%', 'nan | 0.05 mol%', '6 mM', '50 vol%', '8 mol%', '0.66; 0.08', '0.1 mol%', '1.3', '0.035; 30 mg/ml', '1.2 wt%', '0.5; 0.5', '12 mM', '0.00075', '3.5 m%', '12 mg/ml', '0.0175', '4 µmol/ml', '5- AVAI', '12.5 wt%', '0.042', '18 mM', '5 %; 0.1 %', '0.6 mg/ml', '3.3 vol%', '0.6 wt%', '0.097 mol%', '0.5 %; nan', '0.66; 0.075', '0.0026344676180021956', 'nan; 10 mg/ml', '0.05 vol%', '4 mg/ml', 'nan; 9 mM', '0.5 M', '0.0004', 'nan; 114 mM', '0.05; 0.15', '1.5 %; 6 %', '0.3 M', '20 mg/ml', '0.04; 0.003', '100 ppm', '7 mg/ml', 'nan; 1 %', 'nan; 0.1 mg/ml', 'nan | 0.1 mol%', '2.5 wt%; 0.2 wt%', '6 µmol/ml', '7.5 mol%', '0.018; nan', '0.5 vol%', '0.08', '2 wt%', '0.04 %', '0.025', '10 mmg/perovskite_mmol', 'nan; 0.033 %', 'nan; 0.2 mg/ml', '0.014 mol%', '0.3', '12.5', '0.59 uM', '0.12', '0.0285 vol% | nan', '0.0076', '0.015 M', '0.06', '10 %; 20 %', 'nan; 0.015', '1 e-05', '6 wt%', '0.10; 0.01', '75 vol%', '10 mg/ml', '0.4 mg/ml', 'nan; 0.15 %', '0.25 mM', '5.69 vol%', '3.5 vol%', '0.75; nan', '0.66; 1; 0.0075', 'nan; 2 mg/ml', 'nan; 0.33 %', 'nan; 0.00166 %', '0.0125; 0.05', '1.35', '8 %; nan', '1000 ppm', '16.67 %; nan', '2.5 wt%; 0.1 wt%', '0.08 mg/ml', '0.125; 20 mg/ml', 'nan; 0.0075', '2.5', '90 mol%', 'nan; 2 wt%', '1.5 mol%', '2 mg/mlantisolvent', '25 mg/ml', '0.0011', '0.15; 0.15', '0.0015', '20 vol%', 'nan; 4 %', '15 mol%; 10 mol%', '0.075; nan', '6', '1.25; nan', 'nan; 2', '0.35; 0.019', '0.02; 0.03', '0.4 mM', '50 mg/ml; nan', '8 vol%', '5 %; 7.5 %', '0.3 mol', '0.006; nan', '0.25', 'nan; 18 mM', '2.5 mg/ml', '0.09; nan; nan', '1 wt%; nan', '2 vol%', '0.01 | 0.1', 'nan; 1.5', 'nan | 0.2 mol%', '0.017', '0.01 vol%', '7.5', '0.009', '13.7 mg/ml', '0.027', '0.25; 0.04', '0.15; 0.15; 0.004', '6.67', '30 wt%', 'nan; 0.005', '1.0 wt%', '1 %; 0.1 %', '0.48 mg/ml; nan', '5 mol%', '0.067 mol%', 'nan; 36 mM', '0.9', 'nan; 0.5', '0.011', '0.035', '0.4 wt%', '0.004', '0.33; 0.33', '6 %; nan', '0.04; 0.0001875', 'nan; 0.025', 'nan; 0.05', '100 mol%', 'nan; 5 mol%', '0.75; 0.25', '20 uL', '0.0375', '1.67 mol%', '0.14 M', '3 wt%', '0.01; 0.05', '4 %; nan', '0.0028', '0.15; 0.075', '0.07 wt%', '1; nan', 'nan | 0.5 mol%', '0.35; 0.057', '0.0002', 'nan; 2 %', '0.8 M', '3.5 mol%; 10 mol%', '25 vol%', '0.04; 0.000375', '10 µmol/ml', '0.15; 0.025', '0.5 mg/m', '0.8', '25', '9', '16 mg/ml', '6 e-05', '0.5; nan', '2 mol%; 2 mol%', '1', '0.33; 0.003', '0.064', '0.6 wt.%', '5.0', '0.1; 0.02', 'nan; 3 %', '0.6', '0.66; 0.0625', '0.84 vol%; nan', '1.25 mg/ml', '76 mg/ml', '15 µmol/ml', '0.20; 0.15', 'nan | 2.5 mol%', 'nan; 0.54 %', '0.1 mg/perovskite_mmol', '12 mg/ml; 12 mg/ml; 12 mg/ml', '0.2', '0.24 mg/ml; nan', 'nan; 1.66 %', '5 mg/ml', '1 mol%', '2 mg/ml; nan', '100 vol%', '1.5', '0.33', '0.00035', '0.167', '0.284', '2.0', 'nan; 9', '1 mg/ml; nan', 'nan; 1.5 mg/ml', '0.00067', '1 e-06', '5 mol%; 10 mol%', 'nan; 10 mol%', '0.003', '0.66; 0.016', '0.66', 'nan; 72 mM', '3.5 mol%', '0.05 | nan', '0.018', '12 mg/ml; 12 mg/ml', '0.0010537870472008782', '0.4', '0.0112', '0.012', '0.1 mg/ml', '0.5; 0.16', '0.26', '4 wt%', '0.02 mg/ml', 'nan; 0.54 %; 0.11 %', '0.034; 0.05', '2.8 mol%', '8 mg/ml', '100', '0.075', '0.077', '0.0615; 0.0185', '0.764 M', 'nan; 1.5 %', '8 µmol/ml', '0.6 mol%', '0.25; 0.75', '0.15', '0.032', '2.5 wt%', 'nan; 7 %', '0.3 mM', '0.5 mg/mlantisolvent', '0.014; nan', '40 mol%', 'nan; 0.25 %', '0.125 mol%', '3.6 vol%', '42.9', '0.05; 0.1', '0.5', '0.46', '20; 10', '5.268935236004391 e-05', '4 mol%', 'nan; 0.03', '0.0008', '0.00026344676180021956', '0.00027', '3 mol%; 10 mol%', '0.66; 0.1', 'nan; 0.125 %', 'nan; 3 wt%', 'nan; 0.5 mg/ml', '0.0075', '0.3 wt%', '0.04', '3', '3.0', '3.2', '2.5 wt%; 0.4 wt%', 'nan; 0.01', '0.39999999999999997', '0.3 mg/ml; 5 mg/ml', '1.009 M', '3 vol%', '0.034 mol%', '0.16666666666666666', '0.023', '0.6 M', '0.333; 0.003', '0.25 mg/ml', '0.07 mol%', '0.03 wt%', '0.003; 0.05', '50 mM', '5; 10', 'nan; 2.5 %', '0.126', '2 mg/ml', '9.6', 'nan; 0.2', '0.001', '0.063', '2.0 mg/ml', '0.15; 0.05', '30 mg/ml', '30 mol%', '0.21', '0.05 mg/ml', '0.05; nan; nan', '0.5 mol%', '0.031', '0.2 mg/ml', '0.0035', '0.75.0.01', '0.01 M', '0.75 M', '0.8 mg/ml', '1 mol%; 10 mol%', 'nan; 4 wt%', '0.11', '0.1 mM', '29 wt%', '16.67', '0.66; 0.05', 'nan; 0.75', '40 wt%', '0.5 mmol', '0.01; nan', '1.4 %5%', '0.04; 0.00075', '4.2 vol%', '1 wt%', '6.0', '0.008', '2 %; nan', '0.23', 'nan; 0.048 %', 'nan; 0.1 %', '0.66; 0.04', 'nan; 0.3 mg/ml', 'nan; 4 mg/ml', '1.5 mg/ml', '50 mg/ml', '15 mol%', '0.67', '12.5 mM', '2 wt%; nan', '5 g/ml', '2.5 wt%; 3 wt%', '0.0005', '1 mM', '33.3', '750 vol%', 'nan | 0.0003', '0.0056', '1 M', '15.0', '6.6 mol%', '1.00 wt%', '1 %; nan', '0.66; 0.01', '0.000285 mol%', '25 mol%', '0.1; 0.005', 'nan; 3 mol%', '0.17500000000000002', '0.002', '0.15000000000000002', '1.4 vol%; nan', '2 µmol/ml', '2.5 mol%', '0.10; 0.15', '0.9 M', '0.034', '3 mol%', '0.1; 0.075', '33 ul.57wt%', '1 mg/ml', '4 vol%', '0.0005268935236004391', '0.17', '0.068', '75', 'nan; 0.5 %', '2.21 wt%', '1.3 %; nan', '2 e-05', '0.15 wt%; 0.05 wt%', '1.4 %; 6 %', '0.1; 0.1', '10; 10', '0.01', '0.25; nan', '6 mol%', '0.028; 0.0003', '20 wt%', '0.014', '0.1', '5 mol%; 10 mg/ml', '0.079', 'nan; 4', '1.5 wt%; 1.0 wt%; 0.15 wt%; nan', '8 wt%', '0.28', '0.13', '0.033', '0.8 vol%', '1.2 wt.%', '9 mg/ml', '0.1; nan', '0.075 wt%', '8 %; 33 %', '3 mM', '10 mol%', 'nan; 2 %; 6 %', 'nan; 10 %', '0.1; 0.01', '10.0', '0.125; 40 mg/ml', '0.024', '2.5 wt%; 0.15 wt%', '0.5 mg/ml', '2 mol%', '20', '0.14 mg/ml', '0.75 mol%', '0.00010537870472008782', '0.1; 0.03', '0.015', '1.2 mg/ml', '0.57', '7.5 g/ml', '0.01 wt%', '9 mol%', '0.19', 'nan; 0.4 mg/ml', '0.071', '350', '0.35; 0.038', '200 mol%', '2.5 mol%; 10 mol%', '0.14 mol%', '0.15 wt%; 0.05 wt%; 0.83 mol%', '4', '0.1; 0.04', '0.0068', '10', '0.25 mol%', '0.045', '20 mol%', '150 vol%', '0.15 M', '0.66; 0.16', '0.2 wt%', '0.1; 0.05', 'nan; 5', '2 %; 2 %', '1 vol%', '2.8 vol%; nan', '0.66; 0.008', '0.6 mg/ml; 10 mg/ml', '2.5 g/ml', 'nan; 2.0', '0.2 mol', '0.01; 0.04', '0.375', '0.001; 0.05', '12', '0.055', 'nan | 1 mol%', '0.1 M', '0.03; nan', '1 mg/perovskite_mmol', '0.03; 0.003', '0.8 wt.%', '0.029', '0.142', 'nan; 0.4', '0.04 M', '0.07500000000000001', 'nan; 7', '0.000175', '1.6', '0.0001', '0.1 wt%', '0.02; nan', '7.8 mg/ml', '10 mol%; 10 mol%', '1.75 mM', '1 %; 10 %', '5', '0.333; 0.018', '0.005; nan', '1.5 wt%', '0.15 wt%; 0.05 wt%; 1.67 mol%', '0.15; 0.1', '0.021', '0.1 w%', '60', '0.005 vol%', 'nan; 0.3', '0.12 mg/ml; nan', '0.04; 0.01', 'nan; 0.1', '0.4; 0.7', '1.61 e-05', '3 %; 3 %', '70 mol%', '0.02 wt%', '7.5 W%', '0.75', '0.009; nan', '0.04 mg/ml', '0.0025', '2.4 mol%', '33 mol%', '1.8 mg/ml', '0.77'])))

    thickness = Quantity(
        type=str,
        shape=[],
        description="""
    The thickness of the perovskite layer
- If the perovskite contains more than one layer, separate those by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- State thicknesses in nm
- Every layer in the stack have a thickness. If it is unknown, state this as ‘nan’
- If there are uncertainties, state the best estimate, e.g write 100 and not 90-110
- For cells where the perovskite infiltrates a mesoporous scaffold, state the thickness as starting from the bottom of the infiltrated mesoporous layer to the top of the perovskite layer (i.e. include the thickness of the infiltrated mesoporous layer)
Example
200
500 |20
600 | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '90.0', '446.0', '325.0', '50.0', '220.0', '520.0', '137.0', '180.0', '45.0', '167.0', '668.0', '387.0', '191.0', '540.0', '353.0', '515.0', '233.0', '601.0', '354.9', '560.0', '461.0', '1020.0', '391.0', '297.0', '6000.0', '385.0', '1800.0', '107.0', '212.0', '95.0', '0.05', '760.0', '551.0', '274.0', '70.0', '447.0', '573.0', '500.0', '306.0', '275.0', '1300.0', '524.0', '2000.0', '788.0', '665.6', '54.0', '507.0', '484.0', '266.1', '291.0', '260.0', '175.0', '630.0', '1410.0', '267.0', '283.0', '455.0', '240.0', '615.0', '303.0', '150.0 | 200.0', '3600.0', '330.0', '810.0', '89.8', '775.0', '334.0', '408.0', '420.0', '26.0', '5500000.0', '85.0', '321.0', '1015.0', '463.0', '505.0', '150.0 | 500.0', '895.0', '354.0', '620.0', '709.0', '278.0', '56.0', '269.4', '650.0', '500.0 | nan', '126.0', '52.0', '20.0', '295.0', '165.0', '135.7', '265.0', '1400.0', '287.0', '372.0', '685.0', '1100.0', '359.0', '150.0 | 300.0', '602.3', '314.0', '238.0', '250.0', '229.0', '375.0', '142.0', '2130.0', '65.0', '399.0', '317.0', '875.0', '150.0', '435.0', '464.0', '379.7', '769.0', '273.0', '12000.0', '610.0', '363.0', '965.0', '880.0', '247.0', '244.0', '50000.0', '302.2', '35.0', '750.0', '468.0', '135.0', '600.0 | 20.0', '730.0', '339.0', '870.0', '258.0', '324.0', '309.4', '183.0', '481.0', '660.0', '457.0', '407.0', '342.0', '336.0', '285.0', '401.0', '25.0', '294.0', '983.0', '3100.0', '172.5', '405.0', '40.0', '67.0', '230.0', '522.0', '75.0', '334.2', '272.0', '288.0', '577.0', '331.0', '150.0 | 400.0', '480.0', '298.0', '190.0', '604.0', '365.0', '301.0', '223.0', '380.0', '700.0', '93.0', '575.0', '155.0', '529.0', '780.0', '217.0', '314.8', '599.8', '280.0', '60.0', '125.0', '465.0', '3370.0', '40000.0', '440.0', '900.0', '122.0', '400.0', '950.0', '572.0', '538.0', '510.0', '296.0', '840.0', '315.0', '498.0', '121.0', '242.0', '425.0', '512.0', '213.0', '600.0', '850.0', '100.0', '105.0', '740.0', '720.0', '485.0', '1150.0', '252.0', '417.0', '607.0', '459.3', '526.0', '511.4', '338.0', '690.0', '200.0', '453.0', '350.0', '104.0', '290.0', '195.0', '311.0', '890.0', '585.0', 'nan | nan', '547.0', '340.0', '202.0', '596.0', '160.0', '209.0', '38.0', '584.0', '653.0', '270.0', '170.0', '390.0 | 10.0', '337.0', '670.0', '151.0', '326.0', '24500.0', '386.0', '1205.0', '725.0', '171.0', '413.0', '445.0', '248.0', '640.0', '2080.0', '355.0', '150.0 | 100.0', '394.0', '765.0', '293.0', '478.0', '1130.0', '563.0', '550.0', '531.0', '215.0', '542.0', '300.0', '513.6', '245.0', '430.0', '1010.0', '395.0', '185.0', '308.0', '289.0', '530.0', '120.0', '450.0', '501.0', '218.0', '153.0', '130.0', '110.0', '310.0', '150.0 | 600.0', '141.0', '199.0', '11000.0', '490.0', '345.0', '580.0', '1000.0', '232.6', '188.0', '66.0', '225.0', '235.0', '60000.0', '528.0', '590.0', '770.0', '1200.0', '444.0', '845.0', '473.0', '1070.0', '390.0', '10.0', '43.0', '1650.0', '1393.0', '236.0', '266.0', '742.6', '412.0', '20000.0', '210.0', '397.0', '357.0', '424.0', '2200.0', '567.0', '800.0', '586.0', '525.0', '388.0', '378.0', '370.0', '129.0', '348.0', '960.0', '149.0', '477.0', '570.0', '487.0', '454.0', '7000.0', '97.0', '410.0', '30.0', '462.0', '510.2', '792.0', '360.0', '470.0', '335.0', '680.0', '438.0', '460.0', '224.0', '400.0 | 2.0', '147.0', '474.0', '1510.0', '140.0', '578.0', '492.0', '80.0', '404.0', '320.0', '4000.0', '138.0', '448.0', '10000.0', '226.0', '276.0'])))

    band_gap = Quantity(
        type=str,
        shape=[],
        description="""
    The band gap of the perovskite
- If the perovskite contains more than one layer, separate the band gaps for the respective layer by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If there are uncertainties, state the best estimate, e.g. write 1.62 and not 1.6-1.64
Example
1.62
1.57 | 2.3
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1.764', '1.189', '1.516', '2', '1.578', '1.645', '1.583', '2.115', '1.715', '1.621', '2.44', '1.533', '1.486', '1.481', '2.6', '1.555', '1.73', '1.63 | nan', '1.633', '1.61', '2.04', '1.595', '2.78', '1.617', '1.526', '2.296', '1.626', '2.4', '1.72', '1.46', '1.613', '1.47', '1.91 | 1.8', '1.79', '1.623', '2.37', '1.17', '2.27', '2.42', '1.608', '1.515', '1.855', '1.525', '1.8', '1.56', '2.2', '1.605', '1.632', '1.523', '1.671', '1.503', '1.27', '2.13', '1.569', '1.684', '1.881', '1.635', '1.597', '1.627', '1.652', '2.43', '2.28', '1.775', '2.277', '1.588', '1.88', '3.04', '2.09', '1.655', '2.9', '2.3', '2.07', '2.31', '1.58', '2.38', '1.57', '1.3', '2.49', '1.901', '1.641', '1.55', '1.89', '1.66', '1.556', '1.546', '1.94', '1.498', '1.49', '1.77', '2.92', '2.12', '1.64', '1.45', '1.919', '1.549', '1.76', '1.33', '1.41', '2.39', '1.584', '1.579', '1.638', '1.592', '1.624', '1.965', '1.65', '1.26', '1.656', '1.644', '1.596', '1.906', '1.598', '1.54', '1.35', '1.21', '1.96', '2.5', '1.51', '2.268', '2.01', '2.273', '2.278', '1.674', '2.161', '1.865', '2.154', '1.99', '1.71', '1.683', '1.739', '1.78', '1.36', '1.848', '2.23', '2.19', '1.924', '1.92', '1.599', '1.872', '1.564', '1.582', '1.791', '1.612', '2.66', '2.153', '1.69', '1.565', '1.651', '2.46', '1.839', '2.14', '1.895', '1.589', '2.26', '2.36', '2.15', '1.812', '1.87', '1.548', '1.4', '1.5', '2.25', '2.027', '2.0', '1.594', '1.9', '1.581', '1.591', '1.28', '2.287', '2.48', '1.614', '1.39', '1.58 | 2.08', '1.86', '1.68', '1.25', '1.576', '1.575', '1.629', '1.375', '1.74', '1.16', '1.38', '1.957', '1.667', '1.609', '1.7', '1.557', '1.604', '2.139', '2.033', '1.602', '1.97', '2.34', '1.896', '1.634', '1.586', '1.2', '1.85', '1.535', '2.05', '1.18', '1.553', '2.21', '2.7', '1.607', '2.47', '1.606', '1.529', '1.639', '1.611', '1.752', '1.688', '1.32', '1.93', '2.22', '1.75', '1.616', '1.574', '2.308', '1.524', '1.23', '1.976', '1.845', '1.59', '2.105', '2.03', '1.725', '1.676', '2.18', '1.672', '1.34', '1.682', '2.1', '1.662', '2.54', '2.536', '2.8', '1.53', '2.113', '1.531', '1.505', '1.858', '1.67', '1.528', '2.35', '1.585', '1.751', '1.63', '1.52', '1.615', '2.02', '1.571', '1.866', '1.42', '1.619', '1.272', '1.573', '1.777', '1.893', '1.746', '1.517', '1.84', '1.509', '1.889', '1.31', '1.636', '1.628', '1.48', '1.62', '1.29', '1.82', '1.95', '1.668', '1.969', '1.91', '1.512', '1.593', '2.83', '2.141', '1.6 | 1.68', '2.29', '2.24', '2.08', '1.22', '2.288', '1.6', '1.587', '2.58', '1.757', '2.52', '1.98', '2.55', '1.37', '1.974', '1.989', '1.43', '1.24', '1.625', '1.701', '1.554', '1.637', '1.44', '2.17', '1.81', '2.32', '1.558', '2.33', '1.83'])))

    band_gap_graded = Quantity(
        type=str,
        shape=[],
        description="""
    TRUE if the band gap varies as a function of the vertical position in the perovskite layer.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['true', 'false'])))

    band_gap_estimation_basis = Quantity(
        type=str,
        shape=[],
        description="""
    The method by which the band gap was estimated. The band gap can be estimated from absorption data, EQE-data, UPS-data, or it can be estimated based on literature values for the recipe, or it could be inferred from the composition and what we know of similar but not identical compositions.
Example
Absorption Tauc-plot
Literature
Composition
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Composition', 'Literature', 'EQE', 'Absorption', 'Absorption Tauc-plot', 'UPS', 'Absorption Tauc-plot | UPS'])))

    pl_max = Quantity(
        type=str,
        shape=[],
        description="""
    The maximum from steady-state PL measurements
- If more than one PL-max, separate those by a semicolon
Example
780
550; 770
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '779.8', '795', '520.0', '761.0', '632.6', '779.0', '695', '828.0', '728', '782.0', '582.1', '589', '772', '952', '553', '540.0', '774.5', '825', '610', '1.61', '560.0', '731', '767', '722.0', '640', '729', '746', '659.5', '781.5', '760.0', '819', '544', '716', '786.7', '590.4', '824', '683', '708.5', '758.5', '746.0', '788.0', '538', 'nan | 740', '795.0', '757.0', '793', '602', '518', '543.8', '626', '673', '794.8', '790', '813', '770', '805.5', '763.5', '934.0', '819.5', '595', '663.1', '830', '715.4', '762', '884', '744', '540', '810.0', '775.0', '762.5', '761', '724.0', '800.8', '766', '765.4', '777.0', '719', '570', '727', '918.0', '815.8', '890', '774.0', '780.3', '825.0', '806', '817', '580', '620.0', '820.0', '910.0', '625', '650.0', '803', '538.2', '781.3', '760.4', '775.4', '799', '882', '754.0', '965', '932', '538.6', '584.9', '755.6', '770.1', '531', '573', '976', '644.0', '632', '667', '810', '718', '755', '654', '767.1', '794', '692', '828', '636.0', '798', '636', '729.4', '777', '752.0', '778.8', '769.0', '539.1', '870', '723.0', '715', '639.1', '465', '741', '587.6', '685', '778', '880.0', '769.9', '764.8', '784', '716.0', '751', '728.0', '614', '780', '804', '751.0', '723', '736.0', '838', '722', '750.0', '776.2', '805.0', '730.0', '801.8', '529', '870.0', '764', '549', '833', '771.2', '769.4', '832', '743.5', '946.0', '915', '900', '639', '768.8', '807.0', '767.0', '551', '630', '984.0', '691', '635', '595.0', '843.5', '525', '735', '638', '635.8', '913', '657.0', '778.0', '757.5', '786', '637', '804.3', '759.0', '731.0', '783', '746.9', '661', '799.0', '300', '624', '943', '704.5', '807', '1005', '642', '677.5', '537.3', '732', '877', '672.0', '800', '894', '692.0', '713', '700.0', '537.4', '686', '22.3', '748', '780.0', '712', '558.0', '772.0', '552', '766.0', '797', '750', '645', '1200', '0', '533.0', '805', '756.6', '996.4', '726.0', '440.0', '605', '670.2', '750 | nan', '868', '526', '725', '720.9', '771.8', '743', '541', '771', '530', '796', '761.4', '680', '802', '682', '575', '754', '832.2', '690', '550', '657', '722.1', '515', '512.0', '985.7', '600.0', '699', '740.0', '740', '822', '753.0', '633', '720.0', '977', '785', '765', '959', '880', '811', '527.0', '769', '975', '961', '775.6', '655', '840', '600', '912', '650', '752', '790.0', '873', '725.1', '535', '583', '896', '539', '806.0', '950', '704', '546', '812', '787', '755.0', '620', '820', '962', '774', '768.0', '670.0', '643', '200', '759', '920', '778; 710', '543.0', '77', '708', '789', '964', '545', '768.3', '725.0', '769.1', '1025', '843', '982', '615', '532', '506', '710.0', '764.0', '664.0', '445.0', '756', '765.0', '865', '785.3', '647.0', '21.4', '675', '649.4', '550.0', '531.0', '834', '763', '781', '726', '756.7', '596', '490', '707', '530.0', '805.1', '779', '527', '1058', '960', '714', '719.0', '969', '889', '792', '420', '666.6', '520', '541.4', '758.0', '576.7', '751.5', '946', '955', '757', '809', '730', '1020', '815', '528.0', '770.0', '802.0', '781.1', '894.0', '651', '753', '724', '652', '767.9', '627', '788', '543', '821', '967', '768', '742', '646', '800.0', '816', '748.0', '922', '525.0', '1016', '660', '665', '727.0', '745', '705', '768.4', '781.0', '663', '534', '785.7', '779.9', '700', '791', '522', '784.0', '696', '710', '782', '776', '775', '968.8', '659', '792.0', '814', '777.6', '537', '717', '763.0', '720', '745.0', '980', '756.8', '680.0', '729.0', '749', '773', '538.7', '547', '800 | 620', '860', '653', '796.0', '670', '762.0', '519', '776.0', '485', '524', '738.0', '808', '658', '794.0', '629.4', '655.0', '758', '895', '528', '760', '779.2', '738', '733', '700.5', '776.5', '712.7', '773.0', '801'])))

    storage_time_until_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    The time between the perovskite stack is finalised and the next layer is deposited
- If there are uncertainties, state the best estimate, e.g. write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['1200.0', '144.0', '300.0', '1440.0', '24.0', '2400.0', '3120.0', '2160.0', '960.0', '120.0', 'Unknown', '12.0', '72.0', '240.0', '720.0', '1920.0', '432.0', '480.0', '168.0', '48.0', '2880.0', '1680.0', '192.0', '4.0'])))

    storage_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere in which the sample with the finalised perovskite stack is stored until the next deposition step.
Example
Air
N2
Vacuum
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown', 'Air', 'Ambient', 'Vacuum', 'N2', 'O2', 'Ar'])))

    storage_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The time between the perovskite stack is finalised and the next layer is deposited
- If there are uncertainties, state the best estimate, e.g write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '40.0', '30.0', '20.0', '75.0', '45.0', '90.0', '24.0', '60.0', '50.0', '10.0', '70.0', '80.0', '35.0'])))

    surface_treatment_before_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    Description of any type of surface treatment or other treatment the sample with the finalised perovskite stack undergoes before the next deposition step.
- If more than one treatment, list the treatments and separate them by a double forward angel bracket (‘ >> ‘)
- If no special treatment, state that as ‘none’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Examples:
none
UV
Ozone
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'UV', 'Ar plasma'])))

    def normalize(self, archive, logger):
        from .formula_normalizer import PerovskiteFormulaNormalizer
        from nomad.atomutils import Formula

        add_solar_cell(archive)
        add_band_gap(archive, self.band_gap)

        if self.composition_short_form:
            archive.results.properties.optoelectronic.solar_cell.absorber = self.composition_short_form.split(' | ')

        if self.composition_long_form:
            if not archive.results.material:
                archive.results.material = Material()

            if self.dimension_3D or self.dimension_2D:
                if archive.results.material.structural_type is None:
                    archive.results.material.structural_type = 'not processed'
                elif self.dimension_3D:
                    archive.results.material.structural_type = 'bulk'
                elif self.dimension_2D and self.dimension_3D is False:
                    archive.results.material.structural_type = '2D'

            if self.composition_perovskite_ABC3_structure:
                if not archive.results.material.symmetry:
                    archive.results.material.symmetry = Symmetry()
                    if archive.results.material.symmetry.structure_name is None:
                        archive.results.material.symmetry.structure_name = 'perovskite'
                    # remove archive.results.material.material_name if == 'perovskite'
                    if archive.results.material.material_name == 'perovskite':
                        archive.results.material.material_name = None

            if archive.results.material.functional_type is None:
                archive.results.material.functional_type = ['semiconductor', 'solar cell']

            formula_cleaner = PerovskiteFormulaNormalizer(self.composition_long_form)
            final_formula = formula_cleaner.clean_formula()
            chemical_formula_reduced = final_formula[0]
            archive.results.material.elements = final_formula[1]
            try:
                archive.results.material.chemical_formula_hill = Formula(chemical_formula_reduced).format('hill')
                archive.results.material.chemical_formula_reduced = Formula(chemical_formula_reduced).format('reduced')
                archive.results.material.chemical_formula_iupac = Formula(chemical_formula_reduced).format('iupac')
                archive.results.material.chemical_formula_anonymous = Formula(chemical_formula_reduced).format('anonymous')
                archive.results.material.chemical_formula_descriptive = formula_cleaner.pre_process_formula()
            except Exception as e:
                logger.warn('could not analyse chemical formula', exc_info=e)


class PerovskiteDeposition(MSection):
    """
    This section contains information about the deposition of the perovskite layer.
    """

    number_of_deposition_steps = Quantity(
        type=np.dtype(np.int64),
        shape=[],
        description="""
    The number of production steps involved in making the perovskite-stack
- A spin coating program that are composed of several segments with different spin speed are still counted as one step (1)
- A spin coating program involving an antisolvent step counts as a 1-step method (1).
- Depositing PbI2 first and subsequently converting it to a perovskite count as a 2-step procedure (2)
- Thermal annealing is considered separately. The motivation for this is that every step is considered to have its own thermal history.
                    """,
        a_eln=dict(
            component='NumberEditQuantity'))

    procedure = Quantity(
        type=str,
        shape=[],
        description="""
    The deposition procedures for the perovskite block.
- The perovskite stack is considered as one block/layer when we consider the synthesis. Thus, even if the perovskite is layered, consider it as one block, i.e. no vertical bars in this field
- When more than one reaction step, separate them by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- There should be as many reaction steps described here as indicated in the field “Perovskite. Deposition. Number of deposition steps”
- Thermal annealing is generally not considered as an individual reaction step. The philosophy behind this is that every deposition step has a thermal history, which is specified in a separate filed. In exceptional cases with thermal annealing procedures clearly disconnected from other procedures, state ‘Thermal annealing’ as a separate reaction step.
- Antisolvent treatment is considered in a separate filed. The motivation for that is that it usually is conducted simultaneously as a spin-coating procedure, and thus acts as an additional aspect of reaction step already accounted for. Exception to this is if there is an antisolvent step that is distinctly separated in time, e.g. a film with a spin-coated perovskite solution is immersed in an antisolvent. In that case, this could eb added as a dipp-coating event, while also being reported in the antisolvent field.
- Even if the most common deposition procedures have been used for 95 % of all reported devise, do not be surprised if you do not find your deposition procedure in the list of reported deposition procedure, as the original dataset tended to use a simplified notation.
- A few clarifications
- Air brush spray
- Deposition with something looking like an old perfume bottle. Classified as a solution technique.
- Brush painting
o A precursor ink is applied with a brush
- CBD
- Chemical bath deposition. Refers to procedures where a film has been immersed in a solution where a reaction occurs. The typical example is when a PbI2 film is immerse in an IPA solution with MAI in which the PbI2 is converted to the perovskite.
- Co-evaporation
- Simultaneous evaporation from multiple sources with line of sight deposition.
- CVD
o Chemical vapour deposition. A gas phase process where a chemical reaction is occurring in the gas phase. If a MA-containing compound is evaporated and reacted with PbI2 where another species is released to the gas phase, it is labeled as CVD. A process where MAI in gas phase react with PbI2 in gas phase is labelled as CVD. A process where MAI or MA gas is reacting with solid PbI2 is instead labelled as a gas reaction as no chemical reaction is occurring the gas phase. Note that all reactions labelled as CVD in the literature may not be CVD even if it is conducted in a CVD reactor, and should instead be labelled as a gas reaction.
- Diffusion
o Solid state reaction where two solid components are mixed. E.g. solid MAI is bought in direct contact with solid PbI2
- Diffusion-gas reaction
- A special case. Where one compound, e.g. MAI is placed on top of another e.g. PbI2 where it is evaporated. It is thus a combination of a gas phase reaction and solid-solid diffusion.
- Dipp-coating
o The thing that separates dipp-coating from CBD is the occurrence of a reaction. If you have component A in solution, dip your substrate in the solution, take it up and you have component A on your substrate, then you have done a dipp-coating. If you have substance A in solution, dip your substrate in the solution, take it up and have something else than A on your substrate, you have done a CBD (e.g. PbI2 dipped in MAI/IPA which gives MAPbI3 and not MAI on the substrate)
- Dropcasting
o A drop is applied to a substrate where it is left to dry without any additional procedures.
- Drop-infiltration
- A mesoporous scaffold in which a drop of the precursor solution is infiltrated without the aid of spin-coating.
- Doctor blading
- There is a family of related techniques, but if it could be described as doctor blading, that is the label to use.
- Evaporation
- Refers to thermal evaporation with line-of-sigh deposition. i.e. PVD
- Flash evaporation
- Fast evaporation (in a flash) of a perovskite that sublimes on another substrate. Line of sight deposition.
- Closed space sublimation
- Evaporation of a well controlled amount of substance (usually in the form of a thin film) in a small container containing the final substrate.
- Gas reaction
- A gas phase reaction. Not a line of sight deposition. In the typical case, MAI is evaporated and the MAI gas builds up a pressure in the reaction chamber in which it reacts with a PbI2 film forming the perovskite.
- Ion exchange
- One perovskite is dipped into a solution (or exposed to a gas) which leads to an ion exchange, e.g. I is replaced by Br.
- Lamination
- A readymade film is transferred directly to the device stack. A rather broad concept. An everyday kitchen related example of lamination would eb to place a thin plastic film over a slice of pie.
- Recrystallization
- A perovskite that already have been formed is deformed and then recrystallised. E.g. MAPbI3 is exposed to Methylamine gas for a short while which dissolved the perovskite which then can crystallize again
- Rinsing
- Cleaning step with a solvent
- Sandwiching
- When a readymade top stack simply is placed on top of the device stack. Could be held together with clams.
- Ultrasonic spray
- A bit like air brush spray but with better control of droplet size. Classified as a solution technique.
Example
Spin-coating
Spin-coating >> Spin-coating
Spin-coating >> CBD
Spin-coating >> Gas reaction
Drop-infiltration
Co-evaporation
Doctor blading
Evaporation >> Evaporation
Evaporation >> Spin-coating
Evaporation >> Gas reaction
Slot-die coating
Spray-coating
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Spin-coating | Spin-coating >> Spin-coating', 'Gas-assisted fast crystallisation', 'Electrospraying >> Dropcasting', 'Slot-die coating >> Doctor blading', 'Spin-coating >> Spin-coating', 'Spin-coating >> Evaporation', 'Roller-coating >> Roller-coating', 'Spin-coating >> Drop-infiltration >> Recrystallization', 'Co-evaporation >> Evaporation >> Spin-coating', 'Evaporation >> Evaporation >> Evaporation', 'Spin-coating >> Vapour annealing >> CBD', 'Spin-coating >> CBD >> Spin-coating >> Spin-coating', 'Evaporation >> Gas reaction >> Washing', 'Spin-coating >> Diffusion-gas reaction', 'Spin-coating | Spin-coating', 'Spin-coating >> Gas reaction >> Ion exchange', 'Spin-coating >> Spin-coating >> Rinsing >> Spin-coating', 'Slot-die coating >> Dipp-coating', 'Spin-coating >> Spin-coating >> Gas reaction', 'Spin-coating >> Dipp-coating >> Rinsing', 'Spin-coating >> Spin-coating | Spin-coating', 'Evaporation >> Evaporation >> Gas reaction >> Dipp-coating', 'Slot-die coating', 'Brush painting', 'Ultrasonic spray >> Gas reaction', 'Spin-coating >> CBD', 'Co-evaporation >> Gas reaction >> Spin-coating', 'Evaporation >> Flash evaporation >> Evaporation >> Flash evaporation', 'Spin-coating >> Spray-coating', 'Ultrasonic spray', 'Spin-coating | Evaporation', 'Spin-coating >> CBD >> Spray-coating', 'Space-limited inverse temperature crystallization', 'Evaporation | Spin-coating', 'Spin-coating | Dripping', 'Meniscus coating', 'Spin-coating | Spin-coating >> IPA washing', 'Evaporation >> Sandwiching >> Rinsing', 'Co-evaporation', 'Evaporation >> CBD >> CBD', 'Co-evaporation >> Diffusion-gas reaction', 'Ultrasonic spray >> Ultrasonic spray', 'Spin-coating >> CBD >> Gas reaction', 'Evaporation >> Diffusion-gas reaction', 'Sputtering >> Gas reaction', 'Spin-coating >> CVD', 'Sputtering >> Sulfurization', 'Spin-coating >> Air brush spray >> Air brush spray', 'Spray-coating >> Gas reaction', 'Spin-coating >> Recrystallization', 'Inkjet printing', 'Spin-coating >> Spin-coating >> Spin-coating', 'Evaporation >> Spin-coating', 'Spin-coating >> Gas reaction >> Solvent annealing >> Recrystallization', 'Pulsed laser deposition >> Gas reaction', 'Dipp-coating >> Dipp-coating', 'Spin-coating >> CBD >> CBD', 'Spin-coating >> Closed space sublimation', 'Evaporation >> Evaporation >> Evaporation >> Evaporation >> Evaporation >> Evaporation', 'Co-evaporation >> Spin-coating', 'Air brush spray', 'Spin-coating | Spray-coating', 'Flash evaporation >> CBD', 'Doctor blading >> Doctor blading', 'Dropcasting >> Spin-coating', 'Spray-coating >> Spin-coating', 'Ultrasonic spray >> CBD', 'Spin-coating >> Dipp-coating', 'Single-source thermal evaporation', 'Inverse temperture crysilization >> Lamination', 'LT-SCD >> LT-SCD', 'Evaporation >> Gas reaction >> Gas reaction', 'Spin-coating >> Dropcasting >> Spin-coating', 'Vacuum flash evaporation', 'Drop-infiltration >> Recrystallization', 'Spray-coating >> Spray-coating', 'Evaporation >> Ultrasonic spray', 'Electrodeposition >> Spin-coating', 'Spin-coating >> Dipp-coating >> Spin-coating', 'Evaporation >> Spin-coating >> Spin-coating', 'Spin-coating >> Dropcasting', 'Dropcasting >> CBD', 'Spin-coating >> Evaporation >> Ultrasonic spray', 'Solvent evaporation crystallization', 'GC-LCG', 'Drop-infiltration >> Drop-infiltration', 'Dropcasting >> Pneumatic pressure', 'spin-coatng', 'Spin-coating >> Spin-coating >> Diffusion', 'Evaporation >> Evaporation >> Gas reaction', 'Evaporation >> Gas reaction', 'Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Dropcasting >> Rinsing', 'Doctor blading >> CBD', 'Spin-coating >> CBD >> Washing', 'PVD', 'Dipp-coating >> CBD', 'Spin-coating >> Inkjet printing', 'Spin-coating >> CBD >> Rinsing', 'Dipp-coating', 'Spin-coating >> Air brush spray', 'Hot-casting', 'Spin-coating >> Spin-coating >> Ion exchange', 'Springkling >> Recrystallization', 'Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Dropcasting >> Rinsing', 'Electrodeposition >> Dipp-coating >> CBD', 'Evaporation >> CBD >> CBD >> CBD', 'Drop-infiltration >> CBD', 'Spin-coating | Gas reaction', 'Spin-coating >> Spin-coating >> Air brush spray', 'Electrospraying >> Gas reaction', 'Spin-coating >> Diffusion', 'Evaporation', 'Spin-coating >> Evaporation >> Spray-coating', 'Electrodeposition >> Gas reaction >> CBD', 'Electrospraying', 'Unknown', 'Spray-pyrolys', 'Evaporation >> Inkjet printing', 'CBD', 'Drop-infiltration', 'Spin-coating >> CBD >> Ion exchange', 'Spin-coating >> Evaporation >> CBD', 'CVD', 'Roller-coating', 'Co-evaporation >> Co-evaporation', 'Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating', 'Unknown >> Unknown', 'Spin-coating >> Sandwiching', 'Spin-coating >> Spin-coating >> CBD', 'Electrodeposition >> CBD', 'Spin-coating >> Co-evaporation', 'Doctor blading', 'Spin-coating >> Printing', 'Magnetron sputtering', 'Evaporation >> Evaporation', 'Spin-coating >> Drop-infiltration', 'Spin-coating >> Ligand exchange >> Dipp-coating', 'Soft-cover deposition', 'Spin-coating >> Spin-coating >> Dipp-coating >> Dipp-coating >> Spin-coating >> Dipp-coating >> Dipp-coating >> Spin-coating >> Dipp-coating >> Dipp-coating', 'Slot-die coating >> Spin-coating', 'Evaporation >> Electrodeposition', 'Spin-coating >> Dipp-coating >> Dipp-coating >> Spin-coating >> Dipp-coating >> Dipp-coating >> Spin-coating >> Dipp-coating >> Dipp-coating', 'Spin-coating >> Evaporation >> Evaporation', 'Spin-coating >> Gas reaction', 'Space-confined single crystal formation', 'Electrospinning', 'Spin-coating >> CBD >> Spin-coating', 'Electrodeposition >> Gas reaction >> Spin-coating', 'Closed space sublimation', 'Spin-coating >> Spin-coating >> Spray-coating', 'Spin-coating >> CBD >> Recrystallization', 'Spin-coating | Spin-coating >> IPA washing | Spin-coating >> IPA washing | Spin-coating >> IPA washing', 'Spin-coating >> Spin-coating >> Evaporation', 'Co-evaporation >> Co-evaporation >> Co-evaporation', 'Spin-coating >> CBD >> Spin-coating >> Gas reaction', 'Spin-coating | Dropcasting', 'CBD >> Spin-coating >> Gas reaction', 'Substrate vibration assisted dropcasting >> Substrate vibration assisted dropcasting', 'Roller-coating >> Spin-coating', 'Spray-coating >> CBD', 'Spin-coating >> Gas reaction >> Gas reaction >> Gas reaction', 'Spin-coating >> Spin-coating >> Dropcasting >> Rinsing', 'Spin-coating | CBD', 'Dropcasting', 'Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating', 'Sputtering >> Spin-coating', 'Pulsed laser deposition', 'Single crystal growth', 'Evaporation >> Diffusion', 'Spin-coating >> Electrospraying', 'Blow-drying', 'Spin-coating >> Spin-coating >> Close space sublimation', 'Dipp-coating >> Spin-coating', 'Spin-coating | Spin-coating >> IPA washing | Spin-coating >> IPA washing', 'Inkjet printing >> Diffusion-gas reaction', 'Spin-coating >> Ion exchange', 'Drop-infiltration >> Spin-coating', 'Spin-coating >> Spin-coating >> Recrystallization', 'Flash evaporation', 'Spin-coating', 'Spin-coating >> Gas reaction >> Spin-coating', 'Electrodeposition >> CBD >> CBD', 'Crystallization >> Recrystallization', 'Slot-die coating >> CBD', 'Hot-pressed', 'Dipp-coating >> Gas reaction', 'Spray-coating', 'Spin-coating >> Spin-coating >> Spin-coating >> Dropcasting >> Rinsing', 'Air brush spray >> Air brush spray', 'Electrodeposition >> Electrodeposition', 'Sputtering >> CBD', 'Spin-coating >> Ultrasonic spray', 'Evaporation >> CBD', 'Spin-coating >> Dipp-coating >> Dipp-coating', 'Evaporation >> Evaporation >> Evaporation >> Evaporation', 'Slot-die coating >> Slot-die coating', 'Spin-coating >> Gas reaction >> Gas reaction', 'Spin-coating >> Condensation >> CBD'])))

    aggregation_state_of_reactants = Quantity(
        type=str,
        shape=[],
        description="""
    The physical state of the reactants
- The three basic categories are Solid/Liquid/Gas
- The perovskite stack is considered as one block/layer when we consider the synthesis. Thus, even if the perovskite is layered, consider it as one block, i.e. no vertical bars in this field
- When more than one reaction step, separate the aggregation state associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Most cases are clear cut, e.g. spin-coating involves species in solution and evaporation involves species in gas phase. For less clear-cut cases, consider where the reaction really is happening as in:
o For a spray-coating procedure, it is droplets of liquid that enters the substrate (thus a liquid phase reaction)
o For sputtering and thermal evaporation, it is species in gas phase that reaches the substrate (thus a gas phase reaction)
Example
Liquid
Gas >> Liquid
Liquid >> Liquid >> Liquid
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Liquid >> Unknown', 'Liquid >> Gas >> Gas >> Gas', 'Liquid | Liquid >> Liquid | Liquid >> Liquid', 'Liquid | Liquid', 'Liquid >> Liquid >> Solid', 'Gas', 'Liquid | Liquid >> Liquid', 'Liquid | Liquid >> Liquid | Liquid >> Liquid | Liquid >> Liquid', 'Gas >> Liquid >> Liquid >> Liquid', 'Liquid >> Liquid >> Liquid', 'Liquid >> Liquid >> Liquid >> Gas', 'Gas >> Solid >> Liquid', 'Solid >> Liquid', 'Liquid | Gas', 'Unknown', 'Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid', 'Liquid >> Liquid >> Liquid >> Liquid', 'Liquid >> Liquid', 'Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid', 'Gas >> Gas >> Liquid', 'Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid', 'Liquid >> Gas', 'Liquid >> Gas >> Gas', 'Liquid | Liquid >> Liquid | Liquid', 'Liquid >> Gas >> Liquid', 'Solid', 'Liquid', 'Gas >> Gas >> Gas >> Gas', 'Liquid >> Liquid | Liquid', 'Gas >> Gas >> Gas >> Liquid', 'Liquid >> Liquid >> Gas', 'Unknown >> Liquid', 'Liquid >> Liquid >> Liquid >> Liquid >> Liquid', 'Gas >> Gas >> Gas', 'Gas >> Gas >> Gas >> Gas >> Gas >> Gas', 'Liquid >> Solid', 'Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid >> Liquid', 'Gas >> Liquid', 'Gas >> Gas', 'Gas >> Liquid >> Liquid', 'Gas | Liquid', 'Gas >> Solid'])))

    synthesis_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The synthesis atmosphere
- When more than one reaction step, separate the atmospheres associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Air
N2 >> N2
Vacuum
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['N2 | Vacuum', 'Inert', 'DMSO; N2 >> N2', 'Air >> MAI; N2', 'FAI >> Unknown', 'FABr', 'FAI >> FABr @ 75 >> 25', 'Methylamin; N2', 'Vacuum >> MAI; toluene', 'Ar; MAI; Pbl2', 'N2 >> N2; TBP', 'Air >> Ar', 'Ar; MAI; PbI2', 'Methylamin; N2 >> Methylamin; N2', 'Acetone; N2 >> N2', 'FAI; N2; Vacuum', 'N2 >> Vacuum', 'Air >> MAI; NH4Cl; Vacuum', 'Vacuum >> MAI; Vacuum >> Vacuum; BAI', 'Unknown >> Air', 'Air >> Ar; Methylamin', 'Vacuum >> FAI', 'N2 >> Air; Methylamin; HI', 'Vacuum >> Vacuum >> IPA', 'N2 >> MABr >> MAI', 'Air >> MAI; Vacuum', 'Unknown >> Vacuum', 'Unknown >> Br2', 'O2', 'Unknown >> Unknown >> Air', 'Air >> MACl >> MAI', 'N2 >> MAI; N2', 'Unknown >> Methylamin', 'N2 >> N2; IPA >> N2', 'N2 >> Dry air', 'N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2', 'Vacuum >> N2', 'N2 >> Air; Methylamin >> Air; HI', 'N2 >> Chlorobenzene; N2 >> N2', 'Dry air | Dry air', 'N2 >> N2 >> N2 >> N2', 'Vacuum >> FAI; Vacuum', 'N2; Vacuum', 'Vacuum >> Vacuum', 'Dry air >> MAI', 'Vacuum >> Vacuum >> N2', 'Ar', 'Unknown >> MAI; Vacuum', 'Air >> Air; I2 >> N2', 'Air >> N2 >> Air', 'N2', 'Ar; MAI; PbCl2', 'Vacuum >> Air; MAI', 'N2 >> FAI; MAI; Vacuum', 'N2 | N2 | N2 | N2', 'Vacuum >> IPA; MAI', 'Vacuum >> TiBr4', 'Air >> Air >> Air', 'FAI', 'N2 >> FA-Ac', 'Air >> MaBr', 'N2 >> N2 >> FAI; Vacuum', 'Vacuum >> Ar; MAI', 'N2 >> MAI', 'Vacuum >> Vacuum >> Unknown', 'N2 | N2 | N2', 'Liquid Air', 'N2 >> MACl', 'Vacuum >> N2; MAI', 'N2 >> N2; Toluene >> N2', 'Air >> Vacuum >> N2', 'Ambient >> Air; MAI', 'N2 >> FAI; Vacuum >> N2', 'Vacuum >> FAI; N2', 'Ar >> MABr', 'Unknown >> Unknown >> Vacuum', 'Unknown >> Unknown >> Methylamin', 'Air >> Vacuum >> Air', 'Vacuum >> Chlorobenzene; MAI', 'Air >> Air; MABr', 'Ambient >> Ambient', 'Air >> MAI; MACl; MABr; Vacuum', 'N2 >> Air; Methylamin', 'Dry air', 'Vacuum >> Vacuum >> Vacuum >> Vacuum >> Vacuum >> Vacuum', 'Unknown >> N2; MAI', 'Unknown >> MAI', 'Vacuum >> Vacuum >> FAI; N2', 'Unknown >> FAI', 'N2 >> Air; DMSO >> Air; DMSO >> Air; Methylamin', 'Vacuum >> N2 >> N2 >> N2', 'Dry air >> Dry air >> Dry air', 'Chlorobenzene; N2', 'DMF; N2 >> N2', 'N2 >> Vacuum >> N2', 'Unknown >> Unknown >> MAI', 'Inert >> Inert', 'Air >> Air >> Pyridine', 'Unknown >> Vacuum >> Air', 'Unknown >> Air; Methylamin', 'Vacuum >> Ar', 'FAI; FABr >> Unknown', 'Air >> Air | Air', 'Unknown >> MABr', 'Vacuum >> Unknown', 'N2 >> N2 >> N2', 'Unknown', 'Ar >> Ar >> Ar', 'Air >> Br2', 'N2 >> N2', 'N2 >> Methylamin; N2', 'N2 >> N2 >> MAI; N2', 'Unknown >> Unknown >> Unknown', 'Dry air >> Dry air', 'Unknown >> Pyridine', 'N2 >> Air; MAI', 'Air >> Air; MAI', 'N2 >> FA', 'Hydrazine; N2', 'N2 >> Ambient', 'Vacuum >> Vacuum >> Vacuum', 'FAI >> FABr @ 25 >> 75', 'Vacuum >> MAI', 'Ambient', 'Unknown >> Unknown', 'N2 >> Vacuum >> Vacuum', 'Air >> Air; MACl', 'Unknown >> MACl', 'Air >> Air; Methylamin', 'Air >> Methyl amine', 'FAI >> FABr @ 50 >> 50', 'Air >> MAI', 'Vacuum >> MAI; Vacuum; FAI >> N2', 'Ait >> Air; MAI', 'Air', 'N2 >> MAI >> MABr', 'Vacuum', 'Air >> FAI; FABr; MACl; Vacuum', 'Dry air >> Air; MAI', 'Vacuum >> Vacuum >> Vacuum >> Vacuum', 'Unknown >> N2; MAI >> Unknown', 'Air >> Air', 'N2 >> Air; DMSO; HBr >> Air; DMSO >> Air; Methylamin', 'N2 >> Air', 'Air >> Air; HI >> Air; Methylamin >> Air; Hi', 'Vacuum >> MAI; N2', 'Vacuum >> Vacuum >> I2; N2 >> N2', 'Vacuum >> Air', 'N2 >> MAI; N2; O2', 'Vacuum >> Vacuum >> FAI; Vacuum', 'Air >> FAI; Vacuum', 'Glovebox', 'Air >> Air >> N2', 'Air >> Methylamine', 'Air >> N2', 'Vacuum >> FAI; N2; Vacuum', 'Air >> Methylamin; Vacuum', 'Unknown >> Unknown >> Unknown >> MAI', 'N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2', 'N2 >> MAI; Vacuum', 'Air | Ari; MAI', 'Air >> MAI; NH4Cl', 'N2 >> FAI; FACl; Vacuum', 'N2 >> N2 >> Vacuum', 'N2 | N2', 'N2 >> FAI; Vacuum', 'Air >> Air >> Air >> Air', 'N2 >> Methylamin', 'N2 >> BEAI2; N2', 'Dry air >> Dry air; Methylamin', 'Air >> Air >> Air >> Air >> Air >> Air', 'N2 >> DMSO; N2 >> N2', 'Air | Air', 'IPA; N2 >> N2', 'Ar >> Ar', 'N2 >> Methylamin; Vacuum', 'Air; O2', 'Air >> Vacuum', 'Vacuum >> N2 >> N2', 'N2 >> Ar; MAI', 'N2 >> FAI; MABr; MACl; N2; Vacuum', 'Ar >> Vacuum', 'Vacuum >> MAI; Vacuum', 'N2 >> MABr', 'Unknown >> Air >> Air', 'N2 >> N2; 1,2-dichlorobenzene >> N2'])))

    synthesis_atmosphere_pressure_total = Quantity(
        type=str,
        shape=[],
        description="""
    The total gas pressure during each reaction step
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of deposition steps must line up with the previous columns.
- Pressures can be stated in different units suited for different situations. Therefore, specify the unit. The preferred units are:
o atm, bar, mbar, mmHg, Pa, torr, psi
- If a pressure is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 100 pa and not 80-120 pa.
Example
1 atm
0.002 torr
1 atm >> 1 atm >> nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1 atm >> 1 atm', '0.000001 mbar', '1 atm | 1 atm', '1 atm >> 0.00003 bar', '1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm', '1 atm >> 60 Pa', '1 atm', '0.000001 mbar >> 1 atm', '1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm', '1 atm >> 1 atm | 1 atm', '0.00001 mbar', 'nan | 700 Pa'])))

    synthesis_atmosphere_pressure_partial = Quantity(
        type=str,
        shape=[],
        description="""
    The partial pressures for the gases present during each reaction step.
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the partial pressures and separate them with semicolons, as in (A; B). The list of partial pressures must line up with the gases they describe.
- In cases where no gas mixtures are used, this field will be the same as the previous filed.
Example
1 atm
0.002 torr; 0.03 torr
0.8 atm; 0.2 atm >> 1 atm >> nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1 atm >> 1 atm', '0.000001 mbar', '1 atm | 1 atm', '1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm', '1 atm', '0.000001 mbar >> 1 atm', '1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm >> 1 atm', '1 atm >> 1 atm | 1 atm', '0.00001 mbar', '1 atm >> 0.00003 bar'])))

    synthesis_atmosphere_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relative humidity during each deposition step
- When more than one reaction step, separate the relative humidity associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of deposition steps must line up with the previous columns
- If the relative humidity for a step is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 35 and not 30-40.
Example
35
0 >> 20
25 >> 25 >> 0
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '40.0', '25', '30 >> 30', '0 >> 40 >> 0', '15', '0 >> 0', '0.4', '0 >> 1', '0 >> 22 >> 0', '0', '1', '5', '85', '32', '10.0', '70', '28', '2', '15 >> 15', '55 >> 55', '80', '30 | 30', '45', '20 >> 20', '40', '75', '20', '60', '65', '36', '45 >> 45', '0 >> 20', '24', '0 >> 16 >> 0', '30', '55', '8', '0 >> 30', '90', '25 >> 25', '35 >> 35', '0 >> 12 >> 0', '10 >> 0', '40 >> 40', '0.01', '52', '70 >> 70', '0 >> 33 >> 0', '27.5 >> 27.5 >> 27.5', '50 >> 50', '10 >> 10', '42', '0.0', 'nan >> 40', '50', '40 >> 40 >> 40', '0 >> 60', '65 >> 65', '35', '12 >> 12', '0 >> 40', '42 >> 42', '30; 40', '60 >> 60', '10'])))

    solvents = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in each deposition procedure for each layer in the stack
- When more than one reaction step, separate the solvents associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the solvents in alphabetic order and separate them with semicolons, as in (A; B)
- The number and order of deposition steps must line up with the previous columns.
- For non-liquid processes with no solvents, state the solvent as ‘none’
- If the solvent is not known, state this as ‘Unknown’
- Use common abbreviations when appropriate but spell it out when risk for confusion
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
DMF; DMSO
GBL
DMF >> IPA
DMF >> none
DMF; DMSO >> IPA
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['DMF; DMSO >> H2O; Methanol', 'DMF >> none >> Unknown', 'Diiodoctane, DMF', 'Unknown >> none', 'n-methyl-2-pyrrolidone', 'acetonitrile; DMF; DMSO >> IPA', 'DMF', 'Octane >> Methyl acetate >> Ethyl acetate', 'DMSO; GBL >> DMF; IPA', 'DMSO | 1-pentanol', 'Ethanol', 'GBL; DMSO; n-butanol', 'DMF; DMSO >> Octane >> IPA', 'DMF; Furfural', 'Octane >> Methyl acetate', 'DMF; DMSO >> H2O; IPA', 'DMF; DMSO >> IPA >> IPA >> IPA', 'Octane >> IPA', 'none; DMF >> IPA', 'DMF >> Methanol', 'acetonitrile', 'acetonitrile; DMF', 'DMSO; N-methyl-2-pyrrolidone', 'DMF; DMSO >> IPA', 'GBL; N-methyl-2-pyrrolidinone', 'DMF; DMSO >> DMF; IPA', 'GBL; Ethanol', 'DMF; DMSO; NMP', 'none >> 2-methoxyethanol; Ethanol', 'DMF >> IPA >> IPA >> IPA', 'DMF >> H2O; IPA >> Unknown', 'DMSO; GBL >> IPA', 'DMF >> Ethyl acetate', 'DMF >> IPA >> Anisole', 'Diethyl ether; DMF; DMSO', 'DMF; Benzaldehyde', 'DMF; g-Valerolacetone', '1-Ethyl-2-pyrrolidone >> none', 'DMF >> DMSO >> IPA', 'Ethanol; Methylamine', 'DMSO; acetonitrile >> IPA', 'BA; Ethanol >> none', 'acetonitrile; NMP', 'Unknown >> Unknown', 'DMF >> n-hexyl alchohol', 'DMSO >> Hexane; Toluene', 'NMP', 'Dimethylacetamide; DMSO', 'Ethanol; H2O', 'DMF; DMSO >> n-butanol', 'DMF >> DMSO; IPA', 'DMSO >> 1-butanol', 'DMF; DMSO >> GBL; IPA', 'DMF >> Cyclohexene; IPA', 'Heptane; Hexane', 'GBL >> DMF', 'GBL', 'DMF; Methylfomramid', 'none >> none >> none >> none >> none >> none', 'DMF; GBL; IPA', '1-Octadecene; n-Heptane >> none', 'Octane >> Octane >> Octane >> Octane', 'Ethyl acetate', 'DMF; DMSO; GBL; NMP', 'DMF >> DMF >> Cyclohexene; IPA', 'Toluene', 'DMSO; NMP', 'DMF; DMSO >> IPA; o-dichlorobenzene', 'Acetic acid; GBL', 'DMF >> IPA; NMP', 'DMF; DMSO >> Methanol', 'DMF >> none >> none', 'N,N-dimethylacetamide', 'DMSO >> Methanol', 'DMF >> IPA; n-hexane', 'Chlorobenzene >> none', 'DMF >> Octane >> Methyl acetate >> Methyl acetate', 'Toluene >> IPA', 'DMAc; NMP', 'DMF >> Octane >> Octane >> Octane >> Octane >> Methyl acetate >> Methyl acetate', 'none', 'DMF >> Toluene', 'DMF; NMP >> IPA', 'acetonitrile; DMF >> IPA', 'DMF; GLB', 'Chlorobenzene', 'methylamine formate; IPA', 'DMF; THF', 'Ethyl amine; HCl', 'DMF; GBL', 'DMF >> H2O', '2-Methoxy-ethanol', '2‐butoxyethano; DMSO', 'DMF; Tetrahydrothiophene-1-oxide', '2-methoxyethanol; DMSO', 'Cyclohexyl-2-pyrrolidone; DMF', 'DMF; Benzylamine', 'DMSO; NMP >> IPA', 'DMF; DMSO >> Hexane', 'DMF >> IPA >> DMF; IPA', 'DMF >> Methanol | Cyclohexane', 'DMF; Tetrahydrothiophene 1-oxid', 'Cl-Cyclohexane; DMF', 'DMSO >> DMSO >> DMSO', 'DMF; NMP', 'DMF; Me-Cyclohexane', 'DMSO >> IPA >> Toluene', 'none >> none', 'DMSO; GBL @ 3; 7 >> IPA', 'DMF; DMSO >> Chloroform', 'DMF | IPA | H2O | Ethanol', 'DMSO; GBL @ 3; 7 >> DMSO; GBL @ 3; 7', 'DMF; DMSO', 'DMF; TBP >> IPA', 'DMSO >> IPA >> Toluene >> none', 'DMF >> IPA >> Chlorobenzene; DMF', 'DMF >> Unknown', 'Ethanol; H2O >> Chlorobenzene; Tert-butanol', 'DMF >> Chex; IPA', 'Unknown >> none >> none', 'acetonitrile; DMF; Methoxyactonitrile', 'none >> Methanol; Ethanol', 'DMSO >> Octane', 'Chlorobenzene >> Methylacetate', 'DMF; GBL; IPA >> IPA', 'Methylammonium acetate', 'DMSO; o-xylene', 'DMSO | Propanetriol', 'Ocatane >> MeOAc', 'DMSO >> IPA', 'DMF; HCl >> IPA', 'DMSO >> Ethanol', '2-methoxyethanol; acetonitrile', 'IPA >> DMF', 'DMF >> Pentan-1-ol >> Unknown', 'none >> IPA', 'DMF; HCl', 'DMF; DMSO >> Cyclohexane; IPA', 'DMF >> Cyclohexane; IPA', 'DMF; DMSO >> Chlorobenzene; IPA', 'DMF; o-DCB', 'DMF >> Methanol >> Methanol', 'Unknown >> IPA >> none', '2-methoxyethanol', 'H2O >> none', 'DMF >> IPA >> DMF >> IPA >> DMF >> IPA', 'n-propylamine', 'DMF >> none >> none >> none', 'THF', 'DMSO >> Toluene', 'DMF >> tert-butanol; Chlorobenzene', 'DMF; Glycerol >> DMF >> IPA', 'Butylamine', 'DMF; o-xylene', 'DMF; DMSO >> none >> none >> none', 'Ethanol; H2O >> IPA', 'IPA', 'DMF; DMSO >> Toluene', 'DMF >> Ethanol; IPA', 'acetonitrile; Ethanol; Methylamine', 'DMSO >> none', 'Hexane; Octane', 'H2O >> IPA', 'NMP >> IPA', 'Octane', 'DMF >> IPA >> Toluene', 'Dimethylacetamide; DMSO; NMP', 'H2O >> none >> IPA', 'DMSO; GBL', 'Dimetyletanamid', 'Dimethylacetamide; DMF', 'DMF; DMSO; HAc >> IPA', 'Diiodomethane; DMF; DMSO', 'DMF; DMSO | IPA', 'DMF; DMSO; HCl >> IPA', 'DMF >> Methanol >> Methanol >> Methanol', 'H2O >> H2O; HI >> IPA', 'DMSO; GBL >> none', 'Methylacetate', 'DMF >> IPA >> DMF >> IPA', 'Methanol; THF', 'IPA >> DMF >> IPA', 'Acetic acid; Ethanol; Water', 'DMF; DMSO; Formamide', 'GBL >> IPA', 'DMF; H2O', 'DMF; DMSO >> Ethanol', '2-methoxyethanol; DMSO; GBL', 'Butylamine; DMF; DMSO >> Butanol', 'Octane >> Pb(OAc)2 satured ethyl acetate solution >> Ethyl acetate >> Octane >> Pb(OAc)2 satured ethyl acetate solution >> Ethyl acetate >> Octane >> Pb(OAc)2 satured ethyl acetate solution >> Ethyl acetate', 'DMF; DMSO >> Methyl acetate', 'DMF; DMSO | IPA >> IPA', 'DMF; THF >> IPA', 'DMF >> Chlorobenzene; IPA', 'DMF; DMSO >> IPA >> DMSO; IPA', 'DMF; NNP', 'DMSO; GBL @ 4; 7 >> IPA', 'GBL; NMP', 'DMF >> IPA >> Methybenene', 'GBL; DMSO', 'Octane >> Ethyl acetate', 'IPA >> IPA', 'Unknown', 'Water >> Ethanol >> IPA', 'H2O; HI; Methylamine', 'DMF; DMA >> IPA', 'Hexane >> Octane', 'DMSO; GBL; IPA', 'IPA >> Ethanol', 'none | IPA', 'Octane >> Octane >> Octane >> Octane >> Octane >> Methyl acetate', 'Terpineol', 'Unknown >> IPA', 'DMF; Acetophenone', 'DMF; N-Methyl-2-pyrrolidone', 'GBL >> GBL', 'none >> IPA >> IPA', 'DMF; DMSO; H2O', 'Hexane', 'DMF | IPA', 'DMF >> DMF; IPA', 'Dimethylacetamide', 'DMF >> IPA; Toluene', 'DMSO; Hac >> IPA', 'DMF >> Ethanol >> Ethanol', 'none >> Ethanol', 'DMSO >> H2O; IPA', 'DMF >> PA', 'Ethyl acetate >> Ethyl acetate', 'DMF >> Ethanol', 'DMF >> DMF', 'DMSO; H2O >> IPA', 'none >> none >> none >> IPA', 'DMF >> n-butyl alchohol', 'DMF; HMPA >> IPA', 'DMF >> IPA; Propanol', 'DMF | DMF; IPA', 'none >> none >> none', 'DMF; DMSO @ 9; 1 >> IPA', 'DMF; IPA', 'Methylamine >> Methylamine', 'Methyl acetate >> Ethyl acetate', 'none >> Hydrophosphorous acid; IPA', 'DMF >> n-amyl alcohol', 'DMF >> IPA; TBP', 'DMSO; GBL; NMP', 'DMSO; BL', 'Aceton; DMF >> IPA', 'DMSO | Butanol; IPA', 'none >> IPA >> IPA >> IPA', 'DMF; DMSO >> Chlorobenzene', 'DMF >> Cyclohexanol; IPA', 'DMF; DMSO >> none', 'DMF >> DMF >> Ethanol', 'Water >> none', 'Octane >> Methyl acetate >> Methyl acetate', 'DMF; DMSO; Pyridin', 'DMF; DMSO; Methanol', 'DMF >> IPA >> none', 'DMF >> Octane >> Pb(OAc)2 satured ethyl acetate solution >> Ethyl acetate >> Octane >> Pb(OAc)2 satured ethyl acetate solution >> Ethyl acetate >> Octane >> Pb(OAc)2 satured ethyl acetate solution >> Ethyl acetate', 'DMF >> Octane >> Octane >> Octane >> Methyl acetate >> Methyl acetate', 'DMF; DMSO >> Chlorobenzene; DMF', 'DMF; DMSO >> DMF; DMSO >> DMSO >> DMSO | IPA', 'none >> Ethanol; Methoxyethanol', 'DMF; Tetraline', 'Dimethylacetamide >> IPA', 'Acetic acid; GBL; Propanol', 'DMF | none', 'DMF; DMSO; N-cyclohexyl-2pyrrolidone', 'DMF; DMSO >> IPA >> IPA', 'DMF; DMSO; Formarmid', 'DMF; NMP >> Ethanol', 'DMF; DMSO >> Ethanol; IPA', 'DMF; DMSO >> IPA >> none', 'DMF >> DMF >> IPA', 'IPA >> DMF >> none', 'Butanol; GBL', 'DMF >> none >> IPA', '1-Octane', 'DMF; DMSO; GBL', 'DMSO >> Octane >> Octane', 'DMF; DMSO >> DMSO; IPA', 'GBL; Polyethylene glycol >> H2O', 'DMF; DMSO; Thiourea', 'DMF >> IPA >> DMF', 'DMF; DMI >> Ethanol', 'Acetic acid; Ethanol; GBL', 'none >> none >> none >> none', 'DMF; n-butanol >> IPA', 'DMF; DMSO; GBL >> IPA', 'Dimethylacetamide; NMP', 'n-butylamine', 'Diiodooctane; DMF', 'acetonitrile; DMF; DMSO', 'DMSO; Formamide', 'DMF | IPA | H2O', 'DMF >> IPA >> IPA', 'DMF; DMSO; n-butyl amine >> n-butanol', 'DMF; HPA', 'GBL >> none', 'acetonitrile; Methylamine', 'acetonitrile; DMSO >> IPA', 'DMF; DMI >> IPA', 'Hexame >> Methyl acetate >> Ethyl acetate', 'DMF >> Octane >> Octane >> Methyl acetate >> Methyl acetate', 'none >> none >> IPA', 'DMF; DMSO >> IPA; H2O', 'DMF >> Acetonitrile, Methylamine', 'IPA >> DMF; DMSO', 'acetonitrile; Methylamine >> IPA', 'DMF; TBP', 'Ethanol; GBL', 'H2O >> none >> none', 'DMF >> TBA', 'DMF >> IPA >> Chlorobenzene; GBL', 'DMSO >> IPA >> IPA', 'DMSO | IPA', 'DMF; Tetramethylene sulfoxide', 'DMSO >> DMF; DMSO', 'DMF >> none', 'DMF >> IPA >> Chlorobenzene', 'DMF; DMSO >> DMF; DMSO >> IPA', 'Methanol; Water', 'DMF >> IPA', 'DMF; DMSO >> none >> IPA', 'DMF; Glycerol >> DMF; DMSO', 'DMF >> Methyl acetate', 'DMSO >> Hexane', 'DMSO', 'DMF >> DMF >> none', 'DMF; HI', 'DMF >> Methanol >> Toluene', 'DMF; DMSO >> IPA | IPA', 'DMF >> IPA >> Chlorobenzene; DMSO', 'DMF >> IPA >> Unknown', 'DMF; DMSO >> DMF; DMSO', 'DMF >> Terpineol', 'DMSO | butanol', 'DMF; DMSO >> DMSO; Methanol', 'DMF; DMSO; Hac >> IPA', 'H2O >> Hexane; IPA', 'acetonitrile; MA(MeOH)'])))

    solvents_mixing_ratios = Quantity(
        type=str,
        shape=[],
        description="""
    The mixing ratios of the solvents used in each deposition procedure for each layer in the stack
- When more than one reaction step, separate the solvent mixing ratios associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of deposition steps must line up with the previous columns.
- For pure solvents, state the mixing ratio as 1
- For non-solvent processes, state the mixing ratio as 1
- For unknown mixing ratios, state the mixing ratio as ‘nan’
- For solvent mixtures, i.e. A and B, state the mixing ratios by using semicolons, as in (VA; VB)
- The preferred metrics is the volume ratios. If that is not available, mass or mol ratios can be used instead, but it the analysis the mixing ratios will be assumed to be based on volumes.
Example
1
4; 1 >> 1
1 >> 5; 2; 0.3 >> 2; 1
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[])))

    solvents_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of all the solvents.
- When more than one reaction step, separate the solvent suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of reaction steps and solvents must line up with the previous columns.
- For non-liquid processes with no solvents, mark the supplier as ‘none’
- If the supplier for a solvent is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Sigma Aldrich
Sigma Aldrich; Fisher >> Acros
none >> Sigma Aldrich; Sigma Aldrich >> Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['J&K Scientific; J&K Scientific >> Unknown', 'ACORS Organic; ACORS Organic', 'Alfa Aesar; Alfa Aesar', 'Sigma Aldrich; Daejung', 'Acros Organics; Sigma Aldrich', 'Sigma Aldrich', 'Unknown >> Alfa Aesar; Sigma Aldrich', "Xi'an Polymer Light Technology >> Xi'an Polymer Light Technology >> Unknown", 'Alfa Aesar', 'ACORS Organic; ACORS Organic; Unknown', 'Unknown', 'Panreac', 'J&K Scientific >> Unknown', 'Millipore Sigma', 'Aladdin >> Sigma Aldrich', 'NanoPac; NanoPac', 'Aldrich', 'Sigma Aldrich; Alfa Aesar', 'Sigma Aldrich; Sigma Aldrich', "Xi'an Polymer Light Technology; Xi'an Polymer Light Technology", 'Wako Pure Chemical Industries >> Wako Pure Chemical Industries', 'Fisher Scientific; Fisher Scientific', 'sigma aldrich; sigma aldrich', 'Sigma Aldrich; Sigma Aldrich >> Unknown', 'Wako >> Wako; Wako', 'Sigma Aldrich >> Sigma Aldrich >> Sigma Aldrich', 'Alfa Aesar; Sigma Aldrich', 'Tokyo Chemical Industry, Japan', 'J&K', 'Wako; Wako', 'Merck >> Merck >> Merck', 'Wako >> Wako', 'Nacalai Tesque', 'Tokyo Chemical Industry >> Sigma Aldrich', 'Alfa Aesar >> Alfa Aesar', 'Kanto Chemical Tokyo', 'J&K; J&K', 'Aladdin; Aladdin', 'Sigma Aldrich; Unknown', 'Sigma; Aladdin', 'Merck >> Merck', 'Tianjin Guangfu Fine Chemical Research Institute; Unknown'])))

    solvents_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the solvents used.
- When more than one reaction step, separate the solvent purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For non-liquid processes with no solvents, state the purity as ‘none’
- If the purity for a solvent is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
Puris; Puris>> Tecnical
none >> Pro analysis; Pro analysis >> Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Pro analysis', '0.99', 'Unknown >> Puris; Puris', 'AR 99% GC', 'Pro analysis; Pro analysis', '99.9%; 99.8%', 'Unknown', 'Reagent Grade >> 99.5% >> 99.8%', '99%; 99%', '0.998', '99.7%; 99%', 'Puris', 'Puris; Puris', '99.8% >> Unkown', '99.99%; 99.5%', 'Puris; Unknown', '99.9%; 99.8% >> Unkown', '99.8%; 99.9%', '99.8', '99.7%; 99%; Unkown', '99%; 99,9%', '99.9%; 99.5%', 'Puris; Puris >> Unknown', '99.8% Anhydrous; 99.5% anhydrous'])))

    reaction_solutions_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    The non-solvent precursor chemicals used in each deposition procedure
- When more than one reaction step, separate the non-solvent chemicals associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several compounds, e.g. A and B, list the associated compounds in alphabetic order and separate them with semicolons, as in (A; B)
- Note that also dopants/additives should be included
- When several precursor solutions are made and mixed before the reaction step, it is the properties of the final mixture used in the reaction we here describe.
- The number and order of reaction steps must line up with the previous columns.
- For gas phase reactions, state the reaction gases as if they were in solution.
- For solid-state reactions, state the compounds as if they were in solution.
- For reaction steps involving only pure solvents, state this as ‘none’
- If the compounds for a deposition step is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
CsI; FAI; MAI; PbBr2; PbI2
PbI2 >> MAI
PbBr2; PbI2 >> FAI; MAI >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'nan >> PEAI', 'FAPbI3; MABr >> PEAI; PbI2', 'CsI; FAI; MABr; PbBr2; PbI2 >> Cs-oleate; PbBr2; PbBI2', 'PEAI; PbI2; MABr >> PEAI; PbI2; MABr', 'CsBr; FAI; PbI2 >> PEAI', 'CsI; MABr; PbBr2; FAI; PbI2', 'PbBr2, MgBr2 >> CsBr', 'MAI; PbI2; TPAI', 'MAI; PbI2 >> MAI', 'SnI2; FASnI', 'SnI2; FAI; SnF2; ethylenediamine; PbI2; MAI; MABr', 'SnI2; FAI; SnF2; ethylenediamine; PbI2; MAI', 'nan >> n-butyl amine', '1,8-octanedithiol; CsI; FAI; PbI2', 'MAI; PbI2; NiI2', 'F5PEAI, PEAI; PbI2', 'SnI2; FAI', 'MAI; PbI2; SnI2', 'MAI; BA; PbI2', 'PbCl2; PbI2 >> MAI', 'SnI2; FAI; SnF2; PbI2; MAI', 'MAI; PbSCN2; PbI2; FAI', 'CsI; CsBr; PbI2; PbBr2', 'PbCl2; PbI2 >> MAI; PMMA', 'CsBr; PbI2 >> CsPbI3-QDs >> Pb(OAc)2 >> nan >> CsPbI3-QDs >> Pb(OAc)2 >> nan >> CsPbI3-QDs >> Pb(OAc)2 >> nan', 'MAI; PbI2; FeI2', 'PbI2; PbBr2; FAI; MABr; g-C3N4', 'CsI; PbBr2 >> CsI', 'CsI; FAI; MABr; PbBr2; PbI2', 'nan >> Guanidinium iodide', 'nan >> Cs2CO3', 'CsI; FAI; PbI2; PbBr2', 'FAI; MAI; PbBr2; PbI2', 'PbI2 >> MAI; MAPbI3-QDs', 'PbI2; FAI; MACl; MABr; PbBr2', 'FAI; MABr; MACl; PbI2; PbBr2', 'CsAc; HPbBr3; PEABr', 'nan >> nan >> nan', 'PbI2 >> MAI >> nan', 'MACl; MAI; PbI2', 'NH4I; PbI2 >> MA', 'MAI; PbI2 >> MAI >> 4-DA', 'CsBr; PbBr2', 'MAI; PbCl2; PbI2', 'FAI; MABr; PbI2; PbBr2 >> CsI', 'MACl; SnCl2', 'MAI; PbI2; CoI2', 'MAI; PbI2 >> nan', 'nan >> CsNO3', 'PbI2; PbBr2; CsI', 'MAI; BDAI; PbI2', 'CsI; PbI2', 'MAI; MnI2', 'nan >> FAI', 'FAI; MABr; PbBr2; PbI2', 'CsAc; HPbBr3', 'PEAI, PbI2', 'MAI; PbCl2 >> IPFB', 'CsI; FAI; MABr; PbBr2; PbI2; RbI >> FABr', 'FAI; PbI2; ThMAI', 'PbI2 >> MAI', 'CsBr; FAI; PbI2', 'PbI2 >> MACl; MAI', 'nan >> MABr', 'CsBr >> PbBr | PQD', 'nan >> GuBr', 'HPbI3; CsI', 'PbI2; PbBr2; FAI; MABr; C3N5', 'nan >> nan >> nan >> Pb(NO3)2', 'PbI2 >> MAI; MACl', 'MAI; MTEACl; PbI2', 'CsI; SnI2', '1,8-diiodooctane; MAI; PbCl2', 'MAI; PbI2; PEAI', 'nan >> ETB; NaI', 'nan >> NH3I(CH2)4NH3I', 'AgI; Bi3', 'CsAc; HPbI3', 'MAI; PbI2', 'MAI; PBI2; SbBr3', 'nan >> FaBr', 'PbBr2, SrBr2 >> CsBr', 'F5PEAI; CsI; FAI; MABr; PbI2; PbBr2; Pb(SCN)2; PEAI', 'CsI; CsBr; PbI2', 'MAI; PbI2 >> PbI2', 'F5PEAI; CsI; FAI; MABr; PbI2; PbBr2; Pb(SCN)2', 'nan >> NH3I(CH2)8NH3I', 'PbI2 >> FAI; MACl', 'F5PEAI, PbI2', 'MAI; PbCl2; Pb(OAc)2.3H2O', 'PbI2 >> 5-AVAI; MAI', 'CsBr; FAI; PbI2 >> PMAI', 'PbCl2 >> MAI', 'MABr; PbBr2', 'CsI; PbBr2; PbI2; FAI >> FAI', 'CsI; FAI; HI; PbI2 >> FA(PF6)', 'CsI; FAI; MABr; PbI2; PbBr2', 'CsBr; CsI; PbI2', 'SnI2; FABr', 'nan >> Eu-pyP', 'PbCl2; PbI2 >> FAI; MAI', 'nan >> CsI', 'PbBr2, CaBr2 >> CsBr', 'MAI; PbCl2; PbI2; Phenol', 'MAI; PbCl2; PbI2; FAI', 'MAI; PbI2 >> MAI; PbI2', 'Cs2CO3; oleic acid; 1-octadecene; PbBr2; oleylamine', 'CsI; PbI2 >> FAI; MABr', 'MAI; PbCl2', 'nan >> Octylammonium iodide', 'MAI; NMA; PbI2', 'C3H5CsO2; HPbBr3; HPbI3', 'CsBr; PbI2 >> CsPbI3-QDs >> Pb(OAc)2 >> nan >> CsPbI3-QDs >> Pb(OAc)2 >> nan >> CsPbI3-QDs >> Pb(OAc)2 >> nan >> nan', 'CsBr; FAI; PbI2 >> PAI', 'nan >> Pb(NO3)2', 'nan >> Ethylammonium Iodide', 'nan >> GAI', 'CsI; FAI; MAI; PbBr2; PbI2', 'MAI; PbI2 >> MAPbI3-QDs', 'CsAc; HPbBr3; HPbI3; PEABr; PEAI', 'nan >> CsAc', 'MAI; PbAc', 'CsI; PbBr2; PbI2; FAI >> FABr', 'MAI; PbCl2; PbI2 >> MAPbI3-QDs', 'nan >> NH3I(CH2)2O(CH2)2O(CH2)2NH3I', 'PbI2 >> FAI; MABr; MACl', 'AuBr3; MABr', 'nan >> MAI', 'MAI; PbI2; MnI2', 'nan >> CsBr', 'CsI; FAI; PbBr2; PbI2', 'CsI; FAI; MABr; MACl; PbI2; PbBr2', 'Pb(NO3)2 >> MACl; MAI', 'PbI2 >> FAI; MAI; MACl', 'CsI; FAI; MABr; PbBr2; PbI2 >> PFPAI', 'CsBr; PbI2', 'FAI; MAI; PbI2', '5-AVAI; MAI; PbI2', 'MAI; PBI2', 'PbI2 >> 5-AVAI; MAI; MACl', 'MAI; 5-AVAI; PbI2', 'PbCl2 >> MAI >> nan', 'AgBr; BiBr3; CsBr', 'MABr; PbBr2; PbI2; FAI', 'CsI; FAI; GaAA3; PbI2', 'CsAc; HPbI3; PEAI', 'CsI; FAI; MAI; PbBr2; PbI2 >> 10%mol TBAI-doped PTzDPPBTz', 'CsPbI3-QDs >> Pb(OAc)2 >> nan >> CsPbI3-QDs >> Pb(OAc)2 >> nan >> CsPbI3-QDs >> Pb(OAc)2 >> nan', 'MAI; PbI2; PbCl2', 'CsAc; HPbBr3; HPbI3', 'PbI2 >> nan >> MAI', 'HPbI3 >> MA', 'C2H3CsO2; HPbBr3; HPbI3', 'CsBr >> PbBr', 'CsI; PbBr2; PbI2', '(BDA)I2; CsI; PbI2; PbBr2', 'CsI; FAI; MABr; PbI2', 'MAI; Pb(OAc)2', 'FAI; MAI; PbI2; Pb(SCN)2', 'HCOOCs; HPbBr3; HPbI3', 'FAI; SnI2', 'MAI; Pb (OAc)2.3H2O', 'PbI2; HI; MA; ethanol; diethyl ether', 'Pb(NO3)2 >> MAI', 'CsI; PbBr2', 'PbI2; FAI; MACl', 'FAI; MAI; PbI2; Pb(SCN)2 >> PMMA', 'KI; FAI; MABr; PbBr2; PbI2', 'SnI2; MASnI', 'PbI2; HI; MA; diethyl ether', 'FAPbI2Br >> MAPbI2Br >> CsPbI2Br >> RbI | BABr', 'CsI; FAI; MAI; PbBr2; PbI2 >> PTABr', 'FAI; PbI2', 'CsI; PbBr2; PbI2; FAI >> FAI; FABr', 'CsBr; HI; PbF2; PbI2', 'nan >> Imidazolium iodide', 'nan >> GABr', 'nan >> MACl', 'PbI2; MAI', 'CsI; FAI; HI; PbI2', 'PbBr2; MAI', 'PbBr2 >> CsBr', 'FAI; MAI; PbBr2; PbI2; Pb(SCN)2 >> PMMA', 'CsI; PbI2 >> FAI; MABr; MACl', 'PbCl2; MAI', 'CsI; FAI; MAI; PbI2', 'nan >> FABr', 'DOI; MABr; PbBr2', 'MAI: PbI2', 'MAI; PbI2 >> BEAI2', 'nan >> nan >> FAI', 'CsBr; HI; PbI2', 'MAPbI3-xClx', 'CsI; FAI; MABr; PbI2; PbBr2; Pb(SCN)2; PEAI; nan', 'CsPbI3 >> Pb(NO3)2 >> FAI', 'CsI; FAI; MAI; PbI2; RbI', 'CsI; FAI; PbI2', 'MAI; BAI; PbI2', 'nan >> EDBE', 'DOI; MAI; PbCl2', 'DMAI; PbI2', 'RbI; CsI; FAI; MABr; PbI2; PbBr2', 'FAI, MABr; PbI2; PbBr2', 'FAI; MACl; MABr; PbI2', 'PbI2; PbCl2; MAI', 'MAI; SnF2; SnI2 >> nan', 'nan >> Pb(NO3)2 >> nan', 'CsI; PbI2 >> MACl; MABr; FAI; MAI', 'PbI2 >> MAI >> TSA', 'nan >> ITIC'])))

    reaction_solutions_compounds_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of the non-solvent chemicals.
- When more than one reaction step, separate the non-solvent chemical suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of reaction steps and solvents must line up with the previous columns.
- For gas phase reactions, state the suppliers for the gases or the targets/evaporation sources that are evaporated/sputtered/etc.
- For solid state reactions, state the suppliers for the compounds in the same way.
- For reaction steps involving only pure solvents, state the supplier as ‘none’ (as that that is entered in a separate filed)
- For chemicals that are lab made, state that as “Lab made” or “Lab made (name of lab)”
- If the supplier for a compound is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Dysole; Sigma Aldrich; Dyenamo; Sigma Aldrich
Sigma Aldrich; Fisher | Acros
Lab made (EPFL) | Sigma Aldrich >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Aldrich; Aldrich; Junsei Chemical; Unknown; Unknown', 'Lab made; synthesiyed; Aladdin; Sigma Aldrich', 'Xian Polymer Light Technology; Alfa Aesar', 'Dyesol; Sigma; Alfa Aesar; Dyesol', 'Lab made; Unknown >> Lab made; Unknown', 'Showa Chemical; Sigma Aldrich', 'Sigma Aldrich; Dyesol; Sigma Aldrich; Sigma Aldrich; Sigma Aldrich; Dyesol; Sigma Aldrich', 'Sigma Aldrich >> Unknown >> Lab made', 'Lab-made; Sigma Aldrich', 'GreatCell Solar; Sigma-Aldrich', 'Greatcell Ltd.; Xi’An Polymer Light Technology Corp.; TCI', 'Youxuan Tech; Youxuan Tech', 'Wako; Tokyo Chemical Industry', 'Dyesol; Sigma; Sigma', 'Sigma Aldrich; Showa Chemical', 'Dyesol; Sigma Aldrich', 'TCI; GreatCell Solar Materials; Dyenamo; GreatCell Solar Materials; TCI', "Xi'an Polymer Light Technology; Alfa Aesar", 'Sigma Aldrich; Alfa-Aesar', 'Sigma Aldrich; Sigma Aldrich; TCI', 'Acros Organics; Acros Organics; Sigma Aldrich; Dyesol', 'Sigma Aldrich; Sigma Aldrich; Sigma Aldrich; Unknown; Unknown', 'Unknown >> Lab-made; Lab-made', 'Dyesol; Sigma; Dyesol; Dyesol', 'Dynamo; Sigma Aldrich', 'Sigma Aldrich; Dyesole; Dyesole; TCI; TCI', 'Alfa Aesar; Lab made', 'Sigma Aldrich >> Sigma Aldrich', 'Xi’an p-OLED >> Lab made', 'Unknown >> Lab made; Unknown', 'Sigma Aldrich; Sigma Aldrich >> Dyesol; Dyesol', 'Synthesiyed; Sigma Aldrich; Sigma Aldrich', 'TCI; Kanto Chemical Tokyo', 'Shanghai Mater. Win. New Materials Corporation; Sigma Aldrich; Sigma Aldrich; Shanghai Mater. Win. New Materials Corporation', 'Sigma Aldrich; Greatcell Solar', 'Sigma Aldrich >> Lab made', 'ACROS Organic; STAREK scientific Co. Ltd.', 'Dyesol; Dyesol; Alfa Aesar', 'Advanced Election Technology Co.. Ltd; GreatCell Solar; Advanced Election Technology Co.. Ltd. TCI.', 'Dyesol; Dyesol; TCI; Alfa Aesar; Sigma Aldrich >> Sigma Aldrich', 'Lab-made; Sigma Aldrich; Sigma Aldrich', 'GreatCell Solar; Sigma Aldrich', 'Sigma Aldrich >> Unknown; Unknown', 'Sigma Aldrich; Lab made; Lab made; Sigma Aldrich; Sigma Aldrich', 'Vizuchem; Vizuchem', 'Sigma Aldrich; Lab made; Sigma Aldrich; Sigma Aldrich >> Greatcell Solar', 'Alfa Aesar; Lumtec; Sigma Aldrich; Luminiscence; Sigma Aldrich', 'Lab made; Sigma Aldrich', 'Dyesol; Dyesol; TCI; Sigma Aldrichch', "Xi'an Polymer Light Technology Corporation.", 'Unknown; Unknown; Unknown', 'Shanghai MaterWin New Materials Co.. Ltd; Sigma Aldrich', 'Alfa Aesar; Greatcell Solar', 'Sigma Aldrich. 1-Material', 'Showa Chemical; Sigma Aldrich; Sigma Aldrich', 'Dynamo; Tokyo Chemical Industry', 'Sigma Aldrich; Daejung', 'Sigma Aldrich; Lab made; Sigma Aldrich; Sigma Aldrich', 'Alfa Aesar; Xi’an Polymer Light Technology; Xi’an Polymer Light Technology', 'Unknown', 'YOUXUAN Technology Co. Ltd.; Sigma Aldrich Co. Ltd.', 'Sigma Aldrich; Lab made; Sigma Aldrich', 'Dynamo; TCI', 'Alfa Aesar; Dyesol', 'Aladdin; Lab made; synthesiyed; Aladdin; Sigma Aldrich', 'Dyesol; Dyesol; TCI', 'Sigma Aldrich; Sigma Aldrich', "Acros Organics; Xi'an Polymer Light Tech. Corp.", 'Lab made; Alfa Aesar', 'Xi’an Polymer Light Technology Corp; Alfa Aesar', 'Alfa Aesar; Alfa Aesar; Alfa Aesar; Alfa Aesar; Alfa Aesar', 'Ossila', 'Sigma Aldrich; Macklin', 'Sigma Aldrich; Dyesol; Sigma Aldrich; Sigma Aldrich; Dyesol', 'Unknown >> PEAI', 'Acros Organics; Acros Organics; Sigma Aldrich; Dyesol; Unknown', 'Unknown >> Lab made', 'Alfa Aesar; Alfa Aesar', 'TCI; Sigma Aldrichch', 'Xian Polymer Light Technology; Xian Polymer Light Technology', 'Dysol; Unknown; Unknown', 'Xi’an p-OLED; Xi’an p-OLED', 'Sigma Aldrich; Dyesol; Sigma Aldrich; Sigma Aldrich; Sigma Aldrich; Dyesol', 'Lab made; Lab made; Sigma Aldrich', 'Dyesol; Alfa Aesar', 'PbI2; MAI', "Xi'an Polymer Light Technology; Xi'an Polymer Light Technology", 'Tokio Chemical Industry; Sigma Aldrich', 'Dyesol; Dyesol; TCI; Sigma Aldrich >> Sigma Aldrich', 'Sigma Aldrich >> Dyesol', 'TCI; TCI', 'Alfa Aesar; Sigma Aldrich', 'Sigma Aldrich; Lab made', "Xi'an Polymer Light Technology Corp.; Xi'an Polymer Light Technology Corp.; Sigma Aldrich; Sigma Aldrich", 'Alfa Aesar; Xian Polymer Light Technology', 'GreatCell Solar; GreatCell Solar; Sigma Aldrich; Sigma Aldrich'])))

    reaction_solutions_compounds_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the chemicals used.
- When more than one reaction step, separate the compound purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, i.e. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of reaction steps and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For reaction steps involving only pure solvents, state this as ‘none’ (as that is stated in another field)
- If the purity for a compound is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
99.999; Puris| Tecnical
Unknown >> Pro analysis; Pro analysis | none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['99,985 >> Unknown', 'Puris; Pro analysis; Puris; Unknown', 'Unknown; 99,9%', '99.5; 99', 'Unknown; Unknown; Puris; Puris', '99,9%; Unknown; Unknown; 99,9%; 99,999%', 'Unknown; 98%', '99.9%; 99.9%', 'Unknown; 99%', 'Puris; technical; technical; Unknown; Unknown', 'Unknown >> Unknown; Unknown', '99.999% >> Unknown', '99.999%; 99.999%', '99.99%; Unknown; 99.9985%', 'Unknown; Unknown; 99,9%; 99,999%', 'Unknown; 99,999%', 'Unknown', '99.99%; Unknown; 99.99%; 99.9985%; Unknown', 'Unknown; 99.9985%', '98%; Unknown', '99.99%; Unknown; 99.99%; 99.9985%', '99.5; 99.99', 'Puris; Pro analysis; Puris; Unknown; Unknown', 'Puris; Puris; Puris', 'Unknown >> 99,999% >> Unknown', '99%; Unknown >> Unkown', 'Puris; Puris; Unknown; Puris; Unknown', 'Unknown; Unknown', 'Puris; Puris', 'Puris; Unknown', 'Unknown; Puris; Puris; Unkown', 'Unknown >> MACl', 'Puris; technical; technical; puris; technical', 'Unknown; Puris', '99 %; 99.5%', 'Unknown; Unknown; 99,9%', 'Unknown; Unknown; Unknown', 'Puris >> Puris', 'Puris; Unknown; Unknown; Unknown; Unknown', 'Unknown; 99.999%', '99,5%; 99,99%', 'Unknown >> ODAI', 'Puris; Puris; Unknown; Puris; Puris; Unknown', 'Unknown >> CsAc', '99,999%; 99,999% >> 99,8%; 99,8%', '99,999% >> Unknown; Unknown', 'Puris; Puris; Unknown; Puris; Puris; Unknown; Puris', 'Unknown; 99%; 99,99%', '99% >> Unknown', '95%; 99.9%'])))

    reaction_solutions_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the non-solvent precursor chemicals.
- When more than one reaction step, separate the concentrations associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of deposition steps and chemicals must line up with the previous columns.
- If a solution contains several dissolved compounds, e.g. A and B, list the associated concentrations and separate them with semicolons, as in (A; B)
- The order of the chemicals must line up to the chemicals in the previous column.
- The order of the compounds must be the same as in the previous filed.
- For reaction steps involving only pure solvents, state this as ‘none’
- For gas phase reactions, state the concentration as ‘none’
- For solid-state reactions, state the concentration as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used. When possible, use one of the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml, wt%, mol%, vol%, ppt, ppm, ppb
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
0.063 M; 1.25 M; 1.25 M; 1.14 M; 1.14 M
1.25 M; 1.25 M >> 1.14 M; 1.14 M; 10 mg/ml
1 M; 1 M >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1 M >> 40 mg/ml', '0.4 M; 0.6 M; 0.8 M', '0.18 M; 1.02 M; 1.2 M', '0.75; 0.788; 0.713', '600 mg/ml >> 1 mg/ml', '1 M >> 0.07 M | nan', '50 wt%', '0.024 M; 1.176 M >> 8 mg/ml', '461 mg/ml; 57 wt%; 33 wt%; nan; nan', '0.063 M; 1.14 M; 1.14 M; 1.25 M; 1.25 M', '0.2 M; 0.06', '1 vol%; 0.05 M; 0.95 M; 1 M', '46.8 mg/ml; 226.9 mg/ml; 0.05 vol%; 608.5 mg/ml >> 1 mg/ml', '0.5 M; 0.5 M', '1 M; 30 mg/ml', '0.442 molal; 1.07 molal; 0.173 molal; 1.36 molal; 13.6 mg/ml >> 1 mg/ml', '1.125 M; 0.75 M', '548.6 mg/ml >> 50 mg/ml', '462 mg/ml >> 35 mg/ml', 'nan; 1 M; 0.2 M; 0.2 M; 1.1 M >> 1000 mM', '1.26 M; 0.14 M; 1.4 M', 'nan; 1 M; 0.2 M; 0.2 M; 1.1 M >> 200 mM', '600 mg/ml >> 10 mg/ml', '1 M >> 0.0031 M; 0.0598 M', '1 M; 1.05 M', '0.5 M >> 20 mg/ml', '573.3 mg/ml; 197.7 mg/ml', '0.1426 M; 1.56 M; 1.74 M >> 4 mM', '1 M >> 10 mg/ml >> 5 mM', '2 wt%; 163.33 mg/ml; 535.2 mg/ml', '0.75; 0.75; 0.75', '0.1426 M; 1.56 M; 1.74 M >> 10 mM', '1 M >> 7 mg/ml', '0.75 M; 0.50 M; 1 M', '1 M >> 8 mg/ml', '0.5 M >> 38 mg/ml', '553 mg/ml; 190 mg/ml', '0.5 M; 0.5 M; 0.6 M; 0.3 M', '1 M >> 10 mg/ml >> 10 mM', '1 M; 1.06 M', 'nan >> 2 mg/ml', '1.25 M; 1.25 M >> nan', '1.5 M; nan', '462 mg/ml >> nan >> 10 mg/ml', '1.4 M; 1.4 M', '1.08 M; 1.08 M; 0.12 M', '1.2 M >> 1.6 mol%', 'nan >> 5 mg/ml', '500 mg/ml >> 10 mg/ml >> nan', '1 M >> 0.07 M', 'nan >> 3 mg/ml', '46.8 mg/ml; 226.9 mg/ml; 0.05 vol%; 608.5 mg/ml >> 7 mg/ml', '0.3 M; 0.06', '1.5 M >> 62.3 M', '3 vol%; 0.05 M; 0.95 M; 1 M', '1 M; 0.2 M; 1 M; 1.1 M', '1.3 M; 1.3 M; 1.3 M; 1.3 M | 1 mg/ml', '1.5 M; 1.5 M', '1.5 M >> 23.5 mM; 70.8 mM', '0.1426 M; 1.56 M; 1.74 M >> 2 mM', '1.35 M; 1.35 M', '1.25 M; 1.3 M', '440 mg/ml; 180 mg/ml', '1.1 M; 0.2 M; 1.15 M; 0.2 M', '50 mg/ml >> nan >> nan >> 50 mg/ml >> nan >> nan >> 50 mg/ml >> nan >> nan', '1.2 M', '400 mg/ml >> 10 mg/ml', '1.0 M; 1.1 M', '462 mg/ml >> 10 mg/ml >> nan', '52.6 mg/ml; 197.4 mg/ml; 622.4 mg/ml', '1 M >> 10 mg/ml >> 2.5 mM', '0.85 M; 0.15 M; 0.15 M; 0.85 M', '0.8 M; 0.3 vol%; 0.12 M; 0.68 M', '0.05 M; 1.1 M; 0.2 M; 0.2 M; 1.25 M', '0.5 M; 0.5 M >> 20 mg/ml', '0.90 M; 0.90 M; 0.10 M', '1.3 M; 0.14 M; 1.26 M; 50 mg/ml', 'nan; 1 M; 0.2 M; 0.2 M; 1.1 M', '0.5 M; 1 M; 1 M', '1.4 M >> 10 mg/ml; nan', '1.8 M; 1.8 M >> nan', '1 M; 0.07 M', '2; 1', '450 mg/ml >> 5 mg/ml; 50 mg/ml', '1 M >> nan', '1 :3molarratio', '2 wt%; 2.43 M; 0.81 M', '0.8 M; 0.3 vol%; 0.28 M; 0.52 M', '0.11 M; 0.91 M; 0.18 M; 0.186 M; 1.014 M', '2.44 M; 2.44 M; 0.9 M', '1.3 M; 1.3 M', '1 M; 57 wt%; 33 wt%; nan; nan', '13.7 mg/ml; 197.5 mg/ml; 573 mg/ml', '0.98 M; 1 M', '70 mg/ml', '1.02 M; 0.18 M; 0.18 M; 1.02 M', '1 mM; 1 mM >> 30 mg/ml', 'nan; 10 mg/ml', '1 M >> 50 mg/ml', '0.95 M; 0.05 M; 0.05 M; 0.95 M', '461 mg/ml; 159 mg/ml', '691.5 mg/ml >> nan; nan', '0.5 M >> 36 mg/ml', '0.4 M; 0.8 M; 0.8 M', '1 M; 50 mM', '0.75 M; 0.75 M', '367 mg/ml', '1.125; 0.75; 0.75', '0.2 M; 0.22 M; 1.1 M; 1 M', '0.5 M >> 50 mg/ml', '45 wt%', '636.4 mg/ml; 90 mg/ml; 89.79 wt%; 8.97 wt%', '33.8 mg/ml; 599 mg/ml >> 60 mg/ml; 6 mg/ml; 6 mg/ml', '1.2 M; 0.28 M', '0.18 M; 1.02 M; 1.2 M; 0.1 M', '1.2 mM', '0.5 M >> 34 mg/ml', '1 M; 0.2 M; 1.1 M; 0.2 M', '66 mg/ml; 187 mg/ml; 12 mg/ml; 80 mg/ml; 568 mg/ml; nan', '0.3 M; 1.2 M >> 40 mg/ml', '0.5 M >> 32 mg/ml', '1.5; 0.75; 0.75', '1.3 M; 0.14 M; 1.26 M', '2.7 M; 0.9 M', '0 M; 0.8 M; 0.8 M', '0.5 M >> 30 mg/ml', '227.14 mg/ml; 80 mg/ml; 527.14 mg/ml', '510 mg/ml >> 8.5 mg/ml >> nan', '0.15 M; 0.85 M; 1 M', '1 M; 1 M; 1 M', 'nan >> 0.5 mg/ml', '0.5 M; 0.5 M; 1 M; 0.2 M', '1.8 M; 0.45 M; 0.45 M', '1 M >> 10 mg/ml', '2.4 M; 0.8 M', '0.6 M; 0.6 M >> 50 mg/ml >> nan >> nan >> 50 mg/ml >> nan >> nan >> 50 mg/ml >> nan >> nan', '1 mol/L; 1 mol/L', '0.7 M; 0.7 M', '50.87 mg/ml; 147.5 mg/ml', '1.42 M; 1.42 M', '0.18 M; 1.32 M; 1.5 M', '0.4 M', '1.25 M; 1 M', '0.05 M; 1.0 M; 0.2 M; 0.2 M; 1.1 M', '450 mg/ml >> 50 mg/ml', '1 M >> 0.0031 M; 0.0126 M; 0.0472 M', '0.06 M; 0.96 M; 0.18 M; 0.186 M; 1.014 M', '1.02 M; 0.18 M; 0.186 M; 1.014 M', '0.375; 0.75; 0.75', '0.5 M >> 40 mg/ml', '159 mg/ml; 461 mg/ml >> 461 mg/ml', '1.467; 1.6', '0.6 M; 0.6 M', '0.05 M; 1 M; 0.2 M; 0.2 M; 1.1 M', '1.59 mol/kg; 1.59 mol/kg', '0.11 M; 1.07 M; 0.19 M; 0.19 M; 1.23 M', '197.6 mg/ml; 787 mg/ml', '0.15 M; 0.75 M; 0.1 M; 1 M', '1.467; 1.6; 2 mol%', '206.2 mg/ml; 597.8 mg/ml', '1.5 M; 0.75 M', '1.53 M; 1.4 M; 0.5 M; 0.0122 M; 0.0122 M', '5 wt%; 1.4 M >> 60 mg/ml; 6 mg/ml', '2.25 M; 0.75 M', 'nan >> 0.16 ml/ml; 9.04 mg/ml; 23.04 mg/ml', '0.8 M; 0.3 vol%; 0.2 M; 0.6 M', '1 M; 1 M', '1.6 M', '9 wt%; 26 wt%', '1.2 M; 1.2 M', '397 mg/ml >> 9 mg/ml', '1.6 mol%; 1.2 M; 1.2 M >> 1.6 mol%; 1.2 M; 1.2 M', '1 wt%; 2.43 M; 0.81 M', '75 mg/ml >> 1 mg/ml >> 1 mg/ml', '0.9 M; 0.3 M; 1.0 M; 0.3 M', '0.14 M; 1.4 M >> 70 mg/ml', '1.3 M; 0.14 M; 1.26 M; 75 mg/ml', '0.5 M; 0.5 M >> 20 mg/ml; 0.005 mg/ml', '6.4 wt%; 33.6 wt%', '1.05 M; nan', '0.9 M; 0.9 M', '636.4 mg/ml; 90 mg/ml; 89.79 wt%; 8.97 wt%; 4 wt%', '1 M; 0.2 M; 0.22 M; 1.1 M', '1.1 M; 10 mg/ml', '1.4 M; 1.45 M', '0.17 M; 0.83 M; 0.5 M; 0.5 M', '0.172 mg/ml; 0.022 mg/ml; 0.507 mg/ml; 0.08 mg/ml >> 1.5 M', '0.072 M; 1.11 M; 0.21 M; 0.21 M; 1.2 M', '0.5 M; 1.2 M; 1 M; 0.2 M', '330 mg/ml >> 10 mg/ml', '1.4 M', '1.3 M; 1 M', '0.05 M; 0.95 M; 1 M', '3 M; 1 M', '450 mg/ml >> 50 mg/ml; 10 mg/ml', '2.1 M; 0.7 M', '596 mg/ml; 200 mg/ml', 'nan >> nan', '159 mg/ml; 461 mg/ml', '0.4 M; 0.4 M; 0.8 M', '1.3 M; 1.35 M', '0.6 M; 0.6 M >> 50 mg/ml >> nan >> nan >> 50 mg/ml >> nan >> nan >> 50 mg/ml >> nan >> nan >> nan', '0.2 M; 0.8 M >> 15 mg/ml', '2.55 M; 0.85 M', '1.3 M; 0.14 M; 1.26 M; 5 mg/ml', '26 mg/ml; 172 mg/ml; 22.4 mg/ml; 16.7 mg/m; 507 mg/ml; 73.4 mg/ml', '0.1426 M; 1.56 M; 1.74 M >> 6 mM', '0.88 M; 1.1 M; 0.44 M', '460 mg/ml >> 10 mg/ml', '199 mg/ml; 605.5 mg/ml', '1 M; 10 mg/ml', '20 mg/ml; 172 mg/ml; 22.4 mg/ml; 507.2 mg/ml; 73.4 mg/ml', '1.1 M; 1.1 M', '0.1125 M; 1.1375 M; 1.25 M', '46.8 mg/ml; 226.9 mg/ml; 0.05 vol%; 608.5 mg/ml >> 28 mg/ml', '1 M; 0.5 M; 0.5 M', '163.33 mg/ml; 535.2 mg/ml', '190 mg/ml; 0.0176 mg/ml; 530 mg/ml', '1.4 M >> 10 mg/ml', '0.54 M; 0.54 M; 0.06 M', '0.2 M; 0.2 M; 1.1 M; 1 M', '470 mg/ml >> 50 mg/ml; 5 mg/ml', '0.196 mg/mLPbI2; 1.5 M', '28 wt%', '20 mg/ml; 1.6 M; 1.6 M; 20 mg/ml', '1.0 M', '0.5 M; 0.75 M; 1 M', '1.4 M; 0.1 M; 0.01 M; 1.4 M; 0.1 M', '40 wt%', '0.214 mg/mLPbI2; 1.5 M', '1 wt%; 163.33 mg/ml; 535.2 mg/ml', '1.875 M; 0.75 M', '1.3 M | 1.3 M', '4 M; 1 M', '5 vol%; 0.05 M; 0.95 M; 1 M', '1.3 M >> 60 mg/ml; 6 mg/ml; 6 mg/ml', '1.2 M; 1.2 M >> nan', '238.5 mg/ml; 726 mg/ml', '1.8 M >> nan', '0.5 M; 0.5 M; 0.4 M', '0.21 M; 0.81 M; 0.18 M; 0.186 M; 1.014 M', '687 mg/ml; 237 mg/ml', 'nan; nan', '1.2 M >> 8 mg/ml', '691.5 mg/ml >> 70 mg/ml', 'nan >> 7 mg/ml', '1.25 M; 1.25 M', '60 mg/ml >> 1.3 M', '0.348 mg/ml; 0.922 mg/ml', '0.075 M; 1 M; 0.2 M; 1 M; 1.1 M', '254.4 mg/ml; 737.6 mg/ml', '1 :1mol%', '0.1 M; 0.06', '20 wt%', '1.24 M', '1; 0.2; 1; 1.1', '0.1426 M; 1.56 M; 1.74 M', '0.8 M; 0.3 vol%; 0.18 M; 0.62 M', '1 M >> 0.0126 M; 0.0503 M', '1.4 M >> 70 mg/ml', '0.9 M; 0.1 M; 0.1 M; 0.9 M', '1.35 M; 0.14 M; 1.26 M', '0.70 M; 0.17 M; 0.10 M; 1.30 M', '0.18 M; 1.02 M; 1.5 mg/ml; 1.2 M', '1 M; 0.33 M; 0.66 M', '1 M; 1 M >> nan', '46.8 mg/ml; 226.9 mg/ml; 0.05 vol%; 608.5 mg/ml', '21.25 wt%; 21.25 wt%', '3.75 M; 1.25 M', '100 mg/ml; 300 mg/ml', '0.375 M; 0.75 M', '19.4 mg/ml; 172 mg/ml; 22.4 mg/ml; 73.4 mg/ml; 507 mg/ml; nan', '2.2 M; 2 M', '112 mg/ml; 38 mg/ml', 'nan; 1 M; 0.2 M; 0.2 M; 1.1 M >> 100 mM', '0.063 M; 1.25 M; 1.25 M; 1.14 M; 1.14 M', '1.2 M; 0.3 M; 0.3 M', '400 mg/ml >> nan', '7 mg/ml; 1.2 M; 0.2 M', '1.3 M; 0.14 M; 1.26 M; 20 mg/ml', '0.12 M; 1.08 M >> 8 mg/ml', 'nan; 1 M; 0.2 M; 0.2 M; 1.1 M >> 50 mM', '0.24 M; 1.36 M >> 30 mg/ml; 70 mg/ml', '1.467; 1.6; 4 mol%', '0.06 M; 1.14 M >> 8 mg/ml', '1.2 M; 8 mg/ml', '190.8 mg/ml; 553 mg/ml', '197.6 mg/ml; 787 mg/ml; 19.1 mg/ml', '46.8 mg/ml; 226.9 mg/ml; 0.05 vol%; 608.5 mg/ml >> 14 mg/ml', '0.3 M; 0.3 M', '32 wt%', '16.89 mg/ml; 600 mg/ml >> 1 mg/ml; 10.75 mg/ml', '198 mg/ml; 577 mg/ml', '0.3 M; 0.3 M; 0.6 M', '1.10 M; 0.20 M; 0.20 M; 1.15 M', '0.8 M; 0.2 M; 0.2 M; 0.8 M', '1.8 M; 0.45 M; 0.45 M >> nan', '0.72 M; 0.72 M; 0.08 M', '0.06 M; 0.95 M; 0.19 M; 0.2 M; 1 M', '4.07 mg/ml; 9.66 vol%; 48.85 vol%; 6.9 mg/ml; 10 vol%', '0.07 M; 1.4 M >> 70 mg/ml', '1 vol%; 1.1 M; 1.1 M', '0.22 M; 0.66 M', 'nan; 1 M; 0.2 M; 0.22 M; 1.1 M', '462 mg/ml >> 20 mg/ml', '1.2 M >> 45 mg/ml', 'nan; 1 M; 0.2 M; 0.2 M; 1.1 M >> 500 mM', '460 mg/ml >> 0.15 M', '1.38 M; 1.38 M', '0.8 M; 0.3 vol%; 0.8 M', '460 mg/ml >> nan; nan', '0.039 M; 0.091 M; 1.1 M; 0.2 M; 1.2 M; 0.2 M', '39.5 mg/ml; 115.7 mg/ml', '0.442 molal; 1.07 molal; 1.48 molal; 13.6 mg/ml', '6.4 wt%; 33.6 wt% >> nan', '0.1426 M; 1.56 M; 1.74 M >> 8 mM', '0.3 M; 1.2 M >> 40 mg/ml >> 2 mg/ml', '0.442 molal; 1.07 molal; 1.48 molal; 13.6 mg/ml >> 1 mg/ml', '1 M.0.03M; 0.07 M', 'nan >> 1 mg/ml', '2.43 M; 0.81 M', '0.8 M; 10 mg/ml', '1 M; 0.952 M; 0.048 M', '1 M >> 30 mg/ml', '460 mg/ml >> 8.3 mg/ml >> nan', '2 M; 1 M', '0.06 M; 1 M; 0.2 M; 0.2 M; 1.1 M', '0.8 M; 0.8 M'])))

    reaction_solutions_volumes = Quantity(
        type=str,
        shape=[],
        description="""
    The volume of the reaction solutions used. used in each deposition procedure
- When more than one reaction step, separate the volumes associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The volumes refer the volumes used, not the volume of the stock solutions. Thus if 0.15 ml of a solution is spin-coated, the volume is 0.15 ml
- For reaction steps without solvents, state the volume as ‘nan’
- When volumes are unknown, state that as ‘nan’
Example
0.04
nan >> 0.1
nan >> 10
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['0.02', '0.045', '0.07; Unknown', 'Unknown >> 0.05', 'Unknown >> 0.2', '0.06 >> Unknown', '0.75; 0.15; 0.1; 0.05 | 0.09', '1.25', '5.0', '0.05', 'Unknown >> 0.04', 'Unknown', '0.065', '0.08', '0.04 >> 0.04', '0.075', '0.035', 'Unknown >> 0.0157 >> 0.5', 'Unknown >> Unknown >> Unknown', '0.05 >> 0.1', 'Unknown >> Unknown >> 50.0', '0.25', 'Unknown >> Unknown', '0.0025; 0.0025', '0.06', '0.01', '0.08 >> 0.1', '1.0', '0.003', 'Unknown >> 8.0', '0.1', 'Unknown >> 0.00942 >> 0.5', '0.08 >> Unknown >> Unknown', '0.03 >> 0.2', '0.03'])))

    reaction_solutions_age = Quantity(
        type=str,
        shape=[],
        description="""
    The age of the solutions used in the deposition
- When more than one reaction step, separate the age of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- As a general guideline, the age refers to the time from the preparation of the final precursor mixture to the reaction procedure.
- When the age of a solution is not known, state that as ‘nan’
- For reaction steps where no solvents are involved, state this as ‘nan’
- For solutions that is stored a long time, an order of magnitude estimate is adequate.
Example
0.5
nan >> 10
10000 >> nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['0.0 | 0.0', '24.0', '3.0', '1.0 >> 2.0', '60.0; 25.0', '600.0', '12.0', 'Unknown', '0.08', 'Unknown >> 60.0', '2.0 >> Unknown', '8.0', '70.0', '6.0', '0.167', 'Unknown >> Unknown', '2.0', 'Unknown >> 30.0', '12.0 >> Unknown >> Unknown', '0.5', '12.0 >> 0.033', '1.0', '48.0', 'Unknown >> 45.0', '8.0; 12.0'])))

    reaction_solutions_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the reaction solutions.
- If there is more than one reaction step involved, list the solution temperatures and separate the data for each step by a double forward angel bracket (‘ >> ‘)
- If a reaction solution undergoes a temperature program, list the temperatures (e.g. start, end, and other important points) and separate them with a semicolon (e.g. heated to 80°C and cooled to room temperature before used would be80; 25)
- When the temperature of a solution is not known, state that as ‘nan’
- For reaction steps where no solvents are involved, state the temperature of the gas or the solid if that make sense. Otherwise mark this with ‘nan’
- Assume that an undetermined room temperature is 25
Example
25
nan >> 50
80; 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['80 | Unknown', '25', '310 >> 150 >> 25', '25 >> 13', '0 | Unknown', 'Unknown; 50', '40 | Unknown', '70 >> 25 >> 25', '70 >> 80', '50.0', 'Unknown >> 150', '70', '125.0', 'Unknown', '60; Unknown', '25; 25', '100', '130', '75.0', '60; 25', '100.0', 'Unknown >> 160', '70 | Unknown', '12', '75', '60', '65', '70 >> 25 >> 70', 'Unknown >> 120', '25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25', '90', '25 >> 25', '80 >> 25', 'Unknown >> 140', 'Unknown >> 25', '150.0', '70; 25', '100 >> 25', '25.0', '80; 25', '75 >> 25', '70 >> 25', '50', '70 >> Unknown', '55; 100 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25', '55; 100', 'Unknown; 150', '60 >> 60', '60 >> 70'])))

    substrate_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the substrate on which the perovskite is deposited.
- When more than one reaction step, separate the temperatures of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a reaction solution undergoes a temperature program, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons, e.g. 25; 100
- When the temperature of a solution is unknown, state that as ‘nan’
- For reaction steps where no solvents are involved, state the temperature of the gas or the solid if that make sense. Otherwise state this as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume an undetermined room temperature to be 25
Example
25
70 >> 25
nan >> 40
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['80 | Unknown', '', '25', '0 | Unknown', '25 >> 13', '85 >> 25', '40 | Unknown', '0', '25 >> 50', '85', '70', '25 >> 110', '-10', '21', 'Unknown', '100', '25 >> Unknown', '120', 'Unknown >> 20; 110', '40', '150', '25 >> 130', '140.0', '70 | Unknown', '25 >> 75', '100 >> Unknown', '70.0', '60', 'Unknown >> 70', '190.0', '90', 'Unknown >> Unknown', '25 >> 25', '80 >> 25', '25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25', '25 >> 90', 'Unknown; 25 >> 25; 150 >> 25', '175', 'Unknown >> 25', '25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25', '100 >> 25', '70 >> 25', '90 | 25', '25 >> 150', '50', 'Unknown >> 25 >> 25', '70 >> Unknown', '70 >> 150', '90 >> 25', '10'])))

    quenching_induced_crystallisation = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE is measures were taken to discontinuously accelerate the crystallisation process without significantly changing the temperature. i.e. an antisolvent treatment or an analogue process was used.
- The most common case is the antisolvent treatment where a volume of a solvent in which the perovskite is not soluble is poured on the substrate during spin coating.
- The same effect can also be achieved by blowing a gas on the sample
- If the sample quickly after spin coating is subjected to a vacuum, this also counts as quenched induced crystallisation
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    quenching_media = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in the antisolvent treatment
- If the antisolvent is a mixture of different solvents, e.g. A and B, list the solvents in alphabetic order and separate them with semicolonsas in (A; B)
- If gas quenching was used, state the gas used
- If the sample quickly after spin coating was subjected to a vacuum, state this as ‘Vacuum’
- If an antisolvent was used but it is unknown which one, stat this as “Antisolvent”
- If no antisolvent was used, leave this field blank
Example
Chlorobenzene
Toluene
Diethyl ether
Ethyl acetate
N2
Vacuum
Anisole
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Tetraethyl orthosilicate', '2-butylalcohol', 'Chlorobenzene; TBP', 'Chlorobenzene; DMSO', 'Butyl acetate', 'Air plasma', 'Tetrachloroethane', '2-Butanol; Ethyl ether', 'Ethyl ether; Hexane', 'Ethyl acetate; IPA', 'Bromobenzene', 'Tetrafluorotoluene', 'Thermal radiation', 'p-xylene', 'MeOAc', 'Chlorobenzene; H2O', 'Chlorobenzene', 'Methyl-phenoxide', 'Chlorobenzene; Ethyl acetate', 'Chloroform; Toluene', 'Ethanol', 'Pumping solvent', 'Acetone', 'Ar', 'Ethyl ether', 'Chloroform', 'N2', 'Gas', 'Anisole; Toluene', 'o-xylene', 'Diethyl ether; Chlorobenzene', 'NIR', 'Chlorobenzene; 2-Butanol', 'He', 'Di-n-butyl ehter', 'n-Butyl alcohol', 'Diethyl ether', 'Rotating magnetic field', 'Dry air', 'Diphenyl ether', 'liquid N2; N2', 'Chlorobenzene; Ether', 'Chlorobenzene; N2', 'Chlorobenzene; Toluene', 'Chloroform; Hexane', 'Trifluorotoluene', 'Dichlorobenzene', '2-Butanol; Chlorobenzene; Ethyl ether', 'acetonitrile; Chlorobenzene', 'Unknown', 'Ethyl benzene', 'Trimethylbenzene', 'Diclorobenzene; Toluene', 'Ethyl acetate; Petroleum ether', 'Petroleum ether', 'N2 >> Chlorobenzene', 'TEOS', 'Diethyl ether; Methanol', '2-Butanol', 'Air', 'Diisopropyl ether', 'Vacuum', '2-Butanol; Chlorobenzene', 'Hot air', 'Diethyl ether; Toluene', 'Ether', 'Hexane', 'Ethanol; Toluene', 'Ethyl acetate; Toluene', 'Dichloromethane', 'Di-n-propyl', 'Chlorobenzene; Diethyl ether', 'Anisole', 'Chlorobenzene; Ethanol', 'Anisole; Chlorobenzene', 'Chlorobenzene; Diiodooctane', 'n-BA', 'Flash infrared annealling', 'Antisolvent', 'Methanol', 'Chlorobenzene; IPA', 'IR', 'Ethyl acetate', 'Anisole >> N2', 'Triochloromethane', 'Methyl acetate', 'Chlorobenzene; Chloroform', 'Propyl acetate', 'Toluene', 'Isopropyl acetate; Toluene', 'IPA', 'Hot substrate', 'Iodobenzene', 'Chlorobenzene; Acetic acid'])))

    quenching_media_mixing_ratios = Quantity(
        type=str,
        shape=[],
        description="""
    The mixing ratios of the antisolvent
- The order of the solvent must line up with the previous column
- For solvent mixtures, i.e. A and B, state the mixing ratios by using semicolons, as in (VA; VB)
- The preferred metrics is the volume ratios. If that is not available, mass or mol ratios can be used instead, but it the analysis the mixing ratios will be assumed to be based on volumes.
- For pure solvents, give the mixing ratio as 1
- For non-solvent processes, give the mixing ratio as 1
Example
1
4; 1
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1; 0.08', '1; 0.2', '990; 10', '1; 0.02', '3; 7', '90; 10', '9; 1', '1', '1; 1', '10; 90', '98; 2', '75; 25', '49; 1', '4; 1', '1; 0.04', '1; 0.01', '9; 5; 5', '1; 0.5', '6; 94', '96; 4', '30; 70', '47; 3', '1; 0.12', '97; 3', '15; 1', '1; 0.10', '25; 75', '1; 0.3', '1; 0.4', '2; 3', '20; 80', '95; 5', '99; 1', '40; 60', '1; 3', '50; 50', '1; 0.06', '5; 95', '70; 30', '24; 1', '2; 25', '1; 0.7', '92; 8', '1; 4'])))

    quenching_media_volume = Quantity(
        type=str,
        shape=[],
        description="""
    The volume of the antisolvent
- For gas and vacuum assisted quenching, stat the volume as ‘nan’
- If the sample is dipped or soaked in the antisolvent, state the volume of the entire solution
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['40.0', '90.0', '300.0', '350.0', '50.0', '425.0', '125.0', '0.6', '600.0', '120.0', 'Unknown', '450.0', '180.0', '45.0', '100.0', '900.0', '130.0', '400.0', '140.0', '240.0', '30000.0', '70.0', '110.0', '80.0', '250.0', '190.0', '30.0', '500.0', '150.0', '750.0', '700.0', '1.0', '1000.0', '20.0', '160.0', '200.0'])))

    quenching_media_additives_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    List of the dopants and additives in the antisolvent
- If several dopants/additives, e.g. A and B, are present, list the dopants/additives in alphabetic order and separate them with semicolonsas in (A; B)
- If no dopants/additives, leave the field blank
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'FABr', 'CsPbBr3-QDs', 'F-PDI', '2,9,16,23-tetra-tert-butyl-29H,31H-phthalocyanine', 'Poly(TA)', 'K2Rrubrene', 'L‐Leucine', 'Undoped', 'Acetylene black', 'ITIC', 'CsI', 'MAI', 'PbS-QDs', 'MABr', '6TIC-4F', 'Poly(9-vinylcarbazole)', 'Hex', 'PDMS', 'AQ310', 'P3HT', 'FAI', 'Cl-functionalized C-np', 'ITIC; PCBM-60', 'ThFAI; MAI, MACl', 'SM', 'Au-np', 'Rubrene', 'PEG', 'PCBM-60', 'Graphdiyne', 'Carbon black', 'MEH-PPV', 'FAI; MABr', 'PAMS', 'Graphydine-QDs', 'SWCNTs', 'bis-PCBM-60', 'La:BaSnO3-np', 'NPB', 'Spiro-MeOTAD', 'FAPbBr3-QDs', 'C60', 'BAI', 'MAPbBr3-QDs', 'PS', 'PBTI', 'BiFeO3-np', 'BHT', 'DF-C60', 'CuPc', 'tFM-PMAI', 'PTB7; ITIC', 'PTAA', 'IDIC-Th', 'CsPbBr3-np', 'PBDB-T', 'Carbon-nt', 'PCBM-60; TIPD', 'PABr', 'ADAHCl', 'MACl', '[M4N]BF4', 'IEICO-4F', 'H2O', 'HI', 'PEAI', 'Au@CdS', 'I2', 'PFA', 'TPFPB', 'IDIC', 'Br passivated C-np', 'DPPS', 'HEA', 'Polyurethane', 'EABr', 'MA', 'F16CuPc', 'C60; PEG', 'PTB7', 'AQ', 'PMMA', 'MAPbI3-QDs', '(PEA)2PbI4'])))

    quenching_media_additives_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the dopants/additives in the antisolvent
- If more than one dopant/additive in the layer, e.g. A and B, separate the concentration for each dopant/additive with semicolons, as in (A; B)
- For each dopant/additive, state the concentration.
- The order of the dopants/additives must be the same as in the previous filed.
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used.
- The preferred way to state the concentration of a dopant/additive is to refer to the amount in the final product, i.e. the material in the layer. When possible, use on the preferred units
o wt%, mol%, vol%, ppt, ppm, ppb
- When the concentration of the dopant/additive in the final product is unknown, but where the concentration of the dopant/additive in the solution is known, state that concentration instead. When possible, use on the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.1 mg/ml', '9 mg/ml', '0.024 mM', '40 mg/ml', '0.24 mg/ml; 0.24 mg/ml', '25 mg/ml', '20 mg/ml', '0.32 mg/ml; 0.16 mg/ml', '20 vol%', '1 vol%', '0.25 M', '0.00001 M', '0.5 mg/ml', '0.05', '0.005', '0.5 vol%', '8 mg/ml', '0.001 M', '1 mg/ml', '0.25 mg/ml', '4 vol%', '5 mg/ml', '0.36 mg/ml; 0.12 mg/ml', '0.1 M', '0.015', '2 mg/ml', '0.3 mg/ml', '6 mg/ml', '0.03', '0.000001 M', '15 mg/ml', '12 mg/ml', '30 mg/ml', '2 vol%', '10 mg/ml', '10 wt%', '3 mg/ml', '0.0001 M', '5 wt%', '0.7 mg/ml', '4 mg/ml', '0.5 M'])))

    thermal_annealing_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperatures of the thermal annealing program associated with each deposition step
- When more than one reaction step, separate the annealing temperatures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- If no thermal annealing is occurring after the deposition of a layer, state that by stating the room temperature (assumed to 25°C if not further specified)
- If the thermal annealing program is not known, state that by ‘nan’
Example
100
70; 100 >> 100
25 >> 90; 150
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['70.0 >> 70.0; 90', '70; 100 >> 150', '320; 200', '105.0 >> 105.0 >> Unknown', '90.0', '170.0 >> 100.0', '65 | 130', '45; 105', 'Unknown >> 105', '325.0', 'Unknown >> 25.0', '85', '50.0', 'Unknown >> 280.0', '70.0 >> 165.0', '100.0 >> Unknown', '220.0', '70.0 >> 100.0', '70.0 >> 85.0', '45; 100 >> 100', '100; 125', '150.0 >> 150', 'Unknown >> 100', '180.0', '65.0; 150', '60.0; 80.0', '45.0', '320; 250', '100 >> 165.0', '240; 105', '120.0 | Unknown', '25 >> 130', '70 | Unknown', '90; 25 >> 25 >> 150', '150.0 >> 100', '85.0 >> 105.0', '40 >> 180', 'Unknown | 100.0', 'Unknown >> 90.0', '85; 100', '30; 160', '65; 100 >> Unknown', '95', 'Unknown >> 115', '150; 100', '70.0 >> 110.0', '70.0 | 140.0', 'Unknown >> Unknown >> 100', '135', '105.0 >> 105', '40.0; 100 >> 90.0', '65.0; 135.0', '200; 105', '70.0 >> 70.0; 70.0', '100.0 >> 135.0', '50 >> 260', '45; 65; 100', '75 >> 75; 70', '25.0; 280.0', '100.0 >> 185.0', '400.0 >> 400.0', '70 | 130', '100; 120', '80 >> 110', '70.0 >> 70', '115; 100', '80.0 >> 135.0', 'Unknown >> 350', '100.0 >> 75.0; 100', 'Unknown >> 130', '90.0 >> 250.0 >> 250.0 >> 250.0', 'Unknown >> 50; 100', '95.0', '25; 300', '90.0 >> Unknown', '70.0 >> 80.0', '40; 100 >> 40; 100', '140.0 >> Unknown', '40; 100 >> 100', '40.0 >> 150.0', '70.0', '100 >> 150', '140.0 >> 145.0', '60.0; 80.0; 100.0', '42.0; 160.0', '160; 100', '70; 90 >> 250', '500.0', '70 >> 100.0', '90 >> 250', '60.0 >> 90.0', '90.0 >> 90.0', '0 >> 100', 'Unknown >> 60', '70 >> Unknown', '145 >> 100', '100.0 >> 100.0 >> 100.0', '130.0 >> Unknown >> 100.0', '70.0; 100 >> 100.0', '60; 85', '150.0 >> 100.0', '120; 100', '80 >> 100', '260.0', '75; 105', '80.0 >> 110.0', '85.0 >> 90.0', '175.0', '100.0 >> 100', 'Unknown >> 275.0', '160.0 >> 160.0', 'Unknown >> 150.0', '40 >> 180 >> 130', '20.0; 70.0; 100.0; 120.0', '90 >> 250 | 85', 'Unknown >> 320.0', '20; 70', '45 >> 160', '90; 25 >> 25 >> 90', '65; 135', '240.0', 'Unknown >> 70', '38.0; 160.0', '110.0 >> 250.0', '70 >> 25 >> Unknown', '70.0; 100 >> 90', '25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> Unknown', '60.0 >> 60.0 >> Unknown', '90', 'Unknown >> Unknown', '130 >> 70', '120.0 >> 100.0', 'Unknown >> 325.0', '330.0', 'Unknown >> Unknown >> 120', '40; 60; 70; 100', '75.0 >> 250.0', '70.0; 90.0 >> 90.0', '70 >> 100', 'Unknown >> 85', '120 >> 80', '100; 100', '90 >> 95', '25.0 >> 100.0', '55; 240', '85.0', '40 | 100 | 100', '40.0; 95.0', '25; 50; 75; 100', '260.0 >> Unknown >> Unknown', '70.0 >> 90.0', '90.0 >> 250.0', '100 >> 70; 100', '90.0; 125.0', '20; 100', '100.0 >> 55.0', '70; 250', 'Unknown >> 40; 100; 130', '120.0 >> 80.0', 'Unknown >> 70.0; 150', '60 >> 250', '280', '25.0 >> 170.0 >> 170.0 >> 170.0', '80; 120; 180', '60.0 >> 100.0', '27.0 >> 100.0', '52; 105', '150.0; 100.0', '70.0 | 120.0', '80.0 >> 140.0', '100 >> 70', '65; 105', '90.0 >> 60.0', 'Unknown >> Unknown >> 70.0', '25; 90', '25; 95', '180 >> 280', '70 >> 120', '110.0 | 25.0 | 25.0 | 110.0', 'Unknown >> 120.0', 'Unknown | 150', '60; 80; 100 >> Unknown', 'Unknown >> 104.0', '70; 300', '160; 230', '45.0; 160.0', '80.0; 120.0', '20.0', '25; 100', '165.0', '50', '20.0 >> 100.0', '50; 100 >> 100', '125.0 >> 100.0', '70.0 >> 130.0', '110; 70', '60; 140', '100 >> 65', '110.0 >> 70.0', '92.0', '70 >> 60; 100', '25; 60', '150 >> 150 >> 150 >> 150', '105.0 >> 105.0', '80 >> 80', '0.0; 60.0; 70.0; 80.0; 90.0', '60; 80', '180', '70; 110; 150', '100.0 >> 250.0', '40; 90 >> 95', '100; 330', '100.0; 100', '130; 150', '70.0 >> 75.0', '50.0 >> Unknown', '72.0 >> 140.0', '70.0; 100.0 >> 100.0', '25 >> 90.0', '100; 110; 120', '300.0 >> Unknown', '90; 250 >> 250', '20; 90', '65', '250.0', '20; 70; 100; 120', '60 >> 100', '70.0 >> 145.0', '100.0 >> 100.0 >> 175.0', '65.0', '50; 150; 270', '71.0 >> 70.0', '200; 150', '150.0', '35; 120 >> 120', '60.0; 70.0; 80.0; 90.0', '90 >> 25 >> 90', '85.0 >> 85.0', '28.0 >> 100.0', '60; 60; 90', '110.0 >> 150.0', '240 >> 120', '70; 95 >> 100 >> 100', '60; 100 >> 150', '90.0 >> 150.0', 'Unknown >> 70.0', '25 >> 70 | Unknown', '60; 70; 80; 90; 100; 100', '105', '180.0 >> 180.0', '110 >> 70; 100', '60.0; 100', '150.0 >> 150.0', '67.0; 100.0', '25; 50; 60', 'Unknown >> 80.0', 'Unknown >> 210.0', '150; 100 >> 100', '65.0 >> 100 >> Unknown', '110 >> 150', '50.0 >> 50.0 >> Unknown', '120 >> 100', 'Unknown >> Unknown >> Unknown >> 120.0', '90.0; 120.0', '70.0 >> 125.0', '60.0; 85', 'Unknown >> 170', '60.0 >> 60.0', '25; 90; 120', '70.0 >> 100.0 >> 100.0', 'Unknown >> 160', '40; 100 >> 150', '100.0 >> 100.0 >> 1000 >> Unknown', '60; 70; 80; 90; 100', 'Unknown >> 60; 80; 100', '100.0 >> 100.0', '26.0 >> 100.0', '70; 100 >> 70; 100', 'Unknown >> Unknown >> 105.0', '100 >> 120; 100', '50.0; 100.0', '70; 100 >> 145', '125.0; 100.0', 'Unknown >> 50.0', 'Unknown >> 140.0', '110.0 >> 105.0', '20; 80', '175', '90.0 >> 250.0 >> 250.0', '160; 105', '35.0', '75 >> 150', '140; 180', '120 >> 150', '90 >> 85', '135.0', '80.0 >> 110.0 >> 110', '250', '70 | 145', '40; 160', 'Unknown >> Unknown >> Unknown >> 130.0', '90 >> 250.0', '75.0 >> 100.0', '100.0 >> 100.0 >> Unknown', '33.0', '100 >> Unknown >> Unknown >> 100', '320', '140; 100', '75 >> 250', '25.0 >> Unknown', 'Unknown >> 70.0 >> 70.0', '70.0; 110', '60.0; 120', '66.0; 100.0', '100.0 >> 25.0 >> 100.0', '51.0 >> 70.0', '30; 40; 50; 60; 70; 80; 90; 100', '70 >> 108', '60; 100', 'Unknown >> Unknown >> 60.0 >> 60', '100 >> 80; 100', '75.0 >> 135.0', '70.0 >> 120.0', '40', '70 >> 90 >> 90', '70 >> 100 >> 100', '100 >> Unknown', '40; 60; 80; 100', '60; 150', '100 | 100', '70.0 >> 70.0 >> 80.0', '80 >> 40; 140', '90.0 >> 100.0', '50; 160', '70.0 | 135.0', '110; 25 >> 110', '80.0 >> 170.0', '100.0 >> Unknown >> Unknown >> Unknown', '80.0 >> 80.0 >> 80.0 >> Unknown', '90.0 >> 80.0', 'Unknown >> Unknown >> Unknown >> 350.0', '70.0 >> 200.0', '285.0', '70 >> 70', '120.0 >> 85.0', '25.0', '80.0 >> 120.0', '130.0 >> 130.0', 'Unknown >> 175', '115', '60; 70 >> 70; 115', '35; 120', '70 >> 85', '80.0 >> 250.0', '70 >> 150', 'Unknown >> 35.0', '40.0', '25 >> 115', '60.0; 100 >> 100; 140', '300', '70; 130', '80.0 >> 90.0', '70 >> 145', 'Unknown >> Unknown >> 150.0', 'Unknown >> 150', '70.0 >> 50.0', '70; 100; 120', '100; 120; 140', '100 | 65', '75.0', '110', 'Unknown >> 60.0', '60.0; 100 >> Unknown >> 100.0', '70.0 >> 105.0', '25.0 >> 250.0', '50; 100 | 50; 100', '35; 280', '40 | 105', '190 >> 330.0', 'Unknown >> 100.0 >> 70.0', '120; 130', '70; 130 >> 145', '25; 70 >> 85', '25 >> 100 >> 70', '160.0 >> 100.0', '150.0 >> 110.0', '130; 160', '60; 25; 100', '80; 150', '190.0', '25; 60; 65', '70.0 | Unknown | 130.0', '90.0; 100.0', '90 | 150', '100 >> 100', '330', '40; 55; 75; 100', 'Unknown >> 110.0; 120', '70 >> 140', 'Unknown >> 90', 'Unknown >> Unknown >> 130.0', '70 >> 75', '150.0 >> 90.0', '170; 100', '65.0 >> 70.0', '93.0', '60; 80; 100', '155.0', '100.0 >> 130.0', '25.0 >> 25.0', '70.0 | 70.0', '25; 130', '25 >> 150', '145.0', '45; 160', '65.0 >> 125.0', 'Unknown >> 130.0', '120; 40', '90.0 >> 85.0; 160', '180.0 >> 150.0', 'Unknown >> 90.0 >> 90.0', '110.0; 100.0', '70 >> 70.0', '25', '70.0 >> 160.0', '120.0 >> 150.0', '100.0 >> 150.0', '140.0 >> 140.0', '35; 120; 165', '280.0', 'Unknown >> 60; 70; 80; 90; 100', '60.0', '125.0', '100 >> 90', '80.0 >> 60.0', '120.0 >> Unknown', '70.0 >> Unknown', 'Unknown >> 180; 150', '130', '70.0 >> 95.0', 'Unknown >> 250', 'Unknown >> Unknown >> 350.0', '100.0 >> 75.0 >> Unknown', '60.0; 100 >> 100.0', 'Unknown >> 150.0; 100', '400.0', '70 | 150', '75 >> 75', 'Unknown >> 180.0', '80; 25 >> 80', 'Unknown >> 100.0', 'Unknown | 100', '50.0; 100 >> 150.0', '90.0 >> 110.0', '40.0 >> 250.0', '65; 125', '50; 85', 'Unknown >> 75.0', '50; 120', '115.0', '70.0 >> 140.0', 'Unknown >> Unknown >> 160', '50.0; 150.0', '25; 70; 100; 120', 'Unknown >> 140', '90 >> Unknown', '100.0 >> 70.0 >> 130.0', '90 >> 90', '60.0; 85.0', '265.0 >> 100.0', '80.0 | 150.0', '100; 180', '42; 160', '75; 90', 'Unknown >> 100.0; 140.0', '140', '100.0 >> 70', '70.0 >> 170.0', '70; 275', '80 >> 280', '285', '55; 250', '65 >> 100', '110; 115', '130.0 >> 120.0', '150.0 >> 170.0 >> 170.0 >> 170.0', '80.0 >> 350.0', 'Unknown >> 150; 120', 'Unknown >> 170.0 >> 170.0 >> 170.0', '550', '40 >> 180 >> 150', '120; 200', '40; 55; 70; 100', 'Unknown >> 150 >> 100', '25.0; 90.0', 'Unknown >> 160.0 >> 160.0', 'Unknown >> 135.0', '100 >> 280', '60 >> 200', '70 >> 70 >> Unknown', 'Unknown >> 20.0', '100.0 >> 165.0', '100', '75 >> 180', '100.0', '105.0', '40; 70 >> 40; 100', '65; 185', '70 >> 100; 120', '200; 130', '75 >> 200', '300.0 >> 150.0', '50 >> 200; 150', '80.0 >> Unknown', '70; 25 >> 25 >> 70', '60.0; 80', '60.0 >> 110.0', '90 >> 160', '70; 120', '80 >> 250', '25; 280', 'Unknown >> Unknown >> 135', '75; 125', '50; 100 >> 50; 100', 'Unknown >> Unknown >> 100.0', '70 >> 150 >> 100', '70.0 >> 100.0 >> 100', '300.0 >> 150', '80 | 80', '150 >> 25', '60.0 >> 150.0', '90.0 >> 70.0', '70; 100 | 100', '300 >> Unknown', '120.0 >> 90.0', '90.0 >> 250.0 >> Unknown', '100.0 >> 120.0', '200.0', '70.0; 100 >> Unknown', '300 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> 25 >> Unknown', '70.0; 90.0; 100.0', 'Unknown >> 50.0; 80', '95; 120', '350.0', '40; 120', '72.0 >> Unknown >> 140.0', '90.0; 120', '60; 125', 'Unknown', '70 >> 70; 100; 120', '90.0 >> 90.0 >> 90.0', '60; 90', '90; 100', '150 >> 280', '50; 150', '50; 130', '40.0; 100.0 >> 40.0; 100.0', '50; 100', '110.0; 130.0', '75 >> 130', '70.0 >> 115.0', '100; 140', '75.0 >> 70.0', '100.0 >> Unknown >> Unknown >> Unknown >> Unknown', '100; 85', '130; 170', '105.0 >> 110.0', '25 >> 25', '25 >> 135', '60 >> 100.0', '340.0', 'Unknown >> 170.0', '70.0 >> 25.0', '120.0 >> 120', '101.0 >> 100.0', '70; 175', 'Unknown >> 125.0', 'Unknown >> 160.0', '110.0 >> 100.0', '85.0 >> 150.0', '80 >> 90', '160.0', '40 >> 100', '70 | 100', '25; 80; 85; 90', '60.0 >> 105.0', '110.0 >> 75.0', '100; 130', '50; 140', '90; 150', '270.0', '50; 75; 100', '70; 100', '25 >> 25 >> 100', '170.0', 'Unknown >> 110.0', '75; 100', '120.0 >> 90.0 >> 80.0', '100.0 >> 70.0', '150 >> 150 >> 150', '150.0 >> 70', '80 >> 150', '25; 85', '170', '200', '70 >> 130', '320; 300', '25; 60; 100', 'Unknown >> 115.0; 100', '70; 70 >> 145', '100; 300 >> 100', '55.0', '65.0; 100.0', 'Unknown | 115.0', '90 | Unknown', '77', '60; 105', '105.0 >> Unknown', '130; 130', '50; 250', '100.0 >> 140.0', '25; 75 >> 135', '340', '50; 60; 70; 100', '22.5; 70; 180', '100.0; 120.0', '70.0 >> 70.0', '100 >> 250', '180; 105', '95.0 >> 60.0', '40; 110 >> 110', '60.0 >> 60.0 >> 60.0 >> Unknown', '110.0 >> 145.0', '65.0; 100', '145', '80.0 >> 150.0', 'Unknown >> 280', '100; 500 >> 110', '70 >> Unknown >> 150', '25; 150', '80.0 >> 100.0', '25; 100.0', '25; 120', '44.0', '110.0 >> 120.0', 'Unknown >> 100.0; 150', '90.0 >> 70.0; 100.0', '100.0 >> 280.0', '80; 100', '90; 120', 'Unknown >> 65; 100', '50; 100; 160', '35; 85', 'Unknown >> 170.0 >> Unknown', '550.0', '100 >> 25; 25', '70; 150 >> 100', '300.0', '120.0 >> 135.0', 'Unknown >> 40.0; 100', '185.0', '3; 5 >> 5', '90; 120 >> 100', '120.0', '0.0; 90.0', '110.0 >> 110.0', '450.0', '90.0 >> 120.0', '40.0; 100 >> 90.0 >> Unknown', '25 >> 25 >> 70', '150 >> 150', '70; 150 >> 400', '70 >> 90', 'Unknown >> 90.0; 120.0; 150.0', '130.0', '70.0 >> 150.0', 'Unknown >> 95.0', '40.0 >> 100.0', '100.0; 100.0', '60.0; 100.0', '50.0 >> 50.0 >> 50.0 >> Unknown', '25.0 >> 110.0', '75', '110.0', '70.0; 100.0', '310.0', '25.0; 100.0', '100.0; 150.0', '220', '25 >> 70', '70.0 >> 95.0 >> 95.0', '25.0 >> 150.0', 'Unknown >> 200.0', '100; 25 >> 25 >> 25', '345.0', '50.0; 85.0', 'Unknown >> 105.0', '60.0; 100 >> 60; 100', '110; 130; 150', 'Unknown >> 250.0', '100.0; 280.0', '170; 330', '100.0 >> 150', '225.0', '60.0; 100 >> Unknown', '70; 90 >> 70.0', '160', '80.0; 100.0', 'Unknown >> 120.0 >> 145.0', '60.0 >> 70.0', '25 >> 85', '170; 150', '100.0 | 100.0', '70 >> 125', '40.0; 160.0', '40.0 >> 40.0', '56; 240', '100 >> 110; 100', '100 | 100 >> 100', '70 >> 75; 70', '40 >> 100 >> 100', '100.0 >> 155.0 >> 150.0', '70; 100 >> 100', '310.0 >> 150.0', '100 >> 100; 100', 'Unknown >> 265.0', '100 >> 110', '10.0', '70', '140.0 >> 160.0', '110.0 | 150.0', '220; 105', '70 >> 70.0 >> 70', '42.0', '150; 25 >> 100', '70.0 >> 70.0 >> 70.0', '150 >> 100', '70 | 70', '200.0 >> 80.0 >> 80.0', 'Unknown >> 120', '250 | 100', '60; 120', '210.0', 'Unknown >> 100.0 >> Unknown', '85.0 >> 100.0', '75 >> 135', '130.0 >> Unknown', '43 >> 160', '60; 110', '90.0 >> 115.0', '80.0 >> 80.0', '70; 100 >> 135', '120.0; 150.0', 'Unknown >> 85.0', '50.0 >> 70.0', '70.0 >> 100.0; 150.0', '95.0 >> 100.0', '350', '75.0 >> 75.0', '100.0 >> Unknown >> Unknown', '32.0', '100.0 >> 50.0; 250', '40.0 >> Unknown', '100.0 >> 155.0', '60; 250', '100; 160', '90; 125', '60.0 >> 60.0 >> 60.0 >> 60.0', '25.0 >> 70.0', 'Unknown >> 85.0 >> 85.0', '70.0 >> 135.0', '110; 25', '130; 100', 'Unknown >> 110', '70.0 >> 70.0 >> Unknown', '100.0 >> 140', 'Unknown >> 150.12', 'Unknown >> 380.0', '25; 90; 100; 130', 'Unknown >> 350.0', '45; 55; 75; 100', '60; 130', '60; 80; 140', '25 >> 120', '100.0 >> 25.0 >> 25.0', '5; 95.0', '80; 150; 350', '80; 140; 160', '265.0 >> 200.0', '200; 170', '150', '45.0 >> 100.0; 150', '90 >> 100', 'Unknown | 125', '160.0; 100', '260.0 >> Unknown', '250.0 >> 80.0', '80.0 >> 50.0; 120', '65; 100', '85.0 >> 250.0', '150.0 >> Unknown', '50; 100; 120 >> 25', '30.0', '55', 'Unknown >> 80', '80; 90; 100; 110', '60.0 >> 140.0', 'Unknown >> 175.0', '100 >> Unknown >> Unknown', '100; 150', '70; 150', '50; 60; 70; 80; 90; 100', '25; 70', '70.0 | 150.0', '40; 100', '25 >> 140', '100.0 >> 110', '70.0 >> 110.0 >> 40.0', '25 >> 100', '100 >> 75', '335.0', '95.0 >> Unknown', 'Unknown >> 145.0', '70.0 >> 175.0', '125.0 >> 160.0', '80; 120', 'Unknown >> 115.0', 'Unknown >> 300.0', '65.0 >> 100', '25; 280 >> Unknown', '120.0 >> 250.0', '150 | 100', '70.0; Unknown', '90; 25 >> 25 >> 120', '55.0; 60.0; 100.0', '100 >> 100 >> 100', '158.0', '60 >> 150', '50.0; 100 >> 100.0', '80', '75.0; 150.0', '50 >> 280', '120', '1000.0 >> Unknown', '100.0; 150', '60; 100 >> 100', '140.0', '65; 75; 85; 95', '25; 100; 25 >> 100', '80.0', '60', '320.0', '70 >> 150 >> 70', '150.0 >> 120.0', '120.0 >> 120.0', '80.0 >> 70.0', 'Unknown >> 100.0 >> 100.0', '60 >> 60 >> 60.0', '70.0 >> 70.0 >> 70.0 >> Unknown', '125.0 >> 125.0', '100 >> 25', '100.0 >> 100.0 >> 100', '130; 140', '130; 120', 'Unknown >> 60.0; 100', '70.0 >> 250.0', '100.0; 140.0', '310', '45; 120', '60.0 >> Unknown'])))

    thermal_annealing_time = Quantity(
        type=str,
        shape=[],
        description="""
    The time program associated to the thermal annealing.
- When more than one reaction step, separate the annealing times associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the associated times at those temperatures and separate them with semicolons.
- The annealing times must align in terms of layers¸ reaction steps and annealing temperatures in the previous filed.
- If a time is not known, state that by ‘nan’
- If no thermal annealing is occurring after the deposition of a layer, state that by ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 20 and not 10-30.
Example
60
5; 30 >> 60
0 >> 5; 5
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['90.0', 'Unknown >> Unknown >> 60.0', '45.0 >> 5.0', '5.0; 20.0', 'Unknown >> 25.0', '50.0', '3.0 >> 3.0; 35.0', '5.0 >> 5.0 >> 10.0', '600.0 >> 30.0', '1.0; 1.0; 10.0', '180.0', '45.0', '10.0 >> 70.0', '0.083', '0.0; 5.0', '5.0; 5.0; 5.0; 5.0', '0.133', '3.0; 10.0 >> 3.0; 10.0', '1.0; 2.0', 'Unknown >> Unknown >> 20.0', '60.0 >> 60.0; 10.0', '2880.0 >> 1.0 >> 1.0 >> 15.0', 'Unknown >> 90.0', '30.0 >> 60.0 >> 30.0', '2.0; 28.0', '720.0 >> 3.0', '1.0 >> 1.0', '10.0; 10.0 >> 30.0', '4.0; 1.0', 'Unknown >> 720.0', '0.1666', '60.0 >> 2.0 >> Unknown', '3.0 >> 2.0', '5.0; 10.0 >> 10.0', '15.0; 50.0', '20.0 >> 40.0', '30.0; Unknown >> Unknown >> 30.0', '5.0 >> Unknown >> Unknown', '0.0', '0.066', '5760.0', '15.0 >> 15.0 >> 15.0', '60.0 >> 5.0 >> 5.0 >> 5.0', '15.0; 10.0 >> Unknown', '10.0 >> 10.0 >> 30.0', '0.07', 'Unknown >> 15.0; 60.0', '0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0', '0.33; 60.0', '120.0 >> 30.0', '3.0; 20.0', '8.0 >> 120.0', '4.0 >> 30.0', '2.0; 30.0', '30.0; 5.0', '60.0; 10.0 >> 10.0', '1.0 >> 25.0', '5.0 >> 40.0', '95.0', '0.05', '3.0; 5.0 >> 3.0; 5.0', 'Unknown >> 600.0; 10.0', 'Unknown >> 3.0; 15.0', 'Unknown >> 3.0; 30.0', '5.0 >> 30.0 >> 30.0', '3.0 >> 5.0', '70.0', '10.0 >> 30.0', '10.0 >> 60.0 >> Unknown', 'Unknown >> 10.0; 20.0', 'Unknown; Unknown; Unknown; 30.0', '2.0; 20.0', '11.0', '45.0 >> 45.0', '1.5; 2.0', '1.3 >> 10.0 >> 8.0', '15.0; 10.0; 60.0', '15.0 >> 5.0', '60.0 >> 90.0 >> 90.0', '3.0; 10.0 >> 3.0; 30.0', '0.0 >> 40.0', '10.0; 5.0; 2.0; 3.0; 50.0', '5.0; 10.0; 5.0', '0.016 >> 5.0', '1.0', '10.0; 20.0; 10.0', '60.0 >> 90.0', '10.0; 50.0', '1.0 >> 10.0', '60.0; 60.0', '15.0; 25.0; 5.0', '600.0 >> Unknown', '60.0; 35.0', '45.0 >> 10.0', '25.0; 60.0', '2.0 >> 2.0 >> 45.0', '1.5 | 20.0', '10.0 >> 0.5', '5.0; 15.0', '10.0 >> 15.0; 30.0', '30.0 >> 60.0', '10.0; 2.0', 'Unknown >> Unknown >> 15.0 >> 15.0', '15.0 >> 60.0', '0.05; 30.0', '10.0 >> Unknown >> Unknown', 'Unknown >> 15.0', '15.0 >> 4.0', '15.0; 15.0; 15.0; 15.0', '5.0; 30.0', 'Unknown >> 150.0', '0.5 >> 40.0', '3.0 >> 120.0', 'Unknown >> 320.0', '240.0', '5.0; 5.0', '10.0; 10.0; 10.0; 10.0; 10.0; 50.0', '5.0 >> 10.0 >> 10.0', 'Unknown >> 10.0; 40.0', '10.0 >> 30.0 >> 30.0', '10.0; 85.0; 10.0 >> Unknown', '15.0; Unknown', 'Unknown >> Unknown', 'Unknown >> 6.0 >> Unknown', '160.0; 17.0', '15.0; 3.0', '0.5; 30.0', '15.0 | Unknown', '2.0; 60.0', '2.0; 3.0', '10.0 >> 5.0', '20.0 >> 15.0 >> 1.0', '5.0 | Unknown | Unknown | 60.0', 'Unknown >> 240.0', '15.0; 30.0', 'Unknown >> 5.0 >> 100.0', '5.0; 5.0; 30.0', 'Unknown >> 30.0 >> 30.0', '1.0 >> 1.0 >> 1.0 >> 2.0', '3.0; 30.0', '60.0 >> 120.0', '1.0; 15.0', '1.0; 10.0 >> Unknown', '15.0 >> Unknown', '30.0 >> 60.0 >> Unknown', '10.0 >> 2.0', 'Unknown >> 6.0', '9.0', '1.0; 4.0', '50.0 >> 5.0', '30.0 >> 2.0', '1.0; 80.0', 'Unknown >> 2.0 >> 2.0', '10.0; 1.0', '0.5; 30.0 >> 0.5', '30.0 >> 30.0; 0.0', '8.0', '10.0; 10.0 >> 90.0', '2.0; 25.0', '40.0; 10.0; 10.0; 20.0', '5.0 >> 150.0', '60.0 >> 16.0 >> Unknown >> 60.0', '13.0', '0.0 >> 30.0', '0.33 >> 20.0', '1.0; 1.0', '30.0 >> 480.0', '3.0 | 5.0 | 5.0', 'Unknown >> 120.0', 'Unknown >> 30.0', '20.0 >> 45.0', '30.0 >> 65.0', '10.0 >> 10.0 >> 60.0', 'Unknown >> 45.0', '20.0', '30.0; 2.0', '15.0; 5.0', '10.0; 10.0 >> 180.0', '60.0 >> 1.0 >> 1.0 >> 15.0', '20.0 >> 120.0', '2.0 >> 540.0', '15.0; 90.0; 15.0', '90.0; 10.0', '15.0 >> 2.0', '30.0; 30.0; 10.0', '40.0 >> 10.0', '0.5 >> 5.0', '25.0 | 30.0', '5.0; 120.0; 100.0; 130.0', '2.0 >> 15.0', '25.0 >> 30.0', '30.0 >> Unknown', '5.0; 2.0', '30.0 >> 5.0', '5.0; 10.0', '20.0 >> 10.0', '4.0; 20.0', '45.0; 50.0', '0.0; 20.0', '3.0; 5.0; 10.0', '60.0 >> 30.0 >> 5.0', 'Unknown; 2.0', '50.0 >> Unknown', '10.0 >> 90.0', '10.0 | 20.0', '14.0', '0.5 >> 30.0', '0.0 >> 30.0 >> 10.0', '10.0 >> 10.0 >> 90.0', '10.0; 10.0 >> 150.0', '50.0 >> 50.0', '90.0 >> 10.0', '30.0 >> 11.5', '20.0; 10.0', '0.5; 0.5', '360.0 >> 30.0', '5.0 >> 25.0', 'Unknown >> 10.0', '3.0; 5.0 >> 10.0', '10.0 | 10.0', '45.0 | 5.0', '30.0; 30.0', '65.0', '20.0 >> 30.0', 'Unknown; 5.0', '20.0 >> Unknown', '150.0', '10.0; 15.0', '1.0; 60.0', '0.0 >> 60.0', '7.0 >> 30.0', 'Unknown >> 420.0', '3.0; 12.0', '45.0 >> 60.0', 'Unknown >> 70.0', 'Unknown >> 5.0', 'Unknown >> 5.0; 60.0', '60.0; 150.0; 15.0', '4.0; 30.0', '30.0; 180.0', 'Unknown >> 80.0', '150.0; 20.0', '0.3; 50.0', '2.0; 5.0', '3.0 >> 40.0', '0.08; 15.0', '30.0 | 30.0', 'Unknown >> Unknown >> Unknown >> 120.0', '10.0; 10.0; 10.0; 30.0', '5.0 >> 65.0', '10.0; 600.0; 90.0', '60.0 >> 60.0', 'Unknown >> 40.0', '10.0; 5.0', '0.0 | 15.0', '150.0; 15.0 >> 30.0', '1.5; 25.0', '0.16; 14.0', '10.0; 10.0 >> Unknown >> 10.0', '5.0; 10.0 >> 5.0; 10.0', '0.3', '10.0; 10.0 >> 60.0', 'Unknown >> 50.0', '0.12', '10.0 >> 60.0', 'Unknown >> 140.0', '5.0; 5.0 >> 5.0; 5.0', '35.0', '20.0; 20.0', '15.0; 60.0', 'Unknown >> Unknown >> 30.0', '120.0 >> 20.0', '5.0 >> 15.0 >> 15.0', '4.0 >> 20.0', '30.0 >> 45.0', '1.0 >> 40.0', '30.0 | 10.0', '30.0 >> 90.0', '10.0 >> 0.17', '30.0; 60.0', '5.0; 15.0 >> 90.0 >> 5.0', '90.0; 5.0', '30.0 >> 30.0 >> 30.0', '0.0 >> 10.0', '30.0 >> 40.0', '2.0; 2.0; 2.0; 60.0', '33.0', '7200.0', '60.0; 10.0', '1.0 >> 7.0', '2.5', '2.0 >> 20.0', '15.0 >> 12.0', 'Unknown >> 1.0; 20.0', 'Unknown >> 10.0 >> 15.0', '15.0; 10.0', '0.25; 5.0', '10.0 >> 1.0 >> 30.0', 'Unknown | 30.0', '2.0 | 20.0', 'Unknown >> Unknown >> 10.0', '20.0 | 20.0', '10.0 >> 150.0', 'Unknown >> 600.0', '5.0; 5.0; 5.0; 30.0', '10.0 >> 50.0', '181.0 >> 30.0', '0.5; 10.0', '60.0 >> 15.0 >> 10.0', '23.0', '30.0 >> Unknown >> Unknown', '15.0; Unknown >> 15.0', '10.0 | 60.0', '5.0; 2.0; 3.0; 50.0', '0.25', '2.0 >> 10.0', '15.0 >> 120.0', '10.0; 10.0 >> 5.0', '1.0 >> 3.0', '10.0 >> 1.0', '5.0; Unknown >> Unknown >> 90.0', '30.0 | Unknown', '25.0', '30.0; 150.0', '1.3 >> 10.0 >> 10.0', '5.0 >> 1.0', '1.0 | 20.0', '15.0 >> 20.0', 'Unknown >> 0.5', '10.0 >> 5.0 >> Unknown', '15.0 >> 6.0', '1.0 >> 120.0', '80.0 >> 250.0', '0.0 >> 20.0', 'Unknown >> 35.0', '1.0 >> 20.0', '40.0', '1.0 >> 70.0', '20.0; 40.0', '30.0 | 20.0', '2.0 >> 25.0', '5.0 >> 0.0', '120.0 >> 60.0', '144.0', 'Unknown >> Unknown >> 120.0', '2.0; 10.0', 'Unknown >> Unknown; Unknown', 'Unknown >> Unknown >> 150.0', 'Unknown | 20.0', '0.08; 30.0', 'Unknown >> Unknown >> 180.0', '10.0 >> 1.7', '0.33 >> 10.0', '10.0 >> 8.0', '10.0; 20.0; 5.0; 30.0', '2.0 >> 5.0', '75.0', 'Unknown >> 60.0', '30.0 >> 50.0; 30.0', '60.0 >> 30.0', '250.0 >> 30.0', '60.0 >> 20.0', '50.0 >> 20.0', '20.0 >> 120.0; 2.0', '480.0', '90.0; 70.0; 45.0', '60.0 >> 15.0', '3.0; 5.0; 10.0 >> Unknown', '30.0 >> 15.0', '2.0 >> 390.0', '60.0; 90.0', '25.0 >> 25.0', '10.0; 30.0', '75.0; 15.0 >> 5.0', '15.0 >> 0.0', '11.0 >> 30.0', '30.0 >> 150.0', '720.0 >> 15.0', 'Unknown; 0.5; 3.0', '30.0; Unknown', '15.0 >> 15.0', '31.0 >> 30.0', '10.0 >> 10.0 >> 5.0', '10.0; 420.0; 90.0', '50.0; 30.0', '20.0 >> 20.0', '60.0; 2880.0', '45.0 >> 10.0; 30.0', '10.0 >> Unknown', '5.0 >> 60.0', '5.0; 5.0 | 5.0; 5.0', '60.0', '5.0', '0.6', '45.0; 70.0', '7.0', '120.0 >> Unknown', '70.0 >> Unknown', '0.07 >> 0.07', '30.0; 30.0; 30.0', '3.0; 60.0', '3.0; 5.0', '122.0', '60.0 >> 10.0 >> 10.0', '5.0 >> 390.0', 'Unknown >> 180.0', 'Unknown >> 100.0', '15.0 >> 600.0', '20.0; 20.0; 20.0; 40.0', '1.5', 'Unknown >> 75.0', '3.0; 2.0', '3.0 >> Unknown', '2.0', 'Unknown >> 10.0; 10.0; 10.0; 10.0; 40.0', '5.0 >> 10.0', '10.0; 10.0; 50.0', '20.0; 5.0', '10.0 >> 10.0 >> 180.0', '1.0; 8.0', '10.0 >> Unknown >> Unknown >> Unknown', '10.0 >> 80.0', '5.0 >> 5.0; 80.0', '10.0 >> 75.0', '5.0 >> 30.0', '0.66', '5.0 >> 20.0', '4.0', '120.0; 20.0', 'Unknown >> 0.333333333', '121.0', '1.0 | Unknown', '12.0 >> 60.0', '5.0 >> 5.0; 10.0; 10.0', '180.0 >> 30.0', '60.0 >> 5.0 >> Unknown', '10.0; 4.0', '30.0; 90.0', '100.0; 20.0', '15.0 >> 30.0', '6.0 >> 30.0', '10.0 | 1.4', '60.0; 25.0', '600.0', '12.0', '5.0 >> 2.0', 'Unknown | 120.0', '5.0; 5.0 | Unknown', 'Unknown >> 20.0', '16.0 >> 30.0', '100.0', 'Unknown >> 5.0 >> 20.0', '105.0', '10.0 >> 10.0 >> 10.0', '2.0 >> Unknown', '15.0 >> 45.0', '13.0 >> 60.0', '720.0', '10.0 >> 10.0', 'Unknown >> 45.0 >> Unknown', '30.0; Unknown >> 30.0', '10.0; 10.0 >> 120.0', '3.0; 10.0', '3.0; 5.0 >> 30.0', 'Unknown >> Unknown >> 90.0', 'Unknown; Unknown', '10.0 >> 15.0', '20.0 >> 30.0 >> Unknown', '100.0 >> 2.0', '5.0 >> 10.0; 10.0', '15.0; 0.17 >> 4.0', '0.5', '0.3; 2.0', '10.0 >> Unknown >> Unknown >> Unknown >> Unknown', '2.0; 5.0 >> Unknown', 'Unknown >> 5.0; 5.0; 5.0', '0.83; 0.83', '2.0 >> 60.0', '3.0 >> 5.0 >> 5.0', '1.0; 2.0 >> Unknown', '60.0 >> 5.0', 'Unknown >> Unknown >> 5.0', '5.0; 2.0; 5.0', '5.0; 5.0 >> 5.0', '10.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0 >> 0.0', '90.0; 15.0', 'Unknown >> Unknown >> Unknown >> 1.0', '200.0', '5.0 >> 150.0 >> 20.0', '20.0 >> 5.0', 'Unknown; 10.0', '10.0; 20.0; 0.0', '10.0 >> 20.0; 60.0', '15.0; 15.0; 90.0', '1.2', '30.0; 60.0 >> 60.0', '15.0 | 60.0', '3.0; 50.0', '60.0; 1440.0', '3.0', '1.0 >> 2.0', '10.0; 90.0', '5.0; Unknown >> Unknown >> 10.0', 'Unknown', '10.0 | 30.0', '10.0 >> 40.0', 'Unknown >> 260.0', '2.0 >> 100.0', 'Unknown >> 0.0', '0.16 >> 15.0', '60.0; 720.0', '5.0; 60.0', '60.0; 10.0 >> 5.0; 10.0', '2.0; 6.0 >> 8.0', '10.0 >> 4.0', '1.0 | 10.0', '10.0 | 120.0', '1.0; 7.0', '15.0 >> 150.0', 'Unknown >> Unknown >> Unknown >> 20.0', '7.0; 1.0 >> 10.0', '45.0; 45.0', 'Unknown >> 10.0 >> 10.0', 'Unknown >> 40.0; 100.0; 130.0', '5.0; Unknown >> Unknown >> 30.0', '2.0 >> 0.17', '60.0; 15.0', '15.0; 15.0 >> 15.0', '10.0; 10.0; 0.0', '30.0 >> 30.0', '30.0; 150.0; 15.0', '15.0; 15.0', '20.0; 30.0', '2.0; 30.0 >> 2.0; 30.0', '10.0; 45.0', '1.0 >> 30.0', '0.17; 12.0', '4.0 >> 4.0', '15.0 >> 30.0 >> 10.0', '0.17; 75.0', '38.0', '120.0 >> 1.0 >> 1.0 >> 15.0', '10.0 >> 10.0 >> Unknown', '30.0 >> 10.0 >> 10.0 >> 6.0', '270.0', '0.33; 2.0', 'Unknown >> 10.0; 60.0', '15.0; 90.0; 60.0', 'Unknown >> 4.0', '5.0; 30.0; 5.0; 30.0', '3.0 >> 30.0', '4.0; 10.0', '3.0; 5.0 >> 15.0 >> Unknown', '10.0 >> 20.0', '15.0; 15.0; 120.0; 15.0', '10.0 >> 110.0', '4.0 >> 45.0', '55.0', '1.0 | 15.0', '10.0; 60.0; 90.0', '60.0; 30.0', '10.0 >> 3.0', '1.3 >> 10.0', '50.0 | 10.0 >> 10.0', '0.33 >> 15.0', 'Unknown >> 360.0; 10.0', '6.0', '30.0; 20.0', '5.0; 60.0; 10.0', '30.0 >> 18.5', 'Unknown >> 12.0', '30.0; 30.0 >> 30.0', '60.0 >> 10.0', '10.0 >> 10.0 >> 120.0', '30.0 >> 120.0', '3.0 >> 10.0', '15.0 >> 90.0', '30.0 >> 20.0', '3.0 >> 2.0; 20.0', '15.0 >> 10.0', '5.0 | 12.0', '5.0 | 60.0', '30.0 >> 0.0 >> 120.0', '2.0; 2.0', 'Unknown; 20.0 >> 20.0', '3.0; 6.0 >> 10.0', '0.33; 10.0', 'Unknown >> 0.8541666666666666', '2.0; 15.0', '1.0 >> 1.0 >> 2.0', '1.0 >> 15.0', '15.0', '1.0 >> 60.0', '10.0 >> 10.0; 15.0', '45.0 >> 30.0', '30.0 >> 1.0', 'Unknown | 10.0', '0.58 >> 5.0', 'Unknown >> 2.0 >> 3.0', '45.0 >> Unknown', '300.0', '40.0; 100.0 >> 100.0', 'Unknown >> 60.0 >> Unknown', '2.0; 6.0', '120.0 >> 20.0 >> 5.0', '10.0; 20.0; 20.0; 20.0', '20.0 >> 15.0', '15.0; Unknown >> 25.0 >> Unknown', '120.0', '0.5; 60.0', '0.4305555555555556', '20.0; 20.0; 20.0', 'Unknown >> Unknown >> 15.0', 'Unknown >> 10.0; 10.0; 10.0', '5.0; 10.0 >> 30.0', '40.0 >> 120.0', '0.5; 15.0', '1.0; 3.0', '12.0; 10.0', '5.0; Unknown >> Unknown >> 60.0', '2.0; 5.0; 10.0', '2.0 | Unknown', '0.17', '110.0', '5.0 >> 12.0', '20.0 | 15.0', '1.0; 5.0', '120.0 >> Unknown >> Unknown', '1.0; 10.0', '10.0 >> 10.0 >> 240.0', 'Unknown; Unknown >> 30.0', '10.0 >> 10.0; 20.0', '2.0 >> 2.0', '5.0; 0.0', 'Unknown >> 10.0; 120.0', '1.0 >> 16.0', '5.0 >> 95.0', '15.0; 120.0; 15.0', '10.0 >> 0.0', '15.0 >> 10.0 >> 10.0', 'Unknown >> 200.0', '5.0 >> 312.0', 'Unknown >> 105.0', '3.0 | 10.0', '3.0 >> 5.0 >> 90.0', '2.0; 4.0', '50.0; 5.0 >> 5.0', 'Unknown >> 10.0 >> 60.0', '0.1', '10.0; 300.0; 90.0', '15.0; 15.0; 90.0; 10.0', '15.0 | 15.0', '3.0; 5.0; 60.0', '2.0 >> 45.0', '0.13', '2.0 >> 30.0', 'Unknown >> 1.0', '7.0 >> 1.0', '15.0; 45.0', '2.0 >> 120.0', '40.0 >> 40.0', '10.0 >> 10.0 >> 10.0 >> 10.0', '10.0', '10.0; 40.0', '10.0; 10.0; 10.0; 10.0; 40.0', 'Unknown >> 18.0', '5.0 >> 15.0', '5.0 >> 780.0', '30.0; 15.0', 'Unknown >> 2.0', '6.0; 10.0', '3.0; 5.0 >> 15.0', '0.5; 20.0', '0.5; 5.0', '2.0 >> 240.0', '15.0; 0.05 >> 4.0', '210.0', '10.0; 10.0; 10.0', 'Unknown >> 7.0', '0.5 >> 5.0 | 5.0', '15.0; 20.0', '120.0 >> 10.0', '30.0 >> 600.0', '150.0; 15.0', 'Unknown >> 8.0', '10.0; 5.0; Unknown >> 5.0', '40.0; 120.0; 100.0; 130.0', '1.0; 9.0', 'Unknown >> 15.0 >> Unknown', '1.0; 25.0', '10.0 >> 120.0', '1.45; 60.0', '40.0 >> Unknown', '15.0; 60.0 >> 5.0', '10.0; 150.0; 90.0', '5.0 | 10.0', '5.0; 30.0; 2.0', '30.0; 3.0 >> 15.0', 'Unknown | 5.0', '1.0; 20.0', '2.0 >> 4.0', 'Unknown >> 30.0; 30.0', '24.0', '5.0 >> 5.0', '0.5; 2.0', '30.0 >> 30.0; 30.0', '30.0; 80.0', '30.0 >> 30.0; 5.0', '4320.0', '10.0 >> 1.0 >> 10.0', '8.0; 5.0', 'Unknown >> 65.0', '100.0; 10.0', '30.0', '40.0; 10.0', '2.0; 58.0', '2.0 >> 5.0; 60.0', 'Unknown >> 3.0', '1.0 >> 5.0', '16.0 >> 10.0', 'Unknown >> 10.0; 100.0', '1.6', '15.0 >> Unknown >> Unknown', '360.0', '60.0; 20.0', '10.0; 10.0', 'Unknown >> 10.4', '10.0; 10.0; 10.0; 10.0; 10.0; 10.0', 'Unknown >> 1.0 >> 1.0 >> 15.0', '3.0; 10.0 >> 0.25', '10.0; 30.0; 90.0', '600.0 >> 5.0', '30.0 | 5.0', '1.0; 2.0 >> 2.0; 4.0', '20.0 >> 60.0', '0.25; 60.0', '1440.0', '4.0 >> 15.0', '5.0 >> 5.0 | Unknown', '5.0; 5.0; 5.0; 10.0; 10.0; 10.0; 10.0; 90.0', '30.0 | Unknown | 150.0', '5.0 >> Unknown', '4.0; 60.0', '140.0', '120.0; 10.0', '3.0; 7.0', '80.0', '40.0 >> 30.0', '120.0 >> 120.0', 'Unknown >> 15.0 >> 10.0', '5.0 >> 15.0; 75.0', '11.0 >> 60.0', '5.0 >> 45.0', '3.0; 5.0 >> 5.0', '15.0; 0.08 >> 4.0', '60.0 >> 5.0 >> 5.0', '50.0 | 10.0', '10.0 >> 10.0; 60.0', '10.0; 60.0', '1.0; 30.0', '0.08; 20.0', 'Unknown >> Unknown >> 1.0', '60.0; 60.0; 60.0', '30.0 >> 10.0', '180.0; 17.0', '60.0 >> Unknown'])))

    thermal_annealing_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere in which the thermal annealing is conducted.
- When more than one reaction step, separate the atmospheres associated to each annealing step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of deposition steps must line up with the previous columns.
- If the atmosphere is a mixture of different gases, i.e. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas.
- This is often the same as the atmosphere under which the deposition is occurring, but not always.
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Air >> N2
Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['N2', 'Unknown >> O2', 'Vacuum; N2', 'Air >> Air | Air', 'Air >> N2', 'N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2', 'Dry air >> Air', 'Unknown', 'N2 >> Vacuum', 'Vacuum; Unknown; Unknown', 'N2 >> N2', 'N2 | N2', 'N2; Air', 'Unknown >> Air', 'Dry air >> Dry air', 'Ambient', 'Unknown >> Unknown', 'Ar >> Ar', 'Unknown >> N2', 'Dry air', 'N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2 >> N2', 'Vacuum >> N2', 'Air', 'Unknown | 70', 'Vacuum', 'Ar >> Vacuum', 'Unknown >> Air >> Air', 'N2; Ambient', 'Air >> Air', 'Ar', 'N2 >> Air'])))

    thermal_annealing_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relative humidity during the thermal annealing
- If there is more than one annealing step involved, list the associate relative humidity in the surrounding atmosphere and separate them by a double forward angel bracket (‘ >> ‘)
- The number and order of annealing steps must line up with the previous column
- If there are uncertainties, only state the best estimate, e.g. write 35 and not 20-50.
- If a humidity is not known, stat that as ‘nan’
Example
0
35 >> 0
nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0 >> 0', '15', '0; 60', '32', '50.0', '30; 50', '0; 0', '60.0', '43', '0.9', '45', '20 >> 20', '60', '30', '90', '65.0', '0 | 0', '0 >> 25', '50', '10'])))

    thermal_annealing_pressure = Quantity(
        type=str,
        shape=[],
        description="""
    The atmospheric pressure during the thermal annealing
- If there is more than one annealing step involved, list the associate atmospheric pressures and separate them by a double forward angel bracket (‘ >> ‘)
- The number and order of annealing steps must line up with the previous column
- Pressures can be stated in different units suited for different situations. Therefore, specify the unit. The preferred units are:
o atm, bar, mbar, mmHg, Pa, torr, psi
- If a pressure is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 100 pa and not 80-120 pa.
Example
1 atm
1 atm >> 0.002 torr
1 atm >> 1 atm >> nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.1 Torr >> 1 Torr', '0.1 Torr', '1 atm >> 1 atm', ' Vacuum >> 1 atm', '1 bar', '1 atm; 0.2 bar', '1 atm | 1 atm', '50 Pa; nan; nan', '1 atm', '0.001 bar', '1 atm >> 1 atm | 1 atm', 'nan >> 700 Pa'])))

    solvent_annealing = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if there has been a separate solvent annealing step, i.e. a step where the perovskite has been annealing in an atmosphere with a significant amount of solvents. This step should also be included deposition procedure sequence but is also stated separately here to simplify downstream filtering.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    solvent_annealing_timing = Quantity(
        type=str,
        shape=[],
        description="""
    The timing of the solvent annealing with respect to the thermal annealing step under which the perovskite is formed. There are three options.
- The solvent annealing is conducted before the perovskite is formed.
- The solvent annealing is conducted under the same annealing step in which the perovskite is formed
- The solvent annealing is conducted after the perovskite has formed.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Under', 'Before', 'After'])))

    solvent_annealing_solvent_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in the solvent annealing step
- If the solvent atmosphere is a mixture of different solvents and gases, e.g. A and B, list them in alphabetic order and separate them with semicolonsas in (A; B)
Example
DMSO
DMF
DMF; DMSO
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['H2O', 'DMSO; H2O', 'TBP', 'Acetic acid; Chlorobenzene', 'Pyridine', '4‐fluoroaniline', 'Chlorobenzene; DMSO', 'GBL', 'DMSO; IPA', 'Aminobutanol', 'Methylamin', 'Methanol; Methylamin', 'Unknown', 'Hac', 'Triethylenetetramine', 'Diethylenetriamine', 'Methanol', 'DMF', 'DMF; H2O', 'DMSO', 'Chlorobenzene', 'Chlorobenzene; DMF', 'HCl', 'Thiophene', 'DMF; DMSO', 'Toluene', 'DMF; IPA', 'Benzyl alcohol', 'NMP', 'Ethanol', 'Air', 'Air; DMSO', 'Vacuum', 'CCl4', 'MACl', 'DMSO; NMP', 'IPA', 'Hexane'])))

    solvent_annealing_time = Quantity(
        type=str,
        shape=[],
        description="""
    The extend of the solvent annealing step in minutes
- If the time is not known, state that by ‘nan’
- If the solvent annealing involves a temperature program with multiple temperature stages, list the associated times at each temperature and separate them with a semicolon (e.g. 5; 10)
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['40.0', '90.0', '2.5', '50.0', '10.0', '60.0', '5.0', '120.0', 'Unknown', '180.0', '0.16666666666666666', '80.0', '480.0', '1.5', '0.25', '30.0', '0.3', '2.0', '1.0', '20.0', '0.08333333333333333', '15.0'])))

    solvent_annealing_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature during the solvent annealing step
- The temperature refers to the temperature of the sample
- If the solvent annealing involves a temperature program with multiple temperature stages, list the associated temperatures and separate them with a semicolon (e.g. 5; 10) and make sure they align with the times in the previous field.
- If the temperature is not known, state that by ‘nan’
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['200', '120.0', 'Unknown', '90.0', '100', '90', '100.0', '20', '110.0', '150.0'])))

    after_treatment_of_formed_perovskite = Quantity(
        type=str,
        shape=[],
        description="""
    Any after treatment of the formed perovskite. Most possible reaction steps should have been entered before this point. This is an extra category for procedures that just does not fit into any of the other categories.
Examples:
Hot isostatic pressing
Magnetic field
UV radiation
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Washed with Ether', 'Ultrasonic vibration treatment', 'Washed with Toluene', 'DABr', 'Post annealing', 'Dipped in Toluene', 'Washed with DMSO', 'Degradation in air under AM 1.5', 'Magnetic field', 'CF4 plasma treatment', 'Deposition Nanocrystals of CsPbI3', 'Deposition Nanocrystals of CsPbBr3', 'Thermal radiation', 'Washed with Ethyl acetate', 'Washed with chloroform and Ether', 'Spin coating GASCN', 'Annealed under pulsed light', 'Vacuum oven annealing', 'Pressed with flat stamp', 'Pulsed light', 'Fast Cooling', 'Washed with IPA', 'Dipped in octadecene >> Washed with cyclohexane', 'TETA vapour treatment', 'Hot isostatic pressing', 'Drying cabine', 'Washed with IPA >> Thermal Annealing', 'Poling', 'Slow cooling', 'Washed with DMF', 'Photonic curing', 'IPA:ACE @ 1:1 washing', 'UV laser radiation', 'Repeated Spin-coating', 'Hot-pressing', 'Dipped in Chlorobenzene', 'Washed with acetylene black (15 mg/ml) solution in chlorobenzene', 'Micro contact inprinting', 'Spin coating@Guanidinium thiocyanate', 'Gas pump treatment', 'Spin coating GASCN; MACl', 'IR radiation', 'DCM:DEE @ 50:50 washing', 'Vaccum drying', 'fs laser polishing >> Washed with IPA >> Spin-coating solution of CsI, FAI, MABr in IPA >> annealed at 100', 'Washed with ACE', 'Degradation in air under dark', 'H2O2 treatment', 'Gradient thermal annealing', 'Soaking in Isopropyl', 'Washed with Acetone', 'Washed with IPA >>  Spin-coating Dichloromethane', 'Annealed under intense laser pulses', 'Graphdiyne passivation treatment', 'Ethylacetate washing', 'Cold-roll pressing', 'IPFB immersion', 'Washed with GBL', 'Dipped in Ethyl acetate', 'Ultrasonic transducer', 'Atmospheric-pressure dielectric barrier discharge', 'Washed with IPA >> Drying in flow of N2', 'Washed with Anisole', 'Dissolving polystyren template in toluene', 'Spin-coating Ethanol', 'Cold isostatic pressing', 'Pressed with dotted stamp', 'Washed with IPA >> Washed with Dichloromethane', 'Annealed in a perfluorodecalin bath', 'Intense pulsed light annealing', 'Inverted thermal annealing', 'Washed with MABr solution in IPA', 'Exposed to moist air', 'Dried by N2 gas', 'Intense light', 'FABr treatment >> Rinse with 2-propanol >> Thermal annealing >> Slow cooling', 'Washed with MAI solution in IPA', 'Dried under flow of N2', 'Spin coating GABr   >>   annealing', 'Toluene washing', 'Refrigerated', 'Ultrapure water spray', 'Moisture', 'Spin-coating iodopentafluorobenzene', 'Radiative thermal annealing', 'Near Infrared radiation', 'Light soaking', 'Vacuum annealing', 'UV radiation', 'Washed with Chlorobenzene', 'Microwave radiation', 'Dipped in FAI containing ethyl acetate solution', 'Heating >> Light exposure', 'Dipped in Anisole', 'Washed with Methyl acetate', 'Annealed under light', 'Spin-coating Pr-ITC; Ph-DTIC', 'Pressed with hexagonal stamp', 'Washed with IPA and drying with pressurized air', 'IR laser radiation', 'Stored at elavated temperature', 'Laser annealing', 'Dried under flow of clean air'])))

    after_treatment_of_formed_perovskite_met = Quantity(
        type=str,
        shape=[],
        description="""
    Connected to the previous field (After treatment of formed perovskite). The label describing the method should be in the previous filed, and the associated metrics in this one. For exampleThe sample is intense gamma radiation at a flux of X under 45 minutes. The “gamma radiation” is the label, and the time and the flux is the metrics. Give the units when you state the metrics
Examples:
40kHz; 5W; 4 min
45 deg. C
30 min
50 W/cm2; 2.38 s
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '50 min', '50 W/cm2; 2.42 s', 'Colling rate < GHT-3', '40kHz; 5W; 2 min', '40kHz; 10W; 2 min', '10 pulses per sample', '2.5', '40kHz; 10W; 1 min', 'Colling rate < GHT-2', '1', '40kHz; 10W; 3 min', '2.47 s', '5', '105 deg. C', '70 C >> 254 nm', '30% RH; 8 h', '30 min', '100 deg. C', '40kHz; 5W; 1 min', '7s', '30% RH; 12 h', '50 W/cm2; 2.40 s', '50 W/cm2; 2.38 s', '50 W/cm2; 2.50 s', '40kHz; 10W; 4 min', '40kHz; 5W; 3 min', '30% RH; 4 h', '2 min', '85 deg. C in Air 50 % RH', '9s', 'Colling rate < GHT-4', '40kHz; 5W; 4 min', '20 min', '50 W/cm2; 2.45 s', '10s', '10 s', '13s', '45 deg. C', '105 C', '40 mW/cm2', 'Fast cooling in ice 150 >> 0', '80 deg. C; 15 s', '11s', '50 W/cm2; 2.53 s', '50 W/cm2; 2.55 s', '50 W/cm2; 2.47 s', '200 Mpa; 90 deg.C; 60 min', '0.5', '85 deg. C in O2', 'Several hours', 'Fast cooling in air 150 >> 0', '85 deg. C in N2', 'nan >> 100 deg C 5 min', '80W/cm 20.1mm/s', '500W, 30 sec', '200 Mpa;  60 min', 'Spin-coating>> 2 mg/ml>> 100 deg. C>> 10 min', '70 deg. C; 60 min'])))

    def normalize(self, archive, logger):
        add_solar_cell(archive)
        if self.procedure:
            archive.results.properties.optoelectronic.solar_cell.absorber_fabrication = self.procedure.split(' | ')


class HTL(MSection):
    """
    A section to describe information related to the Hole Transport Layer (**HTL**).
    """

    stack_sequence = Quantity(
        type=str,
        shape=[],
        description="""
    The stack sequence describing the hole transport layer. Use the following formatting guidelines
- With the HTL, we refer to any layer between the substrate and the perovskite in a pin-device, and any layer between the perovskite and the back contact in a nip-device.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If two materials, e.g. A and B, are mixed in one layer, list the materials in alphabetic order and separate them with semicolons, as in (A; B)
- If no hole transport layer, state that as ‘non’
- Use common abbreviations when appropriate but spell it out if risk for confusion.
- If a material is doped, or have an additive, state the pure material here and specify the doping in the columns specifically targeting the doping of those layers.
- There is no sharp well-defined boundary between when a material is best considered as doped or as a mixture of two materials. When in doubt if your material is best described as doped or as a mixture, use the notation that best capture the metaphysical essence of the situation.
- There are a lot of stack sequences described in the literature. Try to find your one in the list. If it is not there (i.e. you may have done something new) define a new stack sequence according to the instructions.
Example:
Spiro-MeOTAD
PEDOT:PSS
none
NiO-c
PTAA
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['TAPC | MoO3; TAPC', '1‐adamantylamine hydrochloride | Spiro-MeOTAD', 'MDMO-PPV', 'TTPA-OMeTPA', 'NiO | Br-BA', 'P3CT-K', 'TPB', 'ZnBu4Pc', 'BuO-DATPA', 'FT-OMeTPA', 'BChl-2', 'Graphene oxide | CuBuPc', 'PTZDPP-2', 'Al2O3-mp | Me2N-DATPA', 'po-TPE-4DPA', 'TDAC', 'Au-np | NiO-c', 'B58', 'CuSCN-3D', 'HMDI', 'Me6-ZnPc', 'PMT', 'T103', 'Ag-np; NiO-c', 'NiO-np | TPI-4MEO', 'CsPbI3-QD | Spiro-MeOTAD', 'P(BDTT-SePPD)', 'PDBT-co-TT', 'GD; P3HT', 'KY7F22-np | Spiro-MeOTAD', 'PDBT-T1 | WOx', 'YT3', 'M116', 'CH3O-PEIA | Spiro-MeOTAD', 'HTM-2', 'TaTM | F6-TCNNQ; TaTm', 'PEDOT:PSS | Perylene', 'F6-TCNNQ | TaTm', 'V842', 'PEDOT:PSS | PSS-Na', 'NiO-c | PNP-BC', 'PEDOT:PSS | Au-np; VOx', '[BMPA-BTD]3-TPA', 'YKP06', 'DTBT', 'rGO-flakes | Spiro-MeOTAD', "9,9'-([1,2,5]thiadiazolo[3,4-c]pyridine-4,7-diylbis(4,1- phenylene))bis(N3,N3,N6,N6-tetrakis(4-methoxyphenyl)-9H-carbazole-3,6-diamine)", 'PFO | Al2O3-np', "4,4'-((2-Hexyl-2H-benzo[d][1,2,3]triazole-4,7-diyl)bis(thiophene5,2-diyl))bis(N,N-bis(4-(hexyloxy)phenyl)aniline)", 'TFB', 'V950', 'VOx | X-DVTPD', 'HB-Cz', 'C201', 'V1036', '2-MP | Spiro-MeOTAD', 'SWCNTs | PTAA', 'HTM4', 'H-2,3', 'MoS2', 'CON-10 | PEDOT:PSS', 'PEDOT:PSS | PTB7', 'YKP03', 'TPA-NAP-TPA', 'PTAA | TPFB', 'TTPA-DTP', 'CS05', 'SGT-407', 'CuPcNO2-OBFPh', 'EGO-PPV', 'RE-ZnBu4Pc', 'CdSe-QD | Spiro-MeOTAD', 'NiO-c | 1ab', 'MoOx | F4TCNQ', 'WS2', 'WO3', 'dly-2', '18-crown-6 ether | CuSCN', 'MoO3 | TBDI', 'BDTT:DPPD:TT', 'SY3', 'TTF1', 'PT', 'Tea Catachinrich biomolecules', 'PEH-9', 'NiO@C | Spiro-MeOTAD', 'ST1 (4-(4-(bis(4-(4-(dibutylamino)styryl)phenyl)amino)styryl)-N,N-dibutylaniline)', 'NiO-c | NBP-BC', 'CuPc-DMP', 'SGT-405', 'AL2O3-c | Spiro-MeOTAD', 'PTAA', 'di-TPA', '2-((5-(4-(2-ethylhexyl)-4H-dithieno[3,2-b:2′,3′-d]pyrrol-2-yl) thiophen-2-yl)methylene) malononitrile', 'PBDT(T)(2F)T', 'PPP', 'B[BMPDP]2', 'V873', '9,9’-di{6-[3-(2-(4-methylphenyl)vinyl)-9-carbazol9-yl]hexyl}-[3,3’]bicarbazole)', 'Caffeine | Spiro-MeOTAD', 'Poly-TBD', 'X60', 'HOFP | Spiro-MeOTAD', 'NP1', 'CuBuPc', 'PolyTPD | Spiro-MeOTAD', 'Al2O3-c | Ethyl acetate; I2; LiI; NMBI; Urea', 'PVBT-SO3', 'Spiro-MeOTAD | Cu1.75S', 'FBT-TH4 | CuxO', 'PTB1', 'Z34', 'PSS-g-PANI', 'MEAI | Spiro-MeOTAD', 'NiO-np | PMMA', 'M118', 'DPA-ANR-DPA', 'YR3', 'V997', 'DBT(QT-TPA)2', 'SO10', 'TaTm | TaTm:F6-TCNNQ', 'TBDI', 'PO-Spiro', 'TTPA-DSQT', 'Spiro-MeOTAD | PDPP4T', 'SnS-np; ZnS-np | NiO-np', '1b @ triphenylamine modified azobenzene dyes', 'THY-1', 'SP-11', 'Zeocoat | Graphene | P3HT', 'P3HT | Spiro-MeOTAD', 'H11', 'CuI | PEDOT:PSS', 'PTAA | PFN-P2', 'PTTh', 'TaTm | TPBi | MoO3', 'DEH', 'Perylene', 'PEDOT:PSS | pTPD', 'SDTCz2F', 'TCTA-BVP', 'EDT; PbS', 'PDCBT', 'NiO-c | Au-np', 'MeOAc-Spiro-OMeTAD', 'C102', '3,4-spiro', 'PTPD2', '3,3′,5,5′-tetrasubstituted 1,1′-biphenyl', 'X14', 'H101', 'POZ6-2', '1a @ triphenylamine modified azobenzene dyes', 'PEDOT:PSS | TPA-NPA-TPA', 'SubPc', 'Z10', 'OIPC-Br', '4b @ triphenylamine modified azobenzene dyes', 'X54', 'TPAC3M', '5,7-disubstituted azulene', 'CZTPA-1', 'NiO-np | TPI', 'P3HT | Ta:Wox-np', 'FT73', 'TDT-OMeTAD', 'AQ310 | Spiro-MeOTAD', 'Spiro-MeO-TPD', 'BL40', 'PEDOT:SAF', 'YC-3', 'TPA3C', 'V1056', 'SCPDT-BiT', 'NiO-np | 2,2’-BiPy', 'Fe3O4-np', 'Carbozole @ S14', 'PZn-DPPA', 'X2', 'SFD-Spiro', 'SWCNTs', 'HTB-Ome', 'CZTSSe; rGO', 'Q198', 'EVA; MWCNTs | Spiro-MeOTAD', 'BTPA', 'CsSnBr2I-QDs', 'MeoTPD', 'CuH', 'V2O5', 'HTM2', 'Graphene oxide | PTAA', "N2,N2,N3',N3',N6',N6',N7,N7-octakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2,3',6',7-tetraamine", 'M114', 'Benzylamine | Spiro-MeOTAD', 'pPh-2MODPACz', 'PMA | TaTm', 'DERDTS-TBDT', 'TTB3', 'BMIMBF4 | Spiro-MeOTAD', 'NBP-BC', 'TB-ZnPc', '(OctPhO)8CuPc1', 'PEDOT:PSS | V2O5', 'BTP-1', 'M4', 'Benxylamine | Spiro-MeOTAD', 'BDT-4D', 'oxo-Graphene', "4,4'-(5,10,11-Trihexyl-10,11-dihydro-5H-thieno[2′,3':4,5]pyrrolo [3,2-g]thieno[3,2-b][1,2,3]triazolo[4,5-e]indole-2,8-diyl)bis(N,N-bis(4- (hexyloxy)phenyl)aniline)", 'PEDOT:PSS; PEG', 'Spiro-MeOTAD | Al2O3-c', 'mm-SFX-2PA', 'Spiro-tBuBED', 'PTB-BO', 'V1004', 'CZTS0.5Se0.5; rGO', 'MPA', 'PTAA | PS', 'P-SC6-TiOPc', 'PBDT(2H)T', 'POZ2', 'PIF8-TAA', 'Spiro-PT-OMeTAD', 'BT', 'BTPA-TCNE', 'NiMgO', 'POZ9', 'Thiophene | Spiro-MeOTAD', 'HMe2Pc', 'TCPI | Spiro-MeOTAD', 'NiO-c | 1bb', 'PCDTBT; PFN', 'MoO2-np', 'Carbon-nt; PEDOT:PSS | PEDOT:PSS', 'Graphene oxide; PEDOT:PSS', '4-(4-phenyl-4-alfa-naphthylbutadienyl)-N,N-di(4-methoxyphenyl)-phenylamine', 'YC03', 'tetra-TPA', 'MHGO', 'P:ON', '1,3-disubstituted azulene', 'Ni | Au | Cu', 'M113', '2FBTA-2', 'NiO-c | NiO-np', 'DDOF', 'Trux-OMeTAD', 'PB2T-S', 'PEDOT:PSS | CuI', 'PEDOT:PSS | FrGO', 'PCE-10', 'HAB1', '2,7-BCz-OMeTAD', '2,5-bis (5-(5-(5-hexylthiophen-2-yl) thiophen2-yl) thiophen-2-yl) thiazolo[5,4-d]thiazole', 'BTF2', 'PEDOT:PSS | P3HT', 'CS04', 'PB(NAP-Th)TBT', '[BMPA-EDOT]3-TPA', '18-crown-6 ether | Spiro-MeOTAD', 'T30P', 'MoOx | TaTm', 'Z7', 'P(BDTT-ttPPD)', 'PEDOT:PSS | Porphyrin', 'Q221', 'SQ2 | Spiro-MeOTAD', 'Tetracene', 'TS-CuPc | PEDOT:PSS', 'D1', 'PTAA | Al2O3-mp', 'P3CT', 'MoO3', 'BTPA-5', 'P3HT', 'MoO3 | CuPc', 'V1012', 'EDOT-OMeTPA', 'CdTe@MAPbI3-QDs | Spiro-MeOTAD', 'NPD', 'Zn-Chl | H2-Chl', '3C', 'FBA2', '4-(2-(4-(Bis(4-(hexyloxy)phenyl)methyl)phenyl)-9-methyl-9H-naphtho[2,1-c]carbazol-12-yl)-N,N-bis(4-(hexyloxy)phenyl)aniline', 'ZnChl-3', '2TPA-3-DP', 'Spiro-MeOTAD | P3HT; SWCNTs | PEDOT:PSS', '2D-Sb', 'SY2', 'SFT-TPAM', 'Al2O3-np | CuBuPc', 'Th-PDI', 'TAA14', 'DPP-F', 'B74', 'Al-np; PEDOT:PSS', 'CuSCN | MoOx', 'I2; KI; Propylene carbonate; Polyethylene glycol', 'Z25', 'Black phosphorous 2D | CuSCN', 'Spiro-TTB | VOx', 'TPBC', 'POSS-SH | Spiro-MeOTAD', 'CZ-STA', 'PEDOT:PSS | 3-aminopropanoic acid-SAM', 'A-PDTON | PEDOT:PSS', 'T102', 'NO-Graphene-QDs', 'PTPD', 'Cu2MnSn4-np', 'PffBT4T-20D', 'M101', 'I2; LiI; Methoxypropionitrile', '3,6-di(2H-imidazol-2-ylidene)cyclohexa 1,4-diene-C12', 'TAE', '4-chlorothiophenol', 'X-DVTPD', 'HTM3', '1-(N,N-di-p-methoxyphenylamine)pyrene', 'MEH-PPV | Spiro-MeOTAD', 'pm-TPE-4DPA', 'MoOx', 'P3CT-N', 'TTh101', 'TP1', 'CZTS', 'CT2', 'PEH-1', 'KR355', 'Poly(3-bromothiophene)', 'rGO-4FPH | Spiro-MeOTAD', 'TZ3', 'PEDOT:PSS', 'FTA2', 'FrGO', 'CzPAF-TPA', 'MeO-DATPA', 'PEDOT:LS', 'MTA', 'PCPD2FBT:BCF', 'SO8', '4-F-br-4C', 'Polymer3', "3,3'-(2,7-bis(bis(4-methoxyphenyl)amino)-9H-fluorene-9,9-diyl)bis(N-ethyl-N,N- dimethylpropan-1-aminium) bis(trifluoromethanesulfonyl)imide", 'PTB7-th', 'BTT-5', 'CuInS2 | Al2O3-mp', 'BTF1', 'DCZ-OMeTAD', 'NiO-c | CuI', 'TPA-Pc', 'P', 'Spiro-MeOTAD-F', 'Cu:NiO', 'Graphene-QDs', 'F33', 'PCDTBT8', 'X59', 'NiO-c | BMIMBF4', 'rGO', 'P3DDT', 'X21', 'TPASBP', 'poly(DTSTPD-r-BThTPD)', 'PEDOT:PSS | Propionic acid', '1d @ triphenylamine modified azobenzene dyes', 'CuPc-Bu', 'MoS2; PEDOT:PSS', 'Al2O3-np | Spiro-MeOTAD', '3-Hexylthiophene | Spiro-MeOTAD', 'COTT-1', 'Graphene aerogel | Spiro-MeOTAD', 'PCP-Na', 'X19', 'IDF-SFXPh | MoO3', 'DPA-TPM', 'Co0.817Cu0.183O1.19', 'DBTMT', 'BTT-Me', 'Carbon-QDs', 'Polystyrene | Spiro-MeOTAD', 'PEDOT:MNSF', 'TOPO | TaTm', 'PEDOT:PSS | PEDOT:GSL', 'BTDTP', '9,9-bis(3-(dimethylamino)propyl)-N2,N2,N7,N7-tetrakis(4-methoxyphenyl)-9H-fluorene- 2,7-diamine', 'Spiro-MeOTAD; X60', 'CT1', 'PBDTT', 'PDO1', 'TTB1', 'JY7', '4,8-bis-(5-bromothiophene-2-yl)-benzo thiadiazole', 'TbT-3', 'BL51', 'NiO-c | N719 dye', 'Aniline | Spiro-MeOTAD', 'KM03', 'PF8-TAA', 'H66', 'ZnPcNO2-OBFPh', 'alfa-NPD', 'WO3-nw@PEDOT', 'NP-BC', 'COTT-2', 'PEG | Spiro-MeOTAD', 'PPDI | Spiro-MeOTAD', 'PFBT-Na', 'FePc-Cou', 'PVBT-SB', 'PAH 2', 'N1,N3,N5-tris(4-n-butylphenyl)-N1,N3,N5-triphenylbenzene-1,3,5-triamine', 'Au; NiO', 'IEICO; PBDTTT-E-T', 'HPB-Ome', 'V911', 'Bis-amide-TTF', 'Aminothiazolium iodide | P3HT', 'S5', 'NiO-c | NP-BC', 'TFB | P3HT', 'PEDOT:PSS | PTPAFSONa', 'M2', 'PVP | Spiro-MeOTAD', 'BTTP-CN', 'X36', 'TiS2-np', 'YK2', 'PEDOT:PSS | PEI', 'PEDOT:PSS | CrO3', 'ZnMe2Pc', 'NiO-c | PFN-P2', 'Poly(TA) | Spiro-MeOTAD', '2‐aminoterephthalic acid | Spiro-MeOTAD', 'Co0.39Cu0.61O', 'Ethyl acetate; I2; LiI; NMBI; Urea', 'H-Z2', 'MeO-2PACz', 'PEDOT:PSS; PEI | PEDOT:PSS', 'DTB', 'CuMePc', '3-hydroxypyridine | Spiro-MeOTAD', 'TPB(2-MeOTAD)', 'Mo(tfd-COCF3)3 | SWCNTs | Spiro-MeOTAD', 'Spiro-MeOTAD | Rubrene', 'CA-Br; TPA-PT-C6', 'PEDOT:PSS; V2O5', 'Z28', 'CISZ-QDs', 'PTB7', 'LiNiO-c', 'A101', 'PVBT-TMA', 'SAF-5', 'M117', 'PBDTTT-E-T | MoO3', 'PD-10-DTTE-7 | Spiro-MeOTAD', "4,4'-(1,3,4-Oxadiazole-2,5-diyl)bis(N,N-bis(4-methoxyphenyl)aniline)", 'WY-3', 'X55', 'BL25', 'pBDT-BODIPY | PFN', 'CuGaO2-np', 'NiO-np | PSS', 'pDPA-DBTP', 'BL52', 'CuCo2O4', 'PEDOT:PSS | PBDB-T:ITIC', 'HFB-OMeDPA', 'VOx | PEI', 'ATT-OHex', 'N-CuMe2Pc', 'PEDOT:PSS | Au-nw; VOx', 'CrO2', 'CoO', 'P5', 'M102', 'V1013', 'BX-OMeTAD', 'PEDOT:PSS; Graphene oxide', 'TAE4', 'TPADPP-2', '3-F-br-4C', 'MoS2-QDs; rGO-flakes | Spiro-MeOTAD', 'Polyacrylonitrile-grafted rGO', 'D101', 'V2Ox', 'TbT-2', 'PEDOT:PSS | NiO-c', 'IDT-TPA', 'NaYF4:Yb:Tm-np | Spiro-MeOTAD', 'PBDTTT-CT | Spiro-MeOTAD', 'CzPAF‐SBF', 'BTT(DPP-Th)3-EH', 'PTZ-TPA', 'PBDTT-FTTE', 'SGT-420', 'P2Z1', 'C6TBPH2', 'PMMA | DTPC8-ThTPA', 'PEDOT:PSS | X-QUPD', 'V1209', 'HBT-ZnPc', 'CuInS2@ZnS-QDs', 'PEDOT:PSS | Au-nanobipyramide; VOx', 'ZnPcNO2-OPh', 'Rubrene', 'tert-CuBuPc', 'PCBM-60', 'SM01', 'X51', 'P3HT; Spiro-MeOTAD', 'OMe-TATPyr', 'Azomethine', 'DCZ-OMeTPA', 'SWCNTs | P3HT', 'Co0.817Cu0.183O', 'Y3', 'N2,N,N8,N8-tetrakis[2,2-bis(4-methoxyphenyl)ethenyl]-4,10-dimethyl-6H,12H-5,11-methanodibenzo[b,f][1,5]diazocine2,8-diamine', 'PEDOT:PSS | AuAg@SiO2-np', 'PZn-TPA', 'TPD', 'Ph-TPA-4A', 'IDTT-TPA', 'TRUX1', 'Ta2O5 | Spiro-MeOTAD', '2mF-X59', 'YC-1', 'PMMA | Spiro-MeOTAD', 'LHTM-2', 'In2O3', 'TSHBC-CF3', 'PTAA | PFN', 'Spiro-TBB', 'HPPHT', 'Poly[4,8-bis(2-(4-(2-ethylhexyloxy)phenyl)-5-thienyl)benzo[1,2-b:4,5b’]dithiophene-alt-1,3-bis(4-octylthien-2-yl)-5-(2-ethylhexyl)thieno[3,4-c]pyrrole-4,6-dione', 'SFX-TPA', 'PEDOT; Graphene', '3,6-Cbz-EDOT', 'TPE-2,7-Carbazole W4', 'Choline chloride | Spiro-MeOTAD | SWCNTs', 'H-Z1', 'PEDOT:PSS | Dex-CB-MA', 'TTB2', 'COPV4', 'Cu0.8Cr0.2O2', 'Cz-OMeTAD', 'Cu3PS4-np', 'PDPPDBTE', 'NO2-PEIA | Spiro-MeOTAD', 'P3HT; SWCNTs; Spiro-MeOTAD', 'BL38', 'PEDOT:PSS | V2Ox', 'Co0.939Cu0.061O', 'Carbon-nt | PEDOT:PSS', 'DM2', 'CuI-np', 'PCz', 'H-3,4', 'TPA-ZnPc', 'TTz-1', 'Z1012', 'NH2-POSS | Spiro-MeOTAD', 'EP02', 'Triazine | Spiro-MeOTAD', 'Z7@MWCNTs', 'N3,N3,N9,N9‐tetrakis(4‐methoxyphenyl)xantheno[2,1,9,8‐klmna]xanthene‐3,9‐diamine', 'H65', 'ZnPc | Spiro-MeOTAD', 'AS44', '2FBTA-1', 'V886', 'PFB', 'IEICO; PBDB-T', 'NiO | CuSCN', 'PSQ1', 'M109', 'TRUX2', 'Spiro-MeOTAD | NaYF4', 'Poly-N-vinylcarbazole | Spiro-MeOTAD', 'Azu-Hex', 'p-DTS(FBTTh2)2', 'NiO-c | PCDTBT', 'TCP-OH', 'CuIn1.5Se3-QDs', '3,6-cbz-EDOT', 'SO9', 'DOR3T-TBDT; PCBM-70', 'TPD | HAT-CN', 'AZ2', 'CuPcNO2-OPh', 'TPAC-SAM', 'Yih-1', 'V885', 'NiS', 'IT-4F; PBDB-T-SF', 'CuO', 'HBZ-70', 'NiO-np | ME2', 'OCNR; PEDOT:PSS', 'M:ON', 'P1-2', 'WY-1', 'TZ1', 'pBBTa‐BDT2', 'mm-SFX-3PA', 'EHCz-2EtCz', 'VB-Me-FDPA', 'TPA1C', 'H6-ZnPc', 'Fluorene-dithiophene', 'quart-p-phenylene1', 'NiO-c | NiO-c', 'PDTSTTz', 'PDCBT | MoO3', 'Graphene oxide | PMMA', 'H-FL', 'none', 'BTF3', 'YC02', 'Carbon-nt; P3HT | PEDOT:PSS', 'Al2O3-c | Spiro-MeOTAD', 'PEDOT:PSS | PDPP-DTT', 'Spiro-MeOTAD | Al2O3', 'Tris(4-(5-hexylthiophen-2-yl)phenyl)amine', 'NBNDD', 'TAZ-[MeOTPA]2', 'TPB-2-MOTPA', 'N,N‐di‐p‐methylthiophenylamine', 'TPC', 'SrGO', 'PEH-8', '2PACz | MeO-2PACZ', 'PEDOT:PSS | PEI-HI', 'HL-1', 'Zn-Chl', 'POZ3', 'CZTS; rGO', 'PEDOT:PSS | Ethylene glycol', 'HT-ZnPc', 'LiMgNiO', 'BTTI-C8', 'X60(TFSI)2', 'CZTSe-QDs', 'RCP-BTT', 'NiO-c | YC-1', 'NiO-np | NaCl', 'PEO; KI; I2', 'rGO | PEDOT:PSS', 'KR133', 'Cz-N', 'CuI | Cu@CuI-nw', 'styryl-functionalized GO', 'H18', 'CuPc‐OTPAtBu', 'KR122', 'IDTC4-TPA', 'Vox | X-DVTPD', "N2,N2,N2',N2',N7,N7,N7',N7'-octakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2,2',7,7'-tetraamine", 'ZnPc-th-ZnPc', 'Tetrakis-Triphenylamine', 'CuPc-OBu', 'MeO-PPV | PFN-P2', 'J1', "5,5',5''-(5,5',5''-(nitrilotris(benzene-4,1-diyl))tris(furan-5,2-diyl))tris(2-octylisoindoline-1,3-dione)", "N2,N2,N7,N7-tetrakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2,7-diamine", 'ZPPHT', 'Y2A2', 'BL50', 'LiI; I2; Methoxyacetonitrile', 'TPP-OMeTAD', 'Cu(In0.5Ga0.5)S2-np', 'PBTTT', 'PEDOT:PSS | NiPcS4', 'NiMgLiO-c', 'S2', 'TTA', 'FT37', 'PASQ-IDT', '2D-PT', 'Graphene oxide | P3HT', 'PolyTPD | Al2O3-mp', 'PEDOT:PSS | PTPADCF3FSONa', 'CS03', 'MeO-PheDOT', '1c @ triphenylamine modified azobenzene dyes', 'XY1', 'mp-SFX-2PA', 'F4-TCNQ | CuPc', 'PEDOT:PSS | NaI', 'TTB-TTQ', 'Rubene | P3HT', 'TPA-CN', "4,4'-((2-Hexyl-2H-benzo[d][1,2,3]triazole-4,7-diyl)bis(thiophene5,2-diyl))bis(N,N-bis(4-methoxyphenyl)aniline)", 'M7-TFSI', 'DPA-ANT-DPA', 'CZTS-QDs', 'TAZ-[MeOTPATh]2', 'TPE-4DPA', 'LiCoO2', 'Graphene oxide; MoOx-np', 'TTA3', 'PTB9', 'iPrO-DATPA', 'PEH-2', 'EtO-DATPA', 'S,Si‐heteropentacene', 'XSln847', 'PFB | Al2O3-np', 'NiO-mp | Spiro-MeOTAD', 'PVCz-OMeDPD', 'MoO3 | TPTPA', 'CZTS0.75Se0.25; rGO', 'Graphene oxide | Spiro-MeOTAD', 'NiPcS4 | PEDOT:PSS', 'KR131', 'NiMgO | PVP', 'Adamantane | Spiro-MeOTAD', 'TAE3', 'PM-Spiro', 'PZn-3FTPA', 'TPBS', 'P3HT-MoS2', 'NiO-np | Graphene oxide', 'Al2O3 | CuPc(tBu)4', 'V1036:C2', 'Al2O3 | Sym-HTPcH', 'VO', 'Y4', 'Cu2O', 'GeO2', 'JY5', "N 4 ,N 4'-(4,10-dimethyl-6H,12H-5,11-methanodibenzo[b,f][1,5]diazocine-2,8-diyl)bis(N 4 ,N 4' ,N 4' - tris(4-methoxyphenyl)-[1,1'-biphenyl]-4,4'-diamine)", '2TPA-4-DP', 'Cu2O | PEDOT:PSS', 'TTBCPE', 'PEAI | Spiro-MeOTAD', 'CuO-nw | PEDOT:PSS', 'Cu0.5Cr0.5O2', 'Q216', 'M112', 'Co0.878Cu0.122O', 'Chl‐1', 'C8Br | Spiro-MeOTAD', 'Al2O3 | CuSCN', 'Zr(acac)4', 'THY-5', 'M104', '2,7-Ben', 'XMP', 'BTT-OMe', 'CuInSe2-QDs', '1,2-Bis[3,6-(4,4`-dimethoxydiphenylamino)-9H-carbazol-9-methyl]benzene', 'MC-43', 'CuPc(tBu)4', 'MoO3 | NPB', 'CW5', 'mGO | P3HT', 'PEDOT:PSS | PCDTBT', 'Cu2BaSnS4', 'CuFeO2-np', 'PPyra-ACD', 'P3CT-K | CuZnS', 'NiO-mp', 'P1Z2', 'X1', 'H-Tri', 'Hexakis[4-(N,N-di-p-methoxyphenylamino)phenyl]benzene', 'PMMA | DTPC13-ThTPA', 'CS02', 'BDTS-2DPP', 'XPP', 'CuMePy', 'CH3-PEIA | Spiro-MeOTAD', 'P:OO', 'Spiro-MeOTAD | Cu9S5-np', '2-(3,5-bis(5-(5-hexylthiophen-2-yl)thiophen-2-yl)thiophen-2-yl)-3,5-bis(5-(5-hexylthiophen-2-yl)thiophen-2-yl)thiophene', 'PZn-DPPA-O', 'MEH-PPV | PEDOT:PSS', '4-(5-(5-(5-(5-(5-hexylthiophen-2-yl) thiophen-2-yl) thiophene-2-yl) thiazolo[5,4-d]thiazol-2-yl) thiophene2-yl)-N,N-diphenyl-benzenamine', 'Graphene', 'V1091', '3,6-2,7-PCzTPA', 'PEDOT:PSS | Au@SiO2-np', "Poly[4,8-bis(2-(4-(2-ethylhexyloxy)3,5-fluorophenyl)-5-thienyl)benzo[1,2-b:4,5-b']dithiophenealt-1,3-bis(4-octylthien-2-yl)-5-(2-ethylhexyl)thieno[3,4-c]pyrrole-4,6-dione", 'FeS2', 'NiO | Spiro-MeOTAD', 'Ome-DPA-CuPc', 'PTPD | Spiro-MeOTAD', 'NiO-np | M2', 'Bi2Te3 | Spiro-MeOTAD', 'DPEDOT-B[BMPDP]2', 'Florinated polymer | Spiro-MeOTAD', 'DA-PEDOT:PSS', 'Poly(ethylene oxide)', 'BChl-3', 'H-Tetra', 'Graphene oxide | PTFTS', 'PbPc', 'benzo[1,2b:4,5b′]-dithiophene', 'SGT-410', 'P3HT | Carbon-nt', 'NiO-np | Glycerol', 'PEDOT:PSS | PTAA', 'HS-Ph-CN | Spiro-MeOTAD', 'Theobromine | Spiro-MeOTAD', 'NiMgLiO | 5-AVA', 'PTB7-TH; PFN', 'P6', '[BMMIm]Cl', 'NiO-np | CuSCN', 'M106', 'SM-1', 'VOx | APPA', 'PEDOT:PSS | PEG', 'CJ-01', 'sGO', 'C4Br | Spiro-MeOTAD', 'Graphene oxide | PEDOT:PSS', "(2Z,2'Z)-2,2'-((10-(2-ethylhexyl)-10H-phenothiazine-3,7-diyl) bis(4,1- phenylene)) bis(3-(4-(diphenylamino) phenyl) acrylonitrile", 'SWCNTs | Graphene oxide | PMMA', 'Z9', 'TAPC', 'V1102', 'Ph-TPA-6A', 'nPrO-DATPA', '2,5-bis (5-(5-(5-hexylthiophen-2-yl)thiophen-2-yl) furan-2-yl) thiazolo[5,4-d] thiazole', 'TBP', 'TSHBC-tBu', '(OctPhO)8ZnPc2', 'NiO-c | Al2O3-mp', 'TB(MA)', 'P2', 'ZnBChl', 'PEDOT:PSS | GeO2', 'PPN', 'OMeTP-SAM', 'PEO | Spiro-MeOTAD', 'PEDOT:PSS | PFI', 'TQ1d', 'NiO-np', 'V1021', 'CoOx', 'P3HT | SWCNTs | PMMA', '5,6,11,12-Tetraphenylnaphthacene', 'Spiro-MeOTAD | PEDOT:PSS', 'Black phosphorous nanosheets', 'MC6Cz-TPA', 'CsSnI3-QDs', 'CuPc | PEI', 'ETH44', 'O5H-OMeDPA', 'HTM-3', 'PTB7-Th | MoOx', 'r-GO-HBS', 'PHPT-py', 'TET', 'NO HTL', 'PTAA | PEDOT:PSS', 'NiO-c | PEDOT:PSS', '[Fe(bpyPY4)](OTf)2.5', 'Diketopyrrolopyrrole', 'F101', 'CPE-Na', 'ZnChl-2', '3-Dodecylthiophene | Spiro-MeOTAD', 'BL08', 'PBDB-T | Spiro-MeOTAD', 'CuGaO2-mp', 'CuInS | Spiro-MeOTAD', 'MEH-PPV', 'TAE1', 'CL1-2', 'PII2T8T', 'NiO-c | SY4', '3,8,13-tris[2,2-bis(4-methoxyphenyl)ethenyl]-5,10,15-triethyl-10,15-dihydro-5H-indolo-[3,2-a:3′,2′-c]carbazole', '5,7-bis(9-ethyl-9H-carbazol-3-yl)-2,3-dihydrothieno[3,4-b][1,4]dioxine', 'H-Z3', 'PZn-FTPA', 'Ni | NiMgO', 'CZTS0.25Se0.75; rGO', 'FA-MeOPh', 'PTEG', 'Me-QTPA', 'MeO-PPV', 'H112', 'MoOx | PEDOT:PSS', 'CzPAF‐SBFN', '4,4′-(9-Methyl-9H-naphtho[2,1-c]carbazole-2,12-diyl)bis(N,N-bis(4-methoxyphenyl)aniline)', 'NiO-np | M3', 'PEDOT:PSS | Carbon-nt', 'Au@SiO2-nw | Spiro-MeOTAD', 'Pyridine | Spiro-MeOTAD', 'PEDOT:PSS | PTPD', 'PARA1', 'PEDOT:PSS | PCP-Na', 'AgI-QDs', 'TTPA-BDT', 'M7-Br', 'PANI-PAMPSA', 'C13-FAS | Spiro-MeOTAD', 'B1', 'LD29', 'SM09', 'ZnPc-flu-ZnPc', 'BTT(DPP)3-C8', 'KR360', 'Si-PO-2CN', 'IEICO-4F | Spiro-MeOTAD', 'SWCNTs | PMMA', 'HfO2 | PTAA', 'BzTA', 'DR3TBDTT; PDMS', 'TPA-QA-TPA', 'Cu0.33Cr0.67O2', 'CuSCN | NPB', 'H1', 'P3HT | P3HT; PMMA', 'TPAC0M', 'CZ-TA', 'V2O5 | P3CT-K', 'KR353', 'Spiro-MeOTAD | CuI', 'X61', 'CuS-np', 'Z8', 'NiO-c | CuGaO2-mp', 'C12-carbazole', 'CA-Br', 'Carbon-nt; P3HT', 'ZnChl', 'PDPP-3T', 'pp-TPE-4DPA', 'Triazine-Ph-OMeTPA', 'TIPS-pentacene', 'V1000', 'NiO-np | ME1', 'PIDT-DFBT', 'PCT', 'NiPcS4', 'SiO2', 'CTAB | Spiro-MeOTAD', 'TPA-BP-TPA', 'PPyra-TXA', 'r-GO-BH', 'Crosslinked TCTA-BVP', 'Sym-HTPcH', 'VB-DAAF', 'Spiro-MeOTAD | V2O5', 'OMeTPA-TPA', 'TPA-OMeTPA', 'PCA-1', 'OMeTPA-FA', 'POZ10', 'CuInS2-QDs', 'PAF-86', 'DMFA-FA', 'TATCz3', 'Carbon; NiS', 'BDT2FMeDPA', 'c-TCTA', 'Spiro-MeOTAD-I', 'OMETPA-DPP', 'Z1', 'TFAP', 'NiO-np | PAS', 'Ethyl acetate; I2; LiI; TBP', "N3',N3',N6',N6'-tetrakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-3',6'-diamine", 'H4', 'Carbon-QDs | Spiro-MeOTAD', 'M4; PCBM-60', 'X23', 'ITIC', 'HMPDI', 'Au-np; P3HT', 'Graphene | Spiro-MeOTAD', 'TPA-BP-OXD', 'PEDOT:PSS | MoS2', 'c-OTPD', 'Porphyrin-H1', "2,2'-[(4,5-Bis(2-ethylhexyl)-dithieno[2,3-d:2',3'-d']thieno[3,2-b:4,5-b']dipyrrole-2,7-diyl)-bis(3-hexylthien-5,5'-diyl)bis(methane-1-yl-1-ylidine)]dimalononitrile", 'CuHePc', 'CBP', 'S9', 'ACR-TPA', 'PBTTTV-h', 'PDVT-10', 'SGT-422', 'ZnPc-p-ZnPc', 'CI-GO | PTAA', 'NiO-c | Cysteine', 'DMFA-TPA', 'Alkoxy-PTEG', 'TATF8HBP', 'Ph-OMeTPA', 'Y1', 'FTA1', 'VOx | Cu phtalocyanine', 'TBASBP', 'V1061', 'Spiro-029', 'Cu12Sb4S13', 'PTB-DCB21', 'n-CuBuPc', 'DAHI | Spiro-MeOTAD', 'HTM-1', 'CuAlO2; CuO', 'PEDOT:PSS | Au@poly(4-styrenesulfonate)', 'Au@CuZnSnSe-np', 'M110', 'TFDIB | Spiro-MeOTAD', 'NaYF4:Yb:Er-np | Spiro-MeOTAD', 'Bp-OMe', 'PEDOT:PSS | 5,6,11,12-Tetraphenylnaphthacene', 'BP-DC', 'CuS', 'P3', 'C8-BTBT', 'Ome-TPA-CuPc', 'NiO-c | SY3', 'asy-PBTBDT', 'HS-Ph-SCH3 | Spiro-MeOTAD', 'Poly-N-vinylcarbazole | SP-11', 'PB2T-SO', 'SFX-TPAM', 'BTT(DPP)3-EH', 'V862', 'T40P', 'Z1011', '1,3,6,8-tetrakis-(N,N-di-p-methoxyphenylamine)pyrene', 'MPA-BTI', 'DTP-C6Th', 'CS01', 'M1', 'Poly(2-ethyl-2-oxazoline); PEDOT:PSS', 'TPA4C', 'M103', 'TbT-1', 'MFGO', 'c-OTPD; TPACA', 's-PANI:PSS', 'WT3', 'A102', 'Spiro-MeOTAD | ODA-FeS2-np', 'EHCz-MeFl', 'LD22', 'ZnPc', 'BTT-4', 'M115', 'Triazine-InT', 'Carbon-nt; Graphene oxide', 'MC6Cz-9-NPC', 'XDB', 'WOx', 'YN2', 'Bifluo', 'PEDOT:PSS | TS-CuPc', 'FH-3', 'Graphene oxide | Carbon-np', 'IDT2', 'PMAA; Spiro-MeOTAD | Spiro-MeOTAD', 'X50', 'NiO-np | Choline chloride', 'DEPT-SC', 'Spiro-MeOTAD | MoO3 | CuPc', 'ZnPc(tBu)4', 'BTPA-3', 'H16', 'TTPA-DBQT', 'PEDOT:PSS | PbI2', 'NiO-np | Al2O3-mp', 'L-f', 'Me-BPZTPA', 'Oleic-acid | P3HT', 'SP-01', "2,2'-[(4,5-Bis(2-ethylhexyl)-dithieno[2,3-d:2',3'-d']thieno[3,2-b:4,5-b']dipyrrole-2,7-diyl)-bis(4,3'-dihexyl-2,2'-bithien-5,5'-diyl)bis(methane-1-yl-1-ylidine)]dimalononitrile", 'NiCo2O4', 'Py-C', 'T101', 'BEDN', 'SO7', 'NP-SC6-TiOPc', 'PTT | Spiro-MeOTAD', 'CdZnSe@ZnSe-QDs', 'Py-COF | PTAA', 'Li-TFSI; TBP', 'XSln1453', '2,8-bis-[2,2-bis(4-methoxyphenyl)ethenyl]-5,11-diethyl-5,11-dihidroindolo[3,2-b]carbazole', 'Polymer4', 'PolyTPD | PFN', 'SM13', '2,5‐bis(4,4′‐bis(methoxyphenyl)aminophen‐4′′‐yl)‐3,4‐ethylenedioxythiophene', 'Z3', '2-((2-(4-(2-ethylhexyl)-4H-dithieno[3,2-b:2′,3′-d]pyrrol-2-yl) thiazol-5-yl)methylene) malononitrile', 'H7', 'Theophylline | Spiro-MeOTAD', 'BPAPF', 'Acetonitrile; B2; LiBr', 'OMETPA-BDT', 'CF-BTz-ThR', 'SDTFCz2', 'Si-OMeTPA', 'EGO-PPV | PFN-P2', 'PANI-PAMSA', 'SP-12', 'Ethyl acetate; I2; LiI; TBP; Urea', 'TPTPA | MoO3', 'MC8-TPA', 'CZTS-np', '1,4-di(1H-imidazol-2-yl)benzene-C6', 'PbS-QDs | Spiro-MeOTAD', 'Spiro p-xylene', 'PEDOT:PSS | 4-bromobenzenediazonium tetrafluoroborate', '3,6-Cz-TPA', 'TPTPA | TPTPA; MoO3', 'CuInS2 | ZnS-QDs', 'V1050', 'tri-TPA', 'H6', 'CzP', 'CuPc', 'F23', 'CsBiBr3-QDs', 'NiO-np | Spiro-MeOTAD', 'Triazine-Flu', 'PMMA; rGO', 'TB4-ZnPc', '2,7-Bis(4,4′-dimethoxydiphenylamine)-9- (bis(methylsulfanyl)methylene)fluorene', 'YT2', 'PhNa-1T', 'Ag:CuO-nanofibers | PEDOT:PSS', 'Chl‐2', 'cyclopenta[2,1-b; 3,4-b′]dithiophene', 'COPV6', 'PTAA | TFPPy-ETTA', 'DBFMT', 'Pt-np', 'BDT0FMeDPA', 'PtMePy', 'H-Bi', '3,6-di(2H-imidazol-2-ylidene)cyclohexa 1,4-diene-C6', 'Z2', 'PDPP3T', 'NiO-c | Sn2O3-qd', 'MC8-9-NPC', 'PTAA-1F', 'HTM5', 'NiO-np | TPI-2MEO', 'Spiro-MeOTAD | CANP | Spiro-MeOTAD', 'br-4C', 'Carbozole @ S12', 'IDF-DiDPA | MoO3', 'TPE-S', 'CsPbBr3-np | Spiro-MeOTAD', 'Rubrene | PEDOT:PSS', 'HA2', 'V2Ox | PEDOT:PSS', 'TP-FTzF-TP', 'SFT-TPA', '3,6-Ben', 'NH-2,6', 'FU7', 'MoS2-QDs | Spiro-MeOTAD', 'PCDTBT', 'NiO-c | MoOx', 'Polymer2', 'NiO-c | n-Butylamine', 'MoO3 | PTAA', 'TOPO', 'HBZ-71', 'DFBT(DTS-FBTTh2)2', 'Cobalt–porphyrin', 'TPA-ANR-TPA', 'Spiro-OEtTAD', 'TTE-2', 'NiO-c | NiO-nw', 'MEH-PPV-20', 'PEH-3', 'CsSnBr3-QDs', '3,6-PCzTPA', 'NiO-c | Mercaptoethylamine chlorate', 'HPB-OMeDPA', 'PTh; Graphene', 'SCZF-5', 'Titanylphthalocyanine', 'NiMgLiO', 'BTTP', 'Spiro-TAD', '3-Butylthiophene | Spiro-MeOTAD', 'Graphene oxide; NiO-c', 'BDT-PTZ', 'PMA', 'Cu3SbS4-np', 'Ni-acetate', 'TcTa', 'Ph-TPA-2A', 'P1', 'D205 | Spiro-MeOTAD', 'PDPPT-TT', 'ATT-ODec', 'Spiro-CPDT', 'P3HT; PFN', 'CuPcNO2-OMFPh', 'BChl-1', 'Spiro-OPrTAD', 'Al2O3-c', 'Carbon', 'PDTSTTz-4', 'Spiro-MeOTAD | MWCNTs; Spiro-MeOTAD', '2,7-triphenylamine-carbazole', 'NiO-c | Ni', 'V859', 'DPBTD-B[BMPDP]2', 'CuSCN | rGO', '2,2′-[(4,5-Bis(1-octylnonyl)-dithieno[2,3-d:2′3′-d]thieno[3,2-b:4,5-b′]dipyrrole-2,7-diyl)bis(thien-5,5′-diyl)bis(methane-1-yl-1-ylidine)]dimalononitrile', 'F4-TCNQ', 'PyThTPA', 'PSS-g-PANI:PFI', 'Spiro-TBB | Spiro-TBB', 'PDMS', 'NiO-np | TPI-6MEO', 'Lignosulfonate; PEDOT:PSS; PDA', 'TAPbI3 | Spiro-MeOTAD', 'Spiro-MeOTAD | WO3', 'DPPZnP-TSEH; PCBM-60 | BCP', 'BI25', 'PANI', 'T80P', 'CDTh 1', 'Au-np; PEDOT:PSS', '4-(4-Phenyl-4-alfa-naphthylbutadienyl)-N,N-di(4-tolyl)-phenylamine', 'FBA3', 'Spiro-MeOTAD | WOx', 'LCS01', 'MeO-FDPA', 'TiO2-np | NiO-np', 'BTT-TPA', 'Cu0.67Cr0.33O2', 'P(VDF-TrFE) | Spiro-MeOTAD', 'NPB', 'TQ2', 'alkylammonium bromide | Spiro-MeOTAD', 'TPE-2,7-Carbazole W1', 'H64', 'BTTI-C6', 'Z30', 'CZTPA-2', 'PCBM-60 | bis-C60', 'TPASB', 'NiO-np | PTAA', 'KTM3', 'S197', 'Au-np; Graphene oxide', 'JY6', 'NiO-c | CuGaO2-c', 'TCPBr | Spiro-MeOTAD', 'PET-OMeDPA', 'AIGS-QDs', 'Polyrotaxane', 'PSQ2', 'KM05', 'CPE-K', 'NiO-np | ME3', 'Unknown', 'TPL', 'pTPA-DBTP', 'OAI | DM', 'NPB; PTAA', 'PEDOT:PSS | Al2O3-mp', 'PO-Spiro-OMeTAD', 'Carbon-nt | Spiro-MeOTAD', 'CON-16 | PEDOT:PSS', 'LHTM-1', 'CuGaO2', 'COPV3', 'PFO', 'CIGS', 'NiMgO-c', 'P3TAA', 'Carbon-nt', 'PffBT4T-2OD | WOx', 'C12H10B2O4 | Spiro-MeOTAD', 'NTPA', 'PBT', 'Li4Ti5O12 | Spiro-MeOTAD', '2,2′-[(4,5-Bis(1-octylnonyl)-dithieno[2,3-d:2′3′-d]thieno[3,2-b:4,5-b′]dipyrrole-2,7-diyl)bis(2,3-dihydrothieno[3,4-b][1,4]dioxin-5,5′-diyl)bis(methane-1-yl-1-ylidine)]dimalononitrile', 'CuP', 'YN3', 'ZnChl-4', 'Carbon-np; PEDOT:PSS', '2PACz', 'Spiro-MeOTAD | PbS', 'BTT-1', 'BTF4', 'PTAA; Spiro-MeOTAD', 'Y2', 'DNA', 'NiO-c | SDBS', 'PTZ2', 'NP2', 'IEICO; PBDTTT-E-T | MoO3', 'OIPC-I', '2-F-br-4C', 'T1', 'B3', 'B63', 'COPV7', 'PCDTBT1', 'PBDTT-SeDPP; PCBM-70', 'Graphene oxide', 'DMZ', 'PMMA', 'Yih-2', 'ZnNc', 'PEDOT:PSS | PFN-P1', 'TTE-1', 'PbS', 'DH-MeO-FDPA', 'TPP-SMeTAD', 'DTS', 'SGT-409', 'S,N-heteropentacene', 'Poly-N-vinylcarbazole | SP-12', 'Spiro-MeOTAD | SWCNTs', '1F-SAM | PEDOT:PSS', 'CPEPh-Na', 'KR216', 'TaTm | MoO3', 'SFXDAnCBZ', 'Carbon-nt | PMMA', 'PTAA | Spiro-MeOTAD', 'CuCrO2', 'NiO-c | N749', 'NiO-c | PhNa-1T', 'TQ1', 'PbS-QDs', 'DM', 'DIPO-Ph4', 'DTh101', 'PPyra-XA', 'IDF-TeDPA | MoO3', 'NiPc | V2O5', 'HTM1', 'NiO-c | NiO-mp', 'CuSeCN', 'PTQ10 | PTAA', 'Th101', 'PolyTPD', 'PBDTT-SeDPP', 'EDOT-MPH', '2,7-Cbz-EDOT', 'DR3TBDTT', 'V1221', 'BTT-3', 'P3HT; SWCNTs | Spiro-MeOTAD', 'Ni | NiMgO | PVP', '2F-SAM | PEDOT:PSS', 'BDT:TT', 'AQ | Spiro-MeOTAD', 'Poly(1,4-phenylenevinylene)', 'PCPDTBT', 'PEDOT:PSS | Al2O3-np', '2,7-Pyr', 'JK-216D', 'BTSe-1', 'S:DIB', 'KR321', 'Spiro-TTB', 'PEDOT:P(SS-co-TFPMA)', 'X62', '(BMPA-EDOT)3-TPA', 'TPA-BPFN-TPA', 'CoTh-TTPA', 'THY-4', 'Fu-OMeTPA', 'PEDOT:PSS-NH2-OH', 'VOx', 'TPADPP-1', 'M6', 'WO3-nw@PEDOT | PEDOT:PSS', 'Al2O3-c | PEDOT:PSS', 'HfO2 | Acetonitrile; I2; LiI; TBP', 'T60P', 'Graphene oxide | PFNBr', 'Spiro-MeOTAD | rGO', 'Al2O3-mp | MeO-DATPA', 'NiO-c | BBA', 'YT1', 'BAI | DM', '3F-SAM | PEDOT:PSS', 'P3HT | Al2O3-mp', 'SY1', 'Triazine-Th-OMeTPA', 'Co3O4', 'SGT-411', 'NiO-c | FDA', 'BDT-POZ', 'J61-ITIC', 'SWCNTs | Graphene oxide', 'Spiro-MeOTAD | MoO3', 'CuInSe2-QDss', 'P3OT', 'T(EDOT-TPA)2', 'H5', 'apv-T', 'Graphene | PEDOT:PSS', 'FB-OMeTPA', 'TT80', 'BDT-4MeOTPA', 'CGS', '2,4-spiro', 'TSHBC', 'TCP-OC8', 'YC04', 'SGT-404', 'P1Z1', 'Nafion; PEDOT:PSS', 'T5H-OMeDPA', '(n-BuO)4ZnPc', 'SBFCz2', 'Polymer1', 'M3; PCBM-60', 'CzPF', 'PEDOT:PSS | SrGO', 'n-octylammonium iodide | Spiro-mF', 'CuSCN-nw', 'Py | Spiro-MeOTAD', 'PTAA | MoS2', 'B186', 'P3HT; SWCNTs | PEDOT:PSS', 'Al2O3 | Spiro-MeOTAD', 'ZnO-nw | PEDOT:PSS', 'NiO-c | Mg(AcO)2', 'HTM', 'Au-nw | Spiro-MeOTAD', 'NiO | PS', 'NiO-c | EPA', 'MeO-TPD', 'PTB7-Th', 'CuI | PbPc', 'Cu0.2Cr0.8O2', '1‐adamantylamine | Spiro-MeOTAD', 'NiO-c | DEA', 'Imidazolium iodide | P3HT', 'Graphene; TSHBC  @ 5:1', 'TFM', 'YC01', 'PEDOT:PSS | NPB', 'BTPA-6', 'S,N-Heteroacene 2', 'Ag-np; PEDOT:PSS', 'COTT-1 | COTT-2', 'S7', 'WO3-nw', 'PdMe2Pc', 'PDCBT | WOx', 'PPDT2FBT', 'SWCNTs | Spiro-MeOTAD', 'CAS', 'Graphene; P3HT', 'Pentacene', 'Q219', 'Polythiophene', 'S,N-Heteroacene 1', 'CT3', 'MnS', 'PTAA | LiF', 'NiCoO', 'CuSCN | Spiro-MeOTAD', 'PDI', "Tetrakis(4-methoxyphenyl)spiro[cyclopenta[1,2-b:5,4-b']dipyridine-5,9'-fluorene]-2',7'-diamine", 'V1207', 'Dispiro-OMeTAD', 'Au-np | Spiro-MeOTAD', 'PEDOT', 'SnS', 'NiO-np | Choline chloride; Glycerol', 'BEDCE | Spiro-MeOTAD', 'PZn-TPA-O', 'PTAA | PFN-Br', 'PP-Spiro', 'CuO2', 'MoO3 | TaTm', 'Phenethylamine | Spiro-MeOTAD', 'CdSe-QDs | Spiro-MeOTAD', 'PEDOT:PSS | MoO3', 'p-PFP-O | PTAA', 'PBTI-C', 'PTAA; TPFB', 'F6-TCNNQ; TaTm', 'NiO-c | SY1', 'TZ2', 'mDPA-DBTP', 'Montmorillonite | Spiro-MeOTAD', 'PCA-2', 'InP-np | Spiro-MeOTAD', 'TPA', 'H2', 'IEICO | MoO3', 'CsCuBr3-QDs', 'PolyTDP', 'NiPc-Cou', 'IDT1', 'TBC-1', 'PCPDTBT | PEDOT:PSS', 'Q197', 'P3HT | PEDOT:PSS', 'F16CuPc | Spiro-MeOTAD', 'TPB-4-MOTPA', 'TATSFHBP', 'ZnPc-p-ZnPc 1', 'HL-2', 'Tetracene | Spiro-MeOTAD', 'XOP', '2,7-PCzTPA', 'SY4', 'TPDI', 'C60', 'BTTI-C12', 'Q205', 'YK1', 'PTB7:Th', 'ZnChl-1', 'NiO-c | PTAA', 'iDM1', 'THY-2', 'mp-SFX-3PA', 'CZ-STA; CZ-TA', 'AS37', 'TaTm', 'CuInS2 | Al2O3-np', 'P3HT | WOx', 'Azu-Oct', 'Imidazonium iodide | Spiro-MeOTAD', 'PEDOT; Spiro-MeOTAD', 'Al2O3-np | TPA-ZnPc', 'P3HT; SWCNTs-PhOMe', 'ADAHI', 'JW8', 'COPV5', 'NiO-c | PEAI', 'PTAA-2F', 'Cu2CoSn4-np', 'PBDT(2F)T', 'PEDOT:PSS | Black phosphorous QDs', 'PEDOT:PSS | VOx', 'BTT-2', 'Pentafluorobenzenethiol | Spiro-MeOTAD', 'TFB | Al2O3-np', 'CMO', 'NiCo2O4-np | Spiro-MeOTAD', 'TBC-2', '1,3,6-tris-(N,N-di-p-methoxyphenylamine)pyrene', 'CuSCN | Graphene', 'C12-silane-SAM | Spiro-MeOTAD', 'Theophylline | PTAA', 'ZnPy', 'MPA-BTTI', 'Z1013', 'Ni | Au', 'P8TTT', 'CuSCN', 'P3OFHT', 'NH-2,7', 'NiO-c | CuGaO2', 'NiO-c | PMMA', 'PANI:PSS', 'P3HT | MoO3', 'CuAlO2 | PEDOT:PSS', 'CuSCN | Ta:Wox-np', 'No HTM', 'TPD-4EtCz', 'L-H', 'PT-DC', 'PDQT', 'DNA-CTMA', '2EGO-PPV', 'Co0.695Cu0.305O', 'PDMS | CuSCN', 'PB2T-O', 'NiO-c', 'GO-nanoribbons', 'CMP', 'Azu-Me', 'Ag-nw; PEDOT:PSS', 'Ppy', 'YC06', 'TPD-4MeTPA', 'LGC-D013', 'NiO-np | KCl', 'M108', 'SiTP-OMeTPA', 'TBC-3', 'Cz-Pyr', 'PEDOT:PSS | PEDOT:PSS', '10-butyl-3,7-diphenylphenoxazine', 'HTM-P1', 'apv-EC', 'Grafted rGO; Polyacrylonitrile', '4C', 'CuI; CuSCN', 'P3HT; SWCNTs', 'JK-217D', 'NiO', 'ATT-OBu', 'DFH', 'NiO-c | SY2', 'HfO2 | CuI', 'DFTAB', '0F', 'TAT-t BuSty', 'NiO-c | PTZ-1', 'DOR3T-TBDT', 'SWCNTs | PEDOT:PSS', 'PABA | Spiro-MeOTAD', 'BTBDT', 'PTAA | PMMA', 'PTZ1', 'M3', 'Cu2ZnSn4-np', 'KR374', 'P4', 'SGT-421', 'Al2O3-mp | Spiro-MeOTAD', 'V841', 'CuCrO2-np', 'BTX-OMeTAD', "N1,N1',N1'',N1'''-(ethene-1,1,2,2-tetrayltetrakis(benzene-4,1-diyl))tetrakis(N1-(4-(dimethylamino)phenyl)-N4,N4-dimethylbenzene-1,4-diamine)", 'H-Ca', 'PTAA | Car-ETTA', 'TPA-BPV-TPA', 'MWCNTs; Spiro-MeOTAD', 'M107', 'PEDOT:PSS | Rubrene', 'H111', 'OTPA-ZnPc', 'PTPD | PFN', 'Azu-Bu', 'PEDOT:PSS | VB-DAAF', 'TRUX-E-T', 'DR3T', 'PEDOT:GSL', "Fused-F (Tris[[4-[3,3'-dihexylsilylene-2,2'-bithiophene]-7-[5′′-n-hexyl-(2,2′; 5′,2′′-terthiophen\ne)-5-yl]-benzo[c]-[1,2,5]thiadiazole]-2,6,10-yl]-4,4,8,8,12,12-hexamethyl-4H,8H,12\nHbenzo[1,9]quinolizino [3,4,5,6,7,-defg]acridine )", 'PVDF-HFP | Spiro-MeOTAD', 'PEAI | PTAA', 'TS-CuPc', 'CsOAc | Spiro-MeOTAD', 'Ph-TPA-8A', 'ATT-OMe', 'PEDOT:PSS | PCPDTBT', 'PFN; PTPD', 'HPDI', 'PNP-BC', 'Oleylamine | Spiro-MeOTAD', 'CuOx', 'Z26', 'TPB(2-TPTZ)', 'P3CT-CH3NH2', 'CuMe2Pc', 'G2', 'KM07', 'JW6', 'MoO3 | TPA-2,7-FLTPA-TPA', 'HS-Ph-NO2 | Spiro-MeOTAD', 'DM1P', 'N2,N2,N12,N12-Tetrakis(4-methoxyphenyl)-9-methyl-9H-naphtho[2,1-c]carbazole-2,12-diamine', 'r-GO-NH', 'Ph-inv-OMeTPA', 'KR145', 'ZnP', '3,6-triphenylamine-carbazole', 'Co-Porphyrin', 'D102', 'H-PheDOT', 'P-OR', 'Z33', 'Polypseudorotaxane', 'TDAB', 'Z29', 'TPA‐ANT‐TPA', 'SYN1', 'G1', 'HfO2 | Spiro-MeOTAD', 'V2O5 | PEDOT:PSS', 'CrO3', 'BV-FNPD', 'V1036:C4', 'Si-QDs | Spiro-MeOTAD', 'pDPP5T-2 | WOx', "(2Z,2'Z)-2,2'-(((2,4-dimethylphenyl) azanediyl) bis([1,1'-biphenyl]-4',4-diyl)) bis(3-(4-(diphenylamino) phenyl) acrylonitrile", 'EH44', 'H2Pc-1', 'DPIE', 'DAI | DM', 'AZ1', 'D103', 'N-CuMe2Pc; P3HT', 'Cu2NiSn4-np', 'TT-3,6-TPA', 'Asy-PBTBDT', 'PTAA | PPNBr', 'PTA', 'CuPrPc', 'HA1', 'TPAC2M', 'M:OO', 'Cu:Ni acetate', 'YN1', 'ACE-QA-ACE', 'Spiro-OiPrTAD', 'PDCBT | Ta-Wox', "tetra{4-[N,N-(4,4'-dimethoxydiphenylamino)]phenyl}ethene", 'Q222', 'DPP-Ome', 'P3CT-Na', 'PFN; TT', 'Spiro-MeOTAD | MoOx', 'CdSe-Qd | CsPbI3-QD | Spiro-MeOTAD', 'CoPcNO2-OPh', 'CW4', 'Selenium', '3,6 ´-BCz-OMeTAD', 'SAF‐OMe', '2TPA-2-DP', 'LiF | PEDOT:PSS', 'NiO-c | PTZ-2', 'DM1', 'DPIO', 'PST1', 'PBDTP-DTDPP', 'BTPA-4', 'CsSnI2.95F0.05', 'P3TAA-co-P3HT', 'IDTC6-TPA', 'PBDTTT-C', '2TPA-1-DP', 'NiO-c | Al2O3-mp; Au@SnO2-nw', "4,4'-(5,10,11-Trihexyl-10,11-dihydro-5H-thieno[2′,3':4,5]pyrrolo [3,2-g]thieno[3,2-b][1,2,3]triazolo[4,5-e]indole-2,8-diyl)bis(N,N-bis(4- methoxyphenyl)aniline)", 'FT55', 'BAI | Spiro-MeOTAD', 'Poly TPD-NPD', 'pentaerythritol tetrakis(3-mercaptopropionate) | Spiro-MeOTAD', 'C6Br | Spiro-MeOTAD', 'Spiro-MeOTAD | TS-CuPc', 'TT-2,5-TPA', 'DBC-OMeDPA', 'DPA-QA-DPA', 'PTB7-TH', 'Co(II)P', 'P1C1', 'MTDATA', 'CW3', 'V1225', 'FBA1', 'Spiro-MeOTAD', 'DTS(IIThThHEX)2', 'KR378', 'P3HT | WO3', 'EtheneTTPA', 'M105', 'CT4', 'VB-MeO-FDPA', 'THY-3', 'NiPc', 'MoS2 | Spiro-MeOTAD', 'pBBTa‐BDT1', 'Py-OMe', 'Cu:NiO-np', 'HTM-M1', 'CuSCN-2D', 'TT0', 'P3CT-Na | PASP', 'Spiro-MeOTAD | Cu2O', 'Graphene | AuCl3 | PEDOT:PSS', 'PTAA | CuSCN', 'rGO | PTAA', 'J2', 'DTPC8-ThDTPA', 'CF-Sp-BTh', 'TaTm | F6-TCNNQ; TaTm', 'PEDOT:PSS | Au@SiO2-nw', 'NiCo2O4-np', 'FH-0', 'FEH', 'IrTiOx-c', 'm-MTDATA', 'TPA-TPM', 'Spiro-OBuTAD', 'MoOx | Spiro-MeOTAD', 'Co(II)P; Co(III)P', 'H2Pc', 'Co(III)P', 'NiO-c | UiO-66', 'PQT-12', 'N,N-bis-[7-(4,4′-dimethoxydiphenylamine)-9- (bis(methylsulfanyl)methylene)fluoren-2-yl]-4-methoxyaniline', 'CJ-02', 'M111', 'PVK', 'C202', 'MoO3 | TPA-3,6-FLTPA-TPA', 'P(BDTT-tPPD)', 'PTB7 | WOx', 'DIB; SeS2', 'Polyacrylonitrile', 'rGO | CuSCN', 'TT1', 'PTAA | NiO-c', 'MEH; PPV', 'PTAA | CuGaO2-mp', 'SnS-np | NiO-np', 'PEDOT:PSS | PFN', 'CuPc | PTAA', 'PBTTT-14', 'V1160', 'B2', 'PTB8', 'CuI', "5,5',5''-(5,5',5''-(nitrilotris(benzene-4,1-diyl))tris(furan-5,2-diyl))tris(2-octylisoindoline-1,3-dione", '1,6-di{3-[2-(4- methylphenyl)vinyl]carbazol-9-yl}hexane', 'RCP', 'PCBM-60 | BCP', 'Azu-EH', 'CuPs-TIPS', 'InP-np', 'Hexamethyl-substituted subphthalocyanine', "Poly[4,8-bis(2-(4-(2-ethylhexyloxy)3-fluorophenyl)-5-thienyl)benzo[1,2-b:4,5-b'] dithiophenealt-1,3-bis(4-octylthien-2-yl)-5-(2-ethylhexyl)thieno[3,4-c]pyrrole-4,6-dione", 'HfO2 | Acetonitrile; I2; LiI; PMII; Propylene glycol; TBP', 'BPZTPA', 'PDO2', 'CrOx', 'Z35', 'F6-TCNNQ; TaTm | TaTm', 'ZnPor', 'TPA-NADT-TPA', 'Black phosphorous nanosheets | Spiro-MeOTAD', 'SP-02', 'PCBM-60 | Carbon', 'Spiro-N', 'PEDOT:PSS | TPD', '3-Ethylthiophene | Spiro-MeOTAD', 'Spiro-E', 'CuAlO2', 'PdMePy', 'NDT', 'HS-Ph-OCH3 | Spiro-MeOTAD', 'TPE-2,7-Carbazole W2', 'CsSnBrI2-QDs', 'PEDOT:PSS | VB-MeO-FDPA', 'TTA2', 'TPA2C', 'BDT-2D', 'Vox', 'YC-2', 'Spiro-s', 'PII2T8TSi', 'PEDOT:PSS | CuSCN', 'SrCl2 | Spiro-MeOTAD', 'V852', '2H-MoS2 | Spiro-MeOTAD', 'V866', 'ZnPc-DPP-ZnPc', 'PEA2PBI4', 'PTPAANT', 'FA-CN', 'Diazo-OMeTPA', 'EVA; SWCNTs | Spiro-MeOTAD', 'H3', 'Red Phosphorous-QDs', '3EGO-PPV', 'Pyrmidine | Spiro-MeOTAD', 'WY-2', 'NiO-c | MOF-808', 'FDT', 'POSS-NH2 | Spiro-MeOTAD', 'MEH-PPV; PFN', 'dly-1', 'Spiro-MeOTAD | VOx', 'COPV2', 'ODA-FeS2-np', 'PEDOT:PSS | Pyrene', 'F8T2e', 'P3HT; PCBM-60', 'SM', "N2',N2',N7',N7'-tetrakis(4-methoxyphenyl)spiro[fluorene-9,9'-xanthene]-2',7'-diamine", 'PPV', '1-Donecyl Mercaptan | Spiro-MeOTAD', 'X25', 'PEDOT:PSS | Ca', 'F22', 'DPPS | Spiro-MeOTAD', 'TPA-MeOPh', 'BT41', 'NiO-nanowalls | Diethanolamine', 'S101', 'P3TI', '3-Methylthiophene | Spiro-MeOTAD', 'H-Di', 'X26', 'NiO-np | Br-BPA-SAM', 'COPV1', 'MeO-BPZTPA', 'PEDOT:PSS | PTMA-BP', 'C5PcH2 | MoOx', 'BL07', 'Porphyrin', 'JW7', 'PEDOT:PSS | Na3C6H5O7', 'tetra-substituted azulene', 'X18', 'P-R', 'BDT-C1', 'Phosphor-QDs', 'PCBZANT', 'TPD-4MeOTPA', 'MoO3 | TPBi | TaTm', 'Bifluo-OMeTAD | MoO3', 'ACE‐ANT‐ACE', 'Cu2ZnSnS4', 'CIGGSe-np', 'DMF; I2; PVA; TBAI', 'PAH 1', 'NiO-c | Al2O3-mp; Au@SnO2-np', 'DORDTS–TFBT', 'PEDOT:PSS | Graphene oxide; PEG', 'PEDOT:PSS | Graphene oxide', 'TPA-TVT-TPA', 'I2-electrolyte', 'PDBD-T', 'WO3 | Spiro-MeOTAD', 'TPA-AZO', 'YT4', 'Cs-oleate | Spiro-MeOTAD', 'TPDCN', 'YD2-o-C8 | Spiro-MeOTAD', 'Porphyrin-H2', 'PEDOT:PSS | PolyTPD', 'Th-OMeTPA', 'PDCBT | Ta:WOx', 'MoO3 | PEDOT:PSS', 'PVAc', 'Ph-TPM', 'CuPc | Spiro-MeOTAD', 'CuEtPc', 'TTA1', '3,6-Pyr', 'CuGaO2-np | CuSCN', 'TPFPB | Spiro-MeOTAD', 'PhCz-4MeOTPA', 'EtheneDTPA', 'CDTh-EtHex 2', 'PZn-2FTPA', 'X22', 'NiO-c | PS', '2F', 'PS | Spiro-MeOTAD', 'C101', 'NiO-nanowalls'])))

    thickness_list = Quantity(
        type=str,
        shape=[],
        description="""
    A list of thicknesses of the individual layers in the stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous filed.
- State thicknesses in nm
- Every layer in the stack have a thickness. If it is unknown, state this as ‘nan’
- If there are uncertainties, state the best estimate, e.g write 100 and not 90-110
Example
200
nan |250
100 | 5 | 8
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '7.8', 'nan | 250.0', '2.5 | 30.0', '15.0 | 10.0', '89.0', '90.0', '2.0 | 50.0', '250.0 | 70.0', '100.0 | 10.0', '40.0 | nan', '325.0', '50.0', '220.0', '50.0 | 200.0', '137.0', '180.0', '45.0', '20.0 | 40.0', 'nan | 130.0', '167.0', 'nan | 33.0', '168.0', 'nan | 160.0', '803.0', '0.0', '10.0 | 6.0 | 1.0', '259.0', '10.0 | 160.0', '300.0 | 15.0', '15.0 | 150.0', '65.0 | 1400.0', '35.6', '60.0 | 10.0', '14.0 | 3.0', '300.0 | 5.0', '25.0 | 4.0', '95.0', 'nan | 44.0', '2.4', '5.0 | 2.0', '87.0', '10.0 | 90.0', '70.0', '41.0', '127.0', '193.0', '150.0 | 40.0', '11.0', '25.0 | 260.0', '5.7', '500.0', '5.0 | nan', '2000.0 | nan', '73.0', '184.0', '4.0 | nan', '76.0', 'True', '2000.0', '1.0', '54.0', '234.0', '10.0 | 5.0', '0.7 | nan', '260.0', '175.0', '267.0', '20.5', '2.0 | 400.0', '240.0', '50.0 | 50.0', '2.5 | 20.0', '9.0 | nan', '35.0 | 8.0', '600.0 | nan', '1.5 | 180.0', '330.0', '810.0', '21.0', '80.0 | 280.0', '26.0', '101.0', '809.0', '85.0', '250.0 | 40.0', '321.0', '30.0 | 40.0', '80.0 | 120.0', 'nan | 150.0', '9.0', '12.0 | 3.0', '5.0 | 40.0', '176.0', '136.0 | 5.0', '8.0', '10.0 | 7.0', 'nan | 8.0', 'nan | 3.0', '278.0', '13.0', '56.0', 'nan | 80.0', '35.0 | 10.0', 'nan | nan | nan', '650.0', '49.4', '20.0', '136.0', '20.0 | 10.0', '295.0', '52.0', '265.0', '165.0', '70.0 | nan', '40.4', '200.0 | nan', '82.0', 'nan | 6.5', '36.0', '6.5', '239.0', '14.0', '20.0 | 2.0', '1.5 | nan', 'nan | 100.0', '250.0', '2.0 | 60.0', '21.6 | nan', '10.0 | 10.0', '65.0', '9.2 | 10.0', '16.0 | 200.0', '150.0', '42.4 | nan', '100.0 | 7.0', '7.0 | 10.0', '47.35', '108.0', '9.0 | 200.0', '14.0 | 7.0', '110.0 | 7.0', '811.0', '25.0 | 15.0', 'nan | 140.0', '7.9', '12.0 | 7.0', 'nan | 20.0', 'nan | 150.3', '33.1', '15.0 | nan', '232.0', '2.5 | 40.0', '5.0 | 20.0', '10.0 | 2.0', '1.0 | 180.0', '33.5', '244.0', '50000.0', '40.0 | 10.0', '47.0', 'nan | 7.0', '35.0', '111.0', '750.0', '57.0', '135.0', '2.0 | nan', '805.0', '1.0 | nan', '30.0 | 10.0', '258.0', '33.0', 'nan | 2.0', '51.0', '2.5', '282.0', 'uknnown', '60.0 | nan', '807.0', '2.0 | 20.0', '160.0 | nan', '183.0', '80.0 | 350.0', '35.2 | nan', '20.0 | 20.0', '49.0', '200.0 | 10.0', '23.0', '59.0', '285.0', '90.0 | 80.0', '25.0', '2.8 | nan', '7.5', '40.0 | 350.0', '80.0 | 20.0', '40.0', '98.0', '35.0 | nan', 'nan | 150.1', '16.0 | 250.0', '230.0', '75.0', '20.0 | 5.0', '10.0 | nan', '37.5', '60.0 | 5.0', '40.0 | 200.0', '28.0', '480.0', '190.0', '50.0 | nan', 'nan | 180.0', 'nan | 300.0', '3.0 | nan', '8.0 | 4.0', '0.8 | 180.0', '15000.0', '10.0 | 325.0', '380.0', '700.0', '155.0', '211.0', '14.0 | 14.0', '22.0', '10.0 | 40.0', '145.0', '131.0', '63.0', '60.0 | 3.0', '34.0', '0.8', '1500.0', '37.0', '60.0 | 7.0', '280.0', '60.0', '5.0', '125.0', 'nan | 30.0', '7.0', '804.0', '160.0 | 8.0', '400.0', '143.0', '10.2', '25000.0', 'nan | 15.0', '71.0', '115.0', '2.0', '30.0 | 14.0', '20.0 | nan', '250.0 | 50.0', '130.0 | 10.0', '315.0', '4.0', '50.0 | 150.0', '253.0', '15.0 | 40.0', '34.4', '242.0', '40.9', '352.0', 'nan | 200.0', '12.0', '600.0', '300.0 | 8.0', '100.0', '105.0', '8.0 | nan', '485.0', '120.0 | 350.0', '252.0', '48.0', '96.0', '10.0 | 2.0 | 5.0', '200.0', '70.0 | 10.0', '150.0 | nan', '113.0', '80.0 | 8.0', '24.0 | 3.0', '1.2', '350.0', '290.0', '8.0 | 130.0', '3.0', '178.0', '10.0 | 30.0', '50.0 | 100.0', '10.0 | 120.0', '200.0 | 350.0', 'nan | nan', '48.3', '0.0 | 700.0', '806.0', '340.0', '35.0 | 210.0', '94.0', '181.0', '160.0', '38.0', 'nan | 15000.0', '100.0 | 8.0', 'nan | 50.0', 'nan | 14.0', '270.0', '170.0', '200.0 | 0.0', '7.0 | 115.0', '2.5 | 10.0', '15.0 | 45.0', 'nan | 170.0', '55.0', '0.4 | 180.0', '25.71', '7.0 | 2.0 | 10.0', '6.0', '80.0 | nan', '66.52', '20.0 | 100.0', '7.0 | nan', '60.0 | 1.0', '250.0 | 60.0', 'nan | 9.0', '200.0 | 700.0', '17.0', '44.0', '5.5 | 10.0', '293.0', '6.0 | 200.0', '12.0 | 200.0', '550.0', '15.0', '25.0 | 250.0', '215.0', '300.0', 'nan | 150.2', '245.0', '10.0 | 5.0 | 5.0', '185.0', '120.0', '180.0 | 160.0', '450.0', '18.0', '130.0', '153.0', '39.0', 'nan | 350.0', '110.0', '20.0 | 15.0', '310.0', '23.8', '808.0', '11000.0', '58.0', '345.0', '5.3', '3.0 | 10.0', '31.6', '1000.0', '25.0 | nan', '27.0', '188.0', '66.0', '225.0', '261.0', '46.2', '235.0', '60000.0', 'nan | 4.0', '802.0', '1200.0', '70.0 | 20.0', '390.0', '10.0', '43.0', '90.0 | 7.0', '0.2 | 180.0', '1.8 | 180.0', '30.0 | nan', '432.0', 'nan | 11.0', '173.0', '210.0', '397.0', '10.0 | 240.0', '2.5 | nan', '800.0', 'nan | 190.0', '198.0', '80.0 | 7.0', '271.0', '32.0', '5.0 | 10.0', '25.0 | 20.0', 'nan | 295.0', '163.0', '50.2', '24.0', '5.2', '40.0 | 20.0', '72.0', 'nan | 10.0', '99.0', '20.0 | 60.0', '200.0 | 5.0', '97.0', '196.0', '30.0', 'nan | 22.0', '18.3 | 10.0', '1.6', '2.0 | 180.0', '470.0', '680.0', '333.0', '35.0 | 348.0', '174.0', '158.0', '21.5', '24.0 | 7.0', '67.5', '35.0 | 475.0', '150.0 | 9.0', '116.0', '140.0', '100.0 | nan', '80.0', '801.0', 'nan | 5.0', '46.0', '320.0', '138.0', 'nan | 400.0', '45.6', '10.0 | 5.0 | 2.0', '70.0 | 7.0'])))

    additives_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    List of the dopants and additives that are in each layer of the HTL-stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous fields.
- If several dopants/additives, e.g. A and B, are present in one layer, list the dopants/additives in alphabetic order and separate them with semicolons, as in (A; B)
- If no dopants/additives, state that as “Undoped”
- If the doping situation is unknown, stat that as‘Unknown’
Example
Li-TFSI; TBP
FK209; Li-TFSI; TBP
F4-TCNQ
Undoped
Cu | Ag; Cu
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Li-TFSI; PEG; TBP', 'Co2f; Li-TFSI; TBP', 'FK209; Li-TFSI; Rutin-Ag-np; TBP', 'AMH', 'AgNO3', 'D102; Li-TFSI; TBP', 'Cu | Ag; Cu', '2,6-ludidine; Li-TFSI', 'Graphene Oxide', 'TBA-PF6', 'Ag-TFSI; Li-TFSI; TNP', 'Poly(ethylene glycol) tridecyl ether', 'Tb', 'Li(CF3SO2)2N; TBP', 'Co1f; Li-TFSI; TBP', 'NDP9', 'PSS-Na', 'TBP; Triphenylamine', 'B', 'Pb', 'DMPS', 'Zn(TFSI)2', 'Diethylene glycol', 'TBP | Unknown', 'rGO-PhOHex', 'EMIC', 'Unknown | T4-TCNQ', 'Carbon-nano onions', 'Pyridine', 'Y', 'Undoped | Undoped', 'Ag-TFSI; Li-TFSI; TBP', 'Unknown | FK209; Li-TFSI; TBP', 'Ni', 'Ag', 'LiN(SO2CF3)2; t-BtPy', 'Cu-2Cl; Li-TFSI; TBP', 'Mg', 'Pd-PVP', 'AgI; Li-TFSI; TBP', 'Co-TFSI; TBP', 'FK209; Li-TFSI; TBP; POM@Cu-BTC', '2-6-lutidin; Li-TFSI', 'Unknown | Zn', 'Cs', 'Silane', 'MoS2', 'Li-TFSI; TPB', 'NH3', 'CSA', 'Mo(tfd-COCF3)3', 'NiOx', 'Undoped | Li-TFSI; TBP', 'Unknown | Li-TSFI; TBP', 'Li-bis; Li-TFSI; TBP', 'TFMS | Li-TFSI; TBP', 'Urea', 'BF4; TBP; TEMPO', 'Chlorobenzol; Li-TFSI; TBP', 'EDTA', 'K102; Li-TFSI', 'PTAA', 'Ethanolamine', 'FK209; Li-TFSI; TBP; CuPc', 'PEO', 'Ti', 'Unknown | 2-Py; Li-TFSI', 'BuPyIm-TFSI', 'FK269; Li-TFSI; nan; TBP', 'Sr', 'FK211', 'TBP; Li-TFSI; Co(III) TFSI', 'KMnO4; Li-TFSI; TBP', 'f-SWCNTs', 'acetylacetone', 'PSSH', 'GeO2-np', 'Co(PyPyz)3[TFSI]3; Li-TFSI; TBP', 'JQ3; Li-TFSI; TBP', 'Li-TFSI; PMo11V; TBP', 'aYF4:Yb,Er@NaYF4; Li-TFSI; TBP', 'PVA', 'Li-TFSI; Polystyrene; TBP', 'LiN(CF3SO2)2N; TBP', 'Li@C60-TFSI; TBP', 'CuSCN', 'Li-TFSI; TBP | Undoped | Li-TFSI; TBP', 'Nd', 'NaLuF4:Yb,Er@NaLuF4; Li-TFSI; TBP', 'FK212', 'TiO2@MoO3-np', 'ethanolamine', 'LiNO3', 'FK209; FN-Br; TBP', 'WOx', 'CMP', 'F8BT', 'nan | DPITPFB', 'Co-TFSI; Li-TFSI; TBP', 'TMAH', 'PFI', 'MoO3-np', 'Graphene; Li-TFSI; TBP', 'Co-TFSI', 'acetylacetonate', 'Co-TPTB; Li-TFSI; TBP', 'FK209; Li-TFIS; TBP', 'Rb', 'TCNQ | Unknown', 'Li-TFSI; TBP; FK102(II)PF6', 'Li', 'Ag-TFSI; TBP', 'Ag; Li', 'Tetrafluoro-tetracyanoquinodimethane', 'Li-TFSI; TBP', 'PZ0.020', 'F6-TCNNQ', 'Er; Yb | Li-TFSI; TBP', 'FK102', 'FK209; H3BTC; Li-TFSI; TBP; nan', 'Eu', 'PFN-P2', 'Li-TFSI; TBP; TEMPO', 'D2; Li-TFSI; TBP', 'Graphdiyne', 'NaYF4:Yb,Er; Li-TFSI; TBP', 'CNT', 'Li-TFSI; Ni-nanobelts; TBP', 'Li-TFSI; TPBA', 'AgOTf-doped GO', 'Cu(bpcm)2; Li-TFSI; TBP', 'TPFB', 'PMPS', 'TS-CuPc', 'Cu | Unknown', 'Yb', 'Li-TFSI; TBP; FK102', 'F-graphene', 'SrCl2', 'FK209; Li-TFSI; TBP | N', 'CZTS-np', 'C3-SAM', 'D-TBP; Li-TFSI', 'WO3-np', 'Li-TFSI', 'Li-TFSI; PTAA; TBP', 'ox-Carbon-nano onions', 'Thiourea', '4-Py', 'TBFB', 'HCl', 'Li-TFSI; SpiroTFSI2; TBP', 'BCF', 'PFPPY', 'Li-bisLi-TFSI; TBP', 'Li; Co', 'FK209; 3PO4; Li-TFSI; TBP', 'NE', 'CrO3', 'D1; Li-TFSI; TBP', 'Undoped; Unknown', 'Sc3N@C80', 'DIO', 'Co(II)(dpzpyr)2; Li-TFSI; TBP', 'NaCl', 'Glycerol', 'Benzoyl peroxide', 'Glucose; Graphene oxide', 'EDA', 'PDA', 'Butylamine', 'Co(III)(pztbpy)3; Li-TFSI; TBP', 'Unknown | TPFB', 'Li-TFSI; MoS2; TBP', 'LAD', 'I2; Li-TFSI; TBP', 'H-TFSI; TBP', 'Zn(TFSI)2; TBP', 'nan | Graphene oxide', 'Ag-np', 'Cu; Li', 'Cu(Oac)2', 'F4-TCNQ | Undoped', 'Fe', 'Cu', 'Zn', 'EHCz-2EtCz-ox', 'DOPA', 'Unknown | Ta', 'GeO2', 'WOx | Unknown', 'Undoped', 'FK209; Li-TFSI; rGO; TBP', 'BMPyTFSI', 'TPACA', 'D-sorbitol', 'FK209; Li-TFS; TBP', 'Li-TFSI; TBP; V2O5', 'FK209; Li-TFSI', 'F4-TCNQ', 'LiClO4', '2-amylpyridine; Li-TFSI', 'Ca', 'Unknown | Undoped', 'Glucose', 'MoO3', 'PDMS', 'Li; Mg', 'Cu9S5-np', 'DOBD', 'Polydopamine', 'Carbon-nt-G; Li-TFSI; TBP', 'FK209; Li-TFSI; TBP | Er3+; Yb3+', 'Co-LTFSI; Li-LTFSI; TBP', 'Ce', 'Diphenyliodonium-hexafluorophosphat', 'PEG', 'Li-TFSI; Li(Gd, Y)F4-Yb; TBP', 'Li-TFSI; TBA', 'Li; Pb', 'NPB', 'Pd', 'IrCp*Cl(PyPyz)[TFSI]; Li-TFSI; TBP', 'Unknown | AgSbF6', 'F4-TCNQ | Unknown', 'Li-TFSI; TBP; FK209', 'Co3f; Li-TFSI; TBP', 'Unknown | Li', 'CuH; Li-TFSI; TBP', 'Unknown | FK269; Li-TFSI; TBP', 'Unknown', 'FK209', 'Undoped | Cu', 'ClO4-', 'FK210', 'Undoped | glacial acetic acid', 'CuI; Li-TFSI', 'EHCz-MeFl-ox', 'CI', 'FK102; Li-TFS', 'K', 'Zonyl FS-300', 'GSL', 'Cu | Cu', 'Cu-2Cl; Li-TFSI', 'FK209; LiNO3', 'Graphene', 'F2-TCNQ | Unknown', 'FK209; Li-TFSI; TBP', 'EDT', 'Co; Li-TFSI; TBP', 'Mo(tfd-CO2Me)3', 'FK209; Zn(TFSI)2; TBP', 'EHCz-3EtCz-ox', 'Li-TFSI; TBP | D-sorbitol', 'Ba', 'Carbon-nt; Li-TFSI; TBP', 'Co(III)(pztbpy)3; LiNTf2; TBP', '4-isopropyl-4′-methyldiphenyliodonium tetrakis(pentafluorophenyl)borate', 'Au-np; Li-TFSI; TBP', 'JQ1; Li-TFSI; TBP', 'La', 'In10-2,4,6; Li-TFSI; TBP', 'FK102; Li-TFSI; TBP', 'CuPc; Li-TFSI; TBP', '2-Py', 'Li-TFSI; O2; TBP', 'n-Butylamine', 'sGO', 'nan | Li-TFSI; TBP', 'CuI', 'Li-TFSI; TBP; TeCA', 'PMMA', 'P3HT | Unknown', 'Fe(ttb)(TFSI)3; Li-TFSI; TBP', 'Co-TFSI; Li-TFSI', 'N', 'Lithium acetate', 'TAPC', 'Unknown | MoO3', 'Co', 'Li-TFSI; TBP | Undoped', 'TPE-NM3; Mo(tfdCOCF3)3', 'JQ2; Li-TFSI; TBP', 'TBP', 'Unknown | TBP', 'FK102; TBP', 'Unknown | Et4N-TFSI; H-TFSI', 'D4; Li-TFSI; TBP', 'Unknown | JQ1; Li-TFSI; TBP', 'FK209; Li-TFSI; TBP | Undoped', 'CuAlO2', 'FK209; H2SO4; Li-TFSI; TBP', 'Li-TFSI; TBP | Undoped | Unknown', 'rGO', 'FK269; Li-TFSI; TBP', 'Unknown | VOx', 'HA', 'Unknown | Unknown | TBP', 'CF3PA; FK209; Li-TFSI; TBP', 'Li-TFSI; TBP | Cu2O', 'O2', 'Li-TFSI; Si NPs', 'FN-Br', 'CsI', 'TBA-BF4', 'N2', 'Ethylene glycol | Unknown', 'PolyTPD', 'Triethanolamine', 'Unknown | FK209; Li-TSFI; TBP', 'Carbon-QDs', 'Alanine', '2-Py; Li-TFSI', 'Spiro-(TFSI)2; TBP', 'AuAg-np', 'Au-np', 'Li-TFSI; TBP; acetonitrile', 'Br-BA', 'NPh2O2C2H6', 'Unknown | FK102; Li-TFSI; TBP', 'DPITPFB', 'Cu-Bix; Li-TFSI; TBP', 'Amonia | Unknown', 'SrCl2 | Undoped', 'Li-TFSI; TBP | Li-TFSI; TBP', 'VOx', 'Spiro-(TFSI)2', 'Mo(tfd-COCF3)3; TBP', 'H2O2', 'Triton-X', 'Oleylamine | Li-TFSI; TBP', 'FK209; TBP', 'Sodium Citrate', 'PS', 'CTAB', 'RbCl', '[In2(phen)3Cl6]CH3CN; Li-TFSI; TBP', 'BCF; Li-TFSI; TBP', 'Ox-SWCNTs', 'PCBTBT', 'SiO-np', 'Carbon-nt@G; Li-TFSI; TBP', 'Cu9S5@SiO2-np', 'FK209; Li-TFSI; Pb(NO3)2; TBP', 'PCDSA', 'Li-TFSI; TBP; MWCNTs', 'Graphydine-QDs; Li-TFSI; TBP; nan', 'Sb', 'FK102; Li-TFSI', 'Li-TFSI; TBP | Unknown', 'Li-TFSI; P4VP; TBP', 'Cu(TFSI)2', 'TEMPOL', 'FK209; Li-TFSI; Pb-MOFs; TBP', 'D3; Li-TFSI; TBP', 'PZ0.025', 'Li-TFSI; TBP; V2O5 | Undoped', 'Unknown | TBA', 'AcOH; FK209; Li-TFSI; TBP', 'F6-TCNNQ | Unknown', 'Cu-2Cl', 'DIO; PCDTBT', 'I2', 'DMC; F4-TCNQ', 'TPFPB', 'DPPS', 'PZ0.030', 'NaLuF4:Yb,Er; Li-TFSI; TBP', 'DMSO', 'FeCl3; Li-TFSI; TBP', 'D-TBP', 'Unknown | Cu', 'NiPcS4', 'rGO-PhBiTh', 'ETH44-ox', 'Unknown | TBP | Unknown', 'Unknown | Li-TFSI; TBP', 'GD-QDs; Li-TFSI; TBP', 'TBA-TFSI'])))

    additives_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the dopants/additives.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If more than one dopant/additive in the layer, e.g. A and B, separate the concentration for each dopant/additive with semicolons, as in (A; B)
- For each dopant/additive in the layer, state the concentration.
- The order of the dopants/additives must be the same as in the previous filed.
- For layers with no dopants/additives, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used.
- The preferred way to state the concentration of a dopant/additive is to refer to the amount in the final product, i.e. the material in the layer. When possible, use on the preferred units
o wt%, mol%, vol%, ppt, ppm, ppb
- When the concentration of the dopant/additive in the final product is unknown, but where the concentration of the dopant/additive in the solution is known, state that concentration instead. When possible, use on the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
5 vol%; nan | 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | 0.3 M
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.1 %; nan', '2 mg/ml; 520 mg/ml; 0.036 vol%', '520 mg/ml; 0.036 mL', '9 uL(520mg/mLACN); 15 uL', '6.76 mg/ml; 0.5 vol%', '1 %; nan', '1.79 vol%; 2.5 vol%', '0.90 vol%; 2.07 vol%; 3.60 vol%', '11.4 mg/ml; 36 µl/ml', '35 uL(260mg/1mLACN); 30 uL/mL', '35 uL(260mg/mLACN); 30 uL', '3 %', 'nan | 2 uL/mL', '2.85 vol%; 1.75 vol% | nan', '520 mg/ml; 2.88 vol%', '11.7 mg/ml; 36 µl/ml', '520 mg/ml; 0.0338 vol%', '0.064 M; 0.198 M', '0.6 %', '8.7 mg/ml; 8.7 mg/ml; 2.88 vol%', '0.064; 0.198 M; nan', '50 vol%', 'nan | 1 vol%', '0.29 vol%; 1.75 vol%; 2.88 vol%', '0.2', '17.5 uL(300mg/1mLACN); 28.8 uL', '3 mol%; 50 mol%; 330 mol%', '9.1 mg/ml; 30 µl/ml', '67 vol%', '30 mM; 30 mM', '0.6 vol%', '5 mg/ml; 520 mg/ml; 0.036 vol%', '8 %', '0.03 M; 0.5 M; 3.3 M', '7 %', '10 %; nan', '32 mM; 28.5 µl/ml', '33 vol%', '0.029 vol%; 28.3 mg/ml; 0.0288 vol%', '1.75 vol%; 2.85 vol%', '6.24 mg/ml; 8 µl/ml', '6 mol%', '0.6 mg/ml', '30 mM; 200 mM', '0.32 mg/ml', '10.4 mg/ml; 0.03 ml/ml', '0.1', '0.5 %; nan', 'nan; nan; 1 mol%', '520 mg/ml; 0.036 vol%', '9.1 mg/ml; 0.029 ml/ml', '9.1 mg/ml; 28.8 µl/ml', '1.7 vol%; 2.0 vol%', 'nan; nan; 2 mol%', '5.2 mg/ml; 0.02 ml/ml', '15.08 mg/ml; 9.1 mg/ml; 28.8 µl/ml', '2.88 vol%', '5 wt%; 1 wt%', '0.32 mg/ml | nan', '0.5 vol%; 6.76 mg/ml; 0.5 vol%', '1.5 %', '0.035 M; 0.231 M', 'nan; 0.8 wt%; nan', '5.3 mol%', '30 mol%; 80 mol%', '7 %; nan', '2 vol%; nan; nan', '10 mol%', '9.1 mg/ml; 0.028 ml/ml', '54 uL(10mg/mLACN); 11.2 uL', '1 mg/ml; 520 mg/ml; 0.036 vol%', '0.05 M; 0.5 M; 3.3 M', '4 wt%', '9.1 mg/ml; 0.03 ml/ml', '17.5 uL(520mg/mLACN); 28.8 uL', '17.5 uL(520mg/mLACN); 29 uL', '0.05', '0.5 mg/ml', 'nan; 0.6 wt%; nan', '45 uL(2mg/mLACN); 10.2 uL', '12 µl/ml', '2 wt%', '30 mM.200mM', '0.075', 'nan; nan; nan', '0.05 wt%', '40 uL(40mg/mLACN); 23 uL(520mg/mLACN); 40 uL', '520 mg/ml; 2.85 vol%', 'nan; 0.4 wt%; nan', '0.025', '300 mg/ml; 520 mg/ml; 0.028 vol%', '2 mM; 2.88 vol%', '0.15', '0.015', '500 mg/ml; 0.03 vol%', '1 vol%; nan; nan', '0.01 %; nan; nan', '2.45 mM; 40 mM; 270 mM', 'nan | 1.75 vol%; 2.88 vol%', '520 mg/ml; 0.0288 vol%', '40 mol%', '7.8 mol%', '6 wt%', '3.38 mg/ml; 22.5 µl/ml', '7.5 mg/ml; 7.65 mg/ml; 0.01 ml/ml', '10 mg/ml', '10 wt%', '32 mM; 195 mM', '520 mg/ml; 334 mol%', '3.0 vol%; 3.6 vol%', '1.8 mM; 30 mM; 200 mM', '2.88 vol%; 1.75 vol%', '12.3 mol%', '0.01 %; nan', '17.5 uL(520mg/mlACN); 28.8 uL', '20 mol%', '50 %; nan', '170 mg/ml; nan', '5.6 mg/ml; 30 mg/ml', '35 mM; 210 mM', '0.4 %', '4 mM; 30 mM; 200 mM | nan', '0.3 wt%', '0.007', '18 uL(1MACN); 29 uL(1MCB)', '2.2 mg/ml', '12 %', '1.44 vol%; 2.88 vol%', '0.5 mg/ml; 0.5 vol%', '8.7 mg/ml; 9.8 mg/ml; 0.029 ml/ml', '11 %', '1 %', '10 uL(300mg/mLACN); 17.5 uL(520mg/mLACN); 28.8 uL', '4 %', '1.5 mM; 14 mM; 56 mM', '4 mM; 30 mM; 200 mM', '3.0 wt%', '29 uL(300mg/mL); 18 uL(520mg/mLACN); 29 uL', '0.0018', '9 mM; 55 mM', '11.4 mg/ml; 0.036 ml/ml', 'nan | nan', '20 uL(517mg/1mLACN); 36 uL; 8 uL(375mg/mLACN)', '6 %', '5.4 mg/ml; 9.36 mg/ml; 0.028 ml/ml', '18 uL(520mg/1mLACN); 30 uL; 29 uL(300mg/MLACN)', '11.34 mg/ml; 0.0176 ml/ml', '2.0 mg/ml', '0.0056 M; 0.031 M; 0.19 M', '30 mol%', '520 mg/ml; 0.285 vol%', '20 uL(517mg/mLACN); 36 uL; 8 uL(376mg/mLACN)', 'nan; nan; 4 mol%', '0.44 M', '170 mg/ml; 0.5 vol%', '31.5 uL(300mg/mLACN); 17.5 uL(520mg/mL/ACN); 28.8 uL', '8.7 mg/m; 9.1 mg/ml; 0.029 ml/ml', '7.5 uL(170mg/mLACN); 4 uL', '8.7 mg/ml; 9.1 mg/ml; 28.8 µl/ml', '9.1 mg/ml; 28.8 µl/ml | 9.1 mg/ml; 28.8 µl/ml', '1.6 vol%; 2.1 vol%; 3.6 vol%', '0.0175 mL/mL; 0.0285 mL/mL', '10.1 mol%', '0.1 wt%', '22.5 uL; 15 uL', '0.5 vol%; nan; nan', '9 %', '0.0175 vol%; 0.0288 vol%', 'nan | 17.5 uL(520mg/mLACN); 28.8 uL', '7.65 mg/ml; 1 vol%', '1.75 vol%; 3.1 vol%', '35 uL(520mg/mLACN); 60 uL', '1.7 mg/ml; 7 µl/ml', '0.0035', '170 mg/ml; 0.004 vol%', '2 %', '1.75 vol%; 2.88 vol%', '0.82 mg/ml; 2 µl/ml', '3 %; nan', '9.1 mg/ml; 0.0288 ml/ml', '0.15 wt%', '23 uL(90.9mg/mLACN); 39 uL', 'nan; nan; 3 mol%', '2.5 %', '17.5 uL(520mg/mLACN); 28.5 uL', '18.2 mg/ml; 8 µl/ml', '1.0 wt%', 'nan; 0.2 wt%; nan', 'nan | 2 vol%', '520 mg/ml; 1.4 vol%', '5 mol%', '0.5 wt%', '0.019 M; 0.007 M; 0.2 wt% | nan', '8.8 mg/ml; 0.028 vol%; 0.035 vol%', 'nan | 20 vol%', '30.2 uL(1756mg/mLACN); 9.7 uL', '1.5 wt%', '11.44 mg/ml; 36 µl/ml', '0.5 mg/ml; 520 mg/ml; 0.036 vol%', '1 wt%', '70 uL(170mg/1mLACN); 20 uL', '5.0 mg/ml', '10 uL(520mg/mLACN); 28 uL; 35 uL(18.8mg/50mLACN)', '17.5 uL(520mg/mLACN); 28.8 uK', '1.5 mg/ml', '30 uL(270mg/mLACN); 35 uL', '3 wt%', '5 at%', '5 %', '10 uL(170mg/mLACN); 5 uL', 'nan | 5 vol%', '9.14 uL(0.25MACN); 21.02 uL(1.8MACN); 35.65 uL', '5 wt%', '8.67 mg/ml; 9.1 mg/ml; 28.8 µl/ml', '6.8 vol%; 3.4 vol%', '0.0052', '0.03; 3.3; 0.5', '12 mol%', '5 %; nan', '0.8 %', 'nan | 10 vol%', '520 mg/ml'])))

    deposition_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    The deposition procedures for the HTL-stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate them by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Thermal annealing is generally not considered as an individual reaction step. The philosophy behind this is that every deposition step has a thermal history, which is specified in a separate filed. In exceptional cases with thermal annealing procedures clearly disconnected from other procedures, state ‘Thermal annealing’ as a separate reaction step.
- Please read the instructions under “Perovskite. Deposition. Procedure” for descriptions and distinctions between common deposition procedures and how they should be labelled for consistency in the database.
Example
Spin-coating
Spin-coating | Spin-coating
Evaporation
Spray-pyrolys
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Spin-coating | Spin-coating >> Spin-coating', 'CBD | Dipp-coating', 'Spin-coating >> Spin-coating', 'Roller coating', 'Spin-coating >> Unknown', 'Spray-coating | Spray-coating', 'Electropolymerisation', 'Spin-coating | Spin-coating', 'Spray-pyrolys | Screen printing', 'Slot-die coating', 'Brush painting', 'Lamination | Spin-coating | Spin-coating', 'Spin-coating | Evaporation', 'Evaporation | Spin-coating', 'Ultrasonic spray', 'Sputtering', 'Spin-coating >> MeOH wash', 'Sputtering | Unknown', 'Spin-coating | Spray-coating | Spin-coating', 'Evaporation | Co-evaporation', 'CBD | Spin-coating', 'Doctor blading | Spin-coating', 'Dropcasting | Spin-coating', 'Spin-coating | Sputtering', 'Spin-coating | Unknown', 'Electrodeposition | Spin-coating', 'Evaporation | ALD', 'Magnetron sputtering >> Gas reaction', 'SILAR', 'Inkjet printing', 'Electrospinning | Spin-coating', 'Evaporation >> Oxidation', 'RF sputtering', 'Spin-coating >> Spin-coating >> Spin-coating', 'Spin-coating | Spin-coating >> Spin-coating >> Spin-coating', 'Evaoration | Spin-coating', 'Spin-coating | Lamination', 'Pulsed laser deposition | Sputtering', 'Evaporation | Evaporation', 'Spin-coating | RF sputtering', 'Electropolymerization', 'Spin-coating | Spin-coating | Lamination', 'Air brush spray', 'Spin-coating | Spray-coating', 'Spin-coating | Doctor blading', 'Sputtering | Sputtering', 'Sprinkling', 'Spray-coating | Spin-coating', 'Spin-coating | Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating', 'Spin-coating | Spin-coating >> Lamination', 'Spin-coating | Spray-pyrolys', 'Spin-coating | Dipp-coating', 'Hydrothermal | Spin-coating', 'Spin-coating >> Blowing hot air', 'Lamination | Spin-coating', 'Spin-coating | Spin-coating | Spin-coating', 'Sputtering | Sputtering | Spin-coating', 'Magnetron sputtering | Spin-coating', 'Lamination', 'Evaporation >> Gas reaction', 'E-beam evaporation | Spin-coating', 'Evaporation | Evaporation | Evaporation', 'PVD', 'Press-transfer | Spin-coating | Spin-coating', 'Dipp-coating >> Spin-drying | Spin-coating', 'Dipp-coating', 'Cryo-controlled quasi-congealing spin-coating', 'Slot-die coating | Evaporation', 'Evaporation', 'Electrospraying', 'Spray-pyrolys', 'Unknown', 'CBD', 'DC Magnetron Sputtering | Spin-coating', 'Drop-infiltration', 'ALD | Spin-coating', 'CVD', 'Screen printing', 'Slot-die coating | Spin-coating', 'Centrifuge-casting', 'Doctor blading', 'Spin-coating | E-beam evaporation', 'Magnetron sputtering', 'Substrate vibration assisted dropcasting', 'Dropcasting | Lamination', 'Sputtering | Spin-coating', 'DC Sputtering >> Oxidation', 'E-beam evaporation', 'Spin-coating | Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating', 'Unknown | Spin-coating', 'Spin-coating | Dropcasting', 'Spray-pyrolys | Dipp-coating', 'Dropcasting', 'Spin-coating >> Spin-coating >> Spin-coating >> Spin-coating', 'Pulsed laser deposition', 'DC Magnetron Sputtering', 'Evaporation >> Gas-reaction', 'Blow-drying', 'Hydrothermal', 'Electrodeposition', 'E-beam evaporation | E-beam evaporation', 'Spray-pyrolys | Spin-coating', 'Dipp-coating | Spin-coating', 'Evaporation >> Polymerisation', 'RF magnetron sputtering', 'Dipp-coating | Evaporation', 'Slot-die coating | Slot-die coating', 'Anti-solvent quenching | Spin-coating', 'Gelation', 'Spin-coating', 'Spin-coating | ALD', 'Doctor blading | Doctor blading', 'Spray-coating', 'Spin-coating | Evaporation | Evaporation', 'ALD', 'Co-evaporation | Evaporation', 'Spin-coating | Drop-infiltration', 'Reactive magnetron sputtering', 'Electrospraying | Spin-coating', 'Hydrolys | Drop-infiltration'])))

    deposition_aggregation_state_of_reactants = Quantity(
        type=str,
        shape=[],
        description="""
    The physical state of the reactants.
- The three basic categories are Solid/Liquid/Gas
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the aggregation state associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Most cases are clear cut, e.g. spin-coating involves species in solution and evaporation involves species in gas phase. For less clear-cut cases, consider where the reaction really is happening as in:
o For a spray-coating procedure, it is droplets of liquid that enters the substrate (thus a liquid phase reaction)
o For sputtering and thermal evaporation, it is species in gas phase that reaches the substrate (thus a gas phase reaction)
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Liquid
Gas | Liquid
Liquid | Liquid >> Liquid
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Solid', 'Liquid', 'Liquid | Liquid', 'Unknown', 'Liquid | Liquid | Liquid', 'Liquid | Liquid >> Liquid >> Liquid >> Liquid >> Liquid', 'Liquid >> Liquid', 'Gas', 'Gas >> Gas', 'Liquid | Liquid >> Liquid >> Liquid >> Liquid', 'Liquid | Liquid >> Liquid >> Liquid', 'Gas | Liquid', 'Gas | Gas', 'Liquid | Gas'])))

    deposition_synthesis_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The synthesis atmosphere.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Vacuum | N2
Air | Ar; H2O >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['N2 | Vacuum', 'Vacuum >> Air', 'N2', 'Vacuum | Ar', 'Unknown', 'Air | N2', 'N2 >> methanol', 'N2 >> N2', 'N2 | N2', 'Ar | Ar', 'Air | Vacuum', 'Ambient', 'Air | Air', 'Dry air', 'Vacuum | Vacuum', 'Air', 'Vacuum', 'Ar; O2', 'Ar'])))

    deposition_synthesis_atmosphere_pressure_total = Quantity(
        type=str,
        shape=[],
        description="""
    The total gas pressure during each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- Pressures can be stated in different units suited for different situations. Therefore, specify the unit. The preferred units are:
o atm, bar, mbar, mmHg, Pa, torr, psi
- If a pressure is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 100 pa and not 80-120 pa.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 1 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.006 Torr', '0.002 Pa', '0.0001 Pa | nan', '1 atm | 0.00004 Torr', '0.000001 mbar | 0.000001 mbar', '1 atm | 0.00003 mbar', '1 atm | 1 atm', '6 Pa', '1 atm | 0.0001 Pa', '1 atm | 0.000001 atm', '1 atm | 0.000002 Torr', '0.0001 Pa', '1 atm', '4 Pa', '1 atm | 0.000001 Torr', ' N2', '2 Pa'])))

    deposition_synthesis_atmosphere_pressure_partial = Quantity(
        type=str,
        shape=[],
        description="""
    The partial pressures for the gases present during each reaction step.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the partial pressures and separate them with semicolons, as in (A; B). The list of partial pressures must line up with the gases they describe.
- In cases where no gas mixtures are used, this field will be the same as the previous filed.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 0.99 atm; 0.01 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1 atm | 0.00004 Torr', '0.000001 mbar | 0.000001 mbar', '1 atm | 0.00003 mbar', '0.003 Torr; 0.003 Torr', '1 atm | 1 atm', '1 atm | 0.000001 atm', '1 atm | 0.000002 Torr', '1 atm', '1 atn', '1 atm | 0.000001 Torr', '1 atm | 0.0001 Pa'])))

    deposition_synthesis_atmosphere_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relative humidity during each deposition step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the relative humidity associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns
- If the relative humidity for a step is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 35 and not 30-40.
Example
35
0 | 20
nan >> 25 | 0
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '30 | 30', '0.9', '25', '15', '30.0', '90.0', '20.0', '30', '0.35', '30 | 0', '50.0', '65.0', '10.0', '80.0', '0 | 0'])))

    deposition_solvents = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in each deposition procedure for each layer in the stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvents associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the solvents in alphabetic order and separate them with semicolons, as in (A; B)
- The number and order of layers and deposition steps must line up with the previous columns.
- For non-liquid processes with no solvents, state the solvent as ‘none’
- If the solvent is not known, state this as ‘Unknown’
- Use common abbreviations when appropriate but spell it out when risk for confusion
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Chlorobenzene
Acetonitile; Ethanol | Chlorobenzene
none >> Ethanol; Methanol; H2O | DMF; DMSO
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Ethanol; Water', 'Chloroform', 'Ethylene glycol', '2-methoxyethanol', 'Etyl cellulose; Terpineol', 'Diethyl sulfide', 'Dichlorobenzene', 'Unknown | Chlorobenzene', 'acetonitrile; Chlorobenzene', '2-methoxyethanol; monoethanolamine', 'IPA | Chlorobenzene', 'Unknown', 'Chlorobenzene | none', 'Ethanol | TMAOH solution', 'Chlorobenzene | Unknown', 'Toluene | Methanol', 'none', '2-methoxyethanol; ethanolamine | none', 'acetonitrile', 'Methanol', 'IPA; Water', 'Water | Chlorobenzene', 'Water; Methanol', 'IPA | Diethyl sulfide', 'Toluene | DMF', 'DMF', 'Water | 2-metoxyethanol', '1,2-dichlorobenzene', 'Chlorobenzene >> 2-Butanol', 'IPA | IPA', 'IPA | Unknown', 'Chlorobenzene', 'Chlorobenzene; acetonitrile', 'Hexanethiol', 'Ethyl acetate', '2-methoxyethanol; ethanolamine', 'Water; IPA', 'DMF | Chlorobenzene', 'Toluene', 'Water', 'Chlorobenzene | Chlorobenzene', 'Ethanol', 'none | 2-metoxyethanol', 'Ethylene glycol | Methanol', 'Water | IPA', 'none | Chlorobenzene', 'Water | Water', 'IPA', 'none | none', 'Hexane | Unknown', 'Chlorobenzene | IPA'])))

    deposition_solvents_mixing_ratios = Quantity(
        type=str,
        shape=[],
        description="""
    The mixing ratios for mixed solvents
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent mixing ratios associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- For pure solvents, state the mixing ratio as 1
- For non-solvent processes, state the mixing ratio as 1
- For unknown mixing ratios, state the mixing ratio as ‘nan’
- For solvent mixtures, i.e. A and B, state the mixing ratios by using semicolons, as in (VA; VB)
- The preferred metrics is the volume ratios. If that is not available, mass or mol ratios can be used instead, but it the analysis the mixing ratios will be assumed to be based on volumes.
Example
1
4; 1 | 1
1 >> 5; 2; 0.3 | 2; 1
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1; 8', '0.1; 51', '1 >> 1', '1; 0.006 | nan', '1 | nan', 'nan | 1', '1', '1; 0.006', '5; 1', '1 | 1', '1; 0.012', '1; 1', '1; 0.1', '1; 3'])))

    deposition_solvents_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of all the solvents.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For non-liquid processes with no solvents, mark the supplier as ‘none’
- If the supplier for a solvent is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Sigma Aldrich
Sigma Aldrich; Fisher | Acros
none >> Sigma Aldrich; Sigma Aldrich | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown | Sigma Aldrich', 'Fisher Chemical | ACROS Organic', 'Unknown', 'Fisher Scientific', 'Heraeus', 'Guangzhou Seaside Technology', 'Nacalai Tesque; Wako Pure Chemical', 'Aladdin', 'Unknown; Sigma Aldrich', 'J&K', 'Sigma Aldrich; Sigma Aldrich', 'Sigma Aldrich', "Xi'an Polymer Light Technology; Xi'an Polymer Light Technology"])))

    deposition_solvents_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the solvents used.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For non-liquid processes with no solvents, state the purity as ‘none’
- If the purity for a solvent is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
Puris; Puris| Tecnical
none >> Pro analysis; Pro analysis | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['anhydrous; 99%', 'Pro analysis', 'Unknown', 'Puris; Puris', '0.998', '99.8%; 99.8%', 'Puris'])))

    deposition_reaction_solutions_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    The non-solvent precursor chemicals used in each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemicals associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several compounds, e.g. A and B, list the associated compounds in alphabetic order and separate them with semicolons, as in (A; B)
- Note that also dopants/additives should be included
- When several precursor solutions are made and mixed before the reaction step, it is the properties of the final mixture used in the reaction we here describe.
- The number and order of layers and reaction steps must line up with the previous columns.
- For gas phase reactions, state the reaction gases as if they were in solution.
- For solid-state reactions, state the compounds as if they were in solution.
- For reaction steps involving only pure solvents, state this as ‘none’
- If the compounds for a deposition step is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
FK209; Li-TFSI; Spiro-MeOTAD; TBP
NiO-np
PTAA | CuSCN
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'NiPc | Vanadium(V)\noxytriisopropoxide; IPA', 'FK209; Li-TFSI; CS03; TBP', 'Spiro-oF; Li-TFSI; Co-TFSI; TBP', 'FK209; Li-TFSI; CS01; TBP', '(NiAc)4H2O', 'Spiro-MeOTAD; nan', 'Ni(Ac)2·4H2O; SrCl2', 'P3CT-Na', 'Spiro-MeOTAD | MoOx', 'Clevios PVP Al 4083; Black phosphorous QDs', 'Li-TFSI; Spiro-MeOTAD', 'Ni; O2', 'NiO-np', 'Spiro-MeOTAD; Co(III)(pztbpy)3; Li-TFSI; TBP', 'P3HT; rGO-PhBiTh', 'NiO-np | PAS', 'FK209; Li-TFSI; LCS01; TBP', 'NaYF4:Yb,Er; Li-TFSI; TBP; PTAA', 'CuPc | PEI', 'Li-TFSI; H-Z3; TBP', 'FK209; Li-TFSI; H-Lin; TBP', 'Oleylamine | Li-TFSI; Spiro-MeOTAD; TBP', 'TRUX-E-T; Li-TFSI; TBP', 'PEDOT:PSS | TPA-NPA-TPA', 'P3HT; rGO-PhOHex', 'PTAA; Li-TFSI; TBP', 'aYF4:Yb,Er@NaYF4; Li-TFSI; TBP; PTAA', 'PTB7-Th | MoOx', 'PEDOT:PSS | MoO3', 'DIPO-Ph4', 'Nickel acetate tetrahydrate; ethanolamine', 'Spiro-MeOTAD; Li; Co', 'NiO Target', 'P3CT', 'Li-TFSI; NiPc; TBP', 'S; Oleylamine; 1-octadecane; diphenylphosphine; Indium acetate; CuI', 'Spiro-MeOTAD; Li-TFSI; TBP | MoO3', 'IEICO | MoO3', 'MoO3', 'FK209; Li-TFSI; B186; TBP', 'P3HT', 'H-Lin', 'PBTTTV-h', 'CuSO4; Lactic Acid; NaOH', 'Clevios PVP Al 4083', 'rGO-4FPH | Spiro-MeOTAD; Li-TFSI; TBP', 'Rubrene', 'M2; Li-TFSI; TBP', 'CF-BTz-ThR', 'Al(C2H5)3 | nan', 'MEH-PPV', 'Graphene oxide | Carbon dots', 'NiOx-np solution', 'Li-TFSI; H-Z2; TBP', 'Li-TFSI; Spiro-MeOTAD; TBP; Co-TFSI', 'Nickel acetate hexahydrate; ethanolamine | ethylphosphonic acid', 'NaLuF4:Yb,Er@NaLuF4; Li-TFSI; TBP; PTAA', 'Clevios PVP Al 4083 | PEI', 'Spiro-MeOTAD', 'Nickel Nitrate hexahydrate', 'CZ-TA; Li-TFSI; TBP', 'FK102; Li-TFSI; Spiro-MeOTAD; TBP', 'M1; Li-TFSI; TBP', 'Spiro-MeOTAD; Li-TFSI; TBP', 'Copper thiocyanate', 'Graphene oxide | PFNBr', 'Graphene oxide | PTAA', 'NiO-np | PTAA', 'CZTS-np', 'Li-TFSI; H-Z1; TBP', 'Spiro-MeOTAD; Li-TFSI; TBP | MoOx', 'Co(PyPz)3(TFSI)3; Li-TFSI; Spiro-MeOTAD; TBP', 'Li-TFSI; MWCNTs; Spiro-MeOTAD; TBP', 'PTAA | PFN', 'Nickel Chloride hexahydrate; HNO3', 'C102; FK209; Li-TFSI; TBP', 'Nickel acetate hexahydrate; ethanolamine', 'InP-np | Li-TFSI; Spiro-MeOTAD; TBP', 'C13-FAS | Spiro-MeOTAD', 'Li-TFSI; PTAA; TBP', 'TaTm | F6-TCNNQ; TaTm', 'FK209; iDM1; Li-TFSI; TBP', 'P3HT; Li-TFSI; TBP', 'H-Star', 'polyacrylonitrile; rGO', 'Ni(CH3COO)2·4H2O', 'Vanadium(V)\noxytriisopropoxide; IPA', 'PTAA', 'PBDT(T)(2F)T', 'CZTS-np; hexanethiol', 'FK209; Li-TFSI; Spiro-MeOTAD; TBP', 'PBDT(2F)T', 'PEDOT:PSS; PEG', 'Li-TFSI; Spiro-MeOTAD; TBP', 'Spiro-MeOTAD; Li-TFSI; Co-TFSI; TBP', 'FK209; Li-TFSI; H-Star; TBP', 'FK209; Li-TFSI; Spiro-MeOTAD; TBP | Vanadium(V)\noxytriisopropoxide; IPA', 'PBDT(2H)T', 'Ni(Ac)2·4H2O; SrCl2 | nan', '2PACz', 'B186', 'Li-TFSI; Spiro-MeOTAD; TBP; V2O3 | Clevios PVP Al 4083', 'Li-TFSI; P3HT; TBP', 'Carbon Paste', 'DTP-C6Th', 'Graphene oxide | PTFTS', 'Spiro-MeOTAD; TBP', 'IEICO; PBDTTT-E-T | MoO3', 'MeO-2PACz', 'nickel acetate tetrahydrate', 'PTAA >> Sb 2D-nanosheets', 'TPE-S', 'Rubrene | PEDOT:PSS', 'nickel acetylacetonate', 'HA2', 'Cu >> iodine', 'Nickel acetate tetrahydrate', 'Spiro-MeOTAD; Co(III)(pztbpy)3; LiNTf2; TBP', 'PEDOT:PSS', 'Graphene oxide', 'Li-TFSI; BTPA-3; TBP', 'HA1', 'Nickel acetate hexahydrate; ethanolamine | 4-Bromobenzoic acid', 'Li-TFSI; TBP; Spiro-MeOTAD; acetonitrile', 'Li-TFSI; Spiro-MeOTAD; TBP | Li-TFSI; MWCNTs; Spiro-MeOTAD; TBP', 'Cu-np; NiOx-np', 'C101; FK209; Li-TFSI; TBP', 'InP-np', 'Oleylamine', 'FK209; Li-TFSI; EP02; TBP', 'PBDTTT-E-T | MoO3', 'nickel (II) acetate tetrahydrate', 'ethanolamine; nickel acetate tetrahydrate', 'Nickel acetate; ethylene glycol; ethylenediamine', 'pentaerythritol tetrakis(3-mercaptopropionate) | Spiro-MeOTAD; Li-TFSI; TBP', 'PTAA | PFN-P2'])))

    deposition_reaction_solutions_compounds_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemical suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For gas phase reactions, state the suppliers for the gases or the targets/evaporation sources that are evaporated/sputtered/etc.
- For solid state reactions, state the suppliers for the compounds in the same way.
- For reaction steps involving only pure solvents, state the supplier as ‘none’ (as that that is entered in a separate filed)
- For chemicals that are lab made, state that as “Lab made” or “Lab made (name of lab)”
- If the supplier for a compound is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Dysole; Sigma Aldrich; Dyenamo; Sigma Aldrich
Sigma Aldrich; Fisher | Acros
Lab made (EPFL) | Sigma Aldrich >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Heraeus | synthesized', 'Energy Chemical; Sigma Aldrich; Sigma Aldrich; Sigma Aldrich', 'Xi’an p-OLED; Sigma Adrich; Xi’an p-OLED; Sigma Adrich', 'Sigma Aldrich', 'Sigma Aldrich; Lumtec; Sigma Aldrich', 'Xi’an p-OLED', 'Unknown', 'Lumtec, Sigma Aldrich, Lumtec, Sigma Aldrich', 'Reike Metals', 'Unknown >> 0.1 mg/ml', 'Clevios PVP', "Xi'an Polymer Light Technology; Xi'an Polymer Light Technology; Xi'an Polymer Light Technology", 'Sigma Adrich; Shenzhen Feiming Science and Technology; Sigma Adrich; Dyesol', 'Dysole; Sigma Aldrich; Dyenamo; Sigma Aldrich', 'Unknown; Borun Chemical; Unknown', 'Clevious', 'Dyesol; Sigma Aldrich; Shenzen Feiminf Science and Technology; Sigma Aldrich', 'Sigma Aldrich; Merck; Sigma Aldrich', 'Sigma Aldrich; Sigma Aldrich; Dalian HeptaChroma SolarTech Co. Ltd.; Sigma Aldrich', 'Advanced Election Technology Co., Ltd; Unknown', 'Tokyo Chemical Industry; Wako Pure Chemical; Wako Pure Chemical', 'Sigma Adrich; Sigma Adrich; Sigma Adrich; Acros Organics', 'Aladdin; Aladdin; Aladdin', 'Alfa-Aesar', 'Unknown | Sigma Aldrich; Sigma Aldrich; Sigma Aldrich', 'Heraeus', '1-Material', 'Kojundo Chemical Lab. Co.', '1-Material, NICT-7', 'Sigma Adrich; Merck; Sigma Adrich', 'Showa Chemical | ACROS Organic', 'Dyesol; Sigma Aldrich; Shenzhen Feiming Science and Technology; Sigma Aldrich', 'Synthesized', 'Guangzhou Seaside Technology', 'Sigma Adrich; Sigma Adrich; Sigma Adrich', 'Dyesol; Aladdin; Merck; Aladdin', 'Tokyo Chemical Industry', 'Aladdin | Aladdin', 'Sigma Aldrich; Sigma Aldrich; Sigma Aldrich', 'Borun Chemicals; Sigma Aldrich; Unknown', 'Baytron', 'Clevios Heraeus', 'Ossila', 'Xi’an p-OLED | Aladdin', '1-Material >> synthesized', 'Synthesized; Synthesized', 'Shanghai Aladdin Bio-Chem. Technology; Shanghai Aladdin Bio-Chem. Technology; Shanghai Aladdin Bio-Chem. Technology; Sigma Aldrich; Sigma Aldrich', 'Sinopharm Chemical Reagent Co. Ltd.,', 'J&K Scientific; Merck; Sigma Aldrich', 'Derthon; Sigma Aldrich; Sigma Aldrich', 'Novaled GmbH | Novaled GmbH', '1-Material; Unknown; Nichem Chemicals; 1-Material'])))

    deposition_reaction_solutions_compounds_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the compound purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, i.e. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For reaction steps involving only pure solvents, state this as ‘none’ (as that is stated in another field)
- If the purity for a compound is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
99.999; Puris| Tecnical
Unknown >> Pro analysis; Pro analysis | none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Puris; Puris; Puris; Pro analysis', 'Unknown', 'Unknown; puris; puris', '99%; 99%', 'Unknown; 96%; Unknown; 99%', 'Puris; Puris; Technical', '99.5; 99.8; Unknown', '99.95%; Unknown; 96%', '0.98', 'Puris; Puris; Puris', '99.9; Pro analysis; Tecnical; Puris', 'Unknown; Pro analysis; Puris; Puris', '0.999'])))

    deposition_reaction_solutions_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the non-solvent precursor chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the concentrations associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated concentrations and separate them with semicolons, as in (A; B)
- The order of the compounds must be the same as in the previous filed.
- For reaction steps involving only pure solvents, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used. When possible, use one of the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml, wt%, mol%, vol%, ppt, ppm, ppb
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
0.2 M; 0.15 M| 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '35 mg/ml', '1.75 vol%; 80 mg/ml; 2.85 vol%', '1.55 wt%', '0.15 M; 1 M', '72 mg/ml; 1.7 vol%; 2.0 vol%', '30 mM; 50 mg/ml; 200 mM', '72.3 mg/ml; 2.85 vol%; 1.75 vol% | nan', '20 mg/ml | 1 mg/ml', '4 mM; 30 mM; 80 mg/ml; 200 mM', '5.4 mg/ml; 9.36 mg/ml; 72.3 mg/ml; 0.028 ml/ml', '30 mol%; 8 wt%; 80 mol%', '0.14 wt%; 2.24 wt%', '51.43 mg/ml; 1.79 vol%; 2.5 vol%', '1 M; 6 mL; 6 mL; 6 mL; 1 mL; 0.3 mmol; 0.3 mmol', '72.3 mg/ml; 6.76 mg/ml; 0.5 vol%', '9.1 mg/ml; 90 mg/ml; 0.029 ml/ml', '0.2 M', '72 mg/ml; 91 mg/ml; 2.85 vol% | nan', '1 mg/ml', '0.82 mg/ml; 10 mg/ml; 2 µl/ml', '15 mg/ml; 1.5 mg/ml', '9.1 mg/ml; 80 mg/ml; 0.0288 ml/ml', '18.2 mg/ml; 72.3 mg/ml; 8 µl/ml', '5 mg/ml', '25 mg/ml; 0.32 mg/ml | 2 mg/ml', '9.1 mg/ml; 72.3 mg/ml; 0.029 ml/ml', '1 mg/ml; 520 mg/ml; 0.036 vol%; 36 mg/ml', '0.15 M; nan', '9.1 mg/ml; 1 wt%; 72.3 mg/ml; 28.8 µl/ml', '90 mg/ml', '5.6 mg/ml; 56 mg/ml; 30 mg/ml', '90 mg/ml; 7.65 mg/ml; 1 vol%', '90 mg/ml; 0.0225 mL; 0.036 mL | nan', 'nan | 72.3 mg/ml; 1.75 vol%; 2.88 vol%', '2 wt% | nan', '70 mg/ml', '72.3 mg/ml; 2.88 vol%; 1.75 vol%', '520 mg/ml; 72.3 mg/ml; 0.0288 vol%', '9.1 mg/ml; 70 mg/ml; 28.8 µl/ml', '72.3 mg/ml; 520 mg/ml; 1', '182 mg/ml; 6 vol%', '11.44 mg/ml; 90 mg/ml; 36 µl/ml', '2 mg/ml | nan', '1.3 mg/ml', '1 mg/ml | 0.05 mg/ml', '17.5 mM', '4 mg/ml', '30 mg/ml; nan', '40 mg/ml; 10 mg/ml', '8.7 mg/ml; 9.1 mg/ml; 72.3 mg/ml; 0.029 ml/ml', '1.5 wt% | 2 mg/ml >> 2 mg/ml >> 2 mg/ml >> 2 mg/ml', '25 mg/ml; 0.32 mg/ml', '11.4 mg/ml; 90 mg/ml; 36 µl/ml', '520 mg/ml; 72.3 mg/ml; 2.88 vol%', '20 mg/ml', '5.2 mg/ml; 52.8 mg/ml; 0.02 ml/ml', '70 mM', '3.83 mg/ml; 50 mg/ml; 22.5 µl/ml', '0.5 vol%; 6.76 mg/ml; 30 mg/ml; 0.5 vol%', '12 µl/ml; 0.2 M', '73 mg/ml', '500 mg/ml; 80 mg/ml; 0.03 vol%', '60 mg/ml', '9.1 mg/ml; 72.3 mg/ml; 28.8 µl/ml | 9.1 mg/ml; 0.5 wt%; 72.3 mg/ml; 28.8 µl/ml', '32 mM; 75 mg/ml; 28.5 µl/ml', '8 mg/ml', '9.1 mg/ml; 72.3 mg/ml; 0.0288 ml/ml', '8.7 mg/ml; 9.8 mg/ml; 72.3 mg/ml; 0.029 ml/ml', '30 mg/ml | 0.005 vol%; 0.995 vol%', '0.5 vol%; 6.76 mg/ml; 10 mg/ml; 0.5 vol%', '12.5 mg/ml | nan', '73.2 mg/ml | 5 mg/ml', '9.1 mg/ml; 2 wt%; 72.3 mg/ml; 28.8 µl/ml', '1 mg/ml | 0.1 mg/ml', '5 mg/ml >> nan', '11.34 mg/ml; 72.3 mg/ml; 0.0176 ml/ml', '1.5 wt% | 2 mg/ml >> 2 mg/ml >> 2 mg/ml', '2.5 wt%', '1.2 mg/ml', 'nan; nan; 70 mg/ml; nan', '63 mg/ml; 170 mg/ml; 2 vol%', '80 mg/ml; 1.45 mg/ml; 2.85 mg/ml', '15 mg/ml; 0.75 vol%; 0.75 vol%', '72 mg/ml; 1.44 vol%; 2.88 vol%', '9.1 mg/ml; 80 mg/ml; 0.03 ml/ml', '1.7 mg/ml; 10 mg/ml; 7 µl/ml', '63 mg/ml', '0.035 M; 0.07 M; 0.231 M', '10 mg/ml', '2 mg/ml; 520 mg/ml; 0.036 vol%; 36 mg/ml', '0.3 M; 2 M; nan', '12.5 mg/ml; 10 mg/ml | nan', '0.2 M | 0.03 M', '30 mM; 85 mg/ml; 30 mM', '182 mg/ml; 6 vol% | 0.5 mg/ml', '9.1 mg/ml; 80 mg/ml; 0.028 ml/ml', '78 mM', '8.8 mg/ml; 0.028 vol%; 70 mg/ml; 0.035 vol%', '0.058 M; 0.0056 M; 0.031 M; 0.19 M', 'nan; 90 mg/ml; nan', '9.1 mg/ml; 60 mM; 0.029 ml/ml', '9.1 mg/ml; 72.3 mg/ml; 28.8 µl/ml', '520 mg/ml; 82 mg/ml; 1.4 vol%', '15 mg/ml; 0.5 mg/ml; 0.5 vol%', '80 mg/ml; 5 mg/ml', '25 mg/ml', '0.2 M | 0.01 M', '3 mol%; 50 mol%; 70 mM; 330 mol%', '0.2 M | 0.05 M', '10 mg/ml; 170 mg/ml; 0.004 vol%', '35 mM', '8.75 mM', '200 mg/ml', '72.3 mg/ml; 2.88 vol%', '1.8 mM; 30 mM; 60 mM; 200 mM', '72.5 mg/ml', '1.5 mg/ml | 0.5 mg/ml', '6.43 mg/ml', '80 mg/ml', '7.5 mg/ml; 7.65 mg/ml; 90 mg/ml; 0.01 ml/ml', '0.064 M; 0.17 M; 0.198 M', '8.7 mg/ml; 9.1 mg/ml; 72 mg/ml; 28.8 µl/ml', '0.1 M', '0.0175 vol%; 72.3 mg/ml; 0.0288 vol%', '2 mg/ml', '8.7 mg/ml; 8.7 mg/ml; 72.3 mg/ml; 2.88 vol%', '9.1 mg/ml; 72.3 mg/ml; 30 µl/ml', '0.3 mg/ml', '1.5 wt% | 2 mg/ml >> 2 mg/ml >> 2 mg/ml >> 2 mg/ml >> 2 mg/ml', '15 mM', '10.4 mg/ml; 60 mg/ml; 0.03 mg/ml', '102 mg/ml', '2.5 mg/ml', '10 mg/ml | nan', '30 mg/ml', '1 mg/ml | 0.025 mg/ml', '72.3 mg/ml; 520 mg/ml; 0.3 vol%', '0.44 M', '11.4 mg/ml; 90 mg/ml; 0.036 mg/ml', '0.90 vol%; 2.07 vol%; 0.091 mg/ml; 3.60 vol%', '72.3 mg/ml; 1.75 vol%; 2.88 vol%', '9.1 mg/ml; 72.3 mg/ml; 28.8 µl/ml | 9.1 mg/ml; 1 wt%; 72.3 mg/ml; 28.8 µl/ml', '10.4 mg/ml; 80 mg/ml; 0.03 ml/ml', '4 mM; 30 mM; 80 mg/ml; 200 mM | 0.005 vol%; 0.995 vol%', '35 mM; 35 mM; 210 mM', '72.3 mg/ml; 2 mM; 2.88 vol%', '80 mg/ml; 10 mg/ml', '68 mM; 9 mM; 55 mM', '1.6 vol%mM; 2.1 vol%; 91 mg/ml; 3.6 vol%', '2 mg/ml | 0.5 mg/ml', '0.2 M | 0.02 M', '0.2 M | 0.04 M', '100 mg/ml', '72.3 mg/ml; 0.029 vol%; 28.3 mg/ml; 0.0288 vol%', '2.45 mM; 40 mM; 81.6 mM; 270 mM', '72.3 mg/ml', '60 mM; 32 mM; 195 mM', '0.5 mg/ml; 520 mg/ml; 0.036 vol%; 36 mg/ml', '9.1 mg/ml; 72 mg/ml; 0.028 mg/ml', '0.170 M; 0.064 M; 0.198 M', '9.1 mg/ml; 0.5 wt%; 72.3 mg/ml; 28.8 µl/ml', '1.5 wt% | 2 mg/ml', '2.38 wt%', '300 mg/ml; 520 mg/ml; 72.3 mg/ml; 0.028 vol%', '0.5 wt%', '73.2 mg/ml', '1.5 wt%', '54 mol%; 30 mg/ml; 334 mol%', '72.3 mg/ml; 1.7 vol%; 2.8 vol%', '2.8 vol%mM; 1.85 vol%; 72.3 mg/ml; 2.9 vol%', '8.67 mg/ml; 9.1 mg/ml; 72.3 mg/ml; 28.8 µl/ml', '520 mg/ml; 83.2 mg/ml; 0.0338 vol%', '0.175 vol%; 80 mg/ml; 0.285 vol%', '1.5 wt% | 2 mg/ml >> 2 mg/ml', '9.1 mg/ml; 72.5 mg/ml; 0.028 mg/ml', '6 mg/ml', '15 mg/ml', '520 mg/ml; 36 mg/ml; 0.036 vol%', '1 mg/ml; 1 mg/ml', '9 mM; 68 mM; 55 mM', '1.5 mg/ml', '20 mg/ml; 170 mg/ml; 34.78 vol%', '50 mg/ml', '32 mM; 15 mg/ml; 28.5 µl/ml', '72.3 mg/ml; 1.75 vol%; 3.1 vol%', '72 mg/ml', '97 mg/ml', '30 mM; 72.3 mg/ml; 200 mM', '15.08 mg/ml; 9.1 mg/ml; 72.3 mg/ml; 28.8 µl/ml', '1 mM', '6.24 mg/ml; 72 mg/ml; 8 µl/ml', '0.005 vol%; 0.995 vol%', '5 mg/ml; 520 mg/ml; 0.036 vol%; 36 mg/ml', '9.1 mg/ml; 72.3 mg/ml; 28.8 µl/ml | 9.1 mg/ml; 2 wt%; 72.3 mg/ml; 28.8 µl/ml', '20 mg/ml; 6.8 vol%; 3.4 vol%'])))

    deposition_reaction_solutions_volumes = Quantity(
        type=str,
        shape=[],
        description="""
    The volume of the reaction solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the volumes associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The volumes refer the volumes used, not the volume of the stock solutions. Thus if 0.15 ml of a solution is spin-coated, the volume is 0.15 ml
- For reaction steps without solvents, state the volume as ‘nan’
- When volumes are unknown, state that as ‘nan’
Example
0.1
0.1 >> 0.05 | 0.05
nan | 0.15
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['0.02', '0.045', '33.0', '0.0175; 0.9537; 0.0288', '0.029; 0.0175; Unknown; 0.288', '0.03; 0.94; 0.03', '1.0; 0.015; 0.008', '0.0088; 0.0144', '0.05', 'Unknown', '0.065', '0.95; 0.0075; 0.004', '0.035', '0.018; 0.018; 0.936; 0.028', '1.3', '1.5', '0.006; 0.0175; 0.9485; 0.028', '1.7', '0.06', '19.0; 7.0; 8.0; 0.2 | 33.33', '0.0175; Unknown; 0.0285', '1.0', '0.0075; 0.47; 0.0169', '0.0175; 0.95; 0.0288', '0.018; 0.94; 0.028; 0.018', '0.018; Unknown; 0.028', '0.0175; 0.028', '0.92; 0.029; 0.0175; 0.0288'])))

    deposition_reaction_solutions_age = Quantity(
        type=str,
        shape=[],
        description="""
    The age of the solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the age of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- As a general guideline, the age refers to the time from the preparation of the final precursor mixture to the reaction procedure.
- When the age of a solution is not known, state that as ‘nan’
- For reaction steps where no solvents are involved, state this as ‘nan’
- For solutions that is stored a long time, an order of magnitude estimate is adequate.
Example
2
0.25 |1000 >> 10000
nan | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['48.0', 'Unknown', '0.33', '10.0', '0.0167', '4.0', '2.0', '4.0 | 0.0', '24.0', '3.0', '0.5'])))

    deposition_reaction_solutions_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the reaction solutions.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a reaction solution undergoes a temperature program, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons, e.g. 25; 100
- When the temperature of a solution is unknown, state that as ‘nan’
- For reaction steps where no solvents are involved, state the temperature of the gas or the solid if that make sense. Otherwise state this as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume an undetermined room temperature to be 25
Example
25
100; 50 | 25
nan | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['70; 25', '25', 'Unknown', '70 | 25', '90', '60; 25', '50', '25 | Unknown', '70 | Unknown', '70', '60', '24', '25 | 25'])))

    deposition_substrate_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the substrate.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the substrates (i.e. the last deposited layer) associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The temperature of the substrate refers to the temperature when the deposition of the layer is occurring.
- If a substrate undergoes a temperature program before the deposition, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- When the temperature of a substrate is not known, state that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume that an undetermined room temperature is 25
Example
25
nan
125; 325; 375; 450 | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '25', 'Unknown', '25 | 15', '120 | 25', '25 | 25 >> 25', '120', '25 >> 100', '25 | 25 >> 25 >> 25', '25 | 25 >> 25 >> 25 >> 25 >> 25', '25 | 25 >> 25 >> 25 >> 25', '25 | 25', '21'])))

    deposition_thermal_annealing_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperatures of the thermal annealing program associated with depositing the layers
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing temperatures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- If no thermal annealing is occurring after the deposition of a layer, state that by stating the room temperature (assumed to 25°C if not further specified)
- If the thermal annealing program is not known, state that by ‘nan’
Example
25
50 | nan
450 | 125; 325; 375; 450 >> 125; 325; 375; 450
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['25', '550', '25 >> 80', '300', '475', '13 | Unknown', '18 | Unknown', '160', '25 >> 550', '140 | 100', '5', '600', '140 | 100 >> 100 >> 100 >> 100', '650', '5 | 0', '25 >> 250', '70', '325', '75; 120; 300', '25 | 25', '400', '80', '200', '140 | 100 >> 100 >> 100 >> 100 >> 100', 'Unknown', '130 | 60', '100', '130', '120', '500', '150', '235 | 15', '165', '100 | 100', '110.0', '60', '25 | 55', '25 >> 650', '120 | 150', '150 | Unknown', '30 | Unknown', '90', '100 >> 100', '100 | 25', '25 >> 450', '12 | Unknown', '95', '235', '145', '450', '7 | Unknown', '125', '135', '140 | 100 >> 100', '235 | 25', '300 | 120', '140', '25; 100', '350', '140 | 100 >> 100 >> 100', '25 >> 100', '11 | Unknown', '50', '120 | 60', '9 | Unknown', '15 | Unknown', '130 | 25', '200.0'])))

    deposition_thermal_annealing_time = Quantity(
        type=str,
        shape=[],
        description="""
    The time program associated to the thermal annealing program.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing times associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the associated times at those temperatures and separate them with semicolons.
- The annealing times must align in terms of layers¸ reaction steps and annealing temperatures in the previous filed.
- If a time is not known, state that by ‘nan’
- If no thermal annealing is occurring after the deposition of a layer, state that by ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 20 and not 10-30.
Example
nan
60 | 1000
30 | 5; 5; 5; 30 >> 5; 5; 5; 30
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['40.0', '10.0 | 5.0 >> 5.0 >> 5.0', '10.0 | 5.0 >> 5.0 >> 5.0 >> 5.0 >> 5.0', '10.0', '5.0', '60.0', '45.0 | 0.0', '12.0', 'Unknown', '5.0 >> 2.0', '120.0', '45.0', '10.0 | 0.0', '15.0 | 25.0', '10.0 | 5.0', '30.0', '10.0 | 10.0', '0.0 >> 30.0', 'Unknown | 25.0', '30.0; 30.0', '10.0; 15.0; 60.0', '10.0 | Unknown', '60.0 | 15.0', '10.0 | 5.0 >> 5.0 >> 5.0 >> 5.0', '15.0 | 5.0', '1.0', '25.0', '20.0', '20.0 | 10.0', '10.0 | 5.0 >> 5.0', '2.0 | 2.0', '15.0'])))

    deposition_thermal_annealing_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere during thermal annealing
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each annealing step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the atmosphere is a mixture of different gases, i.e. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas.
- This is often the same as the atmosphere under which the deposition is occurring, but not always.
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Vacuum | N2
Air | Ar >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Dry air', 'Unknown', 'Air | Vacuum', 'Air | N2', 'Air', 'Ambient', 'N2', 'Ambient | Ar', 'O2', 'Vacuum', 'N2 >> N2', 'Air | Air', 'N2 | N2', 'Ar | Ar', 'Ar'])))

    storage_time_until_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    The time between the HTL stack is finalised and the next layer is deposited
- If there are uncertainties, only state the best estimate, e.g. write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['12.0', 'Unknown', '20.0', '24.0', '4.0', '15.0'])))

    storage_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere in which the sample with the finalised HTL stack is stored until the next deposition step.
Example
Air
N2
Vacuum
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Dry air', 'Unknown', 'Air', 'Ambient', 'Vacuum', 'N2', 'O2'])))

    storage_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relive humidity under which the sample with the finalised HTL stack is stored until next deposition step
- If there are uncertainties, only state the best estimate, e.g write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '20.0', '10.0'])))

    surface_treatment_before_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    Description of any type of surface treatment or other treatment the sample with the finalised HTL stack undergoes before the next deposition step.
- If more than one treatment, list the treatments and separate them by a double forward angel bracket (‘ >> ‘)
- If no special treatment, state that as ‘none’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example:
none
Ar plasma
UV-ozone
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Plasma', 'Ar plasma', 'Ozone', 'UV-Ozone', 'He plasma', 'Washed with methanol', 'IPA dipping', 'DMF'])))

    def normalize(self, archive, logger):
        add_solar_cell(archive)
        if self.stack_sequence:
            archive.results.properties.optoelectronic.solar_cell.hole_transport_layer = self.stack_sequence.split(' | ')


class Backcontact(MSection):
    """
    A section to describe information related to the back contact of the solar cell.
    """

    stack_sequence = Quantity(
        type=str,
        shape=[],
        description="""
    The stack sequence describing the back contact.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If two materials, e.g. A and B, are mixed in one layer, list the materials in alphabetic order and separate them with semicolons, as in (A; B)
- If no back contact, state that as ‘non’
- Use common abbreviations when appropriate but spell it out if risk for confusion.
- If a material is doped, or have an additive, state the pure material here and specify the doping in the columns specifically targeting the doping of those layers.
- There is no sharp well-defined boundary between when a material is best considered as doped or as a mixture of two materials. When in doubt if your material is best described as doped or as a mixture, use the notation that best capture the metaphysical essence of the situation.
- There are a lot of stack sequences described in the literature. Try to find your one in the list. If it is not there (i.e. you may have done something new) define a new stack sequence according to the instructions.
Example:
Au
Ag
Al
Carbon
MoO3 | Ag
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Au | ITO', 'rGO | Au', 'MoO3 | Ag | WO3', 'Carbon; WO3-np', 'ITO | MgF2', 'Al | Al2O3', 'Ag | Alq3', 'AgAu', 'MoOx | Ag', 'Carbon-nt; PCBM-60', 'ITO', 'AgAl', 'Cr | Cu', 'AgZn | Al', 'Mo | Ag', 'Cu-CFN', "Field's metal", 'Al | Ag', 'Ag-nw | ZnO-np', 'Carbon; NiS', 'PEDOT:PSS | PDMS', 'Carbon; Graphite; PANI', 'MoOx | Au', 'Ni-grid', 'GaIn', 'Cu', 'Ag | MoO3', 'CNTs', 'none', 'MoOx | Al', 'MWCNTs; ONC1', 'Cu | Au', 'Cr | Au', 'MoOx | Au | Cu | MoOx', 'Pt-sheet', 'MoO3 | Ag', 'Carbon | IPA', 'KIPIG', 'T-MWCNTs', 'AZO', 'CSCNT@SnO2', 'B-MWCNTs', 'Carbon black | Carbon', 'Ag | SiO2 | ZnS | Ag | ZnS', 'Carbon-nt | PMMA', 'MoO3 | Ag | MoO3', 'PEDOT:PSS | Graphene', 'Pb', 'ITO | SLG', 'Carbon; NiO:rGO', 'Carbon; NiO', 'NiO', 'Cu; Cu2O', 'Graphene | PDMS', 'Carbon black; Graphite', 'N-Graphene', 'CuPc | Carbon', 'Carbon', 'Carbon-paper', 'Au | Al', 'Carbon; WO2-np', 'FTO', 'CNTs | Mxene', 'Graphite | Cu-tape', 'H2PtCl6', 'PTAA | FTO | SLG', 'PEI | PEDOT:PSS | PDMS', 'Bi2Te3', 'Pt', 'Carbon-nw', 'Cu | Ag | MoO3', 'MoO2 | ITO', 'Ca | Al', 'AZO | Ni | Al | Ni', 'MoOx | Cu', 'Carbon-tape', 'Carbon | Au', 'Carbon | CuSCN', 'PEDOT:PSS | Al', 'TFSA-Graphene | PET | Ag', 'ITO | Ag-grid', 'Ca', 'Mo2O3 | Ag', 'SnO2-c | Ag | SnO2-c', 'Candle soot | FTO | SLG', 'MoOx | IZO', 'Au | Ag-nw', 'Au', 'TETA-Graphene | PET', 'W', 'Transparent Conductive Adhesive | PET:Ni mesh', 'Graphene | PMMA | PDMS', 'MoO3 | Au | MoO3', 'Au | LiF', 'MoO3 ∣ ITO', 'Au-np', 'Mg | Al', 'Carbon | Ag', 'TETA-Graphene | PET | Ag', 'Graphite | Cu', 'Ag-nw', 'Ni', 'Na@Carbon-nanowalls', 'Carbon black', 'Au | Organosilicate', 'AZO-c', 'Carbon; NiO-np', 'Ag', 'Ag | FTO', 'IZTO', 'ITO | MWCNTs', 'Ag-nw | PCBM-60', 'PEDOT:PSS | PEDOT:PSS | PDMS', 'Carbon | FTO | SLG', 'SWCNTs', 'Cr', 'IZO | Ag', 'FTO | SLG', 'Graphite | FTO', 'AZO | Ni | Al', 'Pd', 'Carbon | FAAc', 'Carbon black; Carbon-nt; Graphite', 'Carbon black; Graphite | MWCNTs', 'TeO2 | Ag', 'Au | MoO3', 'AZO | NiAl', 'Bi | Au', 'Carbon; MAI | Carbon', 'NiO | Ag | NiO', 'MoO3 | ITO | MgF2', 'PH 1000', 'Ag | ITO | Ag', 'ITO | LiF', 'Ag | IZO', 'LiF | Al', 'H:MoO3-nanobelts', 'Carbon | CNTs', 'Ti | Au', 'Carbon | Al', 'MoOx | Au | MoOx', 'Ag-sheet', 'TFSA-Graphene | PET', 'Graphene | PET', 'Graphene | PEDOT', 'MoOx | Ag | MoOx', 'MoO3 | AuAg | MoO3', 'Carbon | Carbon-fibre', 'MoO3 | Au | Ag', 'Ba | Ag', 'Graphen', 'Mg | Ag', 'Carbon | MAAc', 'MoO3 | Au', 'MoOx | ITO', 'Graphite | Pt', 'MWCNTs; ONC2', 'IZO', 'Ag | Ni', 'PANI | FTO | SLG', 'Al | Au', 'MoO3 ∣ Au ∣ Ag ∣ MoO3 | Alq3', 'PEDOT:PSS | Ag-nw | PDMS', 'MWCNTs', 'Cr | Pt | FTO', 'SnO2-c | Ag', 'Carbon | Sn', 'Unknown', 'Carbon; PEMA', 'Carbon | Graphite', 'MoP3 | Ag', 'Ag | Ta2O3', 'Cr2O3:Cr', 'Graphene | Au', 'PEDOT:PSS | FTO | SLG', 'Carbon | CNTs | Mxene', 'AlAg', 'AZO | Au', 'AV-Carbon; MAI', 'Graphite; Carbon black@5:1', 'NiO | Ag | NiO | NaYF4 | Ag', 'AV-carbon; MAI', 'SWCNTs | PMMA', 'Ba | Al', 'Pt-Carbon-nt', 'Sb', 'Carbon-epoxy | Ag', 'ITO | Ni | Al', 'Ni | Al', 'Ag | V2O5', 'Pt | FTO | SLG', 'PEDOT:PSS | ITO | SLG', 'Graphene', 'Cu | Au | BCP', 'IZO | Au', 'Au | Ni', 'Au | ITO | Au', 'MoO3 | AZO | AlNi-grid', 'WO3 | Ag', 'Au | Ag', 'SnO2-c | Cu | SnO2-c', 'Carbon | FTO', 'CSCNT@Al2O3-c | CSCNT@SnO2', 'Ti', 'Carbon-nt', 'AZO-np | Ag', 'Carbon; NiPt-nw', 'D-Sorbito; PEDOT:PSSl | Ag-nw | PET', 'Carbon; LPP', 'MoOx | Cu | MoOx', 'PEDOT:PSS | ITO | PET', 'In', 'Carbon-nanowalls', 'NiS | Cr | Pt | FTO | SLG', 'Ag | ITO', 'Ca | Ag', 'Ag-nw | C60', 'MoOx | Ag | ZnS', 'Au | FTO', 'Pt | Si', 'Al', 'Ag@Au-np', 'Graphite', 'PEDOT:PSS:PSA', 'MnO3 | Ag', 'AgAu-mp', 'ITO | Al', 'Ag | SnO2-c', 'ICO', 'Ti-grid', 'ITO | Au', 'MoO3 | Au | Ag | MoO3 | Alq3', 'Au-np; NiO', 'Carbon-nt | Carbon', 'MoOx | ITO | Au', 'PEDOT:PSS; Sorbitol | Ag-grid | PET', 'MoO3 | ITO', 'Pt | FTO', 'MoOx | IO | ITO | Au', 'Carbon | Galinstan', 'P3HT | FTO | SLG', 'ITO | Ag', 'Ni | Au', 'Ag | Au', 'SWCNTs | Ag', 'Ag | Al', 'MoOx | IAI', 'Cu | Ag', 'Perovskite | PEDOT:PSS | ITO | SLG', 'Graphene oxide | Carbon', 'Ag-nanocubes | Ag | MoO3', 'MoO3 | IZO', 'Carbon-mp', 'MoOx | ITO | MgF2', 'NbS2', 'MoO3 | Al', 'Ti3C2', 'MWCNTs; ONC3', 'Carbon | PEDOT:PSS | FTO | SLG', 'MoOx | IZO | Au', 'LiF | Ag', 'ITO | Au-grid', 'PEDOT:PSS', '3D potassium-ion preintercalated graphene (KIPIG)', 'PEI | PH 1000', "Filed's metal", 'Ti | Cu', 'Au | Au-wire', 'AlNi-grid', 'Carbon-nt | PMMA | Au', 'ITO | PEN', 'Carbon black; MWCNTs', 'Carbon | Silica gel electrolyte | Carbon', 'CSCNT@Al2O3-c | CSCNT', 'Ag-np | ITO', 'Carbon; MAI', 'PTCBI | Ag | WO3 | PTCBI | Ag', 'Carbon | KAc', 'Ag | CsF', 'Liq | Al', 'Graphite | FTO | SLG', 'Cu-ribbon', 'ITO | Cu', 'Mo', 'Metal', 'AZO | Ag | AZO', 'Graphite ribbon', 'Pt-np | FTO | SLG', 'Carbon | CsAc'])))

    thickness_list = Quantity(
        type=str,
        shape=[],
        description="""
    A list of thicknesses of the individual layers in the stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous filed.
- State thicknesses in nm
- Every layer in the stack have a thickness. If it is unknown, state this as ‘nan’
- If there are uncertainties, state the best estimate, e.g write 100 and not 90-110
Example
100
10 | 80
nan | 100
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '89.0', '90.0', '40.0 | 11.0', '5.0 | 130.0', '50.0', '70.0 | 10.0 | 10.0', '220.0', '230.0 | 100.0', '81.0', '180.0', '45.0', '15.0 | 80.0', '14000.0', '50.0 | 4000.0', '7.0 | 80.0', '24.0 | 15.0 | 20.0 | 480.0 | 8.0', '10.0 | 100.0 | 200.0', '203.0', '10.0 | 120.0 | 70.0', '11.0 | 30.0', '1.0 | 200.0', '30.0 | 120.0 | nan', '60.9', '11000.0 | nan', '1.3 | 100.0', '300.0 | 250.0', '2.0 | 1.0 | 7.0 | 5.0 | 50.0', '15.0 | 150.0', '60.4', '35.0 | 103.0 | 35.0', '2.5 | 154.0', '6.0 | 11.0 | 20.0', '1800.0', '22000.0', '11.0 | 50.0', '60.5', '95.0', '20.0 | 10.0 | 20.0', '10.0 | 110.0', '20.0 | 14.0', '87.0', '10.0 | 90.0', '70.0', '12.0 | 40.0', '21.0 | 250.0', '11.0', '500.0', '2000.0 | nan', '5.0 | nan', '53150.0', '2000.0', '20.0 | 15.0 | 20.0 | 480.0 | 8.0', '10.0 | 80.0', '1.0 | 7.0 | 40.0', '21.0 | 15.0 | 20.0 | 480.0 | 8.0', '8.0 | 120.0', '5.0 | 150.0', 'nan | 6.0 | nan', '61210.0', '260.0', '201.0', '3.0 | 1.0 | 7.0 | 5.0 | 50.0', '2.0 | 100.0', '7.0 | 60.0', '100.0 | 1000.0', '16500.0 | nan', '10.0 | 15.0 | 30.0', '240.0', '7.2 | 70.0', '50.0 | 50.0', '11.0 | nan', '9.0 | nan', '35.0 | 95.0 | 35.0', '330.0', '10.0 | 30000.0', '60.3', '85.0', '60.1', '130.0 | 100.0', '2.5 | 154.0 | 50.0', 'nan | 150.0', '15.0 | 12.0', '10.0 | 70.0', '9.0', '15.0 | 12.0 | 60.0', '150.0 | 500.0', '8.0', '0.5 | 60.0', '10.0 | 200.0', '13.0', '7.0 | 12.0 | 30.0', '820.0', '1.0 | 6.0', 'nan | 80.0', '13.0 | 80.0', '25.0 | 15.0 | 20.0 | 480.0 | 8.0', '2.0 | 250.0', '93.2', '500.0 | 150.0', '5000.0', '20.0', '52.0', '7.0 | 18.0 | 30.0', '21.0 | 7.0 | 20.0', '60.11', '82.0', '15.0 | 12.0 | 20.0', '6.0 | 1.5 | 9.5 | 20.0', '7870.0', '8.0 | 20.0 | 115.0 | 8.0 | 20.0', '30.0 | 120.0', '30.0 | 80.0', '10.0 | 20.0', '14.0', '8.0 | 20.0 | 65.0 | 8.0 | 20.0', '80.0 | 10.0', '2001.0 | nan', 'nan | 100.0', '250.0', '10.0 | 10.0', 'nan | 220.0', '65.0', '150.0', '15.0 | 50.0', '4000000000.0', '15.0 | 200.0', '46530.0', 'nan | 70.0', '23.0 | 15.0 | 20.0 | 480.0 | 8.0', '35.0 | 10.0 | 35.0', '100000.0 | nan', '88.0 | 700.0', '5.0 | 12.0 | 40.0', '1.0 | 250.0', '8000.0 | nan', '60.8', '12000.0', '15.0 | nan', '60.7', 'nan | 25000.0', '1.0 | 100.0', '30.0 | 70.0', '50000.0', '30.0 | 135.0', '8.0 | 200.0', '35.0', '180.0 | 50.0', '30.0 | 7.0 | 80.0', '750.0', '32000.0', '1.0 | nan', '5.0 | 90.0', '8.0 | 80.0', '150.0 | 4900.0', '90.0 | 200.0', '51.0', '22.0 | 15.0 | 20.0 | 480.0 | 8.0', '4500.0', '16000.0', '24000.0', '30000.0', '10.0 | 60.0', '12.0 | 100.0', '7.0 | 120.0', '9.0 | 100.0', '6.7 | 100.0', '154.0 | 50.0', '25.0', '10.0 | 202.0', '20.0 | 10.0 | 35.0', '80.0 | 20.0', '40.0', '65.0 | nan', '83.0', '11200.0', '8.0 | 90.0', '20.0 | 7.0 | 20.0', '230.0', '75.0', '8.0 | 60.0', '190.0', '6.0 | 100.0', '7.0 | 1.0', '3.0 | nan', '15000.0', '35.0 | 84.0 | 35.0', '60.6', '9.0 | 80.0', 'nan | 400.0 | 500.0', '10.0 | 40.0', '145.0', 'nan | 50.0 | 4000.0', '1500.0', '100.0 | 125.0', '2.0 | 200.0', '28.0 | 15.0 | 20.0 | 480.0 | 8.0', '7.0 | 70.0', '60.0', '125.0', '5.0', '40000.0', '45.0 | nan', '35.0 | 76.0 | 35.0', '900.0', '400.0', '105.0 | 90.0', '25000.0', '8.0 | 30.0 | 100.0', '180.0 | 100.0', '2.0', '20.0 | nan', '80.0 | 150.0', '30.0 | 100.0', '11.0 | 10.0', '5.0 | 120.0', '4.0', '1.0 | 10.0 | 20.0', '14.0 | 20.0', '15.0 | 40.0', '40.0 | 120.0', '20.0 | 250.0', '600.0', 'nan | 200.0', '12.0', '12000.0 | nan', '6.0 | nan', '100.0', '105.0', '8.0 | 110.0', '15.0 | 120.0', '37500.0', '23000.0 | nan', '5.0 | 10.0 | 35.0', '10.0 | 40.0 | nan', '80.0 | 50.0', '10000.0 | nan', '8.0 | 20.0 | 85.0 | 8.0 | 20.0', '28000.0', '21000.0', '48.0', '9.0 | 25.0', '4.0 | 80.0', '15.0 | 100.0', '96.0', '3.0 | 80.0', '200.0', '70.0 | 10.0', '4000.0 | nan', '150.0 | nan', '80.0 | 8.0', '350.0', '110.0 | 120.0', '1.0 | 10.0 | 35.0', '90.0 | 100.0', '50.0 | 100.0', '80000.0', '10.0 | 120.0', '202.0', '200000.0', '8000.0', '35.0 | 120.0 | 70.0', '160.0', '10.0 | 55.0', '3.5 | 100.0', '2000.0 | 100.0 | nan', '10.0 | 150.0', '25.0 | 100.0', '12.0 | 80.0', '5.0 | 10.0 | 40.0', '170.0', '35.0 | 64.0 | 35.0', '1500-2000', '55.0', '9000.0', '6.0', '4900.0', '10.0 | 100.0 | 10.0 | 100.0', '100.0 | 20.0', '20.0 | 100.0', '7.0 | nan', '34.0 | 20.0 | 25.0 | nan | 8.0', '3333.0', '2.5 | 80.0', '20.0 | 80.0', '18.0 | 10.0 | 19.0', '150.0 | 100.0', '5.0 | 60.0', '6.0 | 10.0 | 35.0', '3500.0', '550.0', '15.0', '12500.0', '35.0 | 120.0', '8.0 | 100.0', '7.0 | 24.0 | 30.0', '300.0', '11.0 | 80.0', '50.0 | 101.0', '185.0', '71600.0', '77.0 | nan', '5.0 | 100.0', '120.0', '88.0', '130.0', '3000.0', '110.0', '20.0 | 15.0', '120000.0', '300000.0', '10.0 | 100.0', '26.0 | 15.0 | 20.0 | 480.0 | 8.0', '11000.0', '6500.0', '1000.0', '5.0 | 10.0 | 5.0', '10.0 | 1.0 | 10.0 | 40.0', '34.0 | 20.0 | 55.0 | nan | 15.0', '6.3 | 70.0', '60000.0', '100.0 | 100.0', '10.0 | 35.0', '1200.0', '60.2', '390.0', '10.0', '11.0 | 20.0', '25800.0', '35.0 | 160.0', '35.0 | 53.0 | 35.0', '30.0 | nan', '1.0 | 120.0', '120.0 | 70.0', '20000.0', '5.0 | 20.0 | 35.0', '210.0', '2500.0', '5.0 | 300.0', '5.0 | 80.0', '10.0 | 10.0 | 35.0', '800.0', '1.9 | 70.0', '1.45 | nan', '9.0 | 110.0', '40.0 | 4000.0', '3.0 | 100.0', '20.0 | 14.0 | 20.0', '15.0 | 350.0', '53.0', '70.0 | 30.0', '250.0 | 50.0 | 1000.0', '86.0', '1600.0', '3.0 | 200.0', '7000.0', '97.0', '30.0', '6.0 | 60.0', '7.0 | 100.0', '3.0 | 150.0', '10.0 | nan | nan', '10.0 | 75.0', '10.0 | 50.0', '70000.0', '9600.0', '140.0', '50.0 | 120.0', '100.0 | nan', '80.0', 'nan | 5.0', '27.0 | 15.0 | 20.0 | 480.0 | 8.0', '80.0 | 80.0', '6.0 | 80.0', '13000.0', '35.0 | 88.0 | 35.0', '4000.0', '100000.0', '60.0 | 40.0', '84.0', '16.0', '10.0 | 201.0', '10000.0', '34.0 | 20.0 | 75.0 | nan | 35.0', '10.0 | 8.5 | 10.0'])))

    additives_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    List of the dopants and additives that are in each layer of the HTL-stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous fields.
- If several dopants/additives, e.g. A and B, are present in one layer, list the dopants/additives in alphabetic order and separate them with semicolons, as in (A; B)
- If no dopants/additives, state that as “Undoped”
- If the doping situation is unknown, stat that as‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template, even if to most common back contacts is undoped metals
Example
CuS
B; P
Au-np | Undoped
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Undoped | Undoped', 'TiO2-np', 'Undoped | Undoped | Undoped | Undoped | Undoped', 'B; P', 'C; NiO', 'B4C', 'WO3-np', 'P', 'B', 'Undoped', 'CuS'])))

    additives_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the dopants/additives.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If more than one dopant/additive in the layer, e.g. A and B, separate the concentration for each dopant/additive with semicolons, as in (A; B)
- For each dopant/additive in the layer, state the concentration.
- The order of the dopants/additives must be the same as in the previous filed.
- For layers with no dopants/additives, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used.
- The preferred way to state the concentration of a dopant/additive is to refer to the amount in the final product, i.e. the material in the layer. When possible, use on the preferred units
o wt%, mol%, vol%, ppt, ppm, ppb
- When the concentration of the dopant/additive in the final product is unknown, but where the concentration of the dopant/additive in the solution is known, state that concentration instead. When possible, use on the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
5 vol%; nan | 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | 0.3 M
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '4; 1', '10 wt%', '2 wt%', '7; 3', '5 wt%', '0.1 wt%', '9; 1', '3.5 wt%', '7.5 wt%', '8 wt%', '6.5 wt%', '0.5 wt%'])))

    deposition_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    The deposition procedures for the HTL-stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate them by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Thermal annealing is generally not considered as an individual reaction step. The philosophy behind this is that every deposition step has a thermal history, which is specified in a separate filed. In exceptional cases with thermal annealing procedures clearly disconnected from other procedures, state ‘Thermal annealing’ as a separate reaction step.
- Please read the instructions under “Perovskite. Deposition. Procedure” for descriptions and distinctions between common deposition procedures and how they should be labelled for consistency in the database.
- A few additional clarifications:
- Lamination
o A readymade film is transferred directly to the device stack. A rather broad concept. An everyday kitchen related example of lamination would eb to place a thin plastic film over a slice of pie.
- Sandwiching
o When a readymade top stack simply is placed on top of the device stack. Could be held together with clams. The typical example is a when a “Carbon | FTO | SLG” is placed on top of the device stack. Standard procedure in the DSSC filed.
Example
Evaporation
Evaporation | Evaporation
Doctor blading
Screen printing
Sputtering
Lamination
E-beam evaporation
Sandwiching
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Evaporation | Evaporation | Evaporation | Evaporation', 'Doctor blading | Doctor blading | Doctor blading', 'Pressed', 'Spray-coating | Spray-coating', 'Spin-coating | Spin-coating', 'Electropolymerisation | Sandwiching', 'Brush painting', 'Evaporation | DC Sputtering', 'Screen printing | Lamination', 'Spin-coating | Evaporation', 'Sputtering | Sputtering | Sputtering', 'Evaporation | Spin-coating', 'Sputtering', 'Screen printing | Spray-coating', 'DC Sputtering', 'Evaporation | Sputtering | Sputtering', 'Evaporation | RF sputtering | E-beam evaporation', 'Doctor blading | Spin-coating', 'Dropp casting', 'Lamination | Lamination', 'RF sputtering | Evaporation | Evaporation', 'Evaporation | ALD', 'Inkjet printing', 'RF sputtering', 'E-beam evaporation | E-beam evaporation | E-beam evaporation | PVD | Evaporation', 'Sputtering | Spray-coating', 'Candle burning | Sandwiching', 'Brush painting | Brush painting', 'Evaporation | Evaporation', 'E-beam evaporation | E-beam evaporation | E-beam evaporation', 'CVD | Spin-coating >> reactive ion etching', 'Evaporation | Activated reactive evaporation', 'Spin-coating | Doctor blading', 'Sputtering | Sputtering', 'Drop coated', 'Sandwithcing', 'CVD >> Lamination', 'Unknown | Ultrasonic  welding', 'Sputtering | Evaporation | Evaporation | Evaporation', 'Evaporation | Magnetron sputtering', 'Candel burning >> Sandwiching | Unknown | Unknown', 'Evaporation | RF magnetron sputtering', 'Candle burning | Lamination', 'ALD | Evaporation | ALD', 'Spray-pyrolys | Sandwiching', 'Spin-coating | Screen printing', 'Lamination | Spin-coating', 'Evaporation | DC Magnetron Sputtering', 'Lamination | Painting', 'Lamination', 'Doctor blading | Ultrasonic  welding', 'Evaporation | Evaporation | Evaporation', 'Brush painting | Unknown', 'Evaporation | Sputtering', 'PVD', 'Dipp-coating', 'Screen printing >> Lamination', 'Springkling | Sandwiching', 'Magnetron sputtering | Magnetron sputtering', 'Sputtering | Lamination', 'Spin-coating | Sandwiching', 'Evaporation', 'Unknown', 'Unknown | Doctor blading', 'Evaporation | Magnetron sputtering | E-beam evaporation', 'Lamination | Evaporation', 'CVD', 'Lamination >> Isostatic pressing', 'ALD | Evaporation', 'Screen printing', 'Evaporation | Sputtering | Evaporation', 'Unknown | Evaporation', 'Evaporation | Sputtering | Sputtering | Evaporation', 'Doctor blading', 'Magnetron sputtering', 'Evaporation | Evaporation | Evaporation | Evaporation | Evaporation', 'RF Magnetron Sputtering', 'DC Sputtering | Evaporation', 'Candle burning >> Sandwiching', 'Dropcasting | Lamination', 'Lamination | Dropcasting', 'Electrospinning', 'Suttering', 'E-beam evaporation', 'Candle burning >> Lamination', 'Screen printing | Painting', 'Evaporation >> Evaporation >> Oxidation', 'Evaporation | E-beam evaporation | E-beam evaporation | E-beam evaporation | E-beam evaporation', 'DC Magnetron Sputtering | Evaporation', 'Sputtering >> Lamination', 'Evaporation | Sandwiching', 'Dropcasting', 'Pulsed laser deposition', 'DC Magnetron Sputtering', 'Screen printing | Unknown', 'E-beam evaporation | E-beam evaporation', 'Sputtering | Evaporation', 'Lamination | Spin-coating | Evaporation', 'Brush painting | Sandwiching', 'Spin-coating', 'Sputtering | E-beam evaporation | E-beam evaporation', 'Doctor blading | Doctor blading', 'Spray-coating', 'Spin-coating | Evaporation | Evaporation', 'Doctor blading | Sandwhiching', 'Sandwiching', 'Painting', 'Mechanical clipping', 'Evaporation | Spray-coating', 'Candel burning | Sandwiching', 'Doctor blading | Lamination', 'Screen printing | Screen printing', 'Evaporation | DC Magnetron Sputtering | E-beam evaporation'])))

    deposition_aggregation_state_of_reactants = Quantity(
        type=str,
        shape=[],
        description="""
    The physical state of the reactants.
- The three basic categories are Solid/Liquid/Gas
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the aggregation state associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Most cases are clear cut, e.g. spin-coating involves species in solution and evaporation involves species in gas phase. For less clear-cut cases, consider where the reaction really is happening as in:
o For a spray-coating procedure, it is droplets of liquid that enters the substrate (thus a liquid phase reaction)
o For sputtering and thermal evaporation, it is species in gas phase that reaches the substrate (thus a gas phase reaction)
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Liquid
Gas | Liquid
Liquid | Liquid >> Liquid
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Solid', 'Liquid', 'Unknown', 'Gas | Gas | Gas | Gas | Gas', 'Solid | Gas', 'Solid | Solid | Solid', 'Liquid | Gas | Gas', 'Liquid | Solid', 'Gas', 'Liquid >> Solid', 'Gas | Gas', 'Liquid | Gas'])))

    deposition_synthesis_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The synthesis atmosphere.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Vacuum
Vacuum | N2
Air | Ar; H2O >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['N2 | Vacuum', 'Vacuum | Vacuum', 'Unknown', 'Air', 'Vacuum', 'Vacuum | Vacuum | Vacuum | Vacuum | Vacuum', 'Air | Vacuum | Vacuum', 'Air >> Air', 'Ar', 'Ar | O2'])))

    deposition_synthesis_atmosphere_pressure_total = Quantity(
        type=str,
        shape=[],
        description="""
    The total gas pressure during each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- Pressures can be stated in different units suited for different situations. Therefore, specify the unit. The preferred units are:
o atm, bar, mbar, mmHg, Pa, torr, psi
- If a pressure is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 100 pa and not 80-120 pa.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 1 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.0006 Pa', '0.000009 mbar', '0.0001 Torr', '0.01 Torr', '0.0000002 Torr', '0.000005 mbar', '0.000002 bar', '0.0001 mbar', '0.0001 Pa | 0.0001 Pa', '0.0005 Pa', '0.00001 mbar', '0.0004 Pa', '0.00005 Torr', '0.0003 bar', '0.0000019 Torr', '0.1 Torr', '0.000001 mbar', '1 atm >> 0.2 MPa', '0.0000001 Torr', '0.00001 Pa', '0.000001 bar', '0.000000001 bar', '0.000006 Torr', '0.000007 Torr', '1.2 mTorr', '0.006 Torr', '0.000004 Torr', '10 E-7Torr', '0.00005 mbar', '0.00001 Torr', '2 e-05', '2 E-10Torr', '0.00005 Pa', '0.000008 bar', '0.00000003 Torr', '0.0001 Pa', '0.000001 Torr', '0.0002 Pa', '0.0000048 Torr', '0.000002 Torr', '0.0003 Pa', '0.00003 mbar', '0.0000001 mbar', '0.000002 mbar', '1 atm', '0.000005 Torr'])))

    deposition_synthesis_atmosphere_pressure_partial = Quantity(
        type=str,
        shape=[],
        description="""
    The partial pressures for the gases present during each reaction step.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the partial pressures and separate them with semicolons, as in (A; B). The list of partial pressures must line up with the gases they describe.
- In cases where no gas mixtures are used, this field will be the same as the previous filed.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 0.99 atm; 0.01 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '0.0001 mbar', '0.0001 Pa | 0.0001 Pa', '0.00001 mbar', '0.0004 Pa', '0.0003 bar', '0.000001 mbar', '1 atm >> 0.2 MPa', '0.00001 Pa', '0.000006 Torr', '0.000007 Torr', '0.006 Torr', '0.000004 Torr', '0.00001 Torr', '0.000001 Torr', '0.000002 Torr', '0.00003 mbar', '0.0003 Pa', '0.0000001 mbar', '0.000002 mbar', '1 atm', '0.000005 Torr'])))

    deposition_synthesis_atmosphere_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relative humidity during each deposition step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the relative humidity associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns
- If the relative humidity for a step is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 35 and not 30-40.
Example
35
0 | 20
nan >> 25 | 0
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '25.0', '30 >> 30', '30.0', '50 | nan | nan', '50.0'])))

    deposition_solvents = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in each deposition procedure for each layer in the stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvents associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the solvents in alphabetic order and separate them with semicolons, as in (A; B)
- The number and order of layers and deposition steps must line up with the previous columns.
- For non-liquid processes with no solvents, state the solvent as ‘none’
- If the solvent is not known, state this as ‘Unknown’
- Use common abbreviations when appropriate but spell it out when risk for confusion
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
none
Acetonitile; Ethanol | Chlorobenzene
none >> Ethanol; Methanol; H2O | DMF; DMSO
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown', 'none', 'Methanol | none | none', 'Ethyl cellulose; Terpineol', 'IPA | none', 'Ethyl cellulose; Terpineol | Unknown'])))

    deposition_solvents_mixing_ratios = Quantity(
        type=str,
        shape=[],
        description="""
    The mixing ratios for mixed solvents
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent mixing ratios associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- For pure solvents, state the mixing ratio as 1
- For non-solvent processes, state the mixing ratio as 1
- For unknown mixing ratios, state the mixing ratio as ‘nan’
- For solvent mixtures, i.e. A and B, state the mixing ratios by using semicolons, as in (VA; VB)
- The preferred metrics is the volume ratios. If that is not available, mass or mol ratios can be used instead, but it the analysis the mixing ratios will be assumed to be based on volumes.
Example
1
4; 1 | 1
1 >> 5; 2; 0.3 | 2; 1
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '1', '1 | 1', '1 | nan | nan'])))

    deposition_solvents_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of all the solvents.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For non-liquid processes with no solvents, mark the supplier as ‘none’
- If the supplier for a solvent is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Sigma Aldrich
Sigma Aldrich; Fisher | Acros
none >> Sigma Aldrich; Sigma Aldrich | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Shanghai MaterWin New Material', 'Guangzhou Seaside Technology', 'Unknown'])))

    deposition_solvents_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the solvents used.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For non-liquid processes with no solvents, state the purity as ‘none’
- If the purity for a solvent is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
Puris; Puris| Tecnical
none >> Pro analysis; Pro analysis | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    deposition_reaction_solutions_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    The non-solvent precursor chemicals used in each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemicals associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several compounds, e.g. A and B, list the associated compounds in alphabetic order and separate them with semicolons, as in (A; B)
- Note that also dopants/additives should be included
- When several precursor solutions are made and mixed before the reaction step, it is the properties of the final mixture used in the reaction we here describe.
- The number and order of layers and reaction steps must line up with the previous columns.
- For gas phase reactions, state the reaction gases as if they were in solution.
- For solid-state reactions, state the compounds as if they were in solution.
- For reaction steps involving only pure solvents, state this as ‘none’
- If the compounds for a deposition step is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Au
CuI
Ag
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Al', 'Graphite', 'Mg | Ag', 'Carbon Paste', 'ITO', 'AgAl', 'Carbon | nan | nan', 'IZO', 'Au', 'Ag | Au', 'Cu', 'Ag | MoO3', 'Cu | Ag', 'Cr | Au', 'Ag-nanocubes | Ag | MoO3', 'Carbon | Ag', 'MoO3 | Al', 'Ag', 'IZTO', 'PEDOT:PSS', 'AZO', 'Adhesive; PEDOT:PSS | PET; Ni-mesh', 'PTCBI | Ag | WO3 | PTCBI | Ag', 'AZO-np | Ag', 'Carbon', 'PEDOT:PSS | Al'])))

    deposition_reaction_solutions_compounds_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemical suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For gas phase reactions, state the suppliers for the gases or the targets/evaporation sources that are evaporated/sputtered/etc.
- For solid state reactions, state the suppliers for the compounds in the same way.
- For reaction steps involving only pure solvents, state the supplier as ‘none’ (as that that is entered in a separate filed)
- For chemicals that are lab made, state that as “Lab made” or “Lab made (name of lab)”
- If the supplier for a compound is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Dysole; Sigma Aldrich; Dyenamo; Sigma Aldrich
Sigma Aldrich; Fisher | Acros
Lab made (EPFL) | Sigma Aldrich >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['DongDaLai company', 'Unknown', 'Solaronix', 'Styccobond; Agfa | Epigem', 'Heraeus', 'Guangzhou Seaside Technology', 'Sigma Aldrich | Unknown', 'Sigma Aldrich', 'Ulet', 'Shanghai MaterWin New Materials Co., Ltd'])))

    deposition_reaction_solutions_compounds_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the compound purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, i.e. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For reaction steps involving only pure solvents, state this as ‘none’ (as that is stated in another field)
- If the purity for a compound is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
99.999; Puris| Tecnical
Unknown >> Pro analysis; Pro analysis | none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown', '99.99'])))

    deposition_reaction_solutions_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the non-solvent precursor chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the concentrations associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated concentrations and separate them with semicolons, as in (A; B)
- The order of the compounds must be the same as in the previous filed.
- For reaction steps involving only pure solvents, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used. When possible, use one of the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml, wt%, mol%, vol%, ppt, ppm, ppb
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
0.2 M; 0.15 M| 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '57.2 wt%; 42.8 wt% | nan', '8 mg/ml | nan'])))

    deposition_reaction_solutions_volumes = Quantity(
        type=str,
        shape=[],
        description="""
    The volume of the reaction solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the volumes associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The volumes refer the volumes used, not the volume of the stock solutions. Thus if 0.15 ml of a solution is spin-coated, the volume is 0.15 ml
- For reaction steps without solvents, state the volume as ‘nan’
- When volumes are unknown, state that as ‘nan’
Example
0.1
0.1 >> 0.05 | 0.05
nan | 0.15
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['15.0 | Unknown | Unknown', 'Unknown'])))

    deposition_reaction_solutions_age = Quantity(
        type=str,
        shape=[],
        description="""
    The age of the solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the age of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- As a general guideline, the age refers to the time from the preparation of the final precursor mixture to the reaction procedure.
- When the age of a solution is not known, state that as ‘nan’
- For reaction steps where no solvents are involved, state this as ‘nan’
- For solutions that is stored a long time, an order of magnitude estimate is adequate.
Example
2
0.25 |1000 >> 10000
nan | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    deposition_reaction_solutions_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the reaction solutions.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a reaction solution undergoes a temperature program, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons, e.g. 25; 100
- When the temperature of a solution is unknown, state that as ‘nan’
- For reaction steps where no solvents are involved, state the temperature of the gas or the solid if that make sense. Otherwise state this as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume an undetermined room temperature to be 25
Example
25
100; 50 | 25
nan | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['25', 'Unknown', '25 | 25'])))

    deposition_substrate_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the substrate.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the substrates (i.e. the last deposited layer) associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The temperature of the substrate refers to the temperature when the deposition of the layer is occurring.
- If a substrate undergoes a temperature program before the deposition, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- When the temperature of a substrate is not known, state that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume that an undetermined room temperature is 25
Example
25
nan
125; 325; 375; 450 | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['25', 'Unknown', '15', '100', '100 | 25', '60', '40', '22', '25 | 25', '80'])))

    deposition_thermal_annealing_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperatures of the thermal annealing program associated with depositing the layers
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing temperatures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- If no thermal annealing is occurring after the deposition of a layer, state that by stating the room temperature (assumed to 25°C if not further specified)
- If the thermal annealing program is not known, state that by ‘nan’
Example
25
50 | nan
450 | 125; 325; 375; 450 >> 125; 325; 375; 450
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '25', '350.0', '85', '120 >> 120', '400', '80', 'Unknown', '450.0', '100', '120', '450 | 25', '60', '250.0', '60; 120', '450', '150.0', '25; 100', '100 | Unknown', '550.0'])))

    deposition_thermal_annealing_time = Quantity(
        type=str,
        shape=[],
        description="""
    The time program associated to the thermal annealing program.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing times associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the associated times at those temperatures and separate them with semicolons.
- The annealing times must align in terms of layers¸ reaction steps and annealing temperatures in the previous filed.
- If a time is not known, state that by ‘nan’
- If no thermal annealing is occurring after the deposition of a layer, state that by ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 20 and not 10-30.
Example
nan
60 | 1000
30 | 5; 5; 5; 30 >> 5; 5; 5; 30
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['40.0', 'Unknown', '30.0', '20.0', '15.0; 5.0', '100.0', '30.0; 30.0', '60.0', '15.0 >> 5.0', '10.0', '10.0 | Unknown', '15.0', '30.0 | Unknown'])))

    deposition_thermal_annealing_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere during thermal annealing
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each annelaing step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the atmosphere is a mixture of different gases, i.e. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas.
- This is often the same as the atmosphere under which the deposition is occurring, but not always.
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Vacuum | N2
Air | Ar >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown', 'Air', 'Vacuum', 'N2', 'Air >> Air'])))

    storage_time_until_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    The time between the back contact is finalised and the next layer is deposited
- If there are uncertainties, only state the best estimate, e.g. write 35 and not 20-50.
- If this is the last layer in the stack, state this as ‘nan’
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown', '24.0'])))

    storage_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere in which the sample with the finalised back contact is stored until the next deposition step or device performance measurement
Example
Air
N2
Vacuum
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Dry air', 'N2', 'Unknown', 'Air'])))

    storage_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relive humidity under which the sample with the finalised back contact is stored until the next deposition step or device performance measurement
- If there are uncertainties, only state the best estimate, e.g write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '5.0', '10', '10.0'])))

    surface_treatment_before_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    Description of any type of surface treatment or other treatment the sample with the finalised back contact is stored until the next deposition step or device performance measurement
- If more than one treatment, list the treatments and separate them by a double forward angel bracket (‘ >> ‘)
- If no special treatment, state that as ‘none’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example:
none
Ar plasma
UV-ozone
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'EDA gas', 'DEDA gas', 'MEA immersion >> 125C 20 min', 'TETA gas'])))

    def normalize(self, archive, logger):
        add_solar_cell(archive)
        if self.stack_sequence:
            archive.results.properties.optoelectronic.solar_cell.back_contact = self.stack_sequence.split(' | ')


class Add(MSection):
    """
    A section to describe **additional layers** present in the device besides the
    *Substrate*, *ETL*, *Perovskite*, *HTL* and *back contact*.
    """

    lay_front = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if there is a functional layer below the substrate, i.e. on the opposite side of the substrate from with respect to the perovskite.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    lay_front_function = Quantity(
        type=str,
        shape=[],
        description="""
    The function of the additional layers on the substrate side
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If a layer has more than one function, e.g. A and B, list the functions in order and separate them with semicolons, as in (A; B)
Example:
A.R.C.
Back reflection
Down conversion
Encapsulation
Light management
Upconversion
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Antireflection', 'Unknown', 'Down conversion', 'A.R.C.', 'Light management'])))

    lay_front_stack_sequence = Quantity(
        type=str,
        shape=[],
        description="""
    The stack sequence describing the additional layers on the substrate side
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If two materials, e.g. A and B, are mixed in one layer, list the materials in alphabetic order and separate them with semicolons, as in (A; B)
- Use common abbreviations when appropriate but spell it out if risk for confusion.
- There are separate filed for doping. Indicate doping with colons. E.g. wither aluminium doped NiO-np as Al:NiO-np
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example:
MgF2
Au-np
NaYF4:Eu-np
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['ZnSe-QDs', 'N-Graphene-QDs', 'Ag-np', 'Moth eye PDMS', 'Polyimide', 'PDMS', 'Unknown', 'NaYF4:Eu-np', 'NaF', 'Eu(TTA)2(Phen)MAA', 'MgF2', 'Y2O3:Eu3', 'CdSeS-QDs', 'Y2O3:Eu3 | Au-np', 'Mica', 'Eu-complex LDL', 'textured antireflective foil', 'Phosphor-in-glass', 'INVAR', 'Mn:CsPbCl3-QDs', 'LiF'])))

    lay_front_thickness_list = Quantity(
        type=str,
        shape=[],
        description="""
    A list of thicknesses of the individual layers in the stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous filed.
- State thicknesses in nm
- Every layer in the stack have a thickness. If it is unknown, state this as ‘nan’
- If there are uncertainties, state the best estimate, e.g write 100 and not 90-110
Example
200
nan |250
100 | 5 | 8
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '90.0', '100.0', '50.0', '80.0'])))

    lay_front_additives_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    List of the dopants and additives that are in each layer of the HTL-stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous fields.
- If several dopants/additives, e.g. A and B, are present in one layer, list the dopants/additives in alphabetic order and separate them with semicolons, as in (A; B)
- If no dopants/additives, state that as “Undoped”
- If the doping situation is unknown, stat that as‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template, even if to most common back contacts is undoped metals
Example
CuS
B; P
Au-np | Undoped
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_additives_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the dopants/additives.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If more than one dopant/additive in the layer, e.g. A and B, separate the concentration for each dopant/additive with semicolons, as in (A; B)
- For each dopant/additive in the layer, state the concentration.
- The order of the dopants/additives must be the same as in the previous filed.
- For layers with no dopants/additives, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used.
- The preferred way to state the concentration of a dopant/additive is to refer to the amount in the final product, i.e. the material in the layer. When possible, use on the preferred units
o wt%, mol%, vol%, ppt, ppm, ppb
- When the concentration of the dopant/additive in the final product is unknown, but where the concentration of the dopant/additive in the solution is known, state that concentration instead. When possible, use on the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
5 vol%; nan | 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | 0.3 M
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_deposition_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    The deposition procedures for the HTL-stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate them by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Thermal annealing is generally not considered as an individual reaction step. The philosophy behind this is that every deposition step has a thermal history, which is specified in a separate filed. In exceptional cases with thermal annealing procedures clearly disconnected from other procedures, state ‘Thermal annealing’ as a separate reaction step.
- Please read the instructions under “Perovskite. Deposition. Procedure” for descriptions and distinctions between common deposition procedures and how they should be labelled for consistency in the database.
- A few additional clarifications:
- Lamination
o A readymade film is transferred directly to the device stack. A rather broad concept. An everyday kitchen related example of lamination would eb to place a thin plastic film over a slice of pie.
- Sandwiching
o When a readymade top stack simply is placed on top of the device stack. Could be held together with clams. The typical example is a when a “Carbon | FTO | SLG” is placed on top of the device stack. Standard procedure in the DSSC filed.
Example
Evaporation
Evaporation | Evaporation
Doctor blading
Screen printing
Sputtering
Lamination
E-beam evaporation
Sandwiching
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Evaporation', 'Unknown'])))

    lay_front_deposition_aggregation_state_of_reactants = Quantity(
        type=str,
        shape=[],
        description="""
    The physical state of the reactants.
- The three basic categories are Solid/Liquid/Gas
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the aggregation state associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Most cases are clear cut, e.g. spin-coating involves species in solution and evaporation involves species in gas phase. For less clear-cut cases, consider where the reaction really is happening as in:
o For a spray-coating procedure, it is droplets of liquid that enters the substrate (thus a liquid phase reaction)
o For sputtering and thermal evaporation, it is species in gas phase that reaches the substrate (thus a gas phase reaction)
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Liquid
Gas | Liquid
Liquid | Liquid >> Liquid
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_synthesis_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The synthesis atmosphere.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Vacuum
Vacuum | N2
Air | Ar; H2O >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_synthesis_atmosphere_pressure_total = Quantity(
        type=str,
        shape=[],
        description="""
    The total gas pressure during each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- Pressures can be stated in different units suited for different situations. Therefore, specify the unit. The preferred units are:
o atm, bar, mbar, mmHg, Pa, torr, psi
- If a pressure is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 100 pa and not 80-120 pa.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 1 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_deposition_synthesis_atmosphere_pressure_partial = Quantity(
        type=str,
        shape=[],
        description="""
    The partial pressures for the gases present during each reaction step.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the partial pressures and separate them with semicolons, as in (A; B). The list of partial pressures must line up with the gases they describe.
- In cases where no gas mixtures are used, this field will be the same as the previous filed.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 0.99 atm; 0.01 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_deposition_synthesis_atmosphere_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relative humidity during each deposition step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the relative humidity associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns
- If the relative humidity for a step is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 35 and not 30-40.
Example
35
0 | 20
nan >> 25 | 0
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_deposition_solvents = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in each deposition procedure for each layer in the stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvents associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the solvents in alphabetic order and separate them with semicolons, as in (A; B)
- The number and order of layers and deposition steps must line up with the previous columns.
- For non-liquid processes with no solvents, state the solvent as ‘none’
- If the solvent is not known, state this as ‘Unknown’
- Use common abbreviations when appropriate but spell it out when risk for confusion
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
none
Acetonitile; Ethanol | Chlorobenzene
none >> Ethanol; Methanol; H2O | DMF; DMSO
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_solvents_mixing_ratios = Quantity(
        type=str,
        shape=[],
        description="""
    The mixing ratios for mixed solvents
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent mixing ratios associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- For pure solvents, state the mixing ratio as 1
- For non-solvent processes, state the mixing ratio as 1
- For unknown mixing ratios, state the mixing ratio as ‘nan’
- For solvent mixtures, i.e. A and B, state the mixing ratios by using semicolons, as in (VA; VB)
- The preferred metrics is the volume ratios. If that is not available, mass or mol ratios can be used instead, but it the analysis the mixing ratios will be assumed to be based on volumes.
Example
1
4; 1 | 1
1 >> 5; 2; 0.3 | 2; 1
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_deposition_solvents_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of all the solvents.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For non-liquid processes with no solvents, mark the supplier as ‘none’
- If the supplier for a solvent is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Sigma Aldrich
Sigma Aldrich; Fisher | Acros
none >> Sigma Aldrich; Sigma Aldrich | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_solvents_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the solvents used.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For non-liquid processes with no solvents, state the purity as ‘none’
- If the purity for a solvent is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
Puris; Puris| Tecnical
none >> Pro analysis; Pro analysis | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_reaction_solutions_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    The non-solvent precursor chemicals used in each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemicals associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several compounds, e.g. A and B, list the associated compounds in alphabetic order and separate them with semicolons, as in (A; B)
- Note that also dopants/additives should be included
- When several precursor solutions are made and mixed before the reaction step, it is the properties of the final mixture used in the reaction we here describe.
- The number and order of layers and reaction steps must line up with the previous columns.
- For gas phase reactions, state the reaction gases as if they were in solution.
- For solid-state reactions, state the compounds as if they were in solution.
- For reaction steps involving only pure solvents, state this as ‘none’
- If the compounds for a deposition step is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Au
CuI
Ag
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_deposition_reaction_solutions_compounds_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemical suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For gas phase reactions, state the suppliers for the gases or the targets/evaporation sources that are evaporated/sputtered/etc.
- For solid state reactions, state the suppliers for the compounds in the same way.
- For reaction steps involving only pure solvents, state the supplier as ‘none’ (as that that is entered in a separate filed)
- For chemicals that are lab made, state that as “Lab made” or “Lab made (name of lab)”
- If the supplier for a compound is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Dysole; Sigma Aldrich; Dyenamo; Sigma Aldrich
Sigma Aldrich; Fisher | Acros
Lab made (EPFL) | Sigma Aldrich >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_reaction_solutions_compounds_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the compound purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, i.e. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For reaction steps involving only pure solvents, state this as ‘none’ (as that is stated in another field)
- If the purity for a compound is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
99.999; Puris| Tecnical
Unknown >> Pro analysis; Pro analysis | none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_reaction_solutions_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the non-solvent precursor chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the concentrations associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated concentrations and separate them with semicolons, as in (A; B)
- The order of the compounds must be the same as in the previous filed.
- For reaction steps involving only pure solvents, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used. When possible, use one of the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml, wt%, mol%, vol%, ppt, ppm, ppb
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
0.2 M; 0.15 M| 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_deposition_reaction_solutions_volumes = Quantity(
        type=str,
        shape=[],
        description="""
    The volume of the reaction solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the volumes associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The volumes refer the volumes used, not the volume of the stock solutions. Thus if 0.15 ml of a solution is spin-coated, the volume is 0.15 ml
- For reaction steps without solvents, state the volume as ‘nan’
- When volumes are unknown, state that as ‘nan’
Example
0.1
0.1 >> 0.05 | 0.05
nan | 0.15
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_reaction_solutions_age = Quantity(
        type=str,
        shape=[],
        description="""
    The age of the solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the age of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- As a general guideline, the age refers to the time from the preparation of the final precursor mixture to the reaction procedure.
- When the age of a solution is not known, state that as ‘nan’
- For reaction steps where no solvents are involved, state this as ‘nan’
- For solutions that is stored a long time, an order of magnitude estimate is adequate.
Example
2
0.25 |1000 >> 10000
nan | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_reaction_solutions_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the reaction solutions.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a reaction solution undergoes a temperature program, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons, e.g. 25; 100
- When the temperature of a solution is unknown, state that as ‘nan’
- For reaction steps where no solvents are involved, state the temperature of the gas or the solid if that make sense. Otherwise state this as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume an undetermined room temperature to be 25
Example
25
100; 50 | 25
nan | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_substrate_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the substrate.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the substrates (i.e. the last deposited layer) associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The temperature of the substrate refers to the temperature when the deposition of the layer is occurring.
- If a substrate undergoes a temperature program before the deposition, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- When the temperature of a substrate is not known, state that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume that an undetermined room temperature is 25
Example
25
nan
125; 325; 375; 450 | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_thermal_annealing_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperatures of the thermal annealing program associated with depositing the layers
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing temperatures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- If no thermal annealing is occurring after the deposition of a layer, state that by stating the room temperature (assumed to 25°C if not further specified)
- If the thermal annealing program is not known, state that by ‘nan’
Example
25
50 | nan
450 | 125; 325; 375; 450 >> 125; 325; 375; 450
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_thermal_annealing_time = Quantity(
        type=str,
        shape=[],
        description="""
    The time program associated to the thermal annealing program.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing times associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the associated times at those temperatures and separate them with semicolons.
- The annealing times must align in terms of layers¸ reaction steps and annealing temperatures in the previous filed.
- If a time is not known, state that by ‘nan’
- If no thermal annealing is occurring after the deposition of a layer, state that by ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 20 and not 10-30.
Example
nan
60 | 1000
30 | 5; 5; 5; 30 >> 5; 5; 5; 30
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_deposition_thermal_annealing_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere during thermal annealing
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each annelaing step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the atmosphere is a mixture of different gases, i.e. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas.
- This is often the same as the atmosphere under which the deposition is occurring, but not always.
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Vacuum | N2
Air | Ar >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_storage_time_until_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    The time between the back contact is finalised and the next layer is deposited
- If there are uncertainties, only state the best estimate, e.g. write 35 and not 20-50.
- If this is the last layer in the stack, state this as ‘nan’
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_storage_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere in which the sample with the finalised back contact is stored until the next deposition step or device performance measurement
Example
Air
N2
Vacuum
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_front_storage_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relive humidity under which the sample with the finalised back contact is stored until the next deposition step or device performance measurement
- If there are uncertainties, only state the best estimate, e.g write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_front_surface_treatment_before_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    Description of any type of surface treatment or other treatment the sample with the finalised back contact is stored until the next deposition step or device performance measurement
- If more than one treatment, list the treatments and separate them by a double forward angel bracket (‘ >> ‘)
- If no special treatment, state that as ‘none’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example:
none
Ar plasma
UV-ozone
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back = Quantity(
        type=bool,
        shape=[],
        description="""
    TRUE if there is a functional layer above the back contact, i.e. layers deposited after the back contact has been finalised.
                    """,
        a_eln=dict(
            component='BoolEditQuantity'))

    lay_back_function = Quantity(
        type=str,
        shape=[],
        description="""
    The function of the additional layers on the backcontact side.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If a layer has more than one function, e.g. A and B, list the functions in order and separate them with semicolons, as in (A; B)
Example:
A.R.C.
Back reflection
Down conversion
Encapsulation
Light management
Upconversion
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', 'Upconversion', 'Dielectric mirror', 'Antireflection'])))

    lay_back_stack_sequence = Quantity(
        type=str,
        shape=[],
        description="""
    The stack sequence describing the additional layers on the backcontact side.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If two materials, e.g. A and B, are mixed in one layer, list the materials in alphabetic order and separate them with semicolons, as in (A; B)
- Use common abbreviations when appropriate but spell it out if risk for confusion.
- There are now separate filed for doping. Indicate doping with colons. E.g. wither aluminium doped NiO-np as Al:NiO-np
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['MgF2', 'Unknown'])))

    lay_back_thickness_list = Quantity(
        type=str,
        shape=[],
        description="""
    A list of thicknesses of the individual layers in the stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous filed.
- State thicknesses in nm
- Every layer in the stack have a thickness. If it is unknown, state this as ‘nan’
- If there are uncertainties, state the best estimate, e.g write 100 and not 90-110
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['', '105.0', '90.0'])))

    lay_back_additives_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    List of the dopants and additives that are in each layer of the HTL-stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- The layers must line up with the previous fields.
- If several dopants/additives, e.g. A and B, are present in one layer, list the dopants/additives in alphabetic order and separate them with semicolons, as in (A; B)
- If no dopants/additives, state that as “Undoped”
- If the doping situation is unknown, stat that as‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template, even if to most common back contacts is undoped metals
Example
CuS
B; P
Au-np | Undoped
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_additives_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the dopants/additives.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- If more than one dopant/additive in the layer, e.g. A and B, separate the concentration for each dopant/additive with semicolons, as in (A; B)
- For each dopant/additive in the layer, state the concentration.
- The order of the dopants/additives must be the same as in the previous filed.
- For layers with no dopants/additives, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used.
- The preferred way to state the concentration of a dopant/additive is to refer to the amount in the final product, i.e. the material in the layer. When possible, use on the preferred units
o wt%, mol%, vol%, ppt, ppm, ppb
- When the concentration of the dopant/additive in the final product is unknown, but where the concentration of the dopant/additive in the solution is known, state that concentration instead. When possible, use on the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
5 vol%; nan | 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | 0.3 M
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_deposition_procedure = Quantity(
        type=str,
        shape=[],
        description="""
    The deposition procedures for the HTL-stack.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate them by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Thermal annealing is generally not considered as an individual reaction step. The philosophy behind this is that every deposition step has a thermal history, which is specified in a separate filed. In exceptional cases with thermal annealing procedures clearly disconnected from other procedures, state ‘Thermal annealing’ as a separate reaction step.
- Please read the instructions under “Perovskite. Deposition. Procedure” for descriptions and distinctions between common deposition procedures and how they should be labelled for consistency in the database.
- A few additional clarifications:
- Lamination
o A readymade film is transferred directly to the device stack. A rather broad concept. An everyday kitchen related example of lamination would eb to place a thin plastic film over a slice of pie.
- Sandwiching
o When a readymade top stack simply is placed on top of the device stack. Could be held together with clams. The typical example is a when a “Carbon | FTO | SLG” is placed on top of the device stack. Standard procedure in the DSSC filed.
Example
Evaporation
Evaporation | Evaporation
Doctor blading
Screen printing
Sputtering
Lamination
E-beam evaporation
Sandwiching
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Evaporation', 'Unknown'])))

    lay_back_deposition_aggregation_state_of_reactants = Quantity(
        type=str,
        shape=[],
        description="""
    The physical state of the reactants.
- The three basic categories are Solid/Liquid/Gas
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the aggregation state associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- Most cases are clear cut, e.g. spin-coating involves species in solution and evaporation involves species in gas phase. For less clear-cut cases, consider where the reaction really is happening as in:
o For a spray-coating procedure, it is droplets of liquid that enters the substrate (thus a liquid phase reaction)
o For sputtering and thermal evaporation, it is species in gas phase that reaches the substrate (thus a gas phase reaction)
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Liquid
Gas | Liquid
Liquid | Liquid >> Liquid
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_synthesis_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The synthesis atmosphere.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Vacuum
Vacuum | N2
Air | Ar; H2O >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_synthesis_atmosphere_pressure_total = Quantity(
        type=str,
        shape=[],
        description="""
    The total gas pressure during each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- Pressures can be stated in different units suited for different situations. Therefore, specify the unit. The preferred units are:
o atm, bar, mbar, mmHg, Pa, torr, psi
- If a pressure is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 100 pa and not 80-120 pa.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 1 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_deposition_synthesis_atmosphere_pressure_partial = Quantity(
        type=str,
        shape=[],
        description="""
    The partial pressures for the gases present during each reaction step.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the pressures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the synthesis atmosphere is a mixture of different gases, e.g. A and B, list the partial pressures and separate them with semicolons, as in (A; B). The list of partial pressures must line up with the gases they describe.
- In cases where no gas mixtures are used, this field will be the same as the previous filed.
Example
1 atm
0.002 torr | 10000 Pa
nan >> 0.99 atm; 0.01 atm | 1 atm
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_deposition_synthesis_atmosphere_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relative humidity during each deposition step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the relative humidity associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns
- If the relative humidity for a step is not known, stat that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 35 and not 30-40.
Example
35
0 | 20
nan >> 25 | 0
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_deposition_solvents = Quantity(
        type=str,
        shape=[],
        description="""
    The solvents used in each deposition procedure for each layer in the stack
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvents associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the solvents in alphabetic order and separate them with semicolons, as in (A; B)
- The number and order of layers and deposition steps must line up with the previous columns.
- For non-liquid processes with no solvents, state the solvent as ‘none’
- If the solvent is not known, state this as ‘Unknown’
- Use common abbreviations when appropriate but spell it out when risk for confusion
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
none
Acetonitile; Ethanol | Chlorobenzene
none >> Ethanol; Methanol; H2O | DMF; DMSO
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_solvents_mixing_ratios = Quantity(
        type=str,
        shape=[],
        description="""
    The mixing ratios for mixed solvents
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent mixing ratios associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- For pure solvents, state the mixing ratio as 1
- For non-solvent processes, state the mixing ratio as 1
- For unknown mixing ratios, state the mixing ratio as ‘nan’
- For solvent mixtures, i.e. A and B, state the mixing ratios by using semicolons, as in (VA; VB)
- The preferred metrics is the volume ratios. If that is not available, mass or mol ratios can be used instead, but it the analysis the mixing ratios will be assumed to be based on volumes.
Example
1
4; 1 | 1
1 >> 5; 2; 0.3 | 2; 1
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_deposition_solvents_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of all the solvents.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For non-liquid processes with no solvents, mark the supplier as ‘none’
- If the supplier for a solvent is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Sigma Aldrich
Sigma Aldrich; Fisher | Acros
none >> Sigma Aldrich; Sigma Aldrich | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_solvents_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the solvents used.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the solvent purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solvent is a mixture of different solvents, e.g. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For non-liquid processes with no solvents, state the purity as ‘none’
- If the purity for a solvent is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
Puris; Puris| Tecnical
none >> Pro analysis; Pro analysis | Unknown
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_reaction_solutions_compounds = Quantity(
        type=str,
        shape=[],
        description="""
    The non-solvent precursor chemicals used in each reaction step
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemicals associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several compounds, e.g. A and B, list the associated compounds in alphabetic order and separate them with semicolons, as in (A; B)
- Note that also dopants/additives should be included
- When several precursor solutions are made and mixed before the reaction step, it is the properties of the final mixture used in the reaction we here describe.
- The number and order of layers and reaction steps must line up with the previous columns.
- For gas phase reactions, state the reaction gases as if they were in solution.
- For solid-state reactions, state the compounds as if they were in solution.
- For reaction steps involving only pure solvents, state this as ‘none’
- If the compounds for a deposition step is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Au
CuI
Ag
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_deposition_reaction_solutions_compounds_supplier = Quantity(
        type=str,
        shape=[],
        description="""
    The suppliers of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the non-solvent chemical suppliers associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated suppliers and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- For gas phase reactions, state the suppliers for the gases or the targets/evaporation sources that are evaporated/sputtered/etc.
- For solid state reactions, state the suppliers for the compounds in the same way.
- For reaction steps involving only pure solvents, state the supplier as ‘none’ (as that that is entered in a separate filed)
- For chemicals that are lab made, state that as “Lab made” or “Lab made (name of lab)”
- If the supplier for a compound is unknown, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Dysole; Sigma Aldrich; Dyenamo; Sigma Aldrich
Sigma Aldrich; Fisher | Acros
Lab made (EPFL) | Sigma Aldrich >> none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_reaction_solutions_compounds_purity = Quantity(
        type=str,
        shape=[],
        description="""
    The purity of the non-solvent chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the compound purities associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, i.e. A and B, list the associated purities and separate them with semicolons, as in (A; B)
- The number and order of layers, reaction steps, and solvents must line up with the previous columns.
- Use standard nomenclature for purities, e.g. pro analysis, puris, extra dry, etc.
- For reaction steps involving only pure solvents, state this as ‘none’ (as that is stated in another field)
- If the purity for a compound is not known, state this as ‘Unknown’
- This category was included after the projects initial phase wherefor the list of reported categories is short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
Pro analysis
99.999; Puris| Tecnical
Unknown >> Pro analysis; Pro analysis | none
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_reaction_solutions_concentrations = Quantity(
        type=str,
        shape=[],
        description="""
    The concentration of the non-solvent precursor chemicals.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the concentrations associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a solution contains several dissolved compounds, e.g. A and B, list the associated concentrations and separate them with semicolons, as in (A; B)
- The order of the compounds must be the same as in the previous filed.
- For reaction steps involving only pure solvents, state this as ‘none’
- When concentrations are unknown, state that as ‘nan’
- Concentrations can be stated in different units suited for different situations. Therefore, specify the unit used. When possible, use one of the preferred units
o M, mM, molal; g/ml, mg/ml, µg/ml, wt%, mol%, vol%, ppt, ppm, ppb
- For values with uncertainties, state the best estimate, e.g write 4 wt% and not 3-5 wt%.
Example
4 wt%
0.2 M; 0.15 M| 10 mg/ml
0.3 mol% | 2 mol%; 0.2 wt% | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_deposition_reaction_solutions_volumes = Quantity(
        type=str,
        shape=[],
        description="""
    The volume of the reaction solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the volumes associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The volumes refer the volumes used, not the volume of the stock solutions. Thus if 0.15 ml of a solution is spin-coated, the volume is 0.15 ml
- For reaction steps without solvents, state the volume as ‘nan’
- When volumes are unknown, state that as ‘nan’
Example
0.1
0.1 >> 0.05 | 0.05
nan | 0.15
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_reaction_solutions_age = Quantity(
        type=str,
        shape=[],
        description="""
    The age of the solutions
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the age of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- As a general guideline, the age refers to the time from the preparation of the final precursor mixture to the reaction procedure.
- When the age of a solution is not known, state that as ‘nan’
- For reaction steps where no solvents are involved, state this as ‘nan’
- For solutions that is stored a long time, an order of magnitude estimate is adequate.
Example
2
0.25 |1000 >> 10000
nan | nan
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_reaction_solutions_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the reaction solutions.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the solutions associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If a reaction solution undergoes a temperature program, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons, e.g. 25; 100
- When the temperature of a solution is unknown, state that as ‘nan’
- For reaction steps where no solvents are involved, state the temperature of the gas or the solid if that make sense. Otherwise state this as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume an undetermined room temperature to be 25
Example
25
100; 50 | 25
nan | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_substrate_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperature of the substrate.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the temperatures of the substrates (i.e. the last deposited layer) associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The temperature of the substrate refers to the temperature when the deposition of the layer is occurring.
- If a substrate undergoes a temperature program before the deposition, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- When the temperature of a substrate is not known, state that as ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- Assume that an undetermined room temperature is 25
Example
25
nan
125; 325; 375; 450 | 25 >> 25
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_thermal_annealing_temperature = Quantity(
        type=str,
        shape=[],
        description="""
    The temperatures of the thermal annealing program associated with depositing the layers
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing temperatures associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the temperatures (e.g. start, end, and other important points) and separate them with semicolons (e.g. 25; 100)
- For values with uncertainties, state the best estimate, e.g. write 120 and not 110-130.
- If no thermal annealing is occurring after the deposition of a layer, state that by stating the room temperature (assumed to 25°C if not further specified)
- If the thermal annealing program is not known, state that by ‘nan’
Example
25
50 | nan
450 | 125; 325; 375; 450 >> 125; 325; 375; 450
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_thermal_annealing_time = Quantity(
        type=str,
        shape=[],
        description="""
    The time program associated to the thermal annealing program.
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the annealing times associated to each reaction step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- If the thermal annealing involves a temperature program with multiple temperature stages, list the associated times at those temperatures and separate them with semicolons.
- The annealing times must align in terms of layers¸ reaction steps and annealing temperatures in the previous filed.
- If a time is not known, state that by ‘nan’
- If no thermal annealing is occurring after the deposition of a layer, state that by ‘nan’
- For values with uncertainties, state the best estimate, e.g. write 20 and not 10-30.
Example
nan
60 | 1000
30 | 5; 5; 5; 30 >> 5; 5; 5; 30
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_deposition_thermal_annealing_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere during thermal annealing
- Every layer should be separated by a space, a vertical bar, and a space, i.e. (‘ | ‘)
- When more than one reaction step, separate the atmospheres associated to each annelaing step by a double forward angel bracket with one blank space on both sides (‘ >> ‘)
- The number and order of layers and deposition steps must line up with the previous columns.
- If the atmosphere is a mixture of different gases, i.e. A and B, list the gases in alphabetic order and separate them with semicolons, as in (A; B)
- “Dry air” represent air with low relative humidity but where the relative humidity is not known
- “Ambient” represent air where the relative humidity is not known. For ambient conditions where the relative humidity is known, state this as “Air”
- “Vacuum” (of unspecified pressure) is for this purpose considered as an atmospheric gas.
- This is often the same as the atmosphere under which the deposition is occurring, but not always.
- This category was included after the projects initial phase wherefor the list of reported categories is
short. Thus, be prepared to expand the given list of alternatives in the data template.
Example
N2
Vacuum | N2
Air | Ar >> Ar
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_storage_time_until_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    The time between the back contact is finalised and the next layer is deposited
- If there are uncertainties, only state the best estimate, e.g. write 35 and not 20-50.
- If this is the last layer in the stack, state this as ‘nan’
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_storage_atmosphere = Quantity(
        type=str,
        shape=[],
        description="""
    The atmosphere in which the sample with the finalised back contact is stored until the next deposition step or device performance measurement
Example
Air
N2
Vacuum
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=['Unknown'])))

    lay_back_storage_relative_humidity = Quantity(
        type=str,
        shape=[],
        description="""
    The relive humidity under which the sample with the finalised back contact is stored until the next deposition step or device performance measurement
- If there are uncertainties, only state the best estimate, e.g write 35 and not 20-50.
                    """,
        a_eln=dict(
            component='EnumEditQuantity', props=dict(suggestions=[''])))

    lay_back_surface_treatment_before_next_deposition_step = Quantity(
        type=str,
        shape=[],
        description="""
    Description of any type of surface treatment or other treatment the sample with the finalised back contact is stored until the next deposition step or device performance measurement
- If more than one treatment, list the treatments and separate them by a double forward angel bracket (‘ >> ‘)
- If no special treatment, state that as ‘none’
- This category was included after the projects initial ph