"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImageBuilder = exports.ImageBuilderComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
const dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
function uniqueName(scope) {
    return cdk.Names.uniqueResourceName(scope, { maxLength: 126, separator: '-', allowedSpecialCharacters: '_-' });
}
class ImageBuilderObjectBase extends cdk.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    version(type, name, data) {
        return new aws_cdk_lib_1.CustomResource(this, 'Version', {
            serviceToken: this.versionFunction().functionArn,
            resourceType: `Custom::ImageBuilder-${type}-Version`,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            properties: {
                ObjectType: type,
                ObjectName: name,
                VersionedData: data,
            },
        }).ref;
    }
    versionFunction() {
        return utils_1.BundledNodejsFunction.singleton(this, 'aws-image-builder-versioner', {
            description: 'Custom resource handler that bumps up Image Builder versions',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'imagebuilder:ListComponents',
                        'imagebuilder:ListContainerRecipes',
                        'imagebuilder:ListImageRecipes',
                    ],
                    resources: ['*'],
                }),
            ],
        });
    }
}
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 */
class ImageBuilderComponent extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.assets = [];
        this.platform = props.platform;
        let steps = [];
        if (props.assets) {
            let inputs = [];
            let extractCommands = [];
            for (const asset of props.assets) {
                this.assets.push(asset.asset);
                if (asset.asset.isFile) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: asset.path,
                    });
                }
                else if (asset.asset.isZipArchive) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: `${asset.path}.zip`,
                    });
                    if (props.platform === 'Windows') {
                        extractCommands.push('$ErrorActionPreference = \'Stop\'');
                        extractCommands.push(`Expand-Archive "${asset.path}.zip" -DestinationPath "${asset.path}"`);
                        extractCommands.push(`del "${asset.path}.zip"`);
                    }
                    else {
                        extractCommands.push(`unzip "${asset.path}.zip" -d "${asset.path}"`);
                        extractCommands.push(`rm "${asset.path}.zip"`);
                    }
                }
                else {
                    throw new Error(`Unknown asset type: ${asset.asset}`);
                }
            }
            steps.push({
                name: 'Download',
                action: 'S3Download',
                inputs,
            });
            if (extractCommands.length > 0) {
                steps.push({
                    name: 'Extract',
                    action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                    inputs: {
                        commands: extractCommands,
                    },
                });
            }
        }
        steps.push({
            name: 'Run',
            action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
            inputs: {
                commands: props.commands,
            },
        });
        const data = {
            name: props.displayName,
            description: props.description,
            schemaVersion: '1.0',
            phases: [
                {
                    name: 'build',
                    steps,
                },
            ],
        };
        const name = uniqueName(this);
        const component = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, 'Component', {
            name: name,
            platform: props.platform,
            version: this.version('Component', name, {
                platform: props.platform,
                data,
            }),
            data: JSON.stringify(data),
        });
        this.arn = component.attrArn;
    }
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee) {
        for (const asset of this.assets) {
            asset.grantRead(grantee);
        }
    }
}
exports.ImageBuilderComponent = ImageBuilderComponent;
_a = JSII_RTTI_SYMBOL_1;
ImageBuilderComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ImageBuilderComponent", version: "0.5.1" };
/**
 * Image builder recipe for a Docker container image.
 */
class ContainerRecipe extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = uniqueName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnContainerRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ContainerRecipe', name, {
                platform: props.platform,
                components,
            }),
            // TODO mcr.microsoft.com/windows/servercore:ltsc2019
            parentImage: 'arn:aws:imagebuilder:us-east-1:aws:image/windows-server-2019-x86-core-ltsc2019-amd64/2020.12.8',
            components,
            containerType: 'DOCKER',
            targetRepository: {
                service: 'ECR',
                repositoryName: props.targetRepository.repositoryName,
            },
            dockerfileTemplateData: dockerfileTemplate,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An image builder that uses Image Builder to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * The CodeBuild builder is better and faster. Only use this one if you have no choice. For example, if you need Windows containers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation. The default components work with CodeBuild and Fargate.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new ContainerImageBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * new CodeBuildRunner(this, 'CodeBuild provider', {
 *     label: 'windows-codebuild',
 *     imageBuilder: builder,
 * });
 * ```
 */
class ContainerImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.components = [];
        // set platform
        this.architecture = props?.architecture ?? common_1.Architecture.X86_64;
        if (!this.architecture.is(common_1.Architecture.X86_64)) {
            throw new Error(`Unsupported architecture: ${this.architecture}. Consider CodeBuild for faster image builds.`);
        }
        this.os = props?.os ?? common_1.Os.LINUX;
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.platform = 'Windows';
        }
        else {
            throw new Error(`Unsupported OS: ${this.os}. Consider CodeBuild for faster image builds.`);
        }
        // set builder options
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (props?.vpc && props?.subnetSelection) {
            this.subnetId = props.vpc.selectSubnets(props.subnetSelection).subnetIds[0];
        }
        if (props?.securityGroup) {
            this.securityGroupIds = [props.securityGroup.securityGroupId];
        }
        this.instanceTypes = [props?.instanceType?.toString() ?? 'm5.large'];
        this.description = `Build image for GitHub Actions runner ${this.node.path} (${this.os.name}/${this.architecture.name})`;
        this.logRetention = props?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        // runner version
        this.runnerVersion = props?.runnerVersion ?? common_1.RunnerVersion.latest();
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove all but the latest image',
                    tagStatus: aws_ecr_1.TagStatus.ANY,
                    maxImageCount: 1,
                },
            ],
        });
        // add all basic components
        this.addBaseWindowsComponents();
    }
    addBaseWindowsComponents() {
        this.addComponent(new ImageBuilderComponent(this, 'AWS CLI', {
            platform: 'Windows',
            displayName: 'AWS CLI',
            description: 'Install latest version of AWS CLI',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
            ],
        }));
        this.addComponent(new ImageBuilderComponent(this, 'GitHub CLI', {
            platform: 'Windows',
            displayName: 'GitHub CLI',
            description: 'Install latest version of gh',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/cli/cli/releases/latest > $Env:TEMP\\latest-gh',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-gh',
                '$GH_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                '$ProgressPreference = \'SilentlyContinue\'',
                'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/cli/cli/releases/download/v${GH_VERSION}/gh_${GH_VERSION}_windows_amd64.msi" -OutFile gh.msi',
                'Start-Process msiexec.exe -Wait -ArgumentList \'/i gh.msi /qn\'',
                'del gh.msi',
            ],
        }));
        this.addComponent(new ImageBuilderComponent(this, 'git', {
            platform: 'Windows',
            displayName: 'Git',
            description: 'Install latest version of git',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/git-for-windows/git/releases/latest > $Env:TEMP\\latest-git',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-git',
                '$GIT_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                '$GIT_VERSION_SHORT = ($GIT_VERSION -Split \'.windows.\')[0]',
                'Invoke-WebRequest -UseBasicParsing -Uri https://github.com/git-for-windows/git/releases/download/v${GIT_VERSION}/Git-${GIT_VERSION_SHORT}-64-bit.exe -OutFile git-setup.exe',
                'Start-Process git-setup.exe -Wait -ArgumentList \'/VERYSILENT\'',
                'del git-setup.exe',
            ],
        }));
        let runnerCommands;
        if (this.runnerVersion.version == common_1.RunnerVersion.latest().version) {
            runnerCommands = [
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest > $Env:TEMP\\latest-gha',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-gha',
                '$RUNNER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
            ];
        }
        else {
            runnerCommands = [`$RUNNER_VERSION = '${this.runnerVersion.version}'`];
        }
        this.addComponent(new ImageBuilderComponent(this, 'GitHub Actions Runner', {
            platform: 'Windows',
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
            ].concat(runnerCommands, [
                'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/actions/runner/releases/download/v${RUNNER_VERSION}/actions-runner-win-x64-${RUNNER_VERSION}.zip" -OutFile actions.zip',
                'Expand-Archive actions.zip -DestinationPath C:\\actions',
                'del actions.zip',
            ]),
        }));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        this.prependComponent(new ImageBuilderComponent(this, 'Extra Certs', {
            platform: this.platform,
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Import-Certificate -FilePath certs\\certs.pem -CertStoreLocation Cert:\\LocalMachine\\Root',
            ],
            assets: [
                {
                    path: 'certs',
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Extra Certs Asset', { path }),
                },
            ],
        }));
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        const infra = this.infrastructure();
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: uniqueName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: this.repository.repositoryName,
                        },
                    },
                },
            ],
        });
        const recipe = new ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform,
            components: this.components,
            targetRepository: this.repository,
        });
        const log = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Log', {
            logGroupName: `/aws/imagebuilder/${recipe.name}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, 'Image', {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            containerRecipeArn: recipe.arn,
        });
        image.node.addDependency(log);
        this.imageCleaner(image, recipe.name);
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'Pipeline', {
            name: uniqueName(this),
            description: this.description,
            containerRecipeArn: recipe.arn,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            schedule: scheduleOptions,
        });
        pipeline.node.addDependency(log);
        this.boundImage = {
            // There are simpler ways to get the ARN, but we want an image object that depends on the newly built image.
            // We want whoever is using this image to automatically wait for Image Builder to finish building before using the image.
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryName(this, 'Dependable Image', 
            // we can't use image.attrName because it comes up with upper case
            cdk.Fn.split(':', cdk.Fn.split('/', image.attrImageUri, 2)[1], 2)[0]),
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
        };
        return this.boundImage;
    }
    infrastructure() {
        let role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
            ],
        });
        for (const component of this.components) {
            component.grantAssetsRead(role);
        }
        return new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: uniqueName(this),
            description: this.description,
            subnetId: this.subnetId,
            securityGroupIds: this.securityGroupIds,
            instanceTypes: this.instanceTypes,
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    role.roleName,
                ],
            }).ref,
        });
    }
    imageCleaner(image, recipeName) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['imagebuilder:ListImages', 'imagebuilder:ListImageBuildVersions', 'imagebuilder:DeleteImage'],
                    resources: ['*'],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Deleter', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageDeleter',
            properties: {
                RepoName: this.repository.repositoryName,
                ImageBuilderName: recipeName,
                DeleteOnly: true,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(image);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
}
exports.ContainerImageBuilder = ContainerImageBuilder;
_b = JSII_RTTI_SYMBOL_1;
ContainerImageBuilder[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.ContainerImageBuilder", version: "0.5.1" };
//# sourceMappingURL=data:application/json;base64,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