import periodictable
from enum import Enum

class CalcType(Enum):
    UNDEFINED = 0
    SP = 1
    OPT = 2
    CONSTR_OPT = 3
    FREQ = 4
    TS = 5
    MEP = 6

    NMR = 10
    UVVIS = 11
    UVVIS_TDA = 12
    MO = 13

    OPTFREQ = 20

ATOMIC_NUMBER = {}
ATOMIC_SYMBOL = {}
LOWERCASE_ATOMIC_SYMBOLS = {}

for el in periodictable.elements:
    ATOMIC_NUMBER[el.symbol] = el.number
    ATOMIC_SYMBOL[el.number] = el.symbol
    LOWERCASE_ATOMIC_SYMBOLS[el.symbol.lower()] = el.symbol

STR_TYPES = {#TODO: more synonyms
        "sp": CalcType.SP,
        "single-point": CalcType.SP,
        "single point": CalcType.SP,
        "single point energy": CalcType.SP,
        "single-point energy": CalcType.SP,
        "opt": CalcType.OPT,
        "optimisation": CalcType.OPT,
        "optimization": CalcType.OPT,
        "geometrical optimisation": CalcType.OPT,
        "geometrical optimization": CalcType.OPT,
        "constr_opt": CalcType.CONSTR_OPT,
        "constrained optimisation": CalcType.CONSTR_OPT,
        "constrained optimization": CalcType.CONSTR_OPT,
        "freq": CalcType.FREQ,
        "frequency calculation": CalcType.FREQ,
        "ts": CalcType.TS,
        "ts optimisation": CalcType.TS,
        "opt ts": CalcType.TS,
        "optts": CalcType.TS,
        "nmr": CalcType.NMR,
        "nmr prediction": CalcType.NMR,
        "nmr calculation": CalcType.NMR,
        "uvvis": CalcType.UVVIS,
        "uvvis prediction": CalcType.UVVIS,
        "uv-vis prediction": CalcType.UVVIS,
        "uvvis calculation": CalcType.UVVIS,
        "uv-vis calculation": CalcType.UVVIS,
        "td": CalcType.UVVIS,
        "tddft": CalcType.UVVIS,
        "td-dft": CalcType.UVVIS,
        "tda": CalcType.UVVIS_TDA,
        "mep": CalcType.MEP,
        "minimum energy path": CalcType.MEP,
        "minimum-energy path": CalcType.MEP,
        "minimum-energy-path": CalcType.MEP,
        "neb": CalcType.MEP,
        "nudged elastic band": CalcType.MEP,
        "mo": CalcType.MO,
        "mo calc": CalcType.MO,
        "mo calculation": CalcType.MO,
        "optfreq": CalcType.OPTFREQ,
        "opt_freq": CalcType.OPTFREQ,
        "opt-freq": CalcType.OPTFREQ,
        "opt+freq": CalcType.OPTFREQ,
        }

INV_STR_TYPES = {}
for cmd, t in STR_TYPES.items():
    if t in INV_STR_TYPES.keys():
        INV_STR_TYPES[t].append(cmd)
    else:
        INV_STR_TYPES[t] = [cmd]

SPECIFICATIONS = {
    'gaussian': {
        'general': {
            'nosymm': 0,
            '5d': 0,
            '6d': 0,
        },
        'scf': {
            'xqc': 0,
            'yqc': 0,
            'tight': 0,
        },
        'pop': {
            'nbo': 0,
            'npa': 0,
            'hirshfeld': 0,
            'esp': 0,
            'hly': 0,
            'hlygat': 0,
        },
        'opt': {
            'maxstep': 1,
            'maxcycles': 1,
            'calcfc': 0,
            'calcall': 0,
            'recalcfc': 1,
            'loose': 0,
            'tight': 0,
            'verytight': 0,
        },
        'freq': {
            'noraman': 0,
        },
    },
    'orca': {
        'general': {
            'sloppyscf': 0,
            'loosescf': 0,
            'normalscf': 0,
            'strongscf': 0,
            'tightscf': 0,
            'verytightscf': 0,
            'extremescf': 0,
            'grid': 1,
        },
        'opt': {
            'looseopt': 0,
            'normalopt': 0,
            'tightopt': 0,
            'verytightopt': 0,
        }

    },
    'xtb': {
        'general': {
            'acc': 1,
            'iterations': 1,
            'gfn2-xtb': 0,
            'gfn1-xtb': 0,
            'gfn0-xtb': 0,
            'gfn-ff': 0,
        },
        'Geometrical Optimisation': {
            'opt(crude)': 0,
            'opt(sloppy)': 0,
            'opt(loose)': 0,
            'opt(lax)': 0,
            'opt(normal)': 0,
            'opt(tight)': 0,
            'opt(vtight)': 0,
            'opt(extreme)': 0,
        },
        'Conformational Search': {
            'gfn2-xtb//gfn-ff': 0,
            'rthr': 1,
            'ewin': 1,
        },
    },
}

THEORY_LEVELS = {
    'hartree-fock': ['hf', 'hartree-fock'],
    # Some old exotic semi-empirical methods in ORCA are missing
    'semi-empirical': ['am1', 'pm3', 'pm6', 'pm7', 'mndo'],
    'xtb': ['gfn2-xtb', 'gfn1-xtb', 'gfn0-xtb', 'gfn-ff'],
    'special': ['hf-3c', 'pbeh-3c', 'r2scan-3c', 'b97-3c'],
    'mp2': ['mp2', 'ri-mp2'],
    'cc': ['ccsd', 'ccsd(t)', 'qcisd', 'qcisd(t)', 'lpno-ccsd', \
            'lpno-ccsd(t)', 'lpno-qcisd', 'lpno-qcisd(t)', \
            'dlpno-ccsd', 'dlpno-ccsd(t)', 'dlpno-qcisd', \
            'dlpno-qcisd(t)'],
    # Everything else is assumed to be DFT, because it most likely is
}

SYN_SOFTWARE = {
    'gaussian': ['g16', 'gaussian16', 'gaussian 16'],
    'orca': ['orca 5', 'orca5'],
        }

SYN_SOLVENTS = {
    'acetone': ['acetone'],
    'acetonitrile': ['acetonitrile', 'acn', 'ch3cn', 'mecn'],
    'benzene': ['benzene', 'c6h6'],
    'dichloromethane': ['ch2cl2', 'dcm'],
    'chloroform': ['chcl3'],
    'carbon disulfide': ['cs2'],
    'dimethylformamide': ['dmf'],
    'dimethylsulfoxide': ['dmso'],
    'diethylether': ['diethyl ether', 'ether', 'et2o'],
    'water': ['h2o'],
    'methanol': ['meoh'],
    'n-hexane': ['hexane', 'hex'],
    'tetrahydrofuran': ['thf'],
    'toluene': ['phme', 'phch3'],

    'ethanol': [],
    'isoquinoline': [],
    'quinoline': [],
    'dichloroethane': [],
    'carbontetrachloride': ['ccl4', 'carbon tetrachloride'],
    'chlorobenzene': ['phcl'],
    'nitromethane': ['meno2'],
    'heptane': [],
    'cyclohexane': [],
    'aniline': [],
    'argon': [],
    'krypton': [],
    'xenon': [],
    'n-octanol': ['1-octanol', 'octanol'],
    '1,1,1-trichloroethane': [],
    '1,1,2-trichloroethane': [],
    '1,2,4-trimethylbenzene': [],
    '1,2-dibromoethane': [],
    '1,2-ethanediol': [],
    '1,4-dioxane': [],
    '1-bromo-2-methylpropane': [],
    '1-bromooctane': [],
    '1-bromopentane': [],
    '1-bromopropane': [],
    '1-butanol': [],
    '1-chlorohexane': [],
    '1-chloropentane': [],
    '1-chloropropane': [],
    '1-decanol': [],
    '1-fluorooctane': [],
    '1-heptanol': [],
    '1-hexanol': [],
    '1-hexene': [],
    '1-hexyne': [],
    '1-iodobutane': [],
    '1-iodohexadecane': [],
    '1-iodopentane': [],
    '1-iodopropane': [],
    '1-nitropropane': [],
    '1-nonanol': [],
    '1-pentanol': [],
    '1-pentene': [],
    '1-propanol': [],
    '2,2,2-trifluoroethanol': [],
    '2,2,4-trimethylpentane': [],
    '2,4-dimethylpentane': [],
    '2,4-dimethylpyridine': [],
    '2,6-dimethylpyridine': [],
    '2-bromopropane': [],
    '2-butanol': [],
    '2-chlorobutane': [],
    '2-heptanone': [],
    '2-hexanone': [],
    '2-methoxyethanol': [],
    '2-methyl-1-propanol': [],
    '2-methyl-2-propanol': [],
    '2-methylpentane': [],
    '2-methylpyridine': [],
    '2-nitropropane': [],
    '2-octanone': [],
    '2-pentanone': [],
    '2-propanol': [],
    '2-propen-1-ol': [],
    '3-methylpyridine': [],
    '3-pentanone': [],
    '4-heptanone': [],
    '4-methyl-2-pentanone': [],
    '4-methylpyridine': [],
    '5-nonanone': [],
    'aceticacid': [],
    'acetophenone': [],
    'a-chlorotoluene': [],
    'anisole': [],
    'benzaldehyde': [],
    'benzonitrile': [],
    'benzylalcohol': [],
    'bromobenzene': [],
    'bromoethane': [],
    'bromoform': [],
    'butanal': [],
    'butanoicacid': [],
    'butanone': [],
    'butanonitrile': [],
    'butylamine': [],
    'butylethanoate': [],
    'carbondisulfide': [],
    'cis-1,2-dimethylcyclohexane': [],
    'cis-decalin': [],
    'cyclohexanone': [],
    'cyclopentane': [],
    'cyclopentanol': [],
    'cyclopentanone': [],
    'decalin-mixture': [],
    'dibromomethane': [],
    'dibutylether': [],
    'diethylamine': [],
    'diethylsulfide': [],
    'diiodomethane': [],
    'diisopropylether': [],
    'dimethyldisulfide': [],
    'diphenylether': [],
    'dipropylamine': [],
    'e-1,2-dichloroethene': [],
    'e-2-pentene': [],
    'ethanethiol': [],
    'ethylbenzene': [],
    'ethylethanoate': [],
    'ethylmethanoate': [],
    'ethylphenylether': [],
    'fluorobenzene': [],
    'formamide': [],
    'formicacid': [],
    'hexanoicacid': [],
    'iodobenzene': [],
    'iodoethane': [],
    'iodomethane': [],
    'isopropylbenzene': [],
    'm-cresol': [],
    'mesitylene': [],
    'methylbenzoate': [],
    'methylbutanoate': [],
    'methylcyclohexane': [],
    'methylethanoate': [],
    'methylmethanoate': [],
    'methylpropanoate': [],
    'm-xylene': [],
    'n-butylbenzene': [],
    'n-decane': [],
    'n-dodecane': [],
    'n-hexadecane': [],
    'nitrobenzene': ['phno2'],
    'nitroethane': [],
    'n-methylaniline': [],
    'n-methylformamide-mixture': [],
    'n,n-dimethylacetamide': [],
    'n,n-dimethylformamide': ['n,n-dimethylformamide', 'dmf'],
    'n-nonane': [],
    'n-octane': [],
    'n-pentadecane': [],
    'n-pentane': [],
    'n-undecane': [],
    'o-chlorotoluene': [],
    'o-cresol': [],
    'o-dichlorobenzene': [],
    'o-nitrotoluene': [],
    'o-xylene': [],
    'pentanal': [],
    'pentanoicacid': [],
    'pentylamine': [],
    'pentylethanoate': [],
    'perfluorobenzene': [],
    'p-isopropyltoluene': [],
    'propanal': [],
    'propanoicacid': [],
    'propanonitrile': [],
    'propylamine': [],
    'propylethanoate': [],
    'p-xylene': [],
    'pyridine': [],
    'sec-butylbenzene': [],
    'tert-butylbenzene': [],
    'tetrachloroethene': [],
    'tetrahydrothiophene-s,s-dioxide': [],
    'tetralin': [],
    'thiophene': [],
    'thiophenol': [],
    'trans-decalin': [],
    'tributylphosphate': [],
    'trichloroethene': [],
    'triethylamine': [],
    'xylene-mixture': [],
    'z-1,2-dichloroethene': [],
}


SYN_METHODS = {
        'hfs': [],
        'xalpha': [],
        'lsd': [],
        'vwn5': [],
        'vwn3': [],
        'bnull': [],
        'bvwn': [],
        'bp': [],
        'blyp': [],
        'gp': [],
        'glyp': [],
        'pbe': [],
        'revpbe': [],
        'rpbe': [],
        'pwp': [],
        'olyp': [],
        'opbe': [],
        'xlyp': [],

        'm05' : [],
        'm052x' : ['m05-2x'],
        'm06' : [],
        'm062x' : ['m06-2x'],
        'm06hf' : ['m06-hf'],
        'm06l': ['m06-l'],
        'm08hx': [],

        'tpss': [],
        'revtpss': [],
        'scanfunc': [],

        'b1lyp' : [],
        'b3lyp' : [],
        'bhandhlyp' : [],
        'cam-b3lyp ' : ['camb3lyp'],
        'lc-blyp' : ['lcblyp'],
        'o3lyp' : [],
        'x3lyp' : [],

        'wb2plyp' : [],
        'wb2gp-plyp ' : ['wb2gpplyp'],

        'b1p' : [],
        'b3p' : [],
        'g1lyp' : [],
        'g3lyp' : [],
        'g1p' : [],
        'g3p' : [],
        'pbe0' : ['pbe1pbe'],
        'pwp1' : [],

        'mpwpw': [],
        'mpwlyp': [],
        'mpw1pw' : [],
        'mpw1lyp' : [],
        'mpw1pw91' : [],
        'mpw1lyp' : [],
        'mpw1pbe' : [],
        'mpw3pbe' : [],
        'pw6b95d3' : ['pw6b95-d3'],
        'pw86pbe': [],
        'rpw86pbe': [],
        'pw91_0' : ['pw91-0', 'pw910'],
        'pw6b95' : [],
        'pwlda': [],
        'pw91': [],

        'b97' : [],
        'b97-d': ['b97d'],
        'b97-d3': ['b97d3'],
        'wb97' : [],
        'wb97x' : [],
        'wb97x-d' : ['wb97xd'],
        'wb97x-d3' : ['wb97xd3'],
        'wb97x-v' : ['wb97xv'],
        'wb97x-d3bj ' : ['wb97xd3bj'],

        'tpssh' : [],
        'tpss0' : [],

        'b2plyp' : [],
        'mpw2plyp ' : [],
        'b2gp-plyp ' : ['b2gpplyp'],
        'b2k-plyp ' : ['b2kplyp'],
        'b2t-plyp ' : ['b2tplyp'],
        'pwpb95' : [],

        'b3p86' : [],
        'hse06' : ['hse1pbe'],
        'ohse2pbe' : [],
        'ohse1pbe' : [],
        'pbeh1pbe' : [],
        'b1b95' : [],
        'apfd' : [],
        'lc-whpbe' : ['lcwhpbe'],
        'thcthhyb' : [],
        'bmk' : [],
        'hissbpbe' : [],
        'mn15' : [],
        'mn15l' : ['mn15-l'],
        'm11' : [],
        'm11l' : ['m11-l'],
        'sogga11x' : [],
        'sogga11' : [],
        'n12sx' : [],
        'n12' : [],
        'mn12sx' : [],
        'mn12l' : ['mn12-l'],
        'vsxc' : [],
        'hcth407' : ['hcth/407', 'hcth-407'],
        'hcth93' : ['hcth/93', 'hcth-93'],
        'hcth147' : ['hcth/147', 'hcth-147'],
        'thcth' : [],

        'hf': ['hartree-fock'],

        'mndo': [],
        'am1': [],
        'pm3': [],
        'pm6': [],
        'pm7': ['pm7r6'],

        'gfn2-xtb': ['xtb2', 'gfn2', 'gfn2xtb'],
        'gfn1-xtb': ['xtb1', 'gfn1', 'gfn1xtb', 'gfn-xtb', 'gfnxtb'],
        'gfn0-xtb': ['xtb0', 'gfn0', 'gfn0xtb'],
        'gfn-ff': ['gfnff'],

        'mp2': [],
        'ri-mp2': ['rimp2'],

        'hf-3c': ['hf3c'],
        'pbeh-3c': ['pbeh3c'],
        'r2scan-3c': ['r2scan3c'],
        'b97-3c': ['b973c'],

        'ccsd': [],
        'ccsd(t)': ['ccsd(t)', 'ccsdt'],
        'qcisd': ['qcisd'],
        'qcisd(t)': ['qcisd(t)', 'qcisdt'],

        'lpno-ccsd': ['lpnoccsd'],
        'lpno-ccsd(t)': ['lpnoccsd(t)', 'lpno-ccsdt', 'lpnoccsdt'],
        'lpno-qcisd': ['lpnoqcisd'],
        'lpno-qcisd(t)': ['lpno-qcisdt', 'lpnoqcisd(t)', 'lpnoqcisdt'],
        'dlpno-ccsd': [],
        'dlpno-ccsd(t)': ['dlpno-ccsdt', 'dlpnoccsd(t)', 'dlpnoccsdt'],
        'dlpno-qcisd': ['dlpnoqcisd'],
        'dlpno-qcisd(t)': ['dlpno-qcisdt', 'dlpnoqcisdt'],
    }



# Gaussian allows the combination of exchange and correlation functionals
# without using any special keyword.

EXCHANGE_FUNCTIONALS = {
        's': 'S',
        'xa': 'XA',
        'b': 'B',
        'pw91': 'PW91',
        'mpw': 'mPW',
        'g96': 'G96',
        'pbe': 'PBE',
        'o': 'O',
        'tpss': 'TPSS',
        'revtpss': 'RevTPSS',
        'brx': 'BRx',
        'pkzb': 'PKZB',
        'wpbeh': 'wPBEh',
        'pbeh': 'PBEh',
}

CORRELATION_FUNCTIONALS = {
        'vwn': 'VWN',
        'vwn5': 'VWN5',
        'lyp': 'LYP',
        'pl': 'PL',
        'p86': 'P86',
        'pw91': 'PW91',
        'b95': 'B95',
        'pbe': 'PBE',
        'tpss': 'TPSS',
        'revtpss': 'RevTPSS',
        'kcis': 'KCIS',
        'brc': 'BRC',
        'pkzb': 'PKZB',
        'vp86': 'VP86',
        'v5lyp': 'V5LYP',
}

SYN_BASIS_SETS = {
        'sto3g': [],

        'mini': [],
        'minis': [],
        'minix': [],
        'midi': [],
        'midix': [],

        '321g': [],
        '321gsp': [],
        '422gsp': [],
        '431g': [],
        '621g': [],

        '631g': [],
        '631gd': ['631g*'],

        'm631g': [],
        'm631gd': ['m631g*'],

        '631gd,p': ['631g**'],
        '631g2d': [],
        '631g2dp': [],
        '631g2d2p': [],
        '631g2df': [],
        '631g2df2p': [],
        '631g2df2pd': [],

        '631+gd': ['631+g*'],
        '631+gdp': ['631+g**'],

        '631+g2d': [],
        '631+g2dp': [],
        '631+g2d2p': [],
        '631+g2df': [],
        '631+g2df2p': [],
        '631+g2df2pd': [],

        '631++gdp': ['631++g**'],
        '631++g2dp': [],
        '631++g2d2p': [],
        '631++g2df': [],
        '631++g2df2p': [],
        '631++g2df2pd': [],

        '6311gd': ['6311g*'],
        '6311gdp': ['6311g**'],
        '6311g2d': [],
        '6311g2dp': [],
        '6311g2d,2p': [],
        '6311g2df': [],
        '6311g2df2p': [],
        '6311g2df2pd': [],
        '6311g3df': [],
        '6311g3df3pd': [],

        '6311+gd': ['6311+g*'],
        '6311+gdp': ['6311+g**'],
        '6311+g2d': [],
        '6311+g2dp': [],
        '6311+g2d2p': [],
        '6311+g2df': [],
        '6311+g2df2p': [],
        '6311+g2df2pd': [],
        '6311+g3df': [],
        '6311+g3df2p': [],
        '6311+g3df3pd': [],

        '6311++gdp': ['6311++g**'],
        '6311++g2d': [],
        '6311++g2dp': [],
        '6311++g2d2p': [],
        '6311++g2df': [],
        '6311++g2df2p': [],
        '6311++g2df2pd': [],
        '6311++g3df': [],
        '6311++g3df3pd': [],

        'havt+dz': [],
        'havq+dz': [],
        'hav5+dz': [],

        'w1dz': [],
        'w1mtsmall': [],
        'w1dz': [],

        'sv': [],
        'svp': [],

        'tzv': [],
        'tzvp': [],
        'tzvpp': [],

        'qzvp': [],
        'qzvpp': [],

        'dkhsvp': [],
        'dkhtzvp': [],
        'dkhtzvpp': [],
        'dkhqzvp': [],
        'dkhqzvpp': [],

        'zorasvp': [],
        'zoratzvp': [],
        'zoratzvpp': [],
        'zoraqzvp': [],
        'zoraqzvpp': [],

        'def2msvp': [],
        'def2mtzvp': [],

        'def2svp': [],
        'def2tzvpf': [],
        'def2tzvp': [],
        'def2tzvpp': [],
        'def2qzvp': [],
        'def2qzvpp': [],

        'def2svpd': [],
        'def2tzvpd': [],
        'def2tzvppd': [],
        'def2qzvpd': [],
        'def2qzvppd': [],

        'def2sv': [],
        'def2svpp': [],
        'def2tzv': [],
        'def2qzv': [],

        'dkhdef2svp': [],
        'dkhdef2tzvpf': [],
        'dkhdef2tzvp': [],
        'dkhdef2tzvpp': [],
        'dkhdef2qzvp': [],
        'dkhdef2qzvpp': [],

        'zoradef2svp': [],
        'zoradef2tzvpf': [],
        'zoradef2tzvp': [],
        'zoradef2tzvpp': [],
        'zoradef2qzvp': [],
        'zoradef2qzvpp': [],

        'madef2msvp': [],
        'madef2svp': [],
        'madef2tzvpf': [],
        'madef2tzvp': [],
        'madef2tzvpp': [],
        'madef2qzvp': [],
        'madef2qzvpp': [],

        'madkhdef2svp': [],
        'madkhdef2tzvpf': [],
        'madkhdef2tzvp': [],
        'madkhdef2tzvpp': [],
        'madkhdef2qzvp': [],
        'madkhdef2qzvpp': [],

        'mazoradef2svp': [],
        'mazoradef2tzvpf': [],
        'mazoradef2tzvp': [],
        'mazoradef2tzvpp': [],
        'mazoradef2qzvpp': [],

        'anopvdz': [],
        'anopvtz': [],
        'anopvqz': [],
        'anopv5z': [],
        'anopv6z': [],

        'auganopvdz': [],
        'auganopvtz': [],
        'auganopvqz': [],
        'auganopv5z': [],

        'sauganopvdz': [],
        'sauganopvtz': [],
        'sauganopvqz': [],
        'sauganopv5z': [],

        'anorccdzp': [],
        'anorcctzp': [],
        'anorccqzp': [],
        'anorccfull': [],

        'pc0': [],
        'pc1': [],
        'pc2': [],
        'pc3': [],
        'pc4': [],

        'augpc0': [],
        'augpc1': [],
        'augpc2': [],
        'augpc3': [],
        'augpc4': [],

        'pcj0': [],
        'pcj1': [],
        'pcj2': [],
        'pcj3': [],
        'pcj4': [],

        'augpcj0': [],
        'augpcj1': [],
        'augpcj2': [],
        'augpcj3': [],
        'augpcj4': [],

        'pcseg0': [],
        'pcseg1': [],
        'pcseg2': [],
        'pcseg3': [],
        'pcseg4': [],

        'augpcseg0': [],
        'augpcseg1': [],
        'augpcseg2': [],
        'augpcseg3': [],
        'augpcseg4': [],

        'pcsseg0': [],
        'pcsseg1': [],
        'pcsseg2': [],
        'pcsseg3': [],
        'pcsseg4': [],

        'augpcsseg0': [],
        'augpcsseg1': [],
        'augpcsseg2': [],
        'augpcsseg3': [],
        'augpcsseg4': [],

        'ccpvdz': [],
        'ccpvtz': [],
        'ccpvqz': [],
        'ccpv5z': [],
        'ccpv6z': [],

        'augccpvdz': [],
        'augccpvtz': [],
        'augccpvqz': [],
        'augccpv5z': [],
        'augccpv6z': [],

        'ccpvd+dz': [],
        'ccpvt+dz': [],
        'ccpvq+dz': [],
        'ccpv5+dz': [],

        'augccpvtzj': [],

        'ccpcvdz': [],
        'ccpcvtz': [],
        'ccpcvqz': [],
        'ccpcv5z': [],
        'ccpcv6z': [],

        'augccpcvdz': [],
        'augccpcvtz': [],
        'augccpcvqz': [],
        'augccpcv5z': [],
        'augccpcv6z': [],

        'ccpwcvdz': [],
        'ccpwcvtz': [],
        'ccpwcvqz': [],
        'ccpwcv5z': [],

        'augccpwcvdz': [],
        'augccpwcvtz': [],
        'augccpwcvqz': [],
        'augccpwcv5z': [],

        'ccpvdzpp': [],
        'ccpvtzpp': [],
        'ccpvqzpp': [],
        'ccpv5zpp': [],

        'augccpvdzpp': [],
        'augccpvtzpp': [],
        'augccpvqzpp': [],
        'augccpv5zpp': [],

        'ccpcvdzpp': [],
        'ccpcvtzpp': [],
        'ccpcvqzpp': [],
        'ccpcv5zpp': [],

        'augccpcvdzpp': [],
        'augccpcvtzpp': [],
        'augccpcvqzpp': [],
        'augccpcv5zpp': [],

        'ccpwvdzpp': [],
        'ccpwvtzpp': [],
        'ccpwvqzpp': [],
        'ccpwv5zpp': [],

        'augccpwvdzpp': [],
        'augccpwvtzpp': [],
        'augccpwvqzpp': [],
        'augccpwv5zpp': [],

        'ccpvdzdk': [],
        'ccpvtzdk': [],
        'ccpvqzdk': [],
        'ccpv5zdk': [],

        'augccpvdzdk': [],
        'augccpvtzdk': [],
        'augccpvqzdk': [],
        'augccpv5zdk': [],

        'ccpwcvdzdk': [],
        'ccpwcvtzdk': [],
        'ccpwcvqzdk': [],
        'ccpwcv5zdk': [],

        'augccpvd+dz': [],
        'augccpvt+dz': [],
        'augccpvq+dz': [],
        'augccpv5+dz': [],
        'augccpv6+dz': [],

        'w1opt': [],
        'w1qz': [],

        'augccpwvdzdk': [],
        'augccpwvtzdk': [],
        'augccpwvqzdk': [],
        'augccpwv5zdk': [],

        'partridge1': [],
        'partridge2': [],
        'partridge3': [],
        'partridge4': [],

        'anosz': [],

        'lanl08': [],
        'lanl08f': [],
        'lanl2dz': [],
        'lanl2tz': [],
        'lanl2tzf': [],

        'lanl2mb': [],

        'sapporodzp2012': [],
        'sapporotzp2012': [],
        'sapporoqzp2012': [],

        'sapporodkh3dzp2012': [],
        'sapporodkh3tzp2012': [],
        'sapporodkh3qzp2012': [],

        'sarcdkhsvp': [],
        'sarcdkhtzvp': [],
        'sarcdkhtzvpp': [],

        'sarc2dkhqzv': [],
        'sarc2dkhqzvp': [],
        'sarc2zoraqzv': [],
        'sarc2zoraqzvp': [],

        'd95': [],
        'd95p': [],

        'eprii': [],
        'epriii': [],

        'igloii': [],
        'igloiii': [],

        'ugbs': [],
        'cp': [],
        'cpppp': [],

        'wachters+f': [],

        'ccpvdzf12': [],
        'ccpvtzf12': [],
        'ccpvqzf12': [],

        'ccpvdzppf12': [],
        'ccpvtzppf12': [],
        'ccpvqzppf12': [],

        'ccpcvdzf12': [],
        'ccpcvtzf12': [],
        'ccpcvqzf12': [],

        'd95v': [],
        'shc': [],
        'cep4g': [],
        'cep31g': [],
        'cep121g': [],

        'sdd': [],
        'sddall': [],

        'dgdzvp': [],
        'dgdzvp2': [],
        'gdtzvp': [],
        'cbsb7': [],

    }

SOFTWARE_METHODS = {
        'orca':
                {
                    'hfs': 'HFS',
                    'xalpha': 'XAlpha',
                    'lsd': 'LSD',
                    'vwn5': 'VWN5',
                    'vwn3': 'VWN3',
                    'pwlda': 'PWLDA',
                    'bnull': 'BNULL',
                    'bvwn': 'BVWN',
                    'bp': 'BP',
                    'pw91': 'PW91',
                    'mpwpw': 'mPWPW',
                    'mpwlyp': 'mPWLYP',
                    'blyp': 'BLYP',
                    'gp': 'GP',
                    'glyp': 'GLYP',
                    'pbe': 'PBE',
                    'revpbe': 'revPBE',
                    'rpbe': 'RPBE',
                    'pwp': 'PWP',
                    'olyp': 'OLYP',
                    'opbe': 'OPBE',
                    'xlyp': 'XLYP',
                    'b97-d': 'B97-D',
                    'b97-d3': 'B97-D3',
                    'pw86pbe': 'PW86PBE',
                    'rpw86pbe': 'RPW86PBE',
                    'm06l': 'M06L',
                    'tpss': 'TPSS',
                    'revtpss': 'revTPSS',
                    'scanfunc': 'SCANfunc',
                    'b1lyp' : 'B1LYP',
                    'b3lyp' : 'B3LYP',
                    'b1p' : 'B1P',
                    'b3p' : 'B3P',
                    'g1lyp' : 'G1LYP',
                    'g3lyp' : 'G3LYP',
                    'g1p' : 'G1P',
                    'g3p' : 'G3P',
                    'pbe0' : 'PBE0',
                    'pwp1' : 'PWP1',
                    'm06' : 'M06',
                    'm062x' : 'M062X',
                    'mpw1pw' : 'mPW1PW',
                    'mpw1lyp' : 'mPW1LYP',
                    'pw91_0' : 'PW91_0',
                    'o3lyp' : 'O3LYP',
                    'x3lyp' : 'X3LYP',
                    'pw6b95' : 'PW6B95',
                    'b97' : 'B97',
                    'bhandhlyp' : 'BHANDHLYP',
                    'wb97' : 'wB97',
                    'wb97x' : 'wB97X',
                    'wb97x-d3' : 'wB97X-D3',
                    'wb97x-v' : 'wB97X-V',
                    'wb97x-d3bj' : 'wB97X-D3BJ',
                    'cam-b3lyp' : 'CAM-B3LYP',
                    'lc-blyp' : 'LC-BLYP',
                    'tpssh' : 'TPSSH',
                    'tpss0' : 'TPSS0',
                    'b2plyp' : 'B2PLYP',
                    'mpw2plyp' : 'mPW2PLYP',
                    'b2gp-plyp' : 'B2GP-PLYP',
                    'b2k-plyp' : 'B2K-PLYP',
                    'b2t-plyp' : 'B2T-PLYP',
                    'pwpb95' : 'PWPB95',
                    'wb2plyp' : 'wB2PLYP',
                    'wb2gp-plyp' : 'wB2GP-PLYP',

                    'am1': 'AM1',
                    'pm3': 'PM3',
                    'mndo': 'MNDO',

                    'gfn2-xtb': 'xtb2',
                    'gfn1-xtb': 'xtb1',
                    #'gfn0-xtb': 'xtb0', # Not available in ORCA 5.0.2
                    'gfn-ff': 'xtbff',

                    'hf': 'HF',

                    'mp2': 'MP2',
                    'ri-mp2': 'RI-MP2',

                    'hf-3c': 'HF-3c',
                    'pbeh-3c': 'PBEh-3c',
                    'r2scan-3c': 'r2SCAN-3c',
                    'b97-3c': 'B97-3c',

                    'ccsd': 'CCSD',
                    'ccsd(t)': 'CCSD(T)',
                    'qcisd': 'QCISD',
                    'qcisd(t)': 'QCISD(T)',

                    'lpno-ccsd': 'LPNO-CCSD',
                    'lpno-ccsd(t)': 'LPNOCCSD(T)',
                    'lpno-qcisd': 'LPNO-QCISD',
                    'lpno-qcisd(t)': 'LPNO-QCISD(T)',
                    'dlpno-ccsd': 'DLPNO-CCSD',
                    'dlpno-ccsd(t)': 'DLPNO-CCSD(T)',
                    'dlpno-qcisd': 'DLPNO-QCISD',
                    'dlpno-qcisd(t)': 'DLPNO-QCISD(T)',

                },
       'gaussian':
                {
                    'b97-d': 'B97D',
                    'b97-d3': 'B97D3',
                    'm06l': 'M06L',
                    'm08hx': 'M08HX',
                    'b1lyp' : 'B1LYP',
                    'b3lyp' : 'B3LYP',
                    'b3p86' : 'B3P86',
                    'pbe0' : 'PBE1PBE',
                    'hse06' : 'HSE1PBE',
                    'ohse2pbe' : 'OHSE2PBE',
                    'ohse1pbe' : 'OHSE1PBE',
                    'pbeh1pbe' : 'PBEh1PBE',
                    'b1b95' : 'B1B95',
                    'm05' : 'M05',
                    'm052x' : 'M052X',
                    'm06' : 'M06',
                    'm06hf' : 'M06HF',
                    'm062x' : 'M062X',
                    'o3lyp' : 'O3LYP',
                    'apfd' : 'APFD',
                    'x3lyp' : 'X3LYP',
                    'pw6b95' : 'PW6B95',
                    'pw6b95d3' : 'PW6B95D3',
                    'mpw1pw91' : 'mPW1PW91',
                    'mpw1lyp' : 'mPW1LYP',
                    'mpw1pbe' : 'mPW1PBE',
                    'mpw3pbe' : 'mPW3PBE',
                    'wb97' : 'wB97',
                    'wb97x' : 'wB97X',
                    'wb97x-d' : 'wB97XD',#Gaussian says dispersion model D2, not D3
                    'cam-b3lyp' : 'CAM-B3LYP',
                    'lc-whpbe' : 'LC-wHPBE',
                    'tpssh' : 'TPSSH',
                    'thcthhyb' : 'tHCTHhyb',
                    'bmk' : 'BMK',
                    'hissbpbe' : 'HISSbPBE',
                    'mn15' : 'MN15',
                    'mn15l' : 'MN15L',
                    'm11' : 'M11',
                    'm11l' : 'M11L',
                    'sogga11x' : 'SOGGA11X',
                    'sogga11' : 'SOGGA11',
                    'n12sx' : 'N12SX',
                    'n12' : 'N12',
                    'mn12sx' : 'MN12SX',
                    'mn12l' : 'MN12L',
                    'vsxc' : 'VSXC',
                    'hcth407' : 'HCTH/407',
                    'hcth93' : 'HCTH/93',
                    'hcth147' : 'HCTH/147',
                    'thcth' : 'tHCTH',

                    'am1': 'AM1',
                    'pm3': 'PM3',
                    'pm6': 'PM6',
                    'pm7': 'PM7',

                    'hf': 'HF',
                }

    }

SOFTWARE_BASIS_SETS = {
        'orca':
                {
                    'sto3g': 'STO-3G',
                    'mini': 'MINI',
                    'minis': 'MINIS',
                    'minix': 'MINIX',
                    'midi': 'MIDI',
                    '321g': '3-21G',
                    '321gsp': '3-21GSP',
                    '422gsp': '4-22GSP',
                    '631g': '6-31G',
                    '631gd': '6-31G(d)',
                    'm631g': 'm6-31G',
                    'm631gd': 'm6-31G(d)',
                    '631gdp': '6-31G(d,p)',
                    '631g2d': '6-31G(2d)',
                    '631g2dp': '6-31G(2d,p)',
                    '631g2d2p': '6-31G(2d,2p)',
                    '631g2df': '6-31G(2df)',
                    '631g2df2p': '6-31G(2df,2p)',
                    '631g2df2pd': '6-31G(2df,2pd)',
                    '631+gd': '6-31+G(d)',
                    '631+gdp': '6-31+G(d,p)',
                    '631+g2d': '6-31+G(2d)',
                    '631+g2dp': '6-31+G(2d,p)',
                    '631+g2d2p': '6-31+G(2d,2p)',
                    '631+g2df': '6-31+G(2df)',
                    '631+g2df2p': '6-31+G(2df,2p)',
                    '631+g2df2pd': '6-31+G(2df,2pd)',
                    '631++gdp': '6-31++G(d,p)',
                    '631++g2dp': '6-31++G(2d,p)',
                    '631++g2d2p': '6-31++G(2d,2p)',
                    '631++g2df': '6-31++G(2df)',
                    '631++g2df2p': '6-31++G(2df,2p)',
                    '631++g2df2pd': '6-31++G(2df,2pd)',
                    '6311gd': '6-311G(d)',
                    '6311gdp': '6-311G(d,p)',
                    '6311g2d': '6-311G(2d)',
                    '6311g2dp': '6-311G(2d,p)',
                    '6311g2d2p': '6-311G(2d,2p)',
                    '6311g2df': '6-311G(2df)',
                    '6311g2df2p': '6-311G(2df,2p)',
                    '6311g2df2pd': '6-311G(2df,2pd)',
                    '6311g3df': '6-311G(3df)',
                    '6311g3df3pd': '6-311G(3df,3pd)',
                    '6311+gd': '6-311+G(d)',
                    '6311+gdp': '6-311+G(d,p)',
                    '6311+g2d': '6-311+G(2d)',
                    '6311+g2dp': '6-311+G(2d,p)',
                    '6311+g2d2p': '6-311+G(2d,2p)',
                    '6311+g2df': '6-311+G(2df)',
                    '6311+g2df2p': '6-311+G(2df,2p)',
                    '6311+g2df2pd': '6-311+G(2df,2pd)',
                    '6311+g3df': '6-311+G(3df)',
                    '6311+g3df2p': '6-311+G(3df,2p)',
                    '6311+g3df3pd': '6-311+G(3df,3pd)',
                    '6311++gdp': '6-311++G(d,p)',
                    '6311++g2d': '6-311++G(2d)',
                    '6311++g2dp': '6-311++G(2d,p)',
                    '6311++g2d2p': '6-311++G(2d,2p)',
                    '6311++g2df': '6-311++G(2df)',
                    '6311++g2df2p': '6-311++G(2df,2p)',
                    '6311++g2df2pd': '6-311++G(2df,2pd)',
                    '6311++g3df': '6-311++G(3df)',
                    '6311++g3df3pd': '6-311++G(3df,3pd)',
                    'havt+dz': 'haV(t+d)z',
                    'havq+dz': 'haV(q+d)z',
                    'hav5+dz': 'haV(5+d)z',
                    'w1dz': 'W1-DZ',
                    'w1mtsmall': 'w1-mtsmall',
                    'w1dz': 'W1-TZ',
                    'sv': 'SV',
                    'svp': 'SVP',
                    'tzv': 'TZV',
                    'tzvp': 'TZVP',
                    'tzvpp': 'TZVPP',
                    'qzvp': 'QZVP',
                    'qzvpp': 'QZVPP',
                    'dkhsvp': 'DKH-SVP',
                    'dkhtzvp': 'DKH-TZVP',
                    'dkhtzv-pp': 'DKH-TZV-PP',
                    'dkhqzvp': 'DKH-QZVP',
                    'dkhqzv-pp': 'DKH-QZV-PP',
                    'zorasvp': 'ZORA-SVP',
                    'zoratzvp': 'ZORA-TZVP',
                    'zoratzvpp': 'ZORA-TZV-PP',
                    'zoraqzvp': 'ZORA-QZVP',
                    'zoraqzvpp': 'ZORA-QZV-PP',
                    'def2msvp': 'Def2-mSVP',
                    'def2mtzvp': 'Def2-mTZVP',
                    'def2svp': 'Def2-SVP',
                    'def2tzvpf': 'Def2-TZVP(-f)',
                    'def2tzvp': 'Def2-TZVP',
                    'def2tzvpp': 'Def2-TZVPP',
                    'def2qzvp': 'Def2-QZVP',
                    'def2qzvpp': 'Def2-QZVPP',
                    'def2svpd': 'Def2-SVPD',
                    'def2tzvpd': 'Def2-TZVPD',
                    'def2tzvppd': 'Def2-TZVPPD',
                    'def2qzvpd': 'Def2-QZVPD',
                    'def2qzvppd': 'Def2-QZVPPD',
                    'dkhdef2svp': 'DKH-Def2-SVP',
                    'dkhdef2tzvp(-f)': 'DKH-Def2-TZVP(-f)',
                    'dkhdef2tzvp': 'DKH-Def2-TZVP',
                    'dkhdef2tzvpp': 'DKH-Def2-TZVPP',
                    'dkhdef2qzvp': 'DKH-Def2-QZVP',
                    'dkhdef2qzvpp': 'DKH-Def2-QZVPP',
                    'zoradef2svp': 'ZORA-Def2-SVP',
                    'zoradef2tzvpf': 'ZORA-Def2-TZVP(-f)',
                    'zoradef2tzvp': 'ZORA-Def2-TZVP',
                    'zoradef2tzvpp': 'ZORA-Def2-TZVPP',
                    'zoradef2qzvp': 'ZORA-Def2-QZVP',
                    'zoradef2qzvpp': 'ZORA-Def2-QZVPP',
                    'madef2msvp': 'ma-Def2-mSVP',
                    'madef2svp': 'ma-Def2-SVP',
                    'madef2tzvpf': 'ma-Def2-TZVP(-f)',
                    'madef2tzvp': 'ma-Def2-TZVP',
                    'madef2tzvpp': 'ma-Def2-TZVPP',
                    'madef2qzvp': 'ma-Def2-QZVP',
                    'madef2qzvpp': 'ma-Def2-QZVPP',
                    'madkhdef2svp': 'ma-DKH-Def2-SVP',
                    'madkhdef2tzvpf': 'ma-DKH-Def2-TZVP(-f)',
                    'madkhdef2tzvp': 'ma-DKH-Def2-TZVP',
                    'madkhdef2tzvpp': 'ma-DKH-Def2-TZVPP',
                    'madkhdef2qzvp': 'ma-DKH-def2-QZVP',
                    'madkhdef2qzvpp': 'ma-DKH-def2-QZVPP',
                    'mazoradef2svp': 'ma-ZORA-def2-SVP',
                    'mazoradef2tzvpf': 'ma-ZORA-Def2-TZVP(-f)',
                    'mazoradef2tzvp': 'ma-ZORA-Def2-TZVP',
                    'mazoradef2tzvpp': 'ma-ZORA-Def2-TZVPP',
                    'mazoradef2qzvpp': 'ma-ZORA-Def2-QZVPP',
                    'anopvdz': 'ANO-pVDZ',
                    'anopvtz': 'ANO-pVTZ',
                    'anopvqz': 'ANO-pVQZ',
                    'anopv5z': 'ANO-pV5Z',
                    'anopv6z': 'ANO-pV6Z',
                    'auganopvdz': 'aug-ANO-pVDZ',
                    'auganopvtz': 'aug-ANO-pVTZ',
                    'auganopvqz': 'aug-ANO-pVQZ',
                    'auganopv5z': 'aug-ANO-pV5Z',
                    'sauganopvdz': 'saug-ANO-pVDZ',
                    'sauganopvtz': 'saug-ANO-pVTZ',
                    'sauganopvqz': 'saug-ANO-pVQZ',
                    'sauganopv5z': 'saug-ANO-pV5Z',
                    'anorccdzp': 'ANO-RCC-DZP',
                    'anorcctzp': 'ANO-RCC-TZP',
                    'anorccqzp': 'ANO-RCC-QZP',
                    'anorccfull': 'ANO-RCC-Full',
                    'pc0': 'pc-0',
                    'pc1': 'pc-1',
                    'pc2': 'pc-2',
                    'pc3': 'pc-3',
                    'pc4': 'pc-4',
                    'augpc0': 'aug-pc-0',
                    'augpc1': 'aug-pc-1',
                    'augpc2': 'aug-pc-2',
                    'augpc3': 'aug-pc-3',
                    'augpc4': 'aug-pc-4',
                    'pcj0': 'pcJ-0',
                    'pcj1': 'pcJ-1',
                    'pcj2': 'pcJ-2',
                    'pcj3': 'pcJ-3',
                    'pcj4': 'pcJ-4',
                    'augpcj0': 'aug-pcJ-0',
                    'augpcj1': 'aug-pcJ-1',
                    'augpcj2': 'aug-pcJ-2',
                    'augpcj3': 'aug-pcJ-3',
                    'augpcj4': 'aug-pcJ-4',
                    'pcseg0': 'pcseg-0',
                    'pcseg1': 'pcseg-1',
                    'pcseg2': 'pcseg-2',
                    'pcseg3': 'pcseg-3',
                    'pcseg4': 'pcseg-4',
                    'augpcseg0': 'aug-pcseg-0',
                    'augpcseg1': 'aug-pcseg-1',
                    'augpcseg2': 'aug-pcseg-2',
                    'augpcseg3': 'aug-pcseg-3',
                    'augpcseg4': 'aug-pcseg-4',
                    'pcsseg0': 'pcSseg-0',
                    'pcsseg1': 'pcSseg-1',
                    'pcsseg2': 'pcSseg-2',
                    'pcsseg3': 'pcSseg-3',
                    'pcsseg4': 'pcSseg-4',
                    'augpcsseg0': 'aug-pcSseg-0',
                    'augpcsseg1': 'aug-pcSseg-1',
                    'augpcsseg2': 'aug-pcSseg-2',
                    'augpcsseg3': 'aug-pcSseg-3',
                    'augpcsseg4': 'aug-pcSseg-4',
                    'ccpvdz': 'cc-pVDZ',
                    'ccpvtz': 'cc-pVTZ',
                    'ccpvqz': 'cc-pVQZ',
                    'ccpv5z': 'cc-pV5Z',
                    'ccpv6z': 'cc-pV6Z',
                    'augccpvdz': 'aug-cc-pVDZ',
                    'augccpvtz': 'aug-cc-pVTZ',
                    'augccpvqz': 'aug-cc-pVQZ',
                    'augccpv5z': 'aug-cc-pV5Z',
                    'augccpv6z': 'aug-cc-pV6Z',
                    'ccpvd+dz': 'cc-pVD(+d)Z',
                    'ccpvt+dz': 'cc-pVT(+d)Z',
                    'ccpvq+dz': 'cc-pVQ(+d)Z',
                    'ccpv5+dz': 'cc-pV5(+d)Z',
                    'augccpvtzj': 'aug-cc-pVTZ-J',
                    'ccpcvdz': 'cc-pCVDZ',
                    'ccpcvtz': 'cc-pCVTZ',
                    'ccpcvqz': 'cc-pCVQZ',
                    'ccpcv5z': 'cc-pCV5Z',
                    'ccpcv6z': 'cc-pCV6Z',
                    'augccpcvdz': 'aug-cc-pCVDZ',
                    'augccpcvtz': 'aug-cc-pCVTZ',
                    'augccpcvqz': 'aug-cc-pCVQZ',
                    'augccpcv5z': 'aug-cc-pCV5Z',
                    'augccpcv6z': 'aug-cc-pCV6Z',
                    'ccpwcvdz': 'cc-pwCVDZ',
                    'ccpwcvtz': 'cc-pwCVTZ',
                    'ccpwcvqz': 'cc-pwCVQZ',
                    'ccpwcv5z': 'cc-pwCV5Z',
                    'augccpwcvdz': 'aug-cc-pwCVDZ',
                    'augccpwcvtz': 'aug-cc-pwCVTZ',
                    'augccpwcvqz': 'aug-cc-pwCVQZ',
                    'augccpwcv5z': 'aug-cc-pwCV5Z',
                    'ccpvdzpp': 'cc-pVDZ-PP',
                    'ccpvtzpp': 'cc-pVTZ-PP',
                    'ccpvqzpp': 'cc-pVQZ-PP',
                    'ccpv5zpp': 'cc-pV5Z-PP',
                    'augccpvdz-pp': 'aug-cc-pVDZ-PP',
                    'augccpvtz-pp': 'aug-cc-pVTZ-PP',
                    'augccpvqz-pp': 'aug-cc-pVQZ-PP',
                    'augccpv5z-pp': 'aug-cc-pV5Z-PP',
                    'ccpcvdzpp': 'cc-pCVDZ-PP',
                    'ccpcvtzpp': 'cc-pCVTZ-PP',
                    'ccpcvqzpp': 'cc-pCVQZ-PP',
                    'ccpcv5zpp': 'cc-pCV5Z-PP',
                    'augccpcvdz-pp': 'aug-cc-pCVDZ-PP',
                    'augccpcvtz-pp': 'aug-cc-pCVTZ-PP',
                    'augccpcvqz-pp': 'aug-cc-pCVQZ-PP',
                    'augccpcv5z-pp': 'aug-cc-pCV5Z-PP',
                    'ccpwvdzpp': 'cc-pwVDZ-PP',
                    'ccpwvtzpp': 'cc-pwVTZ-PP',
                    'ccpwvqzpp': 'cc-pwVQZ-PP',
                    'ccpwv5zpp': 'cc-pwV5Z-PP',
                    'augccpwvdz-pp': 'aug-cc-pwVDZ-PP',
                    'augccpwvtz-pp': 'aug-cc-pwVTZ-PP',
                    'augccpwvqz-pp': 'aug-cc-pwVQZ-PP',
                    'augccpwv5z-pp': 'aug-cc-pwV5Z-PP',
                    'ccpvdzdk': 'cc-pVDZ-DK',
                    'ccpvtzdk': 'cc-pVTZ-DK',
                    'ccpvqzdk': 'cc-pVQZ-DK',
                    'ccpv5zdk': 'cc-pV5Z-DK',
                    'augccpvdzdk': 'aug-cc-pVDZ-DK',
                    'augccpvtzdk': 'aug-cc-pVTZ-DK',
                    'augccpvqzdk': 'aug-cc-pVQZ-DK',
                    'augccpv5zdk': 'aug-cc-pV5Z-DK',
                    'ccpwcvdzdk': 'cc-pwCVDZ-DK',
                    'ccpwcvtzdk': 'cc-pwCVTZ-DK',
                    'ccpwcvqzdk': 'cc-pwCVQZ-DK',
                    'ccpwcv5zdk': 'cc-pwCV5Z-DK',
                    'augccpvd+dz': 'aug-cc-pV(D+d)Z',
                    'augccpvt+dz': 'aug-cc-pV(T+d)Z',
                    'augccpvq+dz': 'aug-cc-pV(Q+d)Z',
                    'augccpv5+dz': 'aug-cc-pV(5+d)Z',
                    'augccpv6+dz': 'aug-cc-pV(6+d)Z',
                    'w1opt': 'W1-Opt',
                    'w1qz': 'W1-QZ',
                    'augccpwvdzdk': 'aug-cc-pwVDZ-DK',
                    'augccpwvtzdk': 'aug-cc-pwVTZ-DK',
                    'augccpwvqzdk': 'aug-cc-pwVQZ-DK',
                    'augccpwv5zdk': 'aug-cc-pwV5Z-DK',
                    'partridge1': 'Partridge-1',
                    'partridge2': 'Partridge-2',
                    'partridge3': 'Partridge-3',
                    'partridge4': 'Partridge-4',
                    'anosz': 'ANO-SZ',
                    'lanl08': 'LANL08',
                    'lanl08f': 'LANL08(f)',
                    'lanl2dz': 'LANL2DZ',
                    'lanl2tz': 'LANL2TZ',
                    'lanl2tzf': 'LANL2TZ(f)',
                    'sapporodzp2012': 'Sapporo-DZP-2012',
                    'sapporotzp2012': 'Sapporo-TZP-2012',
                    'sapporoqzp2012': 'Sapporo-QZP-2012',
                    'sapporodkh3dzp2012': 'Sapporo-DKH3-DZP-2012',
                    'sapporodkh3tzp2012': 'Sapporo-DKH3-TZP-2012',
                    'sapporodkh3qzp2012': 'Sapporo-DKH3-QZP-2012',
                    'sarcdkhsvp': 'SARC-DKH-SVP',
                    'sarcdkhtzvp': 'SARC-DKH-TZVP',
                    'sarcdkhtzvpp': 'SARC-DKH-TZVPP',
                    'sarc2dkhqzv': 'SARC2-DKH-QZV',
                    'sarc2dkhqzvp': 'SARC2-DKH-QZVP',
                    'sarc2zoraqzv': 'SARC2-ZORA-QZV',
                    'sarc2zoraqzvp': 'SARC2-ZORA-QZVP',
                    'd95': 'D95',
                    'd95p': 'D95p',
                    'eprii': 'EPR-II',
                    'epriii': 'EPR-III',
                    'igloii': 'IGLO-II',
                    'igloiii': 'IGLO-III',
                    'ugbs': 'UGBS',
                    'cp': 'CP',
                    'cpppp': 'CP(PPP)',
                    'wachters+f': 'Wachters+f',
                    'ccpvdzf12': 'cc-pVDZ-F12',
                    'ccpvtzf12': 'cc-pVTZ-F12',
                    'ccpvqzf12': 'cc-pVQZ-F12',
                    'ccpvdzppf12': 'cc-pVDZ-PP-F12',
                    'ccpvtzppf12': 'cc-pVTZ-PP-F12',
                    'ccpvqzppf12': 'cc-pVQZ-PP-F12',
                    'ccpcvdzf12': 'cc-pCVDZ-F12',
                    'ccpcvtzf12': 'cc-pCVTZ-F12',
                    'ccpcvqzf12': 'cc-pCVQZ-F12',
                },
        'gaussian':
                {
                    'sto3g': 'STO-3G',
                    'midix': 'MidiX',
                    '321g': '3-21G',
                    '431g': '4-31G',
                    '621g': '6-21G',
                    '631g': '6-31G',
                    #Missing: 6-31G dagger from Petersson
                    '631gd': '6-31G(d)',
                    '631gdp': '6-31G(d,p)',
                    '631g2d': '6-31G(2d)',
                    '631g2dp': '6-31G(2d,p)',
                    '631g2d2p': '6-31G(2d,2p)',
                    '631g2df': '6-31G(2df)',
                    '631g2df2p': '6-31G(2df,2p)',
                    '631g2df2pd': '6-31G(2df,2pd)',
                    '631+gd': '6-31+G(d)',
                    '631+gdp': '6-31+G(d,p)',
                    '631+g2d': '6-31+G(2d)',
                    '631+g2dp': '6-31+G(2d,p)',
                    '631+g2d2p': '6-31+G(2d,2p)',
                    '631+g2df': '6-31+G(2df)',
                    '631+g2df2p': '6-31+G(2df,2p)',
                    '631+g2df2pd': '6-31+G(2df,2pd)',
                    '631++gdp': '6-31++G(d,p)',
                    '631++g2dp': '6-31++G(2d,p)',
                    '631++g2d2p': '6-31++G(2d,2p)',
                    '631++g2df': '6-31++G(2df)',
                    '631++g2df2p': '6-31++G(2df,2p)',
                    '631++g2df2pd': '6-31++G(2df,2pd)',
                    '6311gd': '6-311G(d)',
                    '6311gdp': '6-311G(d,p)',
                    '6311g2d': '6-311G(2d)',
                    '6311g2dp': '6-311G(2d,p)',
                    '6311g2d2p': '6-311G(2d,2p)',
                    '6311g2df': '6-311G(2df)',
                    '6311g2df2p': '6-311G(2df,2p)',
                    '6311g2df2pd': '6-311G(2df,2pd)',
                    '6311g3df': '6-311G(3df)',
                    '6311g3df3pd': '6-311G(3df,3pd)',
                    '6311+gd': '6-311+G(d)',
                    '6311+gdp': '6-311+G(d,p)',
                    '6311+g2d': '6-311+G(2d)',
                    '6311+g2dp': '6-311+G(2d,p)',
                    '6311+g2d2p': '6-311+G(2d,2p)',
                    '6311+g2df': '6-311+G(2df)',
                    '6311+g2df2p': '6-311+G(2df,2p)',
                    '6311+g2df2pd': '6-311+G(2df,2pd)',
                    '6311+g3df': '6-311+G(3df)',
                    '6311+g3df2p': '6-311+G(3df,2p)',
                    '6311+g3df3pd': '6-311+G(3df,3pd)',
                    '6311++gdp': '6-311++G(d,p)',
                    '6311++g2d': '6-311++G(2d)',
                    '6311++g2dp': '6-311++G(2d,p)',
                    '6311++g2d2p': '6-311++G(2d,2p)',
                    '6311++g2df': '6-311++G(2df)',
                    '6311++g2df2p': '6-311++G(2df,2p)',
                    '6311++g2df2pd': '6-311++G(2df,2pd)',
                    '6311++g3df': '6-311++G(3df)',
                    '6311++g3df3pd': '6-311++G(3df,3pd)',
                    'w1mtsmall': 'MTSmall',
                    'sv': 'SV',
                    'svp': 'SVP',
                    'tzv': 'TZV',
                    'tzvp': 'TZVP',
                    'qzvp': 'QZVP',
                    'def2sv': 'Def2SV',
                    'def2svp': 'Def2SVP',
                    'def2svpp': 'Def2SVPP',
                    'def2tzv': 'Def2TZV',
                    'def2tzvp': 'Def2TZVP',
                    'def2tzvpp': 'Def2TZVPP',
                    'def2qzv': 'Def2QZV',
                    'def2qzvp': 'Def2QZVP',
                    'def2qzvpp': 'Def2QZVPP',
                    'ccpvdz': 'cc-pVDZ',
                    'ccpvtz': 'cc-pVTZ',
                    'ccpvqz': 'cc-pVQZ',
                    'ccpv5z': 'cc-pV5Z',
                    'ccpv6z': 'cc-pV6Z',
                    'augccpvdz': 'aug-cc-pVDZ',
                    'augccpvtz': 'aug-cc-pVTZ',
                    'augccpvqz': 'aug-cc-pVQZ',
                    'augccpv5z': 'aug-cc-pV5Z',
                    'augccpv6z': 'aug-cc-pV6Z',
                    'augccpvd+dz': 'dAug-cc-pVDZ',#to verify
                    'augccpvt+dz': 'dAug-cc-pVTZ',#to verify
                    'augccpvq+dz': 'dAug-cc-pVQZ',#to verify
                    'augccpv5+dz': 'dAug-cc-pV5Z',#to verify
                    'lanl2mb': 'LANL2MB',
                    'lanl2dz': 'LANL2DZ',
                    'd95': 'D95',
                    'd95v': 'D95V',
                    'eprii': 'EPR-II',
                    'epriii': 'EPR-III',
                    'ugbs': 'UGBS',#polarization missing
                    'shc': 'SHC',
                    'cep4g': 'CEP-4G',
                    'cep31g': 'CEP-31G',
                    'cep121g': 'CEP-121G',
                    'sdd': 'SDD',
                    'sddall': 'SDDAll',
                    'dgdzvp': 'DGDZVP',
                    'dgdzvp2': 'DGDZVP2',
                    'gdtzvp': 'DGTZVP',
                    'cbsb7': 'cbsb7',
                },
    }

SOFTWARE_SOLVENTS = {
         'gaussian':
                {
                    'water': 'water',
                    'acetonitrile': 'acetonitrile',
                    'methanol': 'methanol',
                    'ethanol': 'ethanol',
                    'isoquinoline': 'isoquinoline',
                    'quinoline': 'quinoline',
                    'chloroform': 'chloroform',
                    'diethylether': 'diethylether',
                    'dichloromethane': 'dichloromethane',
                    'dichloroethane': 'dichloroethane',
                    'carbontetrachloride': 'carbontetrachloride',
                    'benzene': 'benzene',
                    'toluene': 'toluene',
                    'chlorobenzene': 'chlorobenzene',
                    'nitromethane': 'nitromethane',
                    'heptane': 'heptane',
                    'cyclohexane': 'cyclohexane',
                    'aniline': 'aniline',
                    'acetone': 'acetone',
                    'tetrahydrofuran': 'tetrahydrofuran',
                    'dimethylsulfoxide': 'dimethylsulfoxide',
                    'argon': 'argon',
                    'krypton': 'krypton',
                    'xenon': 'xenon',
                    'n-octanol': 'n-octanol',
                    '1,1,1-trichloroethane': '1,1,1-trichloroethane',
                    '1,1,2-trichloroethane': '1,1,2-trichloroethane',
                    '1,2,4-trimethylbenzene': '1,2,4-trimethylbenzene',
                    '1,2-dibromoethane': '1,2-dibromoethane',
                    '1,2-ethanediol': '1,2-ethanediol',
                    '1,4-dioxane': '1,4-dioxane',
                    '1-bromo-2-methylpropane': '1-bromo-2-methylpropane',
                    '1-bromooctane': '1-bromooctane',
                    '1-bromopentane': '1-bromopentane',
                    '1-bromopropane': '1-bromopropane',
                    '1-butanol': '1-butanol',
                    '1-chlorohexane': '1-chlorohexane',
                    '1-chloropentane': '1-chloropentane',
                    '1-chloropropane': '1-chloropropane',
                    '1-decanol': '1-decanol',
                    '1-fluorooctane': '1-fluorooctane',
                    '1-heptanol': '1-heptanol',
                    '1-hexanol': '1-hexanol',
                    '1-hexene': '1-hexene',
                    '1-hexyne': '1-hexyne',
                    '1-iodobutane': '1-iodobutane',
                    '1-iodohexadecane': '1-iodohexadecane',
                    '1-iodopentane': '1-iodopentane',
                    '1-iodopropane': '1-iodopropane',
                    '1-nitropropane': '1-nitropropane',
                    '1-nonanol': '1-nonanol',
                    '1-pentanol': '1-pentanol',
                    '1-pentene': '1-pentene',
                    '1-propanol': '1-propanol',
                    '2,2,2-trifluoroethanol': '2,2,2-trifluoroethanol',
                    '2,2,4-trimethylpentane': '2,2,4-trimethylpentane',
                    '2,4-dimethylpentane': '2,4-dimethylpentane',
                    '2,4-dimethylpyridine': '2,4-dimethylpyridine',
                    '2,6-dimethylpyridine': '2,6-dimethylpyridine',
                    '2-bromopropane': '2-bromopropane',
                    '2-butanol': '2-butanol',
                    '2-chlorobutane': '2-chlorobutane',
                    '2-heptanone': '2-heptanone',
                    '2-hexanone': '2-hexanone',
                    '2-methoxyethanol': '2-methoxyethanol',
                    '2-methyl-1-propanol': '2-methyl-1-propanol',
                    '2-methyl-2-propanol': '2-methyl-2-propanol',
                    '2-methylpentane': '2-methylpentane',
                    '2-methylpyridine': '2-methylpyridine',
                    '2-nitropropane': '2-nitropropane',
                    '2-octanone': '2-octanone',
                    '2-pentanone': '2-pentanone',
                    '2-propanol': '2-propanol',
                    '2-propen-1-ol': '2-propen-1-ol',
                    '3-methylpyridine': '3-methylpyridine',
                    '3-pentanone': '3-pentanone',
                    '4-heptanone': '4-heptanone',
                    '4-methyl-2-pentanone': '4-methyl-2-pentanone',
                    '4-methylpyridine': '4-methylpyridine',
                    '5-nonanone': '5-nonanone',
                    'aceticacid': 'aceticacid',
                    'acetophenone': 'acetophenone',
                    'a-chlorotoluene': 'a-chlorotoluene',
                    'anisole': 'anisole',
                    'benzaldehyde': 'benzaldehyde',
                    'benzonitrile': 'benzonitrile',
                    'benzylalcohol': 'benzylalcohol',
                    'bromobenzene': 'bromobenzene',
                    'bromoethane': 'bromoethane',
                    'bromoform': 'bromoform',
                    'butanal': 'butanal',
                    'butanoicacid': 'butanoicacid',
                    'butanone': 'butanone',
                    'butanonitrile': 'butanonitrile',
                    'butylamine': 'butylamine',
                    'butylethanoate': 'butylethanoate',
                    'carbondisulfide': 'carbondisulfide',
                    'cis-1,2-dimethylcyclohexane': 'cis-1,2-dimethylcyclohexane',
                    'cis-decalin': 'cis-decalin',
                    'cyclohexanone': 'cyclohexanone',
                    'cyclopentane': 'cyclopentane',
                    'cyclopentanol': 'cyclopentanol',
                    'cyclopentanone': 'cyclopentanone',
                    'decalin-mixture': 'decalin-mixture',
                    'dibromomethane': 'dibromomethane',
                    'dibutylether': 'dibutylether',
                    'diethylamine': 'diethylamine',
                    'diethylsulfide': 'diethylsulfide',
                    'diiodomethane': 'diiodomethane',
                    'diisopropylether': 'diisopropylether',
                    'dimethyldisulfide': 'dimethyldisulfide',
                    'diphenylether': 'diphenylether',
                    'dipropylamine': 'dipropylamine',
                    'e-1,2-dichloroethene': 'e-1,2-dichloroethene',
                    'e-2-pentene': 'e-2-pentene',
                    'ethanethiol': 'ethanethiol',
                    'ethylbenzene': 'ethylbenzene',
                    'ethylethanoate': 'ethylethanoate',
                    'ethylmethanoate': 'ethylmethanoate',
                    'ethylphenylether': 'ethylphenylether',
                    'fluorobenzene': 'fluorobenzene',
                    'formamide': 'formamide',
                    'formicacid': 'formicacid',
                    'hexanoicacid': 'hexanoicacid',
                    'iodobenzene': 'iodobenzene',
                    'iodoethane': 'iodoethane',
                    'iodomethane': 'iodomethane',
                    'isopropylbenzene': 'isopropylbenzene',
                    'm-cresol': 'm-cresol',
                    'mesitylene': 'mesitylene',
                    'methylbenzoate': 'methylbenzoate',
                    'methylbutanoate': 'methylbutanoate',
                    'methylcyclohexane': 'methylcyclohexane',
                    'methylethanoate': 'methylethanoate',
                    'methylmethanoate': 'methylmethanoate',
                    'methylpropanoate': 'methylpropanoate',
                    'm-xylene': 'm-xylene',
                    'n-butylbenzene': 'n-butylbenzene',
                    'n-decane': 'n-decane',
                    'n-dodecane': 'n-dodecane',
                    'n-hexadecane': 'n-hexadecane',
                    'n-hexane': 'n-hexane',
                    'nitrobenzene': 'nitrobenzene',
                    'nitroethane': 'nitroethane',
                    'n-methylaniline': 'n-methylaniline',
                    'n-methylformamide-mixture': 'n-methylformamide-mixture',
                    'n,n-dimethylacetamide': 'n,n-dimethylacetamide',
                    'n,n-dimethylformamide': 'n,n-dimethylformamide',
                    'n-nonane': 'n-nonane',
                    'n-octane': 'n-octane',
                    'n-pentadecane': 'n-pentadecane',
                    'n-pentane': 'n-pentane',
                    'n-undecane': 'n-undecane',
                    'o-chlorotoluene': 'o-chlorotoluene',
                    'o-cresol': 'o-cresol',
                    'o-dichlorobenzene': 'o-dichlorobenzene',
                    'o-nitrotoluene': 'o-nitrotoluene',
                    'o-xylene': 'o-xylene',
                    'pentanal': 'pentanal',
                    'pentanoicacid': 'pentanoicacid',
                    'pentylamine': 'pentylamine',
                    'pentylethanoate': 'pentylethanoate',
                    'perfluorobenzene': 'perfluorobenzene',
                    'p-isopropyltoluene': 'p-isopropyltoluene',
                    'propanal': 'propanal',
                    'propanoicacid': 'propanoicacid',
                    'propanonitrile': 'propanonitrile',
                    'propylamine': 'propylamine',
                    'propylethanoate': 'propylethanoate',
                    'p-xylene': 'p-xylene',
                    'pyridine': 'pyridine',
                    'sec-butylbenzene': 'sec-butylbenzene',
                    'tert-butylbenzene': 'tert-butylbenzene',
                    'tetrachloroethene': 'tetrachloroethene',
                    'tetrahydrothiophene-s,s-dioxide': 'tetrahydrothiophene-s,s-dioxide',
                    'tetralin': 'tetralin',
                    'thiophene': 'thiophene',
                    'thiophenol': 'thiophenol',
                    'trans-decalin': 'trans-decalin',
                    'tributylphosphate': 'tributylphosphate',
                    'trichloroethene': 'trichloroethene',
                    'triethylamine': 'triethylamine',
                    'xylene-mixture': 'xylene-mixture',
                    'z-1,2-dichloroethene': 'z-1,2-dichloroethene',
                },
         'orca':
                {
                    'water': 'water',
                    'acetone': 'acetone',
                    'acetonitrile': 'acetonitrile',
                    'ammonia': 'ammonia',
                    'benzene': 'benzene',
                    'carbontetrachloride': 'ccl4',
                    'dichloromethane': 'ch2cl2',
                    'chloroform': 'chloroform',
                    'cyclohexane': 'cyclohexane',
                    'dimethylformamide': 'dmf',
                    'dimethylsulfoxide': 'dmso',
                    'ethanol': 'ethanol',
                    'n-hexane': 'hexane',
                    'methanol': 'methanol',
                    'n-octanol': 'octanol',#CPCM keyword, see calculation_helper.py
                    'pyridine': 'pyridine',
                    'tetrahydrofuran': 'thf',
                    'toluene': 'toluene',

                    '1,1,1-trichloroethane': '1,1,1-trichloroethane',
                    '1,1,2-trichloroethane': '1,1,2-trichloroethane',
                    '1,2,4-trimethylbenzene': '1,2,4-trimethylbenzene',
                    '1,2-dibromoethane': '1,2-dibromoethane',
                    '1,2-dichloroethane': '1,2-dichloroethane',
                    '1,2-ethanediol': '1,2-ethanediol',
                    '1,4-dioxane': '1,4-dioxane',
                    '1-bromo-2-methylpropane': '1-bromo-2-methylpropane',
                    '1-bromooctane': '1-bromooctane',
                    '1-bromopentane': '1-bromopentane',
                    '1-bromopropane': '1-bromopropane',
                    '1-butanol': '1-butanol',
                    '1-chlorohexane': '1-chlorohexane',
                    '1-chloropentane': '1-chloropentane',
                    '1-chloropropane': '1-chloropropane',
                    '1-decanol': '1-decanol',
                    '1-fluorooctane': '1-fluorooctane',
                    '1-heptanol': '1-heptanol',
                    '1-hexanol': '1-hexanol',
                    '1-hexene': '1-hexene',
                    '1-hexyne': '1-hexyne',
                    '1-iodobutane': '1-iodobutane',
                    '1-iodohexadecane': '1-iodohexadecane',
                    '1-iodopentane': '1-iodopentane',
                    '1-iodopropane': '1-iodopropane',
                    '1-nitropropane': '1-nitropropane',
                    '1-nonanol': '1-nonanol',
                    '1-octanol': '1-octanol',
                    '1-pentanol': '1-pentanol',
                    '1-pentene': '1-pentene',
                    '1-propanol': '1-propanol',
                    '2,2,2-trifluoroethanol': '2,2,2-trifluoroethanol',
                    '2,2,4-trimethylpentane': '2,2,4-trimethylpentane',
                    '2,4-dimethylpentane': '2,4-dimethylpentane',
                    '2,4-dimethylpyridine': '2,4-dimethylpyridine',
                    '2,6-dimethylpyridine': '2,6-dimethylpyridine',
                    '2-bromopropane': '2-bromopropane',
                    '2-butanol': '2-butanol',
                    'cyclopentane': 'cyclopentane',
                    'cyclopentanol': 'cyclopentanol',
                    'cyclopentanone': 'cyclopentanone',
                    'decalin (cis/trans mixture)': 'decalin (cis/trans mixture)',
                    'cis-decalin': 'cis-decalin',
                    'n-decane': 'n-decane',
                    'dibromomethane': 'dibromomethane',
                    'dibutylether': 'dibutylether',
                    'o-dichlorobenzene': 'o-dichlorobenzene',
                    'e-1,2-dichloroethene': 'e-1,2-dichloroethene',
                    'z-1,2-dichloroethene': 'z-1,2-dichloroethene',
                    'diethyl sulfide': 'diethyl sulfide',
                    'diethylamine': 'diethylamine',
                    'diiodomethane': 'diiodomethane',
                    'diisopropyl ether': 'diisopropyl ether',
                    'cis-1,2-dimethylcyclohexane': 'cis-1,2-dimethylcyclohexane',
                    'dimethyl disulfide': 'dimethyl disulfide',
                    'n,n-dimethylacetamide': 'n,n-dimethylacetamide',
                    'n,n-dimethylformamide': 'n,n-dimethylformamide',
                    'dimethylsulfoxide': 'dimethylsulfoxide',
                    'diphenylether': 'diphenylether',
                    'dipropylamine': 'dipropylamine',
                    'n-dodecane': 'n-dodecane',
                    'ethanethiol': 'ethanethiol',
                    'ethanol': 'ethanol',
                    'ethyl ethanoate': 'ethyl ethanoate',
                    'ethyl methanoate': 'ethyl methanoate',
                    'ethyl phenyl ether': 'ethyl phenyl ether',
                    'ethylbenzene': 'ethylbenzene',
                    'fluorobenzene': 'fluorobenzene',
                    'formamide': 'formamide',
                    'formic acid': 'formic acid',
                    'n-heptane': 'n-heptane',
                    'n-hexadecane': 'n-hexadecane',
                    'hexanoic acid': 'hexanoic acid',
                    '2-chlorobutane': '2-chlorobutane',
                    '2-heptanone': '2-heptanone',
                    '2-hexanone': '2-hexanone',
                    '2-methoxyethanol': '2-methoxyethanol',
                    '2-methyl-1-propanol': '2-methyl-1-propanol',
                    '2-methyl-2-propanol': '2-methyl-2-propanol',
                    '2-methylpentane': '2-methylpentane',
                    '2-methylpyridine': '2-methylpyridine',
                    '2-nitropropane': '2-nitropropane',
                    '2-octanone': '2-octanone',
                    '2-pentanone': '2-pentanone',
                    '2-propanol': '2-propanol',
                    '2-propen-1-ol': '2-propen-1-ol',
                    'e-2-pentene': 'e-2-pentene',
                    '3-methylpyridine': '3-methylpyridine',
                    '3-pentanone': '3-pentanone',
                    '4-heptanone': '4-heptanone',
                    '4-methyl-2-pentanone': '4-methyl-2-pentanone',
                    '4-methylpyridine': '4-methylpyridine',
                    '5-nonanone': '5-nonanone',
                    'acetic acid': 'acetic acid',
                    ' acetophenone': ' acetophenone',
                    'aniline': 'aniline',
                    'anisole': 'anisole',
                    'benzaldehyde': 'benzaldehyde',
                    'benzonitrile': 'benzonitrile',
                    'benzyl alcohol': 'benzyl alcohol',
                    'bromobenzene': 'bromobenzene',
                    'bromoethane': 'bromoethane',
                    'bromoform': 'bromoform',
                    'butanal': 'butanal',
                    'butanoic acid': 'butanoic acid',
                    'butanone': 'butanone',
                    'butanonitrile': 'butanonitrile',
                    'butyl ethanoate': 'butyl ethanoate',
                    'butylamine': 'butylamine',
                    'n-butylbenzene': 'n-butylbenzene',
                    'sec-butylbenzene': 'sec-butylbenzene',
                    'tert-butylbenzene': 'tert-butylbenzene',
                    'carbon disulfide': 'carbon disulfide',
                    'carbon tetrachloride': 'carbon tetrachloride',
                    ' chlorobenzene': ' chlorobenzene',
                    'chloroform': 'chloroform',
                    'a-chlorotoluene': 'a-chlorotoluene',
                    'o-chlorotoluene': 'o-chlorotoluene',
                    'm-cresol': 'm-cresol',
                    'o-cresol': 'o-cresol',
                    'cyclohexane': 'cyclohexane',
                    'cyclohexanone': 'cyclohexanone',
                    'iodobenzene': 'iodobenzene',
                    'iodoethane': 'iodoethane',
                    'iodomethane': 'iodomethane',
                    'isopropylbenzene': 'isopropylbenzene',
                    'p-isopropyltoluene': 'p-isopropyltoluene',
                    'mesitylene': 'mesitylene',
                    'methyl benzoate': 'methyl benzoate',
                    'methyl butanoate': 'methyl butanoate',
                    'methyl ethanoate': 'methyl ethanoate',
                    'methyl methanoate': 'methyl methanoate',
                    'methyl propanoate': 'methyl propanoate',
                    'n-methylaniline': 'n-methylaniline',
                    'methylcyclohexane': 'methylcyclohexane',
                    'n-methylformamide (e/z mixture)': 'n-methylformamide (e/z mixture)',
                    'nitrobenzene': 'nitrobenzene',
                    'nitroethane': 'nitroethane',
                    'nitromethane': 'nitromethane',
                    'o-nitrotoluene': 'o-nitrotoluene',
                    'n-nonane': 'n-nonane',
                    'n-octane': 'n-octane',
                    'n-pentadecane': 'n-pentadecane',
                    'pentanal': 'pentanal',
                    'n-pentane': 'n-pentane',
                    'pentanoic acid': 'pentanoic acid',
                    'pentyl ethanoate': 'pentyl ethanoate',
                    'pentylamine': 'pentylamine',
                    'perfluorobenzene': 'perfluorobenzene',
                    'propanal': 'propanal',
                    'propanoic acid': 'propanoic acid',
                    'propanonitrile': 'propanonitrile',
                    'propyl ethanoate': 'propyl ethanoate',
                    'propylamine': 'propylamine',
                    'tetrachloroethene': 'tetrachloroethene',
                    'tetrahydrothiophene-s,s-dioxide': 'tetrahydrothiophene-s,s-dioxide',
                    'tetralin': 'tetralin',
                    'thiophene': 'thiophene',
                    'thiophenol': 'thiophenol',
                    'trans-decalin': 'trans-decalin',
                    'tributylphosphate': 'tributylphosphate',
                    'trichloroethene': 'trichloroethene',
                    'triethylamine': 'triethylamine',
                    'n-undecane': 'n-undecane',
                    'xylene (mixture)': 'xylene (mixture)',
                    'm-xylene': 'm-xylene',
                    'o-xylene': 'o-xylene',
                    'p-xylene': 'p-xylene',
                },
        'xtb':
                {
                    'acetone': 'acetone',
                    'acetonitrile': 'acetonitrile',
                    'benzene': 'benzene',
                    'dichloromethane': 'ch2cl2',
                    'chloroform': 'chcl3',
                    'carbon disulfide': 'cs2',
                    'dimethylformamide': 'dmf',
                    'dimethylsulfoxide': 'dmso',
                    'diethylether': 'ether',
                    'water': 'h2o',
                    'methanol': 'methanol',
                    'n-hexane': 'n-hexane',
                    'tetrahydrofuran': 'thf',
                    'toluene': 'toluene',
                },
}

# The two lists below use the "absolute" method identifications.
# See SYN_METHODS for the full list of names and their synonyms.

# These lists contain all the functionals with D3 parameters developed
# by the Grimme group, which are not software specific. As such, the
# parameters might not be available in every QM package (although it could).

# Some functionals were not found in the list of known functionals or synonyms.
# These are indicated with the question mark.
FUNCTIONALS_WITH_DISPERSION_PARAMETERS = {
        'd3': [
                'b1b95',
                'b2gp-plyp',
                'b3lyp',
                'b97-d',
                'bhlyp', # BHandHLYP?
                'blyp',
                'bp86', # ?
                'bpbe', # HISSbPBE?
                'mpwlyp',
                'pbe',
                'pbe0',
                'pw6b95',
                'pwb6k', # ?
                'revpbe',
                'tpss',
                'tpss0',
                'tpssh',
                'bop', # ?
                'mpw1b95', # ?
                'mpwb1k', # ?
                'olyp',
                'opbe',
                'otpss', # ?
                'pbe38', # ?
                'pbesol', # ?
                'revssb', # ?
                'ssb', # ?
                'b3pw91', # ?
                'bmk',
                'cam-b3lyp',
                'lcwpbe', # ?
                'm052x',
                'm05',
                'm062x',
                'm06hf',
                'm06l',
                'm06',
                'hcth120', # ?
                'b2plyp',
                'dsdblyp', # ?
                'ptpss', # ?
                'pwpb95',
                'revpbe0', # ?
                'revpbe38', # ?
                'rpw86pbe',
            ],
        'd3bj': [
                '1b95',
                'b2gp-plyp',
                'b3pw91', # ?
                'bhlyp', # ?
                'bmk',
                'bop', # ?
                'bpbe', # ?
                'cam-b3lyp',
                'lcwpbe', # ?
                'mpw1b95', # ?
                'mpwb1k', # ?
                'mpwlyp',
                'olyp',
                'opbe',
                'otpss', # ?
                'pbe38', # ?
                'pbesol', # ?
                'ptpss', # ?
                'pwb6k', # ?
                'revssb', # ?
                'ssb', # ?
                'tpssh',
                'hcth120', # ?
                'b2plyp',
                'b3lyp',
                'b97-d',
                'blyp',
                'bp86', # ?
                'dsdblyp', # ?
                'pbe0',
                'pbe',
                'pw6b95',
                'pwpb95',
                'revpbe0', # ?
                'revpbe38', # ?
                'revpbe',
                'rpw86pbe',
                'tpss0',
                'tpss',
            ],
}

