"use strict";
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
class ReusableStateMachine extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const choice = new stepfunctions.Choice(this, 'Choice')
            .when(stepfunctions.Condition.stringEquals('$.branch', 'left'), new stepfunctions.Pass(this, 'Left Branch'))
            .when(stepfunctions.Condition.stringEquals('$.branch', 'right'), new stepfunctions.Pass(this, 'Right Branch'));
        this.startState = choice;
        this.endStates = choice.afterwards().endStates;
    }
}
class SimpleChain extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const task1 = new stepfunctions.Task(this, 'Task1', { task: new FakeTask() });
        this.task2 = new stepfunctions.Task(this, 'Task2', { task: new FakeTask() });
        task1.next(this.task2);
        this.startState = task1;
        this.endStates = [this.task2];
    }
    catch(state, props) {
        this.task2.addCatch(state, props);
        return this;
    }
}
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
class FakeTask {
    bind(_task) {
        return {
            resourceArn: 'resource'
        };
    }
}
module.exports = {
    'Basic composition': {
        'A single task is a State Machine'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const chain = new stepfunctions.Pass(stack, 'Some State');
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Some State',
                States: {
                    'Some State': { Type: 'Pass', End: true }
                }
            });
            test.done();
        },
        'A sequence of two tasks is a State Machine'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'You dont need to hold on to the state to render the entire state machine correctly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            task1.next(task2);
            // THEN
            test.deepEqual(render(task1), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'A chain can be appended to'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Pass(stack, 'State Three');
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2)
                .next(task3);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'A state machine can be appended to another state machine'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Wait(stack, 'State Three', {
                time: stepfunctions.WaitTime.duration(cdk.Duration.seconds(10))
            });
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(stepfunctions.Chain.start(task2).next(task3));
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Wait', End: true, Seconds: 10 },
                }
            });
            test.done();
        },
        'A state machine definition can be instantiated and chained'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const before = new stepfunctions.Pass(stack, 'Before');
            const after = new stepfunctions.Pass(stack, 'After');
            // WHEN
            const chain = before.next(new ReusableStateMachine(stack, 'Reusable')).next(after);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Before',
                States: {
                    'Before': { Type: 'Pass', Next: 'Choice' },
                    'Choice': {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.branch', StringEquals: 'left', Next: 'Left Branch' },
                            { Variable: '$.branch', StringEquals: 'right', Next: 'Right Branch' },
                        ]
                    },
                    'Left Branch': { Type: 'Pass', Next: 'After' },
                    'Right Branch': { Type: 'Pass', Next: 'After' },
                    'After': { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'A success state cannot be chained onto'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const succeed = new stepfunctions.Succeed(stack, 'Succeed');
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            test.throws(() => {
                pass.next(succeed).next(pass);
            });
            test.done();
        },
        'A failure state cannot be chained onto'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const fail = new stepfunctions.Fail(stack, 'Fail', { error: 'X', cause: 'Y' });
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            test.throws(() => {
                pass.next(fail).next(pass);
            });
            test.done();
        },
        'Parallels can contain direct states'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            const three = new stepfunctions.Pass(stack, 'Three');
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(one.next(two));
            para.branch(three);
            // THEN
            test.deepEqual(render(para), {
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'One',
                                States: {
                                    One: { Type: 'Pass', Next: 'Two' },
                                    Two: { Type: 'Pass', End: true },
                                }
                            },
                            {
                                StartAt: 'Three',
                                States: {
                                    Three: { Type: 'Pass', End: true }
                                }
                            }
                        ]
                    }
                }
            });
            test.done();
        },
        'Parallels can contain instantiated reusable definitions'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(new ReusableStateMachine(stack, 'Reusable1').prefixStates('Reusable1/'));
            para.branch(new ReusableStateMachine(stack, 'Reusable2').prefixStates('Reusable2/'));
            // THEN
            test.deepEqual(render(para), {
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Reusable1/Choice',
                                States: {
                                    'Reusable1/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable1/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable1/Right Branch' },
                                        ]
                                    },
                                    'Reusable1/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable1/Right Branch': { Type: 'Pass', End: true },
                                }
                            },
                            {
                                StartAt: 'Reusable2/Choice',
                                States: {
                                    'Reusable2/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable2/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable2/Right Branch' },
                                        ]
                                    },
                                    'Reusable2/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable2/Right Branch': { Type: 'Pass', End: true },
                                }
                            },
                        ]
                    }
                }
            });
            test.done();
        },
        'State Machine Fragments can be wrapped in a single state'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const reusable = new SimpleChain(stack, 'Hello');
            const state = reusable.toSingleState();
            test.deepEqual(render(state), {
                StartAt: 'Hello',
                States: {
                    Hello: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Hello: Task1',
                                States: {
                                    'Hello: Task1': { Type: 'Task', Next: 'Hello: Task2', Resource: 'resource' },
                                    'Hello: Task2': { Type: 'Task', End: true, Resource: 'resource' },
                                }
                            }
                        ],
                    },
                }
            });
            test.done();
        },
        'Chaining onto branched failure state ignores failure state'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const yes = new stepfunctions.Pass(stack, 'Yes');
            const no = new stepfunctions.Fail(stack, 'No', { error: 'Failure', cause: 'Wrong branch' });
            const enfin = new stepfunctions.Pass(stack, 'Finally');
            const choice = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), yes)
                .otherwise(no);
            // WHEN
            choice.afterwards().next(enfin);
            // THEN
            test.deepEqual(render(choice), {
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'No',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    No: { Type: 'Fail', Error: 'Failure', Cause: 'Wrong branch' },
                    Finally: { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'Can include OTHERWISE transition for Choice in afterwards()'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const chain = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), new stepfunctions.Pass(stack, 'Yes'))
                .afterwards({ includeOtherwise: true })
                .next(new stepfunctions.Pass(stack, 'Finally'));
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'Finally',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    Finally: { Type: 'Pass', End: true },
                }
            });
            test.done();
        }
    },
    'Goto support': {
        'State machines can have unconstrainted gotos'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            // WHEN
            const chain = one.next(two).next(one);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'One',
                States: {
                    One: { Type: 'Pass', Next: 'Two' },
                    Two: { Type: 'Pass', Next: 'One' },
                }
            });
            test.done();
        },
    },
    'Catches': {
        'States can have error branches'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addCatch(failure);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ]
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    }
                }
            });
            test.done();
        },
        'Retries and errors with a result path'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addRetry({ errors: ['HTTPError'], maxAttempts: 2 }).addCatch(failure, { resultPath: '$.some_error' }).next(failure);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Catch: [{ ErrorEquals: ['States.ALL'], Next: 'Failed', ResultPath: '$.some_error' }],
                        Retry: [{ ErrorEquals: ['HTTPError'], MaxAttempts: 2 }],
                        Next: 'Failed',
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    }
                }
            });
            test.done();
        },
        'Can wrap chain and attach error handler'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.next(task2).toSingleState('Wrapped').addCatch(errorHandler);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Wrapped',
                States: {
                    Wrapped: {
                        Type: 'Parallel',
                        Branches: [
                            {
                                StartAt: 'Task1',
                                States: {
                                    Task1: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        Next: 'Task2',
                                    },
                                    Task2: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        End: true,
                                    },
                                }
                            }
                        ],
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ],
                        End: true
                    },
                    ErrorHandler: { Type: 'Pass', End: true }
                },
            });
            test.done();
        },
        'Chaining does not chain onto error handler state'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler).next(task2);
            // THEN
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ]
                    },
                    Task2: { Type: 'Task', Resource: 'resource', End: true },
                    ErrorHandler: { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'Chaining does not chain onto error handler, extended'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const task3 = new stepfunctions.Task(stack, 'Task3', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler)
                .next(task2.addCatch(errorHandler))
                .next(task3.addCatch(errorHandler));
            // THEN
            const sharedTaskProps = { Type: 'Task', Resource: 'resource', Catch: [{ ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' }] };
            test.deepEqual(render(chain), {
                StartAt: 'Task1',
                States: {
                    Task1: { Next: 'Task2', ...sharedTaskProps },
                    Task2: { Next: 'Task3', ...sharedTaskProps },
                    Task3: { End: true, ...sharedTaskProps },
                    ErrorHandler: { Type: 'Pass', End: true },
                }
            });
            test.done();
        },
        'Error handler with a fragment'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            task1.addCatch(errorHandler)
                .next(new SimpleChain(stack, 'Chain').catch(errorHandler))
                .next(task2.addCatch(errorHandler));
            test.done();
        },
        'Can merge state machines with shared states'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            task1.addCatch(failure);
            task2.addCatch(failure);
            task1.next(task2);
            // THEN
            test.deepEqual(render(task1), {
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ]
                    },
                    Task2: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ]
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    }
                }
            });
            test.done();
        }
    },
    'State machine validation': {
        'No duplicate state IDs'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            state1.next(state2);
            // WHEN
            test.throws(() => {
                render(state1);
            });
            test.done();
        },
        'No duplicate state IDs even across Parallel branches'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            const parallel = new stepfunctions.Parallel(stack, 'Parallel')
                .branch(state1)
                .branch(state2);
            // WHEN
            test.throws(() => {
                render(parallel);
            });
            test.done();
        },
        'No cross-parallel jumps'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const state1 = new stepfunctions.Pass(stack, 'State1');
            const state2 = new stepfunctions.Pass(stack, 'State2');
            test.throws(() => {
                new stepfunctions.Parallel(stack, 'Parallel')
                    .branch(state1.next(state2))
                    .branch(state2);
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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