"use strict";
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
function createAppWithMap(mapFactory) {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'my-stack');
    const map = mapFactory(stack);
    new stepfunctions.StateGraph(map, 'Test Graph');
    return app;
}
module.exports = {
    'State Machine With Map State'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const map = new stepfunctions.Map(stack, 'Map State', {
            maxConcurrency: 1,
            itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
            parameters: {
                foo: 'foo',
                bar: stepfunctions.Data.stringAt('$.bar')
            }
        });
        map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
        // THEN
        test.deepEqual(render(map), {
            StartAt: 'Map State',
            States: {
                'Map State': {
                    Type: 'Map',
                    End: true,
                    Parameters: { foo: 'foo', bar: '$.bar' },
                    Iterator: {
                        StartAt: 'Pass State',
                        States: {
                            'Pass State': {
                                Type: 'Pass',
                                End: true
                            }
                        }
                    },
                    ItemsPath: '$.inputForMap',
                    MaxConcurrency: 1
                }
            }
        });
        test.done();
    },
    'synth is successful'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: 1,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        app.synth();
        test.done();
    },
    'fails in synthesis if iterator is missing'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: 1,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /Map state must have a non-empty iterator/, 'A validation was expected');
        test.done();
    },
    'fails in synthesis when maxConcurrency is a float'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: 1.2,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /maxConcurrency has to be a positive integer/, 'A validation was expected');
        test.done();
    },
    'fails in synthesis when maxConcurrency is a negative integer'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: -1,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /maxConcurrency has to be a positive integer/, 'A validation was expected');
        test.done();
    },
    'fails in synthesis when maxConcurrency is too big to be an integer'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: Number.MAX_VALUE,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /maxConcurrency has to be a positive integer/, 'A validation was expected');
        test.done();
    },
    'isPositiveInteger is false with negative number'(test) {
        test.equals(stepfunctions.isPositiveInteger(-1), false, '-1 is not a valid positive integer');
        test.done();
    },
    'isPositiveInteger is false with decimal number'(test) {
        test.equals(stepfunctions.isPositiveInteger(1.2), false, '1.2 is not a valid positive integer');
        test.done();
    },
    'isPositiveInteger is false with a value greater than safe integer '(test) {
        const valueToTest = Number.MAX_SAFE_INTEGER + 1;
        test.equals(stepfunctions.isPositiveInteger(valueToTest), false, `${valueToTest} is not a valid positive integer`);
        test.done();
    },
    'isPositiveInteger is true with 0'(test) {
        test.equals(stepfunctions.isPositiveInteger(0), true, '0 is expected to be a positive integer');
        test.done();
    },
    'isPositiveInteger is true with 10'(test) {
        test.equals(stepfunctions.isPositiveInteger(10), true, '10 is expected to be a positive integer');
        test.done();
    },
    'isPositiveInteger is true with max integer value'(test) {
        test.equals(stepfunctions.isPositiveInteger(Number.MAX_SAFE_INTEGER), true, `${Number.MAX_SAFE_INTEGER} is expected to be a positive integer`);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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