"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cargoLambdaVersion = exports.Bundling = void 0;
/* eslint-disable no-console */
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    constructor(projectRoot, props) {
        this.projectRoot = projectRoot;
        this.props = props;
        Bundling.runsLocally = Bundling.runsLocally
            ?? cargoLambdaVersion()
            ?? false;
        // Docker bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? props.dockerImage ?? cdk.DockerImage.fromRegistry('calavera/cargo-lambda:latest')
            : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const osPlatform = os.platform();
        const bundlingCommand = this.createBundlingCommand({
            osPlatform,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            packageName: props.packageName,
            binaryName: props.binaryName,
            architecture: props.architecture,
            lambdaExtension: props.lambdaExtension,
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        //Local bundling
        if (!props.forcedDockerBundling) { // only if Docker is not forced
            const createLocalCommand = (outputDir) => {
                return this.createBundlingCommand({
                    osPlatform,
                    outputDir: outputDir,
                    inputDir: projectRoot,
                    packageName: props.packageName,
                    binaryName: props.binaryName,
                    architecture: props.architecture,
                    lambdaExtension: props.lambdaExtension,
                });
            };
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('Rust build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...props.environment ?? {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: projectRoot,
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    static bundle(options) {
        const projectRoot = path.dirname(options.manifestPath);
        const bundling = new Bundling(projectRoot, options);
        return aws_lambda_1.Code.fromAsset(projectRoot, {
            assetHashType: options.assetHashType ?? cdk.AssetHashType.OUTPUT,
            assetHash: options.assetHash,
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
            },
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(props) {
        const buildBinary = [
            'cargo',
            'lambda',
            'build',
            '--release',
            '--lambda-dir',
            props.outputDir,
        ];
        if (props.lambdaExtension) {
            buildBinary.push('--extension');
        }
        if (props.architecture) {
            const targetFlag = props.architecture.name == aws_lambda_1.Architecture.ARM_64.name ? '--arm64' : '--x86-64';
            buildBinary.push(targetFlag);
        }
        let flattenPackage = props.packageName;
        if (props.binaryName) {
            buildBinary.push('--bin');
            buildBinary.push(props.binaryName);
            flattenPackage = props.binaryName;
        }
        if (!props.lambdaExtension && flattenPackage) {
            buildBinary.push('--flatten');
            buildBinary.push(flattenPackage);
        }
        return chain([
            ...this.props.commandHooks?.beforeBundling(props.inputDir, props.outputDir) ?? [],
            buildBinary.join(' '),
            ...this.props.commandHooks?.afterBundling(props.inputDir, props.outputDir) ?? [],
        ]);
    }
}
exports.Bundling = Bundling;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
function cargoLambdaVersion() {
    try {
        const cargo = child_process_1.spawnSync('cargo', ['lambda', '--version']);
        return cargo.status !== 0 || cargo.error ? undefined : true;
    }
    catch (err) {
        return undefined;
    }
}
exports.cargoLambdaVersion = cargoLambdaVersion;
//# sourceMappingURL=data:application/json;base64,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