"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const handler_1 = require("./handler");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html).
 *
 * @stability stable
 */
var Tracing;
(function (Tracing) {
    Tracing["ACTIVE"] = "Active";
    Tracing["PASS_THROUGH"] = "PassThrough";
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 *
 * @stability stable
 */
class Function extends function_base_1.FunctionBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.functionName,
        });
        /**
         * The construct node where permissions are attached.
         *
         * @stability stable
         */
        this.permissionsNode = this.node;
        /**
         * Whether the addPermission() call adds any permissions.
         *
         * True for new Lambdas, false for version $LATEST and imported Lambdas
         * from different accounts.
         *
         * @stability stable
         */
        this.canCreatePermissions = true;
        this.layers = [];
        /**
         * Environment variables for this function
         */
        this.environment = {};
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additonal managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _a;
                    (_a = this.role) === null || _a === void 0 ? void 0 : _a.addToPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfilingEnvironmentVariables(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfilingEnvironmentVariables(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup', {
                computePlatform: aws_codeguruprofiler_1.ComputePlatform.AWS_LAMBDA,
            });
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        const env = { ...profilingGroupEnvironmentVariables, ...props.environment };
        for (const [key, value] of Object.entries(env)) {
            this.addEnvironment(key, value);
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        const UNDEFINED_MARKER = '$$$undefined';
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
                imageUri: (_a = code.image) === null || _a === void 0 ? void 0 : _a.imageUri,
            },
            layers: core_1.Lazy.list({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler === handler_1.Handler.FROM_IMAGE ? UNDEFINED_MARKER : props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            packageType: props.runtime === runtime_1.Runtime.FROM_IMAGE ? 'Image' : undefined,
            runtime: props.runtime === runtime_1.Runtime.FROM_IMAGE ? UNDEFINED_MARKER : (_b = props.runtime) === null || _b === void 0 ? void 0 : _b.name,
            role: this.role.roleArn,
            // Uncached because calling '_checkEdgeCompatibility', which gets called in the resolve of another
            // Token, actually *modifies* the 'environment' map.
            environment: core_1.Lazy.uncachedAny({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
            imageConfig: undefinedIfNoKeys({
                command: (_c = code.image) === null || _c === void 0 ? void 0 : _c.cmd,
                entryPoint: (_d = code.image) === null || _d === void 0 ? void 0 : _d.entrypoint,
            }),
            kmsKeyArn: (_e = props.environmentEncryption) === null || _e === void 0 ? void 0 : _e.keyArn,
        });
        // since patching the CFN spec to make Runtime and Handler optional causes a
        // change in the order of the JSON keys, which results in a change of
        // function hash (and invalidation of all lambda functions everywhere), we
        // are using a marker to indicate this fields needs to be erased using an
        // escape hatch. this should be fixed once the new spec is published and a
        // patch is no longer needed.
        if (resource.runtime === UNDEFINED_MARKER) {
            resource.addPropertyOverride('Runtime', undefined);
        }
        if (resource.handler === UNDEFINED_MARKER) {
            resource.addPropertyOverride('Handler', undefined);
        }
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logRetention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            resource.addPropertyOverride('FileSystemConfigs', [
                {
                    LocalMountPath: config.localMountPath,
                    Arn: config.arn,
                },
            ]);
        }
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this Lambda function. A new version will be created every time the function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     *
     * @stability stable
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        // override the version's logical ID with a lazy string which includes the
        // hash of the function itself, so a new version resource is created when
        // the function configuration changes.
        const cfn = this._currentVersion.node.defaultChild;
        const originalLogicalId = this.stack.resolve(cfn.logicalId);
        cfn.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: () => {
                const hash = function_hash_1.calculateFunctionHash(this);
                const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
                return `${logicalId}${hash}`;
            },
        }));
        return this._currentVersion;
    }
    /**
     * @stability stable
     */
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined within this stack.
     *
     * @param scope The parent construct.
     * @param id The name of the lambda construct.
     * @param attrs the attributes of the function to import.
     * @stability stable
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                var _a;
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = (_a = attrs.sameEnvironment) !== null && _a !== void 0 ? _a : this._isStackAccount();
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda.
     *
     * @stability stable
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     *
     * If this is a ref to a Lambda function, this operation results in a no-op.
     *
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     * @stability stable
     */
    addEnvironment(key, value, options) {
        this.environment[key] = { value, ...options };
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     * @stability stable
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * (deprecated) Add a new version for this Lambda.
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked asynchronously.
     * @returns A new Version object.
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     *
     * @stability stable
     */
    get logGroup() {
        if (!this._logGroup) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logRetention.logGroupArn);
        }
        return this._logGroup;
    }
    /** @internal */
    _checkEdgeCompatibility() {
        // Check env vars
        const envEntries = Object.entries(this.environment);
        for (const [key, config] of envEntries) {
            if (config.removeInEdge) {
                delete this.environment[key];
                core_1.Annotations.of(this).addInfo(`Removed ${key} environment variable for Lambda@Edge compatibility`);
            }
        }
        const envKeys = Object.keys(this.environment);
        if (envKeys.length !== 0) {
            throw new Error(`The function ${this.node.path} contains environment variables [${envKeys}] and is not compatible with Lambda@Edge. \
Environment variables can be marked for removal when used in Lambda@Edge by setting the \'removeInEdge\' property in the \'addEnvironment()\' API.`);
        }
        return;
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        const variables = {};
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it). For backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. Otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        const keys = this._currentVersion
            ? Object.keys(this.environment).sort()
            : Object.keys(this.environment);
        for (const key of keys) {
            variables[key] = this.environment[key].value;
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        var _a;
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        const allowPublicSubnet = (_a = props.allowPublicSubnet) !== null && _a !== void 0 ? _a : false;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId) && !allowPublicSubnet) {
                throw new Error('Lambda Functions in a public subnet can NOT access the internet. ' +
                    'If you are aware of this limitation and would still like to place the function int a public subnet, set `allowPublicSubnet` to true');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfilingEnvironmentVariables(props) {
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, props) {
    // mutually exclusive
    const codeType = [code.inlineCode, code.s3Location, code.image];
    if (codeType.filter(x => !!x).length !== 1) {
        throw new Error('lambda.Code must specify exactly one of: "inlineCode", "s3Location", or "image"');
    }
    if (!!code.image === (props.handler !== handler_1.Handler.FROM_IMAGE)) {
        throw new Error('handler must be `Handler.FROM_IMAGE` when using image asset for Lambda function');
    }
    if (!!code.image === (props.runtime !== runtime_1.Runtime.FROM_IMAGE)) {
        throw new Error('runtime must be `Runtime.FROM_IMAGE` when using image asset for Lambda function');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !props.runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${props.runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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