#
# Copyright (C) 2009-2020 the sqlparse authors and contributors
# <see AUTHORS file>
#
# This module is part of python-sqlparse and is released under
# the BSD License: https://opensource.org/licenses/BSD-3-Clause

from atc.configurator.sql.sqlparse import sql
from atc.configurator.sql.sqlparse import tokens as T


class OutputFilter:
    varname_prefix = ""

    def __init__(self, varname="sql"):
        self.varname = self.varname_prefix + varname
        self.count = 0

    def _process(self, stream, varname, has_nl):
        raise NotImplementedError

    def process(self, stmt):
        self.count += 1
        if self.count > 1:
            varname = "{f.varname}{f.count}".format(f=self)
        else:
            varname = self.varname

        has_nl = len(str(stmt).strip().splitlines()) > 1
        stmt.tokens = self._process(stmt.tokens, varname, has_nl)
        return stmt


class OutputPythonFilter(OutputFilter):
    def _process(self, stream, varname, has_nl):
        # SQL query assignation to varname
        if self.count > 1:
            yield sql.Token(T.Whitespace, "\n")
        yield sql.Token(T.Name, varname)
        yield sql.Token(T.Whitespace, " ")
        yield sql.Token(T.Operator, "=")
        yield sql.Token(T.Whitespace, " ")
        if has_nl:
            yield sql.Token(T.Operator, "(")
        yield sql.Token(T.Text, "'")

        # Print the tokens on the quote
        for token in stream:
            # Token is a new line separator
            if token.is_whitespace and "\n" in token.value:
                # Close quote and add a new line
                yield sql.Token(T.Text, " '")
                yield sql.Token(T.Whitespace, "\n")

                # Quote header on secondary lines
                yield sql.Token(T.Whitespace, " " * (len(varname) + 4))
                yield sql.Token(T.Text, "'")

                # Indentation
                after_lb = token.value.split("\n", 1)[1]
                if after_lb:
                    yield sql.Token(T.Whitespace, after_lb)
                continue

            # Token has escape chars
            elif "'" in token.value:
                token.value = token.value.replace("'", "\\'")

            # Put the token
            yield sql.Token(T.Text, token.value)

        # Close quote
        yield sql.Token(T.Text, "'")
        if has_nl:
            yield sql.Token(T.Operator, ")")


class OutputPHPFilter(OutputFilter):
    varname_prefix = "$"

    def _process(self, stream, varname, has_nl):
        # SQL query assignation to varname (quote header)
        if self.count > 1:
            yield sql.Token(T.Whitespace, "\n")
        yield sql.Token(T.Name, varname)
        yield sql.Token(T.Whitespace, " ")
        if has_nl:
            yield sql.Token(T.Whitespace, " ")
        yield sql.Token(T.Operator, "=")
        yield sql.Token(T.Whitespace, " ")
        yield sql.Token(T.Text, '"')

        # Print the tokens on the quote
        for token in stream:
            # Token is a new line separator
            if token.is_whitespace and "\n" in token.value:
                # Close quote and add a new line
                yield sql.Token(T.Text, ' ";')
                yield sql.Token(T.Whitespace, "\n")

                # Quote header on secondary lines
                yield sql.Token(T.Name, varname)
                yield sql.Token(T.Whitespace, " ")
                yield sql.Token(T.Operator, ".=")
                yield sql.Token(T.Whitespace, " ")
                yield sql.Token(T.Text, '"')

                # Indentation
                after_lb = token.value.split("\n", 1)[1]
                if after_lb:
                    yield sql.Token(T.Whitespace, after_lb)
                continue

            # Token has escape chars
            elif '"' in token.value:
                token.value = token.value.replace('"', '\\"')

            # Put the token
            yield sql.Token(T.Text, token.value)

        # Close quote
        yield sql.Token(T.Text, '"')
        yield sql.Token(T.Punctuation, ";")
