
'''
BLURB GOES HERE.

Copyright (c) 2015 - 2022 Rob "N3X15" Nelson <nexisentertainment@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

'''
import os, sys
from pathlib import Path
from typing import List, Optional, Tuple

from buildtools import os_utils
from buildtools.maestro.base_target import SingleBuildTarget


class NuitkaTarget(SingleBuildTarget):
    BT_LABEL = 'NUITKA'

    def __init__(self, entry_point: str, package_name: str, files: List[str], dependencies: List[str]=[], tmp_dir: Optional[str]=None, single_file: bool=True, nuitka_subdir: str = None) -> None:
        if tmp_dir is None:
            tmp_dir = Path('.tmp')
        else:
            tmp_dir = Path(tmp_dir)
        self.python_path = sys.executable
        self.entry_point: Path = Path(entry_point)
        self.package_name: str = package_name
        self.out_dir: Path = tmp_dir / 'nuitka'
        if nuitka_subdir:
            self.out_dir = self.out_dir / nuitka_subdir
        self.dist_dir: Path = self.out_dir / f'{self.package_name}.dist'
        self.singlefile: bool = single_file
        self.executable_mangled: Path = self.dist_dir / self.package_name if not single_file else (self.out_dir / self.package_name).with_suffix('.bin')

        if os.name == 'nt':
            self.executable_mangled = self.executable_mangled.with_suffix('.exe')

        self.windows_product_name: str = 'Nuitka Executable'
        self.windows_company_name: str = 'Nuitka Executable Contributors'
        self.windows_file_version: Tuple[int, int, int, int] = (0,0,0,0)
        self.windows_file_description: str = 'FIXME: Review target.windows_product_name, target.windows_company_name, target.windows_file_version, and target.windows_file_description.'

        super().__init__(target=str(self.executable_mangled), files=files, dependencies=dependencies)

        self.included_packages: List[str] = []
        self.enabled_plugins: List[str] = []

    def build(self):
        opts: List[str] = [
            '--prefer-source-code',
            '--assume-yes-for-downloads',
            #'--recurse-all',
            #'--follow-imports',
        ]
        if os.name != 'nt':
            opts += ['--static-libpython=yes']
        for pkg in self.included_packages:
            opts += ['--include-package', pkg]
        for plug in self.enabled_plugins:
            opts += ['--enable-plugin', plug]
        opts += [
            '--output-dir', str(self.out_dir),
            #'--show-progress', # *screaming*
            '--standalone'
        ]
        if self.singlefile:
            opts += ['--onefile']
        if os_utils.is_windows():
            opts += [
                f'--windows-product-name={self.windows_product_name}',
                f'--windows-company-name={self.windows_company_name}',
                f'--windows-file-version='+('.'.join(map(str, self.windows_file_version))),
                f'--windows-file-description={self.windows_file_description}',
            ]
        nuitka_cmd: List[str] = [str(sys.executable), '-m', 'nuitka']
        nuitka_cmd += opts
        nuitka_cmd.append(str(self.entry_point))
        os_utils.cmd(nuitka_cmd, echo=True, show_output=True, critical=True, globbify=False)
