"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpsRedirect = void 0;
const crypto = require("crypto");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * Allows creating a domainA -> domainB redirect using CloudFront and S3.
 * You can specify multiple domains to be redirected.
 */
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const domainNames = (_a = props.recordNames) !== null && _a !== void 0 ? _a : [props.zone.zoneName];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`The certificate must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            });
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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