#!/usr/bin/env python3
#
#  __init__.py
"""
A tool to check all modules can be correctly imported.
"""
#
#  Copyright © 2021 Dominic Davis-Foster <dominic@davis-foster.co.uk>
#
#  Permission is hereby granted, free of charge, to any person obtaining a copy
#  of this software and associated documentation files (the "Software"), to deal
#  in the Software without restriction, including without limitation the rights
#  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
#  copies of the Software, and to permit persons to whom the Software is
#  furnished to do so, subject to the following conditions:
#
#  The above copyright notice and this permission notice shall be included in all
#  copies or substantial portions of the Software.
#
#  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
#  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
#  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
#  DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
#  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
#  OR OTHER DEALINGS IN THE SOFTWARE.
#

# stdlib
import contextlib
import importlib
import importlib.machinery
import importlib.util
import traceback
from io import StringIO
from typing import Any, Dict, Iterator, List, Mapping, NamedTuple, Tuple, Union, cast

# 3rd party
import toml
from domdf_python_tools.doctools import prettify_docstrings
from domdf_python_tools.paths import PathPlus
from domdf_python_tools.typing import PathLike
from domdf_python_tools.words import Plural
from packaging.markers import Marker
from typing_extensions import TypedDict

__all__ = ["ConfigDict", "Error", "OK", "check_module", "evaluate_markers", "load_toml", "redirect_output"]

__author__: str = "Dominic Davis-Foster"
__copyright__: str = "2021 Dominic Davis-Foster"
__license__: str = "MIT License"
__version__: str = "0.1.0"
__email__: str = "dominic@davis-foster.co.uk"

_module = Plural("module", "modules")


class ConfigDict(TypedDict, total=False):
	"""
	:class:`typing.TypedDict` representing the configuration mapping parsed from ``pyproject.toml`` or similar.
	"""

	#: List of modules to always try to import.
	always: List[str]

	#: Mapping of :pep:`508` markers to lists of imports to try to import if the markers evaluate to :py:obj:`True`.
	only_if: Mapping[str, List[str]]

	#: Configuration for ``importcheck``.
	config: Dict[str, Any]


def load_toml(filename: PathLike) -> ConfigDict:
	"""
	Load the ``importcheck`` configuration mapping from the given TOML file.

	:param filename:
	"""

	config = toml.loads(PathPlus(filename).read_text())

	if "importcheck" in config:
		return cast(ConfigDict, config["importcheck"])
	elif "tool" in config and "importcheck" in config["tool"]:
		return cast(ConfigDict, config["tool"]["importcheck"])
	else:
		raise KeyError("No such table 'importcheck' or 'tool.importcheck'")


def evaluate_markers(config: ConfigDict) -> List[str]:
	"""
	Evaluate the markers in the ``only_if`` key and return a list of all modules to try to import.

	:param config:
	"""

	modules_to_check: List[str] = []

	if "always" in config:
		modules_to_check.extend(config["always"])

	if "only_if" in config:
		for marker, modules in config["only_if"].items():
			if Marker(marker).evaluate():
				modules_to_check.extend(modules)

	return modules_to_check


@prettify_docstrings
class OK(NamedTuple):
	"""
	Returned by :func:`~.check_module` if the module is successfully imported.
	"""

	#: The name of the module being checked.
	module: str

	@property
	def stdout(self):  # noqa: D102
		raise NotImplementedError

	@property
	def stderr(self):  # noqa: D102
		raise NotImplementedError

	def __bool__(self):
		"""
		:class:`~.OK` objects always evaluate as :py:obj:`False`.
		"""
		return False


@prettify_docstrings
class Error(NamedTuple):
	"""
	Returned by :func:`~.check_module` if the module could not be successfully imported.
	"""

	#: The name of the module being checked.
	module: str

	stdout: str
	"""
	The standard output from importing the module.

	This may also contain standard error if the streams are combined by :func:`~.check_module`
	"""

	stderr: str
	"""
	Standard error generated by importing the module.

	This may also contain standard out if the streams are combined by :func:`~.check_module`.
	"""

	def __bool__(self):
		return True


@contextlib.contextmanager
def redirect_output(combine: bool = False) -> Iterator[Tuple[StringIO, StringIO]]:
	"""
	Context manager to redirect stdout and stderr to two :class:`io.StringIO` objects.

	These are assigned (as a :class:`tuple`) to the target the :keyword:`as` expression.

	Example:

	.. code-block:: python

		with redirect_output() as (stdout, stderr):
			...

	:param combine: If :py:obj:`True` ``stderr`` is combined with ``stdout``.
	"""  # noqa: D400

	if combine:
		stdout = stderr = StringIO()
	else:
		stdout = StringIO()
		stderr = StringIO()

	with contextlib.redirect_stdout(stdout):
		with contextlib.redirect_stderr(stderr):
			yield stdout, stderr


def check_module(module: str, combine_output: bool = False) -> Union[OK, Error]:
	"""
	Try to import ``module``, otherwise handle the resulting error.

	:param module:
	:param combine_output: If :py:obj:`True` ``stderr`` is combined with ``stdout``.
	"""

	with redirect_output(combine_output) as (stdout, stderr):
		try:
			importlib.import_module(module)
			return OK(module)
		except Exception as e:
			traceback_frames = traceback.extract_tb(e.__traceback__)
			tb_e = traceback.TracebackException(
					type(e),
					e,
					e.__traceback__,  # type: ignore
					)

			if traceback_frames[0].filename == __file__:
				del traceback_frames[0]

			buf = ['Traceback (most recent call last):\n']
			buf.extend(traceback.format_list(traceback_frames))

			while buf[-1] == '\n':
				del buf[-1]

			buf.extend(tb_e.format_exception_only())

			print(''.join(buf), file=stderr)

			return Error(module, stdout.getvalue(), stderr.getvalue())
