"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
// CDK parameters
const CONSTRUCT_NAME = 'aws-cdk-dynamodb-global';
// DynamoDB table parameters
const TABLE_NAME = 'GlobalTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: aws_dynamodb_1.AttributeType.STRING };
const STACK_PROPS = {
    partitionKey: TABLE_PARTITION_KEY,
    tableName: TABLE_NAME,
    regions: ['us-east-1', 'us-east-2', 'us-west-2'],
};
module.exports = {
    'Default Global DynamoDB stack': {
        'global dynamo'(test) {
            const stack = new core_1.Stack();
            new lib_1.GlobalTable(stack, CONSTRUCT_NAME, STACK_PROPS);
            const topStack = stack.node.findChild(CONSTRUCT_NAME);
            for (const reg of STACK_PROPS.regions) {
                const tableStack = topStack.node.findChild(CONSTRUCT_NAME + '-' + reg);
                assert_1.expect(tableStack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_AND_OLD_IMAGES',
                    },
                    'TableName': 'GlobalTable',
                }));
            }
            const customResourceStack = stack.node.findChild(CONSTRUCT_NAME + '-CustomResource');
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Description: 'Lambda to make DynamoDB a global table',
                Handler: 'index.handler',
                Timeout: 300,
            }));
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
                Regions: STACK_PROPS.regions,
                ResourceType: 'Custom::DynamoGlobalTableCoordinator',
                TableName: TABLE_NAME,
            }));
            test.done();
        },
    },
    'GlobalTable generated stacks inherit their account from the parent stack'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'GlobalTableStack', { env: { account: '123456789012', region: 'us-east-1' } });
        const globalTable = new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            partitionKey: TABLE_PARTITION_KEY,
            regions: ['us-east-1', 'us-west-2'],
            stream: aws_dynamodb_1.StreamViewType.NEW_AND_OLD_IMAGES,
        });
        new core_1.CfnOutput(stack, 'DynamoDbOutput', {
            // this works, because both `stack` and `regionTables[0]` stack are in the same account & region
            value: globalTable.regionalTables[0].tableStreamArn,
        });
        assert_1.expect(stack).toMatch({
            'Outputs': {
                'DynamoDbOutput': {
                    'Value': {
                        'Fn::ImportValue': 'GlobalTableStackawscdkdynamodbglobalawscdkdynamodbglobaluseast19C1C8A14:awscdkdynamodbglobalawscdkdynamodbglobaluseast1ExportsOutputFnGetAttawscdkdynamodbglobalGlobalTableuseast1FC03DD69StreamArn28E90DB8',
                    },
                },
            },
        });
        test.done();
    },
    'Enforce StreamSpecification': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                    tableName: TABLE_NAME,
                    stream: aws_dynamodb_1.StreamViewType.KEYS_ONLY,
                    partitionKey: TABLE_PARTITION_KEY,
                    regions: ['us-east-1', 'us-east-2', 'us-west-2'],
                });
            }, /dynamoProps.stream MUST be set to dynamodb.StreamViewType.NEW_AND_OLD_IMAGES/);
            test.done();
        },
    },
    'Check getting tables': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            const regTables = new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                partitionKey: TABLE_PARTITION_KEY,
                regions: ['us-east-1', 'us-east-2', 'us-west-2'],
            });
            test.equal(regTables.regionalTables.length, 3);
            for (const table of regTables.regionalTables) {
                test.ok(table instanceof aws_dynamodb_1.Table);
            }
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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