"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const aws_kinesisanalytics_1 = require("@aws-cdk/aws-kinesisanalytics");
const logs = require("@aws-cdk/aws-logs");
const core = require("@aws-cdk/core");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /**
     * (experimental) Implement the convenience {@link IApplication.addToPrincipalPolicy} method.
     *
     * @experimental
     */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        this.applicationName = attrs.applicationName;
    }
}
/**
 * (experimental) The L2 construct for Flink Kinesis Data Applications.
 *
 * @experimental
 * @resource AWS::KinesisAnalyticsV2::Application
 */
class Application extends ApplicationBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, { physicalName: props.applicationName });
        validation_1.validateFlinkApplicationProps(props);
        this.role = (_b = props.role) !== null && _b !== void 0 ? _b : new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        const resource = new aws_kinesisanalytics_1.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: environment_properties_1.environmentProperties(props.propertyGroups),
                flinkApplicationConfiguration: flink_application_configuration_1.flinkApplicationConfiguration({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: (_c = props.snapshotsEnabled) !== null && _c !== void 0 ? _c : true,
                },
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = (_d = props.logGroup) !== null && _d !== void 0 ? _d : new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    sep: ':',
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new aws_kinesisanalytics_1.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
    /**
     * (experimental) Import an existing Flink application defined outside of CDK code by applicationName.
     *
     * @experimental
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn, applicationName });
    }
    /**
     * (experimental) Import an existing application defined outside of CDK code by applicationArn.
     *
     * @experimental
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const applicationName = core.Stack.of(scope).parseArn(applicationArn).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${applicationArn}) must include resource name`);
        }
        return new Import(scope, id, { applicationArn, applicationName });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-kinesisanalytics-flink.Application", version: "1.132.0" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBsaWNhdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHdDQUF3QztBQUN4Qyx3RUFBMEc7QUFDMUcsMENBQTBDO0FBQzFDLHNDQUFzQztBQUd0Qyw2RUFBeUU7QUFDekUsK0ZBQTBGO0FBQzFGLHFEQUFpRztBQWtCakc7OztHQUdHO0FBQ0gsTUFBZSxlQUFnQixTQUFRLElBQUksQ0FBQyxRQUFROzs7Ozs7SUFTM0MsZUFBZSxDQUFDLGVBQW9DO1FBQ3pELElBQUksSUFBSSxDQUFDLElBQUksRUFBRTtZQUNiLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDaEQsT0FBTyxJQUFJLENBQUM7U0FDYjtRQUVELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztDQUNGO0FBcUREOztHQUVHO0FBQ0gsTUFBTSxNQUFPLFNBQVEsZUFBZTtJQU1sQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBEO1FBQ2xHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsa0VBQWtFO1FBQ2xFLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNuRSxJQUFJLENBQUMsSUFBSSxHQUFHLFNBQVMsQ0FBQztRQUV0QixJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7UUFDM0MsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO0lBQy9DLENBQUM7Q0FDRjs7Ozs7OztBQUdELE1BQWEsV0FBWSxTQUFRLGVBQWU7Ozs7SUEwQjlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7O1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsWUFBWSxFQUFFLEtBQUssQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1FBQzFELDBDQUF3QixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRWhDLElBQUksQ0FBQyxJQUFJLFNBQUcsS0FBSyxDQUFDLElBQUksbUNBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDbkQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLGdDQUFnQyxDQUFDO1NBQ3RFLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztRQUVoQyx5Q0FBeUM7UUFDekMsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDckQsT0FBTyxFQUFFLENBQUMsMEJBQTBCLENBQUM7WUFDckMsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBRUosTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFNUIsTUFBTSxRQUFRLEdBQUcsSUFBSSx1Q0FBZ0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3RELGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtZQUN0QyxrQkFBa0IsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUs7WUFDdkMsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ3ZDLHdCQUF3QixFQUFFO2dCQUN4QixHQUFHLElBQUksQ0FBQyxvQ0FBb0M7Z0JBQzVDLHFCQUFxQixFQUFFLDhDQUFxQixDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7Z0JBQ2xFLDZCQUE2QixFQUFFLCtEQUE2QixDQUFDO29CQUMzRCxvQkFBb0IsRUFBRSxLQUFLLENBQUMsb0JBQW9CO29CQUNoRCxrQkFBa0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCO29CQUM1QywwQkFBMEIsRUFBRSxLQUFLLENBQUMsMEJBQTBCO29CQUM1RCxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7b0JBQ3hCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtvQkFDaEMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtvQkFDNUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO29CQUM5QixpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO2lCQUMzQyxDQUFDO2dCQUNGLGdDQUFnQyxFQUFFO29CQUNoQyxnQkFBZ0IsUUFBRSxLQUFLLENBQUMsZ0JBQWdCLG1DQUFJLElBQUk7aUJBQ2pEO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdkMsTUFBTSxRQUFRLFNBQUcsS0FBSyxDQUFDLFFBQVEsbUNBQUksSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUN2RSxNQUFNLFNBQVMsR0FBRyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFFdEUsb0JBQW9CO1FBRXBCLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3JELE9BQU8sRUFBRSxDQUFDLHdCQUF3QixDQUFDO1lBQ25DLFNBQVMsRUFBRTtnQkFDVCxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQzVCLE9BQU8sRUFBRSxNQUFNO29CQUNmLFFBQVEsRUFBRSxXQUFXO29CQUNyQixHQUFHLEVBQUUsR0FBRztvQkFDUixZQUFZLEVBQUUsR0FBRztpQkFDbEIsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNyRCxPQUFPLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQztZQUNwQyxTQUFTLEVBQUUsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDO1NBQ2xDLENBQUMsQ0FBQyxDQUFDO1FBRUosTUFBTSxZQUFZLEdBQUcsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVMsU0FBUyxJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLFVBQVUsY0FBYyxRQUFRLENBQUMsWUFBWSxlQUFlLFNBQVMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUN6SyxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNyRCxPQUFPLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQztZQUM5QixTQUFTLEVBQUUsQ0FBQyxZQUFZLENBQUM7U0FDMUIsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLDhEQUF1QyxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7WUFDakUsZUFBZSxFQUFFLFFBQVEsQ0FBQyxHQUFHO1lBQzdCLHVCQUF1QixFQUFFO2dCQUN2QixZQUFZO2FBQ2I7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbkUsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQ2hELElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsRUFDckUsd0JBQXdCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUM1QyxDQUFDO1FBRUYsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDL0MsT0FBTyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTztTQUNwQyxDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7SUE5R00sTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGVBQXVCO1FBQ3JGLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQyx3QkFBd0IsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO1FBRWpHLE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxlQUFlLEVBQUUsQ0FBQyxDQUFDO0lBQ3BFLENBQUM7Ozs7OztJQUdNLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxjQUFzQjtRQUNuRixNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLENBQUMsWUFBWSxDQUFDO1FBQ25GLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQywwQ0FBMEMsY0FBYyw4QkFBOEIsQ0FBQyxDQUFDO1NBQ3pHO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsY0FBYyxFQUFFLGVBQWUsRUFBRSxDQUFDLENBQUM7SUFDcEUsQ0FBQzs7QUFoQkgsa0NBaUhDOzs7QUFFRCxTQUFTLHdCQUF3QixDQUFDLFlBQW9CO0lBQ3BELE9BQU87UUFDTCxPQUFPLEVBQUUsa0JBQWtCO1FBQzNCLFFBQVEsRUFBRSxhQUFhO1FBQ3ZCLFlBQVk7S0FDYixDQUFDO0FBQ0osQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IENmbkFwcGxpY2F0aW9uQ2xvdWRXYXRjaExvZ2dpbmdPcHRpb25WMiwgQ2ZuQXBwbGljYXRpb25WMiB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1raW5lc2lzYW5hbHl0aWNzJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgY29yZSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQXBwbGljYXRpb25Db2RlIH0gZnJvbSAnLi9hcHBsaWNhdGlvbi1jb2RlJztcbmltcG9ydCB7IGVudmlyb25tZW50UHJvcGVydGllcyB9IGZyb20gJy4vcHJpdmF0ZS9lbnZpcm9ubWVudC1wcm9wZXJ0aWVzJztcbmltcG9ydCB7IGZsaW5rQXBwbGljYXRpb25Db25maWd1cmF0aW9uIH0gZnJvbSAnLi9wcml2YXRlL2ZsaW5rLWFwcGxpY2F0aW9uLWNvbmZpZ3VyYXRpb24nO1xuaW1wb3J0IHsgdmFsaWRhdGVGbGlua0FwcGxpY2F0aW9uUHJvcHMgYXMgdmFsaWRhdGVBcHBsaWNhdGlvblByb3BzIH0gZnJvbSAnLi9wcml2YXRlL3ZhbGlkYXRpb24nO1xuaW1wb3J0IHsgTG9nTGV2ZWwsIE1ldHJpY3NMZXZlbCwgUHJvcGVydHlHcm91cHMsIFJ1bnRpbWUgfSBmcm9tICcuL3R5cGVzJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElBcHBsaWNhdGlvbiBleHRlbmRzIGNvcmUuSVJlc291cmNlLCBpYW0uSUdyYW50YWJsZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uQXJuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhcHBsaWNhdGlvbk5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBhZGRUb1JvbGVQb2xpY3kocG9saWN5U3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBJbXBsZW1lbnRzIHRoZSBmdW5jdGlvbmFsaXR5IHNoYXJlZCBiZXR3ZWVuIENESyBjcmVhdGVkIGFuZCBpbXBvcnRlZFxuICogSUFwcGxpY2F0aW9ucy5cbiAqL1xuYWJzdHJhY3QgY2xhc3MgQXBwbGljYXRpb25CYXNlIGV4dGVuZHMgY29yZS5SZXNvdXJjZSBpbXBsZW1lbnRzIElBcHBsaWNhdGlvbiB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhcHBsaWNhdGlvbkFybjogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYXBwbGljYXRpb25OYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gIC8vIEltcGxlbWVudCBpYW0uSUdyYW50YWJsZSBpbnRlcmZhY2VcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRUb1JvbGVQb2xpY3kocG9saWN5U3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogYm9vbGVhbiB7XG4gICAgaWYgKHRoaXMucm9sZSkge1xuICAgICAgdGhpcy5yb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KHBvbGljeVN0YXRlbWVudCk7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9XG5cbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvblByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhcHBsaWNhdGlvbk5hbWU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBydW50aW1lOiBSdW50aW1lO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29kZTogQXBwbGljYXRpb25Db2RlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2hlY2twb2ludGluZ0VuYWJsZWQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2hlY2twb2ludEludGVydmFsPzogY29yZS5EdXJhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWluUGF1c2VCZXR3ZWVuQ2hlY2twb2ludHM/OiBjb3JlLkR1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ0xldmVsPzogTG9nTGV2ZWw7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtZXRyaWNzTGV2ZWw/OiBNZXRyaWNzTGV2ZWw7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGF1dG9TY2FsaW5nRW5hYmxlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwYXJhbGxlbGlzbT86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBhcmFsbGVsaXNtUGVyS3B1PzogbnVtYmVyXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNuYXBzaG90c0VuYWJsZWQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJvcGVydHlHcm91cHM/OiBQcm9wZXJ0eUdyb3VwcztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IGNvcmUuUmVtb3ZhbFBvbGljeTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ0dyb3VwPzogbG9ncy5JTG9nR3JvdXA7XG59XG5cbi8qKlxuICogQW4gaW1wb3J0ZWQgRmxpbmsgYXBwbGljYXRpb24uXG4gKi9cbmNsYXNzIEltcG9ydCBleHRlbmRzIEFwcGxpY2F0aW9uQmFzZSB7XG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gIHB1YmxpYyByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICBwdWJsaWMgcmVhZG9ubHkgYXBwbGljYXRpb25OYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbkFybjogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiB7IGFwcGxpY2F0aW9uQXJuOiBzdHJpbmcsIGFwcGxpY2F0aW9uTmFtZTogc3RyaW5nIH0pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgLy8gSW1wb3J0ZWQgYXBwbGljYXRpb25zIGhhdmUgbm8gYXNzb2NpYXRlZCByb2xlIG9yIGdyYW50UHJpbmNpcGFsXG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IG5ldyBpYW0uVW5rbm93blByaW5jaXBhbCh7IHJlc291cmNlOiB0aGlzIH0pO1xuICAgIHRoaXMucm9sZSA9IHVuZGVmaW5lZDtcblxuICAgIHRoaXMuYXBwbGljYXRpb25Bcm4gPSBhdHRycy5hcHBsaWNhdGlvbkFybjtcbiAgICB0aGlzLmFwcGxpY2F0aW9uTmFtZSA9IGF0dHJzLmFwcGxpY2F0aW9uTmFtZTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBcHBsaWNhdGlvbiBleHRlbmRzIEFwcGxpY2F0aW9uQmFzZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFwcGxpY2F0aW9uTmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhcHBsaWNhdGlvbk5hbWU6IHN0cmluZyk6IElBcHBsaWNhdGlvbiB7XG4gICAgY29uc3QgYXBwbGljYXRpb25Bcm4gPSBjb3JlLlN0YWNrLm9mKHNjb3BlKS5mb3JtYXRBcm4oYXBwbGljYXRpb25Bcm5Db21wb25lbnRzKGFwcGxpY2F0aW9uTmFtZSkpO1xuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkLCB7IGFwcGxpY2F0aW9uQXJuLCBhcHBsaWNhdGlvbk5hbWUgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFwcGxpY2F0aW9uQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGFwcGxpY2F0aW9uQXJuOiBzdHJpbmcpOiBJQXBwbGljYXRpb24ge1xuICAgIGNvbnN0IGFwcGxpY2F0aW9uTmFtZSA9IGNvcmUuU3RhY2sub2Yoc2NvcGUpLnBhcnNlQXJuKGFwcGxpY2F0aW9uQXJuKS5yZXNvdXJjZU5hbWU7XG4gICAgaWYgKCFhcHBsaWNhdGlvbk5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgYXBwbGljYXRpb25Bcm4gZm9yIGZyb21BcHBsaWNhdGlvbkFybiAoJHthcHBsaWNhdGlvbkFybn0pIG11c3QgaW5jbHVkZSByZXNvdXJjZSBuYW1lYCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkLCB7IGFwcGxpY2F0aW9uQXJuLCBhcHBsaWNhdGlvbk5hbWUgfSk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgYXBwbGljYXRpb25Bcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGFwcGxpY2F0aW9uTmFtZTogc3RyaW5nO1xuXG4gIC8vIFJvbGUgbXVzdCBiZSBvcHRpb25hbCBmb3IgSlNJSSBjb21wYXRpYmlsaXR5XG4gIHB1YmxpYyByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFwcGxpY2F0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHsgcGh5c2ljYWxOYW1lOiBwcm9wcy5hcHBsaWNhdGlvbk5hbWUgfSk7XG4gICAgdmFsaWRhdGVBcHBsaWNhdGlvblByb3BzKHByb3BzKTtcblxuICAgIHRoaXMucm9sZSA9IHByb3BzLnJvbGUgPz8gbmV3IGlhbS5Sb2xlKHRoaXMsICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2tpbmVzaXNhbmFseXRpY3MuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLnJvbGU7XG5cbiAgICAvLyBQZXJtaXQgbWV0cmljIHB1Ymxpc2hpbmcgdG8gQ2xvdWRXYXRjaFxuICAgIHRoaXMucm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ2Nsb3Vkd2F0Y2g6UHV0TWV0cmljRGF0YSddLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCBjb2RlID0gcHJvcHMuY29kZS5iaW5kKHRoaXMpO1xuICAgIGNvZGUuYnVja2V0LmdyYW50UmVhZCh0aGlzKTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkFwcGxpY2F0aW9uVjIodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgYXBwbGljYXRpb25OYW1lOiBwcm9wcy5hcHBsaWNhdGlvbk5hbWUsXG4gICAgICBydW50aW1lRW52aXJvbm1lbnQ6IHByb3BzLnJ1bnRpbWUudmFsdWUsXG4gICAgICBzZXJ2aWNlRXhlY3V0aW9uUm9sZTogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICBhcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgLi4uY29kZS5hcHBsaWNhdGlvbkNvZGVDb25maWd1cmF0aW9uUHJvcGVydHksXG4gICAgICAgIGVudmlyb25tZW50UHJvcGVydGllczogZW52aXJvbm1lbnRQcm9wZXJ0aWVzKHByb3BzLnByb3BlcnR5R3JvdXBzKSxcbiAgICAgICAgZmxpbmtBcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb246IGZsaW5rQXBwbGljYXRpb25Db25maWd1cmF0aW9uKHtcbiAgICAgICAgICBjaGVja3BvaW50aW5nRW5hYmxlZDogcHJvcHMuY2hlY2twb2ludGluZ0VuYWJsZWQsXG4gICAgICAgICAgY2hlY2twb2ludEludGVydmFsOiBwcm9wcy5jaGVja3BvaW50SW50ZXJ2YWwsXG4gICAgICAgICAgbWluUGF1c2VCZXR3ZWVuQ2hlY2twb2ludHM6IHByb3BzLm1pblBhdXNlQmV0d2VlbkNoZWNrcG9pbnRzLFxuICAgICAgICAgIGxvZ0xldmVsOiBwcm9wcy5sb2dMZXZlbCxcbiAgICAgICAgICBtZXRyaWNzTGV2ZWw6IHByb3BzLm1ldHJpY3NMZXZlbCxcbiAgICAgICAgICBhdXRvU2NhbGluZ0VuYWJsZWQ6IHByb3BzLmF1dG9TY2FsaW5nRW5hYmxlZCxcbiAgICAgICAgICBwYXJhbGxlbGlzbTogcHJvcHMucGFyYWxsZWxpc20sXG4gICAgICAgICAgcGFyYWxsZWxpc21QZXJLcHU6IHByb3BzLnBhcmFsbGVsaXNtUGVyS3B1LFxuICAgICAgICB9KSxcbiAgICAgICAgYXBwbGljYXRpb25TbmFwc2hvdENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBzbmFwc2hvdHNFbmFibGVkOiBwcm9wcy5zbmFwc2hvdHNFbmFibGVkID8/IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLnJvbGUpO1xuXG4gICAgY29uc3QgbG9nR3JvdXAgPSBwcm9wcy5sb2dHcm91cCA/PyBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnTG9nR3JvdXAnKTtcbiAgICBjb25zdCBsb2dTdHJlYW0gPSBuZXcgbG9ncy5Mb2dTdHJlYW0odGhpcywgJ0xvZ1N0cmVhbScsIHsgbG9nR3JvdXAgfSk7XG5cbiAgICAvKiBQZXJtaXQgbG9nZ2luZyAqL1xuXG4gICAgdGhpcy5yb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnbG9nczpEZXNjcmliZUxvZ0dyb3VwcyddLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIGNvcmUuU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgICAgICBzZXJ2aWNlOiAnbG9ncycsXG4gICAgICAgICAgcmVzb3VyY2U6ICdsb2ctZ3JvdXAnLFxuICAgICAgICAgIHNlcDogJzonLFxuICAgICAgICAgIHJlc291cmNlTmFtZTogJyonLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGhpcy5yb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnbG9nczpEZXNjcmliZUxvZ1N0cmVhbXMnXSxcbiAgICAgIHJlc291cmNlczogW2xvZ0dyb3VwLmxvZ0dyb3VwQXJuXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCBsb2dTdHJlYW1Bcm4gPSBgYXJuOiR7Y29yZS5Bd3MuUEFSVElUSU9OfTpsb2dzOiR7Y29yZS5Bd3MuUkVHSU9OfToke2NvcmUuQXdzLkFDQ09VTlRfSUR9OmxvZy1ncm91cDoke2xvZ0dyb3VwLmxvZ0dyb3VwTmFtZX06bG9nLXN0cmVhbToke2xvZ1N0cmVhbS5sb2dTdHJlYW1OYW1lfWA7XG4gICAgdGhpcy5yb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnbG9nczpQdXRMb2dFdmVudHMnXSxcbiAgICAgIHJlc291cmNlczogW2xvZ1N0cmVhbUFybl0sXG4gICAgfSkpO1xuXG4gICAgbmV3IENmbkFwcGxpY2F0aW9uQ2xvdWRXYXRjaExvZ2dpbmdPcHRpb25WMih0aGlzLCAnTG9nZ2luZ09wdGlvbicsIHtcbiAgICAgIGFwcGxpY2F0aW9uTmFtZTogcmVzb3VyY2UucmVmLFxuICAgICAgY2xvdWRXYXRjaExvZ2dpbmdPcHRpb246IHtcbiAgICAgICAgbG9nU3RyZWFtQXJuLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMuYXBwbGljYXRpb25OYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUocmVzb3VyY2UucmVmKTtcbiAgICB0aGlzLmFwcGxpY2F0aW9uQXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShcbiAgICAgIGNvcmUuU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKGFwcGxpY2F0aW9uQXJuQ29tcG9uZW50cyhyZXNvdXJjZS5yZWYpKSxcbiAgICAgIGFwcGxpY2F0aW9uQXJuQ29tcG9uZW50cyh0aGlzLnBoeXNpY2FsTmFtZSksXG4gICAgKTtcblxuICAgIHJlc291cmNlLmFwcGx5UmVtb3ZhbFBvbGljeShwcm9wcy5yZW1vdmFsUG9saWN5LCB7XG4gICAgICBkZWZhdWx0OiBjb3JlLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICB9KTtcbiAgfVxufVxuXG5mdW5jdGlvbiBhcHBsaWNhdGlvbkFybkNvbXBvbmVudHMocmVzb3VyY2VOYW1lOiBzdHJpbmcpOiBjb3JlLkFybkNvbXBvbmVudHMge1xuICByZXR1cm4ge1xuICAgIHNlcnZpY2U6ICdraW5lc2lzYW5hbHl0aWNzJyxcbiAgICByZXNvdXJjZTogJ2FwcGxpY2F0aW9uJyxcbiAgICByZXNvdXJjZU5hbWUsXG4gIH07XG59XG4iXX0=