"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrCreateCluster = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const cluster_utils_1 = require("./private/cluster-utils");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 */
class EmrCreateCluster extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EmrCreateClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.visibleToAllUsers = (_b = this.props.visibleToAllUsers) !== null && _b !== void 0 ? _b : true;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this._autoScalingRole = this.props.autoScalingRole;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = (_c = this.props.serviceRole) !== null && _c !== void 0 ? _c : this.createServiceRole();
        this._clusterRole = (_d = this.props.clusterRole) !== null && _d !== void 0 ? _d : this.createClusterRole();
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole();
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        this.taskPolicies = this.createPolicyStatements(this._serviceRole, this._clusterRole, this._autoScalingRole);
        if (this.props.releaseLabel !== undefined && !cdk.Token.isUnresolved(this.props.releaseLabel)) {
            this.validateReleaseLabel(this.props.releaseLabel);
        }
        if (this.props.stepConcurrencyLevel !== undefined && !cdk.Token.isUnresolved(this.props.stepConcurrencyLevel)) {
            if (this.props.stepConcurrencyLevel < 1 || this.props.stepConcurrencyLevel > 256) {
                throw new Error(`Step concurrency level must be in range [1, 256], but got ${this.props.stepConcurrencyLevel}.`);
            }
            if (this.props.releaseLabel && this.props.stepConcurrencyLevel !== 1) {
                const [major, minor] = this.props.releaseLabel.slice(4).split('.');
                if (Number(major) < 5 || (Number(major) === 5 && Number(minor) < 28)) {
                    throw new Error(`Step concurrency is only supported in EMR release version 5.28.0 and above but got ${this.props.releaseLabel}.`);
                }
            }
        }
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    /**
     * @internal
     */
    _renderTask() {
        var _b, _c, _d;
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Instances: cluster_utils_1.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(cluster_utils_1.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_b = this._autoScalingRole) === null || _b === void 0 ? void 0 : _b.roleName),
                BootstrapActions: cdk.listMapper(cluster_utils_1.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(cluster_utils_1.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: (_c = this.props.ebsRootVolumeSize) === null || _c === void 0 ? void 0 : _c.toGibibytes(),
                KerberosAttributes: this.props.kerberosAttributes ? cluster_utils_1.KerberosAttributesPropertyToJson(this.props.kerberosAttributes) : undefined,
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_d = this.props.scaleDownBehavior) === null || _d === void 0 ? void 0 : _d.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                StepConcurrencyLevel: cdk.numberToCloudFormation(this.props.stepConcurrencyLevel),
                ...(this.props.tags ? this.renderTags(this.props.tags) : undefined),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            }),
        };
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['elasticmapreduce:RunJobFlow', 'elasticmapreduce:DescribeCluster', 'elasticmapreduce:TerminateJobFlows'],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [serviceRole.roleArn, clusterRole.roleArn],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole() {
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole() {
        const role = new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole() {
        var _b;
        const role = new iam.Role(this, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')],
        });
        (_b = role.assumeRolePolicy) === null || _b === void 0 ? void 0 : _b.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('application-autoscaling.amazonaws.com')],
            actions: ['sts:AssumeRole'],
        }));
        return role;
    }
    /**
     * Validates the release label string is in proper format.
     * Release labels are in the form `emr-x.x.x`. For example, `emr-5.33.0`.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-release-components.html
     */
    validateReleaseLabel(releaseLabel) {
        const prefix = releaseLabel.slice(0, 4);
        const versions = releaseLabel.slice(4).split('.');
        if (prefix !== 'emr-' || versions.length !== 3 || versions.some((e) => isNotANumber(e))) {
            throw new Error(`The release label must be in the format 'emr-x.x.x' but got ${releaseLabel}`);
        }
        return releaseLabel;
        function isNotANumber(value) {
            return value === '' || isNaN(Number(value));
        }
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
_a = JSII_RTTI_SYMBOL_1;
EmrCreateCluster[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EmrCreateCluster", version: "1.156.0" };
EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
(function (EmrCreateCluster) {
    /**
     * The Cluster ScaleDownBehavior specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity
     * occurs or an instance group is resized.
     *
     * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html#EMR-RunJobFlow-request-ScaleDownBehavior
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR adds nodes to a deny list and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Spot Timeout Actions
     *
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * Spot Allocation Strategies
     *
     * Specifies the strategy to use in launching Spot Instance fleets. For example, "capacity-optimized" launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.
     *
     * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_SpotProvisioningSpecification.html
     *
     */
    let SpotAllocationStrategy;
    (function (SpotAllocationStrategy) {
        /**
         * Capacity-optimized, which launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.
         */
        SpotAllocationStrategy["CAPACITY_OPTIMIZED"] = "capacity-optimized";
    })(SpotAllocationStrategy = EmrCreateCluster.SpotAllocationStrategy || (EmrCreateCluster.SpotAllocationStrategy = {}));
    /**
     * CloudWatch Alarm Comparison Operators
     *
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * EC2 Instance Market
     *
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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