"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTask = exports.EcsEc2LaunchTarget = exports.EcsFargateLaunchTarget = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Configuration for running an ECS task on Fargate
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-fargate
 */
class EcsFargateLaunchTarget {
    constructor(options) {
        this.options = options;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsFargateLaunchTargetOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
    }
    /**
     * Called when the Fargate launch type configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        var _d;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTask(_task);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_LaunchTargetBindOptions(launchTargetOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                LaunchType: 'FARGATE',
                PlatformVersion: (_d = this.options) === null || _d === void 0 ? void 0 : _d.platformVersion,
            },
        };
    }
}
exports.EcsFargateLaunchTarget = EcsFargateLaunchTarget;
_a = JSII_RTTI_SYMBOL_1;
EcsFargateLaunchTarget[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsFargateLaunchTarget", version: "1.156.0" };
/**
 * Configuration for running an ECS task on EC2
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-ec2
 */
class EcsEc2LaunchTarget {
    constructor(options) {
        this.options = options;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsEc2LaunchTargetOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
    }
    /**
     * Called when the EC2 launch type is configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        var _d, _e, _f, _g, _h;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTask(_task);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_LaunchTargetBindOptions(launchTargetOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (!launchTargetOptions.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not compatible with EC2');
        }
        if (!((_d = launchTargetOptions.cluster) === null || _d === void 0 ? void 0 : _d.hasEc2Capacity)) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addCapacity() on the cluster.');
        }
        return {
            parameters: {
                LaunchType: 'EC2',
                // takes an array of placement constraints each of which contain a single item array of constraints, flattens it
                // and renders the Json to be passed as a parameter in the state machine.
                // input: [ecs.PlacementConstraint.distinctInstances()] - distinctInstances() returns [{ type: 'distinctInstance' }]
                // output: {Type: 'distinctInstance'}
                PlacementConstraints: noEmpty(flatten(((_f = (_e = this.options) === null || _e === void 0 ? void 0 : _e.placementConstraints) !== null && _f !== void 0 ? _f : []).map((c) => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten(((_h = (_g = this.options) === null || _g === void 0 ? void 0 : _g.placementStrategies) !== null && _h !== void 0 ? _h : []).map((c) => c.toJson().map(uppercaseKeys)))),
            },
        };
        function uppercaseKeys(obj) {
            const ret = {};
            for (const key of Object.keys(obj)) {
                ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
            }
            return ret;
        }
        function flatten(xs) {
            return Array.prototype.concat([], ...xs);
        }
        function noEmpty(xs) {
            if (xs.length === 0) {
                return undefined;
            }
            return xs;
        }
    }
}
exports.EcsEc2LaunchTarget = EcsEc2LaunchTarget;
_b = JSII_RTTI_SYMBOL_1;
EcsEc2LaunchTarget[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsEc2LaunchTarget", version: "1.156.0" };
/**
 * Run a Task on ECS or Fargate
 */
class EcsRunTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _d, _e, _f;
        super(scope, id, props);
        this.props = props;
        /**
         * Manage allowed network traffic for this service
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTaskProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.integrationPattern = (_d = props.integrationPattern) !== null && _d !== void 0 ? _d : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken((_e = props.containerOverrides) === null || _e === void 0 ? void 0 : _e.map(override => override.environment))) {
            throw new Error('Task Token is required in at least one `containerOverrides.environment` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (!this.props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        if (this.props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking();
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            this.validateNoNetworkingProps();
            this.connections.addSecurityGroup(...this.props.cluster.connections.securityGroups);
        }
        for (const override of (_f = this.props.containerOverrides) !== null && _f !== void 0 ? _f : []) {
            const name = override.containerDefinition.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.findContainer(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('ecs', 'runTask', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.family,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.launchTarget.bind(this, { taskDefinition: this.props.taskDefinition, cluster: this.props.cluster }).parameters,
            }),
        };
    }
    configureAwsVpcNetworking() {
        var _d, _e;
        const subnetSelection = (_d = this.props.subnets) !== null && _d !== void 0 ? _d : { subnetType: this.props.assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: this.props.assignPublicIp ? (this.props.assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.list({ produce: () => { var _d; return (_d = this.securityGroups) === null || _d === void 0 ? void 0 : _d.map(sg => sg.securityGroupId); } }),
            },
        };
        // Make sure we have a security group if we're using AWSVPC networking
        this.securityGroups = (_e = this.props.securityGroups) !== null && _e !== void 0 ? _e : [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.props.cluster.vpc })];
        this.connections.addSecurityGroup(...this.securityGroups);
    }
    validateNoNetworkingProps() {
        if (this.props.subnets !== undefined || this.props.securityGroups !== undefined) {
            throw new Error(`Supplied TaskDefinition must have 'networkMode' of 'AWS_VPC' to use 'vpcSubnets' and 'securityGroup'. Received: ${this.props.taskDefinition.networkMode}`);
        }
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.getTaskDefinitionFamilyArn()],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: this.taskExecutionRoles().map((r) => r.roleArn),
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Returns the ARN of the task definition family by removing the
     * revision from the task definition ARN
     * Before - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world:8
     * After - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world
     */
    getTaskDefinitionFamilyArn() {
        const arnComponents = cdk.Stack.of(this).splitArn(this.props.taskDefinition.taskDefinitionArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        let { resourceName } = arnComponents;
        if (resourceName) {
            resourceName = resourceName.split(':')[0];
        }
        return cdk.Stack.of(this).formatArn({
            partition: arnComponents.partition,
            service: arnComponents.service,
            account: arnComponents.account,
            region: arnComponents.region,
            resource: arnComponents.resource,
            arnFormat: arnComponents.arnFormat,
            resourceName,
        });
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTask = EcsRunTask;
_c = JSII_RTTI_SYMBOL_1;
EcsRunTask[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsRunTask", version: "1.156.0" };
EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderOverrides(containerOverrides) {
    var _d;
    if (!containerOverrides || containerOverrides.length === 0) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerDefinition.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: (_d = override.environment) === null || _d === void 0 ? void 0 : _d.map((e) => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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