"use strict";
const cluster_1 = require("../lib/cluster-resource-handler/cluster");
const mocks = require("./cluster-resource-handler-mocks");
module.exports = {
    setUp(callback) {
        mocks.reset();
        callback();
    },
    create: {
        async 'onCreate: minimal defaults (vpc + role)'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', mocks.MOCK_PROPS));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.configureAssumeRoleRequest, {
                RoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                RoleSessionName: 'AWSCDK.EKSCluster.Create.fake-request-id',
            });
            test.deepEqual(mocks.actualRequest.createClusterRequest, {
                roleArn: 'arn:of:role',
                resourcesVpcConfig: {
                    subnetIds: ['subnet1', 'subnet2'],
                    securityGroupIds: ['sg1', 'sg2', 'sg3'],
                },
                name: 'MyResourceId-fakerequestid',
            });
            test.done();
        },
        async 'generated cluster name does not exceed 100 characters'(test) {
            var _a, _b;
            // GIVEN
            const req = {
                StackId: 'fake-stack-id',
                RequestId: '602c078a-6181-4352-9676-4f00352445aa',
                ResourceType: 'Custom::EKSCluster',
                ServiceToken: 'boom',
                LogicalResourceId: 'hello'.repeat(30),
                ResponseURL: 'http://response-url',
                RequestType: 'Create',
                ResourceProperties: {
                    ServiceToken: 'boom',
                    Config: mocks.MOCK_PROPS,
                    AssumeRoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                },
            };
            // WHEN
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
            await handler.onEvent();
            // THEN
            test.equal((_a = mocks.actualRequest.createClusterRequest) === null || _a === void 0 ? void 0 : _a.name.length, 100);
            test.deepEqual((_b = mocks.actualRequest.createClusterRequest) === null || _b === void 0 ? void 0 : _b.name, 'hellohellohellohellohellohellohellohellohellohellohellohellohellohe-602c078a6181435296764f00352445aa');
            test.done();
        },
        async 'onCreate: explicit cluster name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                name: 'ExplicitCustomName',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.name, 'ExplicitCustomName');
            test.done();
        },
        async 'with no specific version'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                version: '12.34.56',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.version, '12.34.56');
            test.done();
        },
        async 'isCreateComplete still not complete if cluster is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'CREATING';
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { IsComplete: false });
            test.done();
        },
        async 'isCreateComplete is complete when cluster is ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp, {
                IsComplete: true,
                Data: {
                    Name: 'physical-resource-id',
                    Endpoint: 'http://endpoint',
                    Arn: 'arn:cluster-arn',
                    CertificateAuthorityData: 'certificateAuthority-data',
                },
            });
            test.done();
        },
    },
    delete: {
        async 'returns correct physical name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.onEvent();
            test.deepEqual(mocks.actualRequest.deleteClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { PhysicalResourceId: 'physical-resource-id' });
            test.done();
        },
        async 'onDelete ignores ResourceNotFoundException'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.deleteClusterErrorCode = 'ResourceNotFoundException';
            await handler.onEvent();
            test.done();
        },
        async 'isDeleteComplete returns false as long as describeCluster succeeds'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.ok(!resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete returns true when describeCluster throws a ResourceNotFound exception'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'ResourceNotFoundException';
            const resp = await handler.isComplete();
            test.ok(resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete propagates other errors'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'OtherException';
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            test.equal(error.code, 'OtherException');
            test.done();
        },
    },
    update: {
        async 'no change'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', mocks.MOCK_PROPS, mocks.MOCK_PROPS));
            const resp = await handler.onEvent();
            test.equal(resp, undefined);
            test.equal(mocks.actualRequest.createClusterRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterConfigRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
            test.done();
        },
        async 'isUpdateComplete is not complete when status is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'UPDATING';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, false);
            test.done();
        },
        async 'isUpdateComplete waits for ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, true);
            test.done();
        },
        'requires replacement': {
            'name change': {
                async 'explicit name change'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: 'new-cluster-name-1234',
                    }, {
                        name: 'old-cluster-name',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'new-cluster-name-1234',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'new-cluster-name-1234' });
                    test.done();
                },
                async 'from auto-gen name to explicit name'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: undefined,
                    }, {
                        name: 'explicit',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'MyResourceId-fakerequestid',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                    test.done();
                },
            },
            async 'subnets or security groups requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                }, {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1'],
                        securityGroupIds: ['sg2'],
                    },
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'arn:of:role',
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                });
                test.done();
            },
            async '"roleArn" requires a replcement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                }, {
                    roleArn: 'old-arn',
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'fails if cluster has an explicit "name" that is the same as the old "name"'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // THEN
                let err;
                try {
                    await handler.onEvent();
                }
                catch (e) {
                    err = e;
                }
                test.equal(err === null || err === void 0 ? void 0 : err.message, 'Cannot replace cluster "explicit-cluster-name" since it has an explicit physical name. Either rename the cluster or remove the "name" configuration');
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it does not'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: undefined,
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it has a different name'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'new-explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'old-explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'new-explicit-cluster-name' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'new-explicit-cluster-name',
                    roleArn: 'new-arn',
                });
                test.done();
            },
        },
        'isUpdateComplete with EKS update ID': {
            async 'with "Failed" status'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Failed';
                mocks.simulateResponse.describeUpdateResponseMockErrors = [
                    {
                        errorMessage: 'errorMessageMock',
                        errorCode: 'errorCodeMock',
                        resourceIds: [
                            'foo', 'bar',
                        ],
                    },
                ];
                let error;
                try {
                    await isCompleteHandler.isComplete();
                }
                catch (e) {
                    error = e;
                }
                test.ok(error);
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(error.message, 'cluster update id "foobar" failed with errors: [{"errorMessage":"errorMessageMock","errorCode":"errorCodeMock","resourceIds":["foo","bar"]}]');
                test.done();
            },
            async 'with "InProgress" status, returns IsComplete=false'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'InProgress';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(response.IsComplete, false);
                test.done();
            },
            async 'with "Successful" status, returns IsComplete=true'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Successful';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(response.IsComplete, true);
                test.done();
            },
        },
        'in-place': {
            'version change': {
                async 'from undefined to a specific value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '12.34',
                    }, {
                        version: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '12.34',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from a specific value to another value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '2.0',
                    }, {
                        version: '1.1',
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '2.0',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'to a new value that is already the current version'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '1.0',
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.describeClusterRequest, { name: 'physical-resource-id' });
                    test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'fails from specific value to undefined'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: undefined,
                    }, {
                        version: '1.2',
                    }));
                    let error;
                    try {
                        await handler.onEvent();
                    }
                    catch (e) {
                        error = e;
                    }
                    test.equal(error.message, 'Cannot remove cluster version configuration. Current version is 1.2');
                    test.done();
                },
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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