"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const efs = require("@aws-cdk/aws-efs");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const case_1 = require("case");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from other S3 buckets or from local disk.
 *
 * @stability stable
 */
class BucketDeployment extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id);
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        if (props.useEfs && !props.vpc) {
            throw new Error('Vpc must be specified if useEfs is set');
        }
        const accessPointPath = '/lambda';
        let accessPoint;
        if (props.useEfs && props.vpc) {
            const accessMode = '0777';
            const fileSystem = this.getOrCreateEfsFileSystem(scope, {
                vpc: props.vpc,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            accessPoint = fileSystem.addAccessPoint('AccessPoint', {
                path: accessPointPath,
                createAcl: {
                    ownerUid: '1001',
                    ownerGid: '1001',
                    permissions: accessMode,
                },
                posixUser: {
                    uid: '1001',
                    gid: '1001',
                },
            });
            accessPoint.node.addDependency(fileSystem.mountTargetsAvailable);
        }
        // Making VPC dependent on BucketDeployment so that CFN stack deletion is smooth.
        // Refer comments on https://github.com/aws/aws-cdk/pull/15220 for more details.
        if (props.vpc) {
            this.node.addDependency(props.vpc);
        }
        const mountPath = `/mnt${accessPointPath}`;
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit, props.vpc),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_7,
            environment: props.useEfs ? {
                MOUNT_PATH: mountPath,
            } : undefined,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            filesystem: accessPoint ? lambda.FileSystem.fromEfsAccessPoint(accessPoint, mountPath) : undefined,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_d = props.prune) !== null && _d !== void 0 ? _d : true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: (_e = props.distribution) === null || _e === void 0 ? void 0 : _e.distributionId,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit, vpc) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        // if user specify to use VPC, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        // A VPC is a must if EFS storage is used and that's why we are only using VPC in uuid.
        if (vpc) {
            uuid += `-${vpc.node.addr}`;
        }
        return uuid;
    }
    /**
     * Function to get/create a stack singleton instance of EFS FileSystem per vpc.
     *
     * @param scope Construct
     * @param fileSystemProps EFS FileSystemProps
     */
    getOrCreateEfsFileSystem(scope, fileSystemProps) {
        var _d;
        const stack = cdk.Stack.of(scope);
        const uuid = `BucketDeploymentEFS-VPC-${fileSystemProps.vpc.node.addr}`;
        return (_d = stack.node.tryFindChild(uuid)) !== null && _d !== void 0 ? _d : new efs.FileSystem(scope, uuid, fileSystemProps);
    }
}
exports.BucketDeployment = BucketDeployment;
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "@aws-cdk/aws-s3-deployment.BucketDeployment", version: "1.130.0" };
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = case_1.kebab(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
class CacheControl {
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
    /**
     * Sets 'must-revalidate'.
     *
     * @stability stable
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     *
     * @stability stable
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     *
     * @stability stable
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     *
     * @stability stable
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     *
     * @stability stable
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     *
     * @stability stable
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     *
     * @stability stable
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     *
     * @stability stable
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     *
     * @stability stable
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "@aws-cdk/aws-s3-deployment.CacheControl", version: "1.130.0" };
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * (deprecated) Used for HTTP expires header, which influences downstream caches.
 *
 * Does NOT influence deletion of the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
    /**
     * (deprecated) Expire at the specified date.
     *
     * @param d date to expire at.
     * @deprecated
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * (deprecated) Expire at the specified timestamp.
     *
     * @param t timestamp in unix milliseconds.
     * @deprecated
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * (deprecated) Expire once the specified duration has passed since deployment time.
     *
     * @param t the duration to wait before expiring.
     * @deprecated
     */
    static after(t) { return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * (deprecated) Create an expiration date from a raw date string.
     *
     * @deprecated
     */
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "@aws-cdk/aws-s3-deployment.Expires", version: "1.130.0" };
//# sourceMappingURL=data:application/json;base64,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