# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['eventsourcing_eventstoredb']

package_data = \
{'': ['*']}

install_requires = \
['esdbclient==0.10', 'eventsourcing>=9.2.14,<9.3.0']

setup_kwargs = {
    'name': 'eventsourcing-eventstoredb',
    'version': '0.3.4',
    'description': 'Python package for eventsourcing with EventStoreDB',
    'long_description': '# Event Sourcing in Python with EventStoreDB\n\nThis is an extension package for the Python\n[eventsourcing](https://github.com/pyeventsourcing/eventsourcing) library\nthat provides a persistence module for [EventStoreDB](https://www.eventstore.com/).\nIt uses the [esdbclient](https://github.com/pyeventsourcing/esdbclient)\npackage to communicate with EventStoreDB via its gRPC interface.\n\n## Installation\n\nUse pip to install the [stable distribution](https://pypi.org/project/eventsourcing-eventstoredb/)\nfrom the Python Package Index.\n\n    $ pip install eventsourcing-eventstoredb\n\nPlease note, it is recommended to install Python packages into a Python virtual environment.\n\n## Getting started\n\nDefine aggregates and applications in the usual way. Please note, aggregate\nsequences  in EventStoreDB start from position `0`, so set INITIAL_VERSION\non your aggregate classes accordingly.\n\n```python\nfrom eventsourcing.application import Application\nfrom eventsourcing.domain import Aggregate, event\n\n\nclass TrainingSchool(Application):\n    def register(self, name):\n        dog = Dog(name)\n        self.save(dog)\n        return dog.id\n\n    def add_trick(self, dog_id, trick):\n        dog = self.repository.get(dog_id)\n        dog.add_trick(trick)\n        self.save(dog)\n\n    def get_dog(self, dog_id):\n        dog = self.repository.get(dog_id)\n        return {\'name\': dog.name, \'tricks\': list(dog.tricks)}\n\n\nclass Dog(Aggregate):\n    INITIAL_VERSION = 0\n\n    @event(\'Registered\')\n    def __init__(self, name):\n        self.name = name\n        self.tricks = []\n\n    @event(\'TrickAdded\')\n    def add_trick(self, trick):\n        self.tricks.append(trick)\n```\n\nConfigure the application to use EventStoreDB. Set environment variable\n`PERSISTENCE_MODULE` to `\'eventsourcing_eventstoredb\'`, and set\n`EVENTSTOREDB_URI` to the host and port of your EventStoreDB.\n\n```python\nschool = TrainingSchool(env={\n    \'PERSISTENCE_MODULE\': \'eventsourcing_eventstoredb\',\n    \'EVENTSTOREDB_URI\': \'localhost:2113\',\n})\n```\n\n*NB: SSL/TLS not yet supported:* In case you are running against a cluster, or want to use SSL/TLS certificates,\nyou can specify these things in the URI.\n\n```\n    \'EVENTSTOREDB_URI\': \'esdb://localhost:2111,localhost:2112,localhost:2113?tls&rootCertificate=./certs/ca/ca.crt\'\n```\n\nCall application methods from tests and user interfaces.\n\n```python\ndog_id = school.register(\'Fido\')\nschool.add_trick(dog_id, \'roll over\')\nschool.add_trick(dog_id, \'play dead\')\ndog_details = school.get_dog(dog_id)\nassert dog_details[\'name\'] == \'Fido\'\nassert dog_details[\'tricks\'] == [\'roll over\', \'play dead\']\n```\n\nTo see the events have been saved, we can reconstruct the application\nand get Fido\'s details again.\n\n```python\nschool = TrainingSchool(env={\n    \'PERSISTENCE_MODULE\': \'eventsourcing_eventstoredb\',\n    \'EVENTSTOREDB_URI\': \'localhost:2113\',\n})\n\ndog_details = school.get_dog(dog_id)\n\nassert dog_details[\'name\'] == \'Fido\'\nassert dog_details[\'tricks\'] == [\'roll over\', \'play dead\']\n```\n\nFor more information, please refer to the Python\n[eventsourcing](https://github.com/johnbywater/eventsourcing) library\nand the [EventStoreDB](https://www.eventstore.com/) project.\n\n## Developers\n\n### Install Poetry\n\nThe first thing is to check you have Poetry installed.\n\n    $ poetry --version\n\nIf you don\'t, then please [install Poetry](https://python-poetry.org/docs/#installing-with-the-official-installer).\n\nIt will help to make sure Poetry\'s bin directory is in your `PATH` environment variable.\n\nBut in any case, make sure you know the path to the `poetry` executable. The Poetry\ninstaller tells you where it has been installed, and how to configure your shell.\n\nPlease refer to the [Poetry docs](https://python-poetry.org/docs/) for guidance on\nusing Poetry.\n\n### Setup for PyCharm users\n\nYou can easily obtain the project files using PyCharm (menu "Git > Clone...").\nPyCharm will then usually prompt you to open the project.\n\nOpen the project in a new window. PyCharm will then usually prompt you to create\na new virtual environment.\n\nCreate a new Poetry virtual environment for the project. If PyCharm doesn\'t already\nknow where your `poetry` executable is, then set the path to your `poetry` executable\nin the "New Poetry Environment" form input field labelled "Poetry executable". In the\n"New Poetry Environment" form, you will also have the opportunity to select which\nPython executable will be used by the virtual environment.\n\nPyCharm will then create a new Poetry virtual environment for your project, using\na particular version of Python, and also install into this virtual environment the\nproject\'s package dependencies according to the `pyproject.toml` file, or the\n`poetry.lock` file if that exists in the project files.\n\nYou can add different Poetry environments for different Python versions, and switch\nbetween them using the "Python Interpreter" settings of PyCharm. If you want to use\na version of Python that isn\'t installed, either use your favourite package manager,\nor install Python by downloading an installer for recent versions of Python directly\nfrom the [Python website](https://www.python.org/downloads/).\n\nOnce project dependencies have been installed, you should be able to run tests\nfrom within PyCharm (right-click on the `tests` folder and select the \'Run\' option).\n\nBecause of a conflict between pytest and PyCharm\'s debugger and the coverage tool,\nyou may need to add ``--no-cov`` as an option to the test runner template. Alternatively,\njust use the Python Standard Library\'s ``unittest`` module.\n\nYou should also be able to open a terminal window in PyCharm, and run the project\'s\nMakefile commands from the command line (see below).\n\n### Setup from command line\n\nObtain the project files, using Git or suitable alternative.\n\nIn a terminal application, change your current working directory\nto the root folder of the project files. There should be a Makefile\nin this folder.\n\nUse the Makefile to create a new Poetry virtual environment for the\nproject and install the project\'s package dependencies into it,\nusing the following command.\n\n    $ make install-packages\n\nIf you want to skip the installation of your project\'s package, use the\n`--no-root` option.\n\n    $ make install-packages --no-root\n\nPlease note, if you create the virtual environment in this way, and then try to\nopen the project in PyCharm and configure the project to use this virtual\nenvironment as an "Existing Poetry Environment", PyCharm sometimes has some\nissues (don\'t know why) which might be problematic. If you encounter such\nissues, you can resolve these issues by deleting the virtual environment\nand creating the Poetry virtual environment using PyCharm (see above).\n\n### Project Makefile commands\n\nYou can start EventStoreDB using the following command.\n\n    $ make start-eventstoredb\n\nYou can run tests using the following command (needs EventStoreDB to be running).\n\n    $ make test\n\nYou can stop EventStoreDB using the following command.\n\n    $ make stop-eventstoredb\n\nYou can check the formatting of the code using the following command.\n\n    $ make lint\n\nYou can reformat the code using the following command.\n\n    $ make fmt\n\nTests belong in `./tests`. Code-under-test belongs in `./eventsourcing_eventstoredb`.\n\nEdit package dependencies in `pyproject.toml`. Update installed packages (and the\n`poetry.lock` file) using the following command.\n\n    $ make update-packages\n',
    'author': 'John Bywater',
    'author_email': 'john.bywater@appropriatesoftware.net',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/pyeventsourcing/eventsourcing-eventstoredb',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
