"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends constructs_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: props.tableName ?? cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: props.sortStyle ?? this.getDefaultSortStyle(props.tableColumns),
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            util_1.getDistKeyColumn(columns);
        }
        catch (err) {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = util_1.getDistKeyColumn(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Table", version: "2.48.0-alpha.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
//# sourceMappingURL=data:application/json;base64,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