# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/sgd_shapley.ipynb (unless otherwise specified).

__all__ = ['ncr', 'SGDshapley']

# Cell
# Author: Simon Grah <simon.grah@thalesgroup.com>
#         Vincent Thouvenot <vincent.thouvenot@thalesgroup.com>

# MIT License

# Copyright (c) 2020 Thales Six GTS France

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

# Cell
import numpy as np
import pandas as pd
import operator as op
from functools import reduce
import random
from tqdm import tqdm

# Cell
def ncr(n, r):
    """
    Combinatorial computation: number of subsets of size r among n elements
    Efficient algorithm
    """
    r = min(r, n-r)
    numer = reduce(op.mul, range(n, n-r, -1), 1)
    denom = reduce(op.mul, range(1, r+1), 1)
    return numer / denom

# Cell
class SGDshapley():
    """
    Estimate the Shapley Values using a Projected Stochastic Gradient algorithm.
    """

    def __init__(self, d, C):
        """
        Calculate internal values for later purposes
        Those elements depend only on the number of features d

        Parameters
        ----------
        d : integer
            Dimension of the problem. The number of features
        """

        # Store in a dictionary for each size k of coalitions
        dict_ω_k = dict() # weights per size k
        dict_L_k = dict() # L-smooth constant per size k
        D = C * np.sqrt(d)
        for k in range(1, d):
            ω_k = (d - 1) / (ncr(d, k) * k * (d - k))
            L_k = ω_k * np.sqrt(k) * (np.sqrt(k) * D + C)
            dict_ω_k.update({k: ω_k})
            dict_L_k.update({k: L_k})
        # Summation of all L per coalition (closed formula)
        sum_L = np.sum([(d-1)/(np.sqrt(k)*(d-k)) * (np.sqrt(k)*D + C) for k in range(1, d)])
        # Probability distributions for sampling new instance
        # Classic SGD
        p = [ncr(d,k) for k in range(1,d)]
        p /= np.sum(p)
        # Importance Sampling proposal q
        q = np.array(list(dict_L_k.values())) * np.array(p)
        q /= np.sum(q)

        # Save internal attributes
        self.d = d
        self.n = 2**d - 2
        self.dict_ω_k = dict_ω_k
        self.dict_L_k = dict_L_k
        self.sum_L = sum_L
        self.p = p
        self.q = q

    def _F_i(self, Φ, x_i, y_i, ω_i):
        """Function value per instance i"""
        res = .5 * self.n * ω_i * (np.dot(x_i, Φ) - y_i)**2
        return res

    def _grad_F_i(self, Φ, x_i, y_i, ω_i):
        """Gradient vector per instance i"""
        res = ω_i * x_i[:,None].dot(x_i[None,:]).dot(Φ) - ω_i * y_i * x_i
        return res

    def _Π_1(self, x, b):
        """Projection Π on convex set K_1"""
        if np.abs((np.sum(x) - b)) <= 1e-6:
            return x
        else:
            return x - (np.sum(x) - b)/len(x)

    def _Π_2(self, x, D):
        """Projection Π on convex set K_2"""
        if np.linalg.norm(x) > D:
            return x * D / np.linalg.norm(x)
        else:
            return x

    def _Dykstra_proj(self, x, D, b, iter_proj=100, epsilon=1e-6):
        """
        Dykstra's algorithm to find orthogonal projection
        onto intersection of convex sets
        """
        xk = x.copy()
        d = len(x)
        pk, qk = np.zeros(d), np.zeros(d)
        for k in range(iter_proj):
            yk = self._Π_2(xk + pk, D)
            pk = xk + pk - yk
            if np.linalg.norm(self._Π_1(yk + qk, b) - xk, 2) <= epsilon:
                break
            else:
                xk = self._Π_1(yk + qk, b)
                qk = yk + qk - xk
        return xk

    def sgd(self, x, fc, ref, n_iter=100, step=.1, step_type="sqrt",
            callback=None, Φ_0=False):
        """
        Stochastic gradient descent algorithm
        The game is defined for an element x, a reference r and function fc

        """

        # Get general information
        feature_names = list(x.index)
        f_x, f_r = fc(x.values), fc(ref.values)
        v_M = f_x - f_r

        d = self.d
        n = 2**d - 2
        p = self.p
        dict_ω_k = self.dict_ω_k
        q = self.q
        dict_L_k = self.dict_L_k
        sum_L = self.sum_L

        # Store Shapley Values in a pandas Series
        if Φ_0:
            Φ = Φ_0.copy()
        else:
            Φ = np.zeros(d)
        Φ_storage = np.zeros((n_iter,d))

        # projection onto convex set K by using a simple algorithm
        # Φ = self._Dykstra_proj(Φ, D, v_M, iter_proj, epsilon=1e-6)
        Φ = Φ - (np.sum(Φ) - v_M) / d

        # Sample in advance coalition sizes
        list_k = np.random.choice(list(range(1, d)), size=n_iter, p=q)

        for t in tqdm(range(1, n_iter+1)):
            # build x_i
            k = list_k[t-1]
            indexes = np.random.permutation(d)[:k]
            x_i = np.zeros(d)
            x_i[indexes] = 1
            # Compute y_i
            z_S = np.array([x.values[j] if x_i[j] == 1 else ref.values[j] for j in range(d)])
            f_S = fc(z_S)
            y_i = f_S - f_r
            # get weight ω_i
            ω_i = dict_ω_k[k]
            # calculate gradient
            p_i = dict_L_k[k] / sum_L
            grad_i = 1/(p_i) * self._grad_F_i(Φ, x_i, y_i, ω_i)
            # update Φ
            if step_type == "constant":
                Φ = Φ - step * grad_i
            elif step_type == "sqrt":
                Φ = Φ - (step/np.sqrt(t)) * grad_i
            elif step_type == "inverse":
                Φ = Φ - (step/(t)) * grad_i

            # projection onto convex set K
            # Φ = self._Dykstra_proj(Φ, D, v_M, iter_proj, epsilon=1e-6)
            Φ = Φ - (Φ.sum() - v_M) / d

            # update storage of Φ
            Φ_storage[t-1,:] = Φ

            if callback and (t % d == 0):
                callback(pd.Series(np.mean(Φ_storage[:t,:],axis=0), index=feature_names))

        # Average all Φ
        Φ = pd.Series(np.mean(Φ_storage,axis=0), index=feature_names)

        return Φ