"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
// import * as kong_core from '../../../../../../kong-core';
const kong_core = require("kong-core");
const defaultHelmProps = {
    repository: 'https://charts.konghq.com',
    chart: 'kong',
    release: 'kong',
};
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.helmOptions = { ...defaultHelmProps, ...props.HelmOptions };
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        // const route_53_zone = aws_route53.HostedZone.fromLookup(this, 'MyZone', {
        //   domainName: props.endpoints.hostedZoneName,
        // });
        // const admin_cert = new aws_certificatemanager.Certificate(this, 'adminCert', {
        //   domainName: props.endpoints.adminDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const cluster_cert = new aws_certificatemanager.Certificate(this, 'clusterCert', {
        //   domainName: props.endpoints.clusterDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const telemetry_cert = new aws_certificatemanager.Certificate(this, 'telemetryCert', {
        //   domainName: props.endpoints.telemetryDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const manager_cert = new aws_certificatemanager.Certificate(this, 'managerCert', {
        //   domainName: props.endpoints.managerDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${props.license_secret_name}-??????`,
                `${props.rds.secret?.secretFullArn}`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: kong_core.Tls.KONG_CP_CERTNAME,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.clusterDns,
                            props.endpoints.telemetryDns,
                            props.endpoints.hostedZoneName,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: kong_core.Tls.KONG_CP_CLUSTER_ISSUER_NAME,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${kong_core.Tls.KONG_CP_CLUSTER_ISSUER_NAME}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        const kong_client_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongClientCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: kong_core.Tls.KONG_ACME_CERTNAME,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.adminDns,
                            props.endpoints.managerDns,
                            props.endpoints.hostedZoneName,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            // group: 'awspca.cert-manager.io',
                            kind: 'ClusterIssuer',
                            name: kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        kong_client_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${props.rds.secret?.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...this.helmOptions,
            cluster: props.cluster,
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                ingressController: {
                    enabled: false,
                },
                image: {
                    repository: 'kong/kong-gateway',
                    tag: '2.8',
                },
                env: {
                    // cluster_listen: `${props.endpoints.clusterDns}:443 ssl`,
                    // cluster_telemetry_listen: `${props.endpoints.telemetryDns}:443 ssl`,
                    // admin_gui_listen: `${props.endpoints.managerDns}:443 ssl`,
                    // admin_listen: `${props.endpoints.adminDns}:443 ssl`,
                    admin_gui_url: `https://${props.endpoints.managerDns}`,
                    admin_api_uri: `${props.endpoints.adminDns}:443`,
                    cluster_data_plane_purge_delay: 86400,
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: `/etc/secrets/${kong_core.Tls.KONG_CP_CLUSTER_ISSUER_NAME}-secret/tls.crt`,
                    cluster_cert_key: `/etc/secrets/${kong_core.Tls.KONG_CP_CLUSTER_ISSUER_NAME}-secret/tls.key`,
                    cluster_ca_cert: `/etc/secrets/${kong_core.Tls.KONG_CP_CLUSTER_ISSUER_NAME}-secret/ca.crt`,
                    admin_ssl_cert: `/etc/secrets/${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret/tls.crt`,
                    admin_ssl_cert_key: `/etc/secrets/${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret/tls.key`,
                    admin_gui_ssl_cert: `/etc/secrets/${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret/tls.crt`,
                    admin_gui_ssl_cert_key: `/etc/secrets/${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret/tls.key`,
                    // cluster_telemetry_listen: '0.0.0.0:8006',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': cluster_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-cluster`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.clusterDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': telemetry_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.telemetryDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-telemetry`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                proxy: {
                    enabled: false,
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    tls: {
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': admin_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.adminDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-admin`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    // http: {
                    //   enabled: false,
                    // },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': manager_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-manager`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.managerDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    `${kong_core.Tls.KONG_CP_CLUSTER_ISSUER_NAME}-secret`,
                    `${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret`,
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account, kong_certificate, kong_client_certificate);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,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