"""Constants for ChessBoard."""

from platform import system

from .datatypes import Color, Piece, PieceType, Side

COLORS: tuple[Color, Color] = ("white", "black")
SIDES: tuple[Side, Side] = ("queenside", "kingside")
FILES = list("abcdefgh")
SQUARES = [f"{file}{rank}" for rank in range(1, 9) for file in FILES]
PIECE_SYMBOLS: dict[PieceType, str] = {
    "king": "♚",
    "queen": "♛",
    "rook": "♜",
    "bishop": "♝",
    "knight": "♞",
    "pawn": "♟︎" if "Windows" not in system() else ":chess_pawn:",
}
BLACK_SQUARES = [f"{file}{rank}" for file in "aceg" for rank in (1, 3, 5, 7)] + [
    f"{file}{rank}" for file in "bdfh" for rank in (2, 4, 6, 8)
]
WHITE_SQUARES = [sq for sq in SQUARES if sq not in BLACK_SQUARES]
PLAINTEXT_ABBRS: dict[str, str] = {
    "knight": "N",
    "rook": "R",
    "bishop": "B",
    "pawn": "P",
    "queen": "Q",
    "king": "K",
}
ALGEBRAIC_PIECE_ABBRS: dict[str, PieceType] = {
    "K": "king",
    "Q": "queen",
    "R": "rook",
    "B": "bishop",
    "N": "knight",
    "": "pawn",
}
FEN_REPRESENTATIONS: dict[str, Piece] = {
    "N": Piece("knight", "white"),
    "K": Piece("king", "white"),
    "R": Piece("rook", "white"),
    "B": Piece("bishop", "white"),
    "Q": Piece("queen", "white"),
    "P": Piece("pawn", "white"),
    "n": Piece("knight", "black"),
    "k": Piece("king", "black"),
    "r": Piece("rook", "black"),
    "b": Piece("bishop", "black"),
    "q": Piece("queen", "black"),
    "p": Piece("pawn", "black"),
}
PIECES: list[Piece] = [
    *([Piece("rook", "white")] * 2),
    *([Piece("knight", "white")] * 2),
    *([Piece("bishop", "white")] * 2),
    Piece("queen", "white"),
    Piece("king", "white"),
    *([Piece("pawn", "white")] * 8),
    *([Piece("rook", "black")] * 2),
    *([Piece("knight", "black")] * 2),
    *([Piece("bishop", "black")] * 2),
    Piece("queen", "black"),
    Piece("king", "black"),
    *([Piece("pawn", "black")] * 8),
]
CASTLING_FINAL_SQUARES: dict[tuple[Color, Side], tuple[str, str]] = {
    ("white", "kingside"): ("g1", "f1"),  # color, side: rook, king
    ("white", "queenside"): ("c1", "d1"),
    ("black", "kingside"): ("g8", "f8"),
    ("black", "queenside"): ("c8", "d8"),
}
KNIGHT_NAVIGABLE_SQUARES: dict[str, list[str]] = {
    "a1": ["c2", "b3"],
    "b1": ["a3", "d2", "c3"],
    "c1": ["a2", "b3", "e2", "d3"],
    "d1": ["b2", "c3", "f2", "e3"],
    "e1": ["c2", "d3", "g2", "f3"],
    "f1": ["d2", "e3", "h2", "g3"],
    "g1": ["e2", "f3", "h3"],
    "h1": ["f2", "g3"],
    "a2": ["c3", "b4", "c1"],
    "b2": ["a4", "d3", "c4", "d1"],
    "c2": ["a3", "b4", "e3", "d4", "a1", "e1"],
    "d2": ["b3", "c4", "f3", "e4", "b1", "f1"],
    "e2": ["c3", "d4", "g3", "f4", "c1", "g1"],
    "f2": ["d3", "e4", "h3", "g4", "d1", "h1"],
    "g2": ["e3", "f4", "h4", "e1"],
    "h2": ["f3", "g4", "f1"],
    "a3": ["c4", "b5", "c2", "b1"],
    "b3": ["a5", "d4", "c5", "a1", "d2", "c1"],
    "c3": ["a4", "b5", "e4", "d5", "a2", "b1", "e2", "d1"],
    "d3": ["b4", "c5", "f4", "e5", "b2", "c1", "f2", "e1"],
    "e3": ["c4", "d5", "g4", "f5", "c2", "d1", "g2", "f1"],
    "f3": ["d4", "e5", "h4", "g5", "d2", "e1", "h2", "g1"],
    "g3": ["e4", "f5", "h5", "e2", "f1", "h1"],
    "h3": ["f4", "g5", "f2", "g1"],
    "a4": ["c5", "b6", "c3", "b2"],
    "b4": ["a6", "d5", "c6", "a2", "d3", "c2"],
    "c4": ["a5", "b6", "e5", "d6", "a3", "b2", "e3", "d2"],
    "d4": ["b5", "c6", "f5", "e6", "b3", "c2", "f3", "e2"],
    "e4": ["c5", "d6", "g5", "f6", "c3", "d2", "g3", "f2"],
    "f4": ["d5", "e6", "h5", "g6", "d3", "e2", "h3", "g2"],
    "g4": ["e5", "f6", "h6", "e3", "f2", "h2"],
    "h4": ["f5", "g6", "f3", "g2"],
    "a5": ["c6", "b7", "c4", "b3"],
    "b5": ["a7", "d6", "c7", "a3", "d4", "c3"],
    "c5": ["a6", "b7", "e6", "d7", "a4", "b3", "e4", "d3"],
    "d5": ["b6", "c7", "f6", "e7", "b4", "c3", "f4", "e3"],
    "e5": ["c6", "d7", "g6", "f7", "c4", "d3", "g4", "f3"],
    "f5": ["d6", "e7", "h6", "g7", "d4", "e3", "h4", "g3"],
    "g5": ["e6", "f7", "h7", "e4", "f3", "h3"],
    "h5": ["f6", "g7", "f4", "g3"],
    "a6": ["c7", "b8", "c5", "b4"],
    "b6": ["a8", "d7", "c8", "a4", "d5", "c4"],
    "c6": ["a7", "b8", "e7", "d8", "a5", "b4", "e5", "d4"],
    "d6": ["b7", "c8", "f7", "e8", "b5", "c4", "f5", "e4"],
    "e6": ["c7", "d8", "g7", "f8", "c5", "d4", "g5", "f4"],
    "f6": ["d7", "e8", "h7", "g8", "d5", "e4", "h5", "g4"],
    "g6": ["e7", "f8", "h8", "e5", "f4", "h4"],
    "h6": ["f7", "g8", "f5", "g4"],
    "a7": ["c8", "c6", "b5"],
    "b7": ["d8", "a5", "d6", "c5"],
    "c7": ["a8", "e8", "a6", "b5", "e6", "d5"],
    "d7": ["b8", "f8", "b6", "c5", "f6", "e5"],
    "e7": ["c8", "g8", "c6", "d5", "g6", "f5"],
    "f7": ["d8", "h8", "d6", "e5", "h6", "g5"],
    "g7": ["e8", "e6", "f5", "h5"],
    "h7": ["f8", "f6", "g5"],
    "a8": ["c7", "b6"],
    "b8": ["a6", "d7", "c6"],
    "c8": ["a7", "b6", "e7", "d6"],
    "d8": ["b7", "c6", "f7", "e6"],
    "e8": ["c7", "d6", "g7", "f6"],
    "f8": ["d7", "e6", "h7", "g6"],
    "g8": ["e7", "f6", "h6"],
    "h8": ["f7", "g6"],
}
KING_NAVIGABLE_SQUARES: dict[str, list[str]] = {
    "a1": ["a2", "b1", "b2"],
    "b1": ["b2", "c1", "a1", "c2", "a2"],
    "c1": ["c2", "d1", "b1", "d2", "b2"],
    "d1": ["d2", "e1", "c1", "e2", "c2"],
    "e1": ["e2", "f1", "d1", "f2", "d2"],
    "f1": ["f2", "g1", "e1", "g2", "e2"],
    "g1": ["g2", "h1", "f1", "h2", "f2"],
    "h1": ["h2", "g1", "g2"],
    "a2": ["a3", "b2", "a1", "b3", "b1"],
    "b2": ["b3", "c2", "a2", "b1", "c3", "a3", "c1", "a1"],
    "c2": ["c3", "d2", "b2", "c1", "d3", "b3", "d1", "b1"],
    "d2": ["d3", "e2", "c2", "d1", "e3", "c3", "e1", "c1"],
    "e2": ["e3", "f2", "d2", "e1", "f3", "d3", "f1", "d1"],
    "f2": ["f3", "g2", "e2", "f1", "g3", "e3", "g1", "e1"],
    "g2": ["g3", "h2", "f2", "g1", "h3", "f3", "h1", "f1"],
    "h2": ["h3", "g2", "h1", "g3", "g1"],
    "a3": ["a4", "b3", "a2", "b4", "b2"],
    "b3": ["b4", "c3", "a3", "b2", "c4", "a4", "c2", "a2"],
    "c3": ["c4", "d3", "b3", "c2", "d4", "b4", "d2", "b2"],
    "d3": ["d4", "e3", "c3", "d2", "e4", "c4", "e2", "c2"],
    "e3": ["e4", "f3", "d3", "e2", "f4", "d4", "f2", "d2"],
    "f3": ["f4", "g3", "e3", "f2", "g4", "e4", "g2", "e2"],
    "g3": ["g4", "h3", "f3", "g2", "h4", "f4", "h2", "f2"],
    "h3": ["h4", "g3", "h2", "g4", "g2"],
    "a4": ["a5", "b4", "a3", "b5", "b3"],
    "b4": ["b5", "c4", "a4", "b3", "c5", "a5", "c3", "a3"],
    "c4": ["c5", "d4", "b4", "c3", "d5", "b5", "d3", "b3"],
    "d4": ["d5", "e4", "c4", "d3", "e5", "c5", "e3", "c3"],
    "e4": ["e5", "f4", "d4", "e3", "f5", "d5", "f3", "d3"],
    "f4": ["f5", "g4", "e4", "f3", "g5", "e5", "g3", "e3"],
    "g4": ["g5", "h4", "f4", "g3", "h5", "f5", "h3", "f3"],
    "h4": ["h5", "g4", "h3", "g5", "g3"],
    "a5": ["a6", "b5", "a4", "b6", "b4"],
    "b5": ["b6", "c5", "a5", "b4", "c6", "a6", "c4", "a4"],
    "c5": ["c6", "d5", "b5", "c4", "d6", "b6", "d4", "b4"],
    "d5": ["d6", "e5", "c5", "d4", "e6", "c6", "e4", "c4"],
    "e5": ["e6", "f5", "d5", "e4", "f6", "d6", "f4", "d4"],
    "f5": ["f6", "g5", "e5", "f4", "g6", "e6", "g4", "e4"],
    "g5": ["g6", "h5", "f5", "g4", "h6", "f6", "h4", "f4"],
    "h5": ["h6", "g5", "h4", "g6", "g4"],
    "a6": ["a7", "b6", "a5", "b7", "b5"],
    "b6": ["b7", "c6", "a6", "b5", "c7", "a7", "c5", "a5"],
    "c6": ["c7", "d6", "b6", "c5", "d7", "b7", "d5", "b5"],
    "d6": ["d7", "e6", "c6", "d5", "e7", "c7", "e5", "c5"],
    "e6": ["e7", "f6", "d6", "e5", "f7", "d7", "f5", "d5"],
    "f6": ["f7", "g6", "e6", "f5", "g7", "e7", "g5", "e5"],
    "g6": ["g7", "h6", "f6", "g5", "h7", "f7", "h5", "f5"],
    "h6": ["h7", "g6", "h5", "g7", "g5"],
    "a7": ["a8", "b7", "a6", "b8", "b6"],
    "b7": ["b8", "c7", "a7", "b6", "c8", "a8", "c6", "a6"],
    "c7": ["c8", "d7", "b7", "c6", "d8", "b8", "d6", "b6"],
    "d7": ["d8", "e7", "c7", "d6", "e8", "c8", "e6", "c6"],
    "e7": ["e8", "f7", "d7", "e6", "f8", "d8", "f6", "d6"],
    "f7": ["f8", "g7", "e7", "f6", "g8", "e8", "g6", "e6"],
    "g7": ["g8", "h7", "f7", "g6", "h8", "f8", "h6", "f6"],
    "h7": ["h8", "g7", "h6", "g8", "g6"],
    "a8": ["b8", "a7", "b7"],
    "b8": ["c8", "a8", "b7", "c7", "a7"],
    "c8": ["d8", "b8", "c7", "d7", "b7"],
    "d8": ["e8", "c8", "d7", "e7", "c7"],
    "e8": ["f8", "d8", "e7", "f7", "d7"],
    "f8": ["g8", "e8", "f7", "g7", "e7"],
    "g8": ["h8", "f8", "g7", "h7", "f7"],
    "h8": ["g8", "h7", "g7"],
}
PIECES_TO_TRACK: list[tuple[PieceType, Color, Side | None]] = [
    ("king", "white", None),
    ("rook", "white", "kingside"),
    ("rook", "white", "queenside"),
    ("king", "black", None),
    ("rook", "black", "kingside"),
    ("rook", "black", "queenside"),
]
