"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const yaml = require("js-yaml");
const path = require("path");
const resourceType = 'Custom::SSM-Document';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An SSM document
 */
class Document extends cdk.Construct {
    /**
     * Defines a new SSM document
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Name of the document
         */
        this.name = '';
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::SSM-Document');
        this.tags.setTag('CreatedBy', ID);
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const name = this.fixDocumentName(props.name);
        if (name.length < 3 || name.length > 128) {
            this.node.addError(`SSM Document name ${name} is invalid. The name must be between 3 and 128 characters.`);
            return;
        }
        let content = props.content;
        if (typeof content === 'string') {
            content = yaml.safeLoad(content);
        }
        const document = new cfn.CustomResource(this, `SSM-Document-${name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                updateDefaultVersion: props.updateDefaultVersion || true,
                name: name,
                content: content,
                documentType: props.documentType || 'Command',
                targetType: props.targetType || '/',
                StackName: stack,
                tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        this.name = document.getAttString('Name');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'SSM-Document-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'SSM-Document-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ssm:ListDocuments',
                        'ssm:ListTagsForResource',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'ssm:CreateDocument',
                        'ssm:AddTagsToResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:RequestTag/CreatedBy': ID,
                        },
                    },
                }),
                new iam.PolicyStatement({
                    actions: [
                        'ssm:DeleteDocument',
                        'ssm:DescribeDocument',
                        'ssm:GetDocument',
                        'ssm:ListDocumentVersions',
                        'ssm:ModifyDocumentPermission',
                        'ssm:UpdateDocument',
                        'ssm:UpdateDocumentDefaultVersion',
                        'ssm:AddTagsToResource',
                        'ssm:RemoveTagsFromResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/CreatedBy': ID,
                        },
                    },
                }),
            ],
        });
        const role = new iam.Role(stack, 'SSM-Document-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage SSM Documents',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    fixDocumentName(name) {
        return name.replace(/[^a-zA-Z0-9_.-]+/g, '-');
    }
}
exports.Document = Document;
//# sourceMappingURL=data:application/json;base64,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