# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/07_calib.row2row.ipynb (unless otherwise specified).

__all__ = ['steffl_spica_dates', 'steffl_spica_nasa_dates', 'Row2Row', 'create_detector_stack']

# Cell
from functools import cached_property

import matplotlib.pyplot as plt
import numpy as np
import param
from tqdm.auto import tqdm, trange

import holoviews as hv
import hvplot.xarray  # noqa
import pandas as pd
import xarray as xr
from nbdev.showdoc import show_doc
from planetarypy.utils import iso_to_nasa_time
from .greg import StarCalibs
from ..io import UVPDS, UVISObs
from ..pds import CatalogFilter

# Cell
steffl_spica_dates = ["2001-04-03", "2002-07-17", "2003-05-19"]
steffl_spica_nasa_dates = [iso_to_nasa_time(i) for i in steffl_spica_dates]
steffl_spica_nasa_dates

# Cell
class Row2Row:
    def __init__(self, pid, do_sum_windowed=True):
        self.pid = pid
        self.data = UVPDS(pid).xarray.astype("int16")
        self.do_sum_windowed = do_sum_windowed

    @property
    def plot_set(self):
        pmin, pmax = np.percentile(self.data, (0.5, 99.5))
        return self.data.hvplot.image(cmap="viridis", clim=(None, None), title=self.pid)

    @property
    def integrated(self):
        if not self.do_sum_windowed:
            data = self.data
        else:
            data = self.data.isel(spatial=slice(3, 61))
        return data.sum(dim="samples")

    @property
    def summed(self):
        "Alias for `integrated`"
        return self.integrated

    @property
    def plot_integrated(self):
        return self.integrated.hvplot(
            x="spectral", y="spatial", cmap="viridis", title=self.pid
        )

    @property
    def averaged(self):
        return self.integrated.sel(spatial=slice(3, 61)).mean(dim="spatial")

    @property
    def plot_averaged(self):
        return self.averaged.hvplot(title=self.pid)

    @property
    def column_std(self):
        return self.integrated.sel(spatial=slice(3, 61)).std(dim="spatial")

    @property
    def plot_column_std(self):
        return self.column_std.hvplot(title=f"{self.pid}, Column STD")

    @property
    def ff(self):
        data = self.averaged / self.integrated
        data = data.where(np.isfinite(data), other=np.nan)
        data.loc[:, 61:] = 1
        data.loc[:, :3] = 1
        return data

    def plot_ff(self, vmax=4):
        return self.ff.hvplot(
            cmap="viridis", title=self.pid, clim=(None, vmax), logz=True
        )

# Cell
def create_detector_stack(
    data,  # numpy 3D array to embed to xarray
    name,  # name of data
    third_dim,  # name of third dimension ("scan", "along", "across")
    third_dim_coords,  # coordinates for 3rd dim
    orig,  # xarray to copy coords from
):
    """Function to create a 2D detector array with variable 3rd dimensions.

    It uses an `orig` array to copy the coordinates from.
    """
    return xr.DataArray(
        data,
        dims=["spectral", "spatial"] + [third_dim],
        coords={
            "spectral": orig.spectral,
            "spatial": orig.spatial,
            third_dim: third_dim_coords,
        },
        name=name,
    )