import { __awaiter, __generator } from "tslib";
import { HttpResponse } from "@aws-sdk/protocol-http";
import { buildQueryString } from "@aws-sdk/querystring-builder";
import { Agent as hAgent, request as hRequest } from "http";
import { Agent as hsAgent, request as hsRequest } from "https";
import { NODEJS_TIMEOUT_ERROR_CODES } from "./constants";
import { getTransformedHeaders } from "./get-transformed-headers";
import { setConnectionTimeout } from "./set-connection-timeout";
import { setSocketTimeout } from "./set-socket-timeout";
import { writeRequestBody } from "./write-request-body";
var NodeHttpHandler = (function () {
    function NodeHttpHandler(options) {
        var _this = this;
        this.metadata = { handlerProtocol: "http/1.1" };
        this.configProvider = new Promise(function (resolve, reject) {
            if (typeof options === "function") {
                options()
                    .then(function (_options) {
                    resolve(_this.resolveDefaultConfig(_options));
                })
                    .catch(reject);
            }
            else {
                resolve(_this.resolveDefaultConfig(options));
            }
        });
    }
    NodeHttpHandler.prototype.resolveDefaultConfig = function (options) {
        var _a = options || {}, connectionTimeout = _a.connectionTimeout, socketTimeout = _a.socketTimeout, httpAgent = _a.httpAgent, httpsAgent = _a.httpsAgent;
        var keepAlive = true;
        var maxSockets = 50;
        return {
            connectionTimeout: connectionTimeout,
            socketTimeout: socketTimeout,
            httpAgent: httpAgent || new hAgent({ keepAlive: keepAlive, maxSockets: maxSockets }),
            httpsAgent: httpsAgent || new hsAgent({ keepAlive: keepAlive, maxSockets: maxSockets }),
        };
    };
    NodeHttpHandler.prototype.destroy = function () {
        var _a, _b, _c, _d;
        (_b = (_a = this.config) === null || _a === void 0 ? void 0 : _a.httpAgent) === null || _b === void 0 ? void 0 : _b.destroy();
        (_d = (_c = this.config) === null || _c === void 0 ? void 0 : _c.httpsAgent) === null || _d === void 0 ? void 0 : _d.destroy();
    };
    NodeHttpHandler.prototype.handle = function (request, _a) {
        var _b = _a === void 0 ? {} : _a, abortSignal = _b.abortSignal;
        return __awaiter(this, void 0, void 0, function () {
            var _c;
            var _this = this;
            return __generator(this, function (_d) {
                switch (_d.label) {
                    case 0:
                        if (!!this.config) return [3, 2];
                        _c = this;
                        return [4, this.configProvider];
                    case 1:
                        _c.config = _d.sent();
                        _d.label = 2;
                    case 2: return [2, new Promise(function (resolve, reject) {
                            if (!_this.config) {
                                throw new Error("Node HTTP request handler config is not resolved");
                            }
                            if (abortSignal === null || abortSignal === void 0 ? void 0 : abortSignal.aborted) {
                                var abortError = new Error("Request aborted");
                                abortError.name = "AbortError";
                                reject(abortError);
                                return;
                            }
                            var isSSL = request.protocol === "https:";
                            var queryString = buildQueryString(request.query || {});
                            var nodeHttpsOptions = {
                                headers: request.headers,
                                host: request.hostname,
                                method: request.method,
                                path: queryString ? "".concat(request.path, "?").concat(queryString) : request.path,
                                port: request.port,
                                agent: isSSL ? _this.config.httpsAgent : _this.config.httpAgent,
                            };
                            var requestFunc = isSSL ? hsRequest : hRequest;
                            var req = requestFunc(nodeHttpsOptions, function (res) {
                                var httpResponse = new HttpResponse({
                                    statusCode: res.statusCode || -1,
                                    headers: getTransformedHeaders(res.headers),
                                    body: res,
                                });
                                resolve({ response: httpResponse });
                            });
                            req.on("error", function (err) {
                                if (NODEJS_TIMEOUT_ERROR_CODES.includes(err.code)) {
                                    reject(Object.assign(err, { name: "TimeoutError" }));
                                }
                                else {
                                    reject(err);
                                }
                            });
                            setConnectionTimeout(req, reject, _this.config.connectionTimeout);
                            setSocketTimeout(req, reject, _this.config.socketTimeout);
                            if (abortSignal) {
                                abortSignal.onabort = function () {
                                    req.abort();
                                    var abortError = new Error("Request aborted");
                                    abortError.name = "AbortError";
                                    reject(abortError);
                                };
                            }
                            writeRequestBody(req, request);
                        })];
                }
            });
        });
    };
    return NodeHttpHandler;
}());
export { NodeHttpHandler };
