# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fastapi_cachette',
 'fastapi_cachette.backends',
 'fastapi_cachette.codecs',
 'fastapi_cachette.codecs.dataframe']

package_data = \
{'': ['*']}

install_requires = \
['fastapi>=0,<1', 'pydantic>=1.9.0,<2.0.0']

extras_require = \
{'dataframe': ['pyarrow[dataframe]>=7.0.0,<8.0.0'],
 'dataframe:python_version >= "3.7" and python_version < "3.8"': ['pandas[dataframe]<=1.3.5'],
 'dataframe:python_version >= "3.8" and python_version < "4.0"': ['pandas[dataframe]>=1.4.0'],
 'dynamodb': ['aiobotocore[dynamodb,examples]>=2.2.0,<3.0.0'],
 'examples': ['uvicorn[examples]==0.15.0',
              'aiomcache[examples,memcached]>=0.7.0,<0.8.0',
              'aiobotocore[dynamodb,examples]>=2.2.0,<3.0.0',
              'redis[examples,redis]>=4.2.1,<5.0.0'],
 'memcached': ['aiomcache[examples,memcached]>=0.7.0,<0.8.0'],
 'mongodb': ['motor[mongodb,examples]>=2.5.1,<3.0.0'],
 'msgpack': ['msgpack[msgpack]>=1.0.3,<2.0.0'],
 'orjson': ['orjson[orjson]>=3.6.7,<4.0.0'],
 'redis': ['redis[examples,redis]>=4.2.1,<5.0.0']}

setup_kwargs = {
    'name': 'fastapi-cachette',
    'version': '0.1.4',
    'description': 'Cache Implementation Extension for FastAPI Asynchronous Web Framework',
    'long_description': '# FastAPI Cachette\n\n[![Build Status](https://travis-ci.com/aekasitt/fastapi-cachette.svg?branch=master)](https://app.travis-ci.com/github/aekasitt/fastapi-cachette)\n[![Package Vesion](https://img.shields.io/pypi/v/fastapi-cachette)](https://pypi.org/project/fastapi-cachette)\n[![Format](https://img.shields.io/pypi/format/fastapi-cachette)](https://pypi.org/project/fastapi-cachette)\n[![Python Version](https://img.shields.io/pypi/pyversions/fastapi-cachette)](https://pypi.org/project/fastapi-cachette)\n[![License](https://img.shields.io/pypi/l/fastapi-cachette)](https://pypi.org/project/fastapi-cachette)\n\n## Features\n\nOpinionated Cache Extension for FastAPI Asynchronous Web Framework;\nThis is an extension aiming at making cache access on the server\nBy configuration at startup of the FastAPI App instance, you can set the backend and other \nconfiguration options and have it remain a class constant when using FastAPI\'s\nintuitive [Dependency Injection](https://fastapi.tiangolo.com/tutorial/dependencies/) system.\n\nThe design has built-in limitations like fixed codec and backend once the app has been launched and \nencourage developers to design their applications with this in mind.\n\nMost of the Backend implementation is directly lifted from \n[fastapi-cache](https://github.com/long2ice/fastapi-cache) by \n[@long2ice](https://github.com/long2ice) excluding the MongoDB backend option.\n\n## Configuration Options\n\nThe following are the current available configuration keys that can be set on this FastAPI extension\non startup either by using a method which returns a list of tuples or a Pydantic BaseSettings object\n(See examples below or in `examples/` folder)\n\n    backend -- optional; must be one of ["dynamodb", "inmemory", "memcached", "mongodb", "redis"];\n      defaults to using inmemory option which required no extra package dependencies. To use other\n      listed options;\n    codec -- optional; serialization and de-serialization format to have cache values stored in\n      the cache backend of choice as a string of selected encoding. once fetched, will have their\n      decoded values returned of the same format. must be one of ["feather", "msgpack", "parquet",\n      "pickle"]; if none is defined, will vanilla codec of basic string conversion will be used.\n    ttl -- optional; the time-to-live or amount before this cache item expires within the cache;\n      defaults to 60 (seconds) and must be between 1 second to 1 hour (3600 seconds).\n    redis_url -- required when backend set to "redis"; the url set to redis-server instance with\n      or without provided authentication in such formats "redis://user:password@host:port" and\n      "redis://host:port" respectively.\n    memcached_host -- required when backend set to "memcached"; the host endpoint to the memcached\n      distributed memory caching system.\n    table_name -- required when backend set to "dynamodb" or "mongodb"; name of the cache table or\n      collection in case of "mongodb" backend to have key-value pairs stored; defaults to\n      "fastapi-cachette".\n    region -- required when backend set to "dynamodb" and "dynamodb_url" not set; one of Amazon\n      Web Services listed Regions which can be found on this Documentation\n      [Page](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones)\n    dynamodb_url -- required when backend set to "dynamodb" and "region" not set; this option is\n      used when setting up your own DynamoDB Local instance according to this\n      [Guide](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DynamoDBLocal)\n    database_name -- required when backend set to "mongodb"; the database name to be automatically\n      created if not exists on the MongoDB instance and store the cache table; defaults to\n      "fastapi-cachette-database"\n    mongodb_url -- required when backend set to "mongodb"; the url set to MongoDB database\n      instance with or without provided authentication in such formats\n      "mongodb://user:password@host:port" and "mongodb://host:port" respectively.\n\n## Examples\n\nThe following shows and example of setting up FastAPI Cachette in its default configuration, which\nis an In-Memory cache implementation.\n\n```py\nfrom fastapi import FastAPI, Depends\nfrom fastapi.responses import PlainTextResponse\nfrom fastapi_cachette import Cachette\nfrom pydantic import BaseModel\n\napp = FastAPI()\n\n### Routing ###\nclass Payload(BaseModel):\n  key: str\n  value: str\n\n@app.post(\'/\', response_class=PlainTextResponse)\nasync def setter(payload: Payload, cachette: Cachette = Depends()):\n  await cachette.put(payload.key, payload.value)\n  return \'OK\'\n\n@app.get(\'/{key}\', response_class=PlainTextResponse, status_code=200)\nasync def getter(key: str, cachette: Cachette = Depends()):\n  value: str = await cachette.fetch(key)\n  return value\n```\n\nAnd then this is how you set up a FastAPI Cachette with Redis support enabled.\n\n```py\nfrom fastapi import FastAPI, Depends\nfrom fastapi.responses import PlainTextResponse\nfrom fastapi_cachette import Cachette\nfrom pydantic import BaseModel\n\napp = FastAPI()\n\n@Cachette.load_config\ndef get_cachette_config():\n  return [(\'backend\', \'redis\'), (\'redis_url\', \'redis://localhost:6379\')]\n\nclass Payload(BaseModel):\n  key: str\n  value: str\n\n@app.post(\'/\', response_class=PlainTextResponse)\nasync def setter(payload: Payload, cachette: Cachette = Depends()):\n  await cachette.put(payload.key, payload.value)\n  return \'OK\'\n\n@app.get(\'/{key}\', response_class=PlainTextResponse, status_code=200)\nasync def getter(key: str, cachette: Cachette = Depends()):\n  value: str = await cachette.fetch(key)\n  return value\n```\n\n## Upcoming Features (To-Do List)\n\n1. Implement `flush` and `flush_expired` methods on individual backends \n(Not needed for Redis & Memcached backends)\n\n2. Memcached Authentication ([No SASL Support](https://github.com/aio-libs/aiomcache/issues/12))\nChange library?\n\n3. DynamoDB Authentication (Add AWS Access Key ID and AWS Access Secret Key to configuration).\n\n4. Boto3 Version Fix; Current version restrictions vulnerable to `aiohttp` bug.\n\n5. Add behaviors responding to "Cache-Control" request header\n\n6. More character validations for URLs and Database/Table/Collection names in configuration options\n\n## Installation\n\nThe easiest way to start working with this extension with pip\n\n```bash\npip install fastapi-cachette\n# or\npoetry add fastapi-cachette\n```\n\nWhen you familiarize with the basic structure of how to Dependency Inject Cachette within your\nendpoints, please experiment more of using external backends with `extras` installations like\n\n```bash\n# Install FastAPI Cachette\'s extra requirements to Redis support\npip install fastapi-cachette --install-option "--extras-require=redis"\n# or Install FastAPI Cachette\'s support to Memcached\npoetry add fastapi-cachette[memcached]\n# or Special JSON Codec written on Rust at lightning speed\npoetry add fastapi-cachette[orjson]\n# or Include PyArrow package making DataFrame serialization much easier\npip install fastapi-cachette --install-option "--extras-require=dataframe"\n# or MongoDB and DynamoDB supports\npoetry add fastapi-cachette[mongodb]\npip install fastapi-cachette --install-option "--extras-require=dynamodb"\n```\n\n## Getting Started\n\nThis FastAPI extension utilizes "Dependency Injection" (To be continued)\n\nConfiguration of this FastAPI extension must be done at startup using "@Cachette.load_config" \ndecorator (To be continued)\n\nThese are all available options with explanations and validation requirements (To be continued)\n\n## Examples\n\nThe following examples show you how to integrate this extension to a FastAPI App (To be continued)\n\nSee "examples/" folders\n\nTo run examples, first you must install extra dependencies\n\nDo all in one go with this command...\n\n```bash\npip install aiobotocore aiomcache motor uvicorn redis\n# or\npoetry install --extras examples\n```\n\nDo individual example with this command...\n\n```bash\npip install redis\n# or\npoetry install --extras redis\n```\n\n## Contributions\n\nSee features and write tests I guess.\n\n## Test Environment Setup\n\nThis project utilizes multiple external backend services namely AWS DynamoDB, Memcached, MongoDB and\nRedis as backend service options as well as a possible internal option called InMemoryBackend. In\norder to test viability, we must have specific instances of these set up in the background of our\ntesting environment \n\n### With Docker-Compose\n\nUtilize orchestration file attached to reposity and `docker-compose` command to set up testing \ninstances of backend services using the following command...\n\n```bash\ndocker-compose up -d\n```\n\nWhen you are finished, you can stop and remove background running backend instances with the\nfollowing command...\n\n```bash\ndocker-compose down\n```\n\n### Without Docker-Compose\n\nIf you are using `arm64` architecture on your local machine like I am with my fancy MacBook Pro, \nthere is a chance that your `docker-compose` (V1) is not properly configured and have caused you \nmany headaches. The following commands will allow you to replicate docker-compose orchestration\ncommand given above.\n\n\n1. AWS DynamoDB Local\n\n    ```bash\n    docker run --detach --rm -ti -p 8000:8000 --name cachette-dynamodb amazon/dynamodb-local:latest\n    ```\n\n2. Memcached\n\n    ```bash\n    docker run --detach --rm -ti -p 11211:11211 --name cachette-memcached memcached:bullseye\n    ```\n\n3. MongoDB\n\n    ```bash\n    docker run --detach --rm -ti -p 27017:27017 --name cachette-mongodb mongo:latest\n    ```\n\n4. Redis\n\n    ```bash\n    docker run --detach --rm -ti -p 6379:6379 --name cachette-redis redis:bullseye\n    ```\n\nAnd finally, to stop and remove running instances, run the following command\n\n```bash\n[ ! -z $(docker ps -f name="cachette-*" -q | tr -d \'\\n\') ] \\\n  && docker kill $(docker ps -f name="cachette-*" -q)\n```\n\n## Tests\n\nNow that you have background running backend instances, you can proceed with the tests by using\n`pytest` command as such...\n\n```bash\npytest\n```\n\nOr you can configure the command to run specific tests as such...\n\n```bash\npytest -k test_load_invalid_configs\n# or\npytest -k test_set_then_clear\n```\n\nAll test suites must be placed under `tests/` folder or its subfolders.\n\n## License\n\nThis project is licensed under the terms of the MIT license.',
    'author': 'Sitt Guruvanich',
    'author_email': 'aekazitt@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/aekasitt/fastapi-cachette',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
