from spotipy import Spotify

from . import get_current_device, get_instance, formatting

sp: Spotify = get_instance()

assert sp is not None, "Couldn't get the Spotipy instance"

def play(artist_name, song_name):

    result = sp.search(q=f"artist:{artist_name} track:{song_name}", type='track', limit=1)

    if(len(songs := result['tracks']['items']) > 0):
        uri = songs[0]['uri']
    else:
        formatting.fatal("No song matches the search terms!")
        exit(1)

    formatting.info(f"Playing {song_name} by {artist_name}")

    sp.start_playback(uris=[uri], device_id=get_current_device())

def volume(new_vol):
    assert 0 <= new_vol <= 100, "Volume outside bounds"

    formatting.info(f"New volume: {new_vol}")

    sp.volume(volume_percent=new_vol, device_id=get_current_device())

def pause():
    
    formatting.info("Pausing playback")

    sp.pause_playback(device_id=get_current_device())

def resume():
    
    formatting.info("Resuming playback")

    sp.start_playback(device_id=get_current_device())

def skip():

    formatting.info("Skipping current track")

    sp.next_track(device_id=get_current_device())

def prev():

    formatting.info("Playing previous track")

    sp.previous_track(device_id=get_current_device())

def shuffle():

    shuffle_state = sp.current_playback()["shuffle_state"]

    formatting.info(f"Toggling shuffle {'on' if not shuffle_state else 'off'}")

    sp.shuffle(not shuffle_state, device_id=get_current_device())

def repeat(repeat_state):

    formatting.info(f"Setting repeat-state to {repeat_state}")

    sp.repeat(repeat, device_id=get_current_device())

def queue(artist_name, song_name):

    result = sp.search(q=f"artist:{artist_name} track:{song_name}", type='track', limit=1)

    if(len(songs := result['tracks']['items']) > 0):
        uri = songs[0]['uri']
    else:
        formatting.fatal("No song matches the search terms!")
        exit(1)

    formatting.info(f"Playing {song_name} by {artist_name}")

    sp.add_to_queue(uri=uri, device_id=get_current_device())

def fill_queue(count):

    recomendations = sp.recommendations(seed_tracks=[sp.currently_playing()['item']['uri']], limit=count)

    recomendations = list(map(lambda track: track['uri'], recomendations['tracks']))

    formatting.info("Filling queue")

    for uri in recomendations:
        track = sp.track(uri)

        artists = [artist['name'] for artist in track['artists']]

        formatting.info(f"Queueing {track['name']} by {artists[0] if len(artists) == 1 else ' and '.join(artists)}")

        sp.add_to_queue(uri=uri, device_id=get_current_device())

def current():

    current = sp.currently_playing()

    current_song = current['item']['name']

    current_artist = ", ".join([artist['name'] for artist in current['item']['artists']])

    current_progress = str(round(current['progress_ms'] / current['item']['duration_ms'] * 100, 2)) + "%"

    devices = sp.devices()['devices']

    for device in devices:
        if device['is_active']:
            current_device = device['name']
            device_type = device['type']
            volume = device['volume_percent']

    print(f"""Current device: {device_type} - {current_device}
{current_artist} - {current_song}
Progress: {current_progress}
Current volume: {volume}%""")