# AUTOGENERATED! DO NOT EDIT! File to edit: projects/m5/04_out_of_stock_detection.ipynb (unless otherwise specified).

__all__ = ['get_series_df', 'get_stats_df', 'plot_item_series', 'plot_all_item_series', 'fix_oos', 'nan_zeros',
           'mark_streaks', 'make_oos_data']

# Cell
#export
import os
import time
import logging

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
# import seaborn as sns
from fastcore.script import call_parse, Param

from .metric import get_agg, get_df_weights
from ..core import time_taken, pool_func

# Cell
def get_series_df(train_df, rollup_matrix_csr, rollup_index, df_cal=None, fill_xmas=False):
    """Returns a dataframe with series for all 12 levels of aggregation. We also
    replace leading zeros with np.nan and if fill_xmas, replace christmas sales with average
    of the day before and day after christmas"""

    series_df = pd.DataFrame(data=rollup_matrix_csr * train_df.iloc[:, 6:].values,
                         index=rollup_index,
                         columns=train_df.iloc[:, 6:].columns)

    zero_mask = series_df.cumsum(axis=1) * 2 == series_df
    series_df[zero_mask] = np.nan

    if fill_xmas:
        xmas_days = df_cal[df_cal.date.str[-5:] == '12-25'].d.str[2:].astype('int16')
        for x in xmas_days:
            series_df[f'd_{x}'] = (series_df[f'd_{x-1}'] + series_df[f'd_{x+1}']) / 2
    return series_df


def get_stats_df(series_df):
    """Returns a dataframe that shows basic stats for all
    series in sereis_df."""

    percentiles = [.005, .025, .165, .25, .5, .75, .835, .975, .995]
    stats_df = series_df.T.describe(percentiles).T
    stats_df['fraction_0'] = ((series_df == 0).sum(axis = 1) / stats_df['count'])

    return stats_df

# Cell
def plot_item_series(item, series_df, state=None, fillna=False, start=0, end=1941):
    """Plots the level 10-12 series containing the item"""
    item_series_all = series_df.loc[series_df.index.get_level_values(1).str.contains(item)]
    if state:
        state_mask = item_series_all.index.get_level_values(1).str.contains(state)
        if fillna:
            item_series_all.loc[state_mask].iloc[:, start:end].fillna(fillna).T.plot(title=f'{item} overall in {state} and by store')
        else:
            item_series_all.loc[state_mask].iloc[:, start:end].T.plot(title=f'{item} overall in {state} and by store')
        plt.legend(bbox_to_anchor=(1,1.04), loc='lower right', ncol=1)
        for i in [1941 - 364*i for i in range(6) if start < 1941 - 364*i <= end]:
            plt.axvline(i, ls=':')
        plt.show()

    else:
        if fillna:
            item_series_all.iloc[:4, start:end].fillna(fillna).T.plot(title=f'{item} total and by state')
        else:
            item_series_all.iloc[:4, start:end].T.plot(title=f'{item} total and by state')
        plt.legend(bbox_to_anchor=(.5,.99), loc='upper center', ncol=1)
        for i in [1941 - 364*i for i in range(6) if start < 1941 - 364*i <= end]:
            plt.axvline(i, ls=':')
        plt.show()

def plot_all_item_series(item, series_df, fillna=False, start=0, end=1941):
    plot_item_series(item, series_df, state=None, fillna=fillna, start=start, end=end)
    for state in ['CA', 'TX', 'WI']:
        plot_item_series(item, series_df, state=state, fillna=fillna, start=start, end=end)

# Cell
def fix_oos(item, series_df):
    """Processes item and returns series that has np.nan
    where we think out of stock zeros occur"""
    item_series = series_df.loc[item].copy()
    item_mean = np.nanmean(item_series)
    x = True
    while x == True:
        item_series, new_mean, streak_length, item_streaks, limit_99 = nan_zeros(item_series, item_mean)
        x = new_mean > item_mean
        item_mean = new_mean

    return item_series, new_mean, streak_length, item_streaks, limit_99

def nan_zeros(item_series, item_mean):
    """Returns item_series with streaks replaced by nans,
    the new average of item series, and max_streak_length,
    which is the highest streak
    count that was not replaced with nans."""
    # With the mean, we can find the probability
    # of a zero sales, given the item follows
    # the poisson distribution
    prob_0 = np.exp(-item_mean)

    # Adding this to make sure we catch long streaks that
    # artificially decrease our starting mean, leading to
    # an artificially large
    lowest_prob_allowed = .000_001
    lowest_streak_allowed = 1
    while prob_0 ** lowest_streak_allowed > lowest_prob_allowed:
        lowest_streak_allowed += 1

    # Given the probability of a zero, we can find
    # the expected value of the total number of
    # zeros.
    expected_zeros = prob_0 * (~np.isnan(item_series)).sum()

    # Given the number of total zeros should
    # follow the binomial distribution, approximated
    # by the normal distribution, we can assume
    # that total zeros are below mean + 3 standard
    # deviations 99.85 percent of the time.
    std = np.sqrt((~np.isnan(item_series)).sum() * prob_0 * (1-prob_0))
    limit_99 = expected_zeros + 3 * std
    item_streaks = mark_streaks(item_series)
    max_streak_length = 1
    total_zeros = (item_streaks == max_streak_length).sum()
    while (total_zeros < limit_99) & (max_streak_length < lowest_streak_allowed):
        max_streak_length += 1
        total_zeros = (item_streaks == max_streak_length).sum()

    # Now remove the zeros in streaks greater
    # than max_streak_length
    m = min(max_streak_length, lowest_streak_allowed)
    item_series = np.where(item_streaks > m, np.nan, item_series)
    new_mean = np.nanmean(item_series)

    return item_series, new_mean, max_streak_length, item_streaks, limit_99

###### Mark zeros with length of consecutive zeros ######
# New version thanks to @nadare tip in sibmikes notebook,
# where I learned about np.frompyfunc, and how it can
# make python functions run really fast.
def mark_streaks(ts):
    """Returns an array of the same length as ts,
    except positive values are replaced by zero,
    and zeros are replaced by the lenth of the zero
    streak to which they belong.

    ########## Example ############
    ### in ###
    series = np.array([np.nan,3,0,0,0,2,0,0,1,0])
    mark_streaks(series)

    ### out ###
    array([nan,  0.,  3.,  3.,  3.,  0.,  2.,  2.,  0.,  1.])
    """
    ts_nan_mask = np.isnan(ts)
    zeros = ~(ts > 0) * 1
    accum_add_prod = np.frompyfunc(lambda x, y: int((x + y)*y), 2, 1)
    a = accum_add_prod.accumulate(zeros, dtype=np.object)
    a = np.where(a==0, 0, np.where(a < np.roll(a, -1), np.nan, a))
    a = pd.Series(a).fillna(method='bfill').to_numpy()
    item_streaks = np.where(ts_nan_mask, np.nan, a)

    return item_streaks

# Cell
@call_parse
def make_oos_data(PATH_DATA_RAW: Param('Path to raw data', str)='data/raw',
                  PATH_DATA_INTERIM: Param('Path to interim data', str)='data/interim') -> None:
    """Creates 2 csv files and stores them in the `PATH_DATA_INTERIM`.

    The first file is of all time series in the aggregation levels
    10, 11, and 12, stored as 'oos_series_df_level_10_11_12.csv'.

    The second file, 'oos_sales_train_evaluation.csv', has the same
    format as 'sales_train_evaluation.csv', except zero streaks
    that are believed to be caused by a stock-out are marked with
    nan.
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Predicting out-of-stock days for each product and filling with NaNs')
    start_time = time.time()

    df_stv = pd.read_csv(os.path.join(PATH_DATA_RAW, 'sales_train_evaluation.csv'))
    df_cal = pd.read_csv(os.path.join(PATH_DATA_RAW, 'calendar.csv'))
    rollup_matrix_csr, rollup_index  = get_agg(df_stv)
    series_df = get_series_df(df_stv, rollup_matrix_csr, rollup_index, df_cal=df_cal, fill_xmas=True)

    mask = series_df.index.get_level_values(0) > 9
    items = series_df[mask].index
    new_series = np.array([fix_oos(item, series_df)[0] for item in items])
    new_df = pd.DataFrame(new_series, columns=series_df.columns, index=series_df[mask].index)

    new_df.to_csv(os.path.join(PATH_DATA_INTERIM, 'oos_series_df_level_10_11_12.csv'))
    new_df = new_df.loc[12]
    new_df.index = new_df.index + '_evaluation'
    df_stv.iloc[:, 6:] = new_df.reindex(df_stv.id).values
    df_stv.to_csv(os.path.join(PATH_DATA_INTERIM, 'oos_sales_train_evaluation.csv'), index=False)

    logging.info(72 * '#')
    logging.info(time_taken(start_time))