import { Construct } from 'constructs';
import { UserEngine } from './common';
import { IUser } from './user-base';
import { IResource, Resource } from 'aws-cdk-lib/core';
/**
 * Properties for defining an ElastiCache UserGroup
 */
export interface UserGroupProps {
    /**
     * Enforces a particular physical user group name.
     * @default <generated>
     */
    readonly userGroupName?: string;
    /**
     * The engine type for the user group
     * Enum options: UserEngine.VALKEY, UserEngine.REDIS
     *
     * @default UserEngine.VALKEY
     */
    readonly engine?: UserEngine;
    /**
     * List of users inside the user group
     *
     * @default - no users
     */
    readonly users?: IUser[];
}
/**
 * Represents an ElastiCache UserGroup
 */
export interface IUserGroup extends IResource {
    /**
     * The name of the user group
     *
     * @attribute
     */
    readonly userGroupName: string;
    /**
     * The engine type for the user group
     */
    readonly engine?: UserEngine;
    /**
     * List of users in the user group
     */
    readonly users?: IUser[];
    /**
     * The ARN of the user group
     *
     * @attribute
     */
    readonly userGroupArn: string;
    /**
     * Add a user to this user group
     *
     * @param user The user to add
     */
    addUser(user: IUser): void;
}
/**
 * Base class for UserGroup constructs
 */
export declare abstract class UserGroupBase extends Resource implements IUserGroup {
    /**
     * The name of the user group
     *
     * @attribute
     */
    abstract readonly userGroupName: string;
    /**
     * The engine type for the user group
     */
    abstract readonly engine?: UserEngine;
    /**
     * List of users in the user group
     */
    abstract readonly users?: IUser[];
    /**
     * The ARN of the user group
     * @attribute
     */
    abstract readonly userGroupArn: string;
    /**
     * Add a user to this user group
     *
     * @param _user The user to add
     */
    addUser(_user: IUser): void;
}
/**
 * Attributes that can be specified when importing a UserGroup
 */
export interface UserGroupAttributes {
    /**
     * The name of the user group
     *
     * One of `userGroupName` or `userGroupArn` is required.
     *
     * @default - derived from userGroupArn
     */
    readonly userGroupName?: string;
    /**
     * The engine type for the user group
     *
     * @default - engine type is unknown
     */
    readonly engine?: UserEngine;
    /**
     * List of users in the user group
     *
     * @default - users are unknown
     */
    readonly users?: IUser[];
    /**
     * The ARN of the user group
     *
     * One of `userGroupName` or `userGroupArn` is required.
     *
     * @default - derived from userGroupName
     */
    readonly userGroupArn?: string;
}
/**
 * An ElastiCache UserGroup
 *
 * @resource AWS::ElastiCache::UserGroup
 */
export declare class UserGroup extends UserGroupBase {
    /**
     * Uniquely identifies this class
     */
    static readonly PROPERTY_INJECTION_ID: string;
    /**
     * Return whether the given object is a `UserGroup`
     */
    static isUserGroup(x: any): x is UserGroup;
    /**
     * Import an existing user group by name
     *
     * @param scope The parent creating construct (usually `this`)
     * @param id The construct's name
     * @param userGroupName The name of the existing user group
     */
    static fromUserGroupName(scope: Construct, id: string, userGroupName: string): IUserGroup;
    /**
     * Import an existing user group by ARN
     *
     * @param scope The parent creating construct (usually `this`)
     * @param id The construct's name
     * @param userGroupArn The ARN of the existing user group
     */
    static fromUserGroupArn(scope: Construct, id: string, userGroupArn: string): IUserGroup;
    /**
     * Import an existing user group using attributes
     *
     * @param scope The parent creating construct (usually `this`)
     * @param id The construct's name
     * @param attrs A `UserGroupAttributes` object
     */
    static fromUserGroupAttributes(scope: Construct, id: string, attrs: UserGroupAttributes): IUserGroup;
    readonly engine?: UserEngine;
    readonly userGroupName: string;
    private readonly _users;
    /**
     * The ARN of the user group
     *
     * @attribute
     */
    readonly userGroupArn: string;
    /**
     * The status of the user group
     * Can be 'creating', 'active', 'modifying', 'deleting'
     *
     * @attribute
     */
    readonly userGroupStatus: string;
    private readonly resource;
    constructor(scope: Construct, id: string, props?: UserGroupProps);
    /**
     * Add a CloudFormation dependency on the user resource to ensure proper creation order.
     */
    private addUserDependency;
    /**
     * Array of users in the user group
     *
     * Do not push directly to this array.
     * Use addUser() instead to ensure proper validation and dependency management.
     */
    get users(): IUser[] | undefined;
    /**
     * Validates users in the user group for duplicate usernames and Redis-specific requirements.
     */
    private validateUsers;
    /**
     * Add a user to this user group
     *
     * @param user The user to add to the group
     */
    addUser(user: IUser): void;
}
