"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserBase = exports.AccessControl = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("aws-cdk-lib/core");
/**
 * Access control configuration for ElastiCache users.
 */
class AccessControl {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@aws-cdk/aws-elasticache-alpha.AccessControl", version: "2.223.0-alpha.0" };
    /**
     * Create access control from an access string.
     *
     * @param accessString The access string defining user permissions.
     */
    static fromAccessString(accessString) {
        return new AccessControlString(accessString);
    }
}
exports.AccessControl = AccessControl;
/**
 * Access control implementation using a raw access string.
 */
class AccessControlString extends AccessControl {
    /**
     * The access string that defines user's permissions.
     */
    accessString;
    constructor(accessString) {
        super();
        this.accessString = accessString;
    }
}
/**
 * Base class for ElastiCache users.
 */
class UserBase extends core_1.Resource {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@aws-cdk/aws-elasticache-alpha.UserBase", version: "2.223.0-alpha.0" };
    /**
     * Import an existing user by ID.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param userId The ID of the existing user.
     */
    static fromUserId(scope, id, userId) {
        return UserBase.fromUserAttributes(scope, id, { userId });
    }
    /**
     * Import an existing user by ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param userArn The ARN of the existing user.
     */
    static fromUserArn(scope, id, userArn) {
        return UserBase.fromUserAttributes(scope, id, { userArn });
    }
    /**
     * Import an existing user using attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `UserBaseAttributes` object.
     */
    static fromUserAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticache_alpha_UserBaseAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromUserAttributes);
            }
            throw error;
        }
        let userId;
        let userArn;
        const stack = core_1.Stack.of(scope);
        if (attrs.userArn && attrs.userId) {
            throw new core_1.ValidationError('Only one of userArn or userId can be provided.', scope);
        }
        if (attrs.userArn) {
            userArn = attrs.userArn;
            const extractedUserId = stack.splitArn(attrs.userArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!extractedUserId) {
                throw new core_1.ValidationError('Unable to extract user id from ARN.', scope);
            }
            userId = extractedUserId;
        }
        else if (attrs.userId) {
            userId = attrs.userId;
            userArn = stack.formatArn({
                service: 'elasticache',
                resource: 'user',
                resourceName: attrs.userId,
            });
        }
        else {
            throw new core_1.ValidationError('One of userId or userArn is required.', scope);
        }
        class Import extends core_1.Resource {
            engine;
            userId;
            userArn;
            userName;
            constructor(_userArn, _userId) {
                super(scope, id);
                this.userArn = _userArn;
                this.userId = _userId;
                this.engine = attrs.engine;
                this.userName = attrs.userName;
            }
        }
        return new Import(userArn, userId);
    }
}
exports.UserBase = UserBase;
//# sourceMappingURL=data:application/json;base64,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